# Copyright (c) 2009 Movable Type ACME Plugin Project, All rights reserved.
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

package Manifest::App;
use strict;
use warnings;

use utf8;
use Encode;

sub _hdlr_manifests {
	my ($ctx, $args, $cond) = @_;
	$ctx->tag('Entries', $args, $cond);
}

sub _hdlr_proxy {
	my ($ctx, $args, $cond) = @_;
	my $tag = lc($ctx->stash('tag'));
	$tag =~ s/manifest/entry/;
	$ctx->tag($tag, $args, $cond);
}

sub _hdlr_manifest {
	my ($ctx, $args, $cond) = @_;
	my $current = $ctx->{__stash}{entry};

	if ($args->{revision} eq 'original') {
		my $rev = $current->load_revision(
			{ rev_number => $current->first_revision }
		);
		if ( $rev && @$rev ) {
			$ctx->{__stash}{entry} = $rev->[0];
		}
	}

	my $result = $ctx->slurp($args, $cond);

	$ctx->{__stash}{entry} = $current;

	$result;
}

sub _hdlr_if_modified {
	my ($ctx, $args, $cond) = @_;

	my $entry = $ctx->stash('entry')
        or return $ctx->_no_entry_error();
	$entry->revision != $entry->first_revision;
}

sub _hdlr_party {
	my ($ctx, $args) = @_;

	my $entry = $ctx->stash('entry')
        or return $ctx->_no_entry_error();
	$entry->party;
}

sub enabled {
	my $blog_id = shift;
	if (! $blog_id) {
		my $blog = MT->instance->blog
			or return 0;
		$blog_id = $blog->id;
	}
	my $plugin = MT->component('Manifest');
	$plugin->get_config_value('enabled', 'blog:' . $blog_id);
}

sub pre_save {
	my ($cb, $obj, $original) = @_;

	if (! $obj->first_revision) {
		require MT::Entry;
		if ($obj->status == MT::Entry::RELEASE()) {
			my $revision = $obj->revision || 0;
			$obj->first_revision($revision + 1);
		}
		else {
			$obj->first_revision(0);
		}
	}

	if (! $obj->party) {
		$obj->party('');
	}

	if ($obj->id && &enabled($obj->blog_id)) {
		delete($obj->{changed_cols}->{'title'});
		delete($obj->{changed_cols}->{'party'});
	}

	1;
}

sub save_filter {
	return 1 unless &enabled();

	my ($cb, $app) = @_;
	my $plugin = MT->component('Manifest');
	
	my $party = $app->param('party') || '';
	$party =~ s/　|\s//g;

	if (! $party) {
		return $cb->error($plugin->translate('Party is required.'));
	}

	1;
}

sub remove_filter {
	my ($cb, $app, $obj) = @_;
	my $plugin = MT->component('Manifest');

	if ($obj->id && &enabled($obj->blog_id)) {
		return $cb->error($plugin->translate('The manifest is not editable.'));
	}

	1;
}

sub pre_run {
	return 1 unless &enabled();

	my $plugin = MT->component('Manifest');
	my $lh = MT->language_handle;

	my $file = File::Spec->catfile(
		$plugin->{full_path}, 'l10n', $lh->language_tag . '.yaml'
	);
	if (! -f $file) {
		$file = File::Spec->catfile(
			$plugin->{full_path}, 'l10n', 'en_us.yaml'
		);
	}

	require YAML::Tiny;
	my $yaml = YAML::Tiny->read($file);
	my $map = $yaml->[0];
	my $package = ref $lh;

	eval(<<__EOE__);
package $package;
\$Lexicon{\$_} = \$map->{\$_} foreach (keys(%\$map));
__EOE__
}

sub param_edit_entry {
	return 1 unless &enabled();

	my ($cb, $app, $param, $tmpl) = @_;

	return 1 unless $app->param('_type') eq 'entry';

	my $plugin = MT->component('Manifest');

	push(@{ $param->{'field_loop'} }, {
		'field_id' => 'party',
		'lock_field' => '0',
		'field_name' => 'party',
		'show_field' => 1,
		'field_label' => $plugin->translate('Party'),
		'label_class' => 'top-label',
		'required' => '1',
		'field_html' => <<__EOF__,
<input type="text" name="party" id="party" class="full-width" value="<mt:var name="party" escape="html" \>" mt:watch-change="1" autocomplete="off" />
__EOF__
	});

	require MT::Entry;
	return if (! $param->{id}) || ($param->{status} != MT::Entry::RELEASE());

	my $confirm = $plugin->translate(
		'A manifest change is not recommended. \nMay I change really?'
	);
	my $cant_mod_title = $plugin->translate(
		'The title is not editable.'
	);
	my $cant_mod_party = $plugin->translate(
		'The party is not editable.'
	);
	$param->{'js_include'} ||= '';
	$param->{'js_include'} .= <<__EOS__;
<script type="text/javascript">
jQuery(function(j) {
	var messages = {
		'title': '$cant_mod_title',
		'party': '$cant_mod_party'
	};

	j('#title, #party').
		attr('readonly', 'readonly').
		click(function() {
			alert(messages[this.id]);
		});

	j('button.delete').parent().hide();
	j('button.publish, button.draft').click(function(ev) {
		if (! window.confirm('$confirm')) {
			ev.preventDefault();
			ev.stopPropagation();
		}
	});
});
</script>
__EOS__
}

sub param_list_entry {
	return 1 unless &enabled();

	my ($cb, $app, $param, $tmpl) = @_;

	return 1 unless $app->mode eq 'list_entry';

	$param->{'js_include'} ||= '';
	$param->{'js_include'} .= <<__EOS__;
<script type="text/javascript">
jQuery(function(j) {
	j('td.status-publish').parent().find('.cb input').hide();
});
</script>
__EOS__
}

1;
