/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.utils;

import java.util.Calendar;
import java.util.Date;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.utils.MonthUtility;

/**
 * 祝日クラス。<br>
 */
public class HolidayUtility {
	
	/**
	 * スタイル文字列(赤)。
	 */
	public static final String			STYLE_RED						= "style=\"color: red\"";
	
	/**
	 * スタイル文字列(青)。
	 */
	public static final String			STYLE_BLUE						= "style=\"color: blue\"";
	
	/**
	 * スタイル文字列(黄)。
	 */
	public static final String			STYLE_YELLOW					= "style=\"color: darkorange\"";
	
	/**
	 * 文言コード(成人の日)。
	 */
	protected static final String		NAM_COMING_OF_AGE_DAY			= "ComingOfAgeDay";
	
	/**
	 * 文言コード(海の日)。
	 */
	protected static final String		NAM_MARINE_DAY					= "MarineDay";
	
	/**
	 * 文言コード(山の日)。
	 */
	protected static final String		NAM_MOUNT_DAY					= "MountDay";
	
	/**
	 * 文言コード(敬老の日)。
	 */
	protected static final String		NAM_RESPECT_FOR_THE_AGED_DAY	= "RespectForTheAgedDay";
	
	/**
	 * 文言コード(体育の日)。
	 */
	protected static final String		NAM_SPORTS_DAY					= "SportsDay";
	
	/**
	 * 文言コード(春分の日)。
	 */
	protected static final String		NAM_VERNAL_EQUINOX_DAY			= "VernalEquinoxDay";
	
	/**
	 * 文言コード(秋分の日)。
	 */
	protected static final String		NAM_AUTUMNAL_EQUINOX_DAY		= "AutumnalEquinoxDay";
	
	/**
	 * 文言コード(元日)。
	 */
	protected static final String		NAM_NEW_YEARS_DAY				= "NewYearsDay";
	
	/**
	 * 文言コード(建国記念の日)。
	 */
	protected static final String		NAM_NATIONAL_FOUNDATION_DAY		= "NationalFoundationDay";
	
	/**
	 * 文言コード(昭和の日)。
	 */
	protected static final String		NAM_SHOWA_DAY					= "ShowaDay";
	
	/**
	 * 文言コード(憲法記念日)。
	 */
	protected static final String		NAM_CONSTITUTION_DAY			= "ConstitutionDay";
	
	/**
	 * 文言コード(みどりの日)。
	 */
	protected static final String		NAM_GREENERY_DAY				= "GreeneryDay";
	
	/**
	 * 文言コード(こどもの日)。
	 */
	protected static final String		NAM_CHILDRENS_DAY				= "ChildrensDay";
	
	/**
	 * 文言コード(文化の日)。
	 */
	protected static final String		NAM_CULTURE_DAY					= "CultureDay";
	
	/**
	 * 文言コード(勤労感謝の日)。
	 */
	protected static final String		NAM_LABOR_THANKSGIVING_DAY		= "LaborThanksgivingDay";
	
	/**
	 * 文言コード(天皇誕生日)。
	 */
	protected static final String		NAM_EMPERORS_BIRTHDAY			= "EmperorsBirthday";
	
	/**
	 * 文言コード(国民の休日)。
	 */
	protected static final String		NAM_PEOPLES_DAY					= "PeoplesDay";
	
	/**
	 * 文言コード(振替休日)。
	 */
	protected static final String		NAM_OBSERVED_HOLIDAY			= "ObservedHoliday";
	
	/**
	 * 祝日マップ。
	 */
	protected static Map<Date, String>	holidayMap;
	
	/**
	 * 対象年。
	 */
	protected static int				year;
	
	
	/**
	 * 対象日の祝日確認を行う。<br>
	 * @param targetDate 対象日
	 * @param mospParams MosP処理情報
	 * @return 確認結果(true：祝日、false：祝日でない)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static boolean isHoliday(Date targetDate, MospParams mospParams) throws MospException {
		// 対象日付の年度を取得
		int fisicalYear = MonthUtility.getFiscalYear(targetDate, mospParams);
		// 祝日マップ取得
		getHolidayMap(fisicalYear, mospParams);
		// 祝日名取得
		return holidayMap.get(targetDate) != null;
	}
	
	/**
	 * 対象日の年における祝日マップを取得する。<br>
	 * 但し、既に当該年の祝日マップが取得されている場合は、何もしない。<br>
	 * @param fiscalYear 対象年度
	 * @param mospParams MosP処理情報
	 * @return 祝日マップ
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Map<Date, String> getHolidayMap(int fiscalYear, MospParams mospParams) throws MospException {
		// 対象年と比較
		if (year == fiscalYear) {
			// 何もしない
			return holidayMap;
		}
		// 対象年を設定
		year = fiscalYear;
		// 祝日マップ作成
		createHolidayMap(mospParams);
		// 祝日マップ取得
		return holidayMap;
	}
	
	/**
	 * 対象日勤務曜日のスタイル文字列を取得する。<br>
	 * @param targetDate 対象日
	 * @param mospParams MosP処理情報
	 * @return 対象日勤務曜日のスタイル文字列
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static String getWorkDayOfWeekStyle(Date targetDate, MospParams mospParams) throws MospException {
		// 祝日判定
		if (isHoliday(targetDate, mospParams)) {
			return STYLE_RED;
		}
		// 土曜日判定
		if (DateUtility.isSaturday(targetDate)) {
			return STYLE_BLUE;
		}
		// 日曜日判定
		if (DateUtility.isSunday(targetDate)) {
			return STYLE_RED;
		}
		return "";
	}
	
	/**
	 * Calendarインスタンスを取得する。
	 * @return	Calendarインスタンス
	 */
	public static Calendar getCalendar() {
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		return cal;
	}
	
	/**
	 * 祝日Mapを生成する。
	 * @param mospParams MosP処理情報
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static void createHolidayMap(MospParams mospParams) throws MospException {
		
		// 元日
		addHolidayMap(getNewYearsDay(mospParams), mospParams.getName(NAM_NEW_YEARS_DAY));
		// 成人の日
		addHolidayMap(getComingOfAgeDay(mospParams), mospParams.getName(NAM_COMING_OF_AGE_DAY));
		// 建国記念の日
		addHolidayMap(getNationalFoundationDay(mospParams), mospParams.getName(NAM_NATIONAL_FOUNDATION_DAY));
		// 春分の日
		addHolidayMap(getVernalEquinoxDay(mospParams), mospParams.getName(NAM_VERNAL_EQUINOX_DAY));
		// 昭和の日	
		addHolidayMap(getShowaDay(mospParams), mospParams.getName(NAM_SHOWA_DAY));
		// 憲法記念日
		addHolidayMap(getConstitutionDay(mospParams), mospParams.getName(NAM_CONSTITUTION_DAY));
		// みどりの日
		addHolidayMap(getGreeneryDay(mospParams), mospParams.getName(NAM_GREENERY_DAY));
		// こどもの日
		addHolidayMap(getChildrensDay(mospParams), mospParams.getName(NAM_CHILDRENS_DAY));
		// 海の日
		addHolidayMap(getMarineDay(mospParams), mospParams.getName(NAM_MARINE_DAY));
		// 山の日
		// 山の日導入年
		final int showaSwitchYear = 2016;
		if (getYear(year, Calendar.AUGUST, mospParams) >= showaSwitchYear) {
			addHolidayMap(getMountDay(mospParams), mospParams.getName(NAM_MOUNT_DAY));
		}
		// 敬老の日
		addHolidayMap(getRespectForTheAgedDay(mospParams), mospParams.getName(NAM_RESPECT_FOR_THE_AGED_DAY));
		// 秋分の日
		addHolidayMap(getAutumnalEquinoxDay(mospParams), mospParams.getName(NAM_AUTUMNAL_EQUINOX_DAY));
		// 体育の日
		addHolidayMap(getSportsDay(mospParams), mospParams.getName(NAM_SPORTS_DAY));
		// 文化の日
		addHolidayMap(getCultureDay(mospParams), mospParams.getName(NAM_CULTURE_DAY));
		// 勤労感謝の日
		addHolidayMap(getLaborThanksgivingDay(mospParams), mospParams.getName(NAM_LABOR_THANKSGIVING_DAY));
		// 天皇誕生日
		addHolidayMap(getEmperorsBirthday(mospParams), mospParams.getName(NAM_EMPERORS_BIRTHDAY));
		// 振替休日を追加
		addSubstituteDate(mospParams);
	}
	
	/**
	 * Mapに祝日を追加する。
	 * @param date 対象年月日
	 * @param name 対象祝祭日名
	 */
	public static void addHolidayMap(Date date, String name) {
		if (holidayMap == null) {
			holidayMap = new TreeMap<Date, String>();
		}
		holidayMap.put(date, name);
	}
	
	/**
	 * 振替休日を追加する。
	 * @param mospParams MosP処理情報
	 */
	public static void addSubstituteDate(MospParams mospParams) {
		final int switchYear = 2006;
		Calendar cal = getCalendar();
		Set<Date> keySet = holidayMap.keySet();
		TreeMap<Date, String> addMap = new TreeMap<Date, String>();
		Date formerHoliday = null;
		for (Date date : keySet) {
			// 第三条第二項
			// 「国民の祝日」が日曜日に当たるときは、その日後においてその日に最も近い「国民の祝日」でない日を休日とする。
			// 日曜だった場合、最も近い「国民の祝日」でない日に「振替休日」を追加する。
			if (DateUtility.isSunday(date)) {
				cal.setTime(date);
				cal.add(Calendar.DAY_OF_MONTH, 1);
				if (year > switchYear) {
					// 既に登録されている休日と重複した場合、一日ずらす。
					while (holidayMap.containsKey(cal.getTime())) {
						cal.add(Calendar.DAY_OF_MONTH, 1);
					}
				}
				if (!holidayMap.containsKey(cal.getTime())) {
					addMap.put(cal.getTime(), mospParams.getName(NAM_OBSERVED_HOLIDAY));
				}
			}
			// 第三条第三項
			// その前日及び翌日が「国民の祝日」である日（「国民の祝日」でない日に限る。）は、休日とする。
			if (formerHoliday != null) {
				cal.setTime(formerHoliday);
				cal.add(Calendar.DAY_OF_MONTH, 2);
				if (cal.getTime().compareTo(date) == 0) {
					cal.add(Calendar.DAY_OF_MONTH, -1);
					addMap.put(cal.getTime(), mospParams.getName(NAM_PEOPLES_DAY));
				}
			}
			formerHoliday = date;
		}
		holidayMap.putAll(addMap);
	}
	
	/**
	 * 成人の日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	成人の日(1月第2月曜日)
	 * <p>
	 * 1999年以前は(1月15日)
	 * </p>
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getComingOfAgeDay(MospParams mospParams) throws MospException {
		int targetYear = getYear(year, Calendar.JANUARY, mospParams);
		final int switchYear = 2000;
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, targetYear);
		cal.set(Calendar.MONTH, Calendar.JANUARY);
		if (targetYear < switchYear) {
			cal.set(Calendar.DAY_OF_MONTH, 15);
		} else {
			cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
			cal.set(Calendar.DAY_OF_WEEK_IN_MONTH, 2);
		}
		return cal.getTime();
	}
	
	/**
	 * 海の日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	海の日(7月第3月曜日)
	 * <p>
	 * 2002年以前は(7月20日)
	 * </p>
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getMarineDay(MospParams mospParams) throws MospException {
		int targetYear = getYear(year, Calendar.JULY, mospParams);
		final int switchYear = 2003;
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, targetYear);
		cal.set(Calendar.MONTH, Calendar.JULY);
		if (targetYear < switchYear) {
			cal.set(Calendar.DAY_OF_MONTH, 20);
		} else {
			cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
			cal.set(Calendar.DAY_OF_WEEK_IN_MONTH, 3);
		}
		return cal.getTime();
	}
	
	/**
	 * 山の日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	山の日(8月11日)
	 * <p>
	 * 2016年以降適用。<br>
	 * </p>
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getMountDay(MospParams mospParams) throws MospException {
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, getYear(year, Calendar.AUGUST, mospParams));
		cal.set(Calendar.MONTH, Calendar.AUGUST);
		cal.set(Calendar.DAY_OF_MONTH, 11);
		return cal.getTime();
	}
	
	/**
	 * 敬老の日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	敬老の日(9月第3月曜日)
	 * <p>
	 * 2002年以前は(9月15日)
	 * </p>
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getRespectForTheAgedDay(MospParams mospParams) throws MospException {
		int targetYear = getYear(year, Calendar.SEPTEMBER, mospParams);
		final int switchYear = 2003;
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, targetYear);
		cal.set(Calendar.MONTH, Calendar.SEPTEMBER);
		if (targetYear < switchYear) {
			cal.set(Calendar.DAY_OF_MONTH, 15);
		} else {
			cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
			cal.set(Calendar.DAY_OF_WEEK_IN_MONTH, 3);
		}
		return cal.getTime();
	}
	
	/**
	 * 体育の日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	体育の日(10月第2月曜日)
	 * <p>
	 * 1999年以前は(10月10日)
	 * </p>
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getSportsDay(MospParams mospParams) throws MospException {
		int targetYear = getYear(year, Calendar.OCTOBER, mospParams);
		final int switchYear = 2000;
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, targetYear);
		cal.set(Calendar.MONTH, Calendar.OCTOBER);
		if (targetYear < switchYear) {
			cal.set(Calendar.DAY_OF_MONTH, 10);
		} else {
			cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
			cal.set(Calendar.DAY_OF_WEEK_IN_MONTH, 2);
		}
		return cal.getTime();
	}
	
	/**
	 * 春分の日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	春分の日(3月xx日)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getVernalEquinoxDay(MospParams mospParams) throws MospException {
		int targetYear = getYear(year, Calendar.MARCH, mospParams);
		final double param1 = 21.4471d;
		final double param2 = 0.242377d;
		final double param3 = 1900d;
		final double param4 = 4.0d;
		Calendar cal = getCalendar();
		double date = param1 + (param2 * (targetYear - param3)) - Math.floor((targetYear - param3) / param4);
		int dd = (int)Math.floor(date);
		cal.set(Calendar.YEAR, targetYear);
		cal.set(Calendar.MONTH, Calendar.MARCH);
		cal.set(Calendar.DAY_OF_MONTH, dd);
		return cal.getTime();
	}
	
	/**
	 * 秋分の日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return 秋分の日(9月xx日)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getAutumnalEquinoxDay(MospParams mospParams) throws MospException {
		int targetYear = getYear(year, Calendar.SEPTEMBER, mospParams);
		final double param1 = 23.8896d;
		final double param2 = 0.242032d;
		final double param3 = 1900d;
		final double param4 = 4.0d;
		Calendar cal = getCalendar();
		final double date = param1 + (param2 * (targetYear - param3)) - Math.floor((targetYear - param3) / param4);
		int dd = (int)Math.floor(date);
		cal.set(Calendar.YEAR, targetYear);
		cal.set(Calendar.MONTH, Calendar.SEPTEMBER);
		cal.set(Calendar.DAY_OF_MONTH, dd);
		return cal.getTime();
	}
	
	/**
	 * 元日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	元日(1月1日)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getNewYearsDay(MospParams mospParams) throws MospException {
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, getYear(year, Calendar.JANUARY, mospParams));
		cal.set(Calendar.MONTH, Calendar.JANUARY);
		cal.set(Calendar.DAY_OF_MONTH, 1);
		return cal.getTime();
	}
	
	/**
	 * 建国記念の日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	建国記念の日(2月11日)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getNationalFoundationDay(MospParams mospParams) throws MospException {
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, getYear(year, Calendar.FEBRUARY, mospParams));
		cal.set(Calendar.MONTH, Calendar.FEBRUARY);
		cal.set(Calendar.DAY_OF_MONTH, 11);
		return cal.getTime();
	}
	
	/**
	 * 憲法記念日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	憲法記念日(5月3日)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getConstitutionDay(MospParams mospParams) throws MospException {
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, getYear(year, Calendar.MAY, mospParams));
		cal.set(Calendar.MONTH, Calendar.MAY);
		cal.set(Calendar.DAY_OF_MONTH, 3);
		return cal.getTime();
	}
	
	/**
	 * こどもの日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	こどもの日(5月5日)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getChildrensDay(MospParams mospParams) throws MospException {
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, getYear(year, Calendar.MAY, mospParams));
		cal.set(Calendar.MONTH, Calendar.MAY);
		cal.set(Calendar.DAY_OF_MONTH, 5);
		return cal.getTime();
	}
	
	/**
	 * 文化の日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	文化の日(11月3日)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getCultureDay(MospParams mospParams) throws MospException {
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, getYear(year, Calendar.NOVEMBER, mospParams));
		cal.set(Calendar.MONTH, Calendar.NOVEMBER);
		cal.set(Calendar.DAY_OF_MONTH, 3);
		return cal.getTime();
	}
	
	/**
	 * 勤労感謝の日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	勤労感謝の日(11月23日)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getLaborThanksgivingDay(MospParams mospParams) throws MospException {
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, getYear(year, Calendar.NOVEMBER, mospParams));
		cal.set(Calendar.MONTH, Calendar.NOVEMBER);
		cal.set(Calendar.DAY_OF_MONTH, 23);
		return cal.getTime();
	}
	
	/**
	 * 天皇誕生日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	天皇誕生日(12月23日)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getEmperorsBirthday(MospParams mospParams) throws MospException {
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, getYear(year, Calendar.DECEMBER, mospParams));
		cal.set(Calendar.MONTH, Calendar.DECEMBER);
		cal.set(Calendar.DAY_OF_MONTH, 23);
		return cal.getTime();
	}
	
	/**
	 * 昭和の日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	昭和の日(4月29日)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getShowaDay(MospParams mospParams) throws MospException {
		final int dayOfMonth = 29;
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, getYear(year, Calendar.APRIL, mospParams));
		cal.set(Calendar.MONTH, Calendar.APRIL);
		cal.set(Calendar.DAY_OF_MONTH, dayOfMonth);
		return cal.getTime();
	}
	
	/**
	 * みどりの日を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return	みどりの日(5月4日)
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static Date getGreeneryDay(MospParams mospParams) throws MospException {
		Calendar cal = getCalendar();
		cal.set(Calendar.YEAR, getYear(year, Calendar.MAY, mospParams));
		cal.set(Calendar.MONTH, Calendar.MAY);
		cal.set(Calendar.DAY_OF_MONTH, 4);
		return cal.getTime();
	}
	
	/**
	 * 対象年度における対象月が含まれる年を取得する。<br>
	 * @param fiscalYear 対象年度
	 * @param targetMonth 対象月(0:1月、1：2月…)
	 * @param mospParams MosP処理情報
	 * @return 対象年度における対象月が含まれる年
	 * @throws MospException 日付操作に失敗した場合
	 */
	public static int getYear(int fiscalYear, int targetMonth, MospParams mospParams) throws MospException {
		return DateUtility.getYear(MonthUtility.getFiscalYearMonth(fiscalYear, targetMonth + 1, mospParams));
	}
	
}
