/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.input.action;

import java.util.Date;
import java.util.List;
import java.util.Map;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.bean.workflow.WorkflowCommentRegistBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowRegistBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.ApplicationReferenceBeanInterface;
import jp.mosp.time.bean.CutoffUtilBeanInterface;
import jp.mosp.time.bean.HolidayInfoReferenceBeanInterface;
import jp.mosp.time.bean.HolidayReferenceBeanInterface;
import jp.mosp.time.bean.HolidayRequestRegistBeanInterface;
import jp.mosp.time.bean.HolidayRequestSearchBeanInterface;
import jp.mosp.time.bean.PaidHolidayInfoReferenceBeanInterface;
import jp.mosp.time.bean.PaidHolidayReferenceBeanInterface;
import jp.mosp.time.bean.StockHolidayInfoReferenceBeanInterface;
import jp.mosp.time.bean.TimeSettingReferenceBeanInterface;
import jp.mosp.time.comparator.settings.HolidayRequestRequestDateComparator;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.constant.TimeMessageConst;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.CutoffDtoInterface;
import jp.mosp.time.dto.settings.HolidayDataDtoInterface;
import jp.mosp.time.dto.settings.HolidayDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestListDtoInterface;
import jp.mosp.time.dto.settings.PaidHolidayDtoInterface;
import jp.mosp.time.dto.settings.StockHolidayDataDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;
import jp.mosp.time.input.vo.HolidayRequestVo;
import jp.mosp.time.utils.TimeUtility;

/**
 * 休暇申請情報の確認と編集を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SHOW}
 * </li><li>
 * {@link #CMD_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_SEARCH}
 * </li><li>
 * {@link #CMD_RE_SHOW}
 * </li><li>
 * {@link #CMD_DRAFT}
 * </li><li>
 * {@link #CMD_APPLI}
 * </li><li>
 * {@link #CMD_TRANSFER}
 * </li><li>
 * {@link #CMD_WITHDRAWN}
 * </li><li>
 * {@link #CMD_SORT}
 * </li><li>
 * {@link #CMD_PAGE}
 * </li><li>
 * {@link #CMD_SET_ACTIVATION_DATE}
 * </li><li>
 * {@link #CMD_INSERT_MODE}
 * </li><li>
 * {@link #CMD_EDIT_MODE}
 * </li><li>
 * {@link #CMD_BATCH_UPDATE}
 * </li><li>
 * {@link #CMD_SET_VIEW_PERIOD}
 * </li><li>
 * {@link #CMD_SET_TRANSFER_HOLIDAY}
 * </li><li>
 * {@link #CMD_SELECT_ACTIVATION_DATE}
 * </li></ul>
 */
public class HolidayRequestAction extends TimeAction {
	
	/**
	 * 表示コマンド。<br>
	 * <br>
	 * 現在ログインしているユーザの休暇申請画面を表示する。<br>
	 */
	public static final String	CMD_SHOW					= "TM1500";
	
	/**
	 * 選択表示コマンド。<br>
	 * <br>
	 * 休暇申請画面を表示する。<br>
	 */
	public static final String	CMD_SELECT_SHOW				= "TM1501";
	
	/**
	 * 検索コマンド。<br>
	 * <br>
	 * 検索欄に入力した情報を元に休暇申請情報の検索を行う。<br>
	 */
	public static final String	CMD_SEARCH					= "TM1502";
	
	/**
	 * 再表示コマンド。<br>
	 * <br>
	 * 新たな下書情報作成や申請を行った際に検索結果一覧にそれらが反映されるよう再表示を行う。<br>
	 */
	public static final String	CMD_RE_SHOW					= "TM1503";
	
	/**
	 * 下書きマンド。<br>
	 * <br>
	 * 申請欄に入力した内容を休暇情報テーブルに登録し、下書情報として保存する。<br>
	 */
	public static final String	CMD_DRAFT					= "TM1504";
	
	/**
	 * 申請コマンド。<br>
	 * <br>
	 * 申請欄に入力した内容を休暇情報テーブルに登録し、休暇申請を行う。以降、このレコードは上長が差戻をしない限り編集不可となる。<br>
	 * 休暇の申請可能時間を超過している、申請年月日で申請不可な日付が選択されている、<br>
	 * 申請時間が0時間0分のまま、休暇理由の項目が入力されていない、<br>
	 * といった状態で申請を行おうとした場合は場合はエラーメッセージにて通知し、申請は実行されない。<br>
	 */
	public static final String	CMD_APPLI					= "TM1505";
	
	/**
	 * 画面遷移コマンド。<br>
	 * <br>
	 * 現在表示している画面から、ワークフロー番号をMosP処理情報に設定し、画面遷移する。<br>
	 */
	public static final String	CMD_TRANSFER				= "TM1506";
	
	/**
	 * 取下コマンド。<br>
	 * <br>
	 * 下書状態または差戻状態で登録されていたレコードの取下を行う。<br>
	 * 取下後、対象の休暇申請情報は未申請状態へ戻る。<br>
	 */
	public static final String	CMD_WITHDRAWN				= "TM1507";
	
	/**
	 * ソートコマンド。<br>
	 * <br>
	 * それぞれのレコードの値を比較して一覧表示欄の各情報毎に並び替えを行う。<br>
	 * これが実行される度に並び替えが昇順・降順と交互に切り替わる。<br>
	 */
	public static final String	CMD_SORT					= "TM1508";
	
	/**
	 * ページ繰りコマンド。<br>
	 * <br>
	 * 検索処理を行った際に検索結果が100件を超えた場合に分割されるページ間の遷移を行う。<br>
	 */
	public static final String	CMD_PAGE					= "TM1509";
	
	/**
	 * 休暇年月日決定コマンド。<br>
	 * <br>
	 * 休暇年月日入力欄に入力されている日付を元にログイン中のユーザが取得可能な休暇種別を検索、各プルダウンに表示させる。<br>
	 */
	public static final String	CMD_SET_ACTIVATION_DATE		= "TM1590";
	
	/**
	 * 新規登録モード切替コマンド。<br>
	 * <br>
	 * 申請テーブルの各入力欄に表示されているレコード内容をクリアにする。<br>
	 * 申請テーブルヘッダに表示されている新規登録モード切替リンクを非表示にする。<br>
	 */
	public static final String	CMD_INSERT_MODE				= "TM1591";
	
	/**
	 * 編集モード切替コマンド。<br>
	 * <br>
	 * 選択したレコードの内容を申請テーブルの各入力欄にそれぞれ表示させる。申請テーブルヘッダに新規登録モード切替リンクを表示させる。
	 * に表示されているレコード内容をクリアにする。<br>
	 * 申請テーブルヘッダに表示されている新規登録モード切替リンクを非表示にする。<br>
	 */
	public static final String	CMD_EDIT_MODE				= "TM1592";
	
	/**
	 * 一括更新コマンド。<br>
	 * <br>
	 * 一覧にて選択チェックボックスにチェックが入っている下書状態のレコードの申請処理を繰り返し行う。<br>
	 * ひとつもチェックが入っていない状態で一括更新ボタンをクリックした場合はエラーメッセージにて通知し、処理を中断する。<br>
	 */
	public static final String	CMD_BATCH_UPDATE			= "TM1595";
	
	/**
	 * 表示期間決定コマンド。<br>
	 * <br>
	 * 一入力した締期間で検索を行い、残日数のある休暇区分と休暇種別を取得し、プルダウンに表示する。<br>
	 */
	public static final String	CMD_SET_VIEW_PERIOD			= "TM1597";
	
	/**
	 * 休暇種別決定コマンド。<br>
	 * <br>
	 * 休暇情報の連続取得判定を行うために休暇種別プルダウン切替時にその休暇の連続取得区分が"必須/警告/不要"の内どれであるかを判断する。<br>
	 */
	public static final String	CMD_SET_TRANSFER_HOLIDAY	= "TM1598";
	
	/**
	 * 休暇年月日決定コマンド。<br>
	 * <br>
	 * 休暇年月日入力欄に入力されている日付を元にログイン中のユーザが取得可能な休暇種別を検索、各プルダウンに表示させる。<br>
	 */
	public static final String	CMD_SELECT_ACTIVATION_DATE	= "TM1599";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public HolidayRequestAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_RE_SHOW;
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new HolidayRequestVo();
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SHOW)) {
			// 表示
			prepareVo(false, false);
			show();
		} else if (mospParams.getCommand().equals(CMD_SELECT_SHOW)) {
			// 選択表示
			prepareVo(false, false);
			select();
		} else if (mospParams.getCommand().equals(CMD_SEARCH)) {
			// 検索
			prepareVo();
			search();
		} else if (mospParams.getCommand().equals(CMD_RE_SHOW)) {
			// 再表示
			prepareVo(true, false);
			search();
		} else if (mospParams.getCommand().equals(CMD_DRAFT)) {
			// 下書き
			prepareVo();
			draft();
		} else if (mospParams.getCommand().equals(CMD_APPLI)) {
			// 申請
			prepareVo();
			appli();
		} else if (mospParams.getCommand().equals(CMD_TRANSFER)) {
			// 遷移
			prepareVo(true, false);
			transfer();
		} else if (mospParams.getCommand().equals(CMD_WITHDRAWN)) {
			// 取下
			prepareVo();
			withdrawn();
		} else if (mospParams.getCommand().equals(CMD_SORT)) {
			// ソート 
			prepareVo();
			sort();
		} else if (mospParams.getCommand().equals(CMD_PAGE)) {
			// ページ繰り
			prepareVo();
			page();
		} else if (mospParams.getCommand().equals(CMD_SET_ACTIVATION_DATE)) {
			// 休暇年月日
			prepareVo();
			setEditActivationDate();
		} else if (mospParams.getCommand().equals(CMD_INSERT_MODE)) {
			// 新規モード切替
			prepareVo();
			insertMode();
		} else if (mospParams.getCommand().equals(CMD_EDIT_MODE)) {
			// 編集モード切替
			prepareVo();
			editMode();
		} else if (mospParams.getCommand().equals(CMD_BATCH_UPDATE)) {
			// 一括更新
			prepareVo();
			batchUpdate();
		} else if (mospParams.getCommand().equals(CMD_SET_VIEW_PERIOD)) {
			// 表示期間
			prepareVo();
			setSearchActivationDate();
		} else if (mospParams.getCommand().equals(CMD_SET_TRANSFER_HOLIDAY)) {
			//
			prepareVo();
			setHolidayContinue();
		} else if (mospParams.getCommand().equals(CMD_SELECT_ACTIVATION_DATE)) {
			// 休暇年月日
			prepareVo(false, false);
			selectActivationDate();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void show() throws MospException {
		// 個人ID取得(ログインユーザ情報から)
		String personalId = mospParams.getUser().getPersonalId();
		// 対象日取得(システム日付)
		Date targetDate = getSystemDate();
		// 人事情報をVOに設定
		setEmployeeInfo(personalId, targetDate);
		// ページ繰り設定
		setPageInfo(CMD_PAGE, getListLength());
		// 新規登録モード設定
		insertMode();
	}
	
	/**
	 * 選択表示処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void select() throws MospException {
		// リクエストから個人ID及び対象日を取得
		String personalId = getTargetPersonalId();
		// 対象日取得(システム日付)
		Date targetDate = getSystemDate();
		// 個人ID確認
		if (personalId == null || personalId.isEmpty()) {
			// ログインユーザの個人IDを取得
			personalId = mospParams.getUser().getPersonalId();
		}
		// 人事情報をVOに設定
		setEmployeeInfo(personalId, targetDate);
		// ページ繰り設定
		setPageInfo(CMD_PAGE, getListLength());
		// 新規登録モード設定
		insertMode();
	}
	
	/**
	 * @throws MospException 例外処理が発生した場合
	 */
	protected void search() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 検索クラス取得
		HolidayRequestSearchBeanInterface search = timeReference().holidayRequestSearch();
		// VOの値を検索クラスへ設定
		search.setPersonalId(vo.getPersonalId());
		
		String holidayType1 = vo.getPltSearchHolidayType();
		
		search.setHolidayType1(holidayType1);
		search.setHolidayType2("");
		search.setHolidayLength("");
		
		if (holidayType1 != null && !holidayType1.isEmpty()) {
			// 休暇種別の種類を判断
			int holidayType = Integer.parseInt(holidayType1);
			String holidayType2 = "";
			String holidayLength = "";
			
			if (TimeConst.CODE_HOLIDAYTYPE_HOLIDAY == holidayType) {
				// 有給休暇
				holidayType2 = vo.getPltSearchStatusWithPay();
				holidayLength = vo.getPltSearchHolidayRange1();
				
			} else if (TimeConst.CODE_HOLIDAYTYPE_SPECIAL == holidayType) {
				// 特別休暇
				holidayType2 = vo.getPltSearchStatusSpecial();
				holidayLength = vo.getPltSearchHolidayRange2();
				
			} else if (TimeConst.CODE_HOLIDAYTYPE_OTHER == holidayType) {
				// その他
				holidayType2 = vo.getPltSearchSpecialOther();
				holidayLength = vo.getPltSearchHolidayRange2();
				
			} else if (TimeConst.CODE_HOLIDAYTYPE_ABSENCE == holidayType) {
				// 欠勤
				holidayType2 = vo.getPltSearchSpecialAbsence();
				holidayLength = vo.getPltSearchHolidayRange2();
			}
			
			search.setHolidayType2(holidayType2);
			search.setHolidayLength(holidayLength);
		}
		// 変数準備
		int year = Integer.parseInt(vo.getPltSearchYear());
		int startMonth = Integer.parseInt(TimeConst.CODE_DISPLAY_JANUARY);
		int endMonth = Integer.parseInt(TimeConst.CODE_DISPLAY_DECEMBER);
		if (!vo.getPltSearchMonth().isEmpty()) {
			// 月検索ならば
			startMonth = Integer.parseInt(vo.getPltSearchMonth());
			endMonth = startMonth;
		}
		// 締日ユーティリティー取得
		CutoffUtilBeanInterface cutoffUtil = timeReference().cutoffUtil();
		// 締日情報取得
		CutoffDtoInterface startMonthCutoffDto = cutoffUtil
			.getCutoffForPersonalId(vo.getPersonalId(), year, startMonth);
		CutoffDtoInterface endMonthCutoffDto = cutoffUtil.getCutoffForPersonalId(vo.getPersonalId(), year, endMonth);
		String startMonthCutoffCode = startMonthCutoffDto == null ? null : startMonthCutoffDto.getCutoffCode();
		String endMonthCutoffCode = endMonthCutoffDto == null ? null : endMonthCutoffDto.getCutoffCode();
		// 締期間の開始及び最終日
		Date firstDate = null;
		if (startMonthCutoffCode != null) {
			firstDate = cutoffUtil.getCutoffFirstDate(startMonthCutoffCode, year, startMonth);
		}
		// 対象年月及び締日から締期間最終日を取得
		Date lastDate = null;
		if (endMonthCutoffCode != null) {
			lastDate = cutoffUtil.getCutoffLastDate(endMonthCutoffCode, year, endMonth);
		}
		// 締期間を検索範囲に設定
		search.setRequestStartDate(firstDate);
		search.setRequestEndDate(lastDate);
		search.setWorkflowStatus(vo.getPltSearchState());
		
		// 検索条件をもとに検索クラスからマスタリストを取得
		List<HolidayRequestListDtoInterface> list = search.getSearchList();
		// 検索結果リスト設定
		vo.setList(list);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(HolidayRequestRequestDateComparator.class.getName());
		vo.setAscending(false);
		// ソート
		sort();
		// 検索結果確認
		if (list.isEmpty()) {
			// 検索結果無しメッセージ設定
			addNoSearchResultMessage();
		}
	}
	
	/**
	 * 初期値を設定する。<br>
	 * @throws MospException 例外発生時
	 */
	public void setDefaultValues() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		Date date = DateUtility.getSystemDate();
		String personalId = vo.getPersonalId();
		final String nameNoLimit = mospParams.getName("NoLimit");
		PaidHolidayInfoReferenceBeanInterface paidHolidayInfo = timeReference().paidHolidayInfo();
		HolidayInfoReferenceBeanInterface holidayInfo = timeReference().holidayInfo();
		StockHolidayInfoReferenceBeanInterface stockHolidayInfo = timeReference().stockHolidayInfo();
		// 特別/その他休暇
		List<HolidayDataDtoInterface> specialList = holidayInfo.getHolidayPossibleRequestList(personalId, date,
				TimeConst.CODE_HOLIDAYTYPE_SPECIAL);
		List<HolidayDataDtoInterface> otherList = holidayInfo.getHolidayPossibleRequestList(personalId, date,
				TimeConst.CODE_HOLIDAYTYPE_OTHER);
		String[] aryGivingDate = new String[specialList.size() + otherList.size()];
		String[] arySpecialHolidayType = new String[specialList.size() + otherList.size()];
		String[] arySpecialHolidayName = new String[specialList.size() + otherList.size()];
		String[] aryRemainder = new String[specialList.size() + otherList.size()];
		String[] aryLimit = new String[specialList.size() + otherList.size()];
		int cnt = 0;
		for (cnt = 0; cnt < specialList.size(); cnt++) {
			// リストから情報を取得
			HolidayDataDtoInterface dto = specialList.get(cnt);
			aryGivingDate[cnt] = DateUtility.getStringDateAndDay(dto.getActivateDate());
			arySpecialHolidayType[cnt] = mospParams.getName("Specially") + mospParams.getName("Vacation");
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(dto.getHolidayCode(),
					dto.getActivateDate(), dto.getHolidayType());
			if (holidayDto != null) {
				arySpecialHolidayName[cnt] = holidayDto.getHolidayName();
			} else {
				arySpecialHolidayName[cnt] = dto.getHolidayCode();
			}
			if (TimeUtility.getUnlimitedDate().compareTo(dto.getHolidayLimitDate()) == 0) {
				aryRemainder[cnt] = nameNoLimit;
				aryLimit[cnt] = nameNoLimit;
			} else {
				aryRemainder[cnt] = String.valueOf(dto.getGivingDay() - dto.getCancelDay()) + mospParams.getName("Day");
				aryLimit[cnt] = DateUtility.getStringDateAndDay(dto.getHolidayLimitDate());
			}
		}
		for (int i = 0; i < otherList.size(); i++) {
			// リストから情報を取得
			HolidayDataDtoInterface dto = otherList.get(i);
			aryGivingDate[cnt] = DateUtility.getStringDateAndDay(dto.getActivateDate());
			arySpecialHolidayType[cnt] = mospParams.getName("Others") + mospParams.getName("Vacation");
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(dto.getHolidayCode(),
					dto.getActivateDate(), dto.getHolidayType());
			if (holidayDto != null) {
				arySpecialHolidayName[cnt] = holidayDto.getHolidayName();
			} else {
				arySpecialHolidayName[cnt] = dto.getHolidayCode();
			}
			if (TimeUtility.getUnlimitedDate().compareTo(dto.getHolidayLimitDate()) == 0) {
				aryRemainder[cnt] = nameNoLimit;
				aryLimit[cnt] = nameNoLimit;
			} else {
				aryRemainder[cnt] = String.valueOf(dto.getGivingDay() - dto.getCancelDay()) + mospParams.getName("Day");
				aryLimit[cnt] = DateUtility.getStringDateAndDay(dto.getHolidayLimitDate());
			}
			cnt++;
		}
		vo.setRecordId(0);
		vo.setAryLblGivingDate(aryGivingDate);
		vo.setAryLblSpecialHolidayType(arySpecialHolidayType);
		vo.setAryLblSpecialHolidayName(arySpecialHolidayName);
		vo.setAryLblRemainder(aryRemainder);
		vo.setAryLblLimit(aryLimit);
		//
		if (getTargetDate() == null) {
			vo.setPltEditStartYear(String.valueOf(DateUtility.getYear(date)));
			vo.setPltEditStartMonth(String.valueOf(DateUtility.getMonth(date)));
			vo.setPltEditStartDay(String.valueOf(DateUtility.getDay(date)));
			vo.setPltEditEndYear(String.valueOf(DateUtility.getYear(date)));
			vo.setPltEditEndMonth(String.valueOf(DateUtility.getMonth(date)));
			vo.setPltEditEndDay(String.valueOf(DateUtility.getDay(date)));
		} else {
			vo.setPltEditStartYear(String.valueOf(DateUtility.getYear(getTargetDate())));
			vo.setPltEditStartMonth(String.valueOf(DateUtility.getMonth(getTargetDate())));
			vo.setPltEditStartDay(String.valueOf(DateUtility.getDay(getTargetDate())));
			vo.setPltEditEndYear(String.valueOf(DateUtility.getYear(getTargetDate())));
			vo.setPltEditEndMonth(String.valueOf(DateUtility.getMonth(getTargetDate())));
			vo.setPltEditEndDay(String.valueOf(DateUtility.getDay(getTargetDate())));
		}
		// VOを初期化
		initVo();
		// ストック休暇の取得
		Map<String, Object> map = paidHolidayInfo.getPaidHolidayInfo(personalId, date);
		vo.setLblPaidHolidayStock(String.valueOf(stockHolidayInfo.getRemainDay(personalId, (Date)map
			.get(TimeConst.CODE_ACTIVATE_DATE), date)));
		// 有給休暇情報欄表示
		map = paidHolidayInfo.getPaidHolidayPossibleRequest(personalId, date);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		double formerYearDate = ((Double)map.get(TimeConst.CODE_FORMER_YEAR_DATE)).doubleValue();
		int formerYearTime = ((Integer)map.get(TimeConst.CODE_FORMER_YEAR_TIME)).intValue();
		double currentYearDate = ((Double)map.get(TimeConst.CODE_CURRENT_YEAR_DATE)).doubleValue();
		int currentYearTime = ((Integer)map.get(TimeConst.CODE_CURRENT_TIME)).intValue();
		vo.setLblFormerDay(Double.toString(formerYearDate));
		vo.setLblFormerTime(Integer.toString(formerYearTime));
		vo.setLblCurrentDay(Double.toString(currentYearDate));
		vo.setLblCurrentTime(Integer.toString(currentYearTime));
		// 有休申請可能日数
		vo.setLblTotalDay(Double.toString(formerYearDate + currentYearDate));
		// 有休申請可能時間
		vo.setLblTotalTime(Integer.toString(formerYearTime + currentYearTime));
		vo.setLblNextGivingDate(DateUtility.getStringDateAndDay(paidHolidayInfo.getNextGivingDate(personalId)));
		if (mospParams.hasErrorMessage()) {
			return;
		}
		vo.setLblNextGivingAmount(String.valueOf(paidHolidayInfo.getNextGivingDay(personalId)));
		// 次回付与予定日(手動)取得及び確認
		Date nextManualGivingDate = paidHolidayInfo.getNextManualGivingDate(personalId);
		if (null == nextManualGivingDate) {
			return;
		}
		// 次回付与予定日(手動)設定
		vo.setLblNextManualGivingDate(DateUtility.getStringDateAndDay(nextManualGivingDate));
		// 次回付与予定日数(手動)取得及び設定
		vo.setLblNextManualGivingAmount(paidHolidayInfo.getNextManualGivingDaysAndHours(vo.getPersonalId()));
	}
	
	/**
	 * VOの初期化を行う。 
	 */
	protected void initVo() {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		vo.setPltEditHolidayType1(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_HOLIDAY));
		vo.setPltEditStatusWithPay("");
		vo.setPltEditStatusSpecial("");
		vo.setPltEditSpecialOther("");
		vo.setPltEditSpecialAbsence("");
		vo.setPltEditHolidayRange("");
		vo.setPltEditHolidayRangePaidHoliday("");
		vo.setPltEditStartHour("0");
		vo.setPltEditStartMinute("0");
		vo.setPltEditEndHour("0");
		vo.setPltEditEndMinute("0");
		vo.setTxtEditRequestReason("");
		vo.setPltSearchHolidayType1("");
		vo.setPltSearchHolidayType2("");
		vo.setPltSearchHolidayType3("");
		vo.setPltSearchState("");
		vo.setPltSearchYear(DateUtility.getStringYear(getSystemDate()));
		vo.setPltSearchMonth("");
		// 承認者欄の初期化
		String[] aryPltLblApproverSetting = new String[0];
		vo.setAryPltLblApproverSetting(aryPltLblApproverSetting);
		vo.setPltApproverSetting1("");
		vo.setPltApproverSetting2("");
		vo.setPltApproverSetting3("");
		vo.setPltApproverSetting4("");
		vo.setPltApproverSetting5("");
		vo.setPltApproverSetting6("");
		vo.setPltApproverSetting7("");
		vo.setPltApproverSetting8("");
		vo.setPltApproverSetting9("");
		vo.setPltApproverSetting10("");
		// 編集項目設定
		vo.setTxtEditRequestReason("");
		// 有給休暇情報の初期設定
		vo.setLblFormerDay("0");
		vo.setLblFormerTime("0");
		vo.setLblCurrentDay("0");
		vo.setLblCurrentTime("0");
		vo.setLblNextGivingDate("0");
		vo.setLblNextGivingAmount("0");
		final String hyphen = mospParams.getName("Hyphen");
		vo.setLblNextManualGivingDate(hyphen);
		vo.setLblNextManualGivingAmount(hyphen);
	}
	
	/**
	* 下書処理を行う。<br>
	* @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	*/
	protected void draft() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 登録クラス取得
		HolidayRequestRegistBeanInterface regist = time().holidayRequestRegist();
		WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
		Date startDate = getEditStartDate();
		Date endDate = getEditEndDate();
		boolean isPaidHoliday = TimeConst.CODE_HOLIDAYTYPE_HOLIDAY == getInt(vo.getPltEditHolidayType1());
		boolean isSpecialHoliday = TimeConst.CODE_HOLIDAYTYPE_SPECIAL == getInt(vo.getPltEditHolidayType1());
		boolean isOtherHoliday = TimeConst.CODE_HOLIDAYTYPE_OTHER == getInt(vo.getPltEditHolidayType1());
		boolean isAbsence = TimeConst.CODE_HOLIDAYTYPE_ABSENCE == getInt(vo.getPltEditHolidayType1());
		if (isPaidHoliday) {
			// 休暇種別が有給休暇又はストック休暇の場合
			Date targetDate = startDate;
			if (vo.getPltEditStatusWithPay().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_HOLIDAY))) {
				// 休暇種別が有給休暇の場合
				while (!targetDate.after(endDate)) {
					Date acquisitionDate = null;
					double useDay = 0;
					int useHour = 0;
					Map<String, Object> map = timeReference().paidHolidayInfo().getPaidHolidayPossibleRequest(
							vo.getPersonalId(), startDate);
					if (TimeConst.CODE_HOLIDAY_RANGE_ALL == getInt(vo.getPltEditHolidayRangePaidHoliday())) {
						// 全休
						if (((Double)map.get(TimeConst.CODE_FORMER_YEAR_DATE)).doubleValue() >= 1) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Double)map.get(TimeConst.CODE_CURRENT_YEAR_DATE)).doubleValue() >= 1) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = 1;
						useHour = 0;
					} else if (TimeConst.CODE_HOLIDAY_RANGE_AM == getInt(vo.getPltEditHolidayRangePaidHoliday())
							|| TimeConst.CODE_HOLIDAY_RANGE_PM == getInt(vo.getPltEditHolidayRangePaidHoliday())) {
						// 半休
						if (((Double)map.get(TimeConst.CODE_FORMER_YEAR_DATE)).doubleValue() >= TimeConst.HOLIDAY_TIMES_HALF) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Double)map.get(TimeConst.CODE_CURRENT_YEAR_DATE)).doubleValue() >= TimeConst.HOLIDAY_TIMES_HALF) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = TimeConst.HOLIDAY_TIMES_HALF;
						useHour = 0;
					} else if ("4".equals(vo.getPltEditHolidayRangePaidHoliday())) {
						// 時休
						Date startTime = DateUtility.getTime(vo.getPltEditStartHour(), vo.getPltEditStartMinute());
						Date endTime = DateUtility.getTime(vo.getPltEditEndHour(), vo.getPltEditEndMinute());
						int hour = (int)((endTime.getTime() - startTime.getTime()) / (1000 * 60 * 60));
						if (((Integer)map.get(TimeConst.CODE_FORMER_YEAR_TIME)).intValue() >= hour) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Integer)map.get(TimeConst.CODE_CURRENT_TIME)).intValue() >= hour) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = 0;
						useHour = hour;
					} else {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						// 例外メッセージ
						mospParams.addErrorMessage(TimeMessageConst.MSG_RANGE_SELECT, null);
						return;
					}
					// DTOの準備
					HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
					if (dto == null || targetDate.compareTo(startDate) != 0) {
						dto = regist.getInitDto();
					}
					// DTOに値をセット
					setDtoFields(dto, targetDate, targetDate, acquisitionDate, useDay, useHour);
					if (targetDate.compareTo(startDate) != 0) {
						dto.setTmdHolidayRequestId(0);
					}
					// 妥当性チェック
					regist.validate(dto);
					if (mospParams.hasErrorMessage()) {
						return;
					}
					// 申請日チェック
					regist.checkRequestDate(dto);
					if (mospParams.hasErrorMessage()) {
						mospParams.getErrorMessageList().clear();
						// 1日加算
						targetDate = DateUtility.addDay(targetDate, 1);
						continue;
					}
					// 申請の相関チェック
					regist.checkDraft(dto);
					// ワークフローの設定
					WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
					if (workflowDto == null) {
						workflowDto = workflowRegist.getInitDto();
						workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
					}
					workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
					// 登録後ワークフローの取得
					workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
							PlatformConst.WORKFLOW_TYPE_TIME);
					if (workflowDto != null) {
						// ワークフローコメント登録
						platform().workflowCommentRegist().addComment(
								workflowDto,
								mospParams.getUser().getPersonalId(),
								mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
										new String[]{ mospParams.getName("WorkPaper") }));
						// ワークフロー番号セット
						dto.setWorkflow(workflowDto.getWorkflow());
						// 登録
						regist.regist(dto);
					}
					// 登録結果確認
					if (mospParams.hasErrorMessage()) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					// 1日加算
					targetDate = DateUtility.addDay(targetDate, 1);
				}
			} else if (vo.getPltEditStatusWithPay().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_STOCK))) {
				// 休暇種別がストック休暇の場合
				while (!targetDate.after(endDate)) {
					Date acquisitionDate = null;
					double useDay = 0;
					List<StockHolidayDataDtoInterface> list = timeReference().stockHolidayInfo()
						.getStockHolidayPossibleRequest(vo.getPersonalId(), startDate);
					if (list == null || list.isEmpty()) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					if (TimeConst.CODE_HOLIDAY_RANGE_ALL == getInt(vo.getPltEditHolidayRange())) {
						// 全休
						for (StockHolidayDataDtoInterface stockHolidayDataDto : list) {
							if (stockHolidayDataDto.getHoldDay() + stockHolidayDataDto.getGivingDay()
									- stockHolidayDataDto.getCancelDay() - stockHolidayDataDto.getUseDay() >= 1) {
								acquisitionDate = stockHolidayDataDto.getAcquisitionDate();
								break;
							}
						}
						useDay = 1;
					} else if (TimeConst.CODE_HOLIDAY_RANGE_AM == getInt(vo.getPltEditHolidayRange())
							|| TimeConst.CODE_HOLIDAY_RANGE_PM == getInt(vo.getPltEditHolidayRange())) {
						// 半休
						for (StockHolidayDataDtoInterface stockHolidayDataDto : list) {
							if (stockHolidayDataDto.getHoldDay() + stockHolidayDataDto.getGivingDay()
									- stockHolidayDataDto.getCancelDay() - stockHolidayDataDto.getUseDay() >= TimeConst.HOLIDAY_TIMES_HALF) {
								acquisitionDate = stockHolidayDataDto.getAcquisitionDate();
								break;
							}
						}
						useDay = TimeConst.HOLIDAY_TIMES_HALF;
					} else {
						addHolidayNumDaysExcessErrorMessage();
						return;
					}
					if (acquisitionDate == null) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						addHolidayNumDaysExcessErrorMessage(mospParams.getName("Stock")
								+ mospParams.getName("Vacation"));
						
						return;
					}
					// DTOの準備
					HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
					if (dto == null) {
						dto = regist.getInitDto();
					}
					if (targetDate.compareTo(startDate) != 0) {
						dto = regist.getInitDto();
					}
					// DTOに値をセット
					setDtoFields(dto, targetDate, targetDate, acquisitionDate, useDay, 0);
					if (targetDate.compareTo(startDate) != 0) {
						dto.setTmdHolidayRequestId(0);
					}
					// 妥当性チェック
					regist.validate(dto);
					if (mospParams.hasErrorMessage()) {
						return;
					}
					// 申請日チェック
					regist.checkRequestDate(dto);
					if (mospParams.hasErrorMessage()) {
						mospParams.getErrorMessageList().clear();
						// 1日加算
						targetDate = DateUtility.addDay(targetDate, 1);
						continue;
					}
					// 申請の相関チェック
					regist.checkDraft(dto);
					// ワークフローの設定
					WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
					if (workflowDto == null) {
						workflowDto = workflowRegist.getInitDto();
						workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
					}
					workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
					// 登録後ワークフローの取得
					workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
							PlatformConst.WORKFLOW_TYPE_TIME);
					if (workflowDto != null) {
						// ワークフローコメント登録
						platform().workflowCommentRegist().addComment(
								workflowDto,
								mospParams.getUser().getPersonalId(),
								mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
										new String[]{ mospParams.getName("WorkPaper") }));
						// ワークフロー番号セット
						dto.setWorkflow(workflowDto.getWorkflow());
						// 登録
						regist.regist(dto);
					}
					// 登録結果確認
					if (mospParams.hasErrorMessage()) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					// 1日加算
					targetDate = DateUtility.addDay(targetDate, 1);
				}
			} else {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (isSpecialHoliday) {
			// 休暇種別が特別休暇の場合
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(vo.getPltEditStatusSpecial(),
					startDate, Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				// 例外メッセージ
				String errorMes = mospParams.getName("Vacation") + mospParams.getName("Classification");
				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, errorMes, null);
				return;
			}
			HolidayDataDtoInterface holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(
					vo.getPersonalId(), startDate, vo.getPltEditStatusSpecial(),
					Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDataDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
				return;
			}
			double useDay = getHolidayDays(holidayDto, holidayDataDto);
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, holidayDataDto.getActivateDate(), useDay, 0);
			// 妥当性チェック
			regist.validate(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			if (workflowDto != null) {
				// ワークフローコメント登録
				platform().workflowCommentRegist().addComment(
						workflowDto,
						mospParams.getUser().getPersonalId(),
						mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
								new String[]{ mospParams.getName("WorkPaper") }));
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (isOtherHoliday) {
			// 休暇種別がその他休暇の場合
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(vo.getPltEditSpecialOther(),
					startDate, Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				// 例外メッセージ
				String errorMes = mospParams.getName("Vacation") + mospParams.getName("Classification");
				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, errorMes, null);
				return;
			}
			HolidayDataDtoInterface holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(
					vo.getPersonalId(), startDate, vo.getPltEditSpecialOther(),
					Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDataDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
				return;
			}
			double useDay = getHolidayDays(holidayDto, holidayDataDto);
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, holidayDataDto.getActivateDate(), useDay, 0);
			// 妥当性チェック
			regist.validate(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			if (workflowDto != null) {
				// ワークフローコメント登録
				platform().workflowCommentRegist().addComment(
						workflowDto,
						mospParams.getUser().getPersonalId(),
						mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
								new String[]{ mospParams.getName("WorkPaper") }));
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (isAbsence) {
			// 休暇種別が欠勤の場合
			int count = 0;
			Date targetDate = startDate;
			while (!targetDate.after(endDate)) {
				if (mospParams.hasErrorMessage()) {
					return;
				}
				HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
				if (dto == null) {
					dto = regist.getInitDto();
				}
				dto.setPersonalId(vo.getPersonalId());
				dto.setRequestStartDate(targetDate);
				dto.setRequestEndDate(targetDate);
				regist.checkRequestDate(dto);
				if (mospParams.hasErrorMessage()) {
					mospParams.getErrorMessageList().clear();
					// 1日加算
					targetDate = DateUtility.addDay(targetDate, 1);
					continue;
				}
				count++;
				// 1日加算
				targetDate = DateUtility.addDay(targetDate, 1);
			}
			double useDay = 0;
			if (TimeConst.CODE_HOLIDAY_RANGE_ALL == getInt(vo.getPltEditHolidayRange())) {
				useDay = count;
			} else if (TimeConst.CODE_HOLIDAY_RANGE_AM == getInt(vo.getPltEditHolidayRange())
					|| TimeConst.CODE_HOLIDAY_RANGE_PM == getInt(vo.getPltEditHolidayRange())) {
				useDay = count * TimeConst.HOLIDAY_TIMES_HALF;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, DateUtility.getSystemDate(), useDay, 0);
			// 妥当性チェック
			regist.validate(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			if (workflowDto != null) {
				// ワークフローコメント登録
				platform().workflowCommentRegist().addComment(
						workflowDto,
						mospParams.getUser().getPersonalId(),
						mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
								new String[]{ mospParams.getName("WorkPaper") }));
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 下書成功メッセージ設定
		addDraftMessage();
		// 検索
		search();
		
		String holidayType2 = "";
		if (isPaidHoliday) {
			// 有給休暇の場合
			holidayType2 = vo.getPltEditStatusWithPay();
		} else if (isSpecialHoliday) {
			// 特別休暇の場合
			holidayType2 = vo.getPltEditStatusSpecial();
		} else if (isOtherHoliday) {
			// その他休暇の場合
			holidayType2 = vo.getPltEditSpecialOther();
		} else if (isAbsence) {
			// 欠勤の場合
			holidayType2 = vo.getPltEditSpecialAbsence();
		}
		String range = vo.getPltEditHolidayRange();
		if (TimeConst.CODE_HOLIDAYTYPE_HOLIDAY == getInt(vo.getPltEditHolidayType1())
				&& TimeConst.CODE_HOLIDAYTYPE_HOLIDAY == getInt(vo.getPltEditStatusWithPay())) {
			// 有給休暇の場合
			range = vo.getPltEditHolidayRangePaidHoliday();
		}
		// 履歴編集対象を取得
		setEditUpdateMode(startDate, vo.getPltEditHolidayType1(), holidayType2, range, DateUtility
			.getStringTime(DateUtility.getTime(vo.getPltEditStartHour(), vo.getPltEditStartMinute())));
		// 編集モード設定
		vo.setModeCardEdit(TimeConst.MODE_APPLICATION_DRAFT);
	}
	
	/**
	 * 申請処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void appli() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 登録クラス取得
		HolidayRequestRegistBeanInterface regist = time().holidayRequestRegist();
		WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
		Date startDate = getEditStartDate();
		Date endDate = getEditEndDate();
		// 休暇種別が有給休暇又はストック休暇の場合
		if (vo.getPltEditHolidayType1().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_HOLIDAY))) {
			Date targetDate = startDate;
			if (vo.getPltEditStatusWithPay().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_HOLIDAY))) {
				// 休暇種別が有給休暇の場合
				while (!targetDate.after(endDate)) {
					Date acquisitionDate = null;
					double useDay = 0;
					int useHour = 0;
					Map<String, Object> map = timeReference().paidHolidayInfo().getPaidHolidayPossibleRequest(
							vo.getPersonalId(), startDate);
					if (TimeConst.CODE_HOLIDAY_RANGE_ALL == getInt(vo.getPltEditHolidayRangePaidHoliday())) {
						// 全休
						if (((Double)map.get(TimeConst.CODE_FORMER_YEAR_DATE)).doubleValue() >= 1) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Double)map.get(TimeConst.CODE_CURRENT_YEAR_DATE)).doubleValue() >= 1) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = 1;
						useHour = 0;
					} else if (TimeConst.CODE_HOLIDAY_RANGE_AM == getInt(vo.getPltEditHolidayRangePaidHoliday())
							|| TimeConst.CODE_HOLIDAY_RANGE_PM == getInt(vo.getPltEditHolidayRangePaidHoliday())) {
						// 半休
						if (((Double)map.get(TimeConst.CODE_FORMER_YEAR_DATE)).doubleValue() >= TimeConst.HOLIDAY_TIMES_HALF) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Double)map.get(TimeConst.CODE_CURRENT_YEAR_DATE)).doubleValue() >= TimeConst.HOLIDAY_TIMES_HALF) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = TimeConst.HOLIDAY_TIMES_HALF;
						useHour = 0;
					} else if (TimeConst.CODE_HOLIDAY_RANGE_TIME == getInt(vo.getPltEditHolidayRangePaidHoliday())) {
						// 時休
						Date startTime = DateUtility.getTime(vo.getPltEditStartHour(), vo.getPltEditStartMinute());
						Date endTime = DateUtility.getTime(vo.getPltEditEndHour(), vo.getPltEditEndMinute());
						int hour = (int)((endTime.getTime() - startTime.getTime()) / (1000 * TimeConst.CODE_DEFINITION_HOUR * TimeConst.CODE_DEFINITION_HOUR));
						if (((Integer)map.get(TimeConst.CODE_FORMER_YEAR_TIME)).intValue() >= hour) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_FORMER_ACQUISITION_DATE_DATE);
						} else if (((Integer)map.get(TimeConst.CODE_CURRENT_TIME)).intValue() >= hour) {
							acquisitionDate = (Date)map.get(TimeConst.CODE_CURRENT_ACQUISITION_DATE_DATE);
						} else {
							addHolidayNumDaysExcessErrorMessage();
							return;
						}
						useDay = 0;
						useHour = hour;
					} else {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						// 例外メッセージ
						mospParams.addErrorMessage(TimeMessageConst.MSG_RANGE_SELECT, null);
						return;
					}
					// DTOの準備
					HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
					if (dto == null) {
						dto = regist.getInitDto();
					}
					if (targetDate.compareTo(startDate) != 0) {
						dto = regist.getInitDto();
					}
					// DTOに値をセット
					setDtoFields(dto, targetDate, targetDate, acquisitionDate, useDay, useHour);
					if (targetDate.compareTo(startDate) != 0) {
						dto.setTmdHolidayRequestId(0);
					}
					if (mospParams.hasErrorMessage()) {
						return;
					}
					// 申請日チェック
					regist.checkRequestDate(dto);
					if (mospParams.hasErrorMessage()) {
						mospParams.getErrorMessageList().clear();
						// 1日加算
						targetDate = DateUtility.addDay(targetDate, 1);
						continue;
					}
					// 申請の相関チェック
					regist.checkAppli(dto);
					// ワークフローの設定
					WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
					if (workflowDto == null) {
						workflowDto = workflowRegist.getInitDto();
						workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
					}
					workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
					// 登録後ワークフローの取得
					workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
							PlatformConst.WORKFLOW_TYPE_TIME, null);
					if (workflowDto != null) {
						// ワークフロー番号セット
						dto.setWorkflow(workflowDto.getWorkflow());
						// 登録
						regist.regist(dto);
						// 勤怠データ削除
						deleteAttendance(dto);
					}
					// 登録結果確認
					if (mospParams.hasErrorMessage()) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					// 1日加算
					targetDate = DateUtility.addDay(targetDate, 1);
				}
			} else if (vo.getPltEditStatusWithPay().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_STOCK))) {
				
				// 休暇種別がストック休暇の場合
				while (!targetDate.after(endDate)) {
					// 取得日
					Date acquisitionDate = null;
					// 休暇日数(全休・半休)
					double useDay = 0;
					// ストック休暇情報取得
					List<StockHolidayDataDtoInterface> list = timeReference().stockHolidayInfo()
						.getStockHolidayPossibleRequest(vo.getPersonalId(), startDate);
					// ストック情報が存在しない場合
					if (list == null || list.isEmpty()) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						addNotExistHolidayInfoErrorMessage();
						return;
					}
					
					if (TimeConst.CODE_HOLIDAY_RANGE_ALL == getInt(vo.getPltEditHolidayRange())) {
						// 全休
						for (StockHolidayDataDtoInterface stockHolidayDataDto : list) {
							//  付与日数 - 廃棄日数 - 使用日数
							if (stockHolidayDataDto.getHoldDay() + stockHolidayDataDto.getGivingDay()
									- stockHolidayDataDto.getCancelDay() - stockHolidayDataDto.getUseDay() >= 1) {
								// 日数が存在する場合、取得日を取得する
								acquisitionDate = stockHolidayDataDto.getAcquisitionDate();
								break;
							}
						}
						useDay = 1;
					} else if (TimeConst.CODE_HOLIDAY_RANGE_AM == getInt(vo.getPltEditHolidayRange())
							|| TimeConst.CODE_HOLIDAY_RANGE_PM == getInt(vo.getPltEditHolidayRange())) {
						// 半休
						for (StockHolidayDataDtoInterface stockHolidayDataDto : list) {
							//  付与日数 - 廃棄日数 - 使用日数
							if (stockHolidayDataDto.getHoldDay() + stockHolidayDataDto.getGivingDay()
									- stockHolidayDataDto.getCancelDay() - stockHolidayDataDto.getUseDay() >= TimeConst.HOLIDAY_TIMES_HALF) {
								// 日数が存在する場合、取得日を取得する
								acquisitionDate = stockHolidayDataDto.getAcquisitionDate();
								
								break;
							}
						}
						useDay = TimeConst.HOLIDAY_TIMES_HALF;
					} else {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					// 取得日が設定できない場合はエラー
					if (acquisitionDate == null) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						// 登録メッセージ
						addHolidayNumDaysExcessErrorMessage(mospParams.getName("Stock")
								+ mospParams.getName("Vacation"));
						return;
					}
					// DTOの準備
					HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
					if (dto == null) {
						dto = regist.getInitDto();
					}
					if (targetDate.compareTo(startDate) != 0) {
						dto = regist.getInitDto();
					}
					// DTOに値をセット
					setDtoFields(dto, targetDate, targetDate, acquisitionDate, useDay, 0);
					if (targetDate.compareTo(startDate) != 0) {
						dto.setTmdHolidayRequestId(0);
					}
					if (mospParams.hasErrorMessage()) {
						return;
					}
					// 申請日整合性チェック
					regist.checkRequestDate(dto);
					if (mospParams.hasErrorMessage()) {
						mospParams.getErrorMessageList().clear();
						// 1日加算
						targetDate = DateUtility.addDay(targetDate, 1);
						continue;
					}
					// 申請の相関チェック
					regist.checkAppli(dto);
					// ワークフローの設定
					WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
					if (workflowDto == null) {
						workflowDto = workflowRegist.getInitDto();
						workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
					}
					workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
					// 登録後ワークフローの取得
					workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
							PlatformConst.WORKFLOW_TYPE_TIME, null);
					if (workflowDto != null) {
						// ワークフロー番号セット
						dto.setWorkflow(workflowDto.getWorkflow());
						// 登録
						regist.regist(dto);
						// 勤怠データ削除
						deleteAttendance(dto);
					}
					// 登録結果確認
					if (mospParams.hasErrorMessage()) {
						// 登録失敗メッセージ設定
						addInsertFailedMessage();
						return;
					}
					targetDate = DateUtility.addDay(targetDate, 1);
				}
			} else {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (vo.getPltEditHolidayType1().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_SPECIAL))) {
			// 休暇種別が特別休暇の場合
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(vo.getPltEditStatusSpecial(),
					startDate, Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				// 例外メッセージ
				// 休暇種別が存在しません。
				String errorMes = mospParams.getName("Vacation") + mospParams.getName("Classification");
				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, errorMes, null);
				return;
			}
			HolidayDataDtoInterface holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(
					vo.getPersonalId(), startDate, vo.getPltEditStatusSpecial(),
					Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDataDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
				return;
			}
			double useDay = getHolidayDays(holidayDto, holidayDataDto);
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, holidayDataDto.getActivateDate(), useDay, 0);
			// 申請の相関チェック
			regist.checkAppli(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME, null);
			if (workflowDto != null) {
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
				// 勤怠データ削除
				deleteAttendance(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (vo.getPltEditHolidayType1().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_OTHER))) {
			// 休暇種別がその他休暇の場合
			HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(vo.getPltEditSpecialOther(),
					startDate, Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				// 例外メッセージ
				String errorMes = mospParams.getName("Vacation") + mospParams.getName("Classification");
				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, errorMes, null);
				return;
			}
			HolidayDataDtoInterface holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(
					vo.getPersonalId(), startDate, vo.getPltEditSpecialOther(),
					Integer.parseInt(vo.getPltEditHolidayType1()));
			if (holidayDataDto == null) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
				return;
			}
			double useDay = getHolidayDays(holidayDto, holidayDataDto);
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, holidayDataDto.getActivateDate(), useDay, 0);
			// 申請の相関チェック
			regist.checkAppli(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME, null);
			if (workflowDto != null) {
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
				// 勤怠データ削除
				deleteAttendance(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else if (vo.getPltEditHolidayType1().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_ABSENCE))) {
			// 休暇種別が欠勤の場合
			int count = 0;
			Date targetDate = startDate;
			while (!targetDate.after(endDate)) {
				if (mospParams.hasErrorMessage()) {
					return;
				}
				HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
				if (dto == null) {
					dto = regist.getInitDto();
				}
				dto.setPersonalId(vo.getPersonalId());
				dto.setRequestStartDate(targetDate);
				dto.setRequestEndDate(targetDate);
				regist.checkRequestDate(dto);
				if (mospParams.hasErrorMessage()) {
					mospParams.getErrorMessageList().clear();
					// 1日加算
					targetDate = DateUtility.addDay(targetDate, 1);
					continue;
				}
				count++;
				// 1日加算
				targetDate = DateUtility.addDay(targetDate, 1);
			}
			double useDay = 0;
			if (TimeConst.CODE_HOLIDAY_RANGE_ALL == getInt(vo.getPltEditHolidayRange())) {
				useDay = count;
			} else if (TimeConst.CODE_HOLIDAY_RANGE_AM == getInt(vo.getPltEditHolidayRange())
					|| TimeConst.CODE_HOLIDAY_RANGE_PM == getInt(vo.getPltEditHolidayRange())) {
				useDay = count * TimeConst.HOLIDAY_TIMES_HALF;
			}
			// DTOの準備
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値をセット
			setDtoFields(dto, startDate, endDate, DateUtility.getSystemDate(), useDay, 0);
			// 申請の相関チェック
			regist.checkAppli(dto);
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_VACATION);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME, null);
			if (workflowDto != null) {
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
				// 勤怠データ削除
				deleteAttendance(dto);
			}
			// 登録結果確認
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
		} else {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 申請成功メッセージ設定
		addAppliMessage();
		// 登録結果確認
		if (!mospParams.hasErrorMessage()) {
			// 登録が成功した場合、初期状態に戻す。
			insertMode();
			search();
		}
	}
	
	/**
	* 取下処理を行う。<br>
	* @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	*/
	protected void withdrawn() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		HolidayRequestRegistBeanInterface regist = time().holidayRequestRegist();
		WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
		WorkflowCommentRegistBeanInterface workflowCommentRegist = platform().workflowCommentRegist();
		// DTOの準備
		HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
		// 存在確認
		checkSelectedDataExist(dto);
		// 取下の相関チェック
		regist.checkWithdrawn(dto);
		// ワークフロー取得
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		// 存在確認
		checkSelectedDataExist(workflowDto);
		boolean isDraft = PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus());
		if (isDraft) {
			// 下書の場合は削除する
			regist.delete(dto);
			workflowRegist.delete(workflowDto);
			workflowCommentRegist.deleteList(reference().workflowComment().getWorkflowCommentList(
					workflowDto.getWorkflow()));
		} else {
			// 下書でない場合は取下する
			// ワークフロー登録
			workflowDto = workflowRegist.withdrawn(workflowDto);
			if (workflowDto != null) {
				// ワークフローコメント登録
				workflowCommentRegist.addComment(workflowDto, mospParams.getUser().getPersonalId(), mospParams
					.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
							new String[]{ mospParams.getName("TakeDown") }));
			}
		}
		// 削除結果確認
		if (mospParams.hasErrorMessage()) {
			// 削除失敗メッセージ設定
			addDeleteHistoryFailedMessage();
			return;
		}
		// コミット
		commit();
		if (isDraft) {
			// 削除成功メッセージ設定
			addDeleteMessage();
		} else {
			// 取下成功メッセージ設定
			addTakeDownMessage();
		}
		// 新規登録モード設定(編集領域をリセット)
		insertMode();
		// 検索
		search();
	}
	
	/**
	 * 一覧のソート処理を行う。<br>
	 * @throws MospException 比較クラスのインスタンス生成に失敗した場合
	 */
	protected void sort() throws MospException {
		setVoList(sortList(getTransferredSortKey()));
	}
	
	/**
	 * 一覧のページ処理を行う。
	 * @throws MospException 例外発生時
	 */
	protected void page() throws MospException {
		setVoList(pageList());
	}
	
	/**
	 * プルダウン設定
	 * @throws MospException 例外発生時
	 */
	private void setPulldown() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 有効日取得
		Date date = getDate(vo.getPltEditStartYear(), vo.getPltEditStartMonth(), vo.getPltEditStartDay());
		int editRequestYear = DateUtility.getYear(date);
		// 編集項目設定
		vo.setAryPltEditStartYear(getYearArray(editRequestYear));
		vo.setAryPltEditStartMonth(getMonthArray());
		// 31日まで取得
		vo.setAryPltEditStartDay(getDayArray());
		vo.setAryPltEditEndYear(getYearArray(editRequestYear));
		vo.setAryPltEditEndMonth(getMonthArray());
		// 31日まで取得
		vo.setAryPltEditEndDay(getDayArray());
		vo.setAryPltEditHolidayType1(mospParams.getProperties().getCodeArray(TimeConst.CODE_HOLIDAY_TYPE, false));
		// 休暇種別2(有給休暇)プルダウン設定
		vo.setAryPltEditHolidayType2WithPay(mospParams.getProperties().getCodeArray(
				TimeConst.CODE_HOLIDAY_TYPE2_WITHPAY, false));
		int holidayType3Range1Flg = 0;
		vo.setAryPltEditHolidayRangePaidHoliday(mospParams.getProperties().getCodeArray(
				TimeConst.CODE_HOLIDAY_TYPE3_RANGE2, false));
		ApplicationDtoInterface appDto = timeReference().application().findForPerson(vo.getPersonalId(), date);
		if (appDto != null) {
			PaidHolidayDtoInterface paidHolidayDto = timeReference().paidHoliday().getPaidHolidayInfo(
					appDto.getPaidHolidayCode(), date);
			if (paidHolidayDto != null && paidHolidayDto.getTimelyPaidHolidayFlag() == 0) {
				// 時休が有効の場合
				vo.setAryPltEditHolidayRangePaidHoliday(mospParams.getProperties().getCodeArray(
						TimeConst.CODE_HOLIDAY_TYPE3_RANGE1, false));
				holidayType3Range1Flg = 1;
			}
		}
		vo.setAryPltEditHolidayRange(mospParams.getProperties()
			.getCodeArray(TimeConst.CODE_HOLIDAY_TYPE3_RANGE2, false));
		vo.setAryPltEditStartHour(getHourArray());
		// 分は15分単位
		vo.setAryPltEditStartMinute(getMinuteArray(15));
		vo.setAryPltEditEndHour(getHourArray());
		// 分は15分単位
		vo.setAryPltEditEndMinute(getMinuteArray(15));
		
		// 検索項目設定
		vo.setAryPltSearchYear(getYearArray(editRequestYear));
		vo.setAryPltSearchMonth(getMonthArray());
		// 検索欄 休暇種別
		vo.setAryPltSearchHolidayType1(mospParams.getProperties().getCodeArray(TimeConst.CODE_HOLIDAY_TYPE, true));
		vo.setAryPltSearchHolidayType2WithPay(mospParams.getProperties().getCodeArray(
				TimeConst.CODE_HOLIDAY_TYPE2_WITHPAY, true));
		// 時休有効無効によってaryPltSearchHolidayType3Range1の内容変更
		vo.setAryPltSearchHolidayRangePaidHoliday(mospParams.getProperties().getCodeArray(
				TimeConst.CODE_HOLIDAY_TYPE3_RANGE2, true));
		if (holidayType3Range1Flg == 1) {
			// 時休が有効の場合
			vo.setAryPltSearchHolidayRangePaidHoliday(mospParams.getProperties().getCodeArray(
					TimeConst.CODE_HOLIDAY_TYPE3_RANGE1, true));
		}
		vo.setAryPltSearchHolidayRange(mospParams.getProperties().getCodeArray(TimeConst.CODE_HOLIDAY_TYPE3_RANGE2,
				true));
		// 検索欄 承認状況
		vo.setAryPltSearchState(mospParams.getProperties().getCodeArray(TimeConst.CODE_APPROVAL_STATE, true));
	}
	
	/**
	 * 勤務終了予定時刻プルダウン設定
	 * @throws MospException 例外処理が発生した場合
	 */
	private void setHourPulldown() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		ApplicationReferenceBeanInterface application = timeReference().application();
		TimeSettingReferenceBeanInterface timeSetting = timeReference().timeSetting();
		PaidHolidayReferenceBeanInterface paidHoliday = timeReference().paidHoliday();
		vo.setAryPltEditStartHour(getHourArray(48));
		vo.setAryPltEditEndHour(getHourArray(48));
		Date requestDate = getEditStartDate();
		if (requestDate == null) {
			return;
		}
		// 適用されている設定を取得
		ApplicationDtoInterface applicationDto = application.findForPerson(vo.getPersonalId(), requestDate);
		// 設定適用マスタの存在
		application.chkExistApplication(applicationDto, requestDate);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 勤怠マスタ取得
		TimeSettingDtoInterface timeSettingDto = timeSetting.getTimeSettingInfo(applicationDto.getWorkSettingCode(),
				requestDate);
		// 勤怠マスタの存在チェック
		timeSetting.chkExistTimeSetting(timeSettingDto, requestDate);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		PaidHolidayDtoInterface paidHolidayDto = paidHoliday.getPaidHolidayInfo(applicationDto.getPaidHolidayCode(),
				requestDate);
		paidHoliday.chkExistPaidHoliday(paidHolidayDto, requestDate);
		int endDayTime = DateUtility.getHour(timeSettingDto.getStartDayTime()) + 24;
		vo.setAryPltEditStartHour(getHourArray(endDayTime - paidHolidayDto.getTimelyPaidHolidayTime()));
		vo.setAryPltEditEndHour(getHourArray(endDayTime));
	}
	
	/**
	 * 有効日(編集)設定処理を行う。<br>
	 * 保持有効日モードを確認し、モード及びプルダウンの再設定を行う。<br>
	 * @throws MospException プルダウンの取得に失敗した場合
	 */
	protected void setEditActivationDate() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 現在の有効日モードを確認
		if (vo.getModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_CHANGING)) {
			// 開始年月日と終了年月日の比較
			if (chkActivationDateValidate()) {
				return;
			}
			Date targetDate = getEditStartDate();
			while (!targetDate.after(getEditEndDate())) {
				// 退職済み確認
				if (reference().retirement().isRetired(vo.getPersonalId(), targetDate)) {
					addEmployeeRetiredMessage();
					return;
				}
				// 休職期間確認
				if (reference().suspension().isSuspended(vo.getPersonalId(), targetDate)) {
					addEmployeeSuspendedMessage();
					return;
				}
				targetDate = DateUtility.addDay(targetDate, 1);
			}
			if (setApprvalPullDown(vo.getPersonalId(), getEditStartDate()) == false) {
				return;
			}
			HolidayRequestRegistBeanInterface regist = time().holidayRequestRegist();
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			dto.setPersonalId(vo.getPersonalId());
			dto.setRequestStartDate(getEditStartDate());
			dto.setRequestEndDate(getEditStartDate());
//			dto.setHolidayRange(1);
			regist.checkTemporaryClosingFinal(dto);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			regist.checkRequestDate(dto);
			regist.checkRequest(dto);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			if (!getEditStartDate().equals(getEditEndDate())) {
				// 休暇開始日と休暇終了日が異なる場合は休暇終了日でもチェック
				dto.setRequestStartDate(getEditEndDate());
				dto.setRequestEndDate(getEditEndDate());
				regist.checkRequestDate(dto);
				regist.checkRequest(dto);
			}
			if (mospParams.hasErrorMessage()) {
				return;
			}
			if (!getEditStartDate().equals(getEditEndDate())) {
				// 休暇開始日と休暇終了日が異なる場合は全休をセット
				vo.setPltEditHolidayRange(String.valueOf(TimeConst.CODE_HOLIDAY_RANGE_ALL));
				vo.setPltEditHolidayRangePaidHoliday(String.valueOf(TimeConst.CODE_HOLIDAY_RANGE_ALL));
			}
			// 連続取得設定
			setHolidayContinue();
			// 有効日モード設定
			vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
		} else {
			String[] aryPltLblApproverSetting = new String[0];
			vo.setAryPltLblApproverSetting(aryPltLblApproverSetting);
			// 有効日モード設定
			vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		}
		setHourPulldown();
		setEditPulldown();
	}
	
	/**
	 * プルダウン(編集)の設定を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setEditPulldown() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 有効日フラグ確認
		if (vo.getModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_FIXED)) {
			// プルダウン設定
			vo.setAryPltEditHolidayType1(mospParams.getProperties().getCodeArray(TimeConst.CODE_HOLIDAY_TYPE, false));
			vo.setAryPltEditHolidayType2Special(timeReference().holiday().getSelectArray(getEditStartDate(),
					TimeConst.CODE_HOLIDAYTYPE_SPECIAL, false));
			vo.setAryPltEditHolidayType2Other(timeReference().holiday().getSelectArray(getEditStartDate(),
					TimeConst.CODE_HOLIDAYTYPE_OTHER, false));
			vo.setAryPltEditHolidayType2Absence(timeReference().holiday().getSelectArray(getEditStartDate(),
					TimeConst.CODE_HOLIDAYTYPE_ABSENCE, false));
			// プルダウンデータの存在チェック
			boolean specialDeleteFlag = mospParams.getName("NoObjectData").equals(
					vo.getAryPltEditHolidayType2Special()[0][1]);
			boolean otherDeleteFlag = mospParams.getName("NoObjectData").equals(
					vo.getAryPltEditHolidayType2Other()[0][1]);
			boolean absenceDeleteFlag = mospParams.getName("NoObjectData").equals(
					vo.getAryPltEditHolidayType2Absence()[0][1]);
			String[][] holidayArray = vo.getAryPltEditHolidayType1();
			String[][] newHolidayArray = holidayArray;
			int minus = 0;
			if (specialDeleteFlag) {
				minus++;
			}
			if (otherDeleteFlag) {
				minus++;
			}
			if (absenceDeleteFlag) {
				minus++;
			}
			if (minus > 0) {
				newHolidayArray = new String[holidayArray.length - minus][2];
				int i = 0;
				for (String[] holiday : holidayArray) {
					if (specialDeleteFlag) {
						if ("2".equals(holiday[0])) {
							continue;
						}
					}
					if (otherDeleteFlag) {
						if ("3".equals(holiday[0])) {
							continue;
						}
					}
					if (absenceDeleteFlag) {
						if ("4".equals(holiday[0])) {
							continue;
						}
					}
					newHolidayArray[i][0] = holiday[0];
					newHolidayArray[i][1] = holiday[1];
					i++;
				}
			}
			vo.setAryPltEditHolidayType1(newHolidayArray);
			Date startDate = getEditStartDate();
			Date endDate = getEditEndDate();
			if (startDate.compareTo(endDate) == 0) {
				// 同日付に時差出勤が申請されているか確認する。
				getDifferenceRequest1(vo.getPersonalId(), startDate);
			}
		}
	}
	
	/**
	 * 有効日(編集)設定処理を行う。<br>
	 * 保持有効日モードを確認し、モード及びプルダウンの再設定を行う。<br>
	 * @throws MospException プルダウンの取得に失敗した場合
	 */
	protected void setSearchActivationDate() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 現在の有効日モードを確認
		if (vo.getJsSearchModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_CHANGING)) {
			// 有効日モード設定
			vo.setJsSearchModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
		} else {
			// 有効日モード設定
			vo.setJsSearchModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		}
		setSearchPulldown();
	}
	
	/**
	 * 休暇連続取得設定を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void setHolidayContinue() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		HolidayRequestRegistBeanInterface regist = time().holidayRequestRegist();
		// 初期化
		vo.setJsHolidayTerm(Integer.toString(1));
		vo.setJsHolidayContinue(Integer.toString(2));
		vo.setJsHolidayRemainDay("");
		// 有給或いは欠勤の場合
		if (TimeConst.CODE_HOLIDAYTYPE_HOLIDAY == getInt(vo.getPltEditHolidayType1())
				|| TimeConst.CODE_HOLIDAYTYPE_ABSENCE == getInt(vo.getPltEditHolidayType1())) {
			return;
		}
		// 休暇申請の期間取得
		Date startDate = getEditStartDate();
		Date endDate = getEditEndDate();
		// 休暇の期間を取得
		int count = 0;
		Date targetDate = startDate;
		// 終了日が開始日より早くてはだめ
		while (!targetDate.after(endDate)) {
			if (mospParams.hasErrorMessage()) {
				return;
			}
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			dto.setPersonalId(vo.getPersonalId());
			dto.setRequestStartDate(targetDate);
			dto.setRequestEndDate(targetDate);
			regist.checkRequestDate(dto);
			if (mospParams.hasErrorMessage()) {
				mospParams.getErrorMessageList().clear();
				// 1日加算
				targetDate = DateUtility.addDay(targetDate, 1);
				continue;
			}
			count++;
			// 1日加算
			targetDate = DateUtility.addDay(targetDate, 1);
		}
		// 休暇期間を設定
		vo.setJsHolidayTerm(Integer.toString(count));
		// 休暇情報取得及び設定
		String holidayCode = "";
		HolidayDataDtoInterface holidayDataDto = null;
		if (TimeConst.CODE_HOLIDAYTYPE_SPECIAL == getInt(vo.getPltEditHolidayType1())) {
			holidayCode = vo.getPltEditStatusSpecial();
			holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(vo.getPersonalId(),
					getEditStartDate(), vo.getPltEditStatusSpecial(), Integer.parseInt(vo.getPltEditHolidayType1()));
		} else if (TimeConst.CODE_HOLIDAYTYPE_OTHER == getInt(vo.getPltEditHolidayType1())) {
			holidayCode = vo.getPltEditSpecialOther();
			holidayDataDto = timeReference().holidayInfo().getHolidayPossibleRequest(vo.getPersonalId(),
					getEditStartDate(), vo.getPltEditSpecialOther(), Integer.parseInt(vo.getPltEditHolidayType1()));
		}
		if (holidayDataDto == null) {
			vo.setJsHolidayRemainDay("0");
			return;
		}
		HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(holidayCode, getEditStartDate(),
				Integer.parseInt(vo.getPltEditHolidayType1()));
		if (holidayDto == null) {
			vo.setJsHolidayRemainDay("0");
			return;
		}
		vo.setJsHolidayContinue(Integer.toString(holidayDto.getContinuousAcquisition()));
		vo.setJsHolidayRemainDay(Double.toString(holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay()));
	}
	
	/**
	 * 有効日(編集)設定処理を行う。<br>
	 * 保持有効日モードを確認し、モード及びプルダウンの再設定を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void selectActivationDate() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		select();
		String transferredHolidayType1 = getTransferredHolidayType1();
		if (transferredHolidayType1 != null) {
			vo.setPltEditHolidayType1(transferredHolidayType1);
		}
		boolean isPaid = TimeConst.CODE_HOLIDAYTYPE_HOLIDAY == getInt(vo.getPltEditHolidayType1());
		boolean isSpecial = TimeConst.CODE_HOLIDAYTYPE_SPECIAL == getInt(vo.getPltEditHolidayType1());
		boolean isOther = TimeConst.CODE_HOLIDAYTYPE_OTHER == getInt(vo.getPltEditHolidayType1());
		boolean isAbsence = TimeConst.CODE_HOLIDAYTYPE_ABSENCE == getInt(vo.getPltEditHolidayType1());
		String transferredHolidayType2 = getTransferredHolidayType2();
		if (transferredHolidayType2 != null) {
			if (isPaid) {
				// 有給休暇
				vo.setPltEditStatusWithPay(transferredHolidayType2);
			} else if (isSpecial) {
				// 特別休暇
				vo.setPltEditStatusSpecial(transferredHolidayType2);
			} else if (isOther) {
				// その他休暇
				vo.setPltEditSpecialOther(transferredHolidayType2);
			} else if (isAbsence) {
				// 欠勤
				vo.setPltEditSpecialAbsence(transferredHolidayType2);
			}
		}
		String transferredHolidayRange = getTransferredHolidayRange();
		if (transferredHolidayRange != null) {
			if (isPaid) {
				// 有給休暇
				vo.setPltEditHolidayRangePaidHoliday(transferredHolidayRange);
			} else if (isSpecial || isOther || isAbsence) {
				// 特別休暇・その他休暇・欠勤
				vo.setPltEditHolidayRange(transferredHolidayRange);
			}
		}
		setEditActivationDate();
	}
	
	/**
	 * プルダウン(編集)の設定を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setSearchPulldown() throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 有効日フラグ確認
		if (vo.getJsSearchModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_FIXED)) {
			// プルダウン設定
			Date date;
			if (vo.getPltSearchMonth().isEmpty()) {
				date = getDate(vo.getPltSearchYear(), "1", "1");
			} else {
				date = getDate(vo.getPltSearchYear(), vo.getPltSearchMonth(), "1");
			}
			vo.setAryPltSearchHolidayType2Special(timeReference().holiday().getSelectArray(date,
					TimeConst.CODE_HOLIDAYTYPE_SPECIAL, true));
			vo.setAryPltSearchHolidayType2Other(timeReference().holiday().getSelectArray(date,
					TimeConst.CODE_HOLIDAYTYPE_OTHER, true));
			vo.setAryPltSearchHolidayType2Absence(timeReference().holiday().getSelectArray(date,
					TimeConst.CODE_HOLIDAYTYPE_ABSENCE, true));
		}
	}
	
	/**
	 * 新規登録モードに設定する。<br>
	 * @throws MospException 例外発生時
	 */
	protected void insertMode() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// プルダウン設定
		setPulldown();
		// 状態
		vo.setModeCardEdit(TimeConst.MODE_APPLICATION_NEW);
		// 有効日(編集)モード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		// 有効日(検索)モード設定
		vo.setJsSearchModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(HolidayRequestRequestDateComparator.class.getName());
		// 
		vo.setJsModeDifferenceRequest1("");
		//
		setEditPulldown();
		setSearchPulldown();
		// 基本情報チェック
		timeReference().holidayRequest().chkBasicInfo(vo.getPersonalId(), getEditStartDate());
	}
	
	/**
	 * 履歴編集モードで画面を表示する。<br>
	 * 履歴編集対象は、遷移汎用コード及び有効日で取得する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void editMode() throws MospException {
		// リクエストから個人ID及び対象日を取得
		String personalId = getTargetPersonalId();
		// 対象日取得(システム日付)
		Date targetDate = getSystemDate();
		if (personalId != null) {
			// 人事情報をVOに設定
			setEmployeeInfo(personalId, targetDate);
			// ページ繰り設定
			setPageInfo(CMD_PAGE, getListLength());
			// 新規登録モード設定
			insertMode();
		}
		// 遷移汎用コード及び有効日から履歴編集対象を取得し編集モードを設定
		setEditUpdateMode(getDate(getTransferredActivateDate()), getTransferredHolidayType1(),
				getTransferredHolidayType2(), getTransferredHolidayRange(), getTransferredStartTime());
		setEditActivationDate();
	}
	
	/**
	 * 履歴編集モードを設定する。<br>
	 * 申請開始日と休暇種別1と休暇種別2と休暇範囲と時休開始時刻で編集対象情報を取得する。<br>
	 * @param requestStartDate 申請開始日
	 * @param holidayType1 休暇種別1
	 * @param holidayType2 休暇種別2
	 * @param holidayRange 休暇範囲
	 * @param paidTime 休暇時間
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setEditUpdateMode(Date requestStartDate, String holidayType1, String holidayType2,
			String holidayRange, String paidTime) throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// startTimeがHH:MM～HH:MM形式なので開始時間を取得する
		Date startDate = DateUtility.getDateTime(DateUtility.getYear(requestStartDate), DateUtility
			.getMonth(requestStartDate), DateUtility.getDay(requestStartDate), 0, 0);
		if (paidTime != null && !paidTime.isEmpty()) {
			int hourIndex = paidTime.indexOf(":");
			String startHour = paidTime.substring(0, hourIndex);
			String startMinute = paidTime.substring(hourIndex + 1, paidTime.length());
			startDate = DateUtility.getTime(startHour, startMinute);
		}
		// 履歴編集対象取得
		HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKeyOnWorkflow(vo.getPersonalId(),
				requestStartDate, Integer.parseInt(holidayType1), holidayType2, Integer.parseInt(holidayRange),
				startDate);
		// 存在確認
		checkSelectedDataExist(dto);
		// VOにセット
		setVoFields(dto);
		// 有効日(編集)モード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
	}
	
	/**
	 * 一括更新処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void batchUpdate() throws MospException {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// 一括更新処理
		time().holidayRequestRegist().update(getIdArray(vo.getCkbSelect()));
		// 一括更新結果確認
		if (mospParams.hasErrorMessage()) {
			// 更新失敗メッセージ設定
			addBatchUpdateFailedMessage();
			return;
		}
		// コミット
		commit();
		// 更新成功メッセージ設定
		addUpdateMessage();
		// 新規登録モード設定(編集領域をリセット)
		insertMode();
		// 検索
		search();
	}
	
	/**
	 * ワークフロー番号をMosP処理情報に設定し、
	 * 連続実行コマンドを設定する。<br>
	 */
	protected void transfer() {
		// VO準備
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// MosP処理情報に対象ワークフローを設定
		setTargetWorkflow(vo.getAryWorkflow(getTransferredIndex()));
		// 承認履歴画面へ遷移(連続実行コマンド設定)
		mospParams.setNextCommand(ApprovalHistoryAction.CMD_LEAVE_APPROVAL_HISTORY_SELECT_SHOW);
	}
	
	/**
	 * 検索結果リストの内容をVOに設定する。<br>
	 * @param list 対象リスト
	 * @throws MospException 例外発生時
	 */
	protected void setVoList(List<? extends BaseDtoInterface> list) throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// Bean宣言
		HolidayReferenceBeanInterface reference = timeReference().holiday();
		
		// データ配列初期化
		String[] aryCkbRecordId = new String[list.size()];
		String[] aryLblRequestDate = new String[list.size()];
		String[] aryLblHolidayType1 = new String[list.size()];
		String[] aryLblHolidayType2 = new String[list.size()];
		String[] aryLblHolidayType3 = new String[list.size()];
		String[] aryLblHolidayRequestTime = new String[list.size()];
		String[] aryLblRequestReason = new String[list.size()];
		String[] aryLblWorkflowStatus = new String[list.size()];
		String[] aryLblApproverName = new String[list.size()];
		String[] aryLblWorkflow = new String[list.size()];
		String[] aryHolidayType1 = new String[list.size()];
		String[] aryHolidayType2 = new String[list.size()];
		String[] aryHolidayType3 = new String[list.size()];
		String[] aryLblOnOff = new String[list.size()];
		long[] aryWorkflow = new long[list.size()];
		// データ作成
		for (int i = 0; i < list.size(); i++) {
			// リストから情報を取得
			HolidayRequestListDtoInterface dto = (HolidayRequestListDtoInterface)list.get(i);
			// 配列に情報を設定
			aryCkbRecordId[i] = String.valueOf(dto.getTmdHolidayRequestId());
			aryLblRequestDate[i] = DateUtility.getStringDateAndDay(dto.getRequestStartDate())
					+ mospParams.getName("Wave") + DateUtility.getStringDateAndDay(dto.getRequestEndDate());
			aryLblHolidayType1[i] = reference.getHolidayType1NameForHolidayRequest(dto.getHolidayType1(), dto
				.getHolidayType2());
			aryLblHolidayType2[i] = getHolidayType2Abbr(dto.getHolidayType1(), dto.getHolidayType2(), dto
				.getRequestStartDate());
			aryLblHolidayType3[i] = getHolidayType3Name(dto.getHolidayType3());
			aryHolidayType1[i] = String.valueOf(dto.getHolidayType1());
			aryHolidayType2[i] = dto.getHolidayType2();
			aryHolidayType3[i] = String.valueOf(dto.getHolidayType3());
			aryLblHolidayRequestTime[i] = getTimeWaveFormat(dto.getStartTime(), dto.getEndTime());
			aryLblRequestReason[i] = dto.getRequestReason();
			aryLblWorkflowStatus[i] = getStatusStageValueView(dto.getState(), dto.getStage());
			aryLblApproverName[i] = dto.getApproverName();
			aryLblWorkflow[i] = String.valueOf(dto.getWorkflow());
			aryLblOnOff[i] = getButtonOnOff(dto.getState());
			aryWorkflow[i] = dto.getWorkflow();
		}
		// データをVOに設定
		vo.setAryCkbHolidayRequestListId(aryCkbRecordId);
		vo.setAryLblDate(aryLblRequestDate);
		vo.setAryLblHolidayType1(aryLblHolidayType1);
		vo.setAryLblHolidayType2(aryLblHolidayType2);
		vo.setAryLblHolidayType3(aryLblHolidayType3);
		vo.setAryLblHolidayTime(aryLblHolidayRequestTime);
		vo.setAryLblRequestReason(aryLblRequestReason);
		vo.setAryLblState(aryLblWorkflowStatus);
		vo.setAryLblApprover(aryLblApproverName);
		vo.setAryLblWorkflow(aryLblWorkflow);
		vo.setAryHolidayType1(aryHolidayType1);
		vo.setAryHolidayType2(aryHolidayType2);
		vo.setAryHolidayType3(aryHolidayType3);
		vo.setAryLblOnOff(aryLblOnOff);
		vo.setAryWorkflow(aryWorkflow);
	}
	
	/**
	 * @return 休暇開始日を返す
	 */
	protected Date getEditStartDate() {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		return getDate(vo.getPltEditStartYear(), vo.getPltEditStartMonth(), vo.getPltEditStartDay());
	}
	
	/**
	 * @return 休暇終了日を返す
	 */
	protected Date getEditEndDate() {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		return getDate(vo.getPltEditEndYear(), vo.getPltEditEndMonth(), vo.getPltEditEndDay());
	}
	
	/**
	 * @return 休暇開始日から終了日の日にちを返す
	 * @throws MospException 例外発生時
	 */
	protected int getTimeDifference() throws MospException {
		return Integer
			.valueOf((int)((getEditEndDate().getTime() - getEditStartDate().getTime()) / (TimeConst.TIME_DAY_ALL_HOUR
					* TimeConst.CODE_DEFINITION_HOUR * TimeConst.CODE_DEFINITION_HOUR * 1000)));
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param startDate 申請開始日
	 * @param endDate 申請終了日
	 * @param acquisitionDate 取得日
	 * @param useDay 使用日数
	 * @param useHour 使用時間数
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setDtoFields(HolidayRequestDtoInterface dto, Date startDate, Date endDate, Date acquisitionDate,
			double useDay, int useHour) throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// VOの値をDTOに設定
		dto.setTmdHolidayRequestId(vo.getRecordId());
		dto.setPersonalId(vo.getPersonalId());
		dto.setRequestStartDate(startDate);
		dto.setRequestEndDate(endDate);
		dto.setHolidayType1(Integer.parseInt(vo.getPltEditHolidayType1()));
		if (vo.getPltEditHolidayType1().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_HOLIDAY))) {
			dto.setHolidayType2(vo.getPltEditStatusWithPay());
		} else if (vo.getPltEditHolidayType1().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_SPECIAL))) {
			dto.setHolidayType2(vo.getPltEditStatusSpecial());
		} else if (vo.getPltEditHolidayType1().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_OTHER))) {
			dto.setHolidayType2(vo.getPltEditSpecialOther());
		} else {
			dto.setHolidayType2(vo.getPltEditSpecialAbsence());
		}
		if (TimeConst.CODE_HOLIDAYTYPE_HOLIDAY == getInt(vo.getPltEditHolidayType1())
				&& TimeConst.CODE_HOLIDAYTYPE_HOLIDAY == getInt(vo.getPltEditStatusWithPay())) {
			// 有給休暇の場合
			dto.setHolidayRange(Integer.parseInt(vo.getPltEditHolidayRangePaidHoliday()));
		} else {
			// その他の場合
			dto.setHolidayRange(Integer.parseInt(vo.getPltEditHolidayRange()));
		}
		dto.setStartTime(DateUtility.getTime(vo.getPltEditStartHour(), vo.getPltEditStartMinute()));
		dto.setEndTime(DateUtility.getTime(vo.getPltEditEndHour(), vo.getPltEditEndMinute()));
		dto.setHolidayAcquisitionDate(acquisitionDate);
		dto.setUseDay(useDay);
		dto.setUseHour(useHour);
		dto.setRequestReason(vo.getTxtEditRequestReason());
	}
	
	/**
	 * @param type 休暇区分
	 * @return 休暇区分のプルダウンを返す
	 * @throws MospException 例外発生時
	 */
	protected String[][] setHolidayType3Value(int type) throws MospException {
		return mospParams.getProperties().getCodeArray(TimeConst.CODE_HOLIDAY_TYPE2_WITHPAY, false);
	}
	
	/**
	 * @param type 休暇区分
	 * @return 文字列に変換した休暇区分を返す
	 * @throws MospException 例外発生時
	 */
	protected String getHolidayType3Name(int type) throws MospException {
		if (type == 1) {
			return mospParams.getProperties().getName("AllTime");
		} else if (type == 2) {
			return mospParams.getProperties().getName("FrontTime");
		} else if (type == 3) {
			return mospParams.getProperties().getName("BackTime");
		} else {
			return mospParams.getProperties().getName("Hour") + mospParams.getProperties().getName("Rest");
		}
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setVoFields(HolidayRequestDtoInterface dto) throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		// DTOの値をVOに設定
		vo.setRecordId(dto.getTmdHolidayRequestId());
		vo.setPltEditStartYear(String.valueOf(DateUtility.getYear(dto.getRequestStartDate())));
		vo.setPltEditStartMonth(String.valueOf(DateUtility.getMonth(dto.getRequestStartDate())));
		vo.setPltEditStartDay(String.valueOf(DateUtility.getDay(dto.getRequestStartDate())));
		vo.setPltEditEndYear(String.valueOf(DateUtility.getYear(dto.getRequestEndDate())));
		vo.setPltEditEndMonth(String.valueOf(DateUtility.getMonth(dto.getRequestEndDate())));
		vo.setPltEditEndDay(String.valueOf(DateUtility.getDay(dto.getRequestEndDate())));
		vo.setPltEditHolidayType1(String.valueOf(dto.getHolidayType1()));
		if (dto.getHolidayType1() == 1) {
			vo.setPltEditStatusWithPay(dto.getHolidayType2());
		} else if (dto.getHolidayType1() == 2) {
			vo.setPltEditStatusSpecial(dto.getHolidayType2());
		} else if (dto.getHolidayType1() == 3) {
			vo.setPltEditSpecialOther(dto.getHolidayType2());
		} else {
			vo.setPltEditSpecialAbsence(dto.getHolidayType2());
		}
		if (dto.getHolidayType1() == 1 && TimeConst.CODE_HOLIDAYTYPE_HOLIDAY == getInt(dto.getHolidayType2())) {
			// 有給休暇の場合
			vo.setPltEditHolidayRangePaidHoliday(String.valueOf(dto.getHolidayRange()));
		} else {
			// その他の場合
			vo.setPltEditHolidayRange(String.valueOf(dto.getHolidayRange()));
		}
		vo.setPltEditStartHour(String.valueOf(DateUtility.getHour(dto.getStartTime())));
		vo.setPltEditStartMinute(String.valueOf(DateUtility.getMinute(dto.getStartTime())));
		vo.setPltEditEndHour(String.valueOf(DateUtility.getHour(dto.getEndTime())));
		vo.setPltEditEndMinute(String.valueOf(DateUtility.getMinute(dto.getEndTime())));
		vo.setTxtEditRequestReason(dto.getRequestReason());
		vo.setModeCardEdit(getApplicationMode(dto.getWorkflow()));
	}
	
	/**
	 * 開始年月日と終了年月日の比較
	 * @return trueは年月日が超えている。falseの年月日が超えていない。
	 * @throws MospException 例外発生時
	 */
	protected boolean chkActivationDateValidate() throws MospException {
		if (getEditStartDate().after(getEditEndDate())) {
			mospParams.addErrorMessage(TimeMessageConst.MSG_HOLIDAY_ACTIVATIONDATE, null);
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * 休暇日数を取得する。<br>
	 * @param holidayDto 休暇マスタDTO
	 * @param holidayDataDto 休暇データDTO
	 * @return 休暇日数
	 * @throws MospException 例外発生時
	 */
	protected double getHolidayDays(HolidayDtoInterface holidayDto, HolidayDataDtoInterface holidayDataDto)
			throws MospException {
		// VO取得
		HolidayRequestVo vo = (HolidayRequestVo)mospParams.getVo();
		HolidayRequestRegistBeanInterface regist = time().holidayRequestRegist();
		Date startDate = getEditStartDate();
		Date endDate = getEditEndDate();
		int count = 0;
		Date targetDate = startDate;
		while (!targetDate.after(endDate)) {
			if (mospParams.hasErrorMessage()) {
				return 0;
			}
			HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			dto.setPersonalId(vo.getPersonalId());
			dto.setRequestStartDate(targetDate);
			dto.setRequestEndDate(targetDate);
			regist.checkRequestDate(dto);
			if (mospParams.hasErrorMessage()) {
				mospParams.getErrorMessageList().clear();
				// 1日加算
				targetDate = DateUtility.addDay(targetDate, 1);
				continue;
			}
			count++;
			// 1日加算
			targetDate = DateUtility.addDay(targetDate, 1);
		}
		double holidayDays = 0;
		if (TimeConst.CODE_HOLIDAY_RANGE_ALL == getInt(vo.getPltEditHolidayRange())) {
			// 全休の場合
			holidayDays = count;
		} else if (TimeConst.CODE_HOLIDAY_RANGE_AM == getInt(vo.getPltEditHolidayRange())
				|| TimeConst.CODE_HOLIDAY_RANGE_PM == getInt(vo.getPltEditHolidayRange())) {
			// 午前休又は午後休の場合
			holidayDays = count * TimeConst.HOLIDAY_TIMES_HALF;
		} else {
			mospParams.addErrorMessage(TimeMessageConst.MSG_RANGE_SELECT, null);
			return 0;
		}
		if (holidayDto.getContinuousAcquisition() == 0) {
			// 連続取得が必須の場合
			if (holidayDays <= holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay()) {
				return holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay();
			}
			addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
			return 0;
		} else if (holidayDto.getContinuousAcquisition() == 1) {
			// 連続取得が警告の場合
			if (holidayDays <= holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay()) {
				return holidayDays;
			}
			addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
			return 0;
		} else if (holidayDto.getContinuousAcquisition() == 2) {
			// 連続取得が不要の場合
			if (holidayDto.getNoLimit() == 1) {
				// 付与日数が無制限の場合
				return holidayDays;
			}
			// 付与日数が無制限でない場合
			if (holidayDays <= holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay()) {
				return holidayDays;
			}
			addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName());
			return 0;
		}
		mospParams.addErrorMessage("");
		return 0;
	}
	
	/**
	 * 勤怠データを削除する
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void deleteAttendance(HolidayRequestDtoInterface dto) throws MospException {
		Date targetDate = dto.getRequestStartDate();
		while (!targetDate.after(dto.getRequestEndDate())) {
			int range = dto.getHolidayRange();
			boolean holidayAm = range == TimeConst.CODE_HOLIDAY_RANGE_AM;
			boolean holidayPm = range == TimeConst.CODE_HOLIDAY_RANGE_PM;
			if (range == TimeConst.CODE_HOLIDAY_RANGE_ALL) {
				// 全休の場合
				time().attendanceRegist().delete(dto.getPersonalId(), targetDate);
			} else if (holidayAm || holidayPm) {
				// 半休の場合
				List<HolidayRequestDtoInterface> list = timeReference().holidayRequest().getHolidayRequestList(
						dto.getPersonalId(), targetDate);
				for (HolidayRequestDtoInterface holidayRequestDto : list) {
					WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(
							holidayRequestDto.getWorkflow());
					if (workflowDto == null || PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
							|| PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
						continue;
					}
					if ((holidayAm && holidayRequestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_PM)
							|| (holidayPm && holidayRequestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_AM)) {
						// 勤怠消去
						time().attendanceRegist().delete(dto.getPersonalId(), targetDate);
					}
				}
				// 代休取得
				List<SubHolidayRequestDtoInterface> subHolidayRequestList = timeReference().subHolidayRequest()
					.getSubHolidayRequestList(dto.getPersonalId(), targetDate);
				for (SubHolidayRequestDtoInterface subHolidayRequestDto : subHolidayRequestList) {
					WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(
							subHolidayRequestDto.getWorkflow());
					if (workflowDto == null || PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
							|| PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
						continue;
					}
					if ((holidayAm && subHolidayRequestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_PM)
							|| (holidayPm && subHolidayRequestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_AM)) {
						// 勤怠消去
						time().attendanceRegist().delete(dto.getPersonalId(), targetDate);
					}
				}
			}
			targetDate = DateUtility.addDay(targetDate, 1);
		}
	}
	
	/**
	 * 有効な有給休暇付与情報を取得できなかった場合のエラーメッセージを設定する。<br>
	 */
	protected void addNotExistHolidayInfoErrorMessage() {
		mospParams.addErrorMessage(TimeMessageConst.MSG_NOT_EXIST_HOLIDAY_INFO, null);
	}
	
	/**
	 * 休暇申請する際に休暇日数を超過していた場合のメッセージを設定する。<br>
	 */
	protected void addHolidayNumDaysExcessErrorMessage() {
		mospParams.addErrorMessage(TimeMessageConst.MSG_HOLIDAY_NUM_DAYS_EXCESS, mospParams.getName("Salaried")
				+ mospParams.getName("Vacation"), null);
	}
	
	/**
	 * 休暇申請する際に休暇日数を超過していた場合のメッセージを設定する。<br>
	 * @param name 表示する文字
	 */
	protected void addHolidayNumDaysExcessErrorMessage(String name) {
		mospParams.addErrorMessage(TimeMessageConst.MSG_HOLIDAY_NUM_DAYS_EXCESS, name, null);
	}
	
}
