/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.calculation.action;

import java.util.List;

import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.base.TotalTimeBaseAction;
import jp.mosp.time.bean.TotalTimeCorrectionRegistBeanInterface;
import jp.mosp.time.bean.TotalTimeRegistBeanInterface;
import jp.mosp.time.calculation.vo.TotalTimeCardVo;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dto.settings.AllowanceDtoInterface;
import jp.mosp.time.dto.settings.HolidayDtoInterface;
import jp.mosp.time.dto.settings.TotalAbsenceDtoInterface;
import jp.mosp.time.dto.settings.TotalLeaveDtoInterface;
import jp.mosp.time.dto.settings.TotalOtherVacationDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeCorrectionDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeDataDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeDtoInterface;
import jp.mosp.time.dto.settings.impl.TmdTotalAbsenceDto;
import jp.mosp.time.dto.settings.impl.TmdTotalLeaveDto;
import jp.mosp.time.dto.settings.impl.TmdTotalOtherVacationDto;

/**
 * 集計結果一覧画面で選択した従業員の集計月毎の勤怠情報を修正する。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_RE_SHOW}
 * </li><li>
 * {@link #CMD_UPDATE}
 * </li><li>
 * {@link #CMD_INSERT}
 * </li></ul>
 */
public class TotalTimeCardAction extends TotalTimeBaseAction {
	
	/**
	 * 選択表示コマンド。<br>
	 * <br>
	 * 集計結果一覧画面で選択した従業員の集計月の勤怠情報を表示する。<br>
	 */
	public static final String	CMD_SELECT_SHOW	= "TM3131";
	
	/**
	 * 再表示コマンド。<br>
	 * <br>
	 * この画面で入力した内容を破棄し、登録済みの情報を再表示する。<br>
	 * 実行時には確認ダイアログを表示し、編集中の内容が破棄される旨を伝える。<br>
	 */
	public static final String	CMD_RE_SHOW		= "TM3132";
	
	/**
	 * 項目表示切替コマンド。<br>
	 * <br>
	 * 登録済の情報を出力するラベル表示と新たに情報を入力するためのテキストボックス・プルダウンの表示を切り替える。<br>
	 * 該当のボタンをクリックする度に表示が交互に切り替わり、切り替え前後の入出力内容は常に保持しておく。<br>
	 */
	public static final String	CMD_UPDATE		= "TM3133";
	
	/**
	 * 登録コマンド。<br>
	 * <br>
	 * 各種入力欄に入力されている内容を勤怠情報テーブルに登録する。<br>
	 * 入力チェックを行い、全ての項目が入力されていない場合、エラーメッセージにて通知して処理は実行されない。<br>
	 */
	public static final String	CMD_INSERT		= "TM3135";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public TotalTimeCardAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_RE_SHOW;
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new TotalTimeCardVo();
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SELECT_SHOW)) {
			// 選択表示
			prepareVo(false, false);
			show();
		} else if (mospParams.getCommand().equals(CMD_RE_SHOW)) {
			// 再表示
			prepareVo(true, false);
			reShow();
		} else if (mospParams.getCommand().equals(CMD_UPDATE)) {
			// 項目表示切替
			prepareVo(true, false);
			changeMode();
		} else if (mospParams.getCommand().equals(CMD_INSERT)) {
			// 登録
			prepareVo();
			regist();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void show() throws MospException {
		// VO準備
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// MosP処理情報からVOに設定
		getTransferredCutoffParams();
		// 締日関連情報を設定
		setCutoffInfo();
		// 個人ID取得
		String personalId = getTransferredGenericCode();
		// VOから締日コード及び対象年月取得
		String cutoffCode = vo.getCutoffCode();
		int targetYear = vo.getTargetYear();
		int targetMonth = vo.getTargetMonth();
		// 画面上部の社員情報を取得(対象日に締期間における基準日を設定)
		setEmployeeInfo(personalId, vo.getCutoffTermTargetDate());
		// 初期値設定
		setDefaultValues();
		setCorrectionVoFields();
		// 新規登録モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
		// 締状態チェック
		TotalTimeDtoInterface dto = timeReference().totalTimeTransaction().findForKey(targetYear, targetMonth,
				cutoffCode);
		final int fin = 2;
		final String modeStateFin = "fin";
		final String modeStateFix = "fix";
		vo.setJsModeCutoffStateEdit(modeStateFix);
		if (null != dto && fin == dto.getCutoffState()) {
			vo.setJsModeCutoffStateEdit(modeStateFin);
		}
	}
	
	/**
	 * 再表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void reShow() throws MospException {
		// 締日関連情報を設定
		setCutoffInfo();
		// 初期値設定
		setDefaultValues();
		setCorrectionVoFields();
		// VO準備
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 新規登録モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
	}
	
	/**
	 * 新規登録モードに設定する。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void changeMode() throws MospException {
		// VO準備
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 編集モード設定
		if (vo.getModeCardEdit().equals(PlatformConst.MODE_CARD_EDIT_INSERT)) {
			vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_EDIT);
		} else if (vo.getModeCardEdit().equals(PlatformConst.MODE_CARD_EDIT_EDIT)) {
			vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
			setDefaultValues();
		}
	}
	
	/**
	 * 登録処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void regist() throws MospException {
		// VO準備
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 登録クラス取得
		TotalTimeRegistBeanInterface registTotalTime = time().totalTimeRegist();
		TotalTimeCorrectionRegistBeanInterface correctionRegist = time().totalTimeCorrectionRegist();
		// VOから対象年月取得
		String personalId = vo.getPersonalId();
		int targetYear = vo.getTargetYear();
		int targetMonth = vo.getTargetMonth();
		// DTOの準備
		TotalTimeDataDtoInterface oldTotalTimeDto = timeReference().totalTime().findForKey(personalId, targetYear,
				targetMonth);
		List<TotalLeaveDtoInterface> oldTotalLeaveList = timeReference().totalLeave().getTotalLeaveList(personalId,
				targetYear, targetMonth);
		List<TotalOtherVacationDtoInterface> oldTotalOtherVacationList = timeReference().totalOtherVacation()
			.getTotalOtherVacationList(personalId, targetYear, targetMonth);
		List<TotalAbsenceDtoInterface> oldTotalAbsenceList = timeReference().totalAbsence().getTotalAbsenceList(
				personalId, targetYear, targetMonth);
		TotalTimeDataDtoInterface workTimeDto = registTotalTime.getInitDto();
		// DTOに値を設定
		setDtoWorkTimeFields(workTimeDto);
		// 特別休暇の登録処理
		for (int i = 0; i < vo.getAryTxtTimesSpecialLeave().length; i++) {
			TotalLeaveDtoInterface dto = new TmdTotalLeaveDto();
			dto.setPersonalId(workTimeDto.getPersonalId());
			dto.setCalculationYear(workTimeDto.getCalculationYear());
			dto.setCalculationMonth(workTimeDto.getCalculationMonth());
			setDtoSpecial(dto, i);
			time().totalLeaveRegist().regist(dto);
		}
		// その他休暇の登録処理
		for (int i = 0; i < vo.getAryTxtTimesOtherVacation().length; i++) {
			TotalOtherVacationDtoInterface dto = new TmdTotalOtherVacationDto();
			dto.setPersonalId(workTimeDto.getPersonalId());
			dto.setCalculationYear(workTimeDto.getCalculationYear());
			dto.setCalculationMonth(workTimeDto.getCalculationMonth());
			setDtoOther(dto, i);
			time().totalOtherVacationRegist().regist(dto);
		}
		// 欠勤の登録処理
		for (int i = 0; i < vo.getAryTxtDeduction().length; i++) {
			TotalAbsenceDtoInterface dto = new TmdTotalAbsenceDto();
			dto.setPersonalId(workTimeDto.getPersonalId());
			dto.setCalculationYear(workTimeDto.getCalculationYear());
			dto.setCalculationMonth(workTimeDto.getCalculationMonth());
			setDtoAbsence(dto, i);
			time().totalAbsenceRegist().regist(dto);
		}
		
		// 登録処理
		registTotalTime.update(workTimeDto);
		TotalTimeCorrectionDtoInterface totalTimeCorrectionDto = correctionRegist.getInitDto();
		setTotalTimeCorrectionDtoFields(totalTimeCorrectionDto);
		correctionRegist.insertTotalTime(totalTimeCorrectionDto, oldTotalTimeDto, workTimeDto);
		for (TotalLeaveDtoInterface oldDto : oldTotalLeaveList) {
			List<TotalLeaveDtoInterface> list = timeReference().totalLeave().getTotalLeaveList(personalId, targetYear,
					targetMonth);
			for (TotalLeaveDtoInterface newDto : list) {
				if (oldDto.getHolidayCode().equals(newDto.getHolidayCode()) && oldDto.getTimes() != newDto.getTimes()) {
					correctionRegist.insertLeave(totalTimeCorrectionDto, oldDto, newDto);
					break;
				}
			}
		}
		for (TotalOtherVacationDtoInterface oldDto : oldTotalOtherVacationList) {
			List<TotalOtherVacationDtoInterface> list = timeReference().totalOtherVacation().getTotalOtherVacationList(
					personalId, targetYear, targetMonth);
			for (TotalOtherVacationDtoInterface newDto : list) {
				if (oldDto.getHolidayCode().equals(newDto.getHolidayCode()) && oldDto.getTimes() != newDto.getTimes()) {
					correctionRegist.insertOtherVacation(totalTimeCorrectionDto, oldDto, newDto);
					break;
				}
			}
		}
		for (TotalAbsenceDtoInterface oldDto : oldTotalAbsenceList) {
			List<TotalAbsenceDtoInterface> list = timeReference().totalAbsence().getTotalAbsenceList(personalId,
					targetYear, targetMonth);
			for (TotalAbsenceDtoInterface newDto : list) {
				if (oldDto.getAbsenceCode().equals(newDto.getAbsenceCode()) && oldDto.getTimes() != newDto.getTimes()) {
					correctionRegist.insertAbsence(totalTimeCorrectionDto, oldDto, newDto);
				}
			}
		}
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 履歴追加成功メッセージ設定
		addInsertMessage();
		// 登録後の情報をVOに設定
		setVoWorkTimeFields(workTimeDto);
		setLeaveVoFields();
		setCorrectionVoFields();
	}
	
	/**
	 * 初期値を設定する。<br>
	 * @throws MospException 例外処理発生時
	 */
	public void setDefaultValues() throws MospException {
		// VO準備
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 計算年月取得
		int targetYear = vo.getTargetYear();
		int targetMonth = vo.getTargetMonth();
		// VOに初期値を設定
		vo.setLblMonth(targetYear + mospParams.getName("Year") + targetMonth + mospParams.getName("Month"));
		vo.setLblCorrectionHistory("");
		vo.setTxtCorrectionReason("");
		// 勤怠集計情報取得
		TotalTimeDataDtoInterface totalTimeDto = timeReference().totalTime().findForKey(vo.getPersonalId(), targetYear,
				targetMonth);
		if (totalTimeDto != null) {
			setVoWorkTimeFields(totalTimeDto);
		}
		// 休暇
		setLeaveVoFields();
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setDtoWorkTimeFields(TotalTimeDataDtoInterface dto) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// VOの値をDTOに設定
		// レコード識別ID
		dto.setTmdTotalTimeId(vo.getTmdTotalTimeId());
		// 個人ID
		dto.setPersonalId(vo.getPersonalId());
		// 計算年月
		dto.setCalculationYear(vo.getTargetYear());
		dto.setCalculationMonth(vo.getTargetMonth());
		// 集計日
		dto.setCalculationDate(vo.getCalculationDate());
		// 減額対象時間
		dto.setDecreaseTime(getTimeValue(Integer.parseInt(vo.getTxtDecreaseTimeHour()), Integer.parseInt(vo
			.getTxtDecreaseTimeMinute())));
		// 直行回数
		dto.setDirectStart(Integer.parseInt(vo.getTxtTimesNonstop()));
		// 直帰回数
		dto.setDirectEnd(Integer.parseInt(vo.getTxtTimesNoreturn()));
		// 45時間超残業時間
		dto.setFortyFiveHourOvertime(getTimeValue(Integer.parseInt(vo.getTxt45HourOverTimeHour()), Integer.parseInt(vo
			.getTxt45HourOverTimeMinute())));
		
		// 合計遅刻日数
		dto.setLateDays(Integer.parseInt(vo.getTxtLateDays()));
		// 遅刻30分以上日数
		dto.setLateThirtyMinutesOrMore(Integer.parseInt(vo.getTxtLateThirtyMinutesOrMore()));
		// 遅刻30分未満日数
		dto.setLateLessThanThirtyMinutes(Integer.parseInt(vo.getTxtLateLessThanThirtyMinutes()));
		
		// 深夜時間
		dto.setLateNight(getTimeValue(Integer.parseInt(vo.getTxtLateNightHour()), Integer.parseInt(vo
			.getTxtLateNightMinute())));
		
		// 合計遅刻時間
		dto.setLateTime(getTimeValue(Integer.parseInt(vo.getTxtLateTimeHour()), Integer.parseInt(vo
			.getTxtLateTimeMinute())));
		// 遅刻30分以上時間
		dto.setLateThirtyMinutesOrMoreTime(getTimeValue(vo.getTxtLateThirtyMinutesOrMoreTimeHour(), vo
			.getTxtLateThirtyMinutesOrMoreTimeMinute()));
		// 遅刻30分未満時間
		dto.setLateLessThanThirtyMinutesTime(getTimeValue(vo.getTxtLateLessThanThirtyMinutesTimeHour(), vo
			.getTxtLateLessThanThirtyMinutesTimeHour()));
		
		// 合計早退日数
		dto.setLeaveEarlyDays(Integer.parseInt(vo.getTxtLeaveEarlyDays()));
		// 早退30分以上日数
		dto.setLeaveEarlyThirtyMinutesOrMore(Integer.parseInt(vo.getTxtLeaveEarlyThirtyMinutesOrMore()));
		// 早退30分未満日数
		dto.setLeaveEarlyLessThanThirtyMinutes(Integer.parseInt(vo.getTxtLeaveEarlyLessThanThirtyMinutes()));
		
		// 法定休日出勤日数
		dto.setLegalWorkOnHoliday(Double.parseDouble(vo.getTxtLegalWorkOnHoliday()));
		// 有給休暇時間
		dto.setPaidHolidayHour(Integer.parseInt(vo.getTxtPaidholidayHour()));
		// 私用外出時間
		dto.setPrivateTime(getTimeValue(Integer.parseInt(vo.getTxtPrivateHour()), Integer.parseInt(vo
			.getTxtPrivateMinute())));
		// 公用外出時間
		dto.setPublicTime(getTimeValue(Integer.parseInt(vo.getTxtPublicHour()), Integer.parseInt(vo
			.getTxtPublicMinute())));
		// 休憩時間
		dto.setRestTime(getTimeValue(Integer.parseInt(vo.getTxtRestTimeHour()), Integer.parseInt(vo
			.getTxtRestTimeMinute())));
		// 深夜休憩時間
		dto.setRestLateNight(getTimeValue(Integer.parseInt(vo.getTxtRestLateNightHour()), Integer.parseInt(vo
			.getTxtRestLateNightMinute())));
		// 所定休出休憩時間
		dto.setRestWorkOnSpecificHoliday(getTimeValue(Integer.parseInt(vo.getTxtRestWorkOnSpecificHour()), Integer
			.parseInt(vo.getTxtRestWorkOnSpecificMinute())));
		// 法定休出休憩時間
		dto.setRestWorkOnHoliday(getTimeValue(Integer.parseInt(vo.getTxtRestWorkOnLegalHour()), Integer.parseInt(vo
			.getTxtRestWorkOnLegalMinute())));
		// 残業時間
		dto.setOvertime(getTimeValue(Integer.parseInt(vo.getTxtOverTimeHour()), Integer.parseInt(vo
			.getTxtOverTimeMinute())));
		// 法定内残業時間
		dto.setOvertimeIn(getTimeValue(Integer.parseInt(vo.getTxtOverTimeInHour()), Integer.parseInt(vo
			.getTxtOverTimeInMinute())));
		// 法定外残業時間
		dto.setOvertimeOut(getTimeValue(Integer.parseInt(vo.getTxtOverTimeOutHour()), Integer.parseInt(vo
			.getTxtOverTimeOutMinute())));
		// 60時間超残業時間
		dto.setSixtyHourOvertime(getTimeValue(Integer.parseInt(vo.getTxt60HourOverTimeHour()), Integer.parseInt(vo
			.getTxt60HourOverTimeMinute())));
		// 所定休日時間外時間
		dto.setSpecificOvertime(getTimeValue(Integer.parseInt(vo.getTxtSpecificOverTimeHour()), Integer.parseInt(vo
			.getTxtSpecificOverTimeMiunte())));
		// 所定休日出勤日数
		dto.setSpecificWorkOnHoliday(Double.parseDouble(vo.getTxtSpecificWorkOnHoliday()));
		// 所定勤務時間
		// 現在は非表示のため0を設定する。
//		dto.setSpecificWorkTime(getTimeValue(Integer.valueOf(vo.getTxtSpecificWorkTimeHour()), Integer.valueOf(vo
//			.getTxtSpecificWorkTimeMinute())));
		dto.setSpecificWorkTime(0);
		// 代替休暇日数
		dto.setTimesAlternative(Double.parseDouble(vo.getTxtTimesAlternative()));
		// 代休日数
		dto.setTimesCompensation(Double.parseDouble(vo.getTxtTimesCompensation()));
		// 休日日数
		dto.setTimesHoliday(Integer.parseInt(vo.getTxtTimesHoliday()));
		// 合計遅刻回数
//		dto.setTimesLate(Integer.parseInt(vo.getTxtTimesLate()));
		dto.setTimesLate(Integer.parseInt(vo.getTxtLateDays()));
		// 深夜代休日数
		dto.setTimesLateCompensation(Double.parseDouble(vo.getTxtTimesLateCompensation()));
		// 合計早退回数
//		dto.setTimesLeaveEarly(Integer.parseInt(vo.getTxtTimesLeaveEarly()));
		dto.setTimesLeaveEarly(Integer.parseInt(vo.getTxtLeaveEarlyDays()));
		// 法定代休日数
		dto.setTimesLegalCompensation(Double.parseDouble(vo.getTxtTimesLegalCompensation()));
		// 法定休日日数
		dto.setTimesLegalHoliday(Integer.parseInt(vo.getTxtTimesLegalHoliday()));
		// 有給休暇日数
		dto.setTimesPaidHoliday(Double.parseDouble(vo.getTxtTimesPaidHoliday()));
		// 所定代休日数
		dto.setTimesSpecificCompensation(Double.parseDouble(vo.getTxtTimesSpecificCompensation()));
		// 所定休日日数
		dto.setTimesSpecificHoliday(Integer.parseInt(vo.getTxtTimesSpecificHoliday()));
		// ストック休暇日数
		dto.setTimesStockHoliday(Double.parseDouble(vo.getTxtTimesStockHoliday()));
		// 振替休日日数
		dto.setTimesHolidaySubstitute(Double.parseDouble(vo.getTxtTimesSubstitute()));
		// 出勤回数
		dto.setTimesWork(Integer.parseInt(vo.getTxtTimesWork()));
		// 出勤日数
		dto.setTimesWorkDate(Double.parseDouble(vo.getTxtTimesWorkDate()));
		// 平日時間外時間
		dto.setWeekDayOvertime(getTimeValue(Integer.parseInt(vo.getTxtWeekDayOverTimeHour()), Integer.parseInt(vo
			.getTxtWeekDayOverTimeMinute())));
		// 法定休出時間
		dto.setWorkOnHoliday(getTimeValue(Integer.parseInt(vo.getTxtRestWorkOnLegalHour()), Integer.parseInt(vo
			.getTxtRestWorkOnLegalMinute())));
		// 所定休出時間
		dto.setWorkOnSpecificHoliday(getTimeValue(Integer.parseInt(vo.getTxtRestWorkOnSpecificHour()), Integer
			.parseInt(vo.getTxtRestWorkOnSpecificMinute())));
		// 勤務時間
		dto.setWorkTime(getTimeValue(Integer.parseInt(vo.getTxtWorkTimeHour()), Integer.parseInt(vo
			.getTxtWorkTimeMinute())));
		// 出勤対象日数
		dto.setTimesTotalWorkDate(Integer.parseInt(vo.getTxtTimesTotalWorkDate()));
		// 残業回数
		dto.setTimesOvertime(Integer.parseInt(vo.getTxtTimesOvertime()));
		// 法定休出時間
		dto.setWorkOnHoliday(getTimeValue(Integer.parseInt(vo.getTxtWorkOnHolidayHour()), Integer.parseInt(vo
			.getTxtWorkOnHolidayMinute())));
		// 所定休出時間
		dto.setWorkOnSpecificHoliday(getTimeValue(Integer.parseInt(vo.getTxtWorkSpecificOnHolidayHour()), Integer
			.parseInt(vo.getTxtWorkSpecificOnHolidayMinute())));
		// 休日出勤回数
		dto.setTimesWorkingHoliday(Integer.parseInt(vo.getTxtTimesWorkingHoliday()));
		
		// 合計早退時間
		dto.setLeaveEarlyTime(getTimeValue(Integer.parseInt(vo.getTxtLeaveEarlyTimeHour()), Integer.parseInt(vo
			.getTxtLeaveEarlyTimeMinute())));
		// 早退30分以上時間
		dto.setLeaveEarlyThirtyMinutesOrMoreTime(getTimeValue(vo.getTxtLeaveEarlyThirtyMinutesOrMoreTimeHour(), vo
			.getTxtLeaveEarlyThirtyMinutesOrMoreTimeMinute()));
		// 早退30分未満時間
		dto.setLeaveEarlyLessThanThirtyMinutesTime(getTimeValue(vo.getTxtLeaveEarlyLessThanThirtyMinutesTimeHour(), vo
			.getTxtLeaveEarlyLessThanThirtyMinutesTimeMinute()));
		
		// 法定振替休日日数
		dto.setTimesLegalHolidaySubstitute(Double.parseDouble(vo.getTxtTimesLegalHolidaySubstitute()));
		// 所定振替休日日数
		dto.setTimesSpecificHolidaySubstitute(Double.parseDouble(vo.getTxtTimesSpecificHolidaySubstitute()));
		// 法定代休未使用日数
		dto.setLegalCompensationUnused(Double.parseDouble(vo.getTxtLegalCompensationUnused()));
		// 所定代休未使用日数
		dto.setSpecificCompensationUnused(Double.parseDouble(vo.getTxtSpecificCompensationUnused()));
		// 深夜代休未使用日数
		dto.setLateCompensationUnused(Double.parseDouble(vo.getTxtLateCompensationUnused()));
		// 特別休暇合計日数
		dto.setTotalSpecialHoliday(Double.parseDouble(vo.getTxtTotalSpecialHoliday()));
		// その他休暇合計日数
		dto.setTotalOtherHoliday(Double.parseDouble(vo.getTxtTotalOtherHoliday()));
		// 欠勤合計日数
		dto.setTotalAbsence(Double.parseDouble(vo.getTxtTotalDeduction()));
		// 出勤実績日数
		dto.setTimesAchievement(Integer.parseInt(vo.getTxtTimesAchievement()));
		// TODO 法定外休憩、所定休日法定外休憩、所定休日法定外勤務時間は後日追加
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param allowance 手当区分
	 * @throws MospException 例外発生時
	 */
	protected void setDtoAllowanceFields(AllowanceDtoInterface dto, int allowance) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		dto.setTmdAllowanceId(vo.getTmdAllowanceId());
		dto.setPersonalId(vo.getPersonalId());
		if (allowance == 0) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO1);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance1()));
		} else if (allowance == 1) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO2);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance2()));
		} else if (allowance == 2) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO3);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance3()));
		} else if (allowance == 3) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO4);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance4()));
		} else if (allowance == 4) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO5);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance5()));
		} else if (allowance == 5) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO6);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance6()));
		} else if (allowance == 6) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO7);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance7()));
		} else if (allowance == 7) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO8);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance8()));
		} else if (allowance == 8) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO9);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance9()));
		} else if (allowance == 9) {
			dto.setAllowanceCode(TimeConst.CODE_ALLOWANCE_INFO10);
			dto.setAllowance(Integer.parseInt(vo.getTxtTimesAllowance10()));
		}
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param i 特別休暇Index
	 * @throws MospException 例外発生時
	 */
	protected void setDtoSpecial(TotalLeaveDtoInterface dto, int i) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		dto.setTmdTotalLeaveId(vo.getAryTimesSpecialLeaveId()[i]);
		dto.setHolidayCode(vo.getAryTxtTimesSpecialLeaveCode()[i]);
		dto.setTimes(Double.parseDouble(vo.getAryTxtTimesSpecialLeave()[i]));
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param i その他休暇Index
	 * @throws MospException 例外発生時
	 */
	protected void setDtoOther(TotalOtherVacationDtoInterface dto, int i) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		dto.setTmdTotalOtherVacationId(vo.getAryTimesOtherVacationId()[i]);
		dto.setHolidayCode(vo.getAryTxtTimesOtherVacationCode()[i]);
		dto.setTimes(Double.parseDouble(vo.getAryTxtTimesOtherVacation()[i]));
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param i 欠勤Index
	 * @throws MospException 例外発生時
	 */
	protected void setDtoAbsence(TotalAbsenceDtoInterface dto, int i) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		dto.setTmdTotalAbsenceId(vo.getAryDeductionId()[i]);
		dto.setAbsenceCode(vo.getAryTxtDeductionCode()[i]);
		dto.setTimes(Double.parseDouble(vo.getAryTxtDeduction()[i]));
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setTotalTimeCorrectionDtoFields(TotalTimeCorrectionDtoInterface dto) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 個人ID
		dto.setPersonalId(vo.getPersonalId());
		// 年
		dto.setCalculationYear(vo.getTargetYear());
		// 月
		dto.setCalculationMonth(vo.getTargetMonth());
		// 修正番号
		TotalTimeCorrectionDtoInterface latestDto = timeReference().totalTimeCorrection()
			.getLatestTotalTimeCorrectionInfo(vo.getPersonalId(), vo.getTargetYear(), vo.getTargetMonth());
		int correctionTimes = 1;
		if (latestDto != null) {
			correctionTimes += latestDto.getCorrectionTimes();
		}
		dto.setCorrectionTimes(correctionTimes);
		// 修正日時
		dto.setCorrectionDate(DateUtility.getSystemTimeAndSecond());
		// 修正個人ID
		dto.setCorrectionPersonalId(mospParams.getUser().getPersonalId());
		// 修正理由
		dto.setCorrectionReason(vo.getTxtCorrectionReason());
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外処理発生時
	 */
	protected void setVoWorkTimeFields(TotalTimeDataDtoInterface dto) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// DTOの値をVOに設定
		vo.setLblEmployeeCode(getEmployeeCode(dto.getPersonalId()));
		// レコード識別ID
		vo.setTmdTotalTimeId(dto.getTmdTotalTimeId());
		// 減額対象時間
		vo.setTxtDecreaseTimeHour(String.valueOf(dto.getDecreaseTime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtDecreaseTimeMinute(String.valueOf(dto.getDecreaseTime() % TimeConst.CODE_DEFINITION_HOUR));
		// 45時間超残業時間
		vo.setTxt45HourOverTimeHour(String.valueOf(dto.getFortyFiveHourOvertime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxt45HourOverTimeMinute(String.valueOf(dto.getFortyFiveHourOvertime() % TimeConst.CODE_DEFINITION_HOUR));
		
		// 合計遅刻日数
		vo.setTxtLateDays(String.valueOf(dto.getLateDays()));
		// 遅刻30分以上日数
		vo.setTxtLateThirtyMinutesOrMore(String.valueOf(dto.getLateThirtyMinutesOrMore()));
		// 遅刻30分未満日数
		vo.setTxtLateLessThanThirtyMinutes(String.valueOf(dto.getLateLessThanThirtyMinutes()));
		
		// 深夜時間
		vo.setTxtLateNightHour(String.valueOf(dto.getLateNight() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtLateNightMinute(String.valueOf(dto.getLateNight() % TimeConst.CODE_DEFINITION_HOUR));
		
		// 合計遅刻時間
		vo.setTxtLateTimeHour(String.valueOf(dto.getLateTime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtLateTimeMinute(String.valueOf(dto.getLateTime() % TimeConst.CODE_DEFINITION_HOUR));
		// 遅刻30分以上時間
		vo.setTxtLateThirtyMinutesOrMoreTimeHour(String.valueOf(getHour(dto.getLateThirtyMinutesOrMoreTime())));
		vo.setTxtLateThirtyMinutesOrMoreTimeMinute(String.valueOf(getMinute(dto.getLateThirtyMinutesOrMoreTime())));
		// 遅刻30分未満時間
		vo.setTxtLateLessThanThirtyMinutesTimeHour(String.valueOf(getHour(dto.getLateLessThanThirtyMinutesTime())));
		vo.setTxtLateLessThanThirtyMinutesTimeMinute(String.valueOf(getMinute(dto.getLateLessThanThirtyMinutesTime())));
		
		// 合計早退日数
		vo.setTxtLeaveEarlyDays(String.valueOf(dto.getLeaveEarlyDays()));
		// 早退30分以上日数
		vo.setTxtLeaveEarlyThirtyMinutesOrMore(String.valueOf(dto.getLeaveEarlyThirtyMinutesOrMore()));
		// 早退30分未満日数
		vo.setTxtLeaveEarlyLessThanThirtyMinutes(String.valueOf(dto.getLeaveEarlyLessThanThirtyMinutes()));
		
		// 法定休日出勤日数
		vo.setTxtLegalWorkOnHoliday(String.valueOf(dto.getLegalWorkOnHoliday()));
		// 有給休暇時間
		vo.setTxtPaidholidayHour(String.valueOf(dto.getPaidHolidayHour()));
		// 私用外出時間
		vo.setTxtPrivateHour(String.valueOf(dto.getPrivateTime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtPrivateMinute(String.valueOf(dto.getPrivateTime() % TimeConst.CODE_DEFINITION_HOUR));
		// 公用外出時間
		vo.setTxtPublicHour(String.valueOf(dto.getPublicTime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtPublicMinute(String.valueOf(dto.getPublicTime() % TimeConst.CODE_DEFINITION_HOUR));
		// 深夜休憩時間
		vo.setTxtRestLateNightHour(String.valueOf(dto.getRestLateNight() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtRestLateNightMinute(String.valueOf(dto.getRestLateNight() % TimeConst.CODE_DEFINITION_HOUR));
		// 休憩時間
		vo.setTxtRestTimeHour(String.valueOf(dto.getRestTime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtRestTimeMinute(String.valueOf(dto.getRestTime() % TimeConst.CODE_DEFINITION_HOUR));
		// 法定休出休憩時間
		vo.setTxtRestWorkOnLegalHour(String.valueOf(dto.getRestWorkOnHoliday() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtRestWorkOnLegalMinute(String.valueOf(dto.getRestWorkOnHoliday() % TimeConst.CODE_DEFINITION_HOUR));
		// 60時間超残業時間
		vo.setTxt60HourOverTimeHour(String.valueOf(dto.getSixtyHourOvertime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxt60HourOverTimeMinute(String.valueOf(dto.getSixtyHourOvertime() % TimeConst.CODE_DEFINITION_HOUR));
		// 所定休日時間外時間
		vo.setTxtSpecificOverTimeHour(String.valueOf(dto.getSpecificOvertime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtSpecificOverTimeMiunte(String.valueOf(dto.getSpecificOvertime() % TimeConst.CODE_DEFINITION_HOUR));
		// 所定休日時間外時間
		vo.setTxtSpecificWorkOnHoliday(String.valueOf(dto.getSpecificWorkOnHoliday()));
		// 所定勤務時間
		vo.setTxtSpecificWorkTimeHour(String.valueOf(dto.getSpecificWorkTime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtSpecificWorkTimeMinute(String.valueOf(dto.getSpecificWorkTime() % TimeConst.CODE_DEFINITION_HOUR));
		// 代替休暇日数
		vo.setTxtTimesAlternative(String.valueOf(dto.getTimesAlternative()));
		// 代休日数
		vo.setTxtTimesCompensation(String.valueOf(dto.getTimesCompensation()));
		// 	休日日数
		vo.setTxtTimesHoliday(String.valueOf(dto.getTimesHoliday()));
		// 合計遅刻回数
		vo.setTxtTimesLate(String.valueOf(dto.getTimesLate()));
		// 深夜代休日数
		vo.setTxtTimesLateCompensation(String.valueOf(dto.getTimesLateCompensation()));
		// 合計早退回数
		vo.setTxtTimesLeaveEarly(String.valueOf(dto.getTimesLeaveEarly()));
		// 	法定代休日数
		vo.setTxtTimesLegalCompensation(String.valueOf(dto.getTimesLegalCompensation()));
		// 法定休日日数
		vo.setTxtTimesLegalHoliday(String.valueOf(dto.getTimesLegalHoliday()));
		// 直行回数
		vo.setTxtTimesNonstop(String.valueOf(dto.getDirectStart()));
		// 直帰回数
		vo.setTxtTimesNoreturn(String.valueOf(dto.getDirectEnd()));
		// 有給休暇日数
		vo.setTxtTimesPaidHoliday(String.valueOf(dto.getTimesPaidHoliday()));
		// 所定代休日数
		vo.setTxtTimesSpecificCompensation(String.valueOf(dto.getTimesSpecificCompensation()));
		// 所定休日日数
		vo.setTxtTimesSpecificHoliday(String.valueOf(dto.getTimesSpecificHoliday()));
		// ストック休暇日数
		vo.setTxtTimesStockHoliday(String.valueOf(dto.getTimesStockHoliday()));
		// 振替休日日数
		vo.setTxtTimesSubstitute(String.valueOf(dto.getTimesHolidaySubstitute()));
		// 出勤回数
		vo.setTxtTimesWork(String.valueOf(dto.getTimesWork()));
		// 出勤日数
		vo.setTxtTimesWorkDate(String.valueOf(dto.getTimesWorkDate()));
		// 平日時間外時間
		vo.setTxtWeekDayOverTimeHour(String.valueOf(dto.getWeekDayOvertime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtWeekDayOverTimeMinute(String.valueOf(dto.getWeekDayOvertime() % TimeConst.CODE_DEFINITION_HOUR));
		// 所定休出時間
		vo.setTxtRestWorkOnSpecificHour(String.valueOf(dto.getRestWorkOnSpecificHoliday()
				/ TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtRestWorkOnSpecificMinute(String.valueOf(dto.getRestWorkOnSpecificHoliday()
				% TimeConst.CODE_DEFINITION_HOUR));
		// 法定休出時間
		vo.setTxtRestWorkOnLegalHour(String.valueOf(dto.getRestWorkOnHoliday() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtRestWorkOnLegalMinute(String.valueOf(dto.getRestWorkOnHoliday() % TimeConst.CODE_DEFINITION_HOUR));
		// 残業時間
		vo.setTxtOverTimeHour(String.valueOf(dto.getOvertime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtOverTimeMinute(String.valueOf(dto.getOvertime() % TimeConst.CODE_DEFINITION_HOUR));
		// 法定内残業時間
		vo.setTxtOverTimeInHour(String.valueOf(dto.getOvertimeIn() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtOverTimeInMinute(String.valueOf(dto.getOvertimeIn() % TimeConst.CODE_DEFINITION_HOUR));
		// 法定外残業時間
		vo.setTxtOverTimeOutHour(String.valueOf(dto.getOvertimeOut() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtOverTimeOutMinute(String.valueOf(dto.getOvertimeOut() % TimeConst.CODE_DEFINITION_HOUR));
		// 勤務時間
		vo.setTxtWorkTimeHour(String.valueOf(dto.getWorkTime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtWorkTimeMinute(String.valueOf(dto.getWorkTime() % TimeConst.CODE_DEFINITION_HOUR));
		// 出勤実績日数
		vo.setTxtTimesAchievement(String.valueOf(dto.getTimesAchievement()));
		// 出勤対象日数
		vo.setTxtTimesTotalWorkDate(String.valueOf(dto.getTimesTotalWorkDate()));
		// 残業回数
		vo.setTxtTimesOvertime(String.valueOf(dto.getTimesOvertime()));
		// 法定休出時間
		vo.setTxtWorkOnHolidayHour(String.valueOf(dto.getWorkOnHoliday() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtWorkOnHolidayMinute(String.valueOf(dto.getWorkOnHoliday() % TimeConst.CODE_DEFINITION_HOUR));
		// 所定休出時間
		vo.setTxtWorkSpecificOnHolidayHour(String.valueOf(dto.getWorkOnSpecificHoliday()
				/ TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtWorkSpecificOnHolidayMinute(String.valueOf(dto.getWorkOnSpecificHoliday()
				% TimeConst.CODE_DEFINITION_HOUR));
		// 休日出勤回数
		vo.setTxtTimesWorkingHoliday(String.valueOf(dto.getTimesWorkingHoliday()));
		
		// 合計早退時間
		vo.setTxtLeaveEarlyTimeHour(String.valueOf(dto.getLeaveEarlyTime() / TimeConst.CODE_DEFINITION_HOUR));
		vo.setTxtLeaveEarlyTimeMinute(String.valueOf(dto.getLeaveEarlyTime() % TimeConst.CODE_DEFINITION_HOUR));
		// 早退30分以上時間
		vo.setTxtLeaveEarlyThirtyMinutesOrMoreTimeHour(String.valueOf(getHour(dto
			.getLeaveEarlyThirtyMinutesOrMoreTime())));
		vo.setTxtLeaveEarlyThirtyMinutesOrMoreTimeMinute(String.valueOf(getMinute(dto
			.getLeaveEarlyThirtyMinutesOrMoreTime())));
		// 早退30分未満時間
		vo.setTxtLeaveEarlyLessThanThirtyMinutesTimeHour(String.valueOf(getHour(dto
			.getLeaveEarlyLessThanThirtyMinutesTime())));
		vo.setTxtLeaveEarlyLessThanThirtyMinutesTimeMinute(String.valueOf(getMinute(dto
			.getLeaveEarlyLessThanThirtyMinutesTime())));
		
		// 法定振替休日日数
		vo.setTxtTimesLegalHolidaySubstitute(String.valueOf(dto.getTimesLegalHolidaySubstitute()));
		// 所定振替休日日数
		vo.setTxtTimesSpecificHolidaySubstitute(String.valueOf(dto.getTimesSpecificHolidaySubstitute()));
		// 特別休暇合計日数
		vo.setTxtTotalSpecialHoliday(String.valueOf(dto.getTotalSpecialHoliday()));
		// その他休暇合計日数
		vo.setTxtTotalOtherHoliday(String.valueOf(dto.getTotalOtherHoliday()));
		// 欠勤合計日数
		vo.setTxtTotalDeduction(String.valueOf(dto.getTotalAbsence()));
		// 法定代休未使用日数
		vo.setTxtLegalCompensationUnused(String.valueOf(dto.getLegalCompensationUnused()));
		// 所定代休未使用日数
		vo.setTxtSpecificCompensationUnused(String.valueOf(dto.getSpecificCompensationUnused()));
		// 深夜代休未使用日数
		vo.setTxtLateCompensationUnused(String.valueOf(dto.getLateCompensationUnused()));
		// 法定外休憩、所定休日法定外休憩、所定休日法定外勤務時間は未使用
		vo.setTxtOverRestTimeHour("0");
		vo.setTxtOverRestTimeMinute("0");
		vo.setTxtRestWorkOnOverHour("0");
		vo.setTxtRestWorkOnOverMinute("0");
		vo.setTxtWorkOnOverHour("0");
		vo.setTxtWorkOnOverMinute("0");
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setVoAllowanceFields(AllowanceDtoInterface dto) throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		vo.setLblEmployeeCode(getEmployeeCode(dto.getPersonalId()));
		vo.setTmdAllowanceId(dto.getTmdAllowanceId());
		if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO1)) {
			vo.setTxtTimesAllowance1(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO2)) {
			vo.setTxtTimesAllowance2(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO3)) {
			vo.setTxtTimesAllowance3(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO4)) {
			vo.setTxtTimesAllowance4(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO5)) {
			vo.setTxtTimesAllowance5(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO6)) {
			vo.setTxtTimesAllowance6(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO7)) {
			vo.setTxtTimesAllowance7(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO8)) {
			vo.setTxtTimesAllowance8(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO9)) {
			vo.setTxtTimesAllowance9(String.valueOf(dto.getAllowance()));
		} else if (dto.getAllowanceCode().equals(TimeConst.CODE_ALLOWANCE_INFO10)) {
			vo.setTxtTimesAllowance10(String.valueOf(dto.getAllowance()));
		}
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @throws MospException 例外発生時
	 */
	protected void setLeaveVoFields() throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		// 特別休暇情報取得(締期間の基準日取得)
		List<HolidayDtoInterface> specialList = timeReference().holiday().getHolidayList(vo.getTargetDate(),
				TimeConst.CODE_HOLIDAYTYPE_SPECIAL);
		int specialListSize = specialList.size();
		String[] aryTxtTimesSpecialLeave = new String[specialListSize];
		String[] aryTxtTimesSpecialLeaveTitle = new String[specialListSize];
		String[] aryTxtTimesSpecialLeaveCode = new String[specialListSize];
		long[] aryTimesSpecialLeaveId = new long[specialListSize];
		int i = 0;
		for (HolidayDtoInterface dto : specialList) {
			aryTxtTimesSpecialLeaveTitle[i] = dto.getHolidayAbbr();
			aryTxtTimesSpecialLeaveCode[i] = dto.getHolidayCode();
			aryTxtTimesSpecialLeave[i] = Double.toString(0);
			TotalLeaveDtoInterface totalLeaveDto = timeReference().totalLeave().findForKey(vo.getPersonalId(),
					vo.getTargetYear(), vo.getTargetMonth(), dto.getHolidayCode());
			if (totalLeaveDto != null) {
				aryTimesSpecialLeaveId[i] = totalLeaveDto.getTmdTotalLeaveId();
				aryTxtTimesSpecialLeave[i] = String.valueOf(totalLeaveDto.getTimes());
			}
			i++;
		}
		vo.setAryTxtTimesSpecialLeave(aryTxtTimesSpecialLeave);
		vo.setAryTxtTimesSpecialLeaveTitle(aryTxtTimesSpecialLeaveTitle);
		vo.setAryTxtTimesSpecialLeaveCode(aryTxtTimesSpecialLeaveCode);
		vo.setAryTimesSpecialLeaveId(aryTimesSpecialLeaveId);
		// その他休暇情報取得(締期間の基準日取得)
		List<HolidayDtoInterface> otherList = timeReference().holiday().getHolidayList(vo.getTargetDate(),
				TimeConst.CODE_HOLIDAYTYPE_OTHER);
		int otherListSize = otherList.size();
		String[] aryTxtTimesOtherVacation = new String[otherListSize];
		String[] aryTxtTimesOtherVacationTitle = new String[otherListSize];
		String[] aryTxtTimesOtherVacationCode = new String[otherListSize];
		long[] aryTimesOtherVacationId = new long[otherListSize];
		i = 0;
		for (HolidayDtoInterface dto : otherList) {
			aryTxtTimesOtherVacationTitle[i] = dto.getHolidayAbbr();
			aryTxtTimesOtherVacationCode[i] = dto.getHolidayCode();
			aryTxtTimesOtherVacation[i] = Double.toString(0);
			TotalOtherVacationDtoInterface totalOtherVacationDto = timeReference().totalOtherVacation().findForKey(
					vo.getPersonalId(), vo.getTargetYear(), vo.getTargetMonth(), dto.getHolidayCode());
			if (totalOtherVacationDto != null) {
				aryTimesOtherVacationId[i] = totalOtherVacationDto.getTmdTotalOtherVacationId();
				aryTxtTimesOtherVacation[i] = String.valueOf(totalOtherVacationDto.getTimes());
			}
			i++;
		}
		vo.setAryTxtTimesOtherVacation(aryTxtTimesOtherVacation);
		vo.setAryTxtTimesOtherVacationTitle(aryTxtTimesOtherVacationTitle);
		vo.setAryTxtTimesOtherVacationCode(aryTxtTimesOtherVacationCode);
		vo.setAryTimesOtherVacationId(aryTimesOtherVacationId);
		// 欠勤情報取得(締期間の基準日取得)
		List<HolidayDtoInterface> deductionList = timeReference().holiday().getHolidayList(vo.getTargetDate(),
				TimeConst.CODE_HOLIDAYTYPE_ABSENCE);
		int deductionListSize = deductionList.size();
		String[] aryTxtTimesDeduction = new String[deductionListSize];
		String[] aryTxtTimesDeductionTitle = new String[deductionListSize];
		String[] aryTxtTimesDeductionCode = new String[deductionListSize];
		long[] aryDeductionId = new long[deductionListSize];
		i = 0;
		for (HolidayDtoInterface dto : deductionList) {
			aryTxtTimesDeductionTitle[i] = dto.getHolidayAbbr();
			aryTxtTimesDeductionCode[i] = dto.getHolidayCode();
			aryTxtTimesDeduction[i] = Double.toString(0);
			TotalAbsenceDtoInterface totalAbsenceDto = timeReference().totalAbsence().findForKey(vo.getPersonalId(),
					vo.getTargetYear(), vo.getTargetMonth(), dto.getHolidayCode());
			if (totalAbsenceDto != null) {
				aryDeductionId[i] = totalAbsenceDto.getTmdTotalAbsenceId();
				aryTxtTimesDeduction[i] = String.valueOf(totalAbsenceDto.getTimes());
			}
			i++;
		}
		vo.setAryTxtDeduction(aryTxtTimesDeduction);
		vo.setAryTxtDeductionTitle(aryTxtTimesDeductionTitle);
		vo.setAryTxtDeductionCode(aryTxtTimesDeductionCode);
		vo.setAryDeductionId(aryDeductionId);
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @throws MospException 例外発生時
	 */
	protected void setCorrectionVoFields() throws MospException {
		// VO取得
		TotalTimeCardVo vo = (TotalTimeCardVo)mospParams.getVo();
		TotalTimeCorrectionDtoInterface dto = timeReference().totalTimeCorrection().getLatestTotalTimeCorrectionInfo(
				vo.getPersonalId(), vo.getTargetYear(), vo.getTargetMonth());
		if (dto == null) {
			return;
		}
		StringBuffer sb = new StringBuffer();
		sb.append(mospParams.getName("Finality"));
		sb.append(mospParams.getName("Corrector"));
		sb.append(mospParams.getName("Colon"));
		sb.append(reference().human().getHumanName(dto.getCorrectionPersonalId(), dto.getCorrectionDate()));
		sb.append(" ");
		sb.append(mospParams.getName("Day"));
		sb.append(mospParams.getName("Hour"));
		sb.append(mospParams.getName("Colon"));
		sb.append(DateUtility.getStringDateAndDayAndTime(dto.getCorrectionDate()));
		vo.setLblCorrectionHistory(sb.toString());
		vo.setTxtCorrectionReason(dto.getCorrectionReason());
	}
	
	/**
	 * 時間取得。<br>
	 * @param time 対象時間
	 * @return 時間
	 */
	protected int getHour(int time) {
		return time / TimeConst.CODE_DEFINITION_HOUR;
	}
	
	/**
	 * 分取得。<br>
	 * @param time 対象時間
	 * @return 分
	 */
	protected int getMinute(int time) {
		return time % TimeConst.CODE_DEFINITION_HOUR;
	}
	
}
