/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.time.bean.WorkTypeReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dao.settings.WorkTypeDaoInterface;
import jp.mosp.time.dao.settings.WorkTypeItemDaoInterface;
import jp.mosp.time.dto.settings.WorkTypeDtoInterface;
import jp.mosp.time.dto.settings.impl.TmmWorkTypeDto;

/**
 * 勤務形態マスタ参照クラス。
 */
public class WorkTypeReferenceBean extends PlatformBean implements WorkTypeReferenceBeanInterface {
	
	/**
	 * 勤務形態マスタDAO。
	 */
	private WorkTypeDaoInterface		dao;
	
	private WorkTypeItemDaoInterface	workTypeItemDao;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public WorkTypeReferenceBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public WorkTypeReferenceBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (WorkTypeDaoInterface)createDao(WorkTypeDaoInterface.class);
		workTypeItemDao = (WorkTypeItemDaoInterface)createDao(WorkTypeItemDaoInterface.class);
	}
	
	@Override
	public WorkTypeDtoInterface getWorkTypeInfo(String workTypeCode, Date targetDate) throws MospException {
		// 勤務形態確認(法定休日出勤)
		if (workTypeCode.equals(TimeConst.CODE_WORK_ON_LEGAL_HOLIDAY)) {
			return getWorkOnLegalHolidayWorkType();
		}
		// 勤務形態確認(法定休日出勤)
		if (workTypeCode.equals(TimeConst.CODE_WORK_ON_PRESCRIBED_HOLIDAY)) {
			return getWorkOnPrescribedHolidayWorkType();
		}
		// 勤務形態マスタ情報取得
		return dao.findForInfo(workTypeCode, targetDate);
	}
	
	@Override
	public List<WorkTypeDtoInterface> getWorkTypeHistory(String workTypeCode) throws MospException {
		return dao.findForHistory(workTypeCode);
	}
	
	@Override
	public String getWorkTypeAbbr(String workTypeCode, Date targetDate) throws MospException {
		// 勤務形態確認(所定休日)
		if (workTypeCode.equals(TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY)) {
			return getPrescribedHolidayAbbrNaming();
		}
		// 勤務形態確認(法定休日)
		if (workTypeCode.equals(TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY)) {
			return getLegalHolidayAbbrNaming();
		}
		// 勤務形態マスタ情報取得及び確認
		WorkTypeDtoInterface dto = getWorkTypeInfo(workTypeCode, targetDate);
		if (dto == null) {
			// 勤務形態マスタ情報が存在しない場合
			return workTypeCode;
		}
		// 勤務形態マスタ情報略称取得
		return dto.getWorkTypeAbbr();
	}
	
	@Override
	public String getParticularWorkTypeName(String workTypeCode) {
		// 法定休日出勤の場合
		if (TimeConst.CODE_WORK_ON_LEGAL_HOLIDAY.equals(workTypeCode)) {
			return getWorkOnLegalHolidayNaming();
		}
		// 所定休日出勤の場合
		if (TimeConst.CODE_WORK_ON_PRESCRIBED_HOLIDAY.equals(workTypeCode)) {
			return getWorkOnPrescribedHolidayNaming();
		}
		// 法定休日の場合
		if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(workTypeCode)) {
			return getLegalHolidayNaming();
		}
		// 所定休日の場合
		if (TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(workTypeCode)) {
			return getPrescribedHolidayNaming();
		}
		return null;
	}
	
	@Override
	public String getWorkTypeAbbrStartTimeEndTime(String workTypeCode, Date targetDate) throws MospException {
		WorkTypeDtoInterface dto = getWorkTypeInfo(workTypeCode, targetDate);
		if (dto == null) {
			return workTypeCode;
		}
		Date standard = DateUtility.getTime(0, 0);
		Date startTime = workTypeItemDao.findForKey(dto.getWorkTypeCode(), dto.getActivateDate(),
				TimeConst.CODE_WORKSTART).getWorkTypeItemValue();
		Date endTime = workTypeItemDao.findForKey(dto.getWorkTypeCode(), dto.getActivateDate(), TimeConst.CODE_WORKEND)
			.getWorkTypeItemValue();
		return dto.getWorkTypeAbbr() + mospParams.getName("FrontWithCornerParentheses")
				+ DateUtility.getStringTime(startTime, standard) + mospParams.getName("Wave")
				+ DateUtility.getStringTime(endTime, standard) + mospParams.getName("BackWithCornerParentheses");
	}
	
	@Override
	public String[][] getSelectArray(Date targetDate) throws MospException {
		// 一覧取得
		List<WorkTypeDtoInterface> list = dao.findForActivateDate(targetDate);
		// 一覧件数確認
		if (list.size() == 0) {
			// 対象データ無し
			return getNoObjectDataPulldown();
		}
		// 配列及びインデックス宣言
		String[][] array = new String[list.size()][2];
		int idx = 0;
		// 配列作成
		for (WorkTypeDtoInterface dto : list) {
			array[idx][0] = dto.getWorkTypeCode();
			array[idx][1] = dto.getWorkTypeName();
			idx++;
		}
		return array;
	}
	
	@Override
	public String[][] getCodedSelectArray(Date targetDate) throws MospException {
		// 一覧取得
		List<WorkTypeDtoInterface> list = dao.findForActivateDate(targetDate);
		// 一覧件数確認
		if (list.size() == 0) {
			// 対象データ無し
			return getNoObjectDataPulldown();
		}
		// 配列及びインデックス宣言
		String[][] array = new String[list.size()][2];
		int idx = 0;
		// コードの最大文字数確認
		int codeLength = 0;
		for (WorkTypeDtoInterface dto : list) {
			if (dto.getWorkTypeCode().length() > codeLength) {
				codeLength = dto.getWorkTypeCode().length();
			}
		}
		// 配列作成
		for (WorkTypeDtoInterface dto : list) {
			array[idx][0] = dto.getWorkTypeCode();
			array[idx][1] = getCodedName(dto.getWorkTypeCode(), dto.getWorkTypeAbbr(), codeLength);
			idx++;
		}
		return array;
	}
	
	@Override
	public String[][] getStartTimeEndTimeSelectArray(Date targetDate) throws MospException {
		// 一覧取得
		List<WorkTypeDtoInterface> list = dao.findForActivateDate(targetDate);
		// 一覧件数確認
		if (list.size() == 0) {
			// 対象データ無し
			return getNoObjectDataPulldown();
		}
		// 配列及びインデックス宣言
		String[][] array = new String[list.size()][2];
		int idx = 0;
		// 配列作成
		for (WorkTypeDtoInterface dto : list) {
			Date standard = DateUtility.getTime(0, 0);
			Date startTime = workTypeItemDao.findForKey(dto.getWorkTypeCode(), dto.getActivateDate(),
					TimeConst.CODE_WORKSTART).getWorkTypeItemValue();
			Date endTime = workTypeItemDao.findForKey(dto.getWorkTypeCode(), dto.getActivateDate(),
					TimeConst.CODE_WORKEND).getWorkTypeItemValue();
			array[idx][0] = dto.getWorkTypeCode();
			array[idx][1] = dto.getWorkTypeAbbr() + mospParams.getName("FrontWithCornerParentheses")
					+ DateUtility.getStringTime(startTime, standard) + mospParams.getName("Wave")
					+ DateUtility.getStringTime(endTime, standard) + mospParams.getName("BackWithCornerParentheses");
			idx++;
		}
		return array;
	}
	
	@Override
	public WorkTypeDtoInterface findForKey(String workTypeCode, Date activateDate) throws MospException {
		return dao.findForKey(workTypeCode, activateDate);
	}
	
	@Override
	public String[][] getSelectAbbrArray(Date targetDate) throws MospException {
		// 一覧取得
		List<WorkTypeDtoInterface> list = dao.findForActivateDate(targetDate);
		// 一覧件数確認
		if (list.size() == 0) {
			// 対象データ無し
			return getNoObjectDataPulldown();
		}
		// 配列及びインデックス宣言
		String[][] array = new String[list.size()][2];
		int idx = 0;
		// 配列作成
		for (WorkTypeDtoInterface dto : list) {
			array[idx][0] = dto.getWorkTypeCode();
			array[idx][1] = dto.getWorkTypeAbbr();
			idx++;
		}
		return array;
	}
	
	@Override
	public WorkTypeDtoInterface findForInfo(String workTypeCode, Date activateDate) throws MospException {
		return dao.findForInfo(workTypeCode, activateDate);
	}
	
	/**
	 * 法定休日出勤勤務形態情報を取得する。<br>
	 * @return 法定休日出勤勤務形態情報
	 */
	protected WorkTypeDtoInterface getWorkOnLegalHolidayWorkType() {
		// DTO準備
		WorkTypeDtoInterface dto = getInitDto();
		// 勤務形態コード設定
		dto.setWorkTypeCode(TimeConst.CODE_WORK_ON_LEGAL_HOLIDAY);
		// 勤務形態略称設定
		dto.setWorkTypeAbbr(getWorkOnLegalHolidayAbbrNaming());
		return dto;
	}
	
	/**
	 * 所定休日出勤勤務形態情報を取得する。<br>
	 * @return 所定休日出勤勤務形態情報
	 */
	protected WorkTypeDtoInterface getWorkOnPrescribedHolidayWorkType() {
		// DTO準備
		WorkTypeDtoInterface dto = getInitDto();
		// 勤務形態コード設定
		dto.setWorkTypeCode(TimeConst.CODE_WORK_ON_PRESCRIBED_HOLIDAY);
		// 勤務形態略称設定
		dto.setWorkTypeAbbr(getWorkOnPrescribedHolidayAbbrNaming());
		return dto;
	}
	
	/**
	 * 初期化された勤務形態情報を取得する。<br>
	 * @return 初期化された勤務形態情報
	 */
	protected WorkTypeDtoInterface getInitDto() {
		// DTO準備
		WorkTypeDtoInterface dto = new TmmWorkTypeDto();
		return dto;
	}
	
	/**
	 * 所定休日略称を取得する。<br>
	 * @return 所定休日略称
	 */
	protected String getPrescribedHolidayAbbrNaming() {
		return mospParams.getName("PrescribedAbbreviation") + mospParams.getName("Holiday");
	}
	
	/**
	 * 法定休日略称を取得する。<br>
	 * @return 法定休日略称
	 */
	protected String getLegalHolidayAbbrNaming() {
		return mospParams.getName("LegalAbbreviation") + mospParams.getName("Holiday");
	}
	
	/**
	 * 所定休出略称を取得する。<br>
	 * @return 所定代休略称
	 */
	protected String getWorkOnPrescribedHolidayAbbrNaming() {
		return mospParams.getName("PrescribedAbbreviation") + mospParams.getName("WorkingHoliday");
	}
	
	/**
	 * 法定休出略称を取得する。<br>
	 * @return 法定代休略称
	 */
	protected String getWorkOnLegalHolidayAbbrNaming() {
		return mospParams.getName("LegalAbbreviation") + mospParams.getName("WorkingHoliday");
	}
	
	/**
	 * 所定休日名称を取得する。<br>
	 * @return 所定休日名称
	 */
	protected String getPrescribedHolidayNaming() {
		return mospParams.getName("Prescribed") + mospParams.getName("Holiday");
	}
	
	/**
	 * 法定休日名称を取得する。<br>
	 * @return 法定休日名称
	 */
	protected String getLegalHolidayNaming() {
		return mospParams.getName("Legal") + mospParams.getName("Holiday");
	}
	
	/**
	 * 所定休日出勤名称を取得する。<br>
	 * @return 所定休日出勤名称
	 */
	protected String getWorkOnPrescribedHolidayNaming() {
		return mospParams.getName("Prescribed") + mospParams.getName("Holiday") + mospParams.getName("GoingWork");
	}
	
	/**
	 * 法定休日出勤名称を取得する。<br>
	 * @return 法定休日出勤名称
	 */
	protected String getWorkOnLegalHolidayNaming() {
		return mospParams.getName("Legal") + mospParams.getName("Holiday") + mospParams.getName("GoingWork");
	}
	
}
