/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDto;
import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.workflow.WorkflowReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.impl.WorkflowReferenceBean;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.OvertimeRequestReferenceBeanInterface;
import jp.mosp.time.dao.settings.OvertimeRequestDaoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;

/**
 * 残業申請参照クラス。
 */
public class OvertimeRequestReferenceBean extends TimeBean implements OvertimeRequestReferenceBeanInterface {
	
	/**
	 * 残業申請マスタDAOクラス。<br>
	 */
	OvertimeRequestDaoInterface		dao;
	
	/**
	 * ワークフローDAOクラス。<br>
	 */
	WorkflowReferenceBeanInterface	workflowReference;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public OvertimeRequestReferenceBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public OvertimeRequestReferenceBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (OvertimeRequestDaoInterface)createDao(OvertimeRequestDaoInterface.class);
	}
	
	@Override
	public List<OvertimeRequestDtoInterface> getOvertimeRequestList(String personalId, Date firstDate, Date lastDate)
			throws MospException {
		return dao.findForTerm(personalId, firstDate, lastDate);
	}
	
	public OvertimeRequestDtoInterface findForKeyOnWorkflow(String personalId, Date requestDate, int overtimeType)
			throws MospException {
		return dao.findForKeyOnWorkflow(personalId, requestDate, overtimeType);
	}
	
	@Override
	public String getStatusForRequestDate(String personalId, Date requestDate) throws MospException {
		
		workflowReference = (WorkflowReferenceBean)createBean(WorkflowReferenceBean.class);
		// 対象残業申請取得
		List<OvertimeRequestDtoInterface> list = dao.findForList(personalId, requestDate);
		// 申請状態カウント
		int cont = 0;
		// 申請していない場合
		if (list.size() == 0) {
			// 空欄を返す
			return "";
		}
		// 申請状況確認
		for (int i = 0; i < list.size(); i++) {
			OvertimeRequestDtoInterface dto = list.get(i);
			if (workflowReference.getLatestWorkflowInfo(dto.getWorkflow()).getWorkflowStage() == 10
					&& workflowReference.getLatestWorkflowInfo(dto.getWorkflow()).getWorkflowStatus().equals("9")) {
				cont++;
			}
		}
		// 返す場合、文字の最後に半角スペースを入れる
		if (cont == list.size()) {
			return mospParams.getName("Remainder") + mospParams.getName("Finish") + " ";
		} else {
			return mospParams.getName("Remainder") + mospParams.getName("Register") + " ";
		}
	}
	
	@Override
	public OvertimeRequestDtoInterface findForId(long id) throws MospException {
		BaseDtoInterface baseDto = findForKey(dao, id, false);
		checkExclusive(baseDto);
		if (mospParams.hasErrorMessage()) {
			return null;
		}
		return (OvertimeRequestDtoInterface)baseDto;
	}
	
	@Override
	public OvertimeRequestDtoInterface findForWorkflow(long workflow) throws MospException {
		return dao.findForWorkflow(workflow);
	}
	
	@Override
	public List<OvertimeRequestDtoInterface> getListForWorkflowStatus(String personalId, int workflowStage,
			String workflowStatus, String routeCode) throws MospException {
		return dao.findForWorkflowStatus(personalId, workflowStage, workflowStatus, routeCode);
	}
	
	@Override
	public List<OvertimeRequestDtoInterface> getListForNotApproved(String personalId, int approvalStage)
			throws MospException {
		List<OvertimeRequestDtoInterface> list = new ArrayList<OvertimeRequestDtoInterface>();
		if (approvalStage > 0) {
			if (approvalStage == 1) {
				// 未承認
				list.addAll(getListForWorkflowStatus(personalId, approvalStage, "1", ""));
			} else {
				// (stage -1)次承認済
				list.addAll(getListForWorkflowStatus(personalId, approvalStage - 1, "2", ""));
			}
			// (stage +1)次差戻
			list.addAll(getListForWorkflowStatus(personalId, approvalStage + 1, "3", ""));
		}
		return list;
	}
	
	@Override
	public OvertimeRequestDtoInterface findForKey(long id) throws MospException {
		BaseDto dto = findForKey(dao, id, false);
		if (dto == null) {
			return null;
		}
		return (OvertimeRequestDtoInterface)dto;
	}
	
	@Override
	public void chkBasicInfo(String personalId, Date targetDate) throws MospException {
		initial(personalId, targetDate);
	}
}
