/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.bean.impl;

import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.constant.MospConst;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.framework.utils.MospUtility;
import jp.mosp.platform.bean.human.RetirementReferenceBeanInterface;
import jp.mosp.platform.bean.human.SuspensionReferenceBeanInterface;
import jp.mosp.platform.bean.portal.HolidayBeanInterface;
import jp.mosp.platform.bean.system.SectionReferenceBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.dto.human.HumanDtoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.bean.ApprovalInfoReferenceBeanInterface;
import jp.mosp.time.bean.AttendanceBean;
import jp.mosp.time.bean.AttendanceCorrectionReferenceBeanInterface;
import jp.mosp.time.bean.AttendanceListReferenceBeanInterface;
import jp.mosp.time.bean.CutoffReferenceBeanInterface;
import jp.mosp.time.bean.DifferenceRequestReferenceBeanInterface;
import jp.mosp.time.bean.HolidayReferenceBeanInterface;
import jp.mosp.time.bean.HolidayRequestReferenceBeanInterface;
import jp.mosp.time.bean.LimitStandardReferenceBeanInterface;
import jp.mosp.time.bean.OvertimeRequestReferenceBeanInterface;
import jp.mosp.time.bean.SubHolidayRequestReferenceBeanInterface;
import jp.mosp.time.bean.TotalTimeEmployeeTransactionReferenceBeanInterface;
import jp.mosp.time.bean.WorkOnHolidayRequestReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.constant.TimeMessageConst;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.AttendanceCorrectionDtoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.CutoffDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.LimitStandardDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.PaidHolidayDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDateDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.WorkTypeItemDtoInterface;
import jp.mosp.time.dto.settings.impl.AttendanceListDto;
import jp.mosp.time.utils.TimeUtility;

/**
 * 勤怠一覧参照クラス。<br>
 * <br>
 * 締期間における勤怠一覧情報を作成する。<br>
 * 勤怠一覧画面、予定確認画面、出勤簿、予定簿等で用いられる。<br>
 * <br>
 * 勤怠一覧情報作成後に作成情報を取得するためのアクセサメソッドを備える。
 * <br>
 * <br>
 * 勤怠一覧情報を対象日指定、或いは年月指定で取得することができる。<br>
 * 勤怠一覧情報を取得する手順は、以下の通り。<br>
 * <br>
 * 個人ID及び対象日(年月指定の場合は年月の最終日)で、設定適用情報及び締日を取得する。<br>
 * 対象日及び締日から、対象年月を算出する(対象日指定の場合のみ)。<br>
 * 対象年月及び締日から、締期間を算出する。<br>
 * 個人ID及び締期間で、勤怠一覧情報を取得する。<br>
 * <br>
 * 週単位で勤怠一覧情報を取得する場合は、締期間の算出方法が変わる。<br>
 */
public class AttendanceListReferenceBean extends AttendanceBean implements AttendanceListReferenceBeanInterface {
	
	/**
	 * 時間表示時の小数点以下の桁数。<br>
	 */
	public static final int											HOURS_DIGITS				= 2;
	
	/**
	 * 時間表示時の区切文字。<br>
	 */
	public static final String										SEPARATOR_HOURS				= ".";
	
	/**
	 * スタイル文字列(赤)。
	 */
	public static final String										STYLE_RED					= "style=\"color: red\"";
	
	/**
	 * スタイル文字列(青)。
	 */
	public static final String										STYLE_BLUE					= "style=\"color: blue\"";
	
	/**
	 * スタイル文字列(黄)。
	 */
	public static final String										STYLE_YELLOW				= "style=\"color: darkorange\"";
	
	/**
	 * 区切文字(勤怠備考)。
	 */
	public static final String										SEPARATOR_ATTENDANCE_REMARK	= " ";
	
	/**
	 * 限度基準期間(1ヶ月)。
	 */
	public static final String										LIMIT_STANDARD_TERM_MONTH1	= "month1";
	
	/**
	 * 有給休暇設定マスタ参照クラス。
	 */
	protected PaidHolidayReferenceBean								paidHoliday;
	
	/**
	 * 勤怠設定限度基準情報参照クラス。
	 */
	protected LimitStandardReferenceBeanInterface					limitStandard;
	
	/**
	 * 締日マスタ参照クラス。
	 */
	protected CutoffReferenceBeanInterface							cutoff;
	
	/**
	 * 勤怠修正情報参照クラス。
	 */
	protected AttendanceCorrectionReferenceBeanInterface			correction;
	
	/**
	 * 休暇申請参照クラス。
	 */
	protected HolidayRequestReferenceBeanInterface					holidayRequest;
	
	/**
	 * 残業申請参照クラス。
	 */
	protected OvertimeRequestReferenceBeanInterface					overtime;
	
	/**
	 * 代休申請参照クラス。
	 */
	protected SubHolidayRequestReferenceBeanInterface				subHoliday;
	
	/**
	 * 時差出勤申請参照クラス。
	 */
	protected DifferenceRequestReferenceBeanInterface				difference;
	
	/**
	 * 所属マスタ参照クラス。
	 */
	protected SectionReferenceBeanInterface							section;
	
	/**
	 * 休暇マスタ参照クラス。
	 */
	protected HolidayReferenceBeanInterface							holiday;
	
	/**
	 * 人事退職情報参照クラス。
	 */
	protected RetirementReferenceBeanInterface						retirement;
	
	/**
	 * 人事休職情報参照クラス。
	 */
	protected SuspensionReferenceBeanInterface						suspension;
	
	/**
	 * 祝日クラス。
	 */
	protected HolidayBeanInterface									holidayBean;
	
	/**
	 * 設定適用情報。
	 */
	protected ApplicationDtoInterface								applicationDto;
	
	/**
	 * 勤怠設定情報。
	 */
	protected TimeSettingDtoInterface								timeSettingDto;
	
	/**
	 * 締日情報。
	 */
	protected CutoffDtoInterface									cutoffDto;
	
	/**
	 * カレンダ情報。
	 */
	protected ScheduleDtoInterface									scheduleDto;
	
	/**
	 * カレンダ日情報リスト。
	 */
	List<ScheduleDateDtoInterface>									scheduleDateList;
	
	/**
	 * 社員勤怠集計管理参照クラス。
	 */
	protected TotalTimeEmployeeTransactionReferenceBeanInterface	totalTimeEmployee;
	
	/**
	 * 取得対象個人ID。<br>
	 * インターフェースに定義されたメソッドの最初で設定される。<br>
	 */
	protected String												personalId;
	
	/**
	 * 対象年。<br>
	 */
	protected int													targetYear;
	
	/**
	 * 対象月。<br>
	 */
	protected int													targetMonth;
	
	/**
	 * 締期間初日。<br>
	 * {@link #setCutoffTerm()}により設定される。<br>
	 */
	protected Date													firstDate;
	
	/**
	 * 締期間最終日。<br>
	 * {@link #setCutoffTerm()}により設定される。<br>
	 */
	protected Date													lastDate;
	
	/**
	 * 勤怠情報リスト。<br>
	 */
	protected List<AttendanceDtoInterface>							attendanceDtoList;
	
	/**
	 * 休暇申請情報リスト。<br>
	 */
	protected List<HolidayRequestDtoInterface>						holidayRequestList;
	
	/**
	 * 振替休日情報リスト。<br>
	 */
	protected List<SubstituteDtoInterface>							substituteList;
	
	/**
	 * 代休申請情報リスト。<br>
	 */
	protected List<SubHolidayRequestDtoInterface>					subHolidayRequestList;
	
	/**
	 * 休出申請情報リスト
	 */
	protected List<WorkOnHolidayRequestDtoInterface>				workOnHolidayRequestList;
	
	/**
	 * 勤怠一覧情報リスト。<br>
	 * 各メソッドは、ここに勤怠一覧情報を追加していく。<br>
	 */
	protected List<AttendanceListDto>								attendanceList;
	

	@Override
	public void initBean() throws MospException {
		// 継承元クラスのメソッドを実行
		super.initBean();
		// 参照クラス準備
		paidHoliday = (PaidHolidayReferenceBean)createBean(PaidHolidayReferenceBean.class);
		limitStandard = (LimitStandardReferenceBeanInterface)createBean(LimitStandardReferenceBeanInterface.class);
		cutoff = (CutoffReferenceBeanInterface)createBean(CutoffReferenceBeanInterface.class);
		correction = (AttendanceCorrectionReferenceBeanInterface)createBean(AttendanceCorrectionReferenceBeanInterface.class);
		holidayRequest = (HolidayRequestReferenceBeanInterface)createBean(HolidayRequestReferenceBeanInterface.class);
		overtime = (OvertimeRequestReferenceBean)createBean(OvertimeRequestReferenceBean.class);
		workOnHoliday = (WorkOnHolidayRequestReferenceBeanInterface)createBean(WorkOnHolidayRequestReferenceBeanInterface.class);
		subHoliday = (SubHolidayRequestReferenceBeanInterface)createBean(SubHolidayRequestReferenceBeanInterface.class);
		difference = (DifferenceRequestReferenceBeanInterface)createBean(DifferenceRequestReferenceBeanInterface.class);
		section = (SectionReferenceBeanInterface)createBean(SectionReferenceBeanInterface.class);
		holiday = (HolidayReferenceBeanInterface)createBean(HolidayReferenceBeanInterface.class);
		retirement = (RetirementReferenceBeanInterface)createBean(RetirementReferenceBeanInterface.class);
		suspension = (SuspensionReferenceBeanInterface)createBean(SuspensionReferenceBeanInterface.class);
		totalTimeEmployee = (TotalTimeEmployeeTransactionReferenceBeanInterface)createBean(TotalTimeEmployeeTransactionReferenceBeanInterface.class);
		// 祝日クラス準備
		holidayBean = (HolidayBeanInterface)createBean(HolidayBeanInterface.class);
	}
	
	@Override
	public List<AttendanceListDto> getScheduleList(String personalId, Date targetDate) throws MospException {
		// フィールド初期化及び対象個人ID設定
		initFields(personalId);
		// 対象年月及び締期間最終日を仮設定
		initDateFields(targetDate);
		// 勤怠設定情報群を取得
		setApplicationSettings();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 対象日及び締日から対象年月を算出し設定
		setTargetYearMonth(targetDate);
		// 締期間設定
		setCutoffTerm();
		// 勤怠一覧情報リストを初期化
		initAttendanceList();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// カレンダ情報を取得し勤怠一覧情報リストに設定(勤務時間、休憩時間含む)
		addScheduleList(true, true);
		// 申請情報を取得し勤怠一覧情報リストに設定(未承認情報は含めない)
		addApplicationList(false, false);
		// 申請(承認済)によって予定勤務時間表示無し
		setApprovalTime();
		// 勤怠一覧情報リストを集計
		totalAttendanceList();
		// 勤怠一覧情報に表示用文字列を設定
		setDtoStringFields();
		// 帳票用ヘッダー項目設定
		setHeaderFields();
		return attendanceList;
	}
	
	@Override
	public List<AttendanceListDto> getScheduleList(String personalId, int year, int month) throws MospException {
		// フィールド初期化及び対象個人ID設定
		initFields(personalId);
		// 対象年月及び締期間最終日を仮設定
		initDateFields(year, month);
		// 勤怠設定情報群を取得
		setApplicationSettings();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 締期間設定
		setCutoffTerm();
		// 勤怠一覧情報リストを初期化
		initAttendanceList();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// カレンダ情報を取得し勤怠一覧情報リストに設定(勤務時間、休憩時間含む)
		addScheduleList(true, true);
		// 申請情報を取得し勤怠一覧情報リストに設定(未承認情報は含めない)
		addApplicationList(false, false);
		// 申請(承認済)によって予定勤務時間表示無し
		setApprovalTime();
		// 勤怠一覧情報リストを集計
		totalAttendanceList();
		// 勤怠一覧情報に表示用文字列を設定
		setDtoStringFields();
		// 帳票用ヘッダー項目設定
		setHeaderFields();
		return attendanceList;
	}
	
	@Override
	public List<AttendanceListDto> getActualList(String personalId, Date targetDate) throws MospException {
		// フィールド初期化及び対象個人ID設定
		initFields(personalId);
		// 対象年月及び締期間最終日を仮設定
		initDateFields(targetDate);
		// 勤怠設定情報群を取得
		setApplicationSettings();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 対象日及び締日から対象年月を算出し設定
		setTargetYearMonth(targetDate);
		// 締期間設定
		setCutoffTerm();
		// 勤怠一覧情報リストを初期化
		initAttendanceList();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 勤怠情報を取得し勤怠一覧情報リストに設定(未承認情報は含めない)
		addActualList(false);
		// 申請情報を取得し勤怠一覧情報リストに設定(未承認情報は含めない)
		addApplicationList(false, true);
		// カレンダ情報を取得し勤怠一覧情報リストに設定(休日のみ)
		addScheduleList(false, false);
		// 勤怠一覧情報リストを集計
		totalAttendanceList();
		// 勤怠一覧情報に表示用文字列を設定
		setDtoStringFields();
		// 帳票用ヘッダー項目設定
		setHeaderFields();
		
		return attendanceList;
	}
	
	@Override
	public List<AttendanceListDto> getActualList(String personalId, int year, int month) throws MospException {
		// フィールド初期化及び対象個人ID設定
		initFields(personalId);
		// 対象年月及び締期間最終日を仮設定
		initDateFields(year, month);
		// 勤怠設定情報群を取得
		setApplicationSettings();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 対象個人ID及び対象年月で締期間を取得
		setCutoffTerm();
		// 締期間設定
		setCutoffTerm();
		// 勤怠一覧情報リストを初期化
		initAttendanceList();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		
		// 勤怠情報を取得し勤怠一覧情報リストに設定(未承認情報は含めない)
		addActualList(false);
		// 申請情報を取得し勤怠一覧情報リストに設定(未承認情報は含めない)
		addApplicationList(false, true);
		// カレンダ情報を取得し勤怠一覧情報リストに設定(休日のみ)
		addScheduleList(false, false);
		// 勤怠一覧情報リストを集計
		totalAttendanceList();
		// 勤怠一覧情報に表示用文字列を設定
		setDtoStringFields();
		// 帳票用ヘッダー項目設定
		setHeaderFields();
		return attendanceList;
	}
	
	@Override
	public List<AttendanceListDto> getAttendanceList(String personalId, Date targetDate) throws MospException {
		// フィールド初期化及び対象個人ID設定
		initFields(personalId);
		// 対象年月及び締期間最終日を仮設定
		initDateFields(targetDate);
		// 勤怠設定情報群を取得
		setApplicationSettings();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 対象日及び締日から対象年月を算出し設定
		setTargetYearMonth(targetDate);
		// 締期間設定
		setCutoffTerm();
		// 勤怠一覧情報リストを初期化
		initAttendanceList();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 勤怠情報を取得し勤怠一覧情報リストに設定(未承認情報を含める)
		addActualList(true);
		// 申請情報を取得し勤怠一覧情報リストに設定(未承認情報を含める)
		addApplicationList(true, false);
		// 勤怠情報(下書)を取得し勤怠一覧情報リストに設定
		addDraftList();
		// カレンダ情報を取得し勤怠一覧情報リストに設定(勤務時間、休憩時間を除く)
		addScheduleList(true, false);
		// 限度基準情報設定
		setLimitStandard();
		// 勤怠一覧情報リストを集計
		totalAttendanceList();
		// 勤怠一覧情報に表示用文字列を設定
		setDtoStringFields();
		// 勤怠一覧情報の特定の項目にハイフンを設定
		setDtoHyphenFields();
		// チェックボックス設定
		setNeedCheckbox();
		return attendanceList;
	}
	
	@Override
	public List<AttendanceListDto> getAttendanceList(String personalId, int year, int month) throws MospException {
		// フィールド初期化及び対象個人ID設定
		initFields(personalId);
		// 対象年月及び締期間最終日を仮設定
		initDateFields(year, month);
		// 勤怠設定情報群を取得
		setApplicationSettings();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 締期間設定
		setCutoffTerm();
		// 勤怠一覧情報リストを初期化
		initAttendanceList();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 勤怠情報を取得し勤怠一覧情報リストに設定(未承認情報を含める)
		addActualList(true);
		// 申請情報を取得し勤怠一覧情報リストに設定(未承認情報を含める)
		addApplicationList(true, false);
		// 勤怠情報(下書)を取得し勤怠一覧情報リストに設定
		addDraftList();
		// カレンダ情報を取得し勤怠一覧情報リストに設定(勤務時間、休憩時間を除く)
		addScheduleList(true, false);
		// 限度基準情報設定
		setLimitStandard();
		// 勤怠一覧情報リストを集計
		totalAttendanceList();
		// 勤怠一覧情報に表示用文字列を設定
		setDtoStringFields();
		// 勤怠一覧情報の特定の項目にハイフンを設定
		setDtoHyphenFields();
		// チェックボックス設定
		setNeedCheckbox();
		return attendanceList;
	}
	
	@Override
	public List<AttendanceListDto> getWeeklyAttendanceList(String personalId, Date targetDate) throws MospException {
		// フィールド初期化及び対象個人ID設定
		initFields(personalId);
		// 対象年月及び締期間最終日を仮設定
		initDateFields(targetDate);
		// 勤怠設定情報群を取得
		setApplicationSettings();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 対象日から期間(週)を設定
		setWeekTerm(targetDate);
		// 勤怠一覧情報リストを初期化
		initAttendanceList();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 勤怠情報を取得し勤怠一覧情報リストに設定(未承認情報を含める)
		addActualList(true);
		// 申請情報を取得し勤怠一覧情報リストに設定(未承認情報を含める)
		addApplicationList(true, false);
		// 勤怠情報(下書)を取得し勤怠一覧情報リストに設定
		addDraftList();
		// カレンダ情報を取得し勤怠一覧情報リストに設定(勤務時間、休憩時間を除く)
		addScheduleList(true, false);
		// 勤怠一覧情報リストを集計
		totalAttendanceList();
		// 勤怠一覧情報に表示用文字列を設定
		setDtoStringFields();
		return attendanceList;
	}
	
	@Override
	public List<AttendanceListDto> getApprovalAttendanceList(String personalId, int year, int month)
			throws MospException {
		// フィールド初期化及び対象個人ID設定
		initFields(personalId);
		// 対象年月及び締期間最終日を仮設定
		initDateFields(year, month);
		// 勤怠設定情報群を取得
		setApplicationSettings();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		// 締期間設定
		setCutoffTerm();
		// 勤怠一覧情報リストを初期化
		initAttendanceList();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return attendanceList;
		}
		
		// 勤怠情報を取得し勤怠一覧情報リストに設定(未承認情報を含める)
		addActualList(true);
		// 申請情報を取得し勤怠一覧情報リストに設定(未承認情報を含める)
		addApplicationList(true, true);
		// カレンダ情報を取得し勤怠一覧情報リストに設定(休日のみ)
		addScheduleList(false, false);
		// 限度基準情報設定
		setLimitStandard();
		// 勤怠一覧情報リストを集計
		totalAttendanceList();
		// 勤怠一覧情報に表示用文字列を設定
		setDtoStringFields();
		// 勤怠一覧情報の特定の項目にハイフンを設定
		setDtoHyphenFields();
		// チェックボックス設定
		setNeedApprovalCheckbox();
		return attendanceList;
	}
	
	@Override
	public AttendanceListDto getAttendanceListDto(String personalId, Date targetDate) throws MospException {
		// フィールド初期化及び対象個人ID設定
		initFields(personalId);
		// 対象年月及び締期間最終日を仮設定
		initDateFields(targetDate);
		// 勤怠設定情報群を取得
		setApplicationSettings();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return null;
		}
		// 対象日を設定
		firstDate = targetDate;
		lastDate = targetDate;
		// 勤怠一覧情報リストを初期化
		initAttendanceList();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			return null;
		}
		// 勤怠情報を取得し勤怠一覧情報リストに設定(未承認情報を含める)
		addActualList(true);
		// 申請情報を取得し勤怠一覧情報リストに設定(未承認情報を含める)
		addApplicationList(true, false);
		// 勤怠情報(下書)を取得し勤怠一覧情報リストに設定
		addDraftList();
		// カレンダ情報を取得し勤怠一覧情報リストに設定(勤務時間、休憩時間を除く)
		addScheduleList(true, false);
		// 勤怠一覧情報リストを集計
		totalAttendanceList();
		// 勤怠一覧情報に表示用文字列を設定
		setDtoStringFields();
		
		if (attendanceList.size() == 1) {
			return attendanceList.get(0);
		}
		return null;
	}
	
	@Override
	public String getPersonalId() {
		return personalId;
	}
	
	@Override
	public Date getLastDate() {
		return getDateClone(lastDate);
	}
	
	@Override
	public int getTargetYear() {
		return targetYear;
	}
	
	@Override
	public int getTargetMonth() {
		return targetMonth;
	}
	
	@Override
	public String getScheduleName() {
		return scheduleDto == null ? "" : scheduleDto.getScheduleName();
	}
	
	/**
	 * フィールド初期化及び対象個人ID設定を行う。<br>
	 * @param personalId 対象個人ID
	 */
	protected void initFields(String personalId) {
		// 設定適用情報初期化
		applicationDto = null;
		// 勤怠設定情報初期化
		timeSettingDto = null;
		// 締日情報初期化
		cutoffDto = null;
		// カレンダ情報初期化
		scheduleDto = null;
		// カレンダ日情報リスト初期化
		scheduleDateList = null;
		// 締期間初日初期化
		firstDate = null;
		// 締期間最終日初期化
		lastDate = null;
		// 勤怠一覧情報リスト初期化
		attendanceList = new ArrayList<AttendanceListDto>();
		// 個人ID設定
		this.personalId = personalId;
	}
	
	/**
	 * 対象日で対象年月及び締期間最終日を仮設定する。<br>
	 * @param targetDate 対象日
	 */
	protected void initDateFields(Date targetDate) {
		// 対象年月設定(対象日の年月を仮設定)
		// TODO 変更予定
		targetYear = DateUtility.getYear(targetDate);
		targetMonth = DateUtility.getMonth(targetDate);
		// 締期間最終日設定(対象日を仮設定)
		lastDate = targetDate;
	}
	
	/**
	 * 対象年月で対象年月を設定、締期間最終日を仮設定する。<br>
	 * @param year  対象年
	 * @param month 対象月
	 * @throws MospException 日付操作に失敗した場合
	 */
	protected void initDateFields(int year, int month) throws MospException {
		// 対象年月設定
		targetYear = year;
		targetMonth = month;
		// 締期間最終日設定(年月の最終日を仮設定)
		lastDate = DateUtility.getLastDateOfMonth(year, month);
	}
	
	/**
	 * 帳票用ヘッダー項目を設定する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setHeaderFields() throws MospException {
		// 最初の勤怠一覧レコードを取得
		AttendanceListDto dto = attendanceList.get(0);
		// 人事情報取得
		HumanDtoInterface humanDto = getHumanInfo(personalId, lastDate);
		// 年月設定
		dto.setTargetDate(DateUtility.getFirstDateOfMonth(targetYear, targetMonth));
		// 社員氏名設定
		dto.setEmployeeName(MospUtility.getHumansName(humanDto.getFirstName(), humanDto.getLastName()));
		// 所属名設定
		dto.setSectionName(section.getSectionName(humanDto.getSectionCode(), lastDate));
		if (section.useDisplayName()) {
			// 所属表示名称を設定
			dto.setSectionName(section.getSectionDisplay(humanDto.getSectionCode(), lastDate));
		}
	}
	
	// TODO TimeApplicationBeanに作成
	/**
	 * 対象個人ID及び締期間最終日で勤怠設定情報群を取得し、設定する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setApplicationSettings() throws MospException {
		// 勤怠一覧情報リスト初期化
		attendanceList = new ArrayList<AttendanceListDto>();
		// 基準日における取得対象個人IDの設定適用情報を取得
		applicationDto = applicationReference.findForPerson(personalId, lastDate);
		// 設定適用情報確認
		if (applicationDto == null) {
			// エラーメッセージ設定
			addApplicationNotExistErrorMessage(lastDate);
			return;
		}
		// 基準日における勤怠設定情報を取得
		timeSettingDto = timeSettingReference.getTimeSettingInfo(applicationDto.getWorkSettingCode(), lastDate);
		// 勤怠設定情報確認
		if (timeSettingDto == null) {
			// エラーメッセージ設定
			addTimeSettingNotExistErrorMessage(lastDate);
			return;
		}
		// 基準日における締日情報を取得
		cutoffDto = cutoff.getCutoffInfo(timeSettingDto.getCutoffCode(), lastDate);
		// 締日情報確認
		if (cutoffDto == null) {
			// エラーメッセージ設定
			addCutoffNotExistErrorMessage(lastDate);
			return;
		}
	}
	
	/**
	 * 対象日及び締日から対象年月を算出し、設定する。<br>
	 * @param targetDate 対象日
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setTargetYearMonth(Date targetDate) throws MospException {
		// 締日取得
		int cutoffDate = cutoffDto.getCutoffDate();
		// 対象日及び締日から対象年月を算出
		Date yearMonth = TimeUtility.getCutoffMonth(cutoffDate, targetDate);
		// 対象年月設定
		targetYear = DateUtility.getYear(yearMonth);
		targetMonth = DateUtility.getMonth(yearMonth);
	}
	
	/**
	 * 対象年月及び締日から締期間を算出し、設定する。<br>
	 * 勤怠一覧画面等の期間で勤怠情報を取得する場合は、この締期間を利用する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setCutoffTerm() throws MospException {
		// 締日取得
		int cutoffDate = cutoffDto.getCutoffDate();
		// 締期間初日設定
		firstDate = TimeUtility.getCutoffFirstDate(cutoffDate, targetYear, targetMonth);
		// 締期間最終日設定
		lastDate = TimeUtility.getCutoffLastDate(cutoffDate, targetYear, targetMonth);
	}
	
	/**
	 * 勤怠一覧情報リストを初期化する。<br>
	 * 締期間初日～締期間最終日の勤怠一覧情報を作成し、初期化する。
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void initAttendanceList() throws MospException {
		// 初期勤怠一覧情報リスト準備
		attendanceList = new ArrayList<AttendanceListDto>();
		// 勤怠日準備(締期間初日)
		Date workDate = getDateClone(firstDate);
		// 締期間初日から締期間最終日までの初期勤怠一覧情報を作成
		while (workDate.after(lastDate) == false) {
			// 勤怠一覧情報準備
			AttendanceListDto dto = new AttendanceListDto();
			// 初期情報設定
			dto.setWorkDate(workDate);
			// 個人ID(頁単位項目)設定
			dto.setPersonalId(personalId);
			// 初期勤怠一覧情報リストに追加
			attendanceList.add(dto);
			// 勤怠日加算
			workDate = addDay(workDate, 1);
		}
	}
	
	/**
	 * 申請情報を取得し勤怠一覧情報リストに設定する。<br>
	 * 同時に、申請情報の集計を行う。<br>
	 * 実績一覧(出勤簿等)でない場合、申請によって、
	 * 申請日の勤怠が下書き状態であれば下書き情報をリストに加える。<br>
	 * @param containNotApproved 未承認要否フラグ(true：下書等未承認申請も含める、false：承認済のみ)
	 * @param isActualList 実績一覧フラグ(true：実績一覧、false：実績一覧ではない)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void addApplicationList(boolean containNotApproved, boolean isActualList) throws MospException {
		// 休暇申請情報設定
		addHolidayRequestList(containNotApproved);
		// 休日出勤申請情報設定
		addWorkOnHolidayRequestList(containNotApproved, isActualList);
		// 振替休日申請情報設定
		addSubstituteList(containNotApproved);
		// 代休申請情報設定
		addSubHolidayRequestList(containNotApproved);
		// 残業申請情報設定
		addOvertimeRequestList(containNotApproved);
		// 時差出勤申請情報設定
		addDifferenceRequestList(containNotApproved, isActualList);
		// 時間単位年休利用確認
		confirmTimelyPaidHoliday();
		// 同日複数申請確認
		checkPluralRequest(containNotApproved);
	}
	
	/**
	 * 休暇申請情報を取得し勤怠一覧情報リストに設定する。<br>
	 * 同時に、休暇申請情報の集計を行う。<br>
	 * @param containNotApproved 未承認要否フラグ(true：未承認申請も含める、false：承認済のみ)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void addHolidayRequestList(boolean containNotApproved) throws MospException {
		// 集計値準備
		// 有給休暇回数
		float paidHolidays = 0F;
		// 有給時間
		float paidHolidayTime = 0F;
		// 特別休暇回数
		float specialHolidays = 0F;
		// その他休暇回数
		float otherHolidays = 0F;
		// 欠勤回数
		float absenceDays = 0F;
		// 休暇申請情報取得
		holidayRequestList = holidayRequest.getHolidayRequestList(personalId, firstDate, lastDate);
		// 休暇申請毎に処理
		for (HolidayRequestDtoInterface holidayRequestDto : holidayRequestList) {
			// ワークフロー情報取得
			WorkflowDtoInterface workflowDto = workflow.getLatestWorkflowInfo(holidayRequestDto.getWorkflow());
			// 申請情報要否確認
			if (isRequestNeeded(workflowDto, containNotApproved) == false) {
				if (containNotApproved && isDraft(workflowDto)) {
					// 下書の場合
					// 対象日(申請開始日)取得
					Date targetDate = holidayRequestDto.getRequestStartDate();
					// 申請終了日取得
					Date endDate = holidayRequestDto.getRequestEndDate();
					// 開始日から終了日まで日毎に処理
					while (!endDate.before(targetDate)) {
						// 設定適用情報を取得
						ApplicationDtoInterface applicationDto = applicationReference.findForPerson(personalId,
								targetDate);
						// 設定適用情報確認
						if (applicationDto == null) {
							// エラーメッセージ設定
							addApplicationNotExistErrorMessage(targetDate);
							return;
						}
						// カレンダ情報を取得
						ScheduleDtoInterface scheduleDto = scheduleReference.getScheduleInfo(applicationDto
							.getScheduleCode(), targetDate);
						// カレンダ情報確認
						if (scheduleDto == null) {
							// エラーメッセージ設定
							addScheduleNotExistErrorMessage(targetDate);
							return;
						}
						// カレンダ日情報取得及び確認
						ScheduleDateDtoInterface scheduleDateDto = scheduleDateReference.getScheduleDateInfo(
								scheduleDto.getScheduleCode(), scheduleDto.getActivateDate(), targetDate);
						if (scheduleDateDto == null) {
							// エラーメッセージ設定
							addScheduleNotExistErrorMessage(targetDate);
							return;
						}
						if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())
								|| TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())) {
							// 対象日加算
							targetDate = addDay(targetDate, 1);
							continue;
						}
						// 対象日における勤怠一覧情報を取得
						AttendanceListDto dto = getAttendanceListDto(targetDate);
						// 対象日加算
						targetDate = addDay(targetDate, 1);
						// 対象日における勤怠一覧情報の存在を確認
						if (dto == null) {
							continue;
						}
						// 備考設定
						addRemark(dto, getHolidayNaming() + getDraftAbbrNaming());
					}
				}
				continue;
			}
			// 休暇回数取得
			float times = getHolidayTimes(holidayRequestDto);
			// 対象日(申請開始日)取得
			Date targetDate = holidayRequestDto.getRequestStartDate();
			// 申請終了日取得
			Date endDate = holidayRequestDto.getRequestEndDate();
			// 開始日から終了日まで日毎に処理
			while (!endDate.before(targetDate)) {
				// 設定適用情報を取得
				ApplicationDtoInterface applicationDto = applicationReference.findForPerson(personalId, targetDate);
				// 設定適用情報確認
				if (applicationDto == null) {
					// エラーメッセージ設定
					addApplicationNotExistErrorMessage(targetDate);
					return;
				}
				// カレンダ情報を取得
				ScheduleDtoInterface scheduleDto = scheduleReference.getScheduleInfo(applicationDto.getScheduleCode(),
						targetDate);
				// カレンダ情報確認
				if (scheduleDto == null) {
					// エラーメッセージ設定
					addScheduleNotExistErrorMessage(targetDate);
					return;
				}
				// カレンダ日情報取得及び確認
				ScheduleDateDtoInterface scheduleDateDto = scheduleDateReference.getScheduleDateInfo(scheduleDto
					.getScheduleCode(), scheduleDto.getActivateDate(), targetDate);
				if (scheduleDateDto == null) {
					// エラーメッセージ設定
					addScheduleNotExistErrorMessage(targetDate);
					return;
				}
				if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())
						|| TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())) {
					// 対象日加算
					targetDate = addDay(targetDate, 1);
					continue;
				}
				// 対象日における勤怠一覧情報を取得
				AttendanceListDto dto = getAttendanceListDto(targetDate);
				// 対象日加算
				targetDate = addDay(targetDate, 1);
				// 対象日における勤怠一覧情報の存在を確認
				if (dto == null) {
					continue;
				}
				// 休暇申請情報を勤怠一覧情報に設定
				setDtoFields(dto, holidayRequestDto, workflowDto);
				// 休暇区分確認
				switch (holidayRequestDto.getHolidayType1()) {
					case TimeConst.CODE_HOLIDAYTYPE_HOLIDAY:
						// 有給休暇の場合
						// 休暇範囲確認
						if (holidayRequestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_TIME) {
							// 有給時間加算
							paidHolidayTime += times;
						} else {
							// 有給休暇回数加算
							paidHolidays += times;
						}
						break;
					case TimeConst.CODE_HOLIDAYTYPE_SPECIAL:
						// 特別休暇の場合
						// 特別休暇回数加算
						specialHolidays += times;
						break;
					case TimeConst.CODE_HOLIDAYTYPE_OTHER:
						// その他休暇の場合
						// その他休暇回数加算
						otherHolidays += times;
						break;
					case TimeConst.CODE_HOLIDAYTYPE_ABSENCE:
						// 欠勤の場合
						// 欠勤回数加算
						absenceDays += times;
						break;
					default:
						// 処理無し
				}
			}
		}
		// 最終レコード取得
		AttendanceListDto dto = attendanceList.get(attendanceList.size() - 1);
		// 集計値設定
		// 有給休暇回数
		dto.setPaidHolidays(paidHolidays);
		// 有給時間
		dto.setPaidHolidayTime(paidHolidayTime);
		// 特別休暇回数
		dto.setSpecialHolidays(specialHolidays);
		// その他休暇回数
		dto.setOtherHolidays(otherHolidays);
		// 欠勤回数
		dto.setAbsenceDays(absenceDays);
	}
	
	/**
	 * ワークフロー情報から申請情報の要否を確認する。<br>
	 * ワークフロー状況及び未承認要否フラグで判断する。<br>
	 * @param workflowDto        ワークフロー情報
	 * @param containNotApproved 未承認要否フラグ(true：未承認申請も含める、false：承認済のみ)
	 * @return 申請情報の要否(true：必要、false：不要)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected boolean isRequestNeeded(WorkflowDtoInterface workflowDto, boolean containNotApproved)
			throws MospException {
		// ワークフロー状況取得
		String workflowStatus = workflowDto.getWorkflowStatus();
		// ワークフロー状況確認(下書の場合)
		if (isDraft(workflowDto)) {
			// 不要
			return false;
		}
		// ワークフロー状況確認(取下の場合)
		if (workflowStatus.equals(PlatformConst.CODE_STATUS_WITHDRAWN)) {
			// 不要
			return false;
		}
		// 未承認要否フラグ及びワークフロー状況確認(承認済でない場合)
		if (containNotApproved == false && workflowStatus.equals(PlatformConst.CODE_STATUS_COMPLETE) == false) {
			// 不要
			return false;
		}
		return true;
	}
	
	/**
	 * ワークフロー情報から下書かどうかを判断する。<br>
	 * @param dto ワークフロー情報
	 * @return true：下書、false：下書でない
	 */
	protected boolean isDraft(WorkflowDtoInterface dto) {
		return PlatformConst.CODE_STATUS_DRAFT.equals(dto.getWorkflowStatus());
	}
	
	/**
	 * ワークフロー情報から差戻かどうかを判断する。<br>
	 * @param dto ワークフロー情報
	 * @return true：差戻、false：差戻でない
	 */
	protected boolean isRevert(WorkflowDtoInterface dto) {
		return PlatformConst.CODE_STATUS_REVERT.equals(dto.getWorkflowStatus());
	}
	
	/**
	 * 休暇回数を取得する。<br>
	 * 休暇申請情報の休暇範囲から、休暇回数を取得する。<br>
	 * 休暇申請の場合は、時間休も考慮する。<br>
	 * @param holidayRequestDto 休暇申請情報
	 * @return 休暇回数
	 */
	protected float getHolidayTimes(HolidayRequestDtoInterface holidayRequestDto) {
		// 休暇範囲確認
		switch (holidayRequestDto.getHolidayRange()) {
			case TimeConst.CODE_HOLIDAY_RANGE_ALL:
				// 全休の場合
				return TimeConst.HOLIDAY_TIMES_ALL;
			case TimeConst.CODE_HOLIDAY_RANGE_AM:
			case TimeConst.CODE_HOLIDAY_RANGE_PM:
				// 半休の場合
				return TimeConst.HOLIDAY_TIMES_HALF;
			case TimeConst.CODE_HOLIDAY_RANGE_TIME:
				// 時間休の場合
				// 開始時刻と終了時刻の差を取得
				return getDefferenceMinutes(holidayRequestDto.getStartTime(), holidayRequestDto.getEndTime());
			default:
				return 0F;
		}
	}
	
	/**
	 * 休暇回数を取得する。<br>
	 * 休暇申請情報の休暇範囲から、休暇回数を取得する。<br>
	 * @param holidayRange 休暇範囲
	 * @return 休暇回数
	 */
	protected float getHolidayTimes(int holidayRange) {
		// 休暇範囲確認
		switch (holidayRange) {
			case TimeConst.CODE_HOLIDAY_RANGE_ALL:
				// 全休の場合
				return TimeConst.HOLIDAY_TIMES_ALL;
			case TimeConst.CODE_HOLIDAY_RANGE_AM:
			case TimeConst.CODE_HOLIDAY_RANGE_PM:
				// 半休の場合
				return TimeConst.HOLIDAY_TIMES_HALF;
			default:
				return 0F;
		}
	}
	
	/**
	 * 休暇申請情報を勤怠一覧情報に設定する。<br>
	 * ワークフロー情報は、休暇申請情報から取得したものとする。<br>
	 * @param dto               設定対象勤怠一覧情報
	 * @param holidayRequestDto 休暇申請情報
	 * @param workflowDto       ワークフロー情報
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setDtoFields(AttendanceListDto dto, HolidayRequestDtoInterface holidayRequestDto,
			WorkflowDtoInterface workflowDto) throws MospException {
		// 備考設定
		addRemark(dto, getHolidayNaming() + getWorkflowStatusAbbr(workflowDto));
		// 休暇範囲確認
		if (holidayRequestDto.getHolidayRange() != TimeConst.CODE_HOLIDAY_RANGE_ALL) {
			// 全休以外の場合は設定不要
			return;
		}
		// 勤務形態設定(空白)(予定による上書を防ぐため)
		dto.setWorkTypeCode("");
		// チェックボックス要否設定(不要)
		dto.setNeedCheckbox(false);
		// 状態設定
		dto.setApplicationInfo(workflow.getWorkflowStatus(workflowDto.getWorkflowStatus(), workflowDto
			.getWorkflowStage()));
		// 状態リンクコマンド設定
		dto.setNeedStatusLink(false);
		// 休暇申請のレコード識別IDを設定
		dto.setAttendanceRecordId(holidayRequestDto.getTmdHolidayRequestId());
		// 休暇区分毎に勤務形態略称設定
		dto.setWorkTypeAbbr(getWorkTypeAbbr(holidayRequestDto));
	}
	
	/**
	 * ワークフロー状況略称を取得する。<br>
	 * ワークフロー状況によって申、済を返す。<br>
	 * @param workflowDto ワークフロー情報
	 * @return ワークフロー状況略称
	 */
	protected String getWorkflowStatusAbbr(WorkflowDtoInterface workflowDto) {
		if (workflowDto == null) {
			return "";
		}
		if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
			return "";
		}
		// ワークフロー状況確認(下書の場合)
		if (isDraft(workflowDto)) {
			return getDraftAbbrNaming();
		}
		// ワークフロー状況確認(承認済の場合)
		if (PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
			return getCompleteApprovalAbbrNaming();
		}
		// ワークフロー状況確認(1次戻の場合)
		if (isRevert(workflowDto) && workflowDto.getWorkflowStage() == 0) {
			// 1次戻の場合
			return getRevertAbbrNaming();
		}
		return getApprovalAbbrNaming();
	}
	
	/**
	 * 休日出勤申請の略称(備考用)を取得する。
	 * @param dto 休日出勤申請情報
	 * @return 休日出勤の略称(備考用)
	 */
	protected String getWorkOnHolidayAbbr(WorkOnHolidayRequestDtoInterface dto) {
		// 振替申請を取得
		int substitute = dto.getSubstitute();
		// 振替申請の場合
		if (substitute == TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_ON) {
			// 振替出勤略称を設定
			return getSubstituteWorkAbbrNaming();
		}
		// 休日出勤申請の場合
		if (substitute == TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_OFF) {
			// 休日出勤略称を設定
			return getWorkOnHolidayAbbrNaming();
		}
		return "";
	}
	
	/**
	 * 休日出勤申請情報を取得し勤怠一覧情報リストに設定する。<br>
	 * @param containNotApproved 未承認要否フラグ(true：未承認申請も含める、false：承認済のみ)
	 * @param isActualList 実績一覧フラグ(true：実績一覧、false：実績一覧ではない)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void addWorkOnHolidayRequestList(boolean containNotApproved, boolean isActualList) throws MospException {
		// 休日出勤申請情報取得
		workOnHolidayRequestList = workOnHoliday.getWorkOnHolidayRequestList(personalId, firstDate, lastDate);
		// 休日出勤申請毎に処理
		for (WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto : workOnHolidayRequestList) {
			// 対象日における勤怠一覧情報を取得
			AttendanceListDto dto = getAttendanceListDto(workOnHolidayRequestDto.getRequestDate());
			// 休日出勤略称を設定
			String requestAbbr = getWorkOnHolidayAbbr(workOnHolidayRequestDto);
			// ワークフロー情報取得
			WorkflowDtoInterface workflowDto = workflow.getLatestWorkflowInfo(workOnHolidayRequestDto.getWorkflow());
			// 申請情報要否確認
			if (isRequestNeeded(workflowDto, containNotApproved) == false) {
				if (containNotApproved && isDraft(workflowDto)) {
					// 下書の場合
					addRemark(dto, requestAbbr + getDraftAbbrNaming());
				}
				continue;
			}
			// 備考設定
			addRemark(dto, requestAbbr + getWorkflowStatusAbbr(workflowDto));
			// 勤務形態確認
			if (dto.getWorkTypeCode() != null) {
				// 勤務形態が設定されている場合(実績が存在する場合)
				continue;
			}
			// ワークフロー状況確認(承認済の場合)
			boolean isComplete = PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus());
			if (isComplete) {
				// チェックボックス要否設定(要)
				dto.setNeedCheckbox(true);
			}
			// 実績一覧の場合
			if (isActualList) {
				dto.setWorkTypeCode("");
				continue;
			}
			// 休日出勤申請情報から休日出勤時の予定勤務形態を取得して設定
			dto.setWorkTypeCode(getWorkOnHolidayWorkType(workOnHolidayRequestDto));
			// 振替申請確認
			if (workOnHolidayRequestDto.getSubstitute() == TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_OFF) {
				// 振替申請しない場合 
				// 始業予定時間設定
				dto.setStartTime(workOnHolidayRequestDto.getStartTime());
				// 終業予定時間設定
				dto.setEndTime(workOnHolidayRequestDto.getEndTime());
				// 状態設定(予定)
				dto.setApplicationInfo(getScheduleNaming());
			} else {
				// 勤怠一覧情報に勤務形態の内容を設定
				setDtoFields(dto, dto.getWorkTypeCode(), true, false);
				if (!isComplete) {
					// チェックボックス要否設定(不要)
					dto.setNeedCheckbox(false);
				}
			}
			// 勤怠データ情報取得(下書確認)
			AttendanceDtoInterface attendanceDto = getAttendanceDtoListDto(dto.getWorkDate());
			if (attendanceDto != null) {
				// ワークフロー情報取得
				WorkflowDtoInterface attendanceWorkflow = workflow.getLatestWorkflowInfo(attendanceDto.getWorkflow());
				// 勤怠一覧情報に勤怠データ情報を設定(下書で上書)
				setDtoFields(dto, attendanceDto, attendanceWorkflow);
			}
		}
	}
	
	/**
	 * 振替休日情報を取得し勤怠一覧情報リストに設定する。<br>
	 * @param containNotApproved 未承認要否フラグ(true：未承認申請も含める、false：承認済のみ)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void addSubstituteList(boolean containNotApproved) throws MospException {
		// 振替休日情報取得
		substituteList = substituteReference.getSubstituteList(personalId, firstDate, lastDate);
		// 振替休日回数
		float substituteHolidays = 0F;
		// 振替休日毎に処理
		for (SubstituteDtoInterface substituteDto : substituteList) {
			// 振替休日の元である休日出勤申請情報を取得
			WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto = workOnHoliday.findForSubstitute(personalId,
					substituteDto.getWorkDate(), substituteDto.getTimesWork());
			// 休日出勤申請情報確認
			if (workOnHolidayRequestDto == null) {
				continue;
			}
			// ワークフロー情報取得
			WorkflowDtoInterface workflowDto = workflow.getLatestWorkflowInfo(workOnHolidayRequestDto.getWorkflow());
			// 申請情報要否確認
			if (isRequestNeeded(workflowDto, containNotApproved) == false) {
				continue;
			}
			// 振替休日と同日の休日出勤申請情報を取得
			WorkOnHolidayRequestDtoInterface sameDayWorkOnHolidayRequestDto = workOnHoliday.findForKeyOnWorkflow(
					personalId, substituteDto.getSubstituteDate());
			// 振替休日と同日の休日出勤申請情報確認
			if (sameDayWorkOnHolidayRequestDto != null) {
				// 振替休日と同日の休日出勤申請情報のワークフロー情報取得
				WorkflowDtoInterface sameDayWorkflowDto = workflow.getLatestWorkflowInfo(workOnHolidayRequestDto
					.getWorkflow());
				// 申請情報要否確認
				if (isRequestNeeded(sameDayWorkflowDto, containNotApproved)) {
					continue;
				}
			}
			// 対象日における勤怠一覧情報を取得
			AttendanceListDto dto = getAttendanceListDto(substituteDto.getSubstituteDate());
			// 備考設定
			addRemark(dto, getSubstituteAbbrNaming() + getWorkflowStatusAbbr(workflowDto));
			// 振替休日回数加算
			substituteHolidays += getHolidayTimes(substituteDto.getSubstituteRange());
			// 休暇範囲確認
			if (substituteDto.getSubstituteRange() != TimeConst.CODE_HOLIDAY_RANGE_ALL) {
				// 全休以外の場合は設定不要
				continue;
			}
			// 勤務形態コード設定
			dto.setWorkTypeCode("");
			// 振替休日種別確認
			if (substituteDto.getSubstituteType().equals(TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY)) {
				// 勤務形態略称設定
				dto.setWorkTypeAbbr(getLegalSubstituteAbbrNaming());
			} else {
				// 勤務形態略称設定
				dto.setWorkTypeAbbr(getPrescribedSubstituteAbbrNaming());
			}
			// 状態設定
			dto.setApplicationInfo(workflow.getWorkflowStatus(workflowDto.getWorkflowStatus(), workflowDto
				.getWorkflowStage()));
		}
		// 最終レコード取得
		AttendanceListDto dto = attendanceList.get(attendanceList.size() - 1);
		// 集計値設定
		// 休日出勤回数
		dto.setSubstituteHolidays(substituteHolidays);
	}
	
	/**
	 * 代休申請情報を取得し勤怠一覧情報リストに設定する。<br>
	 * @param containNotApproved 未承認要否フラグ(true：未承認申請も含める、false：承認済のみ)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void addSubHolidayRequestList(boolean containNotApproved) throws MospException {
		// 代休申請情報取得
		subHolidayRequestList = subHoliday.getSubHolidayRequestList(personalId, firstDate, lastDate);
		// 代休回数
		float subHolidays = 0F;
		// 代休申請毎に処理
		for (SubHolidayRequestDtoInterface subHolidayRequestDto : subHolidayRequestList) {
			// 対象日における勤怠一覧情報を取得
			AttendanceListDto dto = getAttendanceListDto(subHolidayRequestDto.getRequestDate());
			// ワークフロー情報取得
			WorkflowDtoInterface workflowDto = workflow.getLatestWorkflowInfo(subHolidayRequestDto.getWorkflow());
			// 申請情報要否確認
			if (isRequestNeeded(workflowDto, containNotApproved) == false) {
				if (containNotApproved && isDraft(workflowDto)) {
					// 下書の場合
					addRemark(dto, getSubHolidayAbbrNaming() + getDraftAbbrNaming());
				}
				continue;
			}
			// 備考設定
			addRemark(dto, getSubHolidayAbbrNaming() + getWorkflowStatusAbbr(workflowDto));
			// 代休回数加算
			subHolidays += getHolidayTimes(subHolidayRequestDto.getHolidayRange());
			// 休暇範囲確認
			if (subHolidayRequestDto.getHolidayRange() != TimeConst.CODE_HOLIDAY_RANGE_ALL) {
				// 全休以外の場合は設定不要
				continue;
			}
			// 勤務形態コード設定
			dto.setWorkTypeCode("");
			// 勤務形態略称設定
			dto.setWorkTypeAbbr(getWorkTypeAbbr(subHolidayRequestDto));
			// 状態設定
			dto.setApplicationInfo(workflow.getWorkflowStatus(workflowDto.getWorkflowStatus(), workflowDto
				.getWorkflowStage()));
		}
		// 最終レコード取得
		AttendanceListDto dto = attendanceList.get(attendanceList.size() - 1);
		// 集計値設定
		// 代休回数
		dto.setSubHolidays(subHolidays);
	}
	
	/**
	 * 残業申請情報を取得し勤怠一覧情報リストに設定する。<br>
	 * @param containNotApproved 未承認要否フラグ(true：未承認申請も含める、false：承認済のみ)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void addOvertimeRequestList(boolean containNotApproved) throws MospException {
		// 残業申請情報取得
		List<OvertimeRequestDtoInterface> list = overtime.getOvertimeRequestList(personalId, firstDate, lastDate);
		// 残業申請毎に処理
		for (OvertimeRequestDtoInterface overtimeRequestDto : list) {
			// 対象日における勤怠一覧情報を取得
			AttendanceListDto dto = getAttendanceListDto(overtimeRequestDto.getRequestDate());
			// ワークフロー情報取得
			WorkflowDtoInterface workflowDto = workflow.getLatestWorkflowInfo(overtimeRequestDto.getWorkflow());
			// 申請情報要否確認
			if (isRequestNeeded(workflowDto, containNotApproved) == false) {
				if (containNotApproved && isDraft(workflowDto)) {
					// 下書の場合
					addRemark(dto, getOvertimeAbbrNaming() + getDraftAbbrNaming());
				}
				continue;
			}
			// 備考設定
			addRemark(dto, getOvertimeAbbrNaming() + getWorkflowStatusAbbr(workflowDto));
		}
	}
	
	/**
	 * 時差出勤申請情報を取得し勤怠一覧情報リストに設定する。<br>
	 * @param containNotApproved 未承認要否フラグ(true：未承認申請も含める、false：承認済のみ)
	 * @param isActualList 実績一覧フラグ(true：実績一覧、false：実績一覧ではない)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void addDifferenceRequestList(boolean containNotApproved, boolean isActualList) throws MospException {
		// 時差出勤申請情報取得
		List<DifferenceRequestDtoInterface> list = difference.getDifferenceRequestList(personalId, firstDate, lastDate);
		// 残業申請毎に処理
		for (DifferenceRequestDtoInterface differenceRequestDto : list) {
			// 対象日における勤怠一覧情報を取得
			AttendanceListDto dto = getAttendanceListDto(differenceRequestDto.getRequestDate());
			// ワークフロー情報取得
			WorkflowDtoInterface workflowDto = workflow.getLatestWorkflowInfo(differenceRequestDto.getWorkflow());
			// 申請情報要否確認
			if (isRequestNeeded(workflowDto, containNotApproved) == false) {
				if (containNotApproved && isDraft(workflowDto)) {
					// 下書の場合
					addRemark(dto, getTimeDefferenceAbbrNaming() + getDraftAbbrNaming());
				}
				continue;
			}
			// 備考設定
			addRemark(dto, getTimeDefferenceAbbrNaming() + getWorkflowStatusAbbr(workflowDto));
			// 実績一覧の場合
			if (isActualList) {
				continue;
			}
			// ワークフロー状況確認
			if (PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
				// 勤務形態設定
				dto.setWorkTypeCode(differenceRequestDto.getDifferenceType());
				// 始業予定時間設定
				dto.setStartTime(differenceRequestDto.getRequestStart());
				// 終業予定時間設定
				dto.setEndTime(differenceRequestDto.getRequestEnd());
			}
			// 勤怠データ情報取得
			AttendanceDtoInterface attendanceDto = getAttendanceDtoListDto(dto.getWorkDate());
			// 勤怠データ確認
			if (attendanceDto == null) {
				// 勤怠データが存在しない場合
				dto.setNeedCheckbox(true);
				dto.setApplicationInfo(getScheduleNaming());
			} else {
				// ワークフロー情報取得
				WorkflowDtoInterface attendanceWorkflow = workflow.getLatestWorkflowInfo(attendanceDto.getWorkflow());
				// 勤怠一覧情報に勤怠データ情報を設定(下書で上書)
				setDtoFields(dto, attendanceDto, attendanceWorkflow);
			}
		}
	}
	
	/**
	 * 時間単位年休の利用確認を行う。<br>
	 * 時間単位年休を利用しない場合は、時間単位年休の合計値をnullにする。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void confirmTimelyPaidHoliday() throws MospException {
		// 有給休暇コード取得
		String paidHolidayCode = applicationDto.getPaidHolidayCode();
		// 有給休暇設定取得
		PaidHolidayDtoInterface paidHolidayDto = paidHoliday.getPaidHolidayInfo(paidHolidayCode, lastDate);
		// 時間単位年休利用フラグ確認
		if (paidHolidayDto != null && paidHolidayDto.getTimelyPaidHolidayFlag() == MospConst.INACTIVATE_FLAG_OFF) {
			return;
		}
		// 時間単位年休無効の場合
		// 勤怠一覧情報リスト最終レコード取得
		AttendanceListDto dto = attendanceList.get(attendanceList.size() - 1);
		// 有給時間
		dto.setPaidHolidayTime(null);
	}
	
	/**
	 * 同日複数申請を確認する。<br>
	 * 同日に複数の休暇申請、振替休日、代休申請があり、合わせて終日休暇となる場合に、
	 * 状態及びチェックボックス要否を再設定する。<br>
	 * @param containNotApproved 未承認要否フラグ(true：未承認申請も含める、false：承認済のみ)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void checkPluralRequest(boolean containNotApproved) throws MospException {
		// 勤怠一覧情報毎に処理
		attendanceList: for (AttendanceListDto dto : attendanceList) {
			// 休日日数準備
			float holidayTimes = 0F;
			// 休暇申請情報確認
			for (HolidayRequestDtoInterface holidayRequestDto : holidayRequestList) {
				// 休暇日確認
				if (TimeUtility.isTermContain(dto.getWorkDate(), holidayRequestDto.getRequestStartDate(),
						holidayRequestDto.getRequestEndDate()) == false) {
					continue;
				}
				// ワークフロー情報取得
				WorkflowDtoInterface workflowDto = workflow.getLatestWorkflowInfo(holidayRequestDto.getWorkflow());
				// 申請情報要否確認
				if (!isRequestNeeded(workflowDto, containNotApproved)) {
					continue;
				}
				// 休暇範囲確認
				if (holidayRequestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_ALL) {
					// 全休の場合は処理不要
					continue attendanceList;
				}
				// 休日日数加算
				holidayTimes += getHolidayTimes(holidayRequestDto.getHolidayRange());
				String workTypeAbbrInitial = getWorkTypeAbbr(holidayRequestDto).substring(0, 1);
				if (dto.getWorkTypeAnteAbbr() == null
						&& holidayRequestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_AM) {
					dto.setWorkTypeAnteAbbr(workTypeAbbrInitial);
				}
				if (dto.getWorkTypePostAbbr() == null
						&& holidayRequestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_PM) {
					dto.setWorkTypePostAbbr(workTypeAbbrInitial);
				}
				// 未承認確認
				if (isApprovable(workflowDto)) {
					// 承認状況設定
					dto.setApplicationInfo(workflow.getWorkflowStatus(workflowDto.getWorkflowStatus(), workflowDto
						.getWorkflowStage()));
				}
			}
			// 代休申請情報確認
			for (SubHolidayRequestDtoInterface subHolidayRequestDto : subHolidayRequestList) {
				// 代休日確認
				if (subHolidayRequestDto.getRequestDate().compareTo(dto.getWorkDate()) != 0) {
					continue;
				}
				// ワークフロー情報取得
				WorkflowDtoInterface workflowDto = workflow.getLatestWorkflowInfo(subHolidayRequestDto.getWorkflow());
				// 申請情報要否確認
				if (!isRequestNeeded(workflowDto, containNotApproved)) {
					continue;
				}
				// 休暇範囲確認
				if (subHolidayRequestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_ALL) {
					// 全休の場合は処理不要
					continue attendanceList;
				}
				// 休日日数加算
				holidayTimes += getHolidayTimes(subHolidayRequestDto.getHolidayRange());
				if (dto.getWorkTypeAnteAbbr() == null
						&& subHolidayRequestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_AM) {
					dto.setWorkTypeAnteAbbr(getSubHolidayAbbrNaming());
				}
				if (dto.getWorkTypePostAbbr() == null
						&& subHolidayRequestDto.getHolidayRange() == TimeConst.CODE_HOLIDAY_RANGE_PM) {
					dto.setWorkTypePostAbbr(getSubHolidayAbbrNaming());
				}
				// 未承認確認
				if (isApprovable(workflowDto)) {
					// 承認状況設定
					dto.setApplicationInfo(workflow.getWorkflowStatus(workflowDto.getWorkflowStatus(), workflowDto
						.getWorkflowStage()));
				}
			}
			// 休日日数確認
			if (holidayTimes >= TimeConst.HOLIDAY_TIMES_ALL) {
				// 勤務形態コード設定
				dto.setWorkTypeCode("");
				// 勤務形態略称設定
				dto.setWorkTypeAbbr(dto.getWorkTypeAnteAbbr() + dto.getWorkTypePostAbbr());
				// チェックボックス設定
				dto.setNeedCheckbox(false);
			}
		}
	}
	
	/**
	 * 勤怠情報を取得し勤怠一覧情報リストに設定する。<br>
	 * @param containNotApproved 未承認要否フラグ(true：下書き等未承認勤怠も含める、false：承認済のみ)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void addActualList(boolean containNotApproved) throws MospException {
		// 勤怠情報取得
		attendanceDtoList = attendanceReference.getAttendanceList(personalId, firstDate, lastDate);
		// 勤怠情報毎に処理
		for (AttendanceDtoInterface attendanceDto : attendanceDtoList) {
			// 勤怠情報の勤務日から勤怠一覧情報を取得
			AttendanceListDto dto = getAttendanceListDto(attendanceDto.getWorkDate());
			// ワークフロー情報取得
			WorkflowDtoInterface workflowDto = workflow.getLatestWorkflowInfo(attendanceDto.getWorkflow());
			// 申請情報要否確認
			if (isRequestNeeded(workflowDto, containNotApproved) == false) {
				continue;
			}
			// 勤怠一覧情報に値を設定
			setDtoFields(dto, attendanceDto, workflowDto);
		}
	}
	
	/**
	 * 勤怠情報(下書)を勤怠一覧情報リストに設定する。<br>
	 * 但し、既に対象日の勤怠一覧情報に勤務形態が設定されている場合は設定しない。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void addDraftList() throws MospException {
		// 勤怠情報毎に処理
		for (AttendanceDtoInterface attendanceDto : attendanceDtoList) {
			// 勤怠情報の勤務日から勤怠一覧情報を取得
			AttendanceListDto dto = getAttendanceListDto(attendanceDto.getWorkDate());
			// 勤怠情報(下書)要否確認
			if (dto.getWorkTypeCode() != null) {
				// 勤務形態が既に設定されている場合
				continue;
			}
			// ワークフロー情報取得
			WorkflowDtoInterface workflowDto = workflow.getLatestWorkflowInfo(attendanceDto.getWorkflow());
			// ワークフロー状況確認
			if (isDraft(workflowDto) == false) {
				// 下書でない場合
				continue;
			}
			// 勤怠一覧情報に値を設定
			setDtoFields(dto, attendanceDto, workflowDto);
		}
	}
	
	/**
	 * カレンダ情報を取得し勤怠一覧情報リストに設定する。<br>
	 * 但し、既に対象日の勤怠一覧情報に勤務形態が設定されている場合は設定しない。<br>
	 * 出勤簿では、出勤予定日が不要となる。<br>
	 * 勤怠一覧では、勤務時間及び休憩時間が不要となる。<br>
	 * @param needWorkDay  出勤予定日要否(true：要、false：不要)
	 * @param needWorkTime 勤務時間及び休憩時間要否(true：要、false：不要)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void addScheduleList(boolean needWorkDay, boolean needWorkTime) throws MospException {
		// カレンダコード取得
		String scheduleCode = applicationDto.getScheduleCode();
		// 期間最終日におけるカレンダ情報を取得
		scheduleDto = scheduleReference.getScheduleInfo(scheduleCode, lastDate);
		// カレンダ情報確認
		if (scheduleDto == null) {
			// エラーメッセージ設定
			addScheduleNotExistErrorMessage(lastDate);
			return;
		}
		// カレンダ日情報取得及び確認
		scheduleDateList = scheduleDateReference.findForList(scheduleCode, firstDate, lastDate);
		if (scheduleDateList.isEmpty()) {
			// エラーメッセージ設定
			addScheduleNotExistErrorMessage(lastDate);
			return;
		}
		// 人事退職情報から退職日を取得
		Date retireDate = retirement.getRetireDate(personalId);
		// 設定適用マスタ(対象期間)情報取得
		Map<Date, ApplicationDtoInterface> mapApplication = applicationReference.findForTerm(personalId, firstDate,
				lastDate);
		
		// 勤怠一覧情報にカレンダ日情報を設定
		for (AttendanceListDto dto : attendanceList) {
			// カレンダ日情報要否確認
			if (dto.getWorkTypeCode() != null) {
				// 勤務形態が既に設定されている場合
				continue;
			}
			// カレンダ日情報取得(勤怠一覧情報の勤務日で取得)及び確認
			ScheduleDateDtoInterface scheduleDateDto = getScheduleDateDto(dto.getWorkDate());
			if (scheduleDateDto == null || scheduleDateDto.getWorkTypeCode().isEmpty()) {
				// カレンダ日情報が存在しない場合
				continue;
			}
			// 設定適用マスタ状況確認
			if (mapApplication.get(dto.getWorkDate()) == null) {
				// 対象日に設定適用情報が存在しない場合
				continue;
			}
			
			// 退職日確認
			if (retireDate != null && retireDate.before(dto.getWorkDate())) {
				// 既に退職している場合
				continue;
			}
			// 休職期間確認
			if (suspension.isSuspended(personalId, dto.getWorkDate())) {
				continue;
			}
			
			// 勤怠一覧情報に勤務形態の内容を設定
			setDtoFields(dto, scheduleDateDto.getWorkTypeCode(), needWorkDay, needWorkTime);
		}
	}
	
	/**
	 * 休暇申請情報から勤務形態略称を取得する。<br>
	 * @param dto 休暇申請情報
	 * @return 勤務形態略称
	 * @throws MospException 特別休暇或いはその他休暇の略称取得に失敗した場合
	 */
	protected String getWorkTypeAbbr(HolidayRequestDtoInterface dto) throws MospException {
		// 休暇区分毎に勤務形態略称を取得
		switch (dto.getHolidayType1()) {
			case TimeConst.CODE_HOLIDAYTYPE_HOLIDAY:
				// 有給休暇の場合
				if (dto.getHolidayType2().equals(String.valueOf(TimeConst.CODE_HOLIDAYTYPE_HOLIDAY))) {
					return getPaidHolidayAbbrNaming();
				} else {
					return getStockHolidayAbbrNaming();
				}
			case TimeConst.CODE_HOLIDAYTYPE_SPECIAL:
			case TimeConst.CODE_HOLIDAYTYPE_OTHER:
			case TimeConst.CODE_HOLIDAYTYPE_ABSENCE:
				// 特別休暇・その他休暇・欠勤の場合
				return holiday.getHolidayAbbr(dto.getHolidayType2(), dto.getRequestEndDate(), dto.getHolidayType1());
			default:
				return null;
		}
	}
	
	/**
	 * 代休申請情報から勤務形態略称を取得する。<br>
	 * @param dto 代休申請情報
	 * @return 勤務形態略称
	 */
	protected String getWorkTypeAbbr(SubHolidayRequestDtoInterface dto) {
		// 代休種別確認
		switch (dto.getWorkDateSubHolidayType()) {
			case TimeConst.CODE_PRESCRIBED_SUBHOLIDAY_CODE:
			case TimeConst.CODE_MIDNIGHT_SUBHOLIDAY_CODE:
				// 所定代休或いは深夜代休
				return getPrescribedSubHolidayAbbrNaming();
			case TimeConst.CODE_LEGAL_SUBHOLIDAY_CODE:
				// 法定代休
				return getLegalSubHolidayAbbrNaming();
			default:
				return null;
		}
	}
	
	/**
	 * 未承認確認を行う。<br>
	 * ワークフロー状況及び段階から、承認可能であるかを確認する。<br>
	 * @param dto ワークフロー情報
	 * @return 未承認確認結果(true：未承認、false：未承認でない)
	 */
	protected boolean isApprovable(WorkflowDtoInterface dto) {
		// ワークフロー状況取得
		String status = dto.getWorkflowStatus();
		// 未承認、承認済(完了でない)の場合
		if (status.equals(PlatformConst.CODE_STATUS_APPLY) || status.equals(PlatformConst.CODE_STATUS_APPROVED)) {
			// 未承認
			return true;
		}
		// 差戻、取消(承認解除)の場合
		if (status.equals(PlatformConst.CODE_STATUS_CANCEL) || status.equals(PlatformConst.CODE_STATUS_APPROVED)) {
			// 段階が0より大きい場合(申請者が操作者でない場合)
			if (dto.getWorkflowStage() > 0) {
				return true;
			}
		}
		// それ以外の場合
		return false;
	}
	
	/**
	 * チェックボックス要否(勤怠一覧用)を取得する。<br>
	 * @param dto ワークフロー情報
	 * @return チェックボックス要否(勤怠一覧用)
	 */
	protected boolean isNeedCheckBox(WorkflowDtoInterface dto) {
		// 下書の場合
		if (isDraft(dto)) {
			return true;
		}
		// 差戻の場合
		if (isRevert(dto)) {
			// 段階が0(申請者)の場合
			if (dto.getWorkflowStage() == 0) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * 状態リンク要否(勤怠一覧用)を取得する。<br>
	 * @param dto ワークフロー情報
	 * @return 状態リンク要否(勤怠一覧用)
	 */
	protected boolean getNeedStatusLink(WorkflowDtoInterface dto) {
		return !isDraft(dto);
	}
	
	/**
	 * カレンダ日情報リストから対象日のカレンダ日情報を取得する。<br>
	 * @param targetDate 対象日
	 * @return カレンダ日情報
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected ScheduleDateDtoInterface getScheduleDateDto(Date targetDate) throws MospException {
		// カレンダ日情報リストから対象日のカレンダ日情報を取得
		for (ScheduleDateDtoInterface dto : scheduleDateList) {
			// 勤務日確認
			if (targetDate.equals(dto.getScheduleDate())) {
				return dto;
			}
		}
		return null;
	}
	
	/**
	 * 勤怠データ情報リストから対象日の勤怠一覧情報を取得する。<br>
	 * @param targetDate 対象日
	 * @return 勤怠データ情報
	 */
	protected AttendanceDtoInterface getAttendanceDtoListDto(Date targetDate) {
		// 勤怠一覧情報確認
		if (attendanceDtoList == null) {
			return null;
		}
		// 勤怠一覧情報リストから対象日の勤怠一覧情報を取得
		for (AttendanceDtoInterface dto : attendanceDtoList) {
			// 勤務日確認
			if (targetDate.equals(dto.getWorkDate())) {
				return dto;
			}
		}
		return null;
	}
	
	/**
	 * 勤怠一覧情報リストから対象日の勤怠一覧情報を取得する。<br>
	 * @param targetDate 対象日
	 * @return 勤怠一覧情報
	 */
	protected AttendanceListDto getAttendanceListDto(Date targetDate) {
		// 勤怠一覧情報リストから対象日の勤怠一覧情報を取得
		for (AttendanceListDto dto : attendanceList) {
			// 勤務日確認
			if (targetDate.equals(dto.getWorkDate())) {
				return dto;
			}
		}
		return null;
	}
	
	/**
	 * 限度基準情報を設定する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setLimitStandard() throws MospException {
		// 限度基準取得
		LimitStandardDtoInterface limitStandardDto = limitStandard.findForKey(timeSettingDto.getWorkSettingCode(),
				timeSettingDto.getActivateDate(), LIMIT_STANDARD_TERM_MONTH1);
		if (limitStandardDto == null) {
			return;
		}
		// 注意時間取得
		int attention = limitStandardDto.getAttentionTime();
		// 警告時間取得
		int warning = limitStandardDto.getWarningTime();
		// 法定外残業時間合計値準備
		int overtimeTotal = 0;
		// 勤怠一覧情報毎に残業時間を確認
		for (AttendanceListDto dto : attendanceList) {
			// 法定外残業時間確認
			if (dto.getOvertimeOut() == null || dto.getOvertimeOut().intValue() == 0F) {
				continue;
			}
			// 法定外残業時間加算
			overtimeTotal += dto.getOvertimeOut().intValue();
			// 注意時間確認
			if (overtimeTotal > attention) {
				dto.setOvertimeStyle(STYLE_YELLOW);
			}
			// 警告時間確認
			if (overtimeTotal > warning) {
				dto.setOvertimeStyle(STYLE_RED);
			}
		}
	}
	
	/**
	 * 勤怠一覧情報リストを集計する。<br>
	 * 集計結果は、リストの最終レコードに設定される。<br>
	 */
	protected void totalAttendanceList() {
		// 集計値準備
		// 勤務時間(分)
		int workTimeTotal = 0;
		// 休憩時間(分)
		int restTimeTotal = 0;
		// 私用外出時間(分)
		int privateTimeTotal = 0;
		// 公用外出時間(分)
		int publicTimeTotal = 0;
		// 遅刻時間(分)
		int lateTimeTotal = 0;
		// 早退時間(分)
		int leaveEarlyTimeTotal = 0;
		// 遅刻早退時間(分)
		int lateLeaveEarlyTimeTotal = 0;
		// 残業時間(分)
		int overtimeTotal = 0;
		// 内残時間(分)
		int overtimeInTotal = 0;
		// 外残時間(分)
		int overtimeOutTotal = 0;
		// 休出時間(分)
		int holidayWorkTimeTotal = 0;
		// 深夜時間(分)
		int lateNightTimeTotal = 0;
		// 出勤回数
		int workDays = 0;
		// 遅刻回数
		int lateDays = 0;
		// 早退回数
		int leaveEarlyDays = 0;
		// 残業回数
		int overtimeDays = 0;
		// 休出回数
		int holidayWorkDays = 0;
		// 深夜回数
		int lateNightDays = 0;
		// 所定休日回数
		int prescribedHolidays = 0;
		// 法定休日回数
		int legalHolidays = 0;
		// 勤怠一覧情報毎に集計
		for (AttendanceListDto dto : attendanceList) {
			// 勤務時間
			workTimeTotal += dto.getWorkTime() == null ? 0 : dto.getWorkTime().intValue();
			// 休憩時間
			restTimeTotal += dto.getRestTime() == null ? 0 : dto.getRestTime().intValue();
			// 私用外出時間
			privateTimeTotal += dto.getPrivateTime() == null ? 0 : dto.getPrivateTime().intValue();
			// 公用外出時間
			publicTimeTotal += dto.getPublicTime() == null ? 0 : dto.getPublicTime().intValue();
			// 遅刻時間
			lateTimeTotal += dto.getLateTime() == null ? 0 : dto.getLateTime().intValue();
			// 早退時間
			leaveEarlyTimeTotal += dto.getLeaveEarlyTime() == null ? 0 : dto.getLeaveEarlyTime().intValue();
			// 遅刻早退時間
			lateLeaveEarlyTimeTotal += dto.getLateLeaveEarlyTime() == null ? 0 : dto.getLateLeaveEarlyTime().intValue();
			// 残業時間
			overtimeTotal += dto.getOvertime() == null ? 0 : dto.getOvertime().intValue();
			// 内残時間
			overtimeInTotal += dto.getOvertimeIn() == null ? 0 : dto.getOvertimeIn().intValue();
			// 外残時間
			overtimeOutTotal += dto.getOvertimeOut() == null ? 0 : dto.getOvertimeOut().intValue();
			// 休出時間
			holidayWorkTimeTotal += dto.getHolidayWorkTime() == null ? 0 : dto.getHolidayWorkTime().intValue();
			// 深夜時間
			lateNightTimeTotal += dto.getLateNightTime() == null ? 0 : dto.getLateNightTime().intValue();
			// 出勤回数
			workDays += countHours(dto.getWorkTime());
			// 遅刻回数
			lateDays += countHours(dto.getLateTime());
			// 早退回数
			leaveEarlyDays += countHours(dto.getLeaveEarlyTime());
			// 残業回数
			overtimeDays += countHours(dto.getOvertime());
			// 休出回数
			holidayWorkDays += countHours(dto.getHolidayWorkTime());
			// 深夜回数
			lateNightDays += countHours(dto.getLateNightTime());
			// 勤務形態確認
			if (dto.getWorkTypeCode() == null || dto.getWorkTypeCode().isEmpty()) {
				continue;
			}
			// 所定休日回数
			prescribedHolidays += dto.getWorkTypeCode().equals(TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY) ? 1 : 0;
			// 法定休日回数
			legalHolidays += dto.getWorkTypeCode().equals(TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY) ? 1 : 0;
		}
		// 最終レコード取得
		AttendanceListDto dto = attendanceList.get(attendanceList.size() - 1);
		// 集計値設定
		dto.setWorkTimeTotal(workTimeTotal);
		dto.setRestTimeTotal(restTimeTotal);
		dto.setPrivateTimeTotal(privateTimeTotal);
		dto.setPublicTimeTotal(publicTimeTotal);
		dto.setLateTimeTotal(lateTimeTotal);
		dto.setLeaveEarlyTimeTotal(leaveEarlyTimeTotal);
		dto.setLateLeaveEarlyTimeTotal(lateLeaveEarlyTimeTotal);
		dto.setOvertimeTotal(overtimeTotal);
		dto.setOvertimeInTotal(overtimeInTotal);
		dto.setOvertimeOutTotal(overtimeOutTotal);
		dto.setHolidayWorkTimeTotal(holidayWorkTimeTotal);
		dto.setLateNightTimeTotal(lateNightTimeTotal);
		dto.setWorkDays(workDays);
		dto.setLateDays(lateDays);
		dto.setLeaveEarlyDays(leaveEarlyDays);
		dto.setOvertimeDays(overtimeDays);
		dto.setHolidayWorkDays(holidayWorkDays);
		dto.setLateNightDays(lateNightDays);
		dto.setPrescribedHolidays(prescribedHolidays);
		dto.setLegalHolidays(legalHolidays);
	}
	
	/**
	 * 勤怠一覧情報に勤怠情報の内容を設定する。<br>
	 * @param dto           設定対象勤怠一覧情報
	 * @param attendanceDto 勤怠情報
	 * @param workflowDto   ワークフロー情報
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setDtoFields(AttendanceListDto dto, AttendanceDtoInterface attendanceDto,
			WorkflowDtoInterface workflowDto) throws MospException {
		// 勤務形態
		dto.setWorkTypeCode(attendanceDto.getWorkTypeCode());
		// 始業時間
		dto.setStartTime(attendanceDto.getStartTime());
		// 終業時間
		dto.setEndTime(attendanceDto.getEndTime());
		// 勤務時間
		dto.setWorkTime(attendanceDto.getWorkTime());
		// 休憩時間
		dto.setRestTime(attendanceDto.getRestTime());
		// 私用外出時間
		dto.setPrivateTime(attendanceDto.getPrivateTime());
		// 公用外出時間
		dto.setPublicTime(attendanceDto.getPublicTime());
		// 遅刻時間
		dto.setLateTime(attendanceDto.getLateTime());
		// 早退時間
		dto.setLeaveEarlyTime(attendanceDto.getLeaveEarlyTime());
		// 遅刻早退時間
		dto.setLateLeaveEarlyTime(attendanceDto.getLateTime() + attendanceDto.getLeaveEarlyTime());
		// 残業時間
		dto.setOvertime(attendanceDto.getOvertime());
		// 内残時間
		dto.setOvertimeIn(attendanceDto.getOvertimeIn());
		// 外残時間
		dto.setOvertimeOut(attendanceDto.getOvertimeOut());
		// 休出時間
		dto.setHolidayWorkTime(attendanceDto.getLegalWorkTime());
		// 深夜時間
		dto.setLateNightTime(attendanceDto.getLateNightTime());
		// 備考
		addAttendanceRemark(dto, attendanceDto);
		// 申請情報(ワークフロー情報から取得)
		dto.setApplicationInfo(workflow.getWorkflowStatus(workflowDto.getWorkflowStatus(), workflowDto
			.getWorkflowStage()));
		// チェックボックス要否(勤怠一覧用)
		dto.setNeedCheckbox(isNeedCheckBox(workflowDto));
		// 状態リンク要否(勤怠一覧用)
		dto.setNeedStatusLink(getNeedStatusLink(workflowDto));
		// 勤怠データレコード識別ID
		dto.setAttendanceRecordId(attendanceDto.getTmdAttendanceId());
		// ワークフローレコード識別ID
		dto.setWorkflowRecordId(workflowDto.getPftWorkflowId());
		// ワークフロー番号
		dto.setWorkflow(workflowDto.getWorkflow());
		// ワークフロー状況
		dto.setWorkflowStatus(workflowDto.getWorkflowStatus());
		// ワークフロー段階
		dto.setWorkflowStage(workflowDto.getWorkflowStage());
		// 修正情報
		dto.setCorrectionInfo(getCorrectionInfo(attendanceDto));
	}
	
	/**
	 * 修正情報を取得する。<br>
	 * @param attendanceDto 勤怠情報
	 * @return 修正情報（本/他）
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected String getCorrectionInfo(AttendanceDtoInterface attendanceDto) throws MospException {
		// 勤怠修正情報取得
		AttendanceCorrectionDtoInterface attendanceCorrectionDto = correction.getLatestAttendanceCorrectionInfo(
				personalId, attendanceDto.getWorkDate(), TIMES_WORK_DEFAULT);
		// 勤怠修正情報確認
		if (attendanceCorrectionDto == null) {
			return "";
		}
		// 勤怠修正者確認
		if (attendanceCorrectionDto.getPersonalId().equals(attendanceCorrectionDto.getCorrectionPersonalId())) {
			return getSelfCorrectAbbrNaming();
		}
		return getOtherCorrectAbbrNaming();
	}
	
	/**
	 * 勤怠一覧情報に勤怠情報の備考を追加する。<br>
	 * @param dto           追加対象勤怠一覧情報
	 * @param attendanceDto 勤怠情報
	 */
	protected void addAttendanceRemark(AttendanceListDto dto, AttendanceDtoInterface attendanceDto) {
		// 直行確認
		if (attendanceDto.getDirectStart() == Integer.parseInt(MospConst.CHECKBOX_ON)) {
			addRemark(dto, getDirectStartNaming());
		}
		// 直帰確認
		if (attendanceDto.getDirectEnd() == Integer.parseInt(MospConst.CHECKBOX_ON)) {
			addRemark(dto, getDirectEndNaming());
		}
		// コメント確認
		if (attendanceDto.getTimeComment().isEmpty() == false) {
			addRemark(dto, attendanceDto.getTimeComment());
		}
	}
	
	/**
	 * 勤怠一覧情報に勤務形態情報の内容を設定する。<br>
	 * @param dto          設定対象勤怠一覧情報
	 * @param workTypeCode 勤務形態コード
	 * @param needWorkDay  出勤予定日要否(true：要、false：不要)
	 * @param needWorkTime 勤務時間、休憩時間要否(true：要、false：不要)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setDtoFields(AttendanceListDto dto, String workTypeCode, boolean needWorkDay, boolean needWorkTime)
			throws MospException {
		// 勤務形態確認(所定休日或いは法定休日)
		if (workTypeCode.equals(TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY)
				|| workTypeCode.equals(TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY)) {
			// 勤務形態設定
			dto.setWorkTypeCode(workTypeCode);
			// 状態設定(承認済)
			dto.setApplicationInfo(getApplovedNaming());
			// チェックボックス要否設定(不要)
			dto.setNeedCheckbox(false);
			return;
		}
		// 出勤予定日不要確認
		if (needWorkDay == false) {
			return;
		}
		// 勤務形態設定
		String differenceAbb = difference.getDifferenceAbbr(personalId, dto.getWorkDate());
		if (differenceAbb != null) {
			// 時差出勤存在する場合
			dto.setWorkTypeCode(differenceAbb);
		} else {
			// 時差出勤存在しない場合
			dto.setWorkTypeCode(workTypeCode);
		}
		dto.setWorkTypeCode(workTypeCode);
		// 状態設定(予定)
		dto.setApplicationInfo(getScheduleNaming());
		// チェックボックス要否設定(要)
		dto.setNeedCheckbox(true);
		// 始業時間設定
		if (dto.getStartTime() == null) {
			dto.setStartTime(getWorkTypeTime(dto, TimeConst.CODE_WORKSTART));
		}
		// 終業時間設定
		if (dto.getEndTime() == null) {
			dto.setEndTime(getWorkTypeTime(dto, TimeConst.CODE_WORKEND));
		}
		// 勤務時間要不要確認
		if (needWorkTime == false) {
			return;
		}
		// 勤務時間
		dto.setWorkTime(getWorkTypeMinutes(dto, TimeConst.CODE_WORKTIME));
		// 休憩時間
		dto.setRestTime(getWorkTypeMinutes(dto, TimeConst.CODE_RESTTIME));
	}
	
	/**
	 * 勤怠一覧情報に表示用文字列(日付、時刻、時間等)を設定する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setDtoStringFields() throws MospException {
		// 勤怠一覧情報毎に処理
		for (AttendanceListDto dto : attendanceList) {
			setDtoStringFields(dto);
		}
	}
	
	/**
	 * 勤怠一覧情報の特定の項目にハイフンを設定する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setDtoHyphenFields() throws MospException {
		// 勤怠一覧情報毎に処理
		for (AttendanceListDto dto : attendanceList) {
			if (dto.getCorrectionInfo() == null) {
				// 始業時刻
				dto.setStartTimeString(getHyphenNaming());
				// 終業時刻
				dto.setEndTimeString(getHyphenNaming());
			}
		}
	}
	
	/**
	 * 勤怠一覧情報にチェックボックス要否設定する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setNeedCheckbox() throws MospException {
		Integer state = totalTimeEmployee.getCutoffState(personalId, targetYear, targetMonth);
		if (state == null) {
			return;
		}
		if (state != TimeConst.CODE_CUTOFF_STATE_NOT_TIGHT) {
			for (AttendanceListDto dto : attendanceList) {
				dto.setNeedCheckbox(false);
			}
		}
	}
	
	/**
	 * 勤怠一覧情報(勤怠承認一覧)にチェックボックス要否設定する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setNeedApprovalCheckbox() throws MospException {
		// 締状態取得
		Integer state = totalTimeEmployee.getCutoffState(personalId, targetYear, targetMonth);
		// 締状態確認
		if (state != null && state != TimeConst.CODE_CUTOFF_STATE_NOT_TIGHT) {
			// 未締でない場合
			for (AttendanceListDto dto : attendanceList) {
				dto.setNeedCheckbox(false);
			}
			return;
		}
		// 承認可能ワークフロー情報マップ(勤怠)取得
		Map<Long, WorkflowDtoInterface> approvableMap = getApprovableMap();
		// 勤怠一覧情報毎に処理
		for (AttendanceListDto dto : attendanceList) {
			// チェックボックス要否設定
			dto.setNeedCheckbox(false);
			// ワークフロー確認
			if (dto.getWorkflow() == 0) {
				// ワークフロー番号が設定されていない場合
				continue;
			}
			// 承認可能ワークフロー情報に含まれているか確認
			if (approvableMap.containsKey(dto.getWorkflow())) {
				// チェックボックス要否設定
				dto.setNeedCheckbox(true);
			}
		}
	}
	
	/**
	 * 承認可能ワークフロー情報マップ(勤怠)を取得する。<br>
	 * ログインユーザが承認可能な情報を取得する。<br>
	 * @return 承認可能ワークフロー情報マップ
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected Map<Long, WorkflowDtoInterface> getApprovableMap() throws MospException {
		// ログインユーザ個人ID取得
		String personalId = mospParams.getUser().getPersonalId();
		// 承認情報参照クラス取得
		ApprovalInfoReferenceBeanInterface approvalReference = (ApprovalInfoReferenceBeanInterface)createBean(ApprovalInfoReferenceBeanInterface.class);
		// 承認可能ワークフローリスト(勤怠)取得
		return approvalReference.getApprovableMap(personalId).get(TimeConst.CODE_FUNCTION_WORK_MANGE);
	}
	
	/**
	 * 勤怠一覧情報に表示用文字列(日付、時刻、時間等)を設定する。<br>
	 * @param dto 設定対象勤怠一覧情報
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setDtoStringFields(AttendanceListDto dto) throws MospException {
		// 基準日付取得(勤務日)
		Date standardDate = dto.getWorkDate();
		// 対象日取得(勤務日)
		Date targetDate = dto.getWorkDate();
		// 勤務日
		dto.setWorkDateString(getStringMonthDay(targetDate));
		// 勤務曜日
		dto.setWorkDayOfWeek(getStringDayOfWeek(targetDate));
		// 曜日配色設定
		dto.setWorkDayOfWeekStyle(getWorkDayOfWeekStyle(targetDate));
		// 祝日備考設定
		addRemark(dto, holidayBean.getHolidayName(dto.getWorkDate()));
		// 勤務形態略称
		setWorkTypeAbbr(dto);
		// 始業時間
		dto.setStartTimeString(getStringTime(dto.getStartTime(), standardDate));
		// 終業時間
		dto.setEndTimeString(getStringTime(dto.getEndTime(), standardDate));
		// 勤務時間
		dto.setWorkTimeString(getStringHours(dto.getWorkTime(), true));
		// 休憩時間
		dto.setRestTimeString(getStringHours(dto.getRestTime(), true));
		// 私用外出時間
		dto.setPrivateTimeString(getStringHours(dto.getPrivateTime(), true));
		// 公用外出時間
		dto.setPublicTimeString(getStringHours(dto.getPublicTime(), true));
		// 遅刻時間
		dto.setLateTimeString(getStringHours(dto.getLateTime(), true));
		// 早退時間
		dto.setLeaveEarlyTimeString(getStringHours(dto.getLeaveEarlyTime(), true));
		// 遅刻早退時間
		dto.setLateLeaveEarlyTimeString(getStringHours(dto.getLateLeaveEarlyTime(), true));
		// 残業時間
		dto.setOvertimeString(getStringHours(dto.getOvertime(), true));
		// 内残時間
		dto.setOvertimeInString(getStringHours(dto.getOvertimeIn(), true));
		// 外残時間
		dto.setOvertimeOutString(getStringHours(dto.getOvertimeOut(), true));
		// 休出時間
		dto.setHolidayWorkTimeString(getStringHours(dto.getHolidayWorkTime(), true));
		// 深夜時間
		dto.setLateNightTimeString(getStringHours(dto.getLateNightTime(), true));
		// 勤務時間合計
		dto.setWorkTimeTotalString(getStringHours(dto.getWorkTimeTotal(), false));
		// 休憩時間合計
		dto.setRestTimeTotalString(getStringHours(dto.getRestTimeTotal(), false));
		// 私用外出時間合計
		dto.setPrivateTimeTotalString(getStringHours(dto.getPrivateTimeTotal(), false));
		// 公用外出時間合計
		dto.setPublicTimeTotalString(getStringHours(dto.getPublicTimeTotal(), false));
		// 遅刻時間
		dto.setLateTimeTotalString(getStringHours(dto.getLateTimeTotal(), false));
		// 早退時間
		dto.setLeaveEarlyTimeTotalString(getStringHours(dto.getLeaveEarlyTimeTotal(), false));
		// 遅刻早退時間
		dto.setLateLeaveEarlyTimeTotalString(getStringHours(dto.getLateLeaveEarlyTimeTotal(), false));
		// 残業時間
		dto.setOvertimeTotalString(getStringHours(dto.getOvertimeTotal(), false));
		// 内残時間
		dto.setOvertimeInTotalString(getStringHours(dto.getOvertimeInTotal(), false));
		// 外残時間
		dto.setOvertimeOutTotalString(getStringHours(dto.getOvertimeOutTotal(), false));
		// 休出時間
		dto.setHolidayWorkTimeTotalString(getStringHours(dto.getHolidayWorkTimeTotal(), false));
		// 深夜時間
		dto.setLateNightTimeTotalString(getStringHours(dto.getLateNightTimeTotal(), false));
		// 出勤回数
		dto.setWorkDaysString(getStringTimes(dto.getWorkDays()));
		// 遅刻回数
		dto.setLateDaysString(getStringTimes(dto.getLateDays()));
		// 早退回数
		dto.setLeaveEarlyDaysString(getStringTimes(dto.getLeaveEarlyDays()));
		// 残業回数
		dto.setOvertimeDaysString(getStringTimes(dto.getOvertimeDays()));
		// 休出回数
		dto.setHolidayWorkDaysString(getStringTimes(dto.getHolidayWorkDays()));
		// 深夜回数
		dto.setLateNightDaysString(getStringTimes(dto.getLateNightDays()));
		// 所定休日回数
		dto.setPrescribedHolidaysString(getStringTimes(dto.getPrescribedHolidays()));
		// 法定休日回数
		dto.setLegalHolidaysString(getStringTimes(dto.getLegalHolidays()));
		// 振替休日回数
		dto.setSubstituteHolidaysString(getStringTimes(dto.getSubstituteHolidays()));
		// 有給休暇回数
		dto.setPaidHolidaysString(getStringTimes(dto.getPaidHolidays()));
		// 有給時間
		dto.setPaidHolidayTimeString(getStringTimes(dto.getPaidHolidayTime()));
		// 特別休暇回数
		dto.setSpecialHolidaysString(getStringTimes(dto.getSpecialHolidays()));
		// その他休暇回数
		dto.setOtherHolidaysString(getStringTimes(dto.getOtherHolidays()));
		// 代休回数
		dto.setSubHolidaysString(getStringTimes(dto.getSubHolidays()));
		// 欠勤回数
		dto.setAbsenceDaysString(getStringTimes(dto.getAbsenceDays()));
	}
	
	/**
	 * 対象日勤務曜日のスタイル文字列を取得する。<br>
	 * @param targetDate 対象日
	 * @return 対象日勤務曜日のスタイル文字列
	 */
	protected String getWorkDayOfWeekStyle(Date targetDate) {
		// 祝日判定
		if (holidayBean.isHoliday(targetDate)) {
			return STYLE_RED;
		}
		// 土曜日判定
		if (DateUtility.isSaturday(targetDate)) {
			return STYLE_BLUE;
		}
		// 日曜日判定
		if (DateUtility.isSunday(targetDate)) {
			return STYLE_RED;
		}
		return "";
	}
	
	/**
	 * 勤務形態略称を設定する。<br>
	 * @param dto 勤怠一覧情報
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setWorkTypeAbbr(AttendanceListDto dto) throws MospException {
		// 勤務形態取得
		String workTypeCode = dto.getWorkTypeCode();
		// 時差出勤の略称取得
		String differenceAbb = difference.getDifferenceAbbr(workTypeCode);
		// 勤務形態確認
		if (workTypeCode == null || workTypeCode.isEmpty()) {
			// 何もしない
			return;
		}
		if (!workTypeCode.equals(differenceAbb)) {
			dto.setWorkTypeAbbr(differenceAbb);
			return;
		}
		String workTypeAbbr = workTypeReference.getWorkTypeAbbr(workTypeCode, lastDate);
		String workTypeAbbrInitial = workTypeAbbr.substring(0, 1);
		if (dto.getWorkTypeAnteAbbr() != null && !dto.getWorkTypeAnteAbbr().isEmpty()) {
			StringBuffer sb = new StringBuffer();
			sb.append(dto.getWorkTypeAnteAbbr().substring(0, 1));
			sb.append(workTypeAbbrInitial);
			dto.setWorkTypeAbbr(sb.toString());
			return;
		}
		if (dto.getWorkTypePostAbbr() != null && !dto.getWorkTypePostAbbr().isEmpty()) {
			StringBuffer sb = new StringBuffer();
			sb.append(workTypeAbbrInitial);
			sb.append(dto.getWorkTypePostAbbr().substring(0, 1));
			dto.setWorkTypeAbbr(sb.toString());
			return;
		}
		// 勤務形態略称設定
		dto.setWorkTypeAbbr(workTypeAbbr);
	}
	
	/**
	 * 勤怠時刻を取得する。<br>
	 * 勤務形態項目情報から取得した値と勤怠一覧情報の勤務日を用い、勤怠時刻を取得する。<br>
	 * @param dto              勤怠一覧情報
	 * @param workTypeItemCode 勤務形態項目コード
	 * @return 勤怠時刻
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected Date getWorkTypeTime(AttendanceListDto dto, String workTypeItemCode) throws MospException {
		// 勤務形態項目情報取得
		WorkTypeItemDtoInterface itemDto = workTypeItemReference.getWorkTypeItemInfo(dto.getWorkTypeCode(), lastDate,
				workTypeItemCode);
		if (itemDto == null) {
			return null;
		}
		// 勤怠時刻取得
		return getTime(itemDto.getWorkTypeItemValue(), dto.getWorkDate());
	}
	
	/**
	 * 勤怠時間を取得する。<br>
	 * 勤務形態項目情報から取得した値と勤怠一覧情報の勤務日を用い、勤怠時間を取得する。<br>
	 * @param dto              勤怠一覧情報
	 * @param workTypeItemCode 勤務形態項目コード
	 * @return 勤怠時間
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected Integer getWorkTypeMinutes(AttendanceListDto dto, String workTypeItemCode) throws MospException {
		// 勤務形態項目情報取得
		WorkTypeItemDtoInterface itemDto = workTypeItemReference.getWorkTypeItemInfo(dto.getWorkTypeCode(), lastDate,
				workTypeItemCode);
		if (itemDto == null) {
			return 0;
		}
		// 勤務形態項目情報設定値を取得
		Date date = itemDto.getWorkTypeItemValue();
		// 基準日付との差を取得
		int minutes = getDefferenceMinutes(getDefaultStandardDate(), date);
		// 勤怠時刻取得
		return Integer.valueOf(minutes);
	}
	
	/**
	 * 勤怠一覧情報に備考を追加する。<br>
	 * @param dto    追加対象勤怠一覧情報
	 * @param remark 追加備考
	 */
	protected void addRemark(AttendanceListDto dto, String remark) {
		// 追加要否確認
		if (remark == null || remark.isEmpty()) {
			return;
		}
		// 勤怠一覧情報の備考を取得
		StringBuffer sb = new StringBuffer();
		// 設定済の備考を追加
		if (dto.getRemark() != null && dto.getRemark().isEmpty() == false) {
			sb.append(dto.getRemark());
		}
		// 備考確認
		if (sb.indexOf(remark) >= 0) {
			// 追加する備考が既に設定されている場合
			return;
		}
		// 空でない場合
		if (sb.length() > 0) {
			// 区切文字追加
			sb.append(SEPARATOR_ATTENDANCE_REMARK);
		}
		// 備考追加
		sb.append(remark);
		// 勤怠一覧情報に設定
		dto.setRemark(sb.toString());
	}
	
	/**
	 * 回数を取得する。<br>
	 * 時間がnull、或いは0の場合は0を返す。<br>
	 * それ以外の場合は1を返す。<br>
	 * @param minutes 時間(分)
	 * @return 回数
	 */
	protected int countHours(Integer minutes) {
		// 時間確認
		if (minutes == null || minutes.intValue() == 0) {
			return 0;
		}
		return 1;
	}
	
	/**
	 * 時間文字列を取得する(Integer→String)。<br>
	 * 時間を文字列(小数点以下2桁)で表す。<br>
	 * 小数点以下2桁は、分を表す。<br>
	 * @param minutes    対象時間(分)
	 * @param needHyphen ゼロ時ハイフン表示要否(true：ゼロ時ハイフン、false：ゼロ時はゼロ)
	 * @return 時間文字列(0.00)
	 */
	protected String getStringHours(Integer minutes, boolean needHyphen) {
		// 時間確認
		if (minutes == null) {
			// 時間がnullならハイフン
			return getHyphenNaming();
		}
		if (needHyphen && minutes.intValue() == 0) {
			// 時間が0ならハイフン
			return getHyphenNaming();
		}
		// 時間文字列準備
		StringBuffer sb = new StringBuffer();
		// 時間
		sb.append(minutes.intValue() / TimeConst.CODE_DEFINITION_HOUR);
		// 区切文字
		sb.append(SEPARATOR_HOURS);
		// 分
		int remainder = minutes.intValue() % TimeConst.CODE_DEFINITION_HOUR;
		// 数値フォーマットクラス準備
		NumberFormat nf = NumberFormat.getNumberInstance();
		// 丸め方法指定(切捨)
		nf.setRoundingMode(RoundingMode.DOWN);
		// 桁数指定
		nf.setMinimumIntegerDigits(HOURS_DIGITS);
		sb.append(nf.format(remainder));
		// 時間文字列取得
		return sb.toString();
	}
	
	/**
	 * 回数文字列を取得する(Integer→String)。<br>
	 * 回数を文字列で表す。<br>
	 * @param times 対象回数
	 * @return 回数文字列
	 */
	protected String getStringTimes(Integer times) {
		// 回数確認
		if (times == null) {
			// 回数がnullならハイフン
			return getHyphenNaming();
		}
		// 回数文字列取得
		return times.toString();
	}
	
	/**
	 * 回数文字列を取得する(Integer→String)。<br>
	 * 回数を文字列(小数点以下1桁)で表す。<br>
	 * @param times 対象回数
	 * @return 回数文字列
	 */
	protected String getStringTimes(Float times) {
		// 回数確認
		if (times == null) {
			// 回数がnullならハイフン
			return getHyphenNaming();
		}
		DecimalFormat df = new DecimalFormat("#.#");
		// 日付文字列取得
		return df.format(times);
	}
	
	/**
	 * 週の起算曜日(勤怠設定)から週期間を算出し、設定する。<br>
	 * ポータル画面等の期間で勤怠情報を取得する場合は、この週期間を利用する。<br>
	 * @param targetDate 対象日
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setWeekTerm(Date targetDate) throws MospException {
		// 週の起算曜日を取得
		int startWeek = timeSettingDto.getStartWeek();
		// 曜日確認
		while (!DateUtility.isDayOfWeek(targetDate, startWeek)) {
			// 対象日を減算
			targetDate = addDay(targetDate, -1);
		}
		// 期間初日設定
		firstDate = getDateClone(targetDate);
		// 対象日を加算
		targetDate = addDay(targetDate, 1);
		// 曜日確認
		while (!DateUtility.isDayOfWeek(targetDate, startWeek)) {
			// 対象日を加算
			targetDate = addDay(targetDate, 1);
		}
		// 期間最終日設定
		lastDate = addDay(targetDate, -1);
	}
	
	/**
	 * 月日を取得する。<br>
	 * @param date 日付
	 * @return 月日
	 */
	protected String getStringMonthDay(Date date) {
		return DateUtility.getStringDate(date, "MM/dd");
	}
	
	/**
	 * 曜日を取得する。<br>
	 * @param date 日付
	 * @return 曜日
	 */
	protected String getStringDayOfWeek(Date date) {
		return DateUtility.getStringDayOfWeek(date);
	}
	
	// TODO 消去予定
	/**
	 * 締日情報が存在しない場合のエラーメッセージを設定する。<br>
	 * @param targetDate 基準日
	 */
	protected void addCutoffNotExistErrorMessage(Date targetDate) {
		String[] rep = { getStringDate(targetDate),
			mospParams.getName("CutoffDate") + mospParams.getName("Information") };
		mospParams.addErrorMessage(TimeMessageConst.MSG_SETTING_APPLICATION_DEFECT, rep);
	}
	
	/**
	 * 所定代休略称を取得する。<br>
	 * @return 所定代休略称
	 */
	protected String getPrescribedSubHolidayAbbrNaming() {
		return mospParams.getName("PrescribedAbbreviation") + mospParams.getName("CompensatoryHoliday");
	}
	
	/**
	 * 法定代休略称を取得する。<br>
	 * @return 法定代休略称
	 */
	protected String getLegalSubHolidayAbbrNaming() {
		return mospParams.getName("LegalAbbreviation") + mospParams.getName("CompensatoryHoliday");
	}
	
	/**
	 * 所定振替休日略称を取得する。<br>
	 * @return 所定振替休日略称
	 */
	protected String getPrescribedSubstituteAbbrNaming() {
		return mospParams.getName("PrescribedAbbreviation") + mospParams.getName("ClosedVibration");
	}
	
	/**
	 * 法定振替休日略称を取得する。<br>
	 * @return 法定振替休日略称
	 */
	protected String getLegalSubstituteAbbrNaming() {
		return mospParams.getName("LegalAbbreviation") + mospParams.getName("ClosedVibration");
	}
	
	/**
	 * 予定名称を取得する。<br>
	 * @return 予定名称
	 */
	protected String getScheduleNaming() {
		return mospParams.getName("Schedule");
	}
	
	/**
	 * 承認済名称を取得する。<br>
	 * @return 承認済名称
	 */
	protected String getApplovedNaming() {
		return mospParams.getName("Approval") + mospParams.getName("Finish");
	}
	
	/**
	 * 直行名称を取得する。<br>
	 * @return 直行名称
	 */
	protected String getDirectStartNaming() {
		return mospParams.getName("DirectStart");
	}
	
	/**
	 * 直帰名称を取得する。<br>
	 * @return 直帰名称
	 */
	protected String getDirectEndNaming() {
		return mospParams.getName("DirectEnd");
	}
	
	/**
	 * 有給休暇略称を取得する。<br>
	 * @return 有給休暇略称
	 */
	protected String getPaidHolidayAbbrNaming() {
		return mospParams.getName("PaidHolidayAbbr");
	}
	
	/**
	 * ストック休暇略称を取得する。<br>
	 * @return ストック休暇略称
	 */
	protected String getStockHolidayAbbrNaming() {
		return mospParams.getName("StockHolidayAbbr");
	}
	
	/**
	 * 休暇名称を取得する。<br>
	 * @return 休暇名称
	 */
	protected String getHolidayNaming() {
		return mospParams.getName("Vacation");
	}
	
	/**
	 * 休日出勤略称を取得する。<br>
	 * @return 休日出勤略称
	 */
	protected String getWorkOnHolidayAbbrNaming() {
		return mospParams.getName("WorkingHoliday");
	}
	
	/**
	 * 振替出勤略称を取得する。<br>
	 * @return 振替出勤略称
	 */
	protected String getSubstituteWorkAbbrNaming() {
		return mospParams.getName("SubstituteAbbr") + mospParams.getName("GoingWorkAbbr");
	}
	
	/**
	 * 振替休日略称を取得する。<br>
	 * @return 振替休日略称
	 */
	protected String getSubstituteAbbrNaming() {
		return mospParams.getName("SubstituteAbbr");
	}
	
	/**
	 * 代休略称を取得する。<br>
	 * @return 代休略称
	 */
	protected String getSubHolidayAbbrNaming() {
		return mospParams.getName("Generation");
	}
	
	/**
	 * 残業略称を取得する。<br>
	 * @return 残業略称
	 */
	protected String getOvertimeAbbrNaming() {
		return mospParams.getName("OvertimeAbbr");
	}
	
	/**
	 * 時差出勤略称を取得する。<br>
	 * @return 時差出勤略称
	 */
	protected String getTimeDefferenceAbbrNaming() {
		return mospParams.getName("TimeDefferenceAbbr");
	}
	
	/**
	 * 承認済略称を取得する。<br>
	 * @return 承認済略称
	 */
	protected String getCompleteApprovalAbbrNaming() {
		return mospParams.getName("Finish");
	}
	
	/**
	 * 差戻略称を取得する。<br>
	 * @return 差戻略称
	 */
	protected String getRevertAbbrNaming() {
		return mospParams.getName("Back");
	}
	
	/**
	 * 申請略称を取得する。<br>
	 * @return 申請略称
	 */
	protected String getApprovalAbbrNaming() {
		return mospParams.getName("Register");
	}
	
	/**
	 * 下書略称を取得する。<br>
	 * @return 下書略称
	 */
	protected String getDraftAbbrNaming() {
		return mospParams.getName("Under");
	}
	
	/**
	 * 本人修正略称を取得する。<br>
	 * @return 本人修正略称
	 */
	protected String getSelfCorrectAbbrNaming() {
		return mospParams.getName("CorrectionHistory");
	}
	
	/**
	 * 他人修正略称を取得する。<br>
	 * @return 他人修正略称
	 */
	protected String getOtherCorrectAbbrNaming() {
		return mospParams.getName("Other");
	}
	
	/**
	 * 予定確認や予定簿の表示について、<br>
	 * 申請(承認済)によって勤務時間の表示を決める。<br>
	 * @throws MospException 例外が発生した時
	 */
	protected void setApprovalTime() throws MospException {
		// 休暇申請毎に処理
		for (HolidayRequestDtoInterface holidayRequestDto : holidayRequestList) {
			// 休暇日準備
			Date holidayDate = holidayRequestDto.getRequestStartDate();
			// 休暇日(休暇期間)ごとに処理
			while (holidayDate.compareTo(holidayRequestDto.getRequestEndDate()) <= 0) {
				// 対象日における勤怠一覧情報を取得
				AttendanceListDto dto = getAttendanceListDto(holidayDate);
				// 勤怠一覧情報確認
				if (dto == null) {
					holidayDate = addDay(holidayDate, 1);
					// 処理をしない
					continue;
				}
				// 休暇範囲確認
				switch (holidayRequestDto.getHolidayRange()) {
					case TimeConst.CODE_HOLIDAY_RANGE_AM:
						// 午前休の場合
						// 半休時間取得
						Date backStart = getWorkTypeTime(dto, TimeConst.CODE_BACKSTART);
						Date backEnd = getWorkTypeTime(dto, TimeConst.CODE_BACKEND);
						// 半休時間確認
						if (backStart == null || backEnd == null) {
							break;
						}
						// 始業時間
						dto.setStartTime(backStart);
						// 終業時間
						dto.setEndTime(backEnd);
						// 休憩時間を消去
						dto.setRestTime(null);
						// 勤務時間
						dto.setWorkTime(getDefferenceMinutes(backStart, backEnd));
						break;
					case TimeConst.CODE_HOLIDAY_RANGE_PM:
						// 午後休の場合
						// 半休時間取得
						Date frontStart = getWorkTypeTime(dto, TimeConst.CODE_FRONTSTART);
						Date frontEnd = getWorkTypeTime(dto, TimeConst.CODE_FRONTEND);
						// 半休時間確認
						if (frontStart == null || frontEnd == null) {
							break;
						}
						// 始業時間
						dto.setStartTime(frontStart);
						// 終業時間
						dto.setEndTime(frontEnd);
						// 休憩時間を消去
						dto.setRestTime(null);
						// 勤務時間
						dto.setWorkTime(getDefferenceMinutes(frontStart, frontEnd));
						break;
					default:
						// 始業時間
						dto.setStartTime(null);
						// 終業時間
						dto.setEndTime(null);
						// 勤務時間
						dto.setWorkTime(null);
						// 休憩時間
						dto.setRestTime(null);
				}
				// 休暇日加算
				holidayDate = addDay(holidayDate, 1);
			}
		}
		// 代休申請毎に処理
		for (SubHolidayRequestDtoInterface subHolidayRequestDto : subHolidayRequestList) {
			// 対象日における勤怠一覧情報を取得
			AttendanceListDto dto = getAttendanceListDto(subHolidayRequestDto.getRequestDate());
			// 始業時間
			dto.setStartTime(null);
			// 終業時間
			dto.setEndTime(null);
			// 勤務時間
			dto.setWorkTime(null);
			// 休憩時間
			dto.setRestTime(null);
			
		}
		// 振替休日申請毎に処理
		for (SubstituteDtoInterface substituteDto : substituteList) {
			// 対象日における勤怠一覧情報を取得
			AttendanceListDto dto = getAttendanceListDto(substituteDto.getSubstituteDate());
			// 始業時間
			dto.setStartTime(null);
			// 終業時間
			dto.setEndTime(null);
			// 勤務時間
			dto.setWorkTime(null);
			// 休憩時間
			dto.setRestTime(null);
		}
		// 申請した振出・休出勤務時刻及び勤務形態を表示
		for (WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto : workOnHolidayRequestList) {
			// 対象日における勤怠一覧情報を取得
			AttendanceListDto dto = getAttendanceListDto(workOnHolidayRequestDto.getRequestDate());
			// 振替出勤区別
			switch (workOnHolidayRequestDto.getSubstitute()) {
				case TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_ON:
					// 振替出勤の場合
					// 勤怠一覧情報に勤務形態の内容を設定
					setDtoFields(dto, getWorkOnHolidayWorkType(workOnHolidayRequestDto), true, true);
					break;
				case TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_OFF:
					// 休日出勤の場合
					// 勤怠一覧情報に勤務形態の内容を設定
					setDtoFields(dto, getWorkOnHolidayWorkType(workOnHolidayRequestDto), true, true);
					// 始業時間
					dto.setStartTime(workOnHolidayRequestDto.getStartTime());
					// 終業時間
					dto.setEndTime(workOnHolidayRequestDto.getEndTime());
					// 時間差を計算
					int minutes = getDefferenceMinutes(dto.getStartTime(), dto.getEndTime());
					// 勤務時間
					dto.setWorkTime(minutes);
					break;
				default:
			}
		}
	}
}
