/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.report.action;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.orangesignal.OrangeSignalUtility;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.file.ExportDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.ExportTableReferenceBeanInterface;
import jp.mosp.time.dto.settings.CutoffDtoInterface;
import jp.mosp.time.file.vo.TimeExportListVo;
import jp.mosp.time.utils.TimeUtility;

/**
 * CSV出力を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_EXECUTION}
 * </li></ul>
 */
public class TimeExportAction extends TimeAction {
	
	/**
	 * 実行コマンド。<br>
	 * <br>
	 * エクスポートを実行する。<br>
	 * 実行時にエクスポートマスタが決定していない、対象ファイルが選択されていない場合は
	 * エラーメッセージにて通知し、処理は実行されない。<br>
	 */
	public static final String	CMD_EXECUTION	= "TM3315";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public TimeExportAction() {
		super();
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new TimeExportListVo();
	}
	
	@Override
	public void action() throws MospException {
		if (mospParams.getCommand().equals(CMD_EXECUTION)) {
			// 実行
			prepareVo();
			execution();
		}
	}
	
	/**
	 * エクスポートの実行を行う。<br>
	 * @throws MospException 比較クラスのインスタンス生成に失敗した場合
	 */
	protected void execution() throws MospException {
		// VO準備
		TimeExportListVo vo = (TimeExportListVo)mospParams.getVo();
		// 検索クラス取得
		ExportTableReferenceBeanInterface exportTable = timeReference().exportTable();
		exportTable.setExportCode(vo.getRadSelect());
		exportTable.setStartYear(Integer.parseInt(vo.getTxtStartYear()));
		exportTable.setStartMonth(Integer.parseInt(vo.getTxtStartMonth()));
		exportTable.setEndYear(Integer.parseInt(vo.getTxtEndYear()));
		exportTable.setEndMonth(Integer.parseInt(vo.getTxtEndMonth()));
		exportTable.setCutoffCode(vo.getPltCutoff());
		exportTable.setWorkPlaceCode(vo.getPltWorkPlace());
		exportTable.setEmploymentCode(vo.getPltEmployment());
		exportTable.setSectionCode(vo.getPltSection());
		exportTable.setPositionCode(vo.getPltPosition());
		
		// CSVファイル名作成処理開始
		CutoffDtoInterface cutOffDto = timeReference().cutoff().getCutoffInfo(vo.getPltCutoff(),
				getDate(vo.getTxtStartYear(), vo.getTxtStartMonth(), "1"));
		if (cutOffDto == null) {
			// 該当する締日情報が存在しない
			String errorMes = mospParams.getName("CutoffDate");
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, errorMes, null);
			// エラー発生時はチェックボタンの選択状態を初期化する
			vo.setRadSelect("");
			return;
		}
		Date startDate = TimeUtility.getCutoffFirstDate(cutOffDto.getCutoffDate(), Integer
			.valueOf(vo.getTxtStartYear()), Integer.valueOf(vo.getTxtStartMonth()));
		Date endDate = TimeUtility.getCutoffLastDate(cutOffDto.getCutoffDate(), Integer.valueOf(vo.getTxtEndYear()),
				Integer.valueOf(vo.getTxtEndMonth()));
		// CSVファイル名作成
		StringBuffer sb = new StringBuffer();
		// エクスポートコード
		sb.append(vo.getRadSelect());
		// ハイフン
		sb.append(mospParams.getName("Hyphen"));
		// 開始年
		sb.append(getStringYear(startDate));
		// 開始月
		sb.append(getStringMonth(startDate));
		// 開始日
		sb.append(getStringDay(startDate));
		// ハイフン
		sb.append(mospParams.getName("Hyphen"));
		// 終了年
		sb.append(getStringYear(endDate));
		// 終了月
		sb.append(getStringMonth(endDate));
		// 終了日
		sb.append(getStringDay(endDate));
		ExportDtoInterface dto = reference().export().findForKey(vo.getRadSelect());
		if (dto == null) {
			return;
		} else if ("1".equals(dto.getType())) {
			// CSV
			sb.append(".csv");
		}
		// 送出ファイル名設定
		mospParams.setFileName(sb.toString());
		// CSVデータリスト取得
		List<String[]> csvDataList = exportTable.export();
		if (csvDataList.size() == 0) {
			// 該当するエクスポート情報が存在しない
			String errorMes = mospParams.getName("Export") + mospParams.getName("Information");
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, errorMes, null);
			// エラー発生時はチェックボタンの選択状態を初期化する
			vo.setRadSelect("");
			return;
		}
		// CSVデータリストをMosP処理情報に設定
		mospParams.setFile(OrangeSignalUtility.getOrangeSignalParams(csvDataList));
	}
	
}
