/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.portal.bean.impl;

import java.sql.Connection;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.platform.bean.portal.PortalBeanInterface;
import jp.mosp.platform.bean.portal.impl.PortalBean;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.time.bean.TimeRecordBeanInterface;
import jp.mosp.time.constant.TimeMessageConst;

/**
 * ポータル用タイムカード処理クラス。<br>
 */
public class PortalTimeCardBean extends PortalBean implements PortalBeanInterface {
	
	/**
	 * パス(ポータル用打刻機能JSP)。
	 */
	protected static final String	PATH_PORTAL_VIEW	= "/jsp/time/portal/portalTimeCard.jsp";
	
	/**
	 * パス(ポータル用打刻機能JS)。
	 */
	public static final String		JS_TIME				= "jsTime";
	
	/**
	 * ポータルパラメータキー(出勤)
	 */
	public static final String		RECODE_START_WORK	= "StartWork";
	
	/**
	 * ポータルパラメータキー(退勤)
	 */
	public static final String		RECODE_END_WORK		= "EndWork";
	
	/**
	 * ポータルパラメータキー(休憩入)
	 */
	public static final String		RECODE_START_REST	= "StartRest";
	
	/**
	 * ポータルパラメータキー(休憩戻)
	 */
	public static final String		RECODE_END_REST		= "EndRest";
	
	/**
	 * パラメータキー(押されたボタンの値)
	 */
	public static final String		PRM_RECODE_TYPE		= "RecodeType";
	

	/**
	 * {@link PortalBean#PortalBean()}を実行する。<br>
	 */
	public PortalTimeCardBean() {
		super();
	}
	
	/**
	 * {@link PortalBean#PortalBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosP処理情報
	 * @param connection DBコネクション
	 */
	protected PortalTimeCardBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() {
		// 処理無し
		
	}
	
	@Override
	public void show() {
		// ポータル用JSPパス追加
		addPortalViewList(PATH_PORTAL_VIEW);
		// ポータル用JS
		mospParams.addGeneralParam(JS_TIME, String.valueOf(getSystemTimeAndSecond().getTime()));
	}
	
	@Override
	public void regist() throws MospException {
		
		// VOから値を受け取り変数に詰める
		String recodeType = getPortalParameter(PRM_RECODE_TYPE);
		// コマンド毎の処理
		if (recodeType.equals(RECODE_START_WORK)) {
			// 出勤	
			recordStartWork();
		} else if (recodeType.equals(RECODE_END_WORK)) {
			// 退勤	
			recordEndWork();
		} else if (recodeType.equals(RECODE_START_REST)) {
			// 休憩入
			recordStartRest();
		} else if (recodeType.equals(RECODE_END_REST)) {
			// 休憩戻	
			recordEndRest();
		}
	}
	
	/**
	 * 始業を打刻する。<br>
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	protected void recordStartWork() throws MospException {
		// 始業打刻
		getTimeRecordBean().recordStartWork();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			// 打刻失敗メッセージ設定
			addRecordTimeFailedMessage();
			return;
		}
		// 打刻メッセージ設定
		addRecordTimeMessage(getNameStartWork());
		
	}
	
	/**
	 * 終業を打刻する。<br>
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	protected void recordEndWork() throws MospException {
		// 終業打刻
		getTimeRecordBean().recordEndWork();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			// 打刻失敗メッセージ設定
			addRecordTimeFailedMessage();
			return;
		}
		// 打刻メッセージ設定
		addRecordTimeMessage(getNameEndWork());
	}
	
	/**
	 * 休憩入を打刻する。<br>
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	protected void recordStartRest() throws MospException {
		// 休憩入打刻
		getTimeRecordBean().recordStartRest();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			// 打刻失敗メッセージ設定
			addRecordTimeFailedMessage();
			return;
		}
		// 打刻メッセージ設定
		addRecordTimeMessage(getNameStartRest());
	}
	
	/**
	 * 休憩戻を打刻する。<br>
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	protected void recordEndRest() throws MospException {
		// 休憩戻打刻
		getTimeRecordBean().recordEndRest();
		// 処理結果確認
		if (mospParams.hasErrorMessage()) {
			// 打刻失敗メッセージ設定
			addRecordTimeFailedMessage();
			return;
		}
		// 打刻メッセージ設定
		addRecordTimeMessage(getNameEndRest());
	}
	
	/**
	 * 打刻クラスを取得する。<br>
	 * @return 打刻クラス
	 * @throws MospException インスタンスの取得に失敗した場合
	 */
	protected TimeRecordBeanInterface getTimeRecordBean() throws MospException {
		return (TimeRecordBeanInterface)createBean(TimeRecordBeanInterface.class);
	}
	
	/**
	 * 打刻メッセージを追加する。<br>
	 * @param process 打刻対象処理
	 */
	protected void addRecordTimeMessage(String process) {
		mospParams.addMessage(TimeMessageConst.MSG_RECORD_TIME, process, getNameRecordTime());
	}
	
	/**
	 * 打刻失敗メッセージを追加する。
	 */
	protected void addRecordTimeFailedMessage() {
		mospParams.addMessage(PlatformMessageConst.MSG_PROCESS_FAILED, getNameRecordTime(), null);
	}
	
	/**
	 * 打刻名称を取得する。<br>
	 * @return 打刻名称
	 */
	protected String getNameRecordTime() {
		return mospParams.getName("RecordTime");
	}
	
	/**
	 * 始業名称を取得する。<br>
	 * @return 始業名称
	 */
	protected String getNameStartWork() {
		return mospParams.getName("StartWork");
	}
	
	/**
	 * 終業名称を取得する。<br>
	 * @return 終業名称
	 */
	protected String getNameEndWork() {
		return mospParams.getName("EndWork");
	}
	
	/**
	 * 休憩入名称を取得する。<br>
	 * @return 休憩入名称
	 */
	protected String getNameStartRest() {
		return mospParams.getName("RestTime") + mospParams.getName("Into");
	}
	
	/**
	 * 休憩戻名称を取得する。<br>
	 * @return 休憩戻名称
	 */
	protected String getNameEndRest() {
		return mospParams.getName("RestTime") + mospParams.getName("Return");
	}
	
}
