/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.workflow.WorkflowReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowRegistBeanInterface;
import jp.mosp.platform.bean.workflow.impl.WorkflowReferenceBean;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dao.workflow.WorkflowDaoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.ApplicationReferenceBeanInterface;
import jp.mosp.time.bean.CutoffUtilBeanInterface;
import jp.mosp.time.bean.HolidayRequestReferenceBeanInterface;
import jp.mosp.time.bean.HolidayRequestRegistBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dao.settings.AttendanceDaoInterface;
import jp.mosp.time.dao.settings.DifferenceRequestDaoInterface;
import jp.mosp.time.dao.settings.HolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.OvertimeRequestDaoInterface;
import jp.mosp.time.dao.settings.ScheduleDaoInterface;
import jp.mosp.time.dao.settings.ScheduleDateDaoInterface;
import jp.mosp.time.dao.settings.SubHolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.SubstituteDaoInterface;
import jp.mosp.time.dao.settings.WorkOnHolidayRequestDaoInterface;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDateDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.impl.TmdHolidayRequestDto;

/**
 * 休暇申請登録クラス。
 */
public class HolidayRequestRegistBean extends TimeBean implements HolidayRequestRegistBeanInterface {
	
	/**
	 * 休暇申請DAOクラス。<br>
	 */
	HolidayRequestDaoInterface					dao;
	/**
	 * 休暇申請参照インターフェース。<br>
	 */
	HolidayRequestReferenceBeanInterface		holidayReference;
	/**
	 * ワークフローDAOクラス。<br>
	 */
	private WorkflowDaoInterface				workflowDao;
	
	/**
	 * ワークフロー登録クラス。<br>
	 */
	private WorkflowRegistBeanInterface			workflowRegist;
	/**
	 * ワークフロー参照クラス。<br>
	 */
	private WorkflowReferenceBeanInterface		workflowReference;
	
	/**
	 * 締日ユーティリティ。<br>
	 */
	private CutoffUtilBeanInterface				cutoffUtil;
	
	private ApplicationReferenceBeanInterface	application;
	private ScheduleDaoInterface				scheduleDao;
	private ScheduleDateDaoInterface			scheduleDateDao;
	private WorkOnHolidayRequestDaoInterface	workOnHolidayRequestDao;
	private SubHolidayRequestDaoInterface		subHolidayRequestDao;
	private SubstituteDaoInterface				substituteDao;
	private OvertimeRequestDaoInterface			overtimeRequestDao;
	private DifferenceRequestDaoInterface		differenceRequestDao;
	private AttendanceDaoInterface				attendanceDao;
	private ScheduleDateReferenceBean			scheduleDateReference;
	private ScheduleReferenceBean				scheduleReference;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public HolidayRequestRegistBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public HolidayRequestRegistBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		// DAO準備
		dao = (HolidayRequestDaoInterface)createDao(HolidayRequestDaoInterface.class);
		holidayReference = (HolidayRequestReferenceBeanInterface)createBean(HolidayRequestReferenceBeanInterface.class);
		workflowDao = (WorkflowDaoInterface)createDao(WorkflowDaoInterface.class);
		workflowReference = (WorkflowReferenceBean)createBean(WorkflowReferenceBean.class);
		workOnHolidayRequestDao = (WorkOnHolidayRequestDaoInterface)createDao(WorkOnHolidayRequestDaoInterface.class);
		application = (ApplicationReferenceBeanInterface)createBean(ApplicationReferenceBeanInterface.class);
		scheduleDao = (ScheduleDaoInterface)createDao(ScheduleDaoInterface.class);
		scheduleDateDao = (ScheduleDateDaoInterface)createDao(ScheduleDateDaoInterface.class);
		subHolidayRequestDao = (SubHolidayRequestDaoInterface)createDao(SubHolidayRequestDaoInterface.class);
		substituteDao = (SubstituteDaoInterface)createDao(SubstituteDaoInterface.class);
		overtimeRequestDao = (OvertimeRequestDaoInterface)createDao(OvertimeRequestDaoInterface.class);
		differenceRequestDao = (DifferenceRequestDaoInterface)createDao(DifferenceRequestDaoInterface.class);
		attendanceDao = (AttendanceDaoInterface)createDao(AttendanceDaoInterface.class);
		scheduleDateReference = (ScheduleDateReferenceBean)createBean(ScheduleDateReferenceBean.class);
		scheduleReference = (ScheduleReferenceBean)createBean(ScheduleReferenceBean.class);
		cutoffUtil = (CutoffUtilBeanInterface)createBean(CutoffUtilBeanInterface.class);
	}
	
	@Override
	public HolidayRequestDtoInterface getInitDto() {
		return new TmdHolidayRequestDto();
	}
	
	@Override
	public void insert(HolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 新規登録情報の検証
		checkInsert(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdHolidayRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void update(HolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 履歴更新情報の検証
		checkUpdate(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// 論理削除
		logicalDelete(dao, dto.getTmdHolidayRequestId());
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdHolidayRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void update(long[] idArray) throws MospException {
		if (idArray == null || idArray.length == 0) {
			// 表示例 必ず一件はチェックボックスを選択してください。
			mospParams.addErrorMessage(PlatformMessageConst.MSG_CHECK, null);
			return;
		}
		// Bean初期化
		workflowRegist = (WorkflowRegistBeanInterface)createBean(WorkflowRegistBeanInterface.class);
		// 処理ワークフロー情報リスト準備
		List<WorkflowDtoInterface> workflowList = new ArrayList<WorkflowDtoInterface>();
		for (long id : idArray) {
			// DTOの準備
			BaseDtoInterface baseDto = findForKey(dao, id, true);
			checkExclusive(baseDto);
			if (mospParams.hasErrorMessage()) {
				continue;
			}
			HolidayRequestDtoInterface dto = (HolidayRequestDtoInterface)baseDto;
			// 申請時の確認
			checkAppli(dto);
			if (mospParams.hasErrorMessage()) {
				continue;
			}
			// ワークフローDTOの準備
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(dto.getWorkflow());
			// 申請
			workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestStartDate(),
					PlatformConst.WORKFLOW_TYPE_TIME, null);
			// 処理ワークフロー情報リストへ追加
			if (workflowDto != null) {
				workflowList.add(workflowDto);
			}
		}
	}
	
	@Override
	public void regist(HolidayRequestDtoInterface dto) throws MospException {
		if (dao.findForKey(dto.getTmdHolidayRequestId(), false) == null) {
			// 新規登録
			insert(dto);
		} else {
			// 履歴追加
			add(dto);
		}
	}
	
	@Override
	public void add(HolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 履歴追加情報の検証
		checkAdd(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// 論理削除
		logicalDelete(dao, dto.getTmdHolidayRequestId());
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdHolidayRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	/**
	 * 新規登録時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkInsert(HolidayRequestDtoInterface dto) throws MospException {
		// 対象レコードの有効日が重複していないかを確認
		checkDuplicateInsert(dao.findForKeyOnWorkflow(dto.getPersonalId(), dto.getRequestStartDate(), dto
			.getHolidayType1(), dto.getHolidayType2(), dto.getHolidayRange(), dto.getStartTime()));
	}
	
	/**
	 * 履歴更新時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkUpdate(HolidayRequestDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getTmdHolidayRequestId());
	}
	
	/**
	 * 履歴追加時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkAdd(HolidayRequestDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getTmdHolidayRequestId());
	}
	
	@Override
	public void validate(HolidayRequestDtoInterface dto) throws MospException {
		// 基本情報のチェック
		holidayReference.chkBasicInfo(dto.getPersonalId(), dto.getRequestStartDate());
	}
	
	@Override
	public void checkDraft(HolidayRequestDtoInterface dto) throws MospException {
		checkTemporaryClosingFinal(dto);
		checkRequestDate(dto);
	}
	
	@Override
	public void checkAppli(HolidayRequestDtoInterface dto) throws MospException {
		// 下書き同様の処理を行う。
		checkDraft(dto);
		// 休暇申請の申請期間チェック。
		checkPeriod(dto);
		// 休暇申請の重複チェック。
		checkHolidayOverlap(dto);
		// 他の申請チェック。
		checkRequest(dto);
		// 勤怠の申請チェック
		checkAttendance(dto);
		// 休暇申請の項目の必須チェック。
		checkRequired(dto);
	}
	
	@Override
	public void checkWithdrawn(HolidayRequestDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
	}
	
	@Override
	public void checkApproval(HolidayRequestDtoInterface dto) throws MospException {
		// 申請時と同様の処理を行う
		checkAppli(dto);
	}
	
	@Override
	public void checkCancel(HolidayRequestDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
	}
	
	@Override
	public void checkRequestDate(HolidayRequestDtoInterface dto) throws MospException {
		if (dto.getRequestStartDate().after(dto.getRequestEndDate())) {
			// 休暇申請の重複チェックでメッセージを追加するため、ここのメッセージ設定は無し。
			return;
		}
		Date requestDate = dto.getRequestStartDate();
		while (!requestDate.after(dto.getRequestEndDate())) {
			// 設定適用
			ApplicationDtoInterface applicationDto = application.findForPerson(dto.getPersonalId(), requestDate);
			application.chkExistApplication(applicationDto, requestDate);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			// カレンダ
			ScheduleDtoInterface scheduleDto = scheduleDao.findForInfo(applicationDto.getScheduleCode(), requestDate);
			scheduleReference.chkExistSchedule(scheduleDto, requestDate);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			// カレンダ日
			ScheduleDateDtoInterface scheduleDateDto = scheduleDateDao.findForInfo(scheduleDto.getScheduleCode(),
					scheduleDto.getActivateDate(), requestDate);
			scheduleDateReference.chkExistScheduleDate(scheduleDateDto, requestDate);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			if (scheduleDateDto.getWorkTypeCode() == null || scheduleDateDto.getWorkTypeCode().isEmpty()) {
				// 勤務形態がない場合
				addWorkTypeNotExistErrorMessage(requestDate);
				return;
			}
			if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())
					|| TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())) {
				// 法定休日又は所定休日の場合
				if (requestDate.equals(dto.getRequestStartDate()) || requestDate.equals(dto.getRequestEndDate())) {
					// 休暇開始日又は休暇終了日の場合はエラーとする
					addHolidayTargetWorkDateHolidayErrorMessage(requestDate);
					return;
				}
			}
			// 1日加算
			requestDate = addDay(requestDate, 1);
		}
	}
	
	@Override
	public void checkHolidayOverlap(HolidayRequestDtoInterface dto) throws MospException {
		if (dto.getRequestStartDate().after(dto.getRequestEndDate())) {
			// 開始日が終了日より後の場合
			// 休暇年月日の入力内容が不正です。終了日は開始日よりも後になるよう入力してください。
			addHolidayRequestDateErrorMessage();
			return;
		}
		Date requestDate = dto.getRequestStartDate();
		while (!requestDate.after(dto.getRequestEndDate())) {
			// 休暇申請リスト取得
			List<HolidayRequestDtoInterface> list = dao.findForList(dto.getPersonalId(), requestDate);
			for (HolidayRequestDtoInterface requestDto : list) {
				// ワークフローの取得
				WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
				if (workflowDto == null) {
					continue;
				}
				if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					// 取下の場合
					continue;
				}
				if (dto.getWorkflow() == workflowDto.getWorkflow()) {
					// ワークフロー番号が同じ場合は同じ申請
					continue;
				}
				// 申請の休暇範囲が全休の場合
				if (dto.getHolidayRange() == 1 || requestDto.getHolidayRange() == 1) {
					// 既にデータが登録されている場合
					// 表示例 休暇内容が重複しています。休暇年月日を選択し直してください。
					addHolidayOverlapRange1ErrorMessage();
					return;
				}
				// 申請の休暇範囲が前休、後休の場合 
				if (dto.getHolidayRange() == 2 || dto.getHolidayRange() == 3) {
					// 登録されている休暇範囲が重複している場合
					if (dto.getHolidayRange() == requestDto.getHolidayRange()) {
						// 表示例 休暇内容が重複しています。休暇年月日または休暇範囲を選択し直してください。
						addHolidayOverlapRange2ErrorMessage();
						return;
					}
				}
			}
			// 1日加算
			requestDate = addDay(requestDate, 1);
		}
	}
	
	@Override
	public void checkRequest(HolidayRequestDtoInterface dto) throws MospException {
		if (dto.getRequestStartDate().after(dto.getRequestEndDate())) {
			// 休暇申請の重複チェックでメッセージを追加するため、ここのメッセージ設定は無し。
			return;
		}
		String personalId = dto.getPersonalId();
		Date requestDate = dto.getRequestStartDate();
		while (!requestDate.after(dto.getRequestEndDate())) {
			// 休出申請
			WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto = workOnHolidayRequestDao.findForKeyOnWorkflow(
					personalId, requestDate);
			if (workOnHolidayRequestDto != null) {
				// 休出申請がある場合
				WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(workOnHolidayRequestDto
					.getWorkflow());
				if (workflowDto != null) {
					if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
							&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
						// 下書でなく且つ取下でない場合
						addHolidayTargetDateRequestErrorMessage(requestDate);
						return;
					}
				}
			}
			boolean amFlag = false;
			boolean pmFlag = false;
			// 振替休日
			List<SubstituteDtoInterface> substituteList = substituteDao.findForList(personalId, requestDate);
			for (SubstituteDtoInterface substituteDto : substituteList) {
				// ワークフローの取得
				WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(substituteDto.getWorkflow());
				if (workflowDto == null) {
					continue;
				}
				if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
						&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					// 下書でなく且つ取下でない場合
					if (dto.getHolidayRange() == 1) {
						// 全休の場合
						addHolidayTargetDateRequestErrorMessage(requestDate);
						return;
					}
					int range = substituteDto.getSubstituteRange();
					if (range == 1) {
						// 全休の場合
						addHolidayTargetDateRequestErrorMessage(requestDate);
						return;
					} else if (range == 2 || range == 3) {
						// 午前休又は午後休の場合
						if (range == 2) {
							amFlag = true;
						}
						if (range == 3) {
							pmFlag = true;
						}
						if (dto.getHolidayRange() == range) {
							addHolidayTargetDateRequestErrorMessage(requestDate);
							return;
						}
					}
				}
			}
			// 代休申請
			List<SubHolidayRequestDtoInterface> subHolidayRequestList = subHolidayRequestDao.findForList(personalId,
					requestDate);
			for (SubHolidayRequestDtoInterface requestDto : subHolidayRequestList) {
				// ワークフローの取得
				WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
				if (workflowDto == null) {
					continue;
				}
				if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
						&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					// 下書でなく且つ取下でない場合
					if (dto.getHolidayRange() == 1) {
						// 全休の場合
						addHolidayTargetDateRequestErrorMessage(requestDate);
						return;
					}
					int range = requestDto.getHolidayRange();
					if (range == 1) {
						// 全休の場合
						addHolidayTargetDateRequestErrorMessage(requestDate);
						return;
					} else if (range == 2 || range == 3) {
						// 午前休又は午後休の場合
						if (range == 2) {
							amFlag = true;
						}
						if (range == 3) {
							pmFlag = true;
						}
						if (dto.getHolidayRange() == range) {
							addHolidayTargetDateRequestErrorMessage(requestDate);
							return;
						}
					}
				}
			}
			if (amFlag && pmFlag) {
				addHolidayTargetDateRequestErrorMessage(requestDate);
				return;
			}
			if (dto.getHolidayRange() == 1) {
				// 全休の場合
				// 残業申請
				List<OvertimeRequestDtoInterface> overtimeRequestList = overtimeRequestDao.findForList(personalId,
						requestDate);
				for (OvertimeRequestDtoInterface requestDto : overtimeRequestList) {
					// ワークフローの取得
					WorkflowDtoInterface workflowDto = workflowReference
						.getLatestWorkflowInfo(requestDto.getWorkflow());
					if (workflowDto == null) {
						continue;
					}
					if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
							&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
						// 下書でなく且つ取下でない場合
						addHolidayTargetDateRequestErrorMessage(requestDate);
						return;
					}
				}
				// 時差出勤申請
				DifferenceRequestDtoInterface differenceRequestDto = differenceRequestDao.findForKeyOnWorkflow(
						personalId, requestDate);
				if (differenceRequestDto != null) {
					// 時差出勤申請がある場合
					WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(differenceRequestDto
						.getWorkflow());
					if (workflowDto != null) {
						if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
								&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
							// 下書でなく且つ取下でない場合
							addHolidayTargetDateRequestErrorMessage(requestDate);
							return;
						}
					}
				}
			}
			// 1日加算
			requestDate = addDay(requestDate, 1);
		}
	}
	
	@Override
	public void checkAttendance(HolidayRequestDtoInterface dto) throws MospException {
		if (dto.getRequestStartDate().after(dto.getRequestEndDate())) {
			// 開始日が終了日より後の場合
			return;
		}
		Date requestDate = dto.getRequestStartDate();
		while (!requestDate.after(dto.getRequestEndDate())) {
			Date targetDate = requestDate;
			// 1日加算
			requestDate = addDay(requestDate, 1);
			AttendanceDtoInterface attendanceDto = attendanceDao.findForKey(dto.getPersonalId(), targetDate, 1);
			if (attendanceDto == null) {
				continue;
			}
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(attendanceDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 取下の場合
				continue;
			}
			if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
				// 下書の場合
				continue;
			}
			if (workflowDto.getWorkflowStage() == 0
					&& PlatformConst.CODE_STATUS_REVERT.equals(workflowDto.getWorkflowStatus())) {
				// 1次戻の場合
				continue;
			}
			// （日付）は既に勤怠の申請が行われています。（申請区分毎の日付名称）を選択し直してください。
			addHolidayTargetWorkDateAttendanceRequestErrorMessage(targetDate);
			return;
		}
	}
	
	@Override
	public void checkRequired(HolidayRequestDtoInterface dto) {
		if (dto.getRequestReason().isEmpty()) {
			addHolidayRequestReasonErrorMessage();
		}
	}
	
	@Override
	public void checkPeriod(HolidayRequestDtoInterface dto) {
		if (dto.getRequestStartDate().after(DateUtility.addMonth(DateUtility.getSystemDate(), 6))) {
			addHolidayPeriodErrorMessage();
		}
	}
	
	@Override
	public void checkTemporaryClosingFinal(HolidayRequestDtoInterface dto) throws MospException {
		// 対象個人IDにつき対象日付が未締であるかの確認
		cutoffUtil.checkTighten(dto.getPersonalId(), dto.getRequestStartDate(), getNameVacationDay());
	}
	
	/**
	 * 休暇年月日名称を取得する。
	 * @return 休暇年月日名称
	 */
	protected String getNameVacationDay() {
		return mospParams.getName("Vacation") + mospParams.getName("Year") + mospParams.getName("Month")
				+ mospParams.getName("Day");
	}
	
}
