package jp.mosp.time.bean;

import java.util.Date;

import jp.mosp.framework.base.MospException;

/**
 * 打刻クラスインターフェース。
 */
public interface TimeRecordBeanInterface {
	
	/**
	 * 始業を打刻する。<br>
	 * 対象個人ID及び打刻日時で、打刻する。<br>
	 * <br>
	 * 打刻した日の勤怠情報として、登録する。<br>
	 * 打刻した日の勤怠情報が既に存在する場合は、打刻しない。<br>
	 * <br>
	 * 打刻した勤怠情報は、下書き状態になる。<br>
	 * <br>
	 * @param personalId 対象個人ID
	 * @param recordTime 打刻日時
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	void recordStartWork(String personalId, Date recordTime) throws MospException;
	
	/**
	 * 終業を打刻する。<br>
	 * 対象個人ID及び打刻日時で、打刻する。<br>
	 * <br>
	 * 打刻した日の勤怠情報が存在し終業時間が登録されていない場合は、
	 * 打刻した日の勤怠情報として登録する。<br>
	 * <br>
	 * 打刻した日の勤怠情報が存在しない場合、前日の勤怠情報を確認する。<br>
	 * 前日の勤怠情報が存在し終業時間が登録されていない場合は、
	 * 前日の勤怠情報として登録する。<br>
	 * <br>
	 * 上記の場合以外は、打刻しない。<br>
	 * <br>
	 * 休憩入が入力されており休憩戻が入力されていない場合は、打刻しない。<br>
	 * <br>
	 * 打刻した勤怠情報は、申請状態になる。<br>
	 * <br>
	 * @param personalId 対象個人ID
	 * @param recordTime 打刻日時
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	void recordEndWork(String personalId, Date recordTime) throws MospException;
	
	/**
	 * 休憩入を打刻する。<br>
	 * 対象個人ID及び打刻日時で、打刻する。<br>
	 * <br>
	 * @param personalId 対象個人ID
	 * @param recordTime 打刻日時
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	void recordStartRest(String personalId, Date recordTime) throws MospException;
	
	/**
	 * 休憩戻を打刻する。<br>
	 * 対象個人ID及び打刻日時で、打刻する。<br>
	 * <br>
	 * @param personalId 対象個人ID
	 * @param recordTime 打刻日時
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	void recordEndRest(String personalId, Date recordTime) throws MospException;
	
	/**
	 * 始業を打刻する。<br>
	 * ログインユーザの個人ID及びシステム日付で、打刻する。<br>
	 * 詳細は、{@link TimeRecordBeanInterface#recordStartWork()}参照。<br>
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	void recordStartWork() throws MospException;
	
	/**
	 * 終業を打刻する。<br>
	 * ログインユーザの個人ID及びシステム日付で、打刻する。<br>
	 * 詳細は、{@link TimeRecordBeanInterface#recordEndWork()}参照。<br>
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	void recordEndWork() throws MospException;
	
	/**
	 * 休憩入を打刻する。<br>
	 * ログインユーザの個人ID及びシステム日付で、打刻する。<br>
	 * 詳細は、{@link TimeRecordBeanInterface#recordStartRest()}参照。<br>
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	void recordStartRest() throws MospException;
	
	/**
	 * 休憩戻を打刻する。<br>
	 * ログインユーザの個人ID及びシステム日付で、打刻する。<br>
	 * 詳細は、{@link TimeRecordBeanInterface#recordEndRest()}参照。<br>
	 * @throws MospException インスタンスの取得及びSQL実行に失敗した場合
	 */
	void recordEndRest() throws MospException;
	
}
