/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.platform.bean.portal.impl;

import java.sql.Connection;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.constant.MospConst;
import jp.mosp.framework.utils.MospUtility;
import jp.mosp.platform.bean.message.MessageReferenceBeanInterface;
import jp.mosp.platform.bean.portal.PortalBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.dto.message.MessageDtoInterface;

/**
 * ポータル用メッセージ処理クラス。<br>
 */
public class PortalMessageBean extends PortalBean implements PortalBeanInterface {
	
	/**
	 * パス(ポータル用メッセージJSP)。
	 */
	protected static final String	PATH_PORTAL_VIEW				= "/jsp/platform/portal/portalMessage.jsp";
	
	/**
	 * MosPアプリケーション設定キー(ポータルメッセージ表示件数)。
	 */
	protected static final String	APP_PORTAL_MESSAGE_COUNT		= "PortalMessageCount";
	
	/**
	 * MosPアプリケーション設定キー(ポータルメッセージ比較クラス)。
	 */
	protected static final String	APP_PORTAL_MESSAGE_COMPARATORS	= "PortalMessageComparators";
	
	/**
	 * ポータルパラメータキー(メッセージ公開開始日)。
	 */
	public static final String		PRM_ARY_LBL_MESSAGE_DATE		= "aryLblMessageDate";
	
	/**
	 * ポータルパラメータキー(メッセージNo)。
	 */
	public static final String		PRM_ARY_LBL_MESSAGE_NO			= "aryLblMessageNo";
	
	/**
	 * ポータルパラメータキー(メッセージ区分)。
	 */
	public static final String		PRM_ARY_LBL_MESSAGE_TYPE		= "aryLblMessageType";
	
	/**
	 * ポータルパラメータキー(重要度)。
	 */
	public static final String		PRM_ARY_LBL_MESSAGE_IMPORTANCE	= "aryLblMessageImportance";
	
	/**
	 * ポータルパラメータキー(メッセージタイトル)。
	 */
	public static final String		PRM_ARY_LBL_MESSAGE_TITLE		= "aryLblMessageTitle";
	
	/**
	 * ポータルパラメータキー(メッセージ)。
	 */
	public static final String		PRM_ARY_LBL_MESSAGE				= "aryLblMessage";
	
	/**
	 * ポータルパラメータキー(他メッセージ数)。
	 */
	public static final String		PRM_LBL_MESSAGE_COUNT			= "lblMessageCount";
	

	/**
	 * {@link PortalBean#PortalBean()}を実行する。<br>
	 */
	public PortalMessageBean() {
		super();
	}
	
	/**
	 * {@link PortalBean#PortalBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosP処理情報
	 * @param connection DBコネクション
	 */
	protected PortalMessageBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() {
		// 処理無し
	}
	
	@Override
	public void show() throws MospException {
		// ポータル用JSPパス追加
		addPortalViewList(PATH_PORTAL_VIEW);
		// メッセージリストを取得しVOに設定
		setVoList(getMessageList(), true);
	}
	
	@Override
	public void regist() throws MospException {
		// メッセージリストを取得し全件VOに設定
		setVoList(getMessageList(), false);
	}
	
	/**
	 * メッセージリストを取得する。
	 * @return メッセージリスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected List<MessageDtoInterface> getMessageList() throws MospException {
		// メッセージ参照クラス準備
		MessageReferenceBeanInterface message = (MessageReferenceBeanInterface)createBean(MessageReferenceBeanInterface.class);
		// メッセージ取得(ログインユーザの個人ID及びシステム日付)
		List<MessageDtoInterface> list = message.getMessageList(mospParams.getUser().getPersonalId(), getSystemDate());
		// ポータルメッセージ比較クラス毎にソート
		for (String comparator : getPortalMessageComparators()) {
			// ソート
			sortList(list, comparator, true);
		}
		return list;
	}
	
	/**
	 * 検索結果リストの内容をVOに設定する。<br>
	 * @param list     対象リスト
	 * @param isCapped 上限フラグ(true：上限有り、false：全件表示)
	 */
	protected void setVoList(List<MessageDtoInterface> list, boolean isCapped) {
		// 表示件数確認
		int count = list.size();
		// 非表示件数準備
		int ohterMessageCount = 0;
		// 上限フラグ確認
		if (isCapped) {
			// ポータルメッセージ表示件数取得及び確認
			int portalMessageCount = mospParams.getApplicationProperty(APP_PORTAL_MESSAGE_COUNT, 0);
			if (portalMessageCount > 0 && count > portalMessageCount) {
				// 非表示件数設定
				ohterMessageCount = count - portalMessageCount;
				// 表示件数再設定
				count = portalMessageCount;
			}
		}
		// データ配列初期化
		String[] aryLblMessageDate = new String[count];
		String[] aryLblMessageNo = new String[count];
		String[] aryLblMessageType = new String[count];
		String[] aryLblMessageImportanec = new String[count];
		String[] aryLblMessageTitle = new String[count];
		String[] aryLblMessage = new String[count];
		// データ作成
		for (int i = 0; i < count; i++) {
			// リストから情報を取得
			MessageDtoInterface dto = list.get(i);
			// 配列に情報を設定
			aryLblMessageDate[i] = getStringDate(dto.getStartDate());
			aryLblMessageNo[i] = dto.getMessageNo();
			aryLblMessageType[i] = getCodeName(dto.getMessageType(), PlatformConst.CODE_KEY_MESSAGE_TYPE);
			aryLblMessageImportanec[i] = getCodeName(dto.getMessageImportance(),
					PlatformConst.CODE_KEY_MESSAGE_IMPORTANCE);
			aryLblMessageTitle[i] = dto.getMessageTitle();
			aryLblMessage[i] = dto.getMessageBody();
		}
		// データをVOに設定
		putPortalParameters(PRM_ARY_LBL_MESSAGE_DATE, aryLblMessageDate);
		putPortalParameters(PRM_ARY_LBL_MESSAGE_NO, aryLblMessageNo);
		putPortalParameters(PRM_ARY_LBL_MESSAGE_TYPE, aryLblMessageType);
		putPortalParameters(PRM_ARY_LBL_MESSAGE_IMPORTANCE, aryLblMessageImportanec);
		putPortalParameters(PRM_ARY_LBL_MESSAGE_TITLE, aryLblMessageTitle);
		putPortalParameters(PRM_ARY_LBL_MESSAGE, aryLblMessage);
		putPortalParameter(PRM_LBL_MESSAGE_COUNT, String.valueOf(ohterMessageCount));
	}
	
	/**
	 * MosP設定情報からポータルメッセージ比較クラス群(クラス名)を取得する。<br>
	 * @return ポータルメッセージ比較クラス群(クラス名)
	 */
	protected String[] getPortalMessageComparators() {
		// ポータルメッセージ比較クラス群(クラス名)取得
		String portalBeans = mospParams.getApplicationProperty(APP_PORTAL_MESSAGE_COMPARATORS);
		// 分割
		return MospUtility.split(portalBeans, MospConst.APP_PROPERTY_SEPARATOR);
	}
	
}
