package monalipse.utils;

import java.net.HttpURLConnection;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.swt.widgets.Display;

public class CancelableRunner
{
	private IStatusLineManager manager;
	private Display display;
	private IAction cancelAction;
	private int runCount;
	private List connections = new ArrayList();

	public CancelableRunner(IStatusLineManager manager, Display display, IAction cancelAction)
	{
		this.manager = manager;
		this.display = display;
		this.cancelAction = cancelAction;
		manager.setCancelEnabled(false);
	}
	
	public void setDisplay(Display display)
	{
		this.display = display;
	}
	
	public ICancelableProgressMonitor getProgressMonitor()
	{
		return new CancelableProgressMonitor(manager.getProgressMonitor());
	}
	
	public ICancelableProgressMonitor getSubProgressMonitor(IProgressMonitor monitor, int ticks)
	{
		return new CancelableSubProgressMonitor(monitor, ticks, SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK);
	}

	public void cancel()
	{
		synchronized(connections)
		{
			for(int i = 0; i < connections.size(); i++)
				((HttpURLConnection)connections.get(i)).disconnect();
			connections.clear();
		}
	}
	
	public void openConnection(HttpURLConnection conn)
	{
		synchronized(connections)
		{
			connections.add(conn);
		}
	}
	
	public void closeConnection(HttpURLConnection conn)
	{
		synchronized(connections)
		{
			connections.remove(conn);
			conn.disconnect();
		}
	}

	public void runAndJoin(Object serializeKey, ICancelableRunnableWithProgress run)
	{
		try
		{
			run(serializeKey, run).join();
		}
		catch (InterruptedException e)
		{
			e.printStackTrace();
		}
	}
	
	public Thread run(final Object serializeKey, final ICancelableRunnableWithProgress run)
	{
		Thread thread = new Thread(new Runnable()
			{
				public void run()
				{
					if(serializeKey == null)
					{
						runDirect(run);
					}
					else
					{
						synchronized(serializeKey)
						{
							runDirect(run);
						}
					}
				}
			});
		thread.start();
		return thread;
	}
	
	public void runDirect(ICancelableRunnableWithProgress run)
	{
		try
		{
			synchronized(this)
			{
				cancelAction.setEnabled(true);
				runCount++;
			}
			run.run(getProgressMonitor());
		}
		finally
		{
			synchronized(this)
			{
				runCount--;
				cancelAction.setEnabled(runCount != 0);
			}
		}
	}
	
	public interface ICancelableRunnableWithProgress
	{
		public void run(ICancelableProgressMonitor monitor);
	}

	public interface ICancelableProgressMonitor extends IProgressMonitor
	{
		public CancelableRunner getRunner();
	}

	public class CancelableProgressMonitor implements ICancelableProgressMonitor
	{
		private IProgressMonitor monitor;

		public CancelableProgressMonitor(IProgressMonitor monitor)
		{
			if(monitor == null)
				this.monitor = new NullProgressMonitor();
			else
				this.monitor = monitor;
		}
		
		public CancelableRunner getRunner()
		{
			return CancelableRunner.this;
		}

		public void beginTask(final String name, final int totalWork)
		{
			if(!display.isDisposed())
			{
				Runnable run = new Runnable()
					{
						public void run()
						{
							monitor.beginTask(name, totalWork);
						}
					};
				if(Thread.currentThread() == display.getThread())
					display.syncExec(run);
				else
					display.asyncExec(run);
			}
		}
		
		public void done()
		{
			if(!display.isDisposed())
			{
				Runnable run = new Runnable()
					{
						public void run()
						{
							monitor.done();
						}
					};
				if(Thread.currentThread() == display.getThread())
					display.syncExec(run);
				else
					display.asyncExec(run);
			}
		}
		
		public void internalWorked(final double work)
		{
			if(!display.isDisposed())
			{
				Runnable run = new Runnable()
					{
						public void run()
						{
							monitor.internalWorked(work);
						}
					};
				if(Thread.currentThread() == display.getThread())
					display.syncExec(run);
				else
					display.asyncExec(run);
			}
		}
		
		public boolean isCanceled()
		{
			return !cancelAction.isEnabled();
		}
		
		public void setCanceled(boolean b)
		{
			if(b)
				cancel();
		}
	
		public void setTaskName(final String name)
		{
			if(!display.isDisposed())
			{
				Runnable run = new Runnable()
					{
						public void run()
						{
							monitor.setTaskName(name);
						}
					};
				if(Thread.currentThread() == display.getThread())
					display.syncExec(run);
				else
					display.asyncExec(run);
			}
		}
		
		public void subTask(final String name)
		{
			if(!display.isDisposed())
			{
				Runnable run = new Runnable()
					{
						public void run()
						{
							monitor.subTask(name);
						}
					};
				if(Thread.currentThread() == display.getThread())
					display.syncExec(run);
				else
					display.asyncExec(run);
			}
		}
		
		public void worked(final int work)
		{
			if(!display.isDisposed())
			{
				Runnable run = new Runnable()
					{
						public void run()
						{
							monitor.worked(work);
						}
					};
				if(Thread.currentThread() == display.getThread())
					display.syncExec(run);
				else
					display.asyncExec(run);
			}
		}
	}

	private class CancelableSubProgressMonitor extends SubProgressMonitor implements ICancelableProgressMonitor
	{
		public CancelableSubProgressMonitor(IProgressMonitor monitor, int ticks, int style)
		{
			super(monitor, ticks, style);
		}
		
		public CancelableRunner getRunner()
		{
			return CancelableRunner.this;
		}
	}
}
