package monalipse.wizards;

import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;

import monalipse.MonalipsePlugin;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard;
import org.eclipse.ui.wizards.newresource.BasicNewResourceWizard;

public class NewMonalipseProjectCreationWizard extends BasicNewResourceWizard implements IExecutableExtension
{
	private static final String WINDOW_PROBLEMS_TITLE = MonalipsePlugin.getResourceString("NewProject.errorOpeningWindow"); //$NON-NLS-1$
	private static final String FINAL_PERSPECTIVE = "finalPerspective";  //$NON-NLS-1$
	private static final String PREFERRED_PERSPECTIVES = "preferredPerspectives";  //$NON-NLS-1$
	private static final String INITIAL_PROJECT_NAME = "monalipse.log";

	private IWorkbench workbench;
	private IStructuredSelection selection;
	private WizardNewProjectCreationPage mainPage;
	private IProject newProject;
	private IConfigurationElement configElement;
	
	public NewMonalipseProjectCreationWizard()
	{
	}

	public void addPages()
	{
		super.addPages();

		mainPage = new WizardNewProjectCreationPage("monalipseNewProjectPage") //$NON-NLS-1$
			{
				public void setPageComplete(boolean complete)
				{
					boolean exists = false;
					IProject[] proj = ResourcesPlugin.getWorkspace().getRoot().getProjects();
					for(int i = 0; i < proj.length; i++)
					{
						try
						{
							if(proj[i].getNature(MonalipsePlugin.NATURE_ID) != null)
								exists = true;
						}
						catch (CoreException e)
						{
						}
					}
					super.setPageComplete(complete && !exists);
					if(complete && exists)
						setErrorMessage(MonalipsePlugin.getResourceString("NewProject.errorAlreadyExists")); //$NON-NLS-1$
				}

				public void createControl(Composite parent)
				{
					super.createControl(parent);
					validate(parent);
				}
				
				private void validate(Control control)
				{
					if(control instanceof Composite)
					{
						Control[] children = ((Composite)control).getChildren();
						for(int i = 0; i < children.length; i++)
							validate(children[i]);
					}
					else if(control instanceof Text)
					{
						final Text text = (Text)control;
						if(text.getText().equals(INITIAL_PROJECT_NAME))
						{
							getShell().getDisplay().asyncExec(new Runnable()
								{
									public void run()
									{
										text.setText(INITIAL_PROJECT_NAME);
									}
								});
						}
					}
				}
			};
		mainPage.setTitle(MonalipsePlugin.getResourceString("NewProject.title")); //$NON-NLS-1$
		mainPage.setDescription(MonalipsePlugin.getResourceString("NewProject.description")); //$NON-NLS-1$
		mainPage.setInitialProjectName(INITIAL_PROJECT_NAME);
		addPage(mainPage);
	}

	private IProject createNewProject()
	{
		if(newProject != null)
			return newProject;

		final IProject newProjectHandle = mainPage.getProjectHandle();

		IPath newPath = null;
		if(!mainPage.useDefaults())
			newPath = mainPage.getLocationPath();

		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		final IProjectDescription description = workspace.newProjectDescription(newProjectHandle.getName());
		description.setLocation(newPath);
		
		String[] natureIDs = description.getNatureIds();
		String[] newNatureIDs = new String[natureIDs.length + 1];
		System.arraycopy(natureIDs, 0, newNatureIDs, 0, natureIDs.length);
		newNatureIDs[newNatureIDs.length - 1] = MonalipsePlugin.NATURE_ID;
		description.setNatureIds(newNatureIDs);

		WorkspaceModifyOperation op = new WorkspaceModifyOperation()
			{
				protected void execute(IProgressMonitor monitor) throws CoreException
				{
					createProject(description, newProjectHandle, monitor);
				}
			};

		try
		{
			getContainer().run(true, true, op);
		}
		catch (InterruptedException e)
		{
			return null;
		}
		catch (InvocationTargetException e)
		{
			Throwable t = e.getTargetException();
			if (t instanceof CoreException)
			{
				if (((CoreException)t).getStatus().getCode() == IResourceStatus.CASE_VARIANT_EXISTS)
				{
					MessageDialog.openError(getShell(), MonalipsePlugin.getResourceString("NewProject.errorMessage"), //$NON-NLS-1$
					MonalipsePlugin.formatResourceString("NewProject.caseVariantExistsError", new String[] { newProjectHandle.getName()})); //$NON-NLS-1$,
				}
				else
				{
					ErrorDialog.openError(getShell(), MonalipsePlugin.getResourceString("NewProject.errorMessage"), null, ((CoreException)t).getStatus()); //$NON-NLS-1$
				}
			}
			else
			{
				Platform.getPlugin(PlatformUI.PLUGIN_ID).getLog().log(new Status(Status.ERROR, PlatformUI.PLUGIN_ID, 0, t.toString(), t));
				MessageDialog.openError(getShell(), MonalipsePlugin.getResourceString("NewProject.errorMessage"), //$NON-NLS-1$
				MonalipsePlugin.formatResourceString("NewProject.internalError", new Object[] { t.getMessage()})); //$NON-NLS-1$
			}
			return null;
		}

		newProject = newProjectHandle;

		return newProject;
	}

	private void createProject(IProjectDescription description, IProject projectHandle, IProgressMonitor monitor) throws CoreException, OperationCanceledException
	{
		try
		{
			monitor.beginTask("", 2000); //$NON-NLS-1$

			projectHandle.create(description, new SubProgressMonitor(monitor, 1000));

			if (monitor.isCanceled())
				throw new OperationCanceledException();

			projectHandle.open(new SubProgressMonitor(monitor, 1000));
		}
		finally
		{
			monitor.done();
		}
	}

	public boolean performFinish()
	{
		createNewProject();
	
		if (newProject == null)
			return false;

		BasicNewProjectResourceWizard.updatePerspective(configElement);
		BasicNewResourceWizard.selectAndReveal(newProject, getWorkbench().getActiveWorkbenchWindow());
	
		return true;
	}

	public void setInitializationData(IConfigurationElement config, String propertyName, Object data) throws CoreException
	{
		configElement = config;
	}

	public void init(IWorkbench workbench, IStructuredSelection selection)
	{
		this.workbench = workbench;
		this.selection = selection;

		initializeDefaultPageImageDescriptor();

		setNeedsProgressMonitor(true);
		setWindowTitle(MonalipsePlugin.getResourceString("NewProject.windowTitle")); //$NON-NLS-1$
	}

	protected void initializeDefaultPageImageDescriptor()
	{
		String iconPath = "icons/full/"; //$NON-NLS-1$		
		try
		{
			URL installURL = Platform.getPlugin(PlatformUI.PLUGIN_ID).getDescriptor().getInstallURL();
			URL url = new URL(installURL, iconPath + "wizban/newprj_wiz.gif"); //$NON-NLS-1$
			ImageDescriptor desc = ImageDescriptor.createFromURL(url);
			setDefaultPageImageDescriptor(desc);
		}
		catch (MalformedURLException e)
		{
		}
	}

	public IStructuredSelection getSelection()
	{
		return selection;
	}

	public IWorkbench getWorkbench()
	{
		return workbench;
	}

	public IProject getNewProject()
	{
		return newProject;
	}

}
