package monalipse.editors;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ResourceBundle;
import java.util.logging.Logger;

import monalipse.MonalipseMessages;
import monalipse.MonalipsePlugin;
import monalipse.actions.BookmarkAction;
import monalipse.bookmark.BookmarkManager;
import monalipse.server.IThreadContentProvider;
import monalipse.utils.CancelableRunner;

import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.action.SubStatusLineManager;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.swt.program.Program;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.internal.WorkbenchImages;
import org.eclipse.ui.internal.WorkbenchMessages;
import org.eclipse.ui.part.EditorActionBarContributor;
import org.eclipse.ui.texteditor.FindReplaceAction;
import org.eclipse.ui.texteditor.ITextEditorActionConstants;

public class ThreadViewerEditorActionBarContributor extends EditorActionBarContributor
{
	private static final Logger logger = MonalipsePlugin.getLogger();

	private CancelableRunner cancelable;
	private SubStatusLineManager statusLineManager;
	private IAction reloadAction;
	private IAction abortAction;
	private IAction browserAction;
	private IAction refer1Action;
	private IAction referLatestAction;
	private IAction lockToolTipAction;
	private IAction fullReloadAction;
	private IAction disposeCacheAction;
	private IAction copyAction;
	private IAction selectAllAction;
	private IAction findReplaceAction;
	private BookmarkAction bookmarkAction;
	private IContributionItem urlContribution;
	private ResourceBundle textEditorResourceBundle = ResourceBundle.getBundle("org.eclipse.ui.texteditor.EditorMessages");
	private int lastReferedResponse;
	
	public ThreadViewerEditorActionBarContributor()
	{
		makeActions();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(new IResourceChangeListener()
			{
				public void resourceChanged(IResourceChangeEvent event)
				{
//logger.finest("enter");
					logger.finest("begin");
					if(BookmarkManager.bookmarkChanged(event) || MonalipsePlugin.projectModified(event.getDelta()))
					{
						getPage().getWorkbenchWindow().getShell().getDisplay().asyncExec(new Runnable()
							{
								public void run()
								{
									updateBookmarkDependentActions();
								}
							});
					}
					logger.finest("done");
				}
			});
	}
	
	public void contributeToContextMenu(IMenuManager manager, ThreadViewerEditor editor)
	{
		manager.add(reloadAction);
//		manager.add(fullReloadAction);
		manager.add(disposeCacheAction);
		manager.add(abortAction);

		manager.add(new Separator());
		manager.add(lockToolTipAction);
		manager.add(refer1Action);
		manager.add(referLatestAction);
		
		lastReferedResponse = editor.getResponseCount();
		referLatestAction.setText(">>" + lastReferedResponse);
		referLatestAction.setToolTipText(">>" + lastReferedResponse);

		manager.add(new Separator());
		manager.add(browserAction);
		manager.add(bookmarkAction);

		manager.add(new Separator(ITextEditorActionConstants.GROUP_EDIT));
		manager.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, copyAction);
		manager.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, selectAllAction);
	}
	
	public void contributeToStatusLine(IStatusLineManager manager)
	{
		cancelable = new CancelableRunner(manager, null, abortAction);
	}

	public void contributeToToolBar(IToolBarManager manager)
	{
		manager.add(reloadAction);
		manager.add(abortAction);
		manager.add(browserAction);
	}
	
	public void setActiveEditor(IEditorPart targetEditor)
	{
//logger.finest("enter");
		logger.finest("begin");
		copyAction = createAction(IWorkbenchActionConstants.COPY, ITextOperationTarget.COPY, "IMG_CTOOL_COPY_EDIT", targetEditor);
		selectAllAction = createAction(IWorkbenchActionConstants.SELECT_ALL, ITextOperationTarget.SELECT_ALL, null, targetEditor);
		getActionBars().setGlobalActionHandler(IWorkbenchActionConstants.REFRESH, reloadAction);
		findReplaceAction = new FindReplaceAction(textEditorResourceBundle, "Editor.FindReplace.", targetEditor);
		getActionBars().setGlobalActionHandler(IWorkbenchActionConstants.FIND, findReplaceAction);
		updateBookmarkDependentActions();
		logger.finest("done");
	}
	
	private void makeActions()
	{
		String iconPath = "icons/"; //$NON-NLS-1$		
		URL installURL = Platform.getPlugin(MonalipsePlugin.PLUGIN_ID).getDescriptor().getInstallURL();

		reloadAction = new Action()
			{
				public void run()
				{
					IEditorPart part = getPage().getActiveEditor();
					if(part instanceof ThreadViewerEditor)
						((ThreadViewerEditor)part).updateThread(-1);
				}
			};
		reloadAction.setText("Reload");
		reloadAction.setToolTipText("Reload thread");
		try
		{
			reloadAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "refresh_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		abortAction = new Action()
			{
				public void run()
				{
					cancelable.cancel();
				}
			};
		abortAction.setText("Abort");
		abortAction.setToolTipText("Abort all downloading thread");
		abortAction.setEnabled(false);
		try
		{
			abortAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "stop_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		browserAction = new Action()
			{
				public void run()
				{
					IEditorInput input = getPage().getActiveEditor().getEditorInput();
					if(input instanceof ThreadEditorInput)
					{
						String href = ((ThreadEditorInput)input).getURL();
						IPreferenceStore store = MonalipsePlugin.getDefault().getPreferenceStore();
						if(store.getBoolean(MonalipsePlugin.PREF_SET_WEBBROWSER))
						{
							try
							{
								Runtime.getRuntime().exec(new String[]{store.getString(MonalipsePlugin.PREF_WEBBROWSER_PATH), href});
							}
							catch (IOException e)
							{
								e.printStackTrace();
							}
						}
						else
						{
							Program.launch(href);
						}
					}
				}
			};
		browserAction.setText("Open With Web Browser");
		browserAction.setToolTipText("Open With Web Browser");
		try
		{
			browserAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "webbrowser_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		lockToolTipAction = new Action()
			{
				public void run()
				{
					IEditorPart part = getPage().getActiveEditor();
					if(part instanceof ThreadViewerEditor)
						((ThreadViewerEditor)part).lockToolTip();
				}
			};
		lockToolTipAction.setText("Lock ToolTip");
		lockToolTipAction.setToolTipText("Lock ToolTip");
		lockToolTipAction.setEnabled(false);
		
		refer1Action = new Action()
			{
				public void run()
				{
					IEditorPart part = getPage().getActiveEditor();
					if(part instanceof ThreadViewerEditor)
						((ThreadViewerEditor)part).scrollTo(1);
				}
			};
		refer1Action.setText(">>1");
		refer1Action.setToolTipText(">>1");
		refer1Action.setEnabled(true);
		
		referLatestAction = new Action()
			{
				public void run()
				{
					IEditorPart part = getPage().getActiveEditor();
					if(part instanceof ThreadViewerEditor)
						((ThreadViewerEditor)part).scrollTo(lastReferedResponse);
				}
			};
		referLatestAction.setText(">>0");
		referLatestAction.setToolTipText(">>0");
		referLatestAction.setEnabled(true);

		fullReloadAction = new Action()
			{
				public void run()
				{
					IEditorPart part = getPage().getActiveEditor();
					if(part instanceof ThreadViewerEditor)
					{
						((ThreadViewerEditor)part).disposeCache();
						((ThreadViewerEditor)part).updateThread(-1);
					}
				}
			};
		fullReloadAction.setText("Full Reload");
		fullReloadAction.setToolTipText("Full Reload thread");
		
		disposeCacheAction = new Action()
			{
				public void run()
				{
					IEditorPart part = getPage().getActiveEditor();
					if(part instanceof ThreadViewerEditor)
						((ThreadViewerEditor)part).disposeCache();
				}
			};
		disposeCacheAction.setText("Dispose Cache");
		disposeCacheAction.setToolTipText("Dispose Cache");
		
		bookmarkAction = new BookmarkAction();
	}
	
	private IAction createAction(final String id, final int action, String image, final IEditorPart part)
	{
		Action act = new Action(MonalipseMessages.getString("Workbench." + id))
			{
				public void run()
				{
					if(part instanceof ThreadViewerEditor)
						((ThreadViewerEditor)part).getTextOperationTarget().doOperation(action);
				}
			};
		if(image != null)
		{
			act.setImageDescriptor(WorkbenchImages.getImageDescriptor(image));
			act.setDisabledImageDescriptor(WorkbenchImages.getImageDescriptor(image + "_DISABLED"));
			act.setHoverImageDescriptor(WorkbenchImages.getImageDescriptor(image + "_HOVER"));
		}
		
		
		if(part instanceof ThreadViewerEditor)
			act.setEnabled(((ThreadViewerEditor)part).getTextOperationTarget().canDoOperation(action));

		getActionBars().setGlobalActionHandler(id, act);
		return act;
	}
	
	public void updateBookmarkDependentActions()
	{
		IEditorPart part = getPage().getActiveEditor();
		if(part instanceof ThreadViewerEditor)
		{
			IThreadContentProvider thread = ((ThreadViewerEditor)part).getContentProvider();
			if(thread != null)
			{
				bookmarkAction.updateState(part.getTitle(), thread.getURL(), true);
				return;
			}
		}

		bookmarkAction.updateState(null, null, false);
	}

	public void updateToolTipDependentActions()
	{
		IEditorPart part = getPage().getActiveEditor();
		if(part instanceof ThreadViewerEditor)
			lockToolTipAction.setEnabled(((ThreadViewerEditor)part).isToolTipActive());
		else
			lockToolTipAction.setEnabled(false);
	}

	public void updateSelectionDependentActions()
	{
		if(copyAction != null)
		{
			IEditorPart part = getPage().getActiveEditor();
			if(part instanceof ThreadViewerEditor)
			{
				copyAction.setEnabled(((ThreadViewerEditor)part).getTextOperationTarget().canDoOperation(ITextOperationTarget.COPY));
				selectAllAction.setEnabled(((ThreadViewerEditor)part).getTextOperationTarget().canDoOperation(ITextOperationTarget.SELECT_ALL));
			}
			else
			{
				copyAction.setEnabled(false);
				selectAllAction.setEnabled(false);
			}
		}
	}
	
	public CancelableRunner getCancelable()
	{
		return cancelable;
	}

}
