package monalipse.views;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.logging.Logger;

import monalipse.MonalipsePlugin;
import monalipse.actions.BookmarkAction;
import monalipse.bookmark.BookmarkManager;
import monalipse.editors.ThreadEditorInput;
import monalipse.editors.ThreadViewerEditor;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSBoard;
import monalipse.server.IBBSServer;
import monalipse.server.IBoardTreeNode;
import monalipse.server.nullserver.NullServer;
import monalipse.utils.CancelableRunner;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.XMLMemento;

public class BoardListView extends AbstractTreeView implements SelectionListener
{
	public static final String ID_BOARD_LIST = BoardListView.class.getName();

	private static final Logger logger = MonalipsePlugin.getLogger(BoardListView.class);

	private TreeViewer viewer;
	private IBBSServer server;
	private BoardListContentProvider provider;
	private CancelableRunner cancelable;
//	private DrillDownAdapter drillDownAdapter;
	private IAction reloadAction;
	private IAction abortAction;
	private IAction doubleClickAction;
	private BookmarkAction bookmarkAction;

	public BoardListView()
	{
		logger.finest("<init>");
	}
	
	protected ITreeContentProvider getContentProvider()
	{
		return provider;
	}

	protected ILabelProvider getLabelProvider()
	{
		return provider;
	}

	protected Logger getLogger()
	{
		return logger;
	}

	protected TreeViewer getViewer()
	{
		return viewer;
	}

	public void createPartControl(Composite parent)
	{
		logger.finest("begin");
		String target = getConfigurationElement().getAttribute("target");
		logger.finest("target: " + target);
		server = BBSServerManager.getInstanceOf(target);
		logger.finest("server: " + target);
		if(server == null)
			server = new NullServer(target);
		viewer = new TreeViewer(parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL);
//		drillDownAdapter = new DrillDownAdapter(viewer);
		provider = new BoardListContentProvider();
		viewer.setContentProvider(provider);
		viewer.setLabelProvider(provider);
		viewer.setInput(getServer());
//		viewer.setSorter(new NameSorter());
		makeActions();
		hookContextMenu();
		hookDoubleClickAction();
		contributeToActionBars();
		getSite().setSelectionProvider(viewer);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		viewer.getTree().addSelectionListener(this);
		partCreated();

		viewer.getTree().addKeyListener(new KeyAdapter()
			{
				public void keyPressed(KeyEvent e)
				{
					if(e.keyCode == SWT.F5)
						reloadAction.run();
				}
			});

		if(provider.getElements(viewer.getInput()).length == 0)
		{
			logger.finest("empty board tree");
			getSite().getShell().getDisplay().asyncExec(new Runnable()
				{
					public void run()
					{
						logger.finest("activate");
						getSite().getPage().activate(BoardListView.this);
						reloadAction.run();
					}
				});
		}
		logger.finest("done");
	}
	
	public IBBSServer getServer()
	{
		return server;
	}
	
	public void resourceChanged(IResourceChangeEvent event)
	{
		if(getServer().boardListChanged(event) || MonalipsePlugin.projectModified(event.getDelta()))
		{
			logger.finest("begin");
			IMemento memento = XMLMemento.createWriteRoot(BoardListView.class.getName());
			saveState(memento);
			logger.finest("reset input");
			viewer.setInput(getServer());
			restoreState(memento);
			logger.finest("done");
		}

		if(BookmarkManager.bookmarkChanged(event))
			viewer.refresh(true);
	}
	
	public void widgetSelected(SelectionEvent e)
	{
		bookmarkAction.updateState(getSite().getWorkbenchWindow(), viewer.getSelection());
	}
	
	public void widgetDefaultSelected(SelectionEvent e)
	{
	}
	
	private void hookContextMenu()
	{
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager manager)
				{
					BoardListView.this.fillContextMenu(manager);
				}
			});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	private void contributeToActionBars()
	{
		IActionBars bars = getViewSite().getActionBars();
//		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
		fillLocalStatusBar(bars.getStatusLineManager());
	}

	private void fillLocalStatusBar(IStatusLineManager manager)
	{
		cancelable = new CancelableRunner(manager, getSite().getShell().getDisplay(), abortAction);
	}

	private void fillLocalPullDown(IMenuManager manager)
	{
		manager.add(reloadAction);
		manager.add(new Separator());
		manager.add(abortAction);
	}

	private void fillContextMenu(IMenuManager manager)
	{
		bookmarkAction.updateState(getSite().getWorkbenchWindow(), viewer.getSelection());
		manager.add(reloadAction);
		manager.add(abortAction);
		manager.add(new Separator());
		manager.add(bookmarkAction);
//		manager.add(new Separator());
//		drillDownAdapter.addNavigationActions(manager);
		// Other plug-ins can contribute there actions here
		manager.add(new Separator("Additions"));
	}

	private void fillLocalToolBar(IToolBarManager manager)
	{
		manager.add(reloadAction);
		manager.add(abortAction);
//		manager.add(new Separator());
//		drillDownAdapter.addNavigationActions(manager);
	}

	private void makeActions()
	{
		String iconPath = "icons/"; //$NON-NLS-1$		
		URL installURL = Platform.getPlugin(MonalipsePlugin.PLUGIN_ID).getDescriptor().getInstallURL();

		reloadAction = new Action()
			{
				public void run()
				{
					cancelable.run(cancelable, new CancelableRunner.ICancelableRunnableWithProgress()
						{
							public void run(CancelableRunner.ICancelableProgressMonitor monitor)
							{
								logger.finest("refresh");
								getServer().updateBoardList(getSite().getWorkbenchWindow());
							}
						});
				}
			};
		reloadAction.setText("Reload");
		reloadAction.setToolTipText("Reload board list");
		try
		{
			reloadAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "refresh_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		abortAction = new Action()
			{
				public void run()
				{
					cancelable.cancel();
				}
			};
		abortAction.setText("Abort");
		abortAction.setToolTipText("Abort downloading board list");
		abortAction.setEnabled(false);
		try
		{
			abortAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "stop_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		doubleClickAction = new Action()
			{
				public void run()
				{
					ISelection selection = viewer.getSelection();
					if(selection instanceof IStructuredSelection)
					{
						final Object first = ((IStructuredSelection)selection).getFirstElement();
						if(first instanceof IBBSBoard)
						{
							cancelable.run(cancelable, new CancelableRunner.ICancelableRunnableWithProgress()
								{
									public void run(CancelableRunner.ICancelableProgressMonitor monitor)
									{
										logger.finest("begin");
										((IBBSBoard)first).updateThreadList(getSite().getWorkbenchWindow());
										logger.finest("done");
									}
								});

							IWorkbenchPage page = getSite().getPage().getWorkbenchWindow().getActivePage();
							try
							{
								logger.finest("openView");
								page.showView(ThreadListView.ID_THREAD_LIST);
							}
							catch(PartInitException ex)
							{
								ex.printStackTrace();
							}
						}
						else
						{
							viewer.setExpandedState(first, !viewer.getExpandedState(first));
						}
					}
				}
			};

		bookmarkAction = new BookmarkAction();
	}

	private void hookDoubleClickAction()
	{
		viewer.addDoubleClickListener(new IDoubleClickListener()
			{
				public void doubleClick(DoubleClickEvent event)
				{
					doubleClickAction.run();
				}
			});
	}

	public void setFocus()
	{
		viewer.getControl().setFocus();
	}

	private class NameSorter extends ViewerSorter
	{
	}

	private class BoardListContentProvider extends LabelProvider implements IStructuredContentProvider, ITreeContentProvider
	{
		public Object[] getElements(Object inputElement)
		{
			if(inputElement instanceof IBBSServer)
				return ((IBBSServer)inputElement).getBoardTree().getChildren();
			else
				return new Object[0];
		}

		public Object[] getChildren(Object parentElement)
		{
			if(parentElement instanceof IBoardTreeNode)
				return ((IBoardTreeNode)parentElement).getChildren();
			else
				return new Object[0];
		}

		public Object getParent(Object element)
		{
			if(element instanceof IBoardTreeNode)
				return ((IBoardTreeNode)element).getParent();
			else
				return null;
		}

		public boolean hasChildren(Object element)
		{
			if(element instanceof IBoardTreeNode)
				return ((IBoardTreeNode)element).hasChildren();
			else if(element instanceof IBBSServer)
				return ((IBBSServer)element).getBoardTree().hasChildren();
			else
				return false;
		}

		public Image getImage(Object element)
		{
			if(hasChildren(element))
				return PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJ_FOLDER);
			if(element instanceof IBBSReference)
				return MonalipsePlugin.getDefault().getLabelImageOf((IBBSReference)element);
			return null;
		}

		public String getText(Object element)
		{
			if(element instanceof IBoardTreeNode)
				return ((IBoardTreeNode)element).getName();
			else
				return "";
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
		}

	}

}