/*
 * AbstractExecuter.java
 *
 * Copyright 2014 the Monache.org.
 * http://www.monache.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.monache.nagesen.ececuter;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.monache.lib.api.CoinAPIException;
import org.monache.lib.api.SendFrom;
import org.monache.lib.api.ValidateAddress;
import org.monache.lib.api.WalletLock;
import org.monache.lib.api.WalletPassphrase;
import org.monache.lib.httpclient.BasicAuthenticationHttpClient;
import org.monache.lib.jsonrpc.JsonRPCClient;
import org.monache.lib.jsonrpc.JsonRPCException;
import org.monache.nagesen.AddressData;
import org.monache.nagesen.Constant;
import org.monache.nagesen.NagesenException;
import org.monache.nagesen.amount.Amount;
import org.monache.nagesen.jsonrpc.KumaJsonRPCClient;
import org.monache.nagesen.jsonrpc.MonaJsonRPCClient;
import org.monache.nagesen.log.SendLog;
import org.monache.nagesen.validator.address.AddressValidator;
import org.monache.nagesen.validator.address.AddressValidatorFactory;
import org.monache.nagesen.validator.message.MessageValidator;
import org.monache.nagesen.validator.unique.UniqueValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * 
 * @author monache.org
 * @since 1.0
 */
public abstract class AbstractExecuter implements Executer {

	private static org.apache.log4j.Logger log = org.apache.log4j.Logger.getLogger(AbstractExecuter.class);	
//	private static Logger log = LoggerFactory.getLogger(AbstractExecuter.class);
	
	// 検証処理
	private UniqueValidator uniqueValidator = null;
	private MessageValidator messageValidator = null;

	private Amount amount = null;

	private WalletPassphrase walletPassphrase = null;
	private WalletLock walletLock = null;
	private SendFrom sendFrom = null;
	private ValidateAddress validateAddress = null;
	
	private String passphrase = null;
	private String fromAccount = null;
	
	private MonaJsonRPCClient monaJsonRPCClient = null;
	private KumaJsonRPCClient kumaJsonRPCClient = null;
	
	/**
	 * 
	 * @since 1.0
	 */
	public AbstractExecuter() {
	}

	public int execute(int maxCount, long timeIntervalMillis) throws NagesenException {

		try {

			// 送金済みのリスト
			List<AddressData> dataList2 = new ArrayList<AddressData>();
			
			boolean owari = false;

			for (;;) {
				// データのリスト
				List<AddressData> dataList = getData();
				if (dataList == null) {
					break;
				}
				for (AddressData data : dataList) {
					
					AddressData data2 = new AddressData();
					BeanUtils.copyProperties(data2, data);

					// 投げ銭対象であるかの検証をする
					// ウォレットアドレスの検証
					if ( validateAddress(getMonaJsonRPCClient(), data2.getAddress()) == false) {
						continue;
					}
					// メッセージの検証
					if ( getMessageValidator().validate(data2.getMessage()) == false) {
						continue;
					}
					// ユニークの検証
					if ( getUniqueValidator().validate(dataList2, data2) == false) {
						continue;
					}
					log.debug("Send candidate: " + data2.getCoin() + " " + data2.getAddress());

					// 金額計算
					data2.setAmount(getAmount().calc(data2));

					JsonRPCClient jsonRPCClient = null;
					if (Constant.MONACOIN.equals(data2.getCoin())) {
						jsonRPCClient = getMonaJsonRPCClient();
						if (getMonaJsonRPCClient().getHttpClient() instanceof BasicAuthenticationHttpClient) {
							log.debug("MonaJsonRPCClient " + data2.getCoin() + " " + ((BasicAuthenticationHttpClient)getMonaJsonRPCClient().getHttpClient()).getPort());
						} else {
							log.debug("MonaJsonRPCClient " + data2.getCoin());
						}
						
					} else if (Constant.KUMACOIN.equals(data2.getCoin())) {
						jsonRPCClient = getKumaJsonRPCClient();
						if (getKumaJsonRPCClient().getHttpClient() instanceof BasicAuthenticationHttpClient) {
							log.debug("KumaJsonRPCClient " + data2.getCoin() + " " + ((BasicAuthenticationHttpClient)getKumaJsonRPCClient().getHttpClient()).getPort());
						} else {
							log.debug("KumaJsonRPCClient " + data2.getCoin());
						}
					} else {
						// エラー
						continue;
					}
					
					// 送金処理
					try {
						sendFrom(jsonRPCClient, data2);
					} catch (CoinAPIException e) {
						log.error("Send error " + data2.getAddress(), e);
					}
					SendLog.log(data2);
//					log.info(data2.getCoin() + " " + data2.getAddress() + " " + data2.getAmount());

					// 送金済みデータ登録
					dataList2.add(data2);
					if (dataList2.size() >= maxCount) {
						owari = true;
						break;
					}
					
					if (isContinue(data2) == false) {
						owari = true;
						break;
					}
				}

				if (owari) {
					break;
				}
				
				try {
					Thread.sleep(timeIntervalMillis);
				} catch (InterruptedException e) {
				}

			}
			
		} catch (Exception e) {
			if (e instanceof NagesenException) {
				throw (NagesenException) e;
			}
			throw new NagesenException(e);
		} finally {
			if (getMonaJsonRPCClient() != null) {
				try {
					getMonaJsonRPCClient().close();
				} catch (JsonRPCException e) {
				}
			}
			if (getKumaJsonRPCClient() != null) {
				try {
					getKumaJsonRPCClient().close();
				} catch (JsonRPCException e) {
				}
			}
		}

		return 0;
	}

	protected boolean validateAddress(JsonRPCClient jsonRPCClient, String address) {
		try {
			return getValidateAddress().call(jsonRPCClient, address).getIsvalid();
		} catch (Exception e) {
			log.error("ValidateAddress error " + address, e);
			return false;
		}
	}
	
	protected void sendFrom(JsonRPCClient jsonRPCClient, AddressData data) throws CoinAPIException {

		try {
			if (StringUtils.isEmpty(getPassphrase()) == false) {
				getWalletPassphrase().call(jsonRPCClient, getPassphrase(), 20);
			}
			getSendFrom().call(jsonRPCClient, getFromAccount(), data.getAddress(), data.getAmount());
			data.setTimestamp(Calendar.getInstance());

		} finally {
			if (StringUtils.isEmpty(getPassphrase()) == false) {
				try {
					getWalletLock().call(jsonRPCClient);
				} catch (CoinAPIException e) {
					log.error("WalletLock error " + data.getAddress(), e);
				}
			}
		}
	}
	
	protected abstract List<AddressData> getData() throws NagesenException ;

	protected abstract boolean isContinue(AddressData data) throws NagesenException ;

	
	/**
	 * @return uniqueValidator
	 * @since 1.0
	 */
	public UniqueValidator getUniqueValidator() {
		return uniqueValidator;
	}

	/**
	 * @param uniqueValidator セットする uniqueValidator
	 * @since 1.0
	 */
	public void setUniqueValidator(UniqueValidator uniqueValidator) {
		this.uniqueValidator = uniqueValidator;
	}

	/**
	 * @return messageValidator
	 * @since 1.0
	 */
	public MessageValidator getMessageValidator() {
		return messageValidator;
	}

	/**
	 * @param messageValidator セットする messageValidator
	 * @since 1.0
	 */
	public void setMessageValidator(MessageValidator messageValidator) {
		this.messageValidator = messageValidator;
	}

	/**
	 * @return amount
	 * @since 1.0
	 */
	public Amount getAmount() {
		return amount;
	}

	/**
	 * @param amount セットする amount
	 * @since 1.0
	 */
	public void setAmount(Amount amount) {
		this.amount = amount;
	}

	/**
	 * @return walletPassphrase
	 * @since 1.0
	 */
	public WalletPassphrase getWalletPassphrase() {
		return walletPassphrase;
	}

	/**
	 * @param walletPassphrase セットする walletPassphrase
	 * @since 1.0
	 */
	public void setWalletPassphrase(WalletPassphrase walletPassphrase) {
		this.walletPassphrase = walletPassphrase;
	}

	/**
	 * @return walletLock
	 * @since 1.0
	 */
	public WalletLock getWalletLock() {
		return walletLock;
	}

	/**
	 * @param walletLock セットする walletLock
	 * @since 1.0
	 */
	public void setWalletLock(WalletLock walletLock) {
		this.walletLock = walletLock;
	}

	/**
	 * @return sendFrom
	 * @since 1.0
	 */
	public SendFrom getSendFrom() {
		return sendFrom;
	}

	/**
	 * @param sendFrom セットする sendFrom
	 * @since 1.0
	 */
	public void setSendFrom(SendFrom sendFrom) {
		this.sendFrom = sendFrom;
	}


	/**
	 * @return
	 * @since 1.0
	 */
	public String getPassphrase() {
		return passphrase;
	}

	/**
	 * @param passphrase セットする passphrase
	 * @since 1.0
	 */
	public void setPassphrase(String passphrase) {
		this.passphrase = passphrase;
	}

	/**
	 * @return fromAccount
	 * @since 1.0
	 */
	public String getFromAccount() {
		return fromAccount;
	}

	/**
	 * @param fromAccount セットする fromAccount
	 * @since 1.0
	 */
	public void setFromAccount(String fromAccount) {
		this.fromAccount = fromAccount;
	}

	/**
	 * @return monaJsonRPCClient
	 * @since 1.0
	 */
	public MonaJsonRPCClient getMonaJsonRPCClient() {
		return monaJsonRPCClient;
	}

	/**
	 * @param monaJsonRPCClient セットする monaJsonRPCClient
	 * @since 1.0
	 */
	public void setMonaJsonRPCClient(MonaJsonRPCClient monaJsonRPCClient) {
		this.monaJsonRPCClient = monaJsonRPCClient;
	}

	/**
	 * @return kumaJsonRPCClient
	 * @since 1.0
	 */
	public KumaJsonRPCClient getKumaJsonRPCClient() {
		return kumaJsonRPCClient;
	}

	/**
	 * @param kumaJsonRPCClient セットする kumaJsonRPCClient
	 * @since 1.0
	 */
	public void setKumaJsonRPCClient(KumaJsonRPCClient kumaJsonRPCClient) {
		this.kumaJsonRPCClient = kumaJsonRPCClient;
	}

	/**
	 * @return validateAddress
	 * @since 
	 */
	public ValidateAddress getValidateAddress() {
		return validateAddress;
	}

	/**
	 * @param validateAddress セットする validateAddress
	 * @since 
	 */
	public void setValidateAddress(ValidateAddress validateAddress) {
		this.validateAddress = validateAddress;
	}

	
	
}
