/* vi: set ts=8 sw=8 : */
/**
        libmkimgfs - support library for creating filesystem images

	Copyright (C) 2002-2003 Masuichi Noritoshi <nor@users.sourceforge.jp>

        This program is free software; you can redistribute it and/or
        modify it under the terms of the GNU General Public License
        as published by the Free Software Foundation; either version
        2 of the License, or (at your option) any later version.
**/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#ifdef HAVE_STRING_H
#	include <string.h>
#else
#	ifdef HAVE_STRINGS_H
#		include <strings.h>
#	endif
#endif
#include <stdlib.h>
#include <stdarg.h>

#include <util.h>
#include <mkimgfs.h>

const char * g_ThisExecName;

/** **************************************************************************
************************************************************************** **/
void*
ut_Malloc(size_t s)
{
	return malloc(s);
}

/** **************************************************************************
************************************************************************** **/
void
ut_Free(void* p)
{
	free(p);
}

/** **************************************************************************
	ut_TokenZ :
		buf : address of pointer to linebuffer
			ATTENTION : buffer is modified by function.
	
	return
		NULL : No more token.
		o.w. : address of Token
************************************************************************** **/
char*
ut_TokenZ(char** buf)
{
	char* p;
	char* q;
	char* t = NULL;

	if (!buf || !(p = *buf)) { /* check param. */
		return t;
	}
	
	while (*p == ' ' || *p == '\t') {	/* Skip pre space(s) */
		p++;
	}
	if (*p == '\0') {			/* Reached EOL */
		*buf = p;
		return t;
	}
	t = q = p;
	while (*p != ' ' && *p != '\t' && *p != '\0') {
		if (*p == '\\' && *p != '\0') {
			p++;
		}
		*q++ = *p++;
	}
	if (*p != '\0') {
		p++;
	}
	*q = '\0';

	*buf = p;
	return t;
}

/** **************************************************************************
	ut_RoundUp() : round up to unit
		u : MUST BE 2^x (x = 1, 2, ...)
************************************************************************** **/
inline uint32
ut_RoundUp(uint32 n, uint32 u)
{
	return (n + u -1) & ~(u - 1);
}

/** **************************************************************************
	the length of string p must be 10.
	ex.)
		"crw-rw-rw-"

	This returns that:
	permisson if good string.
	VFS_ERR_CONV if bad.
************************************************************************** **/
uint32
ut_StringToMode(const char *p)
{
	uint32 mode = 0;

	if (!p || strlen(p) != 10) {
		return VFS_ERR_CONV;
	}

	switch (*p++) {				/* I-NODE TYPE */
	case 's':	mode = VFS_IFSOCK;	break;
	case 'l':	mode = VFS_IFLNK;	break;
	case '-':	mode = VFS_IFREG;	break;
	case 'b':	mode = VFS_IFBLK;	break;
	case 'd':	mode = VFS_IFDIR;	break;
	case 'c':	mode = VFS_IFCHR;	break;
	case 'p':	mode = VFS_IFIFO;	break;
	default:	return VFS_ERR_CONV;
	}
	switch (*p++) {				/* PERMISSION : USER */
	case 'r':	mode |= VFS_IRUSR;	break;
	case '-':				break;
	default:	return VFS_ERR_CONV;
	}
	switch (*p++) {
	case 'w':	mode |= VFS_IWUSR;	break;
	case '-':				break;
	default:	return VFS_ERR_CONV;
	}
	switch (*p++) {
	case 'x':	mode |= VFS_IXUSR;		break;
	case 's':	mode |= (VFS_IXUSR | VFS_ISUID);	break;
	case '-':					break;
	default:	return VFS_ERR_CONV;
	}
	switch (*p++) {				/* PERMISSION : GROUP */
	case 'r':	mode |= VFS_IRGRP;	break;
	case '-':				break;
	default:	return VFS_ERR_CONV;
	}
	switch (*p++) {
	case 'w':	mode |= VFS_IWGRP;	break;
	case '-':				break;
	default:	return VFS_ERR_CONV;
	}
	switch (*p++) {
	case 'x':	mode |= VFS_IXGRP;		break;
	case 's':	mode |= (VFS_IXGRP | VFS_ISGID);	break;
	case '-':					break;
	default:	return VFS_ERR_CONV;
	}
	switch (*p++) {				/* PERMISSION : OTHER */
	case 'r':	mode |= VFS_IROTH;	break;
	case '-':				break;
	default:	return VFS_ERR_CONV;
	}
	switch (*p++) {
	case 'w':	mode |= VFS_IWOTH;	break;
	case '-':				break;
	default:	return VFS_ERR_CONV;
	}
	switch (*p++) {
	case 'x':	mode |= VFS_IXOTH;		break;
	case 't':	mode |= (VFS_IXOTH | VFS_ISVTX);	break;
	case '-':					break;
	default:	return VFS_ERR_CONV;
	}

	return mode;
}

/** **************************************************************************
	basename of a path
************************************************************************** **/
char*
ut_GenBasename(const char* s)
{
	char* p;

	p = strrchr(s, '/');
	return strdup(p ? p + 1 : s);
}

/** **************************************************************************
	dirname of a path
************************************************************************** **/
char*
ut_GenDirname(const char* s)
{
	char* p;
	char* ix;

	p = strdup(s);
	if ((ix = strrchr(p, '/'))) {
		*(ix+1) = '\0';
	} else {
		*p = '\0';
	}
	return p;
}

/** **************************************************************************
	Error : Panic !!
************************************************************************** **/
void
errexit(const char *fmt, ...)
{
	va_list ap;
	fprintf(stderr, "%s: ", g_ThisExecName);
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fprintf(stderr, "\n");
	exit(1);
}

/** **************************************************************************
	Error : Panic !!
************************************************************************** **/
void
pexit(const char * fname)
{
	fprintf(stderr, "%s: ", g_ThisExecName);
	perror(fname);
	exit(1);
}

/* END-OF-FILE */

