/* vi: sw=8 ts=8 */
/**
        libmkimgfs - support library for creating filesystem images

	Copyright (C) 2002-2003 Masuichi Noritoshi <nor@users.sourceforge.jp>

        This program is free software; you can redistribute it and/or
        modify it under the terms of the GNU General Public License
        as published by the Free Software Foundation; either version
        2 of the License, or (at your option) any later version.
**/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_STRING_H
#       include <string.h>
#else
#       ifdef HAVE_STRINGS_H
#               include <strings.h>
#       endif
#endif
#include <dirent.h>
#include <stdarg.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#define __USE_GNU
#include <ctype.h>
#include <fcntl.h>

#include <util.h>
#include <srcinode.h>

const char* g_PrefixOfRelatvePath = NULL;

#define XSTRDUP(s)	strdup(s)
#define XGETTOKEN(t,s)	\
	do { if (!((t) = ut_TokenZ(&(s)))) goto L_ERR_CONV; } while(0)


/** **************************************************************************
************************************************************************** **/
void
TSrcInode_Init(TSrcInode* ce)
{
	memset((char*)ce, 0, sizeof(*ce));
}

/** **************************************************************************
************************************************************************** **/
TSrcInode*
TSrcInode_New(void)
{
	TSrcInode* p = XMALLOC(TSrcInode, 1);

	TSrcInode_Init(p);
	return p;
}

/** **************************************************************************
************************************************************************** **/
void
TSrcInode_Fini(TSrcInode* ce)
{
	if (ce) {
		SafeDel_PROP(ce->mSourcePath);
		SafeDel(TSplitPath_Fini, ce->mTargets);
	}
}

/** **************************************************************************
************************************************************************** **/
void
TSrcInode_AddTarget(TSrcInode* ce, const char* s)
{
	TSplitPath* p = TSplitPath_New(s);

	p->mNext = ce->mTargets;
	ce->mTargets = p;
}

/** **************************************************************************
	convert form i-node config string to internal structure.

	1) Directory
		drwxrwxrwx uid gid /path/name [-p]

	2) Block device
		brwxrwxrwx uid gid major minor /path/name [ from-to/step ]

	3) Charactor device
		crwxrwxrwx uid gid major minor /path/name [ from-to/step ]

	4) Symlink
		lrwxrwxrwx uid gid /real/path/name /symlink/path/name

	5) Regular File (with hardlink)
		-rwxrwxrwx uid gid /body/data/file /path/name [/path/name2]..

	6) hardlink to the file which is already in the filesystem image
		H--------- uid gid /body/data/file /path/name [/path/name2]..
		*) permisson-string has no means. but need for passing
		   syntax check.
************************************************************************** **/
TSrcInode*
TSrcInode_New_String(const char* line)
{
	TSrcInode* si;
	char* tkn;
	char* tknh;
	char* s;
	char* sstr;

	sstr = s = XSTRDUP(line);

	si = TSrcInode_New();

	XGETTOKEN(tkn, s);
	if ((si->mMode = ut_StringToMode(tkn)) == VFS_ERR_CONV) {
		goto L_ERR_CONV;
	}
	XGETTOKEN(tkn, s);
	si->mUid = atoi(tkn);
	XGETTOKEN(tkn, s);
	si->mGid = atoi(tkn);

	switch (si->mMode & VFS_IFMT) {
	case VFS_IFDIR:
		XGETTOKEN(tkn, s); TSrcInode_AddTarget(si, tkn);
		break;
	case VFS_IFBLK:
	case VFS_IFCHR:
		XGETTOKEN(tkn, s); si->mMajor = atoi(tkn);
		XGETTOKEN(tkn, s); si->mMinor = atoi(tkn);
		XGETTOKEN(tkn, s); TSrcInode_AddTarget(si, tkn);
		break;
	case VFS_IFLNK:
	case VFS_IFREG:
		XGETTOKEN(tkn, s); si->mSourcePath = XSTRDUP(tkn);
		XGETTOKEN(tkn, s); TSrcInode_AddTarget(si, tkn);
		while ((tkn = ut_TokenZ(&s)) != NULL) {
			TSrcInode_AddTarget(si, tkn);
		}
		break;
	case VFS_IFHLNK:
		XGETTOKEN(tknh, s);
		XGETTOKEN(tkn, s); TSrcInode_AddTarget(si, tkn);
		while ((tkn = ut_TokenZ(&s)) != NULL) {
			TSrcInode_AddTarget(si, tkn);
		}
		TSrcInode_AddTarget(si, tknh);
		break;
	case VFS_IFIFO:
	case VFS_IFSOCK:
		goto L_ERR_CONV;
	default:
		errexit("malformed text input file");
	}

#ifdef DEBUG_LIBMKIMGFS
	printf("### %o %d %d %d %d [%s:%s]\n",
		si->mMode, si->mUid, si->mGid,
		si->mMajor, si->mMinor,
		si->mTargets->mDir, si->mTargets->mBase
	);
#endif

L_ERR_CONV:
	free(sstr);

	return si;
}

/* END-OF-FILE */
