import unittest
from MindMap import *

class XML2MindMapRootTestCase(unittest.TestCase):
  def setUp(self):
    self.xml2mmr = XML2MindMapRoot()

  def testCreateFromXML1(self):
    xml_string = '<map>\n'
    xml_string += ' <node TEXT="parent" color="red"/>\n'
    xml_string += '</map>\n'
    root = self.xml2mmr.createFromXML(xml_string)
    parent = root.getRoot()
    self.assertEqual('parent', parent.message)
    self.assertEqual('red', parent.getColor())

  def testCreateFromXML1n(self):
    xml_string = '<map>\n'
    xml_string += ' <node TEXT="parent\nsecond" color="red"/>\n'
    xml_string += '</map>\n'
    root = self.xml2mmr.createFromXML(xml_string)
    parent = root.getRoot()
    self.assertEqual('parent second', parent.message)
    self.assertEqual('red', parent.getColor())

  def testCreateFromXML2(self):
    xml_string = '<map>\n'
    xml_string += ' <node TEXT="parent" color="red">\n'
    xml_string += '  <node TEXT="child" color="blue"/>\n'
    xml_string += ' </node>\n'
    xml_string += '</map>\n'
    root = self.xml2mmr.createFromXML(xml_string)
    parent = root.getRoot()
    self.assertEqual('parent', parent.message)
    self.assertEqual('red', parent.getColor())

    child = parent.children[0]
    self.assertEqual('child', child.message)
    self.assertEqual('blue', child.getColor())

class MindMapRootTestCase(unittest.TestCase):
  def setUp(self):
    self.mmr = MindMapRoot()
    self.root    = self.mmr.add(None, 'root')
    self.child0  = self.mmr.add(self.root.code,   'child0')
    self.child00 = self.mmr.add(self.child0.code, 'child00')
    self.child1  = self.mmr.add(self.root.code,   'child1')
    self.child10 = self.mmr.add(self.child1.code, 'child10')
    self.child11 = self.mmr.add(self.child1.code, 'child11')

  def testGetRoot(self):
    self.assertEqual('root', self.mmr.getRoot().message)

  def testIdNumber(self):
    self.assertEqual('0', self.root.code)
    self.assertEqual('1', self.child0.code)
    self.assertEqual('2', self.child00.code)
    self.assertEqual('3', self.child1.code)
    self.assertEqual('4', self.child10.code)
    self.assertEqual('5', self.child11.code)

    self.assertEqual('0', self.mmr.decode_map[self.root.code].code)
    self.assertEqual('1', self.mmr.decode_map[self.child0.code].code)
    self.assertEqual('2', self.mmr.decode_map[self.child00.code].code)
    self.assertEqual('3', self.mmr.decode_map[self.child1.code].code)
    self.assertEqual('4', self.mmr.decode_map[self.child10.code].code)
    self.assertEqual('5', self.mmr.decode_map[self.child11.code].code)

  def testColor(self):
    self.assertEqual('black', self.mmr.getRoot().color)
 
  def testChangeRoot(self):
    child = self.mmr.getRoot()
    self.assertEqual('root', child.message)
    self.assertEqual('black', child.color)
    self.assertEqual('', child.link)

    self.mmr.change(self.root.code, 'changed message', 'red', 'link', 'linkType')
    self.assertEqual('changed message', child.message)
    self.assertEqual('red', child.color)
    self.assertEqual('link', child.link)
    self.assertEqual('linkType', child.linkType)

  def testChangeChild10(self):
    child10 = self.mmr.decode_map[self.child10.code]
    self.assertEqual('child10', child10.message)
    self.assertEqual('black', child10.color)

    self.mmr.change(self.child10.code, 'changed message', 'red', 'link', 'linkType')
    self.assertEqual('changed message', child10.message)
    self.assertEqual('red', child10.color)
    self.assertEqual('link', child10.link)
    self.assertEqual('linkType', child10.linkType)

  def testAdd(self):
    child10 = self.mmr.decode_map[self.child10.code]
    child10.color = 'green'

    child100 = self.mmr.add(self.child10.code, 'add message')
    self.assertEqual('6', child100.code)
    self.assertEqual('add message', child100.message)
    self.assertEqual('green', child100.getColor())
    self.assertEqual(child100, self.mmr.decode_map['6'])

class CounterTestCase(unittest.TestCase):
  def setUp(self):
    self.counter = Counter()

  def testGet(self):
    self.assertEqual(0, self.counter.nextCount())
    self.assertEqual(1, self.counter.nextCount())
    self.assertEqual(2, self.counter.nextCount())
    self.assertEqual(3, self.counter.nextCount())
    

class MindMapTestCase(unittest.TestCase):
  def setUp(self):
    self.mindmap = MindMap('parent')
    self.child1 = MindMap('child1')
    self.child11 = MindMap('child11')
    self.child2 = MindMap('child2')
    self.child21 = MindMap('child21')
    self.child22 = MindMap('child22')

    self.mindmap.addChild(self.child1)
    self.mindmap.addChild(self.child2)
    self.child1.addChild(self.child11)
    self.child2.addChild(self.child21)
    self.child2.addChild(self.child22)

    self.mindmap.name = 'parent'
    self.child1.name  = 'child1'
    self.child11.name = 'child11'
    self.child2.name  = 'child2'
    self.child21.name = 'child21'
    self.child22.name = 'child2'
 
  def testMessage(self):
    self.assertEqual('parent', self.mindmap.message)
    self.assertEqual('child1', self.child1.message)
    self.assertEqual('child2', self.child2.message)
    self.assertEqual('child11', self.child11.message)
    self.assertEqual('child21', self.child21.message)
    self.assertEqual('child22', self.child22.message)

  def testColor(self):
    self.assertEqual('black', self.mindmap.color)

  def testSetGetColor(self):
    self.mindmap.setColor('blue')
    self.assertEqual('blue', self.mindmap.color)
    self.assertEqual('blue', self.mindmap.getColor())

    self.mindmap.setColor('red')
    self.assertEqual('red', self.mindmap.color)
    self.assertEqual('red', self.mindmap.getColor())

  def testLink(self):
    self.assertEqual('', self.mindmap.link)
    self.assertEqual('', self.mindmap.linkType)

  def testChange(self):
    import time
    original_time = time.time
    time.time = mockTime

    self.mindmap.change('changed message', 'changed color')
    self.assertEquals('changed message', self.mindmap.message)
    self.assertEquals('changed color', self.mindmap.color)
    self.assertEquals(time.mktime((2004, 1, 11, 14, 38, 44, 6, 11, 0)),
                      self.mindmap.time)

    time.time = original_time

  def testRefresh(self):
    self.mindmap.refresh()
    self.assertEquals([self.child1, self.child2], self.mindmap.children)

  def testRefreshChild1(self):
    self.assertEquals([self.child11], self.child1.children)

    self.child11.message = ""
    self.mindmap.refresh()
    self.assertEquals([], self.child1.children)

  def testRefreshMindmap2(self):
    self.assertEquals([self.child1, self.child2], self.mindmap.children)

    self.child1.message = ""
    self.mindmap.refresh()
    self.assertEquals([self.child1, self.child2], self.mindmap.children)

    self.child11.message = ""
    self.mindmap.refresh()
    self.assertEquals([self.child2], self.mindmap.children)


def mockTime():
  return time.mktime((2004, 1, 11, 14, 38, 44, 6, 11, 0))

class BirthdayTestCase(unittest.TestCase):
  import time
  def setUp(self):
    self.original_time = time.time
    time.time = mockTime
    self.birthday = Birthday()

  def testTime(self):
    self.assertEqual(time.mktime((2004, 1, 11, 14, 38, 44, 6, 11, 0)),
                     self.birthday.time)

  def testAmountOfSecond(self):
    self.birthday.time = time.mktime((2004, 1, 11, 14, 38, 42, 6, 11, 0))
    self.assertEqual(2, self.birthday.amountOfSecond())

  def testRebirth(self):
    self.birthday.time = time.mktime((2004, 1, 11, 14, 38, 42, 6, 11, 0))
    self.birthday.rebirth()
    self.assertEqual(time.mktime((2004, 1, 11, 14, 38, 44, 6, 11, 0)),
                     self.birthday.time)

  def tearDown(self):
    time.time = self.original_time


class CompositeTestCase(unittest.TestCase):
  def setUp(self):
    self.parent  = Composite()
    self.child1  = Composite()
    self.child2  = Composite()
    self.child11 = Composite()
    self.child21 = Composite()
    self.child22 = Composite()

    self.parent.addChild(self.child1)
    self.parent.addChild(self.child2)
    self.child1.addChild(self.child11)
    self.child2.addChild(self.child21)
    self.child2.addChild(self.child22)

  def testChildren(self):
    self.assertEqual([self.child1, self.child2],   self.parent.children)
    self.assertEqual([self.child11],               self.child1.children)
    self.assertEqual([self.child21, self.child22], self.child2.children)
    self.assertEqual([], self.child11.children)
    self.assertEqual([], self.child21.children)
    self.assertEqual([], self.child22.children)

  def testParent(self):
    self.assertEqual(None,        self.parent.parent)
    self.assertEqual(self.parent, self.child1.parent)
    self.assertEqual(self.parent, self.child2.parent)
    self.assertEqual(self.child1, self.child11.parent)
    self.assertEqual(self.child2, self.child21.parent)
    self.assertEqual(self.child2, self.child22.parent)

  def testAddChild(self):
    child12 = Composite()

    self.assertEqual([self.child11], self.child1.children)
    self.assertEqual(None, child12.parent)
    
    self.child1.addChild(child12)
    self.assertEqual([self.child11, child12], self.child1.children)
    self.assertEqual(self.child1, child12.parent)

  def testAddChildWithIndex(self):
    child21_5 = Composite()

    self.assertEqual([self.child21, self.child22], self.child2.children)
    self.assertEqual(None, child21_5.parent)
    
    self.child2.addChild(child21_5, 1)
    self.assertEqual([self.child21, child21_5, self.child22], self.child2.children)
    self.assertEqual(self.child2, child21_5.parent)

  def testRemoveChild(self):
    self.assertEqual([self.child21, self.child22], self.child2.children)
    self.assertEqual(self.child2, self.child21.parent)
    
    self.child2.removeChild(self.child21)
    self.assertEqual([self.child22], self.child2.children)
    self.assertEqual(None, self.child21.parent)

  def testMoveForward(self):
    self.child2.moveForward()
    self.assertEqual([self.child2, self.child1], self.parent.children)

    # don't move
    self.child2.moveForward()
    self.assertEqual([self.child2, self.child1], self.parent.children)

  def testMoveBack(self):
    self.child1.moveBack()
    self.assertEqual([self.child2, self.child1], self.parent.children)

    # don't move
    self.child1.moveBack()
    self.assertEqual([self.child2, self.child1], self.parent.children)

  def testMoveUp(self):
    self.child11.moveUp()
    self.assertEqual([self.child1, self.child11, self.child2], self.parent.children)
    self.assertEqual([], self.child1.children)

    # don't move
    self.child11.moveUp()
    self.assertEqual([self.child1, self.child11, self.child2], self.parent.children)
    self.assertEqual([], self.child1.children)

  def testMoveDown(self):
    self.child2.moveDown()
    self.assertEqual([self.child1], self.parent.children)
    self.assertEqual([self.child11, self.child2], self.child1.children)

    # don't move
    self.child1.moveDown()
    self.assertEqual([self.child1], self.parent.children)
    self.assertEqual([self.child11, self.child2], self.child1.children)

  def testMoveRoot(self):
    self.parent.moveForward()
    self.parent.moveBack()
    self.parent.moveUp()


if __name__ == "__main__":
  unittest.main()
