/*
 * startmenu.c
 *
 * Copyright (C) 2008 MikuInstaller Project. All rights reserved.
* http://mikuinstaller.sourceforge.jp/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <windows.h>
#include <shlwapi.h>
#include <shlobj.h>
#include <tiffio.h>
#include <wine/debug.h>

WINE_DEFAULT_DEBUG_CHANNEL(miku);
#define TRACE WINE_TRACE

#define arraysize(x) (sizeof(x)/sizeof(x[0]))

#ifdef __GNUC__
#define NORETURN __attribute__((noreturn))
#endif

#define ERROR_STDC    ((1U << 29) | 0U)
#define ERROR_LIBTIFF ((1U << 29) | 1U)

static void *
xmalloc(size_t size)
{
	void *p = CoTaskMemAlloc(size);
	if (p == NULL) {
		fputs("CoTaskMemAlloc failed\n", stderr);
		abort();
	}
	return p;
}

static void *
xrealloc(void *p, size_t size)
{
	p = CoTaskMemRealloc(p, size);
	if (p == NULL) {
		fputs("CoTaskMemRealloc failed\n", stderr);
		abort();
	}
	return p;
}

static void
xfree(void *p)
{
	if (p) CoTaskMemFree(p);
}

#define free xfree

static LPWSTR
xstrdupW(LPCWSTR str)
{
	size_t len = lstrlenW(str);
	LPWSTR p = xmalloc((len + 1) * sizeof(WCHAR));
	lstrcpyW(p, str);
	return p;
}

static LPWSTR
strcat_alloc(LPCWSTR str1, WCHAR delim, LPCWSTR str2)
{
	size_t len1, len2;
	LPWSTR ret;

	len1 = lstrlenW(str1);
	len2 = lstrlenW(str2);
	ret = xmalloc((len1 + 1 + len2 + 1) * sizeof(WCHAR));

	memcpy(ret, str1, len1 * sizeof(WCHAR));
	ret[len1] = delim;
	lstrcpyW(&ret[len1 + 1], str2);
	return ret;
}

/* for debug */
const char *
debugstr_w(LPCWSTR str)
{
	static char buf[4][MAX_PATH];
	static int index = 0;
	char *p;

	if (str == NULL) return "(null)";
	p = buf[index];
	memset(p, 0, sizeof(buf[0]));
	WideCharToMultiByte(CP_ACP, 0, str, -1, p, sizeof(buf[0]), NULL, NULL);
	index = (index + 1) % 4;
	return p;
}

static LPSTR
to_mbs(LPCWSTR src)
{
	size_t wlen, mlen;
	LPSTR buf;

	wlen = lstrlenW(src);
	mlen = WideCharToMultiByte(CP_UTF8, 0, src, wlen, NULL, 0, NULL, NULL);

	buf = xmalloc(mlen + 1);
	WideCharToMultiByte(CP_UTF8, 0, src, wlen, buf, mlen, NULL, NULL);
	buf[mlen] = '\0';

	return buf;
}

static LPSTR
onelineA(LPSTR str)
{
	LPSTR p;

	for (p = str; *p != '\0'; p++) {
		if (*p == '\n' || *p == '\r')
			*p = ' ';
	}
	return str;
}

static LPWSTR
basename(LPWSTR str, LPCWSTR suffix)
{
	size_t len, i;
	LPWSTR ret = str;

	len = lstrlenW(str);
	for (i = len; i-- > 0; ) {
		if (str[i] == '/' || str[i] == '\\') {
			ret = &str[i + 1];
			len = len - (i + 1);
			break;
		}
	}

	if (suffix) {
		i = lstrlenW(suffix);
		if (len > i + 1
		    && ret[len - i - 1] == '.'
		    && StrCmpNIW(&ret[len - i], suffix, i) == 0)
			ret[len - i - 1] = '\0';
	}
	return ret;
}


static void
errw32(const char *prefix, DWORD err)
{
	LPSTR buf;

	if (err == 0)
		err = GetLastError();
	switch (err) {
	case ERROR_STDC:
		perror(prefix);
		break;
	case ERROR_LIBTIFF:
		/* error message is already printed. */
		break;
	default:
		buf = NULL;
		FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER
			       | FORMAT_MESSAGE_FROM_SYSTEM
			       | FORMAT_MESSAGE_IGNORE_INSERTS,
			       NULL, err, 0, (LPTSTR)&buf, 0, NULL);
		fprintf(stderr, "%s: error %lu %s\n",
			prefix, (unsigned long)err, buf ? buf : "");
		LocalFree(buf);
		break;
	}
}

static NORETURN void
diebug(const char *msg)
{
	fprintf(stderr, "[BUG] %s\n", msg);
	abort();
}

static NORETURN void
diew32(const char *prefix, DWORD err)
{
	errw32(prefix, err);
	abort();
}

static NORETURN void
diecom(const char *prefix, HRESULT hr)
{
	fprintf(stderr, "%s: error %08lx\n", prefix, (unsigned long)hr);
	abort();
}
struct filesuffix {
	struct filesuffix *next;
	LPWSTR suffix;                 /* suffix */
	LPWSTR mimetype;               /* MIME type */
};

struct filetype {
	struct filetype *next;
	LPWSTR filetype;               /* file type */
	LPWSTR description;            /* description of this type */
	LPWSTR icon_file;              /* filename who has the icon */
	int icon_index;                /* index of icon */
	LPWSTR command;                /* associated command */
	struct filesuffix *suffixes;
};

struct progentry {
	struct progentry *next;
	LPWSTR filename;               /* filename of original file */
	LPWSTR folder;                 /* folder name */
	LPWSTR name;                   /* display name */
	HICON large_icon, small_icon;  /* icons */
	LPWSTR progname;               /* filename of application */
	struct filetype *filetypes;    /* file types accepted by this app */
};

/* for debug */
void
dump_filetype(struct filetype *ft, int indent)
{
	struct filesuffix *fs;

	while (ft) {
		printf("%*stype: %s\n", indent,"", debugstr_w(ft->filetype));
		printf("%*sdesc: %s\n", indent,"", debugstr_w(ft->description));
		printf("%*sicon: %s : %d\n", indent,"",
		       debugstr_w(ft->icon_file), ft->icon_index);
		printf("%*scmnd: %s\n", indent,"", debugstr_w(ft->command));
		printf("%*ssuffix:", indent,"");
		for (fs = ft->suffixes; fs; fs = fs->next)
			printf(" %s[%s]",
			       debugstr_w(fs->suffix),
			       debugstr_w(fs->mimetype));
		printf("\n");
		ft = ft->next;
	}
}

/* for debug */
void
dump_progentry(struct progentry *pe, int indent)
{
	while (pe) {
		printf("%*sorig: %s\n", indent,"", debugstr_w(pe->filename));
		printf("%*sname: %s / %s\n", indent,"",
		       debugstr_w(pe->folder), debugstr_w(pe->name));
		printf("%*sicon: %p %p\n", indent,"",
		       pe->large_icon, pe->small_icon);
		printf("%*sprog: %s\n", indent,"", debugstr_w(pe->progname));
		dump_filetype(pe->filetypes, indent + 2);
		pe = pe->next;
	}
}

static struct filesuffix *
alloc_filesuffix()
{
	struct filesuffix *fs;
	fs = xmalloc(sizeof(struct filesuffix));
	fs->next = NULL;
	fs->suffix = NULL;
	fs->mimetype = NULL;
	return fs;
}

static void
free_filesuffix(struct filesuffix *fs)
{
	struct filesuffix *next;
	while (fs) {
		free(fs->suffix);
		free(fs->mimetype);
		next = fs->next;
		free(fs);
		fs = next;
	}
}

static struct filetype *
alloc_filetype()
{
	struct filetype *ft;

	ft = xmalloc(sizeof(struct filetype));
	ft->next = NULL;
	ft->filetype = NULL;
	ft->description = NULL;
	ft->icon_file = NULL;
	ft->command = NULL;
	ft->suffixes = NULL;

	return ft;
}

static void
free_filetype(struct filetype *ft)
{
	struct filetype *next;

	while (ft) {
		free(ft->filetype);
		free(ft->description);
		free(ft->icon_file);
		free(ft->command);
		free_filesuffix(ft->suffixes);
		next = ft->next;
		free(ft);
		ft = next;
	}
}

static struct progentry *
alloc_progentry()
{
	struct progentry *pe;

	pe = xmalloc(sizeof(struct progentry));
	pe->next = NULL;
	pe->filename = NULL;
	pe->folder = NULL;
	pe->name = NULL;
	pe->large_icon = NULL;
	pe->small_icon = NULL;
	pe->progname = NULL;
	pe->filetypes = NULL;

	return pe;
}

static void
free_progentry(struct progentry *pe)
{
	struct progentry *next;

	while (pe) {
		free(pe->filename);
		free(pe->folder);
		free(pe->name);
		if (pe->large_icon)
			DestroyIcon(pe->large_icon);
		if (pe->small_icon)
			DestroyIcon(pe->small_icon);
		free(pe->progname);
		free_filetype(pe->filetypes);
		next = pe->next;
		free(pe);
		pe = next;
	}
}


static struct filetype *
find_filetype(struct filetype *ft, LPCWSTR type)
{
	while (ft) {
		if (lstrcmpW(ft->filetype, type) == 0)
			break;
		ft = ft->next;
	}
	return ft;
}

static BOOL
match_command(LPCWSTR cmdline, LPCWSTR prog, size_t len)
{
	int m;

	/*
	 * command "c:\Program Files\Internet Explorer/iexplore.exe"
	 * matches against either
	 * "c:\Program Files\Internet Explorer/iexplore.exe ..." or
	 * "\"c:\Program Files\Internet Explorer/iexplore.exe\" ...".
	 */
	if (cmdline[0] == '"') {
		m = StrCmpNIW(&cmdline[1], prog, len);
		if (m == 0 && cmdline[len] == '"')
			return TRUE;
	} else {
		m = StrCmpNIW(cmdline, prog, len);
		if (m == 0 && (cmdline[len] == ' '
			       || cmdline[len] == '\0'))
			return TRUE;
	}
	return FALSE;
}

static struct filetype *
take_filetypes_by_command(struct filetype **ft_all, LPCWSTR command)
{
	size_t len;
	struct filetype *dst, *ft;
	len = lstrlenW(command);
	dst = NULL;

	/*
	 * command "c:\Program Files\Internet Explorer/iexplore.exe"
	 * matches against either
	 * "c:\Program Files\Internet Explorer/iexplore.exe ..." or
	 * "\"c:\Program Files\Internet Explorer/iexplore.exe\" ...".
	 */
	while ((ft = *ft_all)) {
		if (match_command(ft->command, command, len)) {
			*ft_all = ft->next;
			ft->next = dst;
			dst = ft;
		} else {
			ft_all = &ft->next;
		}
	}
	return dst;
}

static void
merge_filetype(struct progentry *pe, struct filetype **ft_all)
{
	struct progentry *p;
	LPCWSTR cmd;

	for (p = pe; p; p = p->next) {
		cmd = p->progname ? p->progname : p->filename;
		p->filetypes = take_filetypes_by_command(ft_all, cmd);
	}
}


static LPWSTR
reg_read_string(HKEY hkey, LPCWSTR name)
{
	LONG r;
	DWORD type;
	DWORD len, len2;
	void *buf, *buf2;

	len = 0;
	r = RegQueryValueExW(hkey, name, NULL, &type, NULL, &len);
	if (r != ERROR_SUCCESS || (type != REG_SZ && type != REG_EXPAND_SZ))
		return NULL;

	buf = xmalloc(len);
	r = RegQueryValueExW(hkey, name, NULL, NULL, buf, &len);
	if (r != ERROR_SUCCESS) {
		free(buf);
		return NULL;
	}
	if (type == REG_EXPAND_SZ) {
		buf2 = xmalloc(len);
		len2 = ExpandEnvironmentStringsW(buf, buf2,
						 len / sizeof(WCHAR));
		if (len2 > len / sizeof(WCHAR)) {
			buf2 = xrealloc(buf2, len2 * sizeof(WCHAR));
			ExpandEnvironmentStringsW(buf, buf2, len2);
		}
		free(buf);
		buf = buf2;
	}

	return buf;
}

static void
parse_default_icon(LPWSTR src, int *index)
{
	size_t len;

	*index = 0;

	len = lstrlenW(src);
	while (len-- > 0) {
		if (src[len] == ',') {
			StrToIntExW(&src[len+1], 0, index);
			src[len] = '\0';
			break;
		}
		if (src[len] == '"')
			break;
	}
	PathUnquoteSpacesW(src);
}

static LPWSTR
reg_read_command(HKEY hkey, LPCWSTR action)
{
	static const WCHAR strCommand[] = {'C','o','m','m','a','n','d',0};
	HKEY hkey2, hkey3;
	LPWSTR ret = NULL;
	LONG r;

	/* "shell/<action>/command","@" : command line */
	r = RegOpenKeyExW(hkey, action, 0, KEY_READ, &hkey2);
	if (r == ERROR_SUCCESS) {
		r = RegOpenKeyExW(hkey2, strCommand, 0, KEY_READ, &hkey3);
		if (r == ERROR_SUCCESS) {
			ret = reg_read_string(hkey3, NULL);
			RegCloseKey(hkey3);
		}
		RegCloseKey(hkey2);
	}

	return ret;
}

static int
reg_read_filetype(struct filetype *ft)
{
	static const WCHAR strDefaultIcon[] =
		{'D','e','f','a','u','l','t','I','c','o','n',0};
	static const WCHAR strShell[] =
		{'s','h','e','l','l',0};
	static const WCHAR strOpen[] =
		{'o','p','e','n',0};
	HKEY hkey, hkey2;
	LONG r;
	LPWSTR action;

	/* Assume that ft->filetype is set. Any other fields except suffixes
	   in ft are set by this function. */

	/* "HKCR/<filetype>" */
	r = RegOpenKeyExW(HKEY_CLASSES_ROOT, ft->filetype, 0, KEY_READ, &hkey);
	if (r != ERROR_SUCCESS)
		return 0;

	/* "HKCR/<filetype>","@" : description of file type */
	ft->description = reg_read_string(hkey, NULL);

	/* "HKCR/<filetype>/DefaultIcon","@" : default icon */
	r = RegOpenKeyExW(hkey, strDefaultIcon, 0, KEY_READ, &hkey2);
	if (r == ERROR_SUCCESS) {
		ft->icon_file = reg_read_string(hkey2, NULL);
		parse_default_icon(ft->icon_file, &ft->icon_index);
		RegCloseKey(hkey2);
	} else {
		ft->icon_file = NULL;
		ft->icon_index = 0;
	}

	/* "HKCR/<filetype>/shell" */
	r = RegOpenKeyExW(hkey, strShell, 0, KEY_READ, &hkey2);
	if (r != ERROR_SUCCESS) {
		RegCloseKey(hkey);
		return 0;
	}

	/* "HKCR/<filetype>/shell","@" : default action */
	action = reg_read_string(hkey2, NULL);

	/* "HKCR/<filetype>/shell/<action>/command","@" : command line */
	if (action)
		ft->command = reg_read_command(hkey2, action);
	/* "open" is default action */
	if (!ft->command)
		ft->command = reg_read_command(hkey2, strOpen);

	free(action);
	RegCloseKey(hkey2);
	RegCloseKey(hkey);

	return (ft->command != NULL);
}

static int
reg_read_suffix(LPWSTR suffix, struct filetype **ft_all)
{
	static const WCHAR strContentType[] =
		{'C','o','n','t','e','n','t',' ','T','y','p','e',0};
	struct filesuffix *fs;
	struct filetype *ft;
	LPWSTR filetype;
	LONG r;
	HKEY hkey;

	if (suffix[0] != '.')
		return 0;

	/* "HKCR/<suffix>" */
	r = RegOpenKeyExW(HKEY_CLASSES_ROOT, suffix, 0, KEY_READ, &hkey);
	if (r != ERROR_SUCCESS)
		return 0;

	/* "HKCR/<suffix>","@" : file type */
	filetype = reg_read_string(hkey, NULL);
	if (filetype == NULL) {
		RegCloseKey(hkey);
		return 0;
	}

	fs = alloc_filesuffix();

	fs->suffix = xstrdupW(&suffix[1]);

	/* "HKCR/<suffix>/Content Type" : MIME type */
	fs->mimetype = reg_read_string(hkey, strContentType);

	RegCloseKey(hkey);

	ft = find_filetype(*ft_all, filetype);
	if (ft) {
		fs->next = ft->suffixes;
		ft->suffixes = fs;
		return 1;
	}

	ft = alloc_filetype();
	ft->filetype = filetype;
	ft->suffixes = fs;

	if (!reg_read_filetype(ft)) {
		free_filetype(ft);
		return 0;
	}

	ft->next = *ft_all;
	*ft_all = ft;
	return 1;
}

static struct filetype *
get_filetypes()
{
	LONG r;
	WCHAR name[256];
	DWORD namelen;
	DWORD index;
	struct filetype *ft = NULL;

	for (index = 0; ; index++) {
		namelen = arraysize(name);
		r = RegEnumKeyExW(HKEY_CLASSES_ROOT, index, name, &namelen,
				  NULL, NULL, NULL, NULL);
		if (r != ERROR_SUCCESS)
			break;

		reg_read_suffix(name, &ft);
	}
	if (r != ERROR_NO_MORE_ITEMS) diew32("RegEnumKeyExW", r);

	return ft;
}

static LPWSTR
get_display_name(IShellFolder *sf, LPITEMIDLIST pidl, SHGDNF flags)
{
	STRRET sr;
	HRESULT hr;
	LPWSTR ret;

	hr = sf->lpVtbl->GetDisplayNameOf(sf, pidl, flags, &sr);
	if (!SUCCEEDED(hr)) diecom("IShellFolder::GetDisplayNameOf", hr);
	hr = StrRetToStrW(&sr, pidl, &ret);
	if (!SUCCEEDED(hr)) diecom("StrRetToStrW", hr);

	return ret;
}

static LPWSTR
get_link_to(LPCWSTR filename)
{
	IShellLink *sl;
	IPersistFile *pf;
	HRESULT hr;
	TCHAR buf[MAX_PATH];
	LPWSTR ret;

	hr = CoCreateInstance(&CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER,
			      &IID_IShellLink, (LPVOID*)&sl);
	if (!SUCCEEDED(hr)) diecom("CoCreateInstance", hr);

	hr = sl->lpVtbl->QueryInterface(sl, &IID_IPersistFile, (LPVOID*)&pf);
	if (!SUCCEEDED(hr)) diecom("QueryInterface", hr);

	hr = pf->lpVtbl->Load(pf, filename, STGM_READ);
	if (hr == E_FAIL)
		return NULL;  /* not a shell link */
	if (!SUCCEEDED(hr))
		return NULL;  /* failed in some reason */

	hr = sl->lpVtbl->GetPath(sl, buf, sizeof(buf), NULL, SLGP_UNCPRIORITY);
	if (!SUCCEEDED(hr)) diecom("IShellLink::GetPath", hr);

	pf->lpVtbl->Release(pf);
	sl->lpVtbl->Release(sl);

	hr = SHStrDup(buf, &ret);
	if (!SUCCEEDED(hr)) diecom("SHStrDup", hr);

	return ret;
}

static void
get_shell_icon(IShellFolder *sf, LPCITEMIDLIST pidl, LPCWSTR filename,
	       HICON *large_icon, HICON *small_icon)
{
	IExtractIcon *ei;
	TCHAR buf[MAX_PATH];
	HRESULT hr;
	int index;
	SHFILEINFOW sfi;

	*large_icon = NULL;
	*small_icon = NULL;

	/* Try IExtractIcon::Extract first in order to avoid to add
	 * shortcut mark to icon.
	 */
	if (sf) {
		hr = sf->lpVtbl->GetUIObjectOf(sf, NULL, 1, &pidl,
					       &IID_IExtractIcon, NULL,
					       (LPVOID*)&ei);
		if (!SUCCEEDED(hr)) diecom("IShellFolder::GetUIObjectOf", hr);

		hr = ei->lpVtbl->GetIconLocation(ei, GIL_FORSHELL,
						 buf, sizeof(buf),
						 &index, NULL);
		if (hr == S_OK) {
			/* FIXME: WINE: Currently nIconSize is ignored. */
			hr = ei->lpVtbl->Extract(ei, buf, index,
						 large_icon, small_icon,
						 MAKELONG(32, 16));
			if (!SUCCEEDED(hr)) {
				*large_icon = NULL;
				*small_icon = NULL;
			}
		}

		ei->lpVtbl->Release(ei);
	}

	if (*large_icon == NULL) {
		/* FIXME: WINE: SHGetFileInfoW returns S_OK without setting
		 * sfi.hIcon even if filename does not exist.
		 */
		sfi.hIcon = NULL;
		hr = SHGetFileInfoW(filename, 0, &sfi, sizeof(sfi),
				    SHGFI_ICON | SHGFI_LARGEICON);
		if (SUCCEEDED(hr))
			*large_icon = sfi.hIcon;
	}
	if (*small_icon == NULL) {
		sfi.hIcon = NULL;
		hr = SHGetFileInfoW(filename, 0, &sfi, sizeof(sfi),
				    SHGFI_ICON | SHGFI_SMALLICON);
		if (SUCCEEDED(hr))
			*small_icon = sfi.hIcon;
	}
}

struct progentry *
visit_programs(IShellFolder *parent, LPITEMIDLIST folder_id,
	       LPCWSTR folder_name, struct progentry *tail)
{
	IShellFolder *sf;
	IEnumIDList *ei;
	LPITEMIDLIST pidl;
	LPWSTR name, folder;
	HRESULT hr;
	ULONG flags;
	struct progentry *pe;

	hr = parent->lpVtbl->BindToObject(parent, folder_id, NULL,
					  &IID_IShellFolder, (LPVOID*)&sf);
	if (!SUCCEEDED(hr)) diecom("IShellFolder::BindToObject", hr);

	hr = sf->lpVtbl->EnumObjects(sf, NULL,
				     SHCONTF_FOLDERS | SHCONTF_NONFOLDERS,
				     &ei);
	if (!SUCCEEDED(hr)) diecom("IShellFolder::EnumObjects", hr);

	while ((hr = ei->lpVtbl->Next(ei, 1, &pidl, NULL)) == S_OK) {
		name = get_display_name(sf, pidl, SHGDN_INFOLDER);

		/* FIXME: WINE: UnixFolder::GetAttributesOf always clear
		 * SFGAO_LINK flag. We cannot use it to determine whether
		 * this id is a shell link or not.
		 */
		flags = SFGAO_FOLDER;
		hr = sf->lpVtbl->GetAttributesOf(sf, 1, (LPCITEMIDLIST*)&pidl,
						 &flags);
		if (!SUCCEEDED(hr)) diecom("IShellFolder::GetAttributesOf", hr);

		if (flags & SFGAO_FOLDER) {
			folder = strcat_alloc(folder_name, '/', name);
			pe = visit_programs(sf, pidl, folder, tail);
			free(folder);
			free(name);
			free(pidl);
			tail = pe;
			continue;
		}

		pe = alloc_progentry();
		pe->folder = xstrdupW(folder_name);
		pe->name = name;
		pe->filename = get_display_name(sf, pidl, SHGDN_FORPARSING);

		/* FIXME: WINE: Obtaining IShellLink interface by
		 * UnixFolder::GetUIObjectOf is not implemented.
		 * Try CoCreateInstance and QueryInterface instead.
		 */
		pe->progname = get_link_to(pe->filename);

		get_shell_icon(sf, pidl, pe->filename,
			       &pe->large_icon, &pe->small_icon);

		pe->next = tail;
		tail = pe;
		free(pidl);
	}
	if (hr != S_FALSE) diecom("IEnumIDList::Next", hr);

	ei->lpVtbl->Release(ei);
	sf->lpVtbl->Release(sf);

	return tail;
}

static struct progentry *
get_prog_entries()
{
	static const WCHAR empty[] = {0};
	IShellFolder *sf;
	LPITEMIDLIST pidl;
	struct progentry *pe;
	HRESULT hr;

	hr = SHGetDesktopFolder(&sf);
	if (!SUCCEEDED(hr)) diecom("SHGetDesktopFolder", hr);

	hr = SHGetFolderLocation(NULL, CSIDL_COMMON_PROGRAMS, NULL, 0, &pidl);
	if (!SUCCEEDED(hr)) diecom("SHGetFolderLocation", hr);

	pe = visit_programs(sf, pidl, empty, NULL);

	free(pidl);
	sf->lpVtbl->Release(sf);

	return pe;
}

static struct progentry *
make_prog_entries(int num_files, char **filename)
{
	static const WCHAR strExe[] = {'e','x','e',0};
	int i;
	HRESULT hr;
	struct progentry *pe, *tail = NULL;

	for (i = 0; i < num_files; i++) {
		pe = alloc_progentry();
		hr = SHStrDupA(filename[i], &pe->filename);
		if (!SUCCEEDED(hr)) diecom("SHStrDup", hr);

		pe->folder = xmalloc(sizeof(WCHAR));
		pe->folder[0] = '\0';
		pe->name = xstrdupW(pe->filename);
		pe->name = basename(pe->name, strExe);

		get_shell_icon(NULL, NULL, pe->filename,
			       &pe->large_icon, &pe->small_icon);

		pe->next = tail;
		tail = pe;
	}

	return tail;
}

static HBITMAP
to_dib(HBITMAP hbitmap)
{
	BITMAP bm;
	BITMAPINFOHEADER bmih;
	HBITMAP hdib, bmold1, bmold2;
	HDC hwnddc, hdc1, hdc2;

	if (GetObjectW(hbitmap, sizeof(bm), &bm) != sizeof(bm)) {
		SetLastError(ERROR_BAD_ARGUMENTS);
		return NULL;
	}

	memset(&bmih, 0, sizeof(bmih));
	bmih.biSize = sizeof(BITMAPINFOHEADER);
	bmih.biWidth = bm.bmWidth;
	bmih.biHeight = -abs(bm.bmHeight);  /* origin is top-left */
	bmih.biPlanes = 1;
	bmih.biBitCount = 32;
	bmih.biCompression = BI_RGB;
	hdib = CreateDIBSection(NULL, (BITMAPINFO*)&bmih, DIB_RGB_COLORS,
				NULL, NULL, 0);
	if (!hdib) abort();

	hwnddc = GetDC(NULL);
	hdc1 = CreateCompatibleDC(hwnddc);
	hdc2 = CreateCompatibleDC(hwnddc);
	ReleaseDC(NULL, hwnddc);
	bmold1 = (HBITMAP)SelectObject(hdc1, hdib);
	bmold2 = (HBITMAP)SelectObject(hdc2, hbitmap);

	BitBlt(hdc1, 0, 0, bm.bmWidth, bm.bmHeight, hdc2, 0, 0, SRCCOPY);

	SelectObject(hdc1, bmold1);
	SelectObject(hdc2, bmold2);
	DeleteDC(hdc1);
	DeleteDC(hdc2);
	GdiFlush();

	return hdib;
}

static void *
rgb_to_abgr(void *color_data, void *mask_data, size_t size)
{
	const DWORD *src = color_data;  /* XOR mask */
	const DWORD *mask = mask_data;  /* AND mask */
	DWORD *dst;
	DWORD pixel, alpha;
	size_t i;

	dst = xmalloc(size);

	for (i = 0; i < size / sizeof(DWORD); i++) {
		pixel = src[i];
		/* 0x__RRGGBB -> 0xAABBGGRR */
		pixel = (pixel & 0x0000ff00)
			| ((pixel >> 16) & 0xff)
			| ((pixel & 0xff) << 16);

		alpha = mask ? mask[i] : 0;
		/* assume R,G,B have same value in mask. */
		alpha = (0xff - (alpha & 0xff)) << 24;

		dst[i] = pixel | alpha;
	}
	return dst;
}

#define CHECK_DIB_RGB32_TOPLEFT(dib)			\
	((dib)->dsBmih.biBitCount == 32			\
	 && (dib)->dsBmih.biCompression == BI_RGB	\
	 && (dib)->dsBmih.biHeight < 0)

static int
write_tiff_image(TIFF *out, HBITMAP hbitmap, HBITMAP hmask)
{
	DIBSECTION dib, dib_mask;
	void *data;

	if (GetObjectW(hbitmap, sizeof(dib), &dib) != sizeof(dib)
	    || !CHECK_DIB_RGB32_TOPLEFT(&dib)) {
		SetLastError(ERROR_BAD_ARGUMENTS);
		return 0;
	}
	if (GetObjectW(hmask, sizeof(dib_mask), &dib_mask) != sizeof(dib_mask)
	    || !CHECK_DIB_RGB32_TOPLEFT(&dib_mask)) {
		SetLastError(ERROR_BAD_ARGUMENTS);
		return 0;
	}
	if (dib.dsBmih.biSizeImage != dib_mask.dsBmih.biSizeImage) {
		SetLastError(ERROR_BAD_ARGUMENTS);
		return 0;
	}

	TIFFSetField(out, TIFFTAG_IMAGEWIDTH, dib.dsBmih.biWidth);
	TIFFSetField(out, TIFFTAG_IMAGELENGTH, -dib.dsBmih.biHeight);
	TIFFSetField(out, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
	TIFFSetField(out, TIFFTAG_SAMPLESPERPIXEL, 4);
	TIFFSetField(out, TIFFTAG_BITSPERSAMPLE, 8);
	TIFFSetField(out, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
	TIFFSetField(out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
	TIFFSetField(out, TIFFTAG_COMPRESSION, COMPRESSION_PACKBITS);
	TIFFSetField(out, TIFFTAG_ROWSPERSTRIP, -dib.dsBmih.biHeight);

	data = rgb_to_abgr(dib.dsBm.bmBits, dib_mask.dsBm.bmBits,
			   dib.dsBmih.biSizeImage);
	TIFFWriteEncodedStrip(out, 0, data, dib.dsBmih.biSizeImage);
	free(data);

	TIFFWriteDirectory(out);
	return 1;
}

static int
write_tiff_icon(TIFF *out, HICON hicon)
{
	ICONINFO icon;
	HBITMAP hbitmap, hmask;
	int ret = 0;

	if (!GetIconInfo(hicon, &icon))
		return 0;

	hbitmap = to_dib(icon.hbmColor);
	if (hbitmap) {
		hmask = to_dib(icon.hbmMask);
		if (hmask) {
			ret = write_tiff_image(out, hbitmap, hmask);
			DeleteObject(hmask);
		}
		DeleteObject(hbitmap);
	}

	DeleteObject(icon.hbmColor);
	DeleteObject(icon.hbmMask);
	return ret;
}

static int
save_icon_tiff(LPCTSTR filename, HICON large_icon, HICON small_icon)
{
	TIFF *out;
	int ret;

	if (large_icon == NULL && small_icon == NULL)
		return -1;

	out = TIFFOpen(filename, "w");
	if (!out) {
		SetLastError(ERROR_LIBTIFF);
		return 0;
	}
	ret = 1;

	if (large_icon != NULL)
		ret = write_tiff_icon(out, large_icon);
	if (small_icon != NULL)
		ret = (ret && write_tiff_icon(out, small_icon));

	TIFFClose(out);

	return ret;
}

/* for debug */
int
write_bmp(HBITMAP hbitmap, const char *filename)
{
	BITMAPFILEHEADER bmh;
	DIBSECTION dib;
	FILE *file;
	int ret;

	if (GetObjectW(hbitmap, sizeof(dib), &dib) != sizeof(dib)) {
		SetLastError(ERROR_BAD_ARGUMENTS);
		return 0;
	}

	memset(&bmh, 0, sizeof(bmh));
	memcpy(&bmh.bfType, "BM", 2);
	bmh.bfType = 0x4d42;   /* 'BM' in little endian */
	bmh.bfOffBits = sizeof(bmh) + dib.dsBmih.biSize;
	bmh.bfSize = bmh.bfOffBits + dib.dsBmih.biSizeImage;

	file = fopen(filename, "wb");
	if (file == NULL) {
		perror("fopen");
		return 0;
	}

	ret = fwrite(&bmh, sizeof(bmh), 1, file);
	ret = (ret && fwrite(&dib.dsBmih, dib.dsBmih.biSize, 1, file));
	ret = (ret && fwrite(dib.dsBm.bmBits, dib.dsBmih.biSizeImage, 1, file));

	if (ret == 0)
		perror("fwrite");

	fclose(file);
	return ret;
}

/* for debug */
int
write_bmp_icon(HICON hicon, const char *basename)
{
	ICONINFO icon;
	HBITMAP hbitmap;
	char buf[MAX_PATH];

	if (!GetIconInfo(hicon, &icon))
		return 0;

	hbitmap = to_dib(icon.hbmColor);
	snprintf(buf, sizeof(buf), "%s_color.bmp", basename);
	write_bmp(hbitmap, buf);
	DeleteObject(hbitmap);

	hbitmap = to_dib(icon.hbmMask);
	snprintf(buf, sizeof(buf), "%s_color.bmp", basename);
	write_bmp(hbitmap, buf);
	DeleteObject(hbitmap);

	DeleteObject(icon.hbmMask);
	DeleteObject(icon.hbmColor);
	return 1;
}

/* for debug */
int
save_icon_bmp(LPCTSTR basename, HICON large_icon, HICON small_icon)
{
	char buf[MAX_PATH];
	int ret;

	if (large_icon == NULL && small_icon == NULL)
		return -1;

	ret = 1;
	if (large_icon != NULL) {
		snprintf(buf, sizeof(buf), "%s_large", basename);
		ret = write_bmp_icon(large_icon, buf);
	}
	if (small_icon != NULL) {
		snprintf(buf, sizeof(buf), "%s_small", basename);
		ret = (ret && write_bmp_icon(small_icon, buf));
	}
	return ret;
}


#define LEN_FILENAME  256   /* large enough to hold 2 integers + 16 chars */

static int
save_filetype(struct filetype *ft, int prog_id, int type_id, FILE *out)
{
	struct filesuffix *fs;
	LPSTR str;
	HICON large_icon, small_icon;
	int ret = 1;
	int i;
	char filename[LEN_FILENAME];

	if (ft->suffixes == NULL)
		diebug("ft->suffixes is NULL");

	if (ft->description) {
		str = onelineA(to_mbs(ft->description));
		fprintf(out, "TYPE:%d %s\n", type_id, str);
		free(str);
	} else {
		fprintf(out, "TYPE:%d\n", type_id);
	}

	for (fs = ft->suffixes, i = 0; fs; fs = fs->next, i++) {
		str = onelineA(to_mbs(fs->suffix));
		fprintf(out, "TYPE:%d:SUFFIX:%d %s\n", type_id, i, str);
		free(str);
		if (fs->mimetype) {
			str = onelineA(to_mbs(fs->suffix));
			fprintf(out, "TYPE:%d:SUFFIX:%d:MIMETYPE %s\n",
				type_id, i, str);
			free(str);
		}
	}

	if (ft->icon_file) {
		snprintf(filename, sizeof(filename),
			 "_e%03dt%03d.tiff", prog_id, type_id);
		large_icon = NULL;
		small_icon = NULL;
		ExtractIconExW(ft->icon_file, ft->icon_index,
			       &large_icon, &small_icon, 1);
		ret = save_icon_tiff(filename, large_icon, small_icon);
		if (ret > 0)
			fprintf(out, "TYPE:%d:ICON %s\n", type_id, filename);
		else if (ret < 0)
			ret = 1;
		DestroyIcon(large_icon);
		DestroyIcon(small_icon);
	}

	return ret;
}

static int
save_progentry(struct progentry *pe, int prog_id)
{
	struct filetype *ft;
	LPSTR str;
	int ret = 1;
	int i;
	char filename[LEN_FILENAME];
	FILE *out;

	snprintf(filename, sizeof(filename), "_Entry%03d", prog_id);
	out = fopen(filename, "wb");
	if (out == NULL) {
		SetLastError(ERROR_STDC);
		return 0;
	}

	str = onelineA(to_mbs(pe->name));
	fprintf(out, "NAME %s\n", str);
	free(str);
	str = onelineA(to_mbs(pe->folder));
	fprintf(out, "FOLDER %s\n", str);
	free(str);
	str = onelineA(to_mbs(pe->filename));
	fprintf(out, "FILE %s\n", str);
	free(str);
	if (pe->progname) {
		str = onelineA(to_mbs(pe->progname));
		fprintf(out, "PROG %s\n", str);
		free(str);
	}

	for (ft = pe->filetypes, i = 0; ft; ft = ft->next, i++) {
		ret = save_filetype(ft, prog_id, i, out);
		if (!ret)
			break;
	}

	if (ret) {
		snprintf(filename, sizeof(filename), "_e%03d.tiff", prog_id);
		ret = save_icon_tiff(filename, pe->large_icon, pe->small_icon);
		if (ret > 0)
			fprintf(out, "ICON %s\n", filename);
		else if (ret < 0)
			ret = 1;
	}

	fclose(out);
	return ret;
}

static int
save_prog_entries(struct progentry *pe)
{
	int i, ret = 1;

	for (i = 0; pe; pe = pe->next, i++) {
		ret = save_progentry(pe, i);
		if (!ret)
			break;
	}

	return ret;
}

int main(int argc, char **argv)
{
	struct filetype *ft;
	struct progentry *pe;
	int r;
	int status = 0;

	CoInitialize(NULL);

	if (argc == 1)
		pe = get_prog_entries();
	else
		pe = make_prog_entries(argc - 1, &argv[1]);

	ft = get_filetypes();

	merge_filetype(pe, &ft);

	/*
	dump_filetype(ft, 0);
	dump_progentry(pe, 0);
	*/

	r = save_prog_entries(pe);
	if (!r) {
		errw32(argv[0], 0);
		status = 1;
	}

	free_progentry(pe);
	free_filetype(ft);

	CoUninitialize();

	return status;
}
