/*
 * shellexec.c
 *
 * Copyright (C) 2008 MikuInstaller Project. All rights reserved.
* http://mikuinstaller.sourceforge.jp/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <tlhelp32.h>
#include <wine/exception.h>
#include <wine/debug.h>

#define arraysize(a) (sizeof(a)/sizeof(a[0]))

void _splitpath(const char*,char*,char*,char*,char*);

WINE_DEFAULT_DEBUG_CHANNEL(miku);

static int
print_error(const char *title, const char *filename, DWORD err)
{
	LPSTR msg, buf;
	int ret = 0;

	msg = NULL;
	FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER
		       | FORMAT_MESSAGE_FROM_SYSTEM
		       | FORMAT_MESSAGE_IGNORE_INSERTS,
		       NULL, err, 0, (LPTSTR)&msg, 0, NULL);
	fprintf(stderr, "%s: %s: error %lu %s\n",
		title, filename, (unsigned long)err, msg ? msg : "");

	buf = malloc(sizeof("Cannot open file ") + lstrlenA(filename) +
		     sizeof(" : ") + lstrlenA(msg));
	if (buf) {
		sprintf(buf, "Cannot open file %s : %s", filename, msg);
		ret = MessageBoxA(NULL, buf, title,
				  MB_OK | MB_ICONERROR | MB_TOPMOST);
	}
	LocalFree(msg);
	return ret;
}

static BOOL
find_pid(DWORD *pids, DWORD len, DWORD pid)
{
	DWORD i;
	for (i = 0; i < len; i++) {
		if (pids[i] == pid)
			return TRUE;
	}
	return FALSE;
}

static DWORD
find_child_processes(DWORD *pids, DWORD offset, DWORD limit)
{
	DWORD pid = GetCurrentProcessId();
	HANDLE ths;
	PROCESSENTRY32 pe;
	DWORD i;
	BOOL r;

	WINE_TRACE("\n");

	ths = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
	if (ths == INVALID_HANDLE_VALUE) {
		WINE_TRACE("CreateToolhelp32Snapshot failed\n");
		return 0;
	}

	i = offset;

	pe.dwSize = sizeof(pe);
	pe.th32ParentProcessID = 0;
	pe.th32ProcessID = 0;
	r = Process32First(ths, &pe);
	while (r) {
		if (pe.th32ParentProcessID == pid
		    && !find_pid(pids, offset, pe.th32ProcessID)) {
			WINE_TRACE("pid %d, ppid %d\n",
				   (int)pe.th32ProcessID,
				   (int)pe.th32ParentProcessID);

			if (i >= limit) {
				WINE_WARN("number of children exceeds.");
				break;
			}
			pids[i++] = pe.th32ProcessID;
		}
		pe.dwSize = sizeof(pe);
		pe.th32ParentProcessID = 0;
		pe.th32ProcessID = 0;
		r = Process32Next(ths, &pe);
	}

	CloseHandle(ths);
	return i;
}

int
main(int argc, char **argv)
{
	const char *progname = argv[0];
	SHELLEXECUTEINFO see;
	DWORD err;
	const char *filename;
	char buf[MAX_PATH];
	char dir[MAX_PATH];
	DWORD pids[MAXIMUM_WAIT_OBJECTS];
	HANDLE hProcess[MAXIMUM_WAIT_OBJECTS];
	volatile DWORD num_pre, num_post;
	volatile DWORD i;
	DWORD status = 128;
	int chdir = 0;

	if (argc >= 2 && strcmp(argv[1], "-c") == 0) {
		argc--;
		argv++;
		chdir = 1;
	}
	if (argc != 2) {
		fprintf(stderr, "usage: %s [-c] <filename>\n", progname);
		return status;
	}
	filename = argv[1];

	if (chdir) {
		_splitpath(filename, buf, dir, NULL, NULL);
		strncat(buf, dir, sizeof(buf));
		if (buf[0] != '\0') {
			SetCurrentDirectory(buf);
			WINE_TRACE("Change directory to %s\n", buf);
		}
	}

	CoInitializeEx(NULL, COINIT_APARTMENTTHREADED | COINIT_DISABLE_OLE1DDE);

	/*
	 * ShellExecuteEx may set hProcess to NULL even if a process
	 * was successfully created. This is the case if
	 * ShellExecuteEx calls IContextMenu::InvokeCommand to start
	 * the process, especially if a shell link is intended to be
	 * opened.
	 * To be certain to obtain the child process handle opened by
	 * ShellExecuteEx, we try to find the child process from
	 * global process list.
	 */
	num_pre = find_child_processes(pids, 0, arraysize(pids));

	memset(&see, 0, sizeof(see));
	see.cbSize = sizeof(see);
	see.fMask = SEE_MASK_NOCLOSEPROCESS
		| SEE_MASK_NOASYNC
		| SEE_MASK_FLAG_NO_UI;
	see.hwnd = NULL;
	see.lpVerb = NULL;
	see.lpFile = filename;
	see.lpParameters = NULL;
	see.lpDirectory = NULL;
	see.nShow = SW_SHOWNORMAL;
	see.hProcess = NULL;

	if (!ShellExecuteExA(&see)) {
		err = GetLastError();
		err = (err != 0) ? err : (DWORD)((int)see.hInstApp);
		if (print_error(progname, filename, err))
			status = 0;
		return status;
	}

	WINE_TRACE("ShellExecuteExA succeeded : hProcess = %p, hInstApp = %p\n",
		   see.hProcess, see.hInstApp);

	if (see.hProcess) {
		hProcess[0] = see.hProcess;
		num_pre = 0;
		num_post = 1;
	} else {
		num_post = find_child_processes(pids, num_pre, arraysize(pids));
		for (i = num_pre; i < num_post; i++) {
			hProcess[i] = OpenProcess(PROCESS_QUERY_INFORMATION
						  | PROCESS_TERMINATE
						  | SYNCHRONIZE,
						  FALSE,
						  pids[i]);
		}
	}

	if (num_pre < num_post) {
		__TRY
		{
			WaitForMultipleObjects(num_post - num_pre,
					       &hProcess[num_pre],
					       TRUE, INFINITE);
			for (i = num_pre; i < num_post; i++) {
				if (GetExitCodeProcess(hProcess[i], &status))
					break;
			}
		}
		__EXCEPT_ALL
		{
			for (i = num_pre; i < num_post; i++) {
				WINE_TRACE("TerminateProcess %p\n",
					   hProcess[i]);
				TerminateProcess(hProcess[i], 255);
			}
		}
		__ENDTRY

		for (i = num_pre; i < num_post; i++)
			CloseHandle(hProcess[i]);
	}

	WINE_TRACE("exit code: %d\n", (int)status);

	CoUninitialize();

	return status;
}
