--
-- frontend of MikuInstaller
--
-- Copyright (C) 2008 MikuInstaller Project. All rights reserved.
-- http://mikuinstaller.sourceforge.jp/
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
--  1. Redistributions of source code must retain the above copyright
--     notice, this list of conditions and the following disclaimer.
--  2. Redistributions in binary form must reproduce the above copyright
--     notice, this list of conditions and the following disclaimer in
--     the documentation and/or other materials provided with the
--     distribution.
-- THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
-- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
-- THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
-- PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
-- BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
-- CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
-- SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
-- BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
-- WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
-- OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
-- IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--

property BASH : "/bin/bash"

property MIKUPREFIX : "default"

property MIKUROOT : missing value            -- set by checkAndInstallWine
property MIKUBUNDLE : missing value          -- set by checkAndInstallWine

property CommandExec : missing value         -- set by awake from nib
property PrefixPane : missing value          -- set by awake from nib
property InfoPane : missing value            -- set by awake from nib

property Initialized : false                 -- for initialize
property NeedUpdatePrefixes : false          -- for initialize
property FileToBeExecuted : missing value    -- for initialize
property NeedCreateAppBundles : false        -- for initialize

property SheetMode : missing value

-------- Utilities --------

on applicationDir()
	set appFolder to path to applications folder from user domain as string
	return POSIX path of (appFolder & "MikuInstaller")
end applicationDir

on winePrefixDir()
	return MIKUROOT & "/prefix"
end winePrefixDir

on winePrefixPath()
	if MIKUPREFIX is missing value then
		error "プレフィックスが選択されていません。"
	else
		return winePrefixDir() & "/" & MIKUPREFIX
	end if
end winePrefixPath

on testExist(filename)
	tell application "System Events"
		set ret to exists disk item ((filename as POSIX file) as string)
	end tell
	return ret
end testExist

on bundleVersion(theBundle)
	set theVersion to call method "objectForInfoDictionaryKey:" of theBundle with parameter "CFBundleVersion"
	try
		return theVersion
	end try
	return missing value
end bundleVersion

on execute(message, command, args)
	set myBundle to call method "mainBundle" of class "NSBundle"
	set myPath to call method "bundlePath" of myBundle
	set myVersion to bundleVersion(myBundle)
	set WINEPREFIX to winePrefixPath()
	set env to call method "dictionaryWithObjects:forKeys:" of class "NSDictionary" with parameters {{MIKUBUNDLE, WINEPREFIX, MIKUROOT, myPath, myVersion}, {"MIKUBUNDLE", "WINEPREFIX", "MIKUROOT", "MIKUINSTALLERAPP", "MIKUINSTALLERVERSION"}}
	set scriptFile to POSIX path of (path to resource "backend.sh")
	set argv to {scriptFile, command} & args
	try
		set status to call method "execute:arguments:environment:message:" of CommandExec with parameters {BASH, argv, env, message}
	on error msg
		error "プログラムの実行でエラーが発生しました: " & msg
	end try
	if status < 0 then
		error "中断されました。"
	else if status is not 0 then
		error "プログラムが異常なステータスで終了しました。(" & status & ")"
	end if
	return status
end execute

-------- Installation --------

on initialize()
	set Initialized to checkAndInstallWine()
	if not Initialized then
		quit me
	else
		if NeedUpdatePrefixes then
			updatePrefixes()
			set NeedUpdatePrefixes to false
		end if
		if FileToBeExecuted is not missing value then
			executeFile(FileToBeExecuted)
		end if
		if NeedCreateAppBundles then
			createAppBundles()
			set NeedCreateAppBundles to false
		end if
		if FileToBeExecuted is not missing value then quit me
	end if
end initialize

on displayUpdateConfirm(oldVersion, newVersion)
	load nib "UpdateConfirm"
	tell window "UpdateConfirm" to center
	set oldVersionBox to box "oldVersionBox" of window "UpdateConfirm"
	set newVersionBox to box "newVersionBox" of window "UpdateConfirm"
	set content of text field "oldVersionNumber" of oldVersionBox to oldVersion
	set content of text field "newVersionNumber" of newVersionBox to newVersion
	return display window "UpdateConfirm"
end displayUpdateConfirm

on displayRemoveConfirm(oldVersion, newVersion)
	load nib "RemoveConfirm"
	tell window "RemoveConfirm" to center
	set oldVersionBox to box "oldVersionBox" of window "RemoveConfirm"
	set newVersionBox to box "newVersionBox" of window "RemoveConfirm"
	set content of text field "oldVersionNumber" of oldVersionBox to oldVersion
	set content of text field "newVersionNumber" of newVersionBox to newVersion
	--set oldVersionIcon to image view "oldVersionIcon" of oldVersionBox
	--set newVersionIcon to image view "newVersionIcon" of newVersionBox
	--set theWorkspace to call method "sharedWorkspace" of class "NSWorkspace"
	--set theImage to call method "iconForFile:" of theWorkspace with parameter thePath
	--call method "setImage:" of oldVersionIcon with parameter theImage
	return display window "RemoveConfirm"
end displayRemoveConfirm

-- After checkAndInstallWine();
-- * one of the following condition holds:
--   * We can find MikuInstaller.app in Applications folder by bundle identifier
--     search, and its Wine.bundle is the latest one, or
--   * There is the latest Wine.bundle in ~/Library/Application Support/MikuInstaller.
-- * the following variables are set:
--   * MIKUROOT is set to the absolute path to appSupport.
--   * MIKUBUNDLE is set to the absolute path to Wine.bundle.
--   * MIKUINSTALLERAPP is set to the absolute path to myself.
--   * MIKUINSTALLERVERSION is set to my version.

on checkAndInstallWine()
	set userAppSupport to path to application support folder from user domain as string
	set userAppSupport to POSIX path of userAppSupport
	set userRoot to userAppSupport & "MikuInstaller"

	set myBundle to call method "mainBundle" of class "NSBundle"
	set myBundleId to call method "bundleIdentifier" of myBundle
	set myPath to call method "bundlePath" of myBundle
	set myVersion to bundleVersion(myBundle)

	set MIKUROOT to userRoot
	set MIKUBUNDLE to missing value

	-- At first, search old version (Wine Kit) and migrate it to new layout.
	set wineKitDir to userAppSupport & "Wine Kit"
	if testExist(wineKitDir) then
		try
			set wineKitBundle to call method "bundleWithPath:" of class "NSBundle" with parameter (wineKitDir & "/Wine.bundle")
			set oldVersion to call method "objectForInfoDictionaryKey:" of theBundle with parameter "CFBundleVersion"
			set oldVersion to oldVersion
		on error msg
			try
				set oldVersion to do shell script BASH & " " & quoted form of (wineKitDir & "/script/version.sh")
			on error msg
				display dialog "インストールされているWine Kitのバージョンを識別できません。アンインストールしてください。" buttons {"終了"} default button "終了" with icon 0
				return false
			end try
		end try
		set theResult to displayUpdateConfirm(oldVersion, myVersion)
		if theResult is 0 then
			execute("Wine Kitをアンインストールしています...", "migrate", {wineKitDir})
			set NeedUpdatePrefixes to true
		else
			return false
		end if
	end if

	-- Search installed MikuInstaller.app.
	set theWorkspace to call method "sharedWorkspace" of class "NSWorkspace"
	set appPath to call method "absolutePathForAppBundleWithIdentifier:" of theWorkspace with parameter myBundleId
	try
		if appPath is myPath and appPath start with "/Volumes/" then set appPath to missing value
	on error msg
		set appPath to missing value
	end try

	try
		set appBundle to call method "bundleWithPath:" of class "NSBundle" with parameter appPath
		set appPath to call method "bundlePath" of appBundle
		set appVersion to bundleVersion(appBundle)
	on error msg
		set appBundle to missing value
		set appPath to missing value
		set appVersion to missing value
	end try

	if appPath is not missing value then
		try
			set appWinePath to call method "pathForResource:ofType:" of appBundle with parameters {"Wine","bundle"}
			set appWineBundle to call method "bundleWithPath:" of class "NSBundle" with parameter appWinePath
			set appWinePath to call method "bundlePath" of appWineBundle
			set appWineVersion to bundleVersion(appWineBundle)
		on error msg
			set appWineBundle to missing value
			set appWinePath to missing value
			set appWineVersion to missing value
		end try
	else
		set appWinePath to missing value
		set appWineBundle to missing value
		set appWineVersion to missing value
	end if

	-- Search Wine.bundle in ~/Library/Application Support.
	try
		set userWinePath to userRoot & "/Wine.bundle"
		set userWineBundle to call method "bundleWithPath:" of class "NSBundle" with parameter userWinePath
		set userWinePath to call method "bundlePath" of userWineBundle
		set userWineVersion to bundleVersion(userWineBundle)
	on error msg
		set userWineBundle to missing value
		set userWinePath to missing value
		set userWineVersion to missing value
	end try

	-- Check user Wine.bundle.
	if userWinePath is not missing value then
		set MIKUBUNDLE to userWinePath
		set wineBundleVersion to userWineVersion
		if appPath is missing value and userWineVersion < myVersion then
			-- Only Wine.bundle is installed.
			-- This means you have selected to install Wine.bundle without MikuInstaller.
			set theResult to displayUpdateConfirm(userWineVersion, myVersion)
			if theResult is 0 then
				execute("Wine.bundleをアンインストールしています...", "uninstall", {})
				set NeedUpdatePrefixes to true
				set userWinePath to missing value
			end if
		else if appWineVersion is not missing value and userWineVersion <= appWineVersion then
			-- Both MikuInstaller.app and user Wine.bundle is installed, but user Wine.bundle is older
			-- than MikuInstaller.app's one. You should remove user Wine.bundle.
			set theResult to displayRemoveConfirm(userWineVersion, appWineVersion)
			if theResult is 0 then
				execute("Wine.bundleをアンインストールしています...", "uninstall", {})
				set NeedUpdatePrefixes to true
				set userWinePath to missing value
			end if
		end if
	end if

	-- Check Wine.bundle in installed MikuInstaller.app.
	if userWinePath is missing value and appWinePath is not missing value then
		set MIKUBUNDLE to appWinePath
		set wineBundleVersion to appWineVersion
		if appWineVersion is not myVersion then
			display dialog "インストールされているMikuInstaller (" & appWineVersion & ") と実行中のMikuInstaller (" & myVersion & ") のバージョンが異なります。MikuInstallerを実行できません。" buttons {"終了"} default button "終了" with icon 0
			return false
		end if
	end if

	-- Neither Wine.bundle nor MikuInstaller.app is installed.
	-- Install Wine.bundle.
	if MIKUBUNDLE is missing value then
		set MIKUBUNDLE to userRoot & "/Wine.bundle"
		set wineBundleVersion to myVersion
		set theBundlePath to POSIX path of (path to resource "Wine.bundle")
		execute("Wine.bundleをインストールしています...", "install", {theBundlePath})
	end if

	if wineBundleVersion is myVersion then return true

	display dialog "インストールされているWine.bundle (" & userWineVersion & ") とMikuInstaller (" & myVersion & ") のバージョンが異なります。MikuInstallerを実行できません。" buttons {"終了"} default button "終了" with icon 0
	return false
end checkAndInstallWine

on checkAndCreatePrefix()
	if not testExist(winePrefixPath()) then
		execute("プレフィックス“" & MIKUPREFIX & "”を初期化しています...", "prefixcreate", {})
	end if
end checkAndCreatePrefix

on updatePrefixes()
	execute("プレフィックスとアプリケーションバンドルを更新しています...", "prefixupdate", {applicationDir()})
end updatePrefixes

on executeFile(exeFile)
	checkAndCreatePrefix()
	execute("プレフィックス“" & MIKUPREFIX & "”でWineを起動しています...", "open", {exeFile})
end executeFile

on createAppBundles()
	set dstDir to applicationDir()
	set oldAppList to do shell script "find " & quoted form of dstDir & " -type d -name '*.app' 2>/dev/null || :"

	execute("“" & MIKUPREFIX & "”のアプリケーションバンドルを作成しています...", "addapp", {dstDir})

	set newAppList to do shell script "find " & quoted form of dstDir & " -type d -name '*.app' 2>/dev/null || :"

	if newAppList is not oldAppList then
		tell application "Finder"
			open folder ((POSIX file dstDir) as string)
			activate
		end tell
	end if
end createAppBundles

on startWineCfg()
	execute("“" & MIKUPREFIX & "”のwinecfgを起動しています...", "winecfg", {})
end startWineCfg

-------- Menu --------

on menuExecute()
	tell open panel
		set can choose files to true
		set can choose directories to false
		set allows multiple selection to false
	end tell
	set theResult to display open panel in directory "~" --for file types ["exe"]
	if theResult is 1 then
		open (path names of open panel as string)
	end if
end menuExecute

on openPreferences()
	load nib "Preferences"
	tell window "Preferences"
		center
		show
		activate
	end tell
end openPreferences

-------- Preferences --------

on updateInfoPane()
	try
		set wineBundle to call method "bundleWithPath:" of class "NSBundle" with parameter MIKUBUNDLE
		set wineVersion to call method "objectForInfoDictionaryKey:" of wineBundle with parameter "CFBundleGetInfoString"
		set wineVersion to wineVersion
	on error msg
		set wineVersion to ""
	end try
	try
		set prefixPath to winePrefixPath()
	on error msg
		set prefixPath to "(未選択)"
	end try
	set content of text field "WineBundleVersion" of InfoPane to wineVersion
	set content of text field "WineBundlePath" of InfoPane to MIKUBUNDLE
	set content of text field "WinePrefixDir" of InfoPane to winePrefixDir()
	set content of text field "WinePrefix" of InfoPane to prefixPath
end updateInfoPane

on updatePrefixList()
	set tableView to table view "list" of scroll view "list" of PrefixPane

	set prefixList to {}
	try
		set prefixFolder to list folder POSIX file winePrefixDir()
	on error msg
		set prefixFolder to {}
	end try
	repeat with i in prefixFolder
		if i does not start with "." then
			set prefixList to prefixList & {i}
		end if
	end repeat

	-- determine which item will be selected.
	set selectedItem to selected row of tableView
	repeat with i from 1 to count of prefixList
		if MIKUPREFIX is item i of prefixList as string then
			set selectedItem to i
		end if
	end repeat

	if selectedItem > (count of prefixList) then
		set selectedItem to count of prefixList
	end if

	-- Setting content of tableView will emit 'selection changed'
	-- event and then call selectPrefix().
	-- We set correct selected row after setting content.
	set content of tableView to prefixList
	set selected row of tableView to selectedItem
	selectPrefix(tableView)

	set availableFlag to ((count of prefixList) > 0)
	tell button "remove" of PrefixPane to set enabled to availableFlag
	tell button "open" of PrefixPane to set enabled to availableFlag
	tell button "makeapps" of PrefixPane to set enabled to availableFlag
	tell button "winecfg" of PrefixPane to set enabled to availableFlag

	updateInfoPane()
end updatePrefixList

on selectPrefix(tableView)
	if selected row of tableView is 0 then
		set MIKUPREFIX to missing value
	else
		set newPrefix to content of data cell 1 of selected data row of tableView as string
		if newPrefix is not "" then set MIKUPREFIX to newPrefix
	end if
	updateInfoPane()
end selectPrefix

on addPrefix()
	set SheetMode to "addPrefix"
	display dialog "プレフィックス名を入力して下さい。" default answer "" attached to window "Preferences"
end addPrefix

on addPrefixResult(result)
	if button returned of result is not "OK" then return
	set newPrefixName to text returned of result

	if (count of words of ("x" & newPrefixName & "x")) is not 1 then
		display alert "不正なプレフィックス名です。" attached to window "Preferences"
		return
	end if

	set newPrefixDir to winePrefixDir() & "/" & newPrefixName
	tell application "System Events"
		set exist to exists disk item ((newPrefixDir as POSIX file) as string)
	end tell

	if exist then
		display alert "プレフィックス“" & MIKUPREFIX & "”はすでに存在しています。" attached to window "Preferences"
		return
	end if

	set MIKUPREFIX to newPrefixName
	checkAndCreatePrefix()
	updatePrefixList()
end addPrefixResult

on removePrefix()
	set SheetMode to "removePrefix"
	display dialog "プレフィックス“" & MIKUPREFIX & "”を削除します。よろしいですか？" default button 1 attached to window "Preferences"
end removePrefix

on removePrefixResult(result)
	if button returned of result is not "OK" then return
	execute("プレフィックス“" & MIKUPREFIX & "”を削除しています...", "prefixremove", {})
	updatePrefixList()
end removePrefixResult

on openPrefix()
	set prefixDir to winePrefixPath() & "/drive_c"
	tell application "Finder"
		open folder ((POSIX file prefixDir) as string)
		activate
	end tell
end openPrefix

-------- Event Handlers --------

on awake from nib theObject
	if name of theObject is "commandExecutePanelController" then
		set CommandExec to theObject
	else if name of theObject is "Preference WINEPREFIX" then
		set PrefixPane to theObject
	else if name of theObject is "Preference Info" then
		set InfoPane to theObject
	end if
end awake from nib

on open theFiles
	if (count of theFiles) is not 1 then return
	if not Initialized then
		set FileToBeExecuted to POSIX path of item 1 of theFiles
		set NeedCreateAppBundles to true
	else
		executeFile(POSIX path of item 1 of theFiles)
		createAppBundles()
	end if
end open

on idle theObject
	if Initialized then return
	try
		initialize()
	on error msg
		activate
		display dialog ("エラー: " & msg) buttons {"終了"} default button "終了" with icon 0
		quit me
	end try
end idle

on will open theObject
	if name of theObject is "Preferences" then
		updatePrefixList()
	end if
end will open

on choose menu item theObject
	if not Initialized then return
	if name of theObject is "Menu execute" then
		menuExecute()
	else if name of theObject is "Menu preferences" then
		openPreferences()
	end if
end choose menu item

on selected tab view item theObject tab view item tabViewItem
	if tabViewItem is PrefixPane then
		updatePrefixList()
	end if
end selected tab view item

on selection changed theObject
	if theObject is table view "list" of scroll view "list" of PrefixPane then
		selectPrefix(theObject)
	end if
end selection changed

on clicked theObject
	if name of theObject is "OK" then
		close panel (window of theObject)
	else if name of theObject is "Cancel" then
		close panel (window of theObject) with result 1
	else if theObject is button "add" of PrefixPane then
		addPrefix()
	else if theObject is button "remove" of PrefixPane then
		removePrefix()
	else if theObject is button "open" of PrefixPane then
		openPrefix()
	else if theObject is button "makeapps" of PrefixPane then
		createAppBundles()
	else if theObject is button "winecfg" of PrefixPane then
		startWineCfg()
	end if
end clicked

on dialog ended theObject with reply withReply
	set mode to SheetMode
	set SheetMode to missing value
	if mode is "addPrefix" then
		addPrefixResult(withReply)
	else if mode is "removePrefix" then
		removePrefixResult(withReply)
	end if
end dialog ended


-- Local Variables:
-- tab-width: 4
-- End:
