/*
 * CommandExecutePanelController.m
 *
 * Copyright (C) 2008 MikuInstaller Project. All rights reserved.
 * http://mikuinstaller.sourceforge.jp/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "CommandExecutePanelController.h"

@implementation CommandExecutePanelController

- (id)init
{
	return [super initWithWindowNibName:@"CommandExecutePanel"];
}

- (void)awakeFromNib
{
	[[self window] center];
}

- (IBAction)toggleDisclosure:(id)sender
{
	NSRect rect = [[self window] frame];
	NSRect detailRect = [detailView frame];
	NSRect newRect;
	BOOL hidden = [sender state] != NSOnState;

	if (!hidden) {
		newRect = NSMakeRect(rect.origin.x,
				     rect.origin.y - detailRect.size.height,
				     rect.size.width,
				     rect.size.height + detailRect.size.height);
	} else {
		newRect = NSMakeRect(rect.origin.x,
				     rect.origin.y + detailRect.size.height,
				     rect.size.width,
				     rect.size.height - detailRect.size.height);
	}

	[[self window] setFrame:newRect display:YES animate:YES];
}

- (void)appendOutput:(NSData *)data
{
	static const NSStringEncoding encodings[] = {
		NSUTF8StringEncoding,
		NSShiftJISStringEncoding,
		NSWindowsCP1251StringEncoding,
		NSWindowsCP1252StringEncoding,
		NSWindowsCP1253StringEncoding,
		NSWindowsCP1254StringEncoding,
		NSWindowsCP1250StringEncoding,
		NSJapaneseEUCStringEncoding,
		NSNonLossyASCIIStringEncoding,
	};

	NSString *s = nil;
	NSAttributedString *as;
	unsigned int i;

	fwrite([data bytes], [data length], 1, stderr); // Logging

	for (i = 0; !s && i < sizeof(encodings)/sizeof(NSStringEncoding); i++) {
		s = [[[NSString alloc] initWithData:data encoding:encodings[i]]
			    autorelease];
	}
	if (!s)
		s = @"<BINARY>";

	as = [[[NSAttributedString alloc] initWithString:s] autorelease];
	[[outputView textStorage] appendAttributedString:as];
	[outputView scrollRangeToVisible:
			    NSMakeRange([[outputView string] length], 0)];
}

- (void)readData:(NSNotification *)notification
{
	NSData *data = [[notification userInfo]
			       objectForKey:NSFileHandleNotificationDataItem];

	if ([data length] == 0) {
		// FIXME: On Tiger, we cannot receive
		// NSTaskDidTerminateNotification when we are in modal
		// run loop. We need to terminate the process here in
		// order to work around this problem.
		if (![currentTask isRunning]) {
			[NSApp abortModal];
			[progressIndicator stopAnimation:self];
			[self close];
		}
		return;
	}

	[self appendOutput:data];
	[[notification object]
		readInBackgroundAndNotifyForModes:
			[NSArray arrayWithObject:NSModalPanelRunLoopMode]];
}

- (void)taskTerminated:(NSNotification *)notification
{
	NSLog([NSString stringWithFormat:@"process terminated status at %d",
			[[notification object] terminationStatus]]);

	@try {
		NSData *data = [[[currentTask standardOutput]
					fileHandleForReading]
				       readDataToEndOfFile];
		[self appendOutput:data];
	} @finally {
		// Terminating modal session is essential.
		[NSApp abortModal];
		[progressIndicator stopAnimation:self];
		[self close];
	}
}

- (void)killByUser:(id)sender
{
	[NSApp stopModal];
	[progressIndicator stopAnimation:self];
	[self close];
}

- (int)execute:(NSString *)command
     arguments:(NSArray *)args
   environment:(NSDictionary *)env
       message:(NSString *)msg
{
	int result;

	if (currentTask) {
		@throw [NSException exceptionWithName:
					    @"CommandExecutePanelException"
				    reason:
					    @"currentTask is not nil"
				    userInfo:nil];
	}

	NSPipe *pipe = [NSPipe pipe];
	NSTask *task = [[[NSTask alloc] init] autorelease];
	[task setStandardInput:[NSFileHandle fileHandleWithNullDevice]];
	[task setStandardOutput:pipe];
	[task setStandardError:pipe];

	[task setLaunchPath:command];
	if (args) {
		[task setArguments:args];
	}
	if (env) {
		NSMutableDictionary *newEnv =
			[NSMutableDictionary dictionaryWithCapacity:10];
		[newEnv setDictionary:
				[[NSProcessInfo processInfo] environment]];
		[newEnv addEntriesFromDictionary:env];
		[task setEnvironment:newEnv];
	}

	[[pipe fileHandleForReading]
		readInBackgroundAndNotifyForModes:
			[NSArray arrayWithObject:NSModalPanelRunLoopMode]];

	NSLog([NSString stringWithFormat:@"executing command %@", command]);

	[outputView setString:@""];
	[[self window] makeKeyAndOrderFront:nil];
	[progressIndicator startAnimation:self];
	[messageField setStringValue:(msg ? msg : @"")];

	[[NSNotificationCenter defaultCenter]
		addObserver:self
		selector:@selector(taskTerminated:)
		name:NSTaskDidTerminateNotification
		object:task];
	[[NSNotificationCenter defaultCenter]
		addObserver:self
		selector:@selector(readData:)
		name:NSFileHandleReadCompletionNotification
		object:[pipe fileHandleForReading]];

	@try {
		NSLog(@"-------- start --------");
		[task launch];
		currentTask = task;
		@try {
			result = [NSApp runModalForWindow:[self window]];
		} @finally {
			currentTask = nil;
			[task terminate];
		}
	} @catch (NSException *e) {
		[self close];
		@throw;
	} @finally {
		NSLog(@"-------- end --------");
		[[NSNotificationCenter defaultCenter] removeObserver:self];
	}

	if (result == NSRunAbortedResponse)
		result = [task terminationStatus];
	NSLog([NSString stringWithFormat:@"exit at status %d", result]);

	return result;
}

- (int)execute:(NSString *)command
     arguments:(NSArray *)args
   environment:(NSDictionary *)env
{
	return [self execute:command arguments:args environment:env
		     message:@""];
}

- (int)execute:(NSString *)command
     arguments:(NSArray *)args
{
	return [self execute:command arguments:args environment:nil
		     message:@""];
}

- (int)execute:(NSString *)command
{
	return [self execute:command arguments:nil environment:nil
		     message:@""];
}

@end
