#
# = コンソール及びファイルにログを出力するためのクラス
# [Author] baku <micle@tohka.info>
# [Version] 0.9.6.1 (2007/05/06)
# [License] {Ruby License}[http://www.ruby-lang.org/ja/LICENSE.txt]
#
# $Id$
#
class Log
	attr_reader :dir, :file, :date
	#
	# == 初期化用メソッド
	# === 引数
	# * なし
	# === 返り値
	# * self
	# === 概要
	# ログが出力されるディレクトリの作成他、各値の初期化。
	#
	def initialize
		@date = Time.now.strftime('%Y%m%d%H%M%S')
		Dir.mkdir("./log") unless FileTest.directory?("./log")
		Dir.mkdir(@dir = "./log/log_#{@date}")
		@file = "#{@dir}/log.txt"
		@index = -1
		message("Micle #{$version}")
		message("="*64)
		message("ログをファイルに出力します。 (#{@file})")
	end

	#
	# == エラー出力用メソッド
	# === 引数
	# * (String) msg
	# * (String) mode=''
	# === 返り値
	# * (SystemExit)
	# === 概要
	# 致命的なエラーが起こったときに呼び出してメッセージを出力させます。このメソッドはプログラムを終了させます。
	#
	# 引数に関してはmessageを参照してください。
	#
	def error(msg, mode='')
		message("erroe: #{msg}", mode)
		exit
	end

	#
	# == 警告出力用メソッド
	# === 引数
	# * (String) msg
	# * (String) mode=''
	# === 返り値
	# * nil
	# === 概要
	# 比較的致命的な出来事が起こったときに呼び出してメッセージを出力させます。エラーまではいかなくとも、このまま処理を継続した場合、支障が出るような事態を想定しています。
	#
	# 引数に関してはmessageを参照してください。
	#
	def alert(msg, mode='')
		message("alert: #{msg}", mode)
	end

	#
	# == 警告出力メソッド
	# === 引数
	# * (String) msg
	# * (String) mode=''
	# === 返り値
	# * nil
	# === 概要
	# 致命的とはいかなくとも、想定外の事態があったときに呼び出してメッセージを出力します。どちらかといえば、warningよりもcautionに近いかもしれません。
	#
	# 引数に関してはmessageを参照してください。
	#
	def warning(msg, mode='')
		message("warning: #{msg}", mode)
	end

	#
	# == メッセージ出力メソッド
	# === 引数
	# * (String) msg
	# * (String) mode=''
	# === 返り値
	# * nil
	# === 概要
	# コンソール及びログファイル(もしくはそのどちらか)にメッセージを出力します。
	#
	# [msg] 出力するメッセージ本文
	# [mode] 出力する媒体('c'でコンソールのみ、'f'でファイルのみ、''で両方)
	#
	def message(msg, mode='')
		msg = "#{Time.now.strftime('%H:%M:%S')} #{msg}"
		fh = open(@file, 'a')
		if mode != 'c'
			fh.puts msg.tosjis
		end
		if mode != 'f'
			puts msg.tosjis
		end
		fh.close
		nil
	end

	#
	# == データダンプ用メソッド
	# === 引数
	# * (String) msg
	# * (String) data
	# * (Boolean) pretty=false
	# * (String) mode=''
	# === 返り値
	# * nil
	# === 概要
	# データを出力します。
	#
	# [msg] ダンプするときに出力するメッセージ
	# [data] 出力するデータ
	# [pretty] 装飾して出力するかどうか。trueならYAML形式で出力します。
	#          配列やハッシュなどのときに有効です。falseなら
	#          そのまま出力します。HTMLなどの文字列のとき有効
	# [mode] 'c'ならコンソールのみ、'f'ならファイルのみに出力
	#
	def dump(msg, data, pretty=false, mode='')
		file = "#{@dir}/_dump_#{(@index += 1)}"
		file += pretty ? ".yml" : ".txt"
		fh = open(file, 'w')
		message("dump: #{msg} (#{file})", mode)
		if pretty
			YAML.dump(data, fh)
		else
			fh.puts data.inspect.gsub(/\<NL\>/, "\r\n")
		end
		fh.close
	end
end
