#
# = コミュニティのリストを管理するクラス
# [Author] baku <micle@tohka.info>
# [Version] 0.9.6.1 (2007/05/06)
# [License] {Ruby License}[http://www.ruby-lang.org/ja/LICENSE.txt]
#
# $Id$
#
class CommunitiesList
	attr_accessor :list
	#
	# == 初期化メソッド
	# === 引数
	# * なし
	# === 返り値
	# * self
	# === 概要
	# @list及び@diffを初期化。
	#
	def initialize
		@list = Array.new
		@diff = Array.new
		self
	end

	#
	# == idで探索しインデックスを返すメソッド
	# === 引数
	# * (String) id
	# === 返り値
	# * (Integer) index / nil
	# === 概要
	# @listの各要素はCommunityクラスのオブジェクトである。
	# その各要素のCommunity#idを探索し、インデックスを返す。
	# idがみつからなければ、nilを返す。
	#
	def index(id)
		index = nil
		@list.each_index do |i|
			if @list[i].id == id
				index = i
				break
			end
		end
		index
	end

	#
	# == リストにコミュニティを破壊的に追加するメソッド
	# === 引数
	# * (Community) community
	# === 返り値
	# * self
	# === 概要
	# 引数で与えられたコミュニティをリストに破壊的に追加する。
	# 既にリストに存在するコミュニティならば上書きする。
	# (存在するかはid比較)
	#
	def add!(community)
		if community.instance_of?(Community)
			index = index(community.id)
			if index.nil?
				@list << community.dup
			else
				community.to_hash.each do |key, value|
					if value.kind_of?(String)
						@list[index][key] = value.dup unless value.empty?
					elsif value.kind_of?(Integer)
						@list[index][key] = value unless value.zero?
					end
				end
			end	
		end
		self
	end

	#
	# == リストに他のリストを破壊的に追加(マージ)するメソッド
	# === 引数
	# * (CommunitiesList) list
	# === 返り値
	# * self
	# === 概要
	# 引数で与えられたコミュニティリストを破壊的に追加する。
	# 既にリストに存在するコミュニティが含まれていたならば、
	# そのコミュニティは上書きする。(存在するかはid比較)
	#
	def merge!(list)
		if list.instance_of?(CommunitiesList)
			list.each do |community|
				index = index(community.id)
				if index.nil?
					@list << community.dup
				else
					community.to_hash.each do |key, value|
						if value.kind_of?(String)
							@list[index][key] = value.dup unless value.empty?
						elsif value.kind_of?(Integer)
							@list[index][key] = value unless value.zero?
						end
					end
				end
			end
			self
		end
	end

	#
	# == リストから廃墟コミュニティを破壊的に削除するメソッド
	# === 引数
	# * (Integer) option=0
	# === 返り値
	# * self
	# === 概要
	# 削除され、メンバーが0人になったコミュニティも
	# コミュニティ検索の結果に含められる。
	# そういった「廃墟」を取り除くためのメソッド。
	# また、廃墟コミュニティはidとメンバー数が0人という
	# 情報しか得られず、CommunityPage#parseもうまくいかない。
	# そのため、set_range!などで日付を参照することは
	# できないので、あらかじめdelete_ruin!(1)をする必要がある。
	#
	# [option = 0] 何も削除しない
	# [option = 1] 廃墟コミュニティを削除する
	# [option = 2] 廃墟コミュニティ以外を削除する
	#
	def delete_ruin!(option=0)
		case option
		when 0
			nil
		when 1
			@list.delete_if do |elm|
				elm.ruin?
			end
		when 2
			@list.delete_if do |elm|
				!elm.ruin?
			end
		end
		$log.dump("CommunitiesList#delete_ruin!", @list, true, 'f')
		self
	end

	#
	# == 範囲を指定し、範囲外の要素を破壊的に取り除くメソッド
	# === 引数
	# * (String) range='-'
	# === 返り値
	# * self
	# === 概要
	# 指定された範囲のみを残すメソッド。範囲の書式はハイフンで
	# 区切った文字列。id及び作成日で範囲を取れる。記述例を示す。
	#
	# [id:10000-id:100000]
	#   idが10000から100000に該当するコミュニティのみ残す
	# [id:10000-date:2007/03/31]
	#   idが10000以上かつ、2007年3月末までに作成されたもののみ
	# [date:2007/01/01-]
	#   2007年以降に作成されたものすべて
	# [-id:100000]
	#   idが100000以下のものすべて
	# [id:10000-id:100]
	#   idが10000以上100未満のものはありえないので、何も削除しない
	# [-]
	#   すべての範囲(何も削除しない)
	#
	# ただし、作成日で範囲をとるときは、自動的に
	# 廃墟コミュニティは削除され、検索結果に入らない。
	# なぜならば、廃墟は日付のデータも既に存在しないので、
	# 日付で範囲を絞ることはできない。
	#
	# また、実際に削除するのは、delete_elements!。
	#
	def set_range!(range='-')
		from, to = range.split('-', 2)
		delete_elements!(from, :lt) unless from.strip.empty?
		delete_elements!(to, :gt) unless to.strip.empty?
		$log.dump("CommunitiesList#set_range!", @list, true, 'f')
		self
	end

	#
	# == 基点から条件に当てはまる要素を破壊的に削除するメソッド
	# === 引数
	# * (String) base
	# * (Symbol) cond
	# === 返り値
	# * self
	# === 概要
	# 基点(base)から条件(cond)に適した要素を削除する。
	# baseの書式はset_range!の範囲の書式に登場した、
	# id:10000やdate:2007/01/01などの、ハイフンの前後の部分。
	#
	# [cond = :el] 基点(base)よりも等しいか小さい(equal less than)
	# [cond = :lt] 基点(base)よりも小さい(less than)
	# [cond = :eg] 基点(base)よりも等しいか大きい(equal grater than)
	# [cond = :gt] 基点(base)よりも大きい(grater than)
	#
	# 基点と各要素との差をとり、その差によって条件を満たすかを
	# 判断する。差をとっているのはget_difference。
	#
	def delete_elements!(base, cond)
		base, mode = analize_base(base)
		delete_ruin!(1) if mode == :date
		sort!
		get_difference(0, @list.size-1, mode, base, cond)
		@list.delete_if do |elm|
			case cond
			when :el
				@diff[index(elm.id)] <= 0
			when :lt
				@diff[index(elm.id)] < 0
			when :eg
				@diff[index(elm.id)] >= 0
			when :gt
				@diff[index(elm.id)] > 0
			end
		end
		@diff.clear
	end

	#
	# == 対象と各要素との差をとるメソッド
	# === 引数
	# * (Integer) first
	# * (Integer) last
	# * (Symbol) mode
	# * (Integer) target
	# * (Symbol) cond
	# === 返り値
	# * self
	# === 概要
	# firstからlastまでのインデックスをもつ要素の値と対象との
	# 差をとって@diffに格納する。要素の値といっても、正確には
	# modeによってidを比較するか日付を比較するか変化する。
	# targetは数値化され、日付の場合は20070101のような整数になっている。
	# condの説明はdelete_elements!を参照してください。
	#
	# このメソッドは二分探索のアルゴリズムをもとに組まれており、
	# 再帰的に差を求めます。また、あらかじめ昇順ソートが必要です。
	#
	# 差といっても、正確な差は求めません。ソートされているので、
	# (less thanのときなどは)比較した要素の方が小さければ、
	# その要素より若いインデックスの要素は、明らかに値は小さくなります。
	# そういったときは、適当な負の値を代入しています。
	#
	def get_difference(first, last, mode, target, cond)
		#
		# 範囲が2以下の場合、すべて調べつくしたことになる
		#
		if (last-first).abs <= 1
			if first == last
				@diff[first] = get_community_data(mode, first) - target
			else
				@diff[first] = get_community_data(mode, first) - target
				@diff[last] = get_community_data(mode, last) - target
			end
			return self
		#
		# 範囲が3以上の場合、中央値をとって比較
		#
		else
			case cond
			when :el, :lt
				median = ((first + last) / 2.0).ceil
			when :eg, :gt
				median = ((first + last) / 2.0).floor
			end
			yardstick = get_community_data(mode, median)
			#
			# 比較した要素の方が大きければ...
			#
			if target < yardstick
				@diff[median] = yardstick - target
				@diff.fill(@diff[median]+1, median+1..last)
				get_difference(first, median-1, mode, target, cond)
			#
			# 比較した要素の方が小さければ...
			#
			elsif target > yardstick
				@diff[median] = yardstick - target
				@diff.fill(@diff[median]-1, first..median-1)
				get_difference(median+1, last, mode, target, cond)
			#
			# 比較した要素と等しければ...
			#
			else
				@diff[median] = 0
				case cond
				#
				# :el, :gtのとき、中央から+1を見ていき、
				# どこで対象より大きくなるかを調べる。
				#
				when :el, :gt
					@diff.fill(-1, first..median-1)
					offset = 1
					if median+offset <= last
						yardstick = get_community_data(mode, median+offset)
						while target == yardstick
							@diff[median+offset] = 0
							offset += 1
							break unless median+offset <= last
							yardstick = get_community_data(mode, median+offset)
						end
						@diff.fill(1, median+offset..last)
					end
					return self
				#
				# :lt, :egのとき、中央から-1を見ていき
				# どこで対象より小さくなるかを調べる。
				#
				when :lt, :eg
					@diff.fill(1, median+1..last)
					offset = 1
					if first <= median-offset
						yardstick = get_community_data(mode, median-offset)
						while target == yardstick
							@diff[median-offset] = 0
							offset += 1
							break unless first <= median-offset
							yardstick = get_community_data(mode, median-offset)
						end
						@diff.fill(-1, first..median-offset)
					end
					return self
				end
			end
		end
	end

	#
	# == コミュニティから特定の値を取り出すメソッド
	# === 引数
	# * (Symbol) mode
	# * (Integer) index
	# === 返り値
	# * (Integer)
	# === 概要
	# 与えられたインデックスのCommunityクラスのオブジェクトに
	# アクセスして、id及び作成日のデータを整数に変換し返す。
	# この時点(SearchPages#parse)ではidは既知だが、日付は
	# 得られていないため、CommunityPageにアクセスして得る。
	#
	def get_community_data(mode, index)
		if mode == :id
			@list[index].id.to_i
		elsif mode == :date
			communitypage = CommunityPage.new(@list[index].id)
			communitypage.community.since.delete('年月日').to_i
		end
	end

	#
	# == 基点の書式を解析するメソッド
	# === 引数
	# * (String) base
	# === 返り値
	# * [(Integer) base, (Symbol) mode]
	# === 概要
	# delete_elements!のbaseの書式を解析する。
	#
	def analize_base(base)
		base.strip
		if base =~ /^id:(\d+)$/
			mode = :id
			base = $1.to_i
		elsif base =~ /^date:(\d+)\/(\d+)\/(\d+)$/
			mode = :date
			base = [$1.to_i, $2.to_i, $3.to_i]
			base[0] += 2000 if base[0] < 100
			base = sprintf("%4d%02d%02d", base[0], base[1], base[2]).to_i
		end
		[base, mode]
	end

	#
	# == 破壊的にidで昇順ソートするメソッド
	# === 引数
	# * なし
	# === 返り値
	# * self
	# === 概要
	# get_differenceのアルゴリズム上、事前に昇順ソートをする必要がある。
	# idでソートすると、作成日でもソートされているため、日付で
	# 比較する際にも問題はない。
	#
	def sort!
		@list.compact!
		@list.sort! do |a,b|
			a.id.to_i <=> b.id.to_i
		end
		self
	end

	#
	# == 未稿
	def get_detail_data
		sort!
		@list.each do |community|
			unless community.ruin?
				$log.message("id:#{community.id}の情報を取得しています。")
				communitypage = CommunityPage.new(community.id)
				merge!(communitypage.list)
			end
		end
		$log.dump("CommunitiesList#get_detail_data", @list, true, 'f')
		self
	end

	#
	# == 出力用メソッド
	# === 引数
	# * (String) key
	# === 返り値
	# * self
	# === 概要
	# HTML及びCSVで結果を出力する。
	#
	def output(key)
		$log.dump("CommunitiesList#output", @list, true, 'f')
		Dir.mkdir("./result") unless FileTest.directory?("./result")
		if $config[:decoded?] > 0
			file = CGI.unescape(key).gsub(/[\s[:punct:]]/, '_').tosjis
		else
			file.gsub!(/[\s[:punct:]]/, '_')
		end
		sort!
		file = "./result/#{file}_#{$log.date}"
		#
		fh = open("#{file}.html", 'w')
		html = Array.new
		html << "<?xml version=\"1.0\" encoding=\"#{$config[:charset]}\"?>"
		html << "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\""
		html << "\t\"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">"
		html << "<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"ja\">"
		html << "\t<head>"
		html << "\t\t<meta http-equiv=\"Content-Type\" content=\"application/xhtml+xml; charset=#{$config[:charset]}\" />"
		html << "\t\t<style type=\"text/css\">"
		html << "\t\t\t<!--"
		html << "\t\t\tbody {"
		html << "\t\t\t\tbackground-color: #ff4500;"
		html << "\t\t\t\tfont-size: 90%;"
		html << "\t\t\t\tline-height: 115%;"
		html << "\t\t\t}"
		html << "\t\t\t#meta {"
		html << "\t\t\t\tmargin: 3%;"
		html << "\t\t\t\tmargin-bottom: 2em;"
		html << "\t\t\t\tpadding: 1ex;"
		html << "\t\t\t\tpadding-left: 5em;"
		html << "\t\t\t\tborder: 6px double #000000;"
		html << "\t\t\t\tbackground-color: #ffdab9;"
		html << "\t\t\t}"
		html << "\t\t\t#meta dt {"
		html << "\t\t\t\tfloat: left;"
		html << "\t\t\t\tclear: left;"
		html << "\t\t\t\twidth: 12em"
		html << "\t\t\t\tpadding: 2px;"
		html << "\t\t\t}"
		html << "\t\t\t#meta dd {"
		html << "\t\t\t\tmargin-left: 12em;"
		html << "\t\t\t\tpadding: 2px;"
		html << "\t\t\t}"
		html << "\t\t\ttable, td, th {"
		html << "\t\t\t\tborder: 1px #000000 solid;"
		html << "\t\t\t\tborder-collapse: collapse;"
		html << "\t\t\t\tbackground-color: #ffdab9;"
		html << "\t\t\t}"
		html << "\t\t\ttable {"
		html << "\t\t\t\twidth: 94%;"
		html << "\t\t\t\tmargin-left: 3%;"
		html << "\t\t\t\tmargin-bottom: 10em;"
		html << "\t\t\t}"
		html << "\t\t\ttd, th {"
		html << "\t\t\t\tpadding: 4px;"
		html << "\t\t\t}"
		html << "\t\t\tth {"
		html << "\t\t\t\ttext-align: center;"
		html << "\t\t\t\tfont-weight: bold;"
		html << "\t\t\t}"
		html << "\t\t\t#index .id, #index .member, #index .since {"
		html << "\t\t\t\ttext-align: right;"
		html << "\t\t\t}"
		html << "\t\t\t.community {"
		html << "\t\t\t\tmargin: 3%;"
		html << "\t\t\t\tmargin-bottom: 5em;"
		html << "\t\t\t\tpadding: 1ex;"
		html << "\t\t\t\tbackground-color: #ffdab9;"
		html << "\t\t\t}"
		html << "\t\t\t.community dt {"
		html << "\t\t\t\tfloat: left;"
		html << "\t\t\t\tclear: left;"
		html << "\t\t\t\twidth: 6em;"
		html << "\t\t\t\tpadding: 4px;"
		html << "\t\t\t}"
		html << "\t\t\t.community dd {"
		html << "\t\t\t\tmargin-left: 6em;"
		html << "\t\t\t\tpadding: 4px;"
		html << "\t\t\t}"
		html << "\t\t\t#footer {"
		html << "\t\t\t\tcolor: #ffdab9;"
		html << "\t\t\t}"
		html << "\t\t\t-->"
		html << "\t\t</style>"
		html << "\t\t<title>「#{CGI.escapeHTML(CGI.unescape(key))}」" +
			"の検索結果</title>"
		html << "\t</head>"
		html << "\t<body>"
		html << "\t\t<div id=\"meta\">"
		html << "\t\t\t<dl>"
		html << "\t\t\t\t<dt>検索キーワード</dt>"
		html << "\t\t\t\t<dd>#{CGI.escapeHTML(CGI.unescape(key))}</dd>"
		html << "\t\t\t\t<dt>検索範囲</dt>"
		html << "\t\t\t\t<dd>#{$config[:range]}</dd>"
		html << "\t\t\t\t<dt>削除済みコミュニティ</dt>"
		case $config[:ruin]
		when 0
			html << "\t\t\t\t<dd>検索結果に含める</dd>"
		when 1
			html << "\t\t\t\t<dd>検索結果から除く</dd>"
		when 2
			html << "\t\t\t\t<dd>削除済みコミュニティのみ扱う</dd>"
		end
		html << "\t\t\t\t<dt>説明文文字数制限</dt>"
		if $config[:desc] > 0
			html << "\t\t\t\t<dd>#{$config[:desc]}バイト</dd>"
		else
			html << "\t\t\t\t<dd>なし</dd>"
		end
		html << "\t\t\t\t<dt>検索結果</dt>"
		html << "\t\t\t\t<dd>#{@list.size}</dd>"
		html << "\t\t\t</dl>"
		html << "\t\t</div>"
		html << "\t\t<table id=\"index\">"
		html << "\t\t\t<tr>"
		html << "\t\t\t\t<th>コミュニティID</th>"
		html << "\t\t\t\t<th>コミュニティ名</th>"
		html << "\t\t\t\t<th>人数</th>"
		html << "\t\t\t\t<th>作成日</th>"
		html << "\t\t\t</tr>"
		@list.each do |elm|
			html << "\t\t\t<tr>"
			html << "\t\t\t\t<td class=\"id\"><a href=\"#id#{elm.id}\">#{elm.id}</a></td>"
			html << "\t\t\t\t<td class=\"name\">#{elm.ruin? ? '&nbsp;' : elm.name}</td>"
			html << "\t\t\t\t<td class=\"member\">#{elm.member}人</td>"
			html << "\t\t\t\t<td class=\"since\">#{elm.ruin? ? '&nbsp;' : elm.since}</td>"
			html << "\t\t\t</tr>"
		end
		html << "\t\t</table>"
		@list.each do |elm|
			html << "\t\t<div class=\"community\" id=\"id#{elm.id}\">"
			html << "\t\t\t<dl>"
			html << "\t\t\t\t<dt>ID</dt>"
			html << "\t\t\t\t<dd>#{elm.id}</dd>"
			html << "\t\t\t\t<dt>名称</dt>"
			if elm.ruin?
				html << "\t\t\t\t<dd>&nbsp;</a></dd>"
			else
				html << "\t\t\t\t<dd><a href=\"#{elm.uri}\">#{elm.name}</a></dd>"
			end
			html << "\t\t\t\t<dt>人数</dt>"
			html << "\t\t\t\t<dd>#{elm.member}人</dd>"
			html << "\t\t\t\t<dt>開設日</dt>"
			if elm.ruin?
				html << "\t\t\t\t<dd>&nbsp;</dd>"
			else
				html << "\t\t\t\t<dd>#{elm.since} (#{elm.span}日)</dd>"
			end
			html << "\t\t\t\t<dt>管理人</dt>"
			if elm.ruin?
				html << "\t\t\t\t<dd>&nbsp;</dd>"
			elsif elm.administer_id.empty?
				html << "\t\t\t\t<dd>#{elm.administer}</dd>"
			else
				html << "\t\t\t\t<dd><a href=\"http://mixi.jp/view_friend.pl?" +
						"id=#{elm.administer_id}\">#{elm.administer}</a></dd>"
			end
			html << "\t\t\t\t<dt>カテゴリー</dt>"
			if elm.ruin?
				html << "\t\t\t\t<dd>&nbsp;</dd>"
			else
				html << "\t\t\t\t<dd>#{elm.category}</dd>"
			end
			html << "\t\t\t\t<dt>公開レベル</dt>"
			if elm.ruin?
				html << "\t\t\t\t<dd>&nbsp;</dd>"
			else
				html << "\t\t\t\t<dd>#{elm.private}</dd>"
			end
			html << "\t\t\t\t<dt>説明文</dt>"
			if elm.ruin?
				html << "\t\t\t\t<dd>&nbsp;</dd>"
			else
				desc = elm.description.gsub(/\<[^>]*\>/, '')
				if $config[:desc] > 0
					desc = desc.jleft($config[:desc])
				end
				desc = CGI.escapeHTML(CGI.unescapeHTML(desc))
				html << "\t\t\t\t<dd>#{desc} </dd>"
			end
			html << "\t\t\t</dl>"
			html << "\t\t\t<p><a href=\"#index\">目次へ</a></p>"
			html << "\t\t</div>"
		end
		html << "\t\t<div id=\"footer\">"
		html << "\t\t\t<p>generated by Micle #{$version}.</p>"
		html << "\t\t</div>"
		html << "\t</body>"
		html << "</html>"
		html.each do |line|
			fh.puts line.kconv($config[:charsetcode])
		end
		fh.close
		$log.message("HTMLファイルを出力しました。 (#{file}.html)")
		fh = open("#{file}.csv", 'w')
		@list.each do |community|
			fh.puts community.to_csv.tosjis
		end
		fh.close
		$log.message("CSVファイルを出力しました。 (#{file}.csv)")
		nil
	end

	#
	# == 配列っぽくアクセス用メソッド
	# === 引数
	# * (Integer) index
	# === 返り値
	# * (Community)
	# === 概要
	# @listの内容にアクセスするメソッド。
	#   list = CommunitiesList.new
	#   list.add!(community)
	#   list[0] #=> community
	# のようなことができる。
	#
	def [](index)
		@list[index]
	end
	
	#
	# == Array化メソッド
	# === 引数
	# * なし
	# === 返り値
	# * (Array)
	# === 概要
	# リストを配列として返す。各要素のコミュニティはHash化される。
	#
	def to_a
		array = Array.new
		@list.each do |elm|
			array << elm.to_hash
		end
		array
	end

	#
	# == イテレータ
	# === 引数
	# * (Community) community
	# === 返り値
	# * self
	# === 概要
	# communityを引数としてブロックを評価します。
	#
	def each
		index = 0
		while @list[index].instance_of?(Community)
			community = @list[index]
			yield community
			index += 1
		end
		self
	end

	#
	# == 未稿
	def kconv!(out_code, in_code=Kconv::AUTO)
		each do |community|
			community.kconv(out_code, in_code)
		end
		self
	end
	def tojis!
		kconv!(Kconv::JIS)
	end
	def toeuc!
		kconv!(Kconv::EUC)
	end
	def tosjis!
		kconv!(Kconv::SJIS)
	end
	def toutf8
		kconv!(Kconv::UTF8)
	end
end
