{ ---------------------------------------------------------------------------------------------- }
{ MHash-384 for Delphi/Pascal                                                                    }
{ Copyright(c) 2016-2018 LoRd_MuldeR <mulder2@gmx.de>                                            }
{                                                                                                }
{ Permission is hereby granted, free of charge, to any person obtaining a copy of this software  }
{ and associated documentation files (the "Software"), to deal in the Software without           }
{ restriction, including without limitation the rights to use, copy, modify, merge, publish,     }
{ distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the  }
{ Software is furnished to do so, subject to the following conditions:                           }
{                                                                                                }
{ The above copyright notice and this permission notice shall be included in all copies or       }
{ substantial portions of the Software.                                                          }
{                                                                                                }
{ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING  }
{ BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND     }
{ NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,   }
{ DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, }
{ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.        }
{ ---------------------------------------------------------------------------------------------- }

unit MHash384;

{ =========================================================================== }
{ INTERFACE                                                                   }
{ =========================================================================== }

interface

uses
  Classes, SyncObjs;

const
  MHASH384_LEN = 48;

type
  TByteArray = array of byte;
  TDigestBuf = array[0..MHASH384_LEN-1] of Byte;
  TProgress = procedure(total: Integer; current: Integer; progress: Integer) of object;

type
  TMHash384 = Class
  public
    constructor Create();
    procedure Update(const input: Byte); overload;
    procedure Update(const input: TByteArray); overload;
    procedure Update(const input: TByteArray; const offset: LongWord; const len: LongWord); overload;
    procedure Update(const input: String); overload;
    procedure Update(const input: TStream); overload;
    function Update(const input: TStream; const count: Cardinal): Cardinal; overload;
    procedure Result(var output: TByteArray);
    procedure Reset();
    class procedure GetVersion(var major: Word; var minor: Word; var patch: Word);
    class procedure SelfTest(const callback: TProgress; const abortFlag: TSimpleEvent);
  private
    m_digest: array[0..1] of TDigestBuf;
    m_rnd: Cardinal;
  end;

{ =========================================================================== }
{ IMPLEMENTATION                                                              }
{ =========================================================================== }

implementation

uses
  SysUtils, Math;

type
  PDigestBuf = ^TDigestBuf;

{ ---------------------------------------------- }
{ Constants                                      }
{ ---------------------------------------------- }

const
  VERSION_MAJOR: Word = 1;
  VERSION_MINOR: Word = 2;
  VERSION_PATCH: Word = 1;

const
  TABLE_INI: array[0..1] of TDigestBuf =
  (
    ( $EF, $DC, $B6, $6D, $4E, $CC, $1A, $A4, $AF, $9B, $3F, $20, $46, $98, $FA, $C6, $A3, $06, $B4, $11, $98, $7C, $28, $2A, $E8, $92, $49, $C2, $64, $A9, $E1, $C8, $A4, $AB, $93, $16, $1F, $35, $96, $77, $35, $2F, $C8, $B5, $6B, $6E, $37, $AE ), {00}
    ( $1E, $E1, $47, $18, $EC, $F4, $37, $F2, $81, $48, $21, $C5, $71, $0E, $28, $A3, $EF, $41, $E3, $0D, $49, $08, $7A, $37, $9D, $50, $EF, $B0, $07, $95, $75, $85, $88, $F9, $5D, $C7, $AE, $A0, $FA, $41, $BF, $81, $9D, $EF, $28, $23, $63, $78 )  {01}
  );
  
const
  TABLE_XOR: array[0..256] of TDigestBuf =
  (
    ( $01, $DC, $DF, $00, $41, $4B, $30, $37, $B1, $B3, $AF, $66, $1B, $8E, $96, $F8, $94, $4D, $28, $73, $DB, $39, $31, $21, $73, $DA, $9A, $36, $66, $2A, $E7, $55, $1F, $4F, $31, $8C, $4E, $CB, $56, $B1, $F0, $97, $43, $D9, $9C, $2A, $A5, $BC ), {00}
    ( $A8, $1F, $BB, $C6, $CB, $BF, $C9, $54, $39, $DE, $43, $64, $89, $59, $ED, $DB, $1A, $64, $1A, $0B, $DA, $01, $82, $2F, $B5, $2E, $60, $72, $66, $93, $26, $58, $2C, $5B, $17, $31, $AC, $80, $20, $84, $C2, $EF, $10, $67, $1F, $C7, $9D, $D4 ), {01}
    ( $CF, $2A, $8D, $8E, $08, $81, $00, $46, $8B, $7E, $9B, $20, $89, $E2, $68, $F6, $93, $04, $61, $65, $2C, $5D, $2D, $EC, $F0, $96, $E4, $22, $23, $BF, $C8, $B9, $8D, $D3, $38, $96, $4C, $FE, $09, $70, $26, $9C, $34, $2F, $7C, $EC, $60, $BD ), {02}
    ( $B9, $70, $A9, $20, $D2, $ED, $05, $52, $01, $0F, $89, $4A, $25, $4D, $A7, $E1, $8D, $5F, $20, $5F, $9D, $1A, $40, $D8, $50, $C3, $3D, $CC, $C3, $FD, $5F, $58, $B4, $9F, $31, $BD, $E7, $D8, $2C, $57, $7C, $DE, $04, $F6, $2A, $95, $90, $33 ), {03}
    ( $49, $FA, $B2, $82, $2B, $9C, $84, $AC, $34, $B8, $64, $8C, $D6, $8C, $BE, $F1, $E5, $12, $11, $47, $BB, $41, $26, $DE, $C0, $B3, $1F, $54, $B2, $FF, $E0, $0F, $2F, $19, $3D, $A3, $8E, $8C, $C6, $32, $05, $8C, $98, $4B, $42, $9B, $8A, $FC ), {04}
    ( $E0, $7F, $9D, $A4, $4D, $8C, $99, $42, $BE, $2A, $F9, $B3, $9A, $CA, $65, $F2, $5D, $3D, $8F, $B1, $46, $6D, $C2, $95, $C0, $05, $1E, $3E, $C3, $F9, $62, $C7, $F8, $FD, $C3, $CC, $D4, $CE, $2B, $B6, $90, $24, $C0, $EF, $C6, $19, $99, $37 ), {05}
    ( $A3, $11, $13, $77, $EF, $01, $F5, $EF, $31, $F5, $9B, $36, $6C, $02, $A3, $FF, $61, $B8, $29, $49, $76, $0D, $16, $DC, $F6, $B9, $58, $AF, $92, $BD, $3B, $DF, $72, $97, $AA, $EF, $EC, $69, $C0, $B2, $FE, $8A, $50, $AD, $9E, $86, $84, $CD ), {06}
    ( $3D, $3D, $D0, $C8, $29, $EA, $9A, $A0, $3E, $77, $59, $7E, $EC, $96, $C7, $A3, $8F, $D0, $38, $23, $1E, $7F, $16, $84, $64, $61, $7B, $13, $1F, $B7, $ED, $E0, $85, $C9, $9C, $E4, $C5, $40, $58, $74, $A5, $87, $35, $D4, $1F, $19, $C5, $E3 ), {07}
    ( $80, $28, $62, $8A, $CA, $F9, $1B, $9A, $19, $4A, $64, $05, $38, $C9, $70, $64, $04, $A1, $5E, $01, $8A, $4F, $16, $80, $F4, $BE, $1B, $04, $C2, $36, $09, $55, $DF, $B2, $4D, $4C, $EF, $58, $1A, $20, $3C, $59, $D0, $A0, $FD, $15, $87, $9D ), {08}
    ( $88, $F8, $E2, $EC, $E7, $8A, $F1, $BC, $B4, $6B, $6E, $22, $29, $73, $64, $C2, $93, $39, $F1, $7F, $92, $6B, $99, $F3, $31, $29, $34, $89, $B3, $B8, $F0, $7C, $39, $09, $CE, $16, $49, $C9, $BC, $F1, $0C, $46, $10, $3B, $FA, $31, $BC, $E9 ), {09}
    ( $4D, $1A, $0B, $8C, $C7, $EF, $E2, $A9, $AF, $D7, $87, $8C, $AD, $55, $E8, $71, $C8, $9C, $FB, $C8, $58, $CF, $41, $53, $C4, $73, $94, $86, $C7, $4F, $75, $D6, $0B, $F7, $19, $2C, $13, $0A, $C9, $F2, $00, $84, $F2, $BC, $5E, $81, $BD, $9A ), {0A}
    ( $4A, $FB, $E9, $75, $01, $4F, $CA, $CF, $41, $DE, $AF, $8C, $FA, $CC, $41, $E4, $3C, $1E, $C2, $3B, $53, $ED, $16, $E9, $78, $B0, $6E, $B3, $0F, $1C, $32, $48, $DD, $11, $16, $5D, $04, $28, $5C, $72, $65, $46, $D9, $B5, $60, $94, $91, $E4 ), {0B}
    ( $FF, $D1, $E2, $E0, $4D, $C8, $D2, $60, $07, $B1, $86, $94, $8A, $74, $EC, $CE, $B5, $12, $0E, $45, $12, $1A, $C5, $31, $BD, $0B, $C8, $63, $30, $81, $0C, $85, $DE, $93, $AF, $DD, $DB, $48, $77, $30, $78, $DE, $B0, $DE, $6E, $B9, $91, $96 ), {0C}
    ( $79, $BA, $A2, $AC, $3F, $DE, $BA, $55, $B1, $B7, $E3, $E1, $C9, $2A, $56, $7E, $A5, $3F, $69, $AB, $4A, $5B, $07, $94, $F0, $DA, $74, $99, $95, $4F, $6D, $DD, $58, $11, $7C, $89, $E1, $13, $22, $48, $D9, $5D, $F5, $F7, $94, $C5, $1A, $6E ), {0D}
    ( $89, $BC, $8A, $0C, $65, $09, $1C, $33, $B0, $4E, $EA, $EE, $06, $31, $35, $C2, $F5, $32, $67, $E0, $4A, $B6, $E6, $89, $B4, $C5, $79, $B7, $20, $7E, $8B, $F2, $3F, $D3, $1E, $51, $34, $3C, $D2, $DF, $11, $9E, $52, $3E, $2F, $81, $97, $FE ), {0E}
    ( $2E, $10, $CB, $1C, $60, $60, $F3, $2D, $BC, $92, $E7, $32, $A9, $4E, $6B, $63, $F3, $22, $0D, $83, $1F, $D0, $42, $67, $50, $2C, $5F, $74, $14, $BC, $E8, $7F, $89, $E0, $65, $1D, $E9, $1D, $24, $57, $75, $9E, $56, $B0, $44, $82, $D9, $15 ), {0F}
    ( $43, $AE, $E3, $2C, $6A, $84, $E8, $03, $0C, $50, $07, $20, $2C, $0B, $D7, $E3, $B4, $F4, $64, $47, $42, $05, $D3, $2A, $7D, $17, $FC, $95, $DE, $38, $6C, $06, $E8, $DF, $BF, $64, $56, $7A, $A5, $45, $55, $BD, $88, $9D, $58, $53, $04, $6F ), {10}
    ( $68, $7A, $BE, $14, $EA, $B8, $DA, $27, $39, $7B, $3A, $B5, $0D, $72, $C3, $44, $50, $5E, $AA, $6D, $1F, $DE, $61, $8D, $61, $BE, $79, $86, $5D, $A1, $3F, $69, $17, $BB, $AB, $29, $B5, $E9, $0E, $2D, $01, $0C, $92, $19, $72, $FA, $8B, $2F ), {11}
    ( $8B, $72, $23, $A4, $F5, $6F, $F4, $53, $29, $1B, $7B, $5C, $B9, $8B, $6F, $E1, $FD, $42, $16, $25, $78, $6F, $BF, $7D, $E3, $3D, $10, $20, $D0, $E8, $CD, $C8, $CA, $53, $0C, $70, $8B, $73, $9E, $87, $87, $8A, $F1, $F3, $04, $B8, $A1, $2F ), {12}
    ( $7E, $CE, $1F, $24, $E5, $6D, $D7, $11, $2E, $08, $69, $24, $1B, $2F, $A6, $DF, $84, $B8, $34, $DC, $C4, $59, $B2, $FD, $E0, $22, $ED, $A6, $31, $9E, $7D, $3B, $59, $83, $9D, $8C, $A0, $3C, $99, $28, $64, $47, $90, $F4, $91, $BB, $C7, $74 ), {13}
    ( $CB, $DC, $6F, $49, $E6, $B0, $DD, $0D, $44, $BA, $2F, $8D, $00, $34, $67, $32, $86, $BC, $C8, $21, $58, $6A, $E6, $1C, $C7, $B7, $49, $12, $85, $CE, $E5, $5B, $ED, $39, $12, $FF, $D9, $7F, $38, $51, $F4, $AF, $01, $86, $BE, $BE, $BC, $BF ), {14}
    ( $CA, $8A, $48, $E5, $4E, $CC, $E5, $16, $BE, $DF, $18, $64, $B7, $F8, $F5, $06, $D5, $6A, $1F, $60, $A4, $B3, $6A, $A4, $49, $B2, $5A, $B5, $FE, $0D, $D9, $D9, $21, $37, $7D, $BB, $5E, $49, $FC, $E1, $70, $8F, $64, $F5, $D5, $9D, $99, $E7 ), {15}
    ( $E9, $F8, $73, $A5, $69, $AF, $E0, $2C, $DA, $66, $BC, $05, $CA, $99, $73, $90, $8C, $88, $17, $47, $56, $D3, $53, $85, $EA, $AA, $F1, $6C, $F4, $FD, $A7, $30, $B3, $9F, $7A, $55, $65, $3A, $35, $12, $F1, $0A, $B7, $27, $BC, $23, $E8, $52 ), {16}
    ( $93, $E9, $6F, $F8, $C5, $BB, $E2, $AF, $A4, $77, $85, $42, $02, $53, $E9, $7C, $70, $4B, $25, $D1, $F7, $7D, $07, $4C, $C0, $B2, $09, $3F, $14, $70, $55, $9C, $8D, $5B, $FD, $D4, $E3, $64, $AA, $CF, $55, $05, $18, $67, $3F, $5B, $1B, $F7 ), {17}
    ( $8D, $AC, $83, $2E, $5B, $E8, $1A, $CB, $58, $8B, $FB, $20, $2B, $75, $83, $D8, $A3, $4D, $8A, $70, $DF, $D1, $D7, $E4, $D0, $3B, $06, $27, $B6, $87, $03, $3E, $E3, $D0, $BE, $7E, $DB, $DC, $75, $CF, $81, $8E, $E8, $C5, $B0, $9F, $8B, $EA ), {18}
    ( $8E, $3B, $02, $E1, $48, $9D, $7D, $31, $53, $36, $75, $2B, $64, $E3, $B5, $32, $E4, $D4, $CC, $79, $5C, $58, $0A, $65, $6D, $AB, $08, $F3, $13, $ED, $76, $7E, $8E, $56, $7E, $88, $FD, $BA, $36, $BF, $25, $94, $90, $F1, $D8, $E9, $33, $D0 ), {19}
    ( $EA, $B4, $37, $D0, $D6, $2C, $AA, $62, $C0, $90, $AD, $28, $B9, $82, $B0, $03, $E2, $55, $D7, $68, $D2, $57, $04, $EC, $A0, $48, $51, $1A, $F6, $25, $6A, $43, $E7, $9F, $07, $8F, $4D, $49, $8B, $5F, $C4, $17, $35, $FB, $75, $B3, $57, $FE ), {1A}
    ( $70, $DA, $9F, $C3, $50, $4F, $F2, $9D, $B9, $AB, $1F, $38, $86, $73, $FF, $25, $36, $92, $2F, $4C, $D1, $73, $57, $BA, $F0, $9C, $3A, $B2, $92, $E7, $E0, $4A, $90, $CE, $0B, $C3, $D9, $BA, $13, $EC, $64, $7C, $4C, $A6, $3C, $91, $8D, $E3 ), {1B}
    ( $D8, $34, $A3, $8D, $D1, $EC, $D6, $88, $A9, $7A, $00, $20, $DE, $46, $AB, $6A, $9C, $DE, $C7, $F6, $E6, $2E, $A7, $1F, $28, $8A, $5A, $6F, $D7, $4D, $C4, $7E, $D4, $4A, $2E, $27, $65, $CE, $50, $F1, $BB, $9B, $50, $B5, $DB, $9E, $4F, $3C ), {1C}
    ( $C6, $6D, $A9, $0E, $41, $DD, $F2, $E6, $5A, $3F, $E8, $6F, $16, $0C, $5C, $09, $6F, $6A, $F0, $40, $51, $08, $CC, $BE, $F9, $38, $38, $2E, $B6, $27, $FC, $7F, $16, $3D, $D6, $34, $61, $7F, $00, $6C, $51, $84, $B1, $FE, $DC, $90, $84, $97 ), {1D}
    ( $C9, $57, $19, $ED, $07, $FC, $B2, $1C, $51, $12, $DF, $04, $3F, $6E, $E7, $EB, $50, $F9, $FD, $60, $01, $23, $34, $CE, $58, $9F, $A8, $51, $04, $D9, $65, $79, $B7, $12, $9E, $44, $D7, $19, $05, $A7, $33, $14, $76, $6E, $07, $33, $52, $8D ), {1E}
    ( $DC, $6C, $80, $14, $C5, $45, $7C, $B8, $D6, $35, $FD, $CD, $28, $6A, $69, $B6, $D6, $6F, $23, $2C, $E2, $7D, $01, $BF, $56, $AF, $4A, $C0, $F6, $82, $EC, $0F, $57, $DF, $1D, $55, $B6, $43, $28, $F5, $65, $1E, $D4, $C5, $2A, $87, $CA, $CA ), {1F}
    ( $26, $D9, $D1, $CC, $79, $EE, $C5, $02, $69, $BF, $34, $0A, $34, $B1, $EB, $FE, $FA, $A5, $AA, $AC, $8E, $39, $71, $74, $D0, $A8, $F9, $BD, $42, $6B, $CF, $6F, $5B, $13, $1F, $46, $4D, $6D, $24, $52, $12, $2D, $D1, $56, $60, $D0, $D6, $DA ), {20}
    ( $6E, $38, $9A, $EC, $5F, $51, $A2, $2F, $7E, $F6, $8F, $1C, $54, $C1, $27, $FB, $98, $6D, $4D, $46, $E0, $48, $5C, $30, $F0, $A4, $7B, $39, $E7, $CF, $8A, $31, $1D, $39, $8D, $FD, $B7, $F2, $A7, $8F, $2F, $C6, $51, $D1, $FB, $B1, $0D, $2E ), {21}
    ( $A4, $4E, $4E, $8D, $1B, $49, $DC, $B0, $07, $A4, $82, $20, $49, $C2, $F3, $43, $C4, $0A, $C0, $4A, $8D, $65, $05, $BA, $D9, $B9, $1D, $3F, $07, $29, $B1, $6C, $AA, $F3, $99, $51, $B5, $0F, $90, $15, $96, $6E, $F5, $D3, $AD, $3F, $90, $76 ), {22}
    ( $EA, $78, $CB, $AC, $0E, $B6, $D0, $09, $A0, $FE, $A6, $72, $5A, $23, $DE, $AB, $CE, $72, $9C, $74, $44, $CB, $94, $D9, $40, $A9, $94, $62, $66, $27, $AA, $0D, $37, $F7, $38, $CD, $E3, $D0, $18, $D5, $4C, $29, $49, $1C, $01, $CD, $B3, $C5 ), {23}
    ( $7C, $97, $92, $AE, $A7, $45, $C8, $7A, $D1, $FF, $56, $20, $C5, $BD, $8F, $D4, $9E, $CA, $84, $E3, $00, $4B, $56, $B9, $5A, $FD, $39, $23, $C2, $28, $B1, $D6, $E5, $DB, $F7, $9E, $B3, $FD, $28, $3B, $44, $17, $12, $E3, $54, $08, $4B, $9F ), {24}
    ( $E3, $5D, $28, $8B, $D8, $E2, $49, $BC, $91, $77, $6C, $14, $53, $A3, $66, $E5, $F5, $D1, $E1, $68, $4E, $95, $EF, $EC, $91, $08, $E1, $17, $D7, $DD, $F6, $06, $81, $B3, $0F, $9D, $A2, $CE, $7C, $8C, $64, $97, $DB, $D7, $86, $81, $8C, $0D ), {25}
    ( $C2, $F8, $91, $FF, $45, $04, $4B, $E0, $75, $A1, $A7, $6D, $2B, $87, $E2, $EB, $85, $CE, $65, $79, $8A, $F3, $C2, $BF, $21, $3F, $53, $2B, $4E, $FD, $09, $DC, $0D, $AA, $1D, $F5, $A5, $3A, $6C, $88, $30, $28, $60, $6A, $50, $D8, $26, $B2 ), {26}
    ( $60, $9A, $62, $61, $63, $79, $F3, $3A, $A3, $39, $A3, $BC, $53, $E4, $51, $6D, $D7, $AD, $92, $61, $6A, $5A, $DB, $EC, $D0, $43, $72, $6D, $86, $E9, $24, $AA, $85, $55, $B5, $64, $F4, $C2, $98, $65, $56, $AA, $12, $AB, $31, $C1, $D6, $B0 ), {27}
    ( $CE, $D9, $ED, $85, $C1, $C1, $7B, $FF, $EB, $52, $23, $25, $AC, $BA, $FF, $C2, $04, $D3, $D8, $F4, $B2, $D1, $53, $94, $D2, $71, $50, $4C, $04, $75, $6E, $EA, $2D, $DB, $A4, $A9, $1A, $F8, $27, $F1, $1F, $67, $D5, $C2, $8F, $80, $02, $E4 ), {28}
    ( $8B, $9C, $08, $AD, $43, $2D, $C0, $8F, $5A, $75, $43, $E2, $97, $96, $BB, $C7, $34, $A6, $DB, $5B, $3C, $19, $67, $DE, $01, $6E, $3B, $C2, $A2, $80, $4E, $E4, $5B, $9B, $CA, $CC, $E5, $17, $2F, $75, $75, $49, $59, $8B, $80, $AD, $BD, $BA ), {29}
    ( $6F, $3F, $B1, $17, $C5, $CD, $D1, $55, $16, $C3, $B0, $A5, $9C, $D6, $EE, $C5, $D9, $A1, $A4, $11, $DE, $53, $87, $69, $93, $8C, $54, $97, $9F, $4A, $C3, $7C, $37, $37, $BC, $C1, $D5, $52, $84, $DB, $6C, $AD, $9F, $8A, $F9, $15, $6B, $B7 ), {2A}
    ( $EB, $BF, $28, $4F, $9C, $75, $EB, $DF, $B3, $83, $EB, $B4, $06, $75, $3D, $E8, $AA, $86, $12, $7A, $EE, $7C, $40, $3C, $10, $BF, $DD, $10, $52, $3D, $E0, $27, $13, $8B, $F6, $C4, $EB, $4A, $8A, $13, $B1, $EF, $F6, $7D, $DB, $78, $B0, $67 ), {2B}
    ( $F6, $D1, $13, $8D, $7A, $A3, $DA, $5E, $BA, $A8, $09, $8D, $8F, $B6, $63, $71, $DA, $E7, $6D, $1B, $8B, $6C, $AA, $F2, $40, $0F, $10, $34, $36, $8D, $1E, $DC, $7C, $93, $7F, $51, $72, $E8, $D2, $77, $7D, $05, $BB, $F8, $3C, $AD, $E6, $EF ), {2C}
    ( $0E, $9C, $2E, $A6, $CF, $34, $B0, $81, $90, $36, $B3, $0D, $58, $F6, $0B, $A0, $DB, $3A, $2C, $58, $48, $F0, $8B, $CA, $C8, $7A, $D1, $B9, $42, $50, $D5, $64, $7C, $89, $2E, $09, $EE, $F9, $61, $66, $26, $DB, $85, $CF, $57, $10, $85, $F3 ), {2D}
    ( $25, $1E, $E3, $F5, $87, $18, $C1, $2A, $F9, $43, $8D, $81, $17, $8A, $2A, $E4, $F0, $92, $9A, $88, $90, $39, $A8, $A8, $F0, $6B, $65, $22, $5E, $BD, $CC, $FD, $2E, $4D, $14, $ED, $F7, $BF, $73, $C6, $A9, $36, $98, $95, $BC, $1D, $FA, $CF ), {2E}
    ( $CA, $E3, $02, $B4, $1D, $69, $79, $CB, $BB, $FA, $5A, $58, $B5, $1E, $E6, $23, $51, $13, $B9, $9D, $C8, $1A, $B5, $2F, $60, $93, $79, $5B, $EC, $17, $A0, $56, $8F, $71, $FB, $4D, $2E, $5E, $35, $5E, $76, $2F, $92, $ED, $BA, $34, $A2, $F2 ), {2F}
    ( $D1, $30, $01, $52, $65, $A4, $D9, $FF, $09, $BE, $A2, $53, $D7, $1F, $26, $C8, $81, $B6, $EA, $ED, $C4, $65, $21, $E6, $FA, $E2, $68, $16, $56, $82, $B8, $A9, $A8, $9C, $3E, $C4, $77, $4A, $B6, $23, $0D, $2E, $45, $E0, $55, $21, $9D, $B2 ), {30}
    ( $2B, $56, $02, $84, $C3, $A6, $92, $AB, $37, $00, $8A, $D0, $B3, $79, $A7, $B8, $AF, $11, $CD, $2C, $30, $F9, $0B, $FC, $7F, $E8, $7A, $25, $0F, $27, $62, $ED, $C3, $FB, $D7, $11, $64, $72, $42, $C3, $74, $ED, $82, $64, $F6, $B3, $22, $BD ), {31}
    ( $28, $19, $5C, $C8, $A7, $AD, $39, $43, $53, $CB, $E8, $08, $46, $4C, $4F, $C4, $D5, $8E, $3D, $7A, $76, $5F, $87, $26, $D8, $30, $52, $F6, $01, $85, $AA, $4F, $EF, $CB, $0D, $85, $22, $3B, $B4, $E7, $5A, $31, $30, $5E, $78, $7F, $AC, $28 ), {32}
    ( $72, $5D, $0E, $E2, $30, $F1, $95, $43, $90, $91, $D2, $C6, $BD, $DF, $34, $E0, $E3, $BE, $49, $C6, $C2, $75, $46, $01, $61, $BE, $30, $0B, $A4, $AD, $56, $6B, $02, $D7, $9D, $75, $51, $FA, $7C, $C1, $65, $43, $91, $0F, $5F, $1C, $DA, $58 ), {33}
    ( $40, $99, $AD, $C4, $44, $81, $B4, $3F, $FE, $13, $61, $92, $2F, $D9, $EB, $81, $A9, $89, $C0, $9E, $44, $1F, $CE, $AC, $44, $9B, $3A, $13, $D3, $CB, $90, $19, $45, $A9, $BE, $39, $6F, $20, $11, $34, $DC, $1A, $D0, $5A, $04, $66, $33, $FE ), {34}
    ( $1A, $56, $3A, $6D, $52, $2F, $3E, $69, $BE, $58, $9E, $07, $9F, $47, $5A, $9E, $75, $A2, $A9, $63, $8E, $4C, $00, $38, $DA, $3B, $62, $02, $57, $7A, $03, $62, $21, $1D, $3F, $1E, $0D, $72, $7A, $F6, $5E, $1F, $FC, $52, $9A, $D9, $92, $33 ), {35}
    ( $47, $B6, $1E, $86, $C6, $D6, $D0, $1B, $43, $7D, $6F, $83, $AD, $AD, $C3, $18, $D5, $A3, $61, $02, $8D, $ED, $73, $8C, $A0, $0D, $4C, $63, $04, $25, $16, $4B, $1A, $69, $AF, $A5, $AF, $4C, $9D, $D2, $F9, $9E, $1C, $67, $F9, $51, $B5, $82 ), {36}
    ( $A6, $6A, $77, $40, $B6, $BD, $EA, $79, $FE, $F7, $FF, $14, $96, $AF, $80, $A3, $05, $AF, $D4, $3E, $EA, $CD, $89, $8C, $B0, $0C, $78, $ED, $31, $AD, $71, $34, $0E, $D3, $1A, $1A, $D7, $84, $66, $73, $74, $B9, $68, $44, $16, $14, $99, $BE ), {37}
    ( $46, $FA, $8D, $6C, $CB, $F6, $D1, $2E, $31, $C2, $FC, $14, $7F, $30, $39, $56, $70, $7F, $44, $01, $DE, $5F, $06, $7F, $3A, $E5, $FE, $C7, $E3, $35, $94, $E9, $28, $E3, $9F, $8A, $63, $53, $17, $14, $B7, $B3, $29, $EA, $1E, $9F, $CA, $B2 ), {38}
    ( $EF, $D8, $F7, $55, $82, $5C, $78, $04, $1F, $5A, $93, $87, $0B, $D3, $0C, $D1, $EF, $BF, $89, $46, $71, $FF, $87, $16, $28, $ED, $61, $7F, $F2, $2B, $DA, $58, $41, $12, $89, $CC, $AE, $5C, $B6, $2E, $95, $DD, $42, $F4, $18, $01, $F2, $F9 ), {39}
    ( $A8, $52, $5B, $86, $45, $FC, $59, $E1, $75, $E6, $2D, $C0, $0A, $5F, $7F, $0C, $09, $C5, $67, $85, $21, $04, $16, $AC, $50, $EF, $76, $E9, $B3, $0D, $76, $26, $2B, $3B, $2C, $DC, $19, $F5, $D6, $65, $A4, $12, $97, $CD, $11, $D8, $F4, $FF ), {3A}
    ( $EA, $C9, $9A, $64, $9E, $EE, $50, $39, $A5, $93, $C9, $2F, $14, $3C, $00, $65, $B3, $14, $73, $52, $03, $07, $12, $06, $EA, $27, $61, $A0, $C7, $64, $A4, $EC, $02, $AA, $7F, $D4, $6C, $AC, $25, $B3, $C6, $8C, $C1, $82, $A9, $6D, $03, $BF ), {3B}
    ( $B2, $87, $3F, $02, $4E, $C8, $3C, $A8, $97, $47, $0A, $B8, $FD, $88, $53, $EB, $18, $FE, $15, $C1, $59, $B3, $05, $BD, $B0, $AB, $08, $F6, $87, $EA, $EA, $FD, $51, $0A, $3F, $DE, $73, $60, $2E, $43, $03, $E1, $B8, $4D, $CC, $F0, $FC, $F0 ), {3C}
    ( $D8, $5B, $BB, $DC, $80, $33, $C0, $D8, $92, $23, $D9, $C3, $9C, $A9, $F3, $4F, $7D, $3B, $CB, $6D, $5B, $63, $C3, $FD, $1C, $30, $F9, $74, $DA, $0C, $0F, $B5, $8B, $24, $BC, $9E, $BE, $FB, $51, $43, $C5, $89, $54, $92, $5B, $7B, $84, $FC ), {3D}
    ( $6A, $BD, $7C, $2E, $08, $44, $D7, $A7, $CC, $F2, $EA, $45, $6C, $DF, $53, $0D, $E8, $93, $8C, $F5, $2B, $39, $21, $B8, $BA, $02, $3C, $A2, $F2, $81, $65, $7C, $EC, $63, $5D, $A6, $75, $D1, $ED, $AE, $B4, $AA, $52, $F2, $2E, $E1, $BE, $6C ), {3E}
    ( $98, $1C, $3A, $C6, $77, $CB, $59, $04, $6A, $92, $B5, $4C, $84, $87, $7B, $49, $74, $5B, $A6, $BB, $40, $C5, $58, $15, $B7, $AF, $55, $0D, $22, $A3, $71, $ED, $D5, $E8, $BD, $87, $C6, $5F, $53, $74, $67, $87, $4A, $37, $F0, $F5, $38, $F5 ), {3F}
    ( $C2, $3B, $BA, $2A, $9D, $EC, $C0, $21, $4E, $61, $0E, $93, $0B, $0E, $34, $50, $1A, $68, $1A, $A9, $14, $77, $57, $7E, $38, $A3, $20, $97, $14, $ED, $C3, $76, $0F, $D1, $55, $63, $EE, $EB, $4A, $B6, $7D, $57, $66, $8A, $01, $D4, $21, $78 ), {40}
    ( $6A, $F8, $8C, $E1, $45, $A0, $98, $B5, $1A, $EB, $85, $8C, $D8, $8B, $8B, $46, $E8, $B7, $33, $AF, $B8, $E2, $D6, $E8, $31, $3F, $AA, $8C, $10, $A7, $EB, $FA, $12, $7D, $37, $5E, $77, $55, $7C, $EA, $96, $BD, $A2, $F7, $0B, $2F, $21, $55 ), {41}
    ( $EC, $89, $03, $97, $8F, $AF, $B6, $36, $C7, $21, $3C, $42, $5C, $07, $97, $63, $76, $03, $84, $03, $6A, $B6, $D1, $7C, $E0, $C6, $3A, $26, $38, $5F, $1F, $49, $29, $98, $77, $D6, $81, $1A, $6D, $F5, $87, $6F, $90, $FC, $53, $04, $B8, $8D ), {42}
    ( $A6, $FA, $BB, $C2, $D6, $E0, $BA, $16, $9B, $70, $C9, $64, $00, $80, $E6, $BC, $29, $B2, $D5, $26, $55, $98, $B2, $7B, $4A, $96, $57, $C7, $26, $E4, $39, $7E, $A8, $01, $CC, $C6, $76, $66, $78, $D5, $80, $0E, $F7, $CC, $72, $61, $99, $98 ), {43}
    ( $23, $59, $31, $A8, $CF, $54, $90, $BF, $E7, $98, $F9, $8E, $0E, $8F, $87, $9F, $C6, $EE, $E2, $9C, $38, $F3, $0C, $A7, $92, $9A, $79, $F2, $D5, $3E, $00, $24, $88, $F2, $E1, $27, $49, $58, $7A, $45, $0B, $85, $B2, $8F, $38, $89, $19, $65 ), {44}
    ( $16, $5E, $03, $03, $E4, $A4, $D8, $27, $67, $99, $4F, $42, $D1, $E8, $43, $6A, $E6, $CC, $8B, $CF, $6E, $13, $0D, $1B, $50, $10, $17, $11, $70, $9D, $DE, $FC, $37, $3B, $DE, $C4, $0C, $D0, $53, $28, $40, $B2, $74, $A4, $AA, $51, $09, $F6 ), {45}
    ( $A9, $F8, $8B, $A0, $08, $FD, $F8, $C8, $EC, $C8, $97, $E3, $47, $6E, $E0, $5A, $BC, $E2, $90, $AB, $69, $D5, $7A, $74, $FA, $44, $DB, $18, $11, $E3, $11, $5D, $62, $67, $AE, $FD, $64, $48, $0C, $88, $26, $97, $D0, $4A, $2D, $3A, $EC, $EB ), {46}
    ( $C0, $78, $2A, $F2, $AB, $CD, $33, $13, $02, $BA, $12, $90, $F2, $F9, $62, $73, $63, $C8, $2F, $1A, $56, $AD, $C2, $B9, $10, $F8, $E8, $C0, $3E, $FE, $51, $C4, $E3, $EB, $34, $86, $25, $CC, $AF, $FD, $93, $D6, $07, $96, $9C, $B8, $E7, $AE ), {47}
    ( $CC, $6E, $17, $94, $43, $E5, $8F, $BC, $D2, $1C, $93, $C6, $55, $A7, $B8, $EE, $2B, $98, $34, $A3, $1F, $2B, $8B, $A4, $C8, $3B, $69, $51, $60, $25, $EC, $EE, $91, $76, $EB, $7B, $42, $7A, $AE, $94, $8C, $B6, $5B, $9E, $30, $B7, $A7, $6E ), {48}
    ( $C1, $A3, $3A, $0A, $D6, $ED, $D9, $89, $18, $B3, $C5, $D9, $58, $13, $B5, $F7, $B0, $24, $BD, $26, $3B, $35, $9A, $8B, $C8, $C1, $7C, $22, $16, $A9, $9B, $50, $71, $F9, $A1, $1D, $58, $23, $77, $29, $3A, $A6, $7C, $76, $18, $28, $42, $90 ), {49}
    ( $99, $B7, $46, $5E, $09, $20, $1C, $7B, $9A, $F8, $9F, $A0, $1C, $A4, $FA, $81, $FC, $2E, $C6, $3E, $76, $1A, $D1, $23, $E2, $A9, $A3, $95, $85, $B1, $7D, $14, $08, $39, $4D, $E5, $29, $F9, $4E, $81, $47, $94, $48, $E6, $97, $94, $FA, $A4 ), {4A}
    ( $23, $CA, $3D, $1C, $4C, $BD, $CA, $BB, $E3, $26, $54, $36, $CE, $1A, $37, $E4, $1B, $BF, $10, $F6, $9E, $8A, $4C, $C9, $05, $A6, $67, $08, $04, $8F, $5C, $4D, $E2, $59, $DC, $DD, $9C, $5B, $FE, $FE, $43, $9E, $65, $FA, $FD, $93, $6E, $FD ), {4B}
    ( $A2, $4D, $73, $B6, $97, $8F, $71, $9C, $3F, $53, $F3, $43, $CC, $B0, $BB, $8E, $BE, $3C, $72, $76, $9E, $E0, $7C, $6A, $FA, $CB, $9E, $53, $9C, $F5, $58, $DD, $67, $B9, $1D, $4E, $30, $DE, $98, $6A, $1D, $B9, $13, $D1, $16, $98, $91, $3A ), {4C}
    ( $98, $BD, $4E, $14, $0D, $C3, $C3, $C6, $14, $2B, $15, $92, $BF, $32, $63, $E8, $CD, $BE, $AC, $59, $ED, $09, $5B, $0E, $90, $07, $63, $F0, $F6, $25, $89, $6A, $E2, $13, $55, $0F, $30, $32, $4E, $39, $8A, $13, $A4, $41, $7A, $80, $31, $95 ), {4D}
    ( $2A, $CD, $98, $ED, $8C, $62, $60, $73, $1C, $AA, $A6, $B4, $C4, $CF, $32, $38, $04, $DC, $B4, $1E, $B6, $77, $EB, $5D, $F8, $8B, $58, $44, $A8, $10, $5B, $68, $98, $1D, $9E, $95, $1A, $06, $1A, $4D, $BC, $94, $71, $89, $4C, $87, $8E, $DB ), {4E}
    ( $49, $59, $FE, $AD, $5D, $6C, $2D, $BD, $6A, $BD, $59, $E2, $8C, $50, $30, $49, $06, $D2, $C5, $49, $4C, $AF, $8B, $34, $70, $E4, $54, $13, $04, $A4, $29, $3C, $52, $0F, $34, $16, $CA, $F2, $F5, $03, $B2, $3D, $09, $D9, $26, $13, $DB, $85 ), {4F}
    ( $26, $B5, $A8, $15, $C3, $2D, $17, $91, $2C, $99, $E7, $55, $5B, $B0, $33, $C6, $09, $CE, $9D, $6A, $00, $02, $51, $4F, $D4, $85, $28, $2B, $2B, $8D, $79, $97, $9C, $5B, $79, $2F, $4A, $4A, $14, $C6, $85, $1D, $9D, $02, $DC, $0B, $B4, $E7 ), {50}
    ( $62, $FE, $B6, $CA, $CF, $B0, $60, $EC, $9D, $97, $7D, $69, $D5, $C6, $61, $EA, $BF, $08, $EF, $D8, $06, $D8, $15, $56, $25, $F1, $EE, $A4, $60, $EA, $57, $18, $A2, $53, $46, $B5, $1F, $5A, $96, $65, $D9, $2F, $9A, $DC, $35, $8C, $A2, $74 ), {51}
    ( $27, $E6, $3D, $FC, $63, $E8, $FF, $A6, $CD, $B9, $CC, $E2, $CE, $99, $FD, $A3, $97, $9D, $5B, $75, $49, $74, $83, $0D, $32, $98, $C8, $40, $7D, $66, $93, $BE, $62, $9D, $5F, $AD, $A3, $9B, $42, $B7, $26, $54, $D3, $12, $71, $CD, $84, $E1 ), {52}
    ( $AB, $1F, $A4, $DA, $F6, $6E, $58, $3C, $EE, $B6, $B7, $A2, $36, $D2, $47, $66, $A9, $07, $38, $CD, $FD, $F5, $C6, $B3, $28, $CB, $A9, $E5, $64, $8E, $2D, $4C, $FD, $E5, $BF, $6C, $0C, $FE, $0D, $A3, $9D, $00, $B8, $63, $D7, $D7, $84, $85 ), {53}
    ( $75, $FB, $BF, $09, $4E, $EA, $16, $AA, $48, $93, $1F, $02, $7C, $D7, $29, $F2, $5D, $36, $06, $79, $00, $9B, $2E, $7F, $DD, $FC, $D1, $48, $BD, $3D, $E2, $1A, $4D, $BF, $F5, $44, $B0, $94, $D0, $E1, $9C, $0E, $5C, $62, $94, $35, $2C, $22 ), {54}
    ( $28, $3A, $27, $FF, $96, $88, $53, $D2, $B0, $96, $0C, $6C, $EA, $0D, $03, $F2, $17, $2B, $BA, $07, $A4, $73, $DB, $38, $68, $8C, $87, $D2, $96, $E6, $F4, $BB, $5C, $B7, $E9, $BC, $5D, $68, $CF, $0F, $57, $A5, $D7, $1B, $0E, $47, $BF, $B4 ), {55}
    ( $DE, $01, $08, $AA, $C1, $E4, $FF, $2F, $D3, $46, $CF, $AB, $EA, $C6, $2B, $99, $B7, $2E, $20, $3F, $98, $B5, $F6, $08, $81, $85, $3D, $8C, $A5, $4B, $29, $BE, $A6, $AE, $D7, $C8, $9F, $AA, $16, $80, $D2, $09, $3B, $15, $5C, $39, $D7, $ED ), {56}
    ( $0B, $AE, $AC, $99, $D4, $97, $4B, $84, $C7, $F2, $58, $A6, $99, $C9, $B4, $DA, $6F, $62, $2C, $5E, $4A, $CC, $F5, $C1, $58, $AB, $39, $7D, $97, $81, $BE, $AA, $BF, $81, $1F, $67, $E1, $01, $FF, $E3, $AF, $BC, $C2, $88, $1C, $3C, $0E, $F3 ), {57}
    ( $26, $B2, $11, $FB, $51, $8D, $6C, $3E, $64, $BA, $DA, $D5, $1A, $10, $78, $4A, $E6, $BE, $4E, $06, $A5, $87, $18, $6C, $D4, $71, $F5, $C6, $13, $43, $CD, $5C, $83, $89, $BB, $0D, $D6, $AA, $ED, $5D, $C8, $81, $12, $67, $89, $14, $A1, $7D ), {58}
    ( $2B, $2D, $0B, $C3, $BB, $88, $D2, $7D, $C5, $A7, $D1, $FA, $FF, $51, $7A, $D2, $96, $F3, $90, $56, $A0, $9F, $82, $AD, $FB, $38, $A6, $1A, $6C, $ED, $4D, $4E, $9D, $30, $8E, $4E, $A6, $F9, $B2, $64, $90, $97, $CE, $29, $4A, $EC, $C6, $B3 ), {59}
    ( $8F, $CA, $2B, $95, $06, $90, $B1, $A2, $29, $3E, $FC, $BF, $03, $D4, $22, $DF, $8C, $91, $25, $B3, $E7, $63, $53, $AB, $3D, $40, $20, $92, $A1, $A7, $01, $73, $9B, $AB, $97, $4C, $AB, $9B, $F6, $76, $5E, $A8, $FC, $C5, $5D, $8C, $58, $6E ), {5A}
    ( $40, $8C, $92, $E8, $C2, $E1, $EC, $8C, $4A, $F4, $C9, $14, $B7, $1B, $43, $50, $51, $86, $AE, $E0, $CD, $FB, $10, $69, $23, $85, $EA, $FA, $B9, $65, $7C, $67, $F7, $08, $E4, $D3, $C8, $98, $CA, $80, $1E, $C8, $B9, $F8, $98, $84, $90, $7E ), {5B}
    ( $46, $E8, $95, $8B, $6A, $2C, $18, $78, $21, $72, $FD, $41, $0F, $78, $A6, $47, $9D, $8E, $9D, $D8, $3A, $29, $90, $04, $39, $09, $13, $C3, $26, $5A, $D0, $25, $D2, $31, $F1, $E2, $30, $77, $CB, $F1, $E7, $EE, $3E, $57, $4E, $80, $D7, $F3 ), {5C}
    ( $5A, $85, $67, $A3, $D8, $5E, $40, $B2, $16, $EC, $F1, $61, $13, $3F, $CF, $73, $52, $DA, $5C, $6F, $BA, $3C, $0D, $D7, $56, $E5, $79, $83, $DE, $B3, $4B, $FB, $83, $25, $4F, $DC, $B7, $68, $D1, $53, $9A, $14, $F9, $5F, $35, $C6, $B8, $2D ), {5D}
    ( $49, $8A, $29, $C6, $E1, $9D, $4A, $E6, $2E, $F4, $AA, $F4, $60, $27, $BA, $11, $BD, $BA, $7D, $AA, $84, $F3, $95, $05, $94, $0B, $2A, $04, $F6, $DC, $94, $4D, $4E, $7E, $D3, $56, $10, $FC, $0D, $53, $BA, $DD, $94, $C2, $90, $7E, $59, $E1 ), {5E}
    ( $14, $DF, $0F, $C4, $3F, $47, $5F, $80, $17, $E2, $AA, $8D, $26, $4B, $F8, $2F, $92, $62, $5B, $DF, $E5, $8B, $93, $4D, $83, $84, $F4, $15, $A4, $AC, $EA, $81, $8E, $9C, $5E, $AE, $C5, $D8, $64, $2B, $4D, $8E, $F5, $5F, $1C, $82, $66, $87 ), {5F}
    ( $4A, $23, $35, $C4, $F7, $71, $28, $D9, $54, $4E, $14, $76, $D2, $9A, $BA, $94, $65, $4E, $C8, $63, $21, $78, $50, $44, $B0, $4A, $D9, $B0, $2F, $80, $44, $5A, $B0, $E0, $1B, $64, $80, $C8, $D0, $20, $59, $6E, $32, $5E, $88, $A3, $CB, $BF ), {60}
    ( $89, $69, $55, $15, $74, $48, $D0, $62, $0D, $B0, $8C, $4C, $0F, $23, $6D, $68, $3B, $A8, $FC, $5B, $3C, $D1, $C4, $A2, $04, $F5, $7C, $53, $E1, $44, $53, $5B, $B7, $D0, $4D, $CC, $7B, $E4, $68, $40, $4B, $BE, $99, $31, $92, $33, $46, $46 ), {61}
    ( $1D, $78, $37, $E6, $AB, $02, $CE, $27, $3E, $A3, $5B, $AE, $D4, $49, $3E, $A4, $D1, $CA, $FD, $B5, $DF, $94, $FA, $BE, $98, $B5, $80, $BB, $62, $17, $0C, $4F, $C3, $C5, $7A, $6C, $A9, $42, $1C, $43, $68, $D6, $5F, $C2, $C1, $20, $16, $34 ), {62}
    ( $FA, $EA, $BA, $BC, $48, $71, $75, $36, $45, $42, $51, $E8, $F6, $2F, $73, $15, $B3, $18, $E8, $A7, $FD, $CD, $C5, $23, $7C, $2E, $83, $20, $13, $C9, $13, $44, $4D, $9E, $5D, $AF, $D1, $69, $90, $52, $12, $26, $2E, $8C, $87, $05, $37, $A7 ), {63}
    ( $8A, $3E, $5D, $0B, $EF, $84, $02, $A2, $A3, $3B, $C5, $FA, $FA, $01, $99, $09, $63, $CB, $E8, $AC, $D0, $07, $62, $F5, $EA, $26, $A3, $F1, $81, $98, $41, $78, $6E, $EB, $78, $D1, $BB, $4A, $F6, $BB, $7E, $CF, $96, $71, $30, $0E, $84, $5F ), {64}
    ( $08, $11, $B6, $7C, $CC, $F5, $D0, $FC, $9F, $8C, $AB, $3F, $34, $96, $BD, $6B, $57, $CB, $7D, $24, $F1, $35, $5C, $2D, $58, $21, $85, $94, $16, $5B, $DE, $80, $FA, $F3, $36, $8A, $65, $3A, $78, $F8, $C0, $4C, $D8, $01, $76, $26, $77, $62 ), {65}
    ( $E6, $41, $7C, $E7, $5A, $AA, $23, $B0, $34, $A7, $BF, $E3, $CB, $A6, $17, $61, $8C, $13, $E3, $96, $F8, $C9, $B6, $ED, $5C, $90, $66, $46, $4B, $09, $ED, $63, $76, $CB, $6A, $64, $2C, $5C, $E2, $83, $49, $8E, $08, $2A, $3E, $B4, $49, $C6 ), {66}
    ( $6F, $2A, $DE, $A6, $35, $7B, $5A, $A0, $54, $DA, $38, $2B, $15, $55, $7B, $69, $30, $2B, $D8, $19, $46, $23, $7A, $AE, $8F, $0C, $BB, $82, $11, $1E, $FE, $DC, $45, $DD, $2D, $AD, $CE, $20, $F2, $D3, $8A, $77, $A5, $E9, $E8, $A2, $D1, $D8 ), {67}
    ( $E1, $EC, $33, $27, $35, $86, $2A, $28, $92, $B6, $8B, $1A, $7E, $9C, $7C, $44, $F4, $56, $18, $DC, $99, $E9, $63, $E3, $7C, $AC, $98, $45, $02, $DD, $1A, $73, $C8, $65, $05, $98, $CD, $70, $84, $0D, $9A, $5D, $A5, $84, $A2, $6D, $4E, $FD ), {68}
    ( $16, $B1, $9B, $01, $07, $40, $C1, $5C, $B4, $54, $4A, $C0, $10, $16, $43, $9A, $22, $1F, $74, $9C, $9E, $2F, $99, $A5, $A6, $3E, $8A, $27, $9A, $65, $57, $0F, $C7, $23, $16, $69, $AD, $D0, $72, $AD, $C5, $BC, $35, $BA, $74, $0B, $C8, $01 ), {69}
    ( $6C, $44, $E7, $5A, $4F, $37, $86, $94, $D2, $7A, $CE, $10, $8A, $57, $76, $47, $17, $C4, $87, $FA, $FA, $7E, $15, $D6, $6A, $36, $54, $D5, $C8, $E2, $9E, $DF, $0C, $E3, $5E, $ED, $CC, $61, $1F, $FA, $D8, $8A, $8C, $03, $C0, $09, $50, $93 ), {6A}
    ( $CF, $10, $69, $48, $BC, $4B, $1F, $2C, $91, $C0, $DC, $99, $90, $B9, $97, $12, $19, $3B, $21, $E3, $E1, $09, $AB, $32, $33, $40, $DE, $06, $08, $DD, $16, $66, $8A, $5B, $B6, $77, $BF, $60, $28, $28, $40, $2C, $41, $0B, $11, $97, $B7, $71 ), {6B}
    ( $EB, $08, $0F, $F4, $9C, $A5, $54, $3E, $B4, $B9, $42, $95, $42, $D6, $CA, $27, $59, $99, $D4, $5D, $C1, $53, $32, $05, $F7, $EA, $9E, $39, $8A, $1B, $EF, $3E, $BE, $88, $17, $77, $54, $76, $DE, $C6, $17, $06, $4D, $77, $90, $C8, $41, $00 ), {6C}
    ( $F3, $32, $8E, $91, $50, $A7, $F8, $D6, $52, $E3, $E6, $1B, $04, $AC, $FD, $F8, $51, $D8, $20, $10, $F3, $CE, $B0, $15, $59, $D6, $73, $33, $66, $76, $D5, $D8, $4C, $B3, $BC, $EF, $1D, $91, $C3, $42, $0C, $58, $9A, $B5, $80, $33, $BE, $49 ), {6D}
    ( $54, $B8, $E7, $0E, $DC, $E0, $38, $55, $7B, $B5, $90, $E9, $96, $87, $FD, $57, $6C, $FF, $08, $68, $8D, $2B, $1F, $DD, $FD, $0F, $6D, $06, $8B, $FE, $99, $4F, $EB, $9B, $CE, $30, $24, $89, $AE, $44, $66, $B2, $1F, $20, $06, $61, $E3, $E4 ), {6E}
    ( $2F, $5E, $00, $60, $18, $96, $69, $AD, $47, $3A, $F1, $D0, $3C, $00, $CA, $E4, $02, $78, $29, $92, $68, $D1, $F3, $B4, $88, $87, $14, $BC, $3A, $7E, $C9, $D2, $9F, $F9, $C7, $F0, $71, $EB, $D2, $D9, $87, $5A, $5D, $C2, $5D, $FF, $DB, $10 ), {6F}
    ( $E2, $A9, $7A, $3E, $46, $83, $99, $D8, $3B, $F7, $EA, $CA, $32, $C8, $0D, $A1, $13, $DC, $AC, $8E, $B6, $C2, $23, $1D, $22, $7E, $C9, $0E, $11, $02, $EE, $97, $B2, $34, $48, $32, $F0, $38, $14, $34, $86, $13, $88, $83, $03, $B1, $90, $EB ), {70}
    ( $3A, $3D, $3B, $6C, $E0, $26, $BF, $FE, $18, $D4, $95, $3B, $9A, $68, $ED, $59, $24, $BB, $7B, $57, $4A, $B7, $77, $A0, $E0, $CB, $7D, $D6, $49, $83, $DC, $B1, $CF, $76, $8C, $43, $98, $69, $AC, $97, $80, $62, $BC, $7A, $90, $0E, $60, $33 ), {71}
    ( $39, $D4, $C3, $B7, $8A, $7A, $33, $C7, $43, $D7, $2E, $F2, $2A, $B0, $B4, $EB, $54, $AE, $81, $84, $DD, $A5, $03, $94, $0C, $2A, $7D, $A0, $83, $C3, $85, $36, $9D, $BC, $6F, $92, $1D, $4A, $D8, $22, $2C, $BB, $61, $FE, $18, $2E, $AA, $42 ), {72}
    ( $D8, $CE, $9A, $80, $6C, $0B, $D2, $4D, $F6, $9D, $65, $A6, $58, $45, $72, $7C, $C3, $FF, $81, $CC, $76, $F2, $B0, $48, $76, $B1, $FD, $C3, $CA, $67, $CE, $58, $CE, $D0, $97, $0A, $FB, $CB, $E7, $8A, $57, $50, $29, $41, $B7, $26, $F5, $F3 ), {73}
    ( $E0, $06, $AE, $C1, $7F, $CE, $FC, $F9, $05, $CA, $A1, $62, $9E, $00, $35, $91, $B7, $05, $0C, $C9, $9F, $58, $53, $12, $66, $92, $60, $40, $1E, $15, $94, $90, $84, $42, $D2, $5A, $A7, $57, $CC, $5A, $22, $86, $55, $CD, $40, $38, $77, $0C ), {74}
    ( $93, $EE, $8D, $67, $D3, $F1, $F3, $A1, $BE, $A4, $6D, $48, $DB, $F8, $D7, $F4, $3C, $91, $F0, $2B, $86, $46, $45, $3C, $6C, $3D, $7C, $1F, $04, $18, $8A, $58, $EF, $A9, $72, $87, $F8, $9C, $EF, $84, $CB, $40, $36, $4E, $10, $8B, $FF, $4B ), {75}
    ( $C6, $DC, $E3, $73, $0D, $4F, $F8, $25, $02, $AF, $54, $F8, $7D, $97, $27, $90, $7D, $69, $D2, $0F, $6F, $4F, $78, $8F, $90, $C2, $55, $C6, $4C, $16, $6E, $8F, $A3, $52, $9F, $BF, $4B, $F9, $C9, $A2, $3E, $CE, $C4, $11, $36, $69, $4F, $6B ), {76}
    ( $3D, $E1, $0A, $5E, $C6, $CA, $7B, $3F, $7E, $19, $60, $81, $D0, $85, $AC, $AA, $DF, $5F, $0D, $E3, $70, $5D, $60, $F7, $39, $3E, $7C, $83, $DC, $C5, $70, $75, $A5, $F3, $3B, $C2, $DC, $B9, $8F, $97, $0A, $EB, $7F, $05, $0D, $12, $04, $C0 ), {77}
    ( $6F, $3B, $3B, $3D, $11, $A8, $BC, $05, $B5, $22, $69, $AB, $2B, $95, $B8, $DC, $12, $ED, $E2, $4E, $B1, $38, $5F, $13, $20, $2B, $BA, $6B, $58, $36, $B5, $E1, $EE, $36, $36, $C5, $92, $5A, $CC, $49, $42, $22, $4C, $F6, $EE, $B5, $09, $BF ), {78}
    ( $5F, $0C, $C3, $BB, $C4, $BE, $9A, $92, $58, $43, $13, $FC, $CC, $54, $DD, $2E, $C1, $1F, $E9, $0F, $00, $39, $40, $36, $33, $71, $66, $7C, $72, $FC, $97, $23, $96, $11, $99, $0B, $62, $AC, $8D, $9F, $4C, $FC, $B9, $EB, $3C, $31, $7F, $AD ), {79}
    ( $CA, $8E, $52, $0A, $89, $4A, $3F, $BA, $BD, $9E, $D1, $B8, $00, $98, $CC, $40, $BD, $F2, $45, $07, $DF, $F3, $75, $7C, $47, $AE, $C5, $72, $E6, $8D, $35, $EC, $F3, $D4, $52, $3D, $27, $B3, $73, $E4, $1A, $B1, $1E, $16, $97, $3A, $05, $AB ), {7A}
    ( $FF, $C2, $93, $A6, $C2, $6B, $81, $7D, $2C, $9E, $9D, $13, $49, $59, $D8, $28, $7F, $A5, $21, $64, $08, $19, $9B, $BF, $A6, $F0, $02, $DE, $0D, $CC, $D8, $61, $BE, $8F, $9D, $C5, $7F, $2C, $F3, $5D, $13, $52, $E2, $DF, $86, $A4, $76, $47 ), {7B}
    ( $84, $B5, $5B, $E1, $01, $70, $8E, $74, $3A, $DE, $C5, $37, $21, $20, $9F, $3E, $B1, $8F, $9A, $1E, $68, $DF, $AD, $BD, $09, $A0, $50, $81, $97, $74, $CF, $2D, $E4, $AB, $29, $5D, $38, $0A, $87, $62, $A3, $60, $5B, $0C, $68, $9C, $23, $9F ), {7C}
    ( $DD, $C7, $03, $1F, $BF, $DF, $FE, $8F, $0B, $17, $5D, $E6, $5B, $83, $2F, $0A, $31, $16, $2A, $BC, $65, $71, $96, $85, $51, $21, $5E, $53, $4B, $BC, $36, $B1, $9F, $2F, $7D, $3B, $5D, $01, $AE, $44, $CF, $43, $A2, $42, $6E, $83, $B6, $1B ), {7D}
    ( $7E, $32, $DB, $67, $2B, $16, $F0, $4A, $CE, $6F, $45, $DE, $0E, $6A, $B7, $88, $25, $71, $85, $48, $B8, $E7, $0B, $41, $D7, $36, $8B, $CF, $39, $A0, $FA, $C4, $95, $68, $63, $EC, $49, $88, $0C, $47, $72, $0E, $33, $57, $96, $34, $16, $74 ), {7E}
    ( $06, $70, $7A, $8E, $33, $D9, $D6, $C6, $B6, $84, $BF, $E2, $6C, $D5, $76, $C6, $44, $F4, $7E, $5E, $CD, $5F, $C4, $6C, $AF, $1B, $23, $A8, $56, $D8, $44, $B7, $98, $A6, $27, $91, $6A, $C5, $65, $7E, $04, $0C, $39, $64, $A1, $12, $7E, $19 ), {7F}
    ( $A5, $DA, $EC, $31, $34, $C0, $A3, $9B, $0C, $A0, $41, $60, $BD, $5A, $DB, $1F, $B5, $0E, $C5, $A9, $F2, $9E, $1A, $CB, $BE, $2F, $A1, $12, $6A, $F7, $BF, $AF, $BE, $FC, $0A, $C4, $C9, $C5, $A4, $B3, $99, $47, $39, $C7, $1F, $B1, $EB, $29 ), {80}
    ( $6F, $EC, $2D, $34, $3E, $83, $A7, $63, $5B, $DB, $A5, $71, $57, $57, $F5, $0C, $D6, $F6, $28, $2E, $E4, $6A, $11, $B3, $A8, $B5, $01, $F5, $92, $2A, $55, $24, $A7, $82, $A2, $10, $06, $B6, $05, $CA, $A1, $0B, $D2, $E8, $96, $97, $5C, $81 ), {81}
    ( $B8, $AA, $E0, $53, $22, $26, $D0, $ED, $89, $18, $31, $C0, $47, $0E, $84, $B7, $74, $C8, $24, $D6, $48, $E8, $FF, $28, $B5, $E4, $E0, $2E, $AD, $39, $06, $EB, $5A, $BB, $08, $6A, $DA, $60, $A7, $13, $A8, $0C, $57, $66, $6A, $9E, $29, $F1 ), {82}
    ( $52, $9E, $3E, $52, $B1, $E7, $23, $0A, $0C, $14, $88, $61, $C9, $F0, $8E, $26, $0C, $FC, $8A, $13, $1B, $AD, $80, $3D, $8C, $09, $F3, $24, $90, $2F, $AA, $9F, $02, $31, $EE, $49, $87, $99, $98, $48, $3B, $06, $88, $49, $2E, $2B, $54, $57 ), {83}
    ( $EF, $A6, $EA, $C5, $03, $68, $14, $CD, $02, $77, $3C, $1F, $8D, $AA, $5D, $F5, $0E, $4E, $ED, $BD, $07, $02, $DE, $31, $BA, $7F, $D7, $57, $D0, $D7, $40, $EF, $A8, $80, $5F, $0C, $74, $00, $5F, $8B, $14, $48, $46, $7B, $FF, $3E, $1E, $F8 ), {84}
    ( $2A, $07, $B7, $66, $01, $6A, $C7, $0D, $64, $21, $5C, $35, $36, $42, $19, $E9, $CD, $6F, $7E, $FE, $35, $FC, $F6, $F1, $F0, $5C, $C0, $60, $84, $C2, $92, $67, $AB, $3B, $F2, $F3, $25, $79, $A4, $44, $AC, $75, $F4, $2D, $9A, $25, $B9, $C9 ), {85}
    ( $EF, $3A, $14, $B5, $ED, $DB, $84, $64, $23, $14, $E0, $80, $2D, $2D, $D0, $E9, $14, $DE, $AE, $A9, $F9, $28, $76, $2A, $57, $63, $EB, $B4, $80, $E1, $5A, $02, $25, $F7, $CA, $14, $E8, $CD, $F5, $E6, $8E, $59, $45, $10, $DC, $61, $E6, $BC ), {86}
    ( $E6, $2C, $38, $DC, $FD, $21, $00, $0B, $7B, $B3, $2A, $E9, $17, $EE, $3D, $A7, $E4, $9F, $15, $E2, $4C, $C9, $B6, $56, $56, $E2, $82, $59, $DC, $A3, $61, $D8, $B4, $3B, $80, $08, $A9, $28, $5F, $48, $0D, $C6, $B4, $AF, $7E, $4A, $E6, $1B ), {87}
    ( $70, $3C, $64, $24, $11, $42, $DC, $AE, $73, $2D, $33, $34, $2C, $45, $06, $3A, $37, $87, $7E, $A1, $62, $45, $67, $CB, $28, $71, $D5, $34, $61, $4D, $D1, $14, $E7, $48, $09, $2A, $1D, $94, $F5, $D1, $45, $24, $05, $6F, $0C, $6D, $1C, $B7 ), {88}
    ( $E3, $25, $B1, $82, $3A, $59, $5D, $F9, $74, $2D, $0D, $D5, $C9, $6F, $39, $7C, $44, $36, $1C, $95, $40, $A9, $F4, $51, $02, $38, $2F, $9B, $F6, $33, $1F, $B9, $8E, $CB, $AF, $BB, $E9, $1A, $04, $67, $52, $8E, $BF, $38, $11, $F9, $04, $A8 ), {89}
    ( $FD, $2B, $C6, $53, $46, $31, $FB, $0D, $27, $A5, $F0, $36, $FE, $EB, $9A, $6C, $D0, $F8, $76, $D7, $91, $1D, $07, $75, $12, $EF, $B3, $A2, $9C, $6E, $0B, $72, $DC, $4B, $CA, $3D, $5E, $87, $1D, $A1, $02, $8F, $B6, $E6, $E6, $08, $F4, $6F ), {8A}
    ( $EF, $17, $EC, $C8, $93, $0A, $7B, $4A, $9D, $97, $B3, $46, $72, $FB, $27, $3D, $C6, $AE, $83, $5F, $35, $A2, $5D, $8F, $6C, $27, $46, $95, $30, $C2, $1F, $5B, $2F, $BC, $16, $A2, $61, $50, $E7, $95, $02, $AD, $93, $AA, $E0, $B5, $C7, $1A ), {8B}
    ( $6D, $24, $BE, $43, $CF, $07, $DD, $56, $63, $68, $1D, $62, $A3, $8D, $2A, $2F, $98, $72, $C9, $B4, $11, $72, $4A, $A0, $B8, $82, $B4, $85, $7C, $19, $69, $0A, $87, $B1, $BA, $8D, $28, $04, $C6, $F4, $D7, $B1, $99, $CC, $36, $F4, $0B, $49 ), {8C}
    ( $EE, $FB, $8D, $85, $73, $FD, $9E, $0F, $93, $34, $03, $19, $9B, $91, $56, $0A, $FF, $0D, $B4, $16, $65, $D5, $24, $8C, $32, $2E, $E1, $05, $EA, $98, $41, $96, $DB, $8C, $E0, $F8, $38, $90, $D8, $9B, $3A, $32, $F8, $98, $3C, $90, $1F, $80 ), {8D}
    ( $08, $2C, $DA, $F9, $3F, $21, $5B, $AC, $67, $C1, $18, $A1, $B9, $27, $4F, $AC, $AF, $74, $50, $1C, $FB, $93, $19, $8A, $53, $52, $5C, $AB, $A0, $E8, $12, $D3, $C9, $AF, $3A, $00, $5E, $FE, $8A, $6E, $F2, $42, $DC, $B6, $0D, $A7, $B2, $FE ), {8E}
    ( $D3, $88, $7F, $BF, $BB, $73, $14, $DF, $DD, $DC, $CC, $F0, $F7, $20, $C3, $42, $B2, $C4, $33, $1C, $33, $C8, $C4, $15, $16, $66, $01, $07, $67, $F4, $78, $5B, $84, $55, $B7, $C1, $FD, $5D, $E4, $87, $A8, $21, $C5, $EA, $18, $18, $75, $F2 ), {8F}
    ( $7E, $28, $98, $31, $41, $85, $62, $F0, $2A, $D1, $2E, $30, $42, $B1, $85, $C3, $7C, $20, $D0, $D7, $35, $A6, $AE, $96, $A6, $8B, $EF, $98, $E2, $2C, $BD, $41, $A1, $41, $1D, $22, $F8, $D9, $32, $43, $D8, $13, $FB, $40, $4F, $3D, $2F, $38 ), {90}
    ( $E1, $3F, $C0, $A7, $6F, $66, $42, $94, $7E, $21, $C9, $D9, $F7, $FD, $DD, $CB, $16, $1E, $68, $B3, $66, $D6, $B1, $F8, $55, $BF, $95, $7E, $B5, $74, $38, $74, $B2, $32, $13, $EF, $83, $64, $D7, $66, $52, $9B, $B9, $8A, $F9, $66, $43, $D4 ), {91}
    ( $03, $6D, $7A, $DD, $AA, $DB, $5C, $33, $05, $25, $83, $5F, $80, $2D, $03, $2E, $7D, $F7, $D0, $D8, $D7, $A2, $BE, $F2, $84, $92, $76, $44, $B2, $76, $96, $B7, $21, $5E, $21, $E4, $D1, $F9, $B5, $B9, $77, $74, $36, $69, $C4, $0E, $B7, $FD ), {92}
    ( $A9, $B3, $53, $4B, $E8, $89, $77, $84, $5B, $FD, $42, $83, $54, $1A, $50, $90, $97, $AF, $FC, $CD, $12, $1C, $97, $78, $C1, $46, $C4, $C9, $63, $79, $89, $C7, $08, $20, $E7, $2F, $CB, $DA, $59, $C7, $55, $26, $E2, $F4, $A0, $AE, $4F, $4F ), {93}
    ( $A4, $73, $9E, $20, $FD, $72, $BD, $C2, $6D, $6E, $E5, $A5, $C1, $A5, $4C, $A6, $70, $A9, $7A, $6F, $CB, $88, $4E, $5C, $2B, $61, $08, $33, $9E, $97, $9C, $48, $93, $A6, $37, $30, $D6, $BB, $23, $A7, $5B, $1D, $CE, $AB, $00, $04, $5E, $E5 ), {94}
    ( $42, $7C, $14, $E4, $F8, $8C, $8B, $DB, $1D, $86, $30, $86, $8E, $03, $9B, $C2, $33, $DB, $40, $A2, $51, $50, $2D, $1B, $E0, $43, $C9, $CC, $B4, $5D, $2B, $3D, $29, $2B, $67, $B6, $EE, $07, $7B, $2D, $1C, $3A, $2F, $BD, $E2, $4C, $74, $2A ), {95}
    ( $3D, $ED, $69, $F3, $70, $16, $D8, $6A, $9A, $94, $7B, $13, $AC, $66, $D7, $C3, $82, $2D, $86, $45, $DF, $4C, $B3, $9C, $2B, $A2, $0F, $98, $F1, $9E, $10, $DA, $67, $03, $13, $8D, $42, $2A, $C4, $C4, $8D, $34, $D6, $13, $8F, $A0, $4A, $1D ), {96}
    ( $28, $E5, $9C, $8B, $25, $7D, $11, $2C, $87, $47, $06, $8C, $C5, $49, $9F, $CF, $D6, $C1, $6E, $B7, $80, $F9, $19, $1A, $B4, $16, $15, $16, $33, $F7, $AF, $08, $A2, $30, $E0, $0D, $6B, $A1, $A1, $C3, $FD, $06, $6F, $B9, $96, $5B, $83, $D2 ), {97}
    ( $70, $F4, $BC, $1B, $7F, $8F, $FC, $37, $38, $DC, $03, $31, $E5, $6B, $0F, $DC, $A9, $AB, $72, $90, $AD, $2B, $0B, $BD, $B3, $07, $97, $3C, $3D, $07, $83, $C6, $BD, $C4, $55, $F6, $CD, $CA, $11, $1F, $23, $F0, $E0, $83, $17, $B8, $F0, $DC ), {98}
    ( $0A, $EE, $C2, $4E, $92, $85, $C5, $0F, $3B, $CD, $A4, $78, $33, $B6, $1A, $CE, $83, $99, $86, $F9, $59, $EE, $07, $23, $C9, $59, $03, $4A, $8D, $7F, $5E, $B9, $D4, $AD, $7E, $05, $B0, $5C, $4F, $B5, $6C, $37, $A3, $D3, $9F, $7A, $0E, $C4 ), {99}
    ( $02, $27, $B7, $23, $0F, $BF, $2D, $07, $28, $D7, $D2, $AD, $63, $2B, $ED, $47, $07, $BD, $8F, $8B, $50, $12, $EF, $D0, $48, $A0, $D4, $3A, $E0, $40, $34, $42, $9B, $89, $39, $20, $7F, $14, $49, $A1, $35, $1E, $AD, $01, $B9, $FD, $F2, $19 ), {9A}
    ( $A7, $11, $9D, $2E, $31, $1C, $EF, $25, $1E, $53, $2C, $D0, $C4, $ED, $04, $79, $22, $72, $F8, $78, $D8, $D3, $0A, $0B, $76, $9C, $41, $2C, $ED, $9C, $4C, $42, $26, $2F, $FB, $FA, $65, $CB, $DD, $F5, $DB, $73, $D8, $67, $21, $EA, $36, $8E ), {9B}
    ( $4B, $DB, $E9, $0B, $3F, $BA, $DC, $B2, $13, $24, $EC, $3A, $8D, $6F, $EA, $57, $6D, $9E, $FB, $E5, $30, $85, $0D, $00, $40, $1A, $88, $AF, $F8, $A4, $C8, $F4, $65, $5C, $B7, $6B, $8A, $2E, $27, $1C, $35, $50, $5B, $6D, $BD, $E1, $6F, $43 ), {9C}
    ( $6E, $15, $E5, $7E, $23, $F5, $70, $37, $49, $62, $73, $73, $62, $C1, $FA, $26, $C9, $62, $37, $2D, $18, $29, $B8, $0B, $A1, $FE, $68, $32, $EA, $4D, $62, $11, $67, $26, $E3, $07, $F9, $6E, $77, $63, $04, $C7, $61, $08, $16, $77, $50, $5B ), {9D}
    ( $42, $E2, $FF, $3A, $8A, $6F, $C1, $64, $FB, $85, $B2, $BC, $9D, $28, $B2, $68, $C5, $59, $CF, $F0, $24, $53, $3A, $28, $2E, $C8, $3F, $39, $11, $DA, $B3, $CE, $AE, $0F, $C7, $4A, $9D, $73, $6A, $27, $DB, $9C, $DD, $04, $8B, $AB, $4C, $CF ), {9E}
    ( $D7, $9C, $52, $22, $1D, $20, $E7, $65, $49, $9E, $DD, $73, $90, $3C, $E7, $04, $9B, $01, $6D, $98, $7D, $F4, $83, $49, $FC, $FA, $B4, $4A, $D1, $2F, $C5, $C1, $81, $12, $93, $F3, $B8, $00, $FD, $F9, $51, $1D, $C6, $19, $CA, $53, $CE, $BE ), {9F}
    ( $A0, $59, $EE, $78, $B8, $26, $ED, $DF, $46, $73, $AF, $29, $4D, $17, $C8, $5A, $5E, $52, $7D, $4E, $4D, $F2, $82, $B5, $DB, $5B, $9A, $26, $93, $F9, $5C, $E3, $65, $51, $D3, $04, $FB, $54, $F2, $96, $AB, $3E, $B7, $0D, $65, $91, $2F, $CC ), {A0}
    ( $7D, $0C, $4F, $67, $B6, $C7, $81, $35, $39, $0C, $AE, $A7, $DE, $30, $4D, $37, $49, $E1, $9F, $AB, $C8, $D4, $94, $FE, $1A, $9E, $1B, $64, $37, $A0, $45, $16, $88, $6C, $C4, $BD, $AB, $6A, $F3, $5A, $05, $29, $21, $73, $44, $F5, $02, $FE ), {A1}
    ( $3C, $ED, $F3, $41, $41, $B5, $2C, $EE, $81, $33, $BA, $92, $47, $53, $57, $3F, $CB, $32, $BE, $22, $BC, $66, $02, $5A, $0C, $48, $01, $83, $DE, $40, $3C, $B3, $BF, $5B, $84, $B4, $27, $DF, $CF, $31, $72, $51, $42, $8D, $B0, $23, $21, $56 ), {A2}
    ( $86, $FC, $E8, $31, $C5, $8E, $25, $CB, $5C, $C4, $3F, $FE, $45, $CB, $FC, $75, $33, $87, $7C, $C0, $42, $F1, $99, $BE, $12, $12, $FA, $7F, $0C, $C2, $2E, $1C, $44, $8E, $AB, $4B, $7D, $1F, $98, $23, $A7, $B1, $36, $3A, $9F, $A7, $59, $9E ), {A3}
    ( $2D, $8C, $2F, $ED, $A0, $E5, $10, $6D, $19, $2E, $36, $68, $38, $BB, $EB, $3F, $36, $22, $6A, $A6, $0A, $CE, $A0, $AF, $E7, $E1, $28, $5D, $C1, $F3, $92, $6A, $90, $03, $71, $FA, $18, $83, $D9, $EC, $BA, $C3, $3B, $1A, $F3, $60, $EB, $66 ), {A4}
    ( $D4, $A2, $A1, $16, $12, $BD, $E0, $E3, $82, $AB, $0D, $A6, $14, $CB, $4C, $B8, $18, $9A, $4D, $50, $AC, $01, $F4, $C6, $E3, $6A, $5D, $A1, $D9, $F6, $A6, $47, $E4, $31, $20, $D6, $B1, $6B, $11, $B6, $7D, $39, $5F, $42, $36, $E7, $53, $78 ), {A5}
    ( $C0, $C1, $55, $CD, $47, $F3, $87, $7F, $4B, $03, $BF, $E5, $C3, $34, $CA, $71, $77, $71, $0F, $1F, $4B, $84, $4F, $F7, $34, $43, $BB, $AB, $72, $0E, $8D, $C5, $F0, $3F, $88, $68, $C5, $86, $34, $06, $0F, $D6, $05, $11, $C8, $72, $EB, $50 ), {A6}
    ( $8C, $25, $3D, $AA, $B5, $28, $63, $06, $9A, $A4, $38, $F5, $4A, $61, $96, $AC, $18, $1D, $08, $C7, $23, $A2, $2C, $5E, $63, $3C, $49, $C8, $8E, $39, $10, $A1, $C9, $F5, $4A, $67, $99, $26, $75, $B0, $1F, $DD, $98, $AC, $BD, $38, $D9, $76 ), {A7}
    ( $A1, $08, $93, $DA, $75, $75, $A9, $F7, $8F, $5F, $4A, $02, $5A, $B2, $A0, $18, $D8, $05, $38, $F0, $33, $6B, $FF, $C0, $0F, $97, $51, $D3, $38, $89, $62, $6F, $30, $38, $3E, $B9, $25, $BF, $91, $1A, $E6, $14, $9F, $68, $CE, $19, $CC, $60 ), {A8}
    ( $B9, $08, $1D, $BA, $C6, $BE, $05, $98, $78, $5D, $D9, $BC, $69, $C7, $14, $92, $8B, $03, $5A, $0C, $A5, $6E, $17, $2B, $89, $46, $78, $35, $00, $72, $48, $88, $AF, $1E, $57, $C9, $58, $65, $05, $69, $E1, $DE, $4E, $94, $4F, $F2, $22, $61 ), {A9}
    ( $EA, $5E, $DC, $4D, $27, $18, $C0, $D2, $CB, $1C, $5D, $4D, $A1, $5A, $8A, $E4, $C6, $27, $23, $82, $F8, $16, $30, $15, $94, $A9, $34, $E5, $05, $7B, $54, $CE, $65, $8E, $48, $1A, $3D, $68, $D1, $0D, $E8, $F2, $49, $29, $E5, $0A, $46, $A0 ), {AA}
    ( $7D, $F1, $46, $28, $1A, $F4, $82, $CD, $01, $4B, $68, $E7, $26, $40, $7B, $06, $6C, $E5, $64, $93, $8C, $70, $DD, $BC, $36, $DA, $D2, $DE, $72, $A5, $DA, $A2, $6D, $57, $3B, $F6, $9C, $0B, $29, $80, $68, $4D, $AB, $14, $B4, $AA, $03, $29 ), {AB}
    ( $9C, $69, $DC, $06, $4E, $73, $8B, $5F, $83, $CC, $16, $BD, $5A, $1C, $36, $F5, $A9, $9B, $36, $5E, $6E, $14, $1B, $12, $27, $48, $FA, $5A, $D0, $FA, $CC, $E8, $26, $D0, $73, $A0, $47, $D9, $9C, $49, $B0, $05, $B1, $82, $50, $5B, $0C, $0C ), {AC}
    ( $15, $B6, $A2, $A2, $0E, $D0, $FD, $1C, $93, $33, $AF, $72, $9B, $D6, $5A, $25, $22, $CC, $33, $32, $93, $BD, $2C, $1B, $D7, $24, $D9, $49, $B1, $5E, $8B, $E1, $69, $D0, $DB, $05, $12, $B9, $71, $17, $85, $AC, $A8, $98, $0D, $D7, $65, $3C ), {AD}
    ( $23, $0E, $C6, $29, $D7, $7B, $B3, $F2, $43, $11, $5B, $99, $1D, $29, $7C, $B2, $A2, $F9, $55, $79, $2C, $53, $C7, $6F, $48, $A7, $67, $28, $EB, $E2, $5B, $A7, $7C, $E6, $62, $A4, $05, $38, $44, $00, $DD, $C0, $6B, $7E, $6B, $F4, $9D, $66 ), {AE}
    ( $20, $DD, $B9, $BD, $76, $44, $41, $0B, $05, $63, $91, $B1, $FA, $2E, $8C, $06, $CA, $4E, $DE, $51, $CF, $16, $7C, $00, $46, $60, $2B, $55, $05, $36, $F8, $70, $50, $40, $67, $25, $97, $C2, $1F, $F4, $0A, $F8, $EC, $6E, $8A, $FB, $84, $4B ), {AF}
    ( $00, $23, $C5, $74, $92, $51, $B8, $83, $33, $5A, $4F, $86, $D6, $6B, $7E, $00, $AE, $35, $3D, $ED, $3E, $FA, $CE, $8F, $3F, $C8, $05, $26, $D6, $7B, $35, $DE, $0D, $90, $78, $FB, $DA, $80, $BC, $53, $46, $79, $00, $DF, $F3, $FE, $4C, $14 ), {B0}
    ( $0F, $9C, $B2, $BE, $6A, $44, $81, $13, $E3, $8D, $54, $1B, $6A, $9A, $58, $29, $67, $39, $53, $DA, $F3, $54, $FC, $0E, $3C, $81, $8A, $27, $7F, $85, $69, $E9, $8D, $16, $EA, $77, $DB, $12, $2A, $3B, $E4, $0A, $86, $03, $18, $B6, $EA, $84 ), {B1}
    ( $78, $CE, $11, $F4, $2D, $7D, $5E, $50, $84, $F7, $6D, $FF, $19, $9C, $99, $8D, $99, $9B, $57, $8E, $3A, $E9, $35, $CB, $D9, $FD, $09, $2C, $1B, $E6, $32, $12, $31, $F3, $3C, $63, $AC, $D3, $16, $D8, $5A, $A0, $80, $30, $B8, $D6, $5C, $0C ), {B2}
    ( $00, $98, $DB, $E1, $9C, $A8, $4F, $E9, $E2, $42, $66, $17, $D1, $14, $21, $37, $63, $C3, $C4, $16, $6A, $78, $E2, $1B, $74, $B1, $45, $35, $3E, $03, $B0, $E4, $F4, $3C, $08, $24, $EA, $E5, $08, $C4, $58, $C1, $E6, $62, $25, $28, $60, $2A ), {B3}
    ( $9E, $27, $EB, $E6, $D1, $42, $6A, $6F, $2A, $6A, $60, $0A, $6B, $5F, $A3, $42, $8F, $F7, $E2, $30, $6B, $A9, $03, $70, $DF, $83, $D9, $1A, $68, $3E, $DD, $DD, $29, $57, $24, $42, $F0, $22, $53, $88, $E9, $CC, $0F, $1B, $64, $37, $32, $0A ), {B4}
    ( $05, $4D, $F3, $80, $E8, $96, $06, $4E, $FA, $B8, $1A, $4A, $A3, $AD, $88, $A4, $F8, $74, $26, $48, $6C, $CA, $15, $6F, $BB, $1B, $3C, $82, $37, $47, $29, $60, $7E, $C0, $B8, $7C, $F7, $3F, $96, $0A, $5C, $57, $D7, $E6, $47, $0F, $78, $08 ), {B5}
    ( $57, $58, $E1, $03, $AC, $61, $4A, $1A, $76, $6A, $EE, $86, $F8, $13, $58, $DF, $20, $3F, $BA, $51, $DC, $74, $39, $6A, $78, $C9, $3D, $F9, $69, $C5, $72, $1F, $E6, $9E, $32, $E2, $30, $19, $65, $97, $E2, $87, $C6, $CE, $CD, $8A, $B9, $5B ), {B6}
    ( $2A, $06, $A7, $C1, $0C, $0D, $CC, $97, $99, $D5, $29, $82, $68, $A6, $74, $5F, $F2, $D8, $18, $BB, $77, $48, $58, $B3, $D5, $2A, $82, $0D, $4F, $64, $D8, $86, $2F, $80, $8E, $F8, $7A, $26, $39, $81, $BB, $91, $20, $6E, $63, $47, $C6, $76 ), {B7}
    ( $08, $47, $C6, $D7, $1C, $E0, $C7, $46, $86, $FD, $45, $1B, $44, $7C, $1E, $11, $C2, $06, $23, $B0, $E2, $85, $6F, $CC, $3A, $DD, $FA, $2D, $03, $98, $18, $1E, $67, $36, $A0, $A0, $6B, $33, $6B, $46, $D1, $C7, $0A, $EE, $B2, $B1, $25, $7D ), {B8}
    ( $56, $33, $26, $0D, $14, $1A, $97, $76, $D5, $30, $80, $5F, $59, $6C, $A3, $DB, $8C, $E3, $3E, $F6, $94, $37, $CE, $46, $F6, $2D, $54, $E9, $7E, $74, $70, $88, $DF, $5C, $93, $18, $48, $9B, $45, $EA, $A4, $AA, $D2, $9F, $0B, $A8, $50, $CA ), {B9}
    ( $BD, $BD, $7B, $16, $76, $7F, $6D, $9F, $F7, $96, $84, $27, $F1, $B7, $B6, $DD, $58, $C7, $65, $99, $B3, $52, $76, $EE, $28, $6F, $4C, $7F, $6C, $AD, $D7, $91, $81, $88, $C0, $40, $17, $42, $11, $7B, $CE, $C4, $F1, $96, $42, $66, $D1, $63 ), {BA}
    ( $97, $E4, $E8, $A6, $B5, $13, $5B, $24, $8A, $8B, $D7, $85, $E5, $29, $79, $77, $45, $45, $C1, $A0, $97, $5B, $C5, $BB, $13, $FA, $E3, $BD, $9F, $59, $E3, $7D, $AF, $D5, $62, $7C, $0E, $91, $DE, $2B, $A2, $23, $AC, $77, $84, $74, $E1, $A9 ), {BB}
    ( $DE, $1B, $F1, $EA, $F8, $6C, $6B, $3B, $A2, $46, $A3, $AC, $D5, $00, $35, $FE, $6F, $80, $17, $9D, $D9, $6A, $21, $CD, $3F, $8D, $B7, $CB, $17, $30, $0D, $03, $49, $7A, $79, $8B, $5D, $94, $50, $6C, $AD, $52, $DC, $C6, $F6, $1A, $E8, $41 ), {BC}
    ( $F4, $A4, $E1, $D0, $8E, $1F, $44, $0B, $5E, $27, $63, $3C, $D5, $64, $22, $E0, $14, $65, $C1, $4F, $1D, $B4, $14, $20, $9A, $93, $90, $43, $98, $8D, $37, $C2, $CB, $E6, $5C, $FA, $24, $5D, $B3, $68, $63, $40, $AE, $DE, $28, $DD, $A8, $55 ), {BD}
    ( $1F, $7A, $B6, $5A, $3F, $89, $24, $54, $D7, $0A, $B4, $16, $7E, $BE, $B5, $A1, $9B, $26, $31, $E8, $24, $C2, $02, $8D, $D5, $D9, $7B, $DE, $E3, $15, $19, $BC, $EA, $2D, $C7, $74, $49, $E4, $05, $8C, $EB, $20, $4F, $2D, $6D, $2F, $BA, $FF ), {BE}
    ( $65, $37, $E6, $91, $71, $A2, $66, $5D, $3F, $D2, $F8, $35, $43, $5A, $3F, $23, $AD, $D5, $DD, $3E, $62, $2D, $6C, $8A, $C5, $22, $CD, $D5, $E5, $E2, $43, $F8, $5A, $EC, $27, $F3, $DB, $FD, $A8, $A2, $47, $7A, $65, $ED, $57, $0E, $14, $45 ), {BF}
    ( $3B, $A7, $CB, $01, $D3, $2E, $9D, $63, $9E, $33, $57, $34, $E7, $B5, $41, $6B, $0E, $D9, $6A, $84, $F9, $45, $39, $F6, $45, $CE, $E2, $E4, $6D, $F5, $A7, $0D, $DE, $14, $2E, $E1, $E9, $AF, $EC, $1C, $78, $D6, $12, $1C, $4F, $DC, $72, $DD ), {C0}
    ( $7B, $B3, $0A, $F6, $53, $39, $0B, $77, $2D, $39, $4F, $2B, $7F, $8F, $7B, $B6, $02, $77, $A3, $C2, $13, $AF, $34, $89, $7D, $F6, $E6, $74, $DD, $56, $D0, $84, $56, $43, $CD, $30, $73, $C4, $24, $51, $FA, $B1, $5F, $8B, $D1, $A1, $DC, $18 ), {C1}
    ( $42, $B4, $53, $AB, $F5, $15, $0D, $8B, $91, $3F, $10, $9C, $11, $88, $E1, $8C, $C2, $7B, $B7, $63, $1F, $B4, $3B, $F9, $EB, $DD, $E6, $85, $EF, $10, $84, $19, $76, $D6, $7C, $87, $C5, $6D, $33, $EA, $95, $EC, $73, $C0, $AF, $40, $F0, $84 ), {C2}
    ( $BC, $E4, $3D, $59, $A1, $F5, $0B, $FB, $BA, $70, $27, $CA, $04, $D8, $46, $00, $FB, $6F, $DB, $98, $A2, $BE, $64, $4B, $D5, $DE, $77, $7E, $99, $3D, $ED, $4A, $FC, $A3, $9F, $1E, $DF, $71, $0F, $3A, $A5, $E5, $89, $3C, $85, $8D, $88, $41 ), {C3}
    ( $C6, $8A, $C7, $76, $E6, $AE, $AC, $FC, $53, $80, $67, $C7, $86, $61, $06, $EB, $D2, $7B, $4A, $35, $2F, $4E, $FD, $E3, $84, $7D, $A2, $B3, $BF, $01, $E3, $78, $3C, $79, $E3, $C1, $36, $92, $6D, $58, $F9, $57, $BC, $87, $26, $AA, $16, $10 ), {C4}
    ( $95, $49, $2C, $42, $03, $C7, $C6, $12, $0D, $D6, $0D, $B1, $EE, $83, $21, $FC, $E1, $D9, $EB, $A9, $02, $F6, $2B, $42, $EA, $2D, $BF, $7D, $0E, $37, $A4, $F2, $E1, $1F, $B9, $09, $8B, $F5, $DA, $48, $DB, $FE, $21, $3F, $81, $8E, $A3, $38 ), {C5}
    ( $17, $CB, $21, $31, $6D, $47, $56, $DD, $B8, $89, $52, $49, $81, $40, $14, $6A, $64, $81, $12, $F5, $80, $84, $42, $88, $49, $47, $AD, $C3, $F7, $D5, $8F, $35, $65, $1C, $CE, $28, $E2, $6A, $53, $77, $0B, $38, $03, $DA, $F3, $37, $F8, $9B ), {C6}
    ( $BE, $AB, $16, $E2, $DC, $E6, $B6, $E3, $8F, $39, $EC, $C8, $E3, $91, $72, $DF, $60, $7C, $C9, $55, $3F, $F2, $9C, $0E, $4B, $FD, $15, $15, $4F, $4F, $0B, $A7, $EE, $62, $30, $B6, $BD, $40, $8C, $E4, $35, $B6, $54, $11, $0D, $16, $4E, $99 ), {C7}
    ( $AD, $DD, $FF, $1B, $D2, $C1, $1C, $D4, $2A, $1A, $26, $2C, $BA, $6E, $1A, $A0, $0B, $F2, $29, $1D, $09, $47, $5A, $46, $4C, $93, $A0, $AB, $AD, $F4, $DE, $32, $73, $EE, $8E, $13, $27, $33, $3E, $63, $F3, $AE, $20, $31, $F5, $D1, $3B, $28 ), {C8}
    ( $24, $6C, $7C, $AB, $B2, $D9, $A5, $5C, $50, $E9, $C7, $28, $2C, $1E, $E0, $F6, $2F, $BD, $A0, $95, $65, $A0, $D3, $D7, $19, $65, $52, $67, $9C, $04, $A4, $EB, $13, $7C, $66, $DA, $29, $A6, $DD, $82, $08, $A7, $6B, $6B, $4B, $DA, $56, $BF ), {C9}
    ( $7C, $A3, $C5, $9B, $E3, $E2, $86, $10, $6A, $DD, $75, $CF, $1F, $7A, $E2, $48, $01, $74, $74, $50, $73, $7A, $64, $35, $A1, $F2, $25, $9C, $B2, $B4, $92, $3B, $E0, $C8, $F5, $5E, $8E, $CE, $72, $10, $D7, $96, $43, $98, $F3, $50, $B6, $9B ), {CA}
    ( $E0, $45, $86, $4E, $D1, $82, $51, $01, $AC, $54, $96, $91, $93, $E1, $A1, $C5, $23, $D8, $5A, $93, $4D, $07, $94, $C7, $B4, $FA, $88, $CB, $73, $4A, $42, $13, $7C, $5C, $BF, $D6, $BD, $A3, $D5, $F9, $66, $60, $7F, $E9, $38, $74, $88, $25 ), {CB}
    ( $BA, $F3, $6F, $D2, $A1, $80, $D4, $81, $EA, $C4, $40, $AC, $1B, $95, $98, $F7, $9A, $A2, $4D, $80, $FF, $B7, $B0, $6C, $79, $60, $1F, $51, $73, $58, $F1, $63, $D1, $07, $18, $31, $41, $8B, $B6, $3B, $81, $96, $09, $A6, $AE, $7D, $3A, $03 ), {CC}
    ( $3E, $91, $52, $D8, $CD, $BA, $E5, $51, $86, $AD, $79, $3F, $20, $3D, $D0, $16, $BE, $3A, $EB, $77, $8A, $D4, $A8, $91, $28, $10, $25, $4D, $D7, $6B, $66, $18, $9B, $5D, $CD, $E3, $66, $36, $C3, $27, $0A, $8A, $AD, $65, $86, $8B, $C5, $8C ), {CD}
    ( $6D, $06, $72, $78, $0D, $93, $15, $2A, $EE, $E7, $05, $24, $7B, $82, $80, $91, $9E, $BD, $B9, $76, $F1, $37, $46, $3F, $A7, $DE, $3E, $73, $A2, $D0, $C1, $BF, $F8, $71, $A0, $0B, $A0, $04, $6A, $C7, $48, $4C, $96, $A8, $03, $F2, $34, $86 ), {CE}
    ( $0F, $C7, $BC, $DA, $BB, $06, $BF, $FB, $F7, $5C, $3F, $FB, $3D, $63, $09, $B3, $EC, $A3, $05, $D1, $03, $10, $91, $62, $37, $3F, $50, $3B, $20, $4F, $FF, $61, $CE, $33, $2C, $9F, $54, $96, $3F, $A2, $9A, $44, $20, $A5, $22, $42, $CD, $B4 ), {CF}
    ( $C7, $1D, $48, $11, $79, $D1, $98, $C1, $50, $5A, $28, $45, $CE, $E9, $25, $69, $F3, $39, $BF, $F6, $DD, $67, $55, $B5, $8B, $EA, $D5, $2B, $8D, $E8, $92, $45, $4B, $68, $6E, $65, $92, $0D, $CA, $2B, $99, $59, $3F, $A4, $3E, $E6, $8A, $37 ), {D0}
    ( $D9, $0A, $68, $D7, $17, $E6, $15, $01, $9B, $B9, $20, $AE, $A1, $91, $61, $A6, $2F, $3D, $6F, $96, $D9, $0E, $B1, $E4, $DF, $15, $EC, $BA, $10, $51, $3D, $7D, $E6, $E5, $D5, $39, $B4, $F0, $18, $31, $C7, $D1, $7A, $75, $28, $FE, $CE, $36 ), {D1}
    ( $A0, $4F, $F0, $BE, $B4, $EB, $FB, $AF, $E5, $E9, $0A, $5B, $3D, $DA, $A3, $CA, $84, $53, $54, $22, $09, $F4, $A1, $45, $80, $A6, $FF, $D7, $2B, $B5, $A7, $07, $14, $E0, $C4, $70, $5A, $1A, $BF, $6A, $D6, $99, $EC, $1F, $C1, $8A, $67, $7D ), {D2}
    ( $70, $21, $A1, $24, $E3, $18, $15, $75, $DC, $7A, $AE, $28, $17, $AD, $94, $5F, $8B, $B5, $52, $1E, $7F, $0D, $56, $5A, $66, $71, $D3, $79, $2F, $08, $05, $EE, $D3, $88, $8E, $A3, $94, $41, $3A, $1A, $CE, $4D, $7E, $47, $B5, $5B, $F9, $CC ), {D3}
    ( $22, $F4, $40, $26, $3C, $AA, $DE, $68, $E7, $7B, $B2, $87, $77, $2E, $AC, $7B, $29, $49, $37, $75, $96, $2A, $40, $E9, $1E, $06, $A2, $7F, $A6, $8C, $B9, $1B, $DD, $EF, $02, $93, $2A, $BD, $B9, $C7, $79, $F0, $3B, $88, $DC, $17, $52, $33 ), {D4}
    ( $65, $F6, $D5, $17, $B5, $3E, $23, $91, $97, $DB, $65, $A2, $F0, $0B, $1C, $39, $1D, $77, $AE, $9B, $85, $AA, $48, $55, $19, $13, $3B, $9B, $3E, $9B, $07, $71, $63, $76, $D9, $F1, $1A, $7D, $B3, $D4, $94, $9A, $D0, $2F, $5A, $E1, $61, $84 ), {D5}
    ( $FE, $44, $34, $CD, $E0, $9D, $92, $3B, $03, $B0, $FC, $FD, $71, $3B, $70, $52, $2D, $71, $32, $90, $D4, $A6, $72, $38, $2B, $56, $94, $6F, $F6, $29, $EE, $96, $60, $A1, $5D, $01, $B2, $B3, $C4, $28, $0B, $1D, $5E, $AF, $79, $39, $33, $A0 ), {D6}
    ( $BC, $40, $FC, $FB, $0E, $0D, $49, $4B, $A3, $1C, $46, $48, $C7, $B3, $D1, $DE, $F1, $11, $3C, $21, $9A, $07, $EC, $8D, $23, $78, $BE, $B1, $A5, $C2, $BD, $1C, $19, $0C, $C3, $47, $80, $70, $A1, $94, $63, $DA, $B6, $E1, $CC, $F5, $63, $29 ), {D7}
    ( $90, $1B, $6B, $9E, $82, $BA, $BF, $91, $87, $2A, $23, $4C, $45, $D6, $10, $01, $6C, $A4, $6A, $95, $C1, $CC, $6D, $6C, $22, $77, $93, $15, $E0, $F0, $22, $95, $60, $A5, $93, $96, $34, $6B, $E6, $AC, $FB, $67, $A5, $03, $CB, $48, $88, $46 ), {D8}
    ( $50, $D4, $40, $F7, $4C, $97, $66, $0B, $E7, $1E, $CA, $BF, $64, $ED, $FE, $0C, $80, $20, $1B, $89, $57, $18, $CE, $22, $A0, $5D, $89, $80, $4D, $35, $D3, $06, $8F, $70, $04, $02, $A2, $B0, $D0, $86, $32, $6F, $CB, $33, $4C, $A4, $DF, $C0 ), {D9}
    ( $BC, $FB, $D0, $2E, $A0, $05, $CD, $D5, $F0, $22, $5A, $46, $75, $55, $31, $15, $08, $E1, $8B, $36, $92, $A7, $AF, $62, $05, $D3, $4A, $82, $0C, $8C, $ED, $0A, $51, $A8, $D7, $CE, $C3, $3E, $80, $EA, $0A, $C0, $07, $50, $3F, $AE, $87, $9C ), {DA}
    ( $F4, $3E, $EF, $B5, $C8, $3C, $52, $1A, $E5, $E9, $B0, $5F, $C4, $88, $41, $AC, $79, $C5, $2C, $38, $BF, $85, $B5, $F9, $26, $CD, $08, $18, $AE, $3B, $F7, $A9, $4F, $38, $5C, $32, $CA, $8F, $5F, $74, $F1, $7B, $22, $10, $7B, $95, $47, $52 ), {DB}
    ( $1A, $48, $FC, $96, $91, $98, $A4, $B0, $D9, $A7, $89, $40, $BB, $0C, $4E, $1C, $42, $78, $1D, $9B, $E6, $0E, $76, $91, $87, $D1, $CA, $F3, $68, $0F, $8A, $30, $D0, $9F, $F1, $93, $60, $6A, $AF, $29, $45, $18, $DA, $BC, $60, $04, $87, $93 ), {DC}
    ( $F0, $5D, $48, $13, $4A, $56, $A0, $34, $89, $A6, $5E, $EB, $91, $DC, $69, $B9, $8F, $C7, $F4, $39, $60, $E6, $3C, $62, $FA, $1C, $6B, $9F, $F9, $41, $5E, $92, $7E, $21, $9D, $4E, $56, $34, $79, $35, $2B, $6A, $48, $D6, $DE, $0A, $EF, $85 ), {DD}
    ( $1A, $7F, $F9, $C5, $4B, $04, $5F, $FD, $44, $A0, $A9, $56, $2E, $94, $68, $B2, $F1, $14, $25, $A2, $2D, $1E, $BF, $92, $20, $8D, $33, $12, $0B, $D2, $8E, $0E, $F2, $D7, $41, $97, $AF, $80, $E1, $62, $CE, $ED, $CA, $73, $DF, $E6, $6C, $93 ), {DE}
    ( $D5, $71, $90, $43, $9D, $29, $C9, $A4, $44, $C0, $07, $DC, $2B, $5E, $AF, $9D, $EF, $6D, $DF, $48, $A7, $80, $CE, $DC, $61, $B2, $05, $E4, $A9, $60, $24, $B1, $18, $85, $B6, $CE, $84, $C3, $FE, $5D, $B8, $B5, $69, $86, $B6, $E2, $CE, $21 ), {DF}
    ( $F3, $6D, $AC, $FA, $34, $23, $7E, $99, $BE, $45, $EB, $52, $53, $BC, $FE, $D0, $40, $2C, $69, $46, $B8, $B2, $1A, $C0, $24, $60, $A6, $FC, $E7, $E9, $CD, $67, $F8, $9A, $6D, $5B, $16, $26, $29, $FC, $F6, $6C, $CE, $A3, $74, $DB, $82, $1E ), {E0}
    ( $16, $E0, $60, $74, $DC, $C3, $1A, $1D, $F1, $72, $01, $7A, $C3, $FA, $38, $C3, $BB, $C1, $CE, $4B, $B7, $84, $ED, $60, $DA, $89, $A8, $BC, $E8, $2A, $E6, $71, $A6, $DA, $CF, $FB, $8D, $26, $C0, $BB, $18, $51, $81, $AE, $96, $09, $F6, $D6 ), {E1}
    ( $F1, $10, $DB, $DD, $94, $D1, $76, $61, $F5, $9F, $BB, $4C, $BA, $69, $F3, $93, $46, $3B, $60, $FB, $3F, $3C, $5E, $00, $1C, $60, $B8, $96, $FE, $8E, $78, $AC, $5E, $B3, $E2, $67, $95, $DE, $5A, $B6, $99, $73, $28, $D4, $65, $4D, $62, $19 ), {E2}
    ( $21, $06, $91, $18, $AB, $E2, $4B, $61, $81, $1C, $B8, $C4, $8F, $CE, $FC, $6A, $48, $3B, $03, $2C, $FB, $56, $F9, $02, $FB, $32, $E8, $48, $19, $8C, $C0, $57, $A6, $20, $81, $54, $62, $A0, $4F, $70, $90, $00, $38, $D1, $89, $49, $59, $E2 ), {E3}
    ( $5A, $D5, $09, $78, $9B, $FF, $EC, $D0, $DD, $CD, $5E, $83, $25, $F6, $9C, $A0, $15, $4D, $8F, $1A, $CD, $9B, $8C, $82, $AC, $7D, $F7, $5E, $94, $CE, $3C, $AF, $6D, $65, $54, $D1, $B3, $87, $54, $BE, $B5, $DB, $64, $AF, $73, $84, $86, $E7 ), {E4}
    ( $35, $A3, $08, $A1, $AC, $9A, $43, $BF, $26, $47, $80, $5A, $B3, $E6, $E4, $92, $4B, $B7, $4A, $61, $6F, $61, $58, $8F, $FA, $46, $02, $EE, $5B, $DB, $F5, $4E, $3F, $DD, $62, $47, $0A, $71, $74, $DB, $57, $95, $43, $3C, $A8, $08, $FA, $AC ), {E5}
    ( $51, $A0, $94, $B8, $77, $4C, $A6, $05, $5F, $07, $97, $4C, $74, $EE, $F2, $25, $02, $2A, $FE, $F7, $AD, $81, $A9, $53, $09, $67, $C4, $4B, $BA, $33, $6F, $D6, $8A, $A3, $27, $91, $8A, $EC, $BA, $3D, $F7, $0B, $84, $36, $57, $3C, $3F, $0A ), {E6}
    ( $CF, $37, $4F, $83, $42, $07, $66, $C3, $71, $F3, $19, $01, $A1, $3E, $F0, $7C, $63, $AD, $56, $C7, $DE, $F9, $DC, $0F, $9E, $5B, $B5, $E8, $59, $F5, $A2, $31, $D0, $BF, $45, $3B, $B9, $89, $3E, $4C, $A1, $E1, $4B, $66, $C2, $71, $97, $60 ), {E7}
    ( $B4, $18, $61, $CC, $73, $FD, $3E, $48, $46, $1D, $79, $A1, $38, $B0, $4B, $E1, $40, $10, $D3, $7D, $37, $FB, $A8, $17, $7D, $96, $22, $AA, $69, $32, $25, $A4, $22, $04, $45, $4B, $81, $26, $79, $9A, $33, $A5, $D4, $87, $DC, $CD, $6E, $B6 ), {E8}
    ( $D2, $91, $D0, $31, $7A, $05, $33, $20, $E2, $76, $78, $F1, $E5, $0D, $1F, $76, $9A, $3D, $66, $3A, $63, $15, $9F, $C7, $AD, $7B, $4D, $3F, $67, $BA, $B4, $52, $26, $9C, $C0, $5E, $2B, $33, $CE, $1C, $0F, $B8, $26, $1C, $D7, $34, $BC, $C3 ), {E9}
    ( $F3, $D0, $54, $6D, $3D, $4A, $25, $EE, $B4, $28, $74, $AD, $28, $C9, $B7, $F2, $73, $EC, $78, $8B, $29, $96, $2D, $28, $4A, $E7, $3A, $48, $13, $2B, $85, $53, $75, $6C, $99, $D7, $A0, $91, $0B, $66, $EC, $A7, $E2, $C2, $71, $2D, $55, $5C ), {EA}
    ( $55, $9F, $A5, $BF, $24, $91, $1F, $DD, $A1, $DD, $F5, $DE, $37, $70, $55, $4B, $C7, $C3, $FD, $13, $93, $66, $B9, $46, $6E, $7E, $CC, $0C, $88, $1D, $2B, $A4, $14, $E7, $6D, $6A, $27, $E5, $4B, $87, $73, $52, $D5, $FB, $C4, $FA, $B8, $78 ), {EB}
    ( $F1, $9A, $62, $2B, $ED, $8D, $AC, $0A, $35, $54, $8E, $5D, $7E, $FC, $5A, $2E, $CA, $C8, $49, $74, $B4, $F0, $57, $B2, $AB, $31, $7E, $D0, $3D, $03, $35, $AE, $71, $0F, $C1, $38, $F2, $C5, $17, $38, $9C, $90, $CC, $49, $5A, $40, $34, $16 ), {EC}
    ( $9F, $A7, $DE, $B9, $36, $F1, $04, $61, $A1, $52, $9B, $0B, $58, $46, $2F, $9D, $9F, $10, $91, $11, $C8, $B9, $EC, $65, $23, $A3, $EB, $28, $44, $4E, $33, $EA, $55, $40, $84, $CA, $75, $11, $89, $37, $59, $9D, $58, $A7, $C9, $46, $EA, $C2 ), {ED}
    ( $6E, $C3, $AA, $BB, $78, $56, $AC, $4E, $98, $0E, $69, $07, $C1, $CB, $CC, $AF, $1F, $85, $57, $AD, $C7, $00, $CB, $F5, $7D, $CB, $1C, $E0, $AF, $48, $D9, $F4, $7F, $B3, $DA, $DF, $81, $99, $AB, $8A, $E6, $B3, $6D, $B8, $FA, $DB, $F3, $12 ), {EE}
    ( $C0, $0F, $0D, $3F, $7A, $10, $16, $60, $60, $5B, $94, $B1, $2D, $B6, $C6, $97, $79, $94, $4F, $7B, $A2, $B6, $5F, $38, $40, $85, $8A, $DE, $DD, $47, $E2, $BC, $1E, $04, $4B, $DB, $0E, $9F, $B0, $2B, $86, $C7, $9D, $01, $A3, $10, $95, $39 ), {EF}
    ( $97, $31, $89, $3D, $5B, $98, $48, $2A, $FB, $8D, $E2, $67, $F9, $79, $03, $26, $87, $80, $F4, $07, $14, $3A, $50, $5D, $A4, $1C, $AE, $FC, $CC, $D3, $A8, $E3, $A0, $42, $F0, $B3, $D7, $B7, $A7, $FE, $3E, $31, $51, $FE, $BB, $19, $A1, $AC ), {F0}
    ( $E7, $ED, $F6, $79, $00, $3A, $69, $50, $BA, $FC, $97, $D4, $A8, $C6, $AB, $12, $13, $C0, $96, $B4, $9C, $79, $55, $9A, $C3, $05, $25, $01, $43, $4B, $50, $19, $12, $80, $FB, $23, $E7, $AD, $FB, $09, $19, $59, $90, $5D, $31, $BD, $2F, $C0 ), {F1}
    ( $57, $5C, $0C, $46, $FC, $FC, $C6, $5B, $FE, $62, $5E, $87, $3F, $34, $B4, $19, $16, $96, $FD, $CC, $7F, $51, $B8, $A3, $C7, $9C, $56, $F3, $0E, $5A, $E7, $C0, $14, $E3, $46, $1C, $D2, $7F, $AD, $15, $1B, $7B, $CC, $B9, $CB, $47, $28, $59 ), {F2}
    ( $38, $06, $FE, $58, $E5, $CC, $8F, $16, $F8, $24, $4E, $D7, $67, $34, $C1, $BF, $4E, $04, $94, $0E, $0F, $5D, $DB, $56, $5B, $D0, $AF, $DD, $C4, $15, $8B, $7B, $A4, $C6, $BA, $94, $99, $11, $C5, $C9, $FF, $6E, $2A, $C1, $55, $AE, $97, $26 ), {F3}
    ( $49, $C7, $C8, $44, $B8, $11, $41, $44, $B4, $50, $E4, $1B, $CA, $35, $CB, $00, $30, $24, $50, $EC, $67, $BE, $F9, $7C, $A8, $66, $20, $49, $DB, $1E, $0D, $8B, $DA, $69, $C0, $22, $52, $8E, $B8, $FA, $6A, $BB, $F1, $65, $85, $C1, $A2, $F7 ), {F4}
    ( $37, $BB, $42, $0D, $F6, $7F, $04, $4E, $DC, $C0, $E9, $F3, $E2, $EF, $07, $B3, $4D, $10, $08, $86, $18, $77, $78, $41, $04, $92, $E5, $37, $93, $05, $DA, $AE, $3D, $A4, $79, $1C, $37, $E4, $12, $8F, $80, $68, $84, $45, $CB, $A4, $EA, $17 ), {F5}
    ( $51, $39, $8A, $7C, $E4, $CF, $8D, $9D, $49, $A5, $FC, $D8, $91, $A6, $9C, $A5, $3D, $72, $A6, $0E, $C2, $39, $2D, $A5, $0E, $82, $96, $B8, $79, $AB, $55, $39, $6B, $CB, $00, $AF, $2E, $DC, $0B, $DE, $BE, $B9, $38, $48, $E5, $4B, $3E, $90 ), {F6}
    ( $7A, $D7, $C5, $2A, $18, $92, $2E, $19, $29, $29, $2C, $57, $C4, $F5, $B8, $F5, $F0, $CF, $1F, $98, $A5, $77, $C1, $0B, $07, $2B, $9F, $29, $3B, $B6, $60, $CD, $09, $B8, $60, $4F, $55, $75, $B6, $FB, $DE, $CB, $39, $6A, $81, $B9, $FC, $DB ), {F7}
    ( $25, $4A, $D7, $AD, $B4, $C2, $20, $DE, $6C, $62, $E2, $0F, $95, $A0, $07, $0D, $AD, $EB, $89, $F3, $39, $30, $9B, $D8, $A2, $F6, $85, $CC, $17, $8B, $28, $9F, $93, $43, $90, $5B, $5D, $EE, $95, $A5, $E0, $C3, $0F, $34, $A2, $97, $7C, $86 ), {F8}
    ( $66, $9C, $D5, $1A, $F7, $CF, $BF, $AA, $E3, $E0, $80, $6F, $68, $80, $27, $1D, $69, $34, $C2, $59, $E0, $98, $BF, $90, $5D, $FE, $EA, $F0, $FB, $CA, $72, $49, $89, $F7, $4B, $94, $8B, $41, $18, $B6, $53, $64, $0A, $EA, $FB, $68, $07, $C3 ), {F9}
    ( $DD, $3B, $AC, $DC, $C0, $4B, $E1, $20, $6D, $49, $49, $BD, $64, $19, $8E, $51, $31, $FD, $B3, $96, $66, $59, $8A, $74, $BB, $BC, $6D, $E9, $C0, $C1, $5A, $81, $F2, $7F, $20, $1C, $61, $C0, $62, $79, $27, $38, $AF, $E3, $E8, $4E, $5C, $DD ), {FA}
    ( $CD, $D7, $1F, $D3, $5A, $64, $11, $DE, $3C, $C0, $12, $79, $3E, $87, $52, $3F, $B0, $CF, $F8, $72, $0F, $CA, $36, $F3, $93, $E8, $5F, $E0, $73, $00, $F0, $12, $E8, $94, $A0, $85, $26, $3F, $09, $0B, $2D, $F6, $0A, $01, $DA, $FA, $90, $EC ), {FB}
    ( $9D, $A5, $0D, $B1, $EE, $B4, $FA, $DD, $E5, $24, $E4, $9C, $99, $74, $79, $9A, $DE, $09, $FF, $F2, $6A, $24, $CB, $B9, $AF, $9D, $71, $E9, $F3, $AC, $E7, $CD, $EB, $62, $B1, $A6, $25, $66, $EC, $9D, $06, $D0, $2A, $B1, $21, $7D, $35, $53 ), {FC}
    ( $DD, $31, $E6, $39, $1A, $E0, $35, $22, $93, $AC, $D1, $06, $5B, $35, $E9, $15, $F4, $EB, $56, $CC, $03, $E7, $92, $18, $07, $17, $81, $5C, $85, $0C, $97, $F1, $BF, $4F, $6A, $8A, $C0, $54, $0A, $6F, $FC, $F8, $AE, $5D, $E9, $50, $7F, $F0 ), {FD}
    ( $AB, $45, $B4, $13, $DC, $50, $B2, $07, $40, $B4, $17, $36, $95, $51, $D8, $D5, $CA, $32, $28, $6A, $10, $8E, $72, $10, $03, $22, $5E, $54, $D8, $D0, $93, $AF, $4B, $6C, $A5, $59, $1E, $A5, $76, $E9, $4E, $12, $AB, $77, $4D, $C4, $E0, $62 ), {FE}
    ( $D9, $F4, $F8, $50, $DF, $6C, $B9, $6C, $8A, $BA, $D8, $1B, $16, $67, $33, $5D, $CB, $40, $79, $CF, $E7, $9C, $72, $E5, $E5, $54, $2F, $76, $3E, $31, $69, $96, $30, $3E, $4B, $79, $B9, $D3, $97, $C4, $E4, $69, $33, $03, $8B, $94, $51, $11 ), {FF}
    ( $75, $B1, $5C, $C5, $3B, $0D, $25, $02, $DA, $1B, $CA, $6B, $A0, $52, $43, $58, $9E, $DA, $97, $75, $56, $C0, $6B, $7E, $6C, $57, $72, $7E, $CF, $0A, $13, $25, $DC, $61, $3D, $5A, $78, $E5, $C3, $F8, $CE, $06, $2D, $94, $A3, $B4, $94, $5A )  {ZZ}
  );

const
  TABLE_MIX: array[0..255] of TDigestBuf =
  (
    ( $02, $04, $23, $0C, $17, $00, $10, $24, $0E, $27, $05, $1B, $0B, $1D, $08, $20, $29, $1F, $2D, $1A, $2F, $18, $2B, $25, $28, $2E, $0F, $11, $0A, $2A, $06, $1C, $09, $19, $1E, $12, $16, $2C, $07, $14, $26, $01, $21, $03, $22, $13, $0D, $15 ), {00}
    ( $16, $1C, $12, $15, $21, $0A, $22, $02, $0B, $0C, $1D, $08, $2C, $11, $00, $2A, $19, $2B, $10, $28, $09, $04, $13, $07, $24, $2D, $2E, $2F, $29, $26, $0F, $17, $27, $0D, $1B, $1A, $03, $05, $25, $1E, $01, $23, $06, $1F, $14, $0E, $20, $18 ), {01}
    ( $15, $18, $0A, $1F, $2E, $26, $11, $28, $22, $13, $25, $1B, $09, $00, $06, $08, $2C, $14, $21, $2D, $27, $2B, $02, $29, $07, $2F, $23, $0B, $2A, $0C, $16, $19, $03, $1D, $1C, $05, $0D, $17, $1A, $0F, $1E, $04, $24, $10, $01, $20, $12, $0E ), {02}
    ( $27, $10, $18, $1E, $09, $0B, $1A, $13, $21, $0D, $12, $2A, $2B, $29, $11, $2E, $2F, $25, $02, $08, $1B, $05, $03, $1F, $26, $16, $06, $0A, $19, $2C, $14, $1C, $17, $0C, $0E, $07, $20, $22, $15, $01, $0F, $24, $04, $00, $1D, $28, $2D, $23 ), {03}
    ( $0A, $0D, $08, $2A, $11, $28, $29, $1A, $1C, $05, $21, $25, $13, $07, $2E, $18, $17, $1B, $0C, $2C, $23, $1F, $1E, $2F, $06, $24, $0F, $00, $1D, $10, $03, $04, $01, $09, $26, $2B, $19, $20, $12, $0E, $16, $2D, $0B, $22, $14, $15, $27, $02 ), {04}
    ( $2B, $25, $11, $1C, $1A, $12, $15, $1F, $0F, $29, $13, $04, $07, $06, $1B, $0D, $0E, $00, $19, $09, $1D, $2F, $21, $01, $26, $27, $16, $03, $2E, $2D, $24, $18, $2A, $0B, $10, $2C, $22, $02, $05, $20, $08, $14, $0C, $1E, $0A, $17, $23, $28 ), {05}
    ( $1A, $1D, $18, $27, $09, $26, $21, $06, $0A, $2C, $2A, $25, $0E, $08, $01, $22, $0B, $1E, $05, $2F, $0F, $2E, $00, $16, $0D, $0C, $28, $20, $23, $1F, $10, $2B, $07, $1C, $24, $19, $2D, $13, $03, $04, $11, $15, $12, $1B, $14, $02, $29, $17 ), {06}
    ( $2D, $02, $14, $1E, $1D, $2B, $22, $18, $07, $15, $11, $0E, $10, $00, $1A, $0D, $09, $16, $23, $0B, $13, $25, $20, $12, $2E, $1B, $2C, $01, $03, $08, $1C, $05, $21, $17, $28, $0C, $06, $2F, $29, $1F, $0A, $2A, $0F, $24, $26, $19, $04, $27 ), {07}
    ( $10, $0E, $19, $23, $1A, $13, $14, $0C, $22, $1D, $2E, $15, $2A, $28, $29, $17, $1F, $08, $06, $09, $0B, $27, $04, $24, $02, $12, $21, $05, $16, $2C, $00, $26, $0F, $1E, $2D, $2F, $0A, $18, $1B, $2B, $03, $25, $0D, $07, $11, $20, $01, $1C ), {08}
    ( $29, $10, $0F, $13, $22, $06, $00, $11, $0E, $03, $20, $18, $1F, $2A, $19, $2F, $0B, $02, $2B, $23, $04, $07, $2C, $25, $08, $21, $24, $1C, $01, $0C, $17, $1D, $1A, $2E, $12, $15, $26, $09, $2D, $1E, $05, $0D, $27, $28, $1B, $0A, $16, $14 ), {09}
    ( $21, $25, $29, $2B, $0F, $0E, $09, $14, $20, $1E, $24, $0C, $15, $10, $03, $0B, $02, $01, $06, $28, $08, $1F, $11, $04, $2F, $18, $19, $2E, $0A, $22, $27, $17, $23, $07, $1A, $2C, $1C, $00, $0D, $2A, $1D, $26, $12, $2D, $13, $1B, $05, $16 ), {0A}
    ( $2A, $22, $15, $1D, $0E, $24, $16, $03, $09, $19, $2C, $01, $26, $1F, $27, $18, $25, $20, $14, $17, $1E, $2B, $0D, $06, $05, $0C, $29, $11, $0B, $00, $1B, $0F, $10, $02, $28, $2F, $2D, $08, $1A, $2E, $12, $21, $04, $07, $1C, $0A, $23, $13 ), {0B}
    ( $1E, $1B, $04, $12, $17, $2A, $24, $05, $03, $26, $0B, $2F, $29, $2E, $0A, $08, $27, $02, $2D, $21, $19, $0C, $0F, $1D, $1C, $0D, $0E, $06, $00, $14, $16, $13, $1F, $22, $2C, $28, $11, $01, $25, $07, $09, $18, $23, $15, $20, $10, $2B, $1A ), {0C}
    ( $29, $0C, $22, $19, $14, $09, $18, $1D, $1C, $28, $2E, $0D, $05, $13, $26, $2B, $0F, $00, $1E, $10, $04, $27, $15, $08, $1B, $1A, $12, $23, $02, $25, $0B, $24, $11, $07, $1F, $0E, $2A, $2F, $17, $0A, $20, $2C, $2D, $01, $21, $16, $03, $06 ), {0D}
    ( $13, $12, $01, $2C, $23, $2A, $21, $27, $1D, $0E, $20, $1C, $28, $2B, $0C, $1B, $07, $22, $0F, $08, $25, $05, $04, $24, $0A, $29, $09, $10, $1F, $19, $02, $2D, $00, $17, $1E, $03, $15, $16, $14, $11, $26, $06, $0B, $2E, $1A, $0D, $2F, $18 ), {0E}
    ( $11, $2A, $1D, $21, $0B, $18, $1B, $2C, $00, $22, $09, $03, $2D, $01, $0F, $0C, $1F, $14, $1C, $1E, $2E, $25, $15, $2F, $19, $0D, $0A, $29, $13, $17, $23, $12, $07, $16, $10, $1A, $20, $05, $06, $04, $27, $2B, $08, $24, $0E, $26, $02, $28 ), {0F}
    ( $08, $28, $1E, $2B, $2C, $24, $0C, $21, $0E, $16, $05, $02, $20, $27, $23, $04, $18, $1A, $25, $00, $10, $14, $09, $0F, $2F, $0B, $1C, $1F, $11, $01, $19, $13, $22, $2D, $07, $26, $15, $1B, $12, $29, $06, $0A, $03, $0D, $2E, $2A, $1D, $17 ), {10}
    ( $17, $0E, $1B, $21, $15, $04, $0B, $0A, $11, $0C, $2F, $06, $22, $09, $05, $1D, $2E, $00, $2A, $2D, $12, $2C, $26, $10, $16, $03, $18, $27, $29, $14, $08, $1A, $19, $1F, $1E, $24, $0F, $0D, $2B, $02, $20, $01, $28, $23, $07, $1C, $25, $13 ), {11}
    ( $23, $19, $0F, $22, $26, $03, $11, $05, $1A, $1F, $28, $16, $2F, $2D, $04, $25, $20, $1C, $1B, $21, $2B, $06, $01, $0C, $1D, $27, $24, $29, $09, $07, $0D, $10, $00, $13, $2E, $0E, $17, $08, $2A, $0A, $02, $14, $18, $12, $1E, $0B, $2C, $15 ), {12}
    ( $11, $0F, $26, $0D, $13, $0C, $19, $23, $14, $2E, $1C, $06, $05, $08, $07, $18, $2A, $1B, $22, $0A, $15, $2C, $17, $29, $1F, $28, $0B, $1E, $27, $1A, $0E, $09, $2D, $25, $10, $02, $01, $1D, $20, $12, $03, $16, $24, $21, $2B, $00, $2F, $04 ), {13}
    ( $15, $0C, $17, $19, $1B, $07, $2B, $05, $1F, $10, $20, $0E, $25, $2C, $2F, $12, $18, $06, $01, $24, $03, $21, $2E, $14, $22, $1E, $0F, $1A, $0B, $13, $23, $29, $2A, $16, $0D, $2D, $09, $1C, $28, $02, $04, $27, $26, $1D, $00, $11, $08, $0A ), {14}
    ( $24, $1F, $2E, $26, $2D, $0C, $25, $15, $21, $17, $1E, $04, $0E, $0A, $0D, $11, $09, $07, $08, $29, $18, $12, $14, $02, $0B, $2B, $20, $27, $16, $2C, $2A, $0F, $03, $1D, $10, $19, $06, $01, $13, $1C, $2F, $28, $00, $22, $23, $05, $1B, $1A ), {15}
    ( $16, $12, $0D, $14, $22, $15, $27, $01, $1C, $00, $1A, $20, $02, $24, $2D, $1F, $1E, $25, $26, $2B, $03, $18, $0C, $29, $28, $23, $1D, $13, $08, $07, $17, $2F, $0E, $0B, $2A, $2E, $21, $1B, $0A, $11, $06, $05, $2C, $0F, $10, $19, $04, $09 ), {16}
    ( $14, $2D, $21, $05, $10, $0E, $2F, $27, $22, $15, $09, $28, $07, $13, $23, $08, $1F, $16, $29, $25, $18, $26, $2A, $03, $1E, $2E, $00, $1C, $0C, $12, $2C, $11, $24, $0A, $0B, $06, $19, $20, $17, $0D, $0F, $01, $1D, $02, $1B, $1A, $2B, $04 ), {17}
    ( $28, $13, $0E, $02, $0A, $19, $1D, $2A, $1C, $25, $08, $09, $1F, $21, $17, $16, $27, $2D, $05, $18, $0F, $22, $20, $2B, $24, $00, $14, $2C, $01, $1A, $0D, $2E, $15, $11, $29, $2F, $04, $0C, $1E, $07, $12, $1B, $0B, $06, $26, $23, $10, $03 ), {18}
    ( $11, $2E, $21, $1A, $20, $07, $28, $18, $09, $29, $17, $2A, $1B, $16, $14, $1D, $26, $0A, $2B, $25, $05, $06, $08, $2D, $0B, $1E, $12, $24, $10, $0E, $03, $15, $0F, $02, $01, $27, $1C, $00, $2C, $19, $1F, $0C, $22, $23, $2F, $13, $04, $0D ), {19}
    ( $14, $26, $1D, $24, $1E, $11, $0A, $16, $00, $18, $20, $2E, $0F, $06, $0C, $1C, $04, $2D, $09, $2A, $0E, $10, $03, $01, $17, $13, $22, $12, $02, $05, $2C, $27, $29, $2B, $23, $1B, $1A, $19, $0B, $2F, $21, $1F, $07, $25, $08, $28, $0D, $15 ), {1A}
    ( $03, $08, $29, $2C, $22, $2D, $02, $12, $2B, $20, $11, $1A, $26, $1F, $1B, $25, $23, $0C, $19, $27, $00, $13, $0E, $1C, $21, $1E, $0D, $15, $14, $16, $04, $2A, $09, $28, $1D, $07, $01, $0F, $17, $18, $2E, $24, $10, $0A, $05, $2F, $06, $0B ), {1B}
    ( $04, $2C, $1F, $08, $27, $1B, $16, $24, $1C, $26, $0E, $13, $23, $21, $00, $14, $18, $07, $20, $06, $05, $1A, $03, $2F, $2A, $12, $1D, $0D, $0F, $2B, $0C, $15, $2E, $2D, $09, $11, $02, $22, $01, $10, $19, $0B, $28, $25, $1E, $17, $0A, $29 ), {1C}
    ( $1F, $0C, $1E, $20, $2A, $2E, $15, $22, $19, $01, $24, $03, $08, $06, $0A, $25, $12, $29, $21, $17, $11, $16, $2F, $13, $07, $1A, $26, $2D, $0D, $27, $04, $1B, $2C, $18, $00, $28, $10, $14, $1D, $09, $0E, $02, $2B, $0F, $0B, $1C, $05, $23 ), {1D}
    ( $24, $29, $27, $0F, $10, $06, $0D, $13, $28, $04, $20, $23, $0A, $25, $1E, $2A, $2B, $0E, $1A, $15, $01, $00, $05, $14, $21, $2D, $09, $2C, $12, $03, $2F, $11, $07, $08, $17, $26, $0C, $18, $2E, $0B, $1D, $19, $1F, $22, $02, $16, $1C, $1B ), {1E}
    ( $28, $24, $09, $29, $2C, $23, $01, $11, $17, $0A, $1B, $07, $02, $04, $08, $06, $0D, $12, $2B, $0F, $13, $1E, $1D, $22, $0E, $05, $20, $16, $0B, $21, $0C, $03, $25, $14, $1C, $2A, $1A, $1F, $19, $2F, $10, $27, $18, $26, $2D, $15, $00, $2E ), {1F}
    ( $1B, $09, $15, $1A, $25, $08, $12, $04, $06, $2B, $1E, $17, $28, $19, $2F, $2D, $18, $14, $0B, $01, $2A, $0D, $10, $20, $07, $29, $0E, $03, $22, $0F, $05, $23, $0A, $11, $2C, $16, $13, $21, $1F, $2E, $1C, $1D, $00, $02, $26, $0C, $24, $27 ), {20}
    ( $11, $29, $1B, $17, $1D, $06, $27, $03, $04, $0D, $19, $0C, $2D, $00, $22, $28, $16, $23, $08, $12, $25, $02, $13, $0B, $1F, $20, $1E, $1C, $09, $05, $21, $24, $2F, $1A, $26, $0F, $07, $14, $15, $2A, $2B, $2E, $0E, $2C, $0A, $01, $10, $18 ), {21}
    ( $2B, $08, $17, $2E, $1E, $16, $19, $06, $0B, $07, $27, $1D, $23, $12, $09, $26, $14, $1B, $00, $15, $2D, $01, $2F, $2A, $0E, $1F, $02, $0C, $0D, $21, $11, $2C, $1A, $13, $18, $05, $1C, $29, $22, $10, $25, $0A, $04, $24, $03, $28, $0F, $20 ), {22}
    ( $1A, $2F, $16, $00, $18, $08, $1C, $2B, $0A, $24, $1D, $17, $29, $21, $0C, $19, $26, $25, $2A, $1E, $07, $1F, $11, $23, $0B, $2C, $03, $2D, $20, $13, $04, $01, $14, $10, $27, $0D, $2E, $0E, $02, $28, $12, $09, $15, $05, $22, $0F, $06, $1B ), {23}
    ( $18, $14, $13, $2A, $0E, $2C, $15, $2D, $24, $0B, $27, $2E, $25, $0F, $2B, $29, $01, $03, $05, $2F, $04, $07, $02, $0C, $16, $23, $11, $17, $21, $28, $1C, $20, $22, $08, $1B, $00, $06, $19, $1D, $0D, $1E, $1A, $0A, $09, $10, $26, $1F, $12 ), {24}
    ( $0C, $22, $14, $06, $2B, $29, $18, $02, $13, $11, $1A, $20, $2C, $1E, $08, $10, $00, $1D, $07, $15, $2E, $12, $1C, $0D, $0A, $1F, $04, $05, $23, $0F, $27, $0B, $19, $25, $03, $09, $17, $21, $28, $2A, $24, $01, $26, $1B, $2D, $16, $2F, $0E ), {25}
    ( $2E, $26, $1F, $23, $15, $28, $2D, $04, $1E, $1C, $10, $07, $29, $09, $02, $16, $14, $13, $11, $0D, $2F, $20, $03, $0B, $00, $18, $27, $22, $05, $21, $12, $08, $1B, $19, $25, $2C, $06, $1A, $17, $2B, $01, $0C, $0E, $2A, $0F, $24, $1D, $0A ), {26}
    ( $0E, $03, $04, $24, $12, $17, $1B, $18, $2A, $27, $28, $0F, $0D, $02, $08, $00, $21, $2C, $16, $1F, $29, $1A, $09, $2B, $06, $14, $11, $1E, $26, $0A, $05, $2F, $15, $22, $2D, $25, $13, $10, $0C, $1C, $0B, $20, $07, $1D, $2E, $19, $01, $23 ), {27}
    ( $28, $05, $03, $1F, $27, $0A, $2E, $0F, $02, $20, $22, $1D, $19, $1B, $0B, $15, $0C, $14, $04, $2C, $16, $21, $2F, $09, $23, $12, $07, $10, $25, $24, $2D, $2A, $18, $06, $0D, $1C, $0E, $11, $2B, $29, $26, $1A, $08, $17, $13, $00, $1E, $01 ), {28}
    ( $03, $1B, $20, $26, $14, $2C, $19, $12, $27, $0E, $1C, $1E, $24, $07, $01, $02, $1A, $18, $1F, $1D, $25, $2F, $28, $21, $10, $05, $0F, $17, $13, $2A, $22, $0A, $0D, $2E, $23, $0C, $04, $00, $16, $2B, $0B, $15, $09, $2D, $06, $29, $11, $08 ), {29}
    ( $21, $02, $00, $01, $06, $0C, $29, $22, $23, $0D, $14, $2F, $1C, $1E, $26, $12, $1B, $20, $03, $09, $0A, $18, $19, $1F, $11, $0E, $1D, $2E, $07, $16, $1A, $28, $27, $2B, $0F, $15, $08, $0B, $04, $10, $2C, $25, $2D, $24, $2A, $13, $17, $05 ), {2A}
    ( $0F, $25, $06, $14, $0B, $20, $00, $0A, $2C, $21, $18, $23, $01, $22, $0D, $02, $2F, $2D, $0E, $2B, $24, $13, $11, $07, $19, $26, $29, $15, $04, $2E, $1C, $1B, $28, $1A, $05, $10, $1F, $08, $03, $17, $2A, $1D, $1E, $12, $09, $27, $16, $0C ), {2B}
    ( $2C, $2B, $11, $2F, $00, $13, $18, $03, $06, $07, $15, $24, $1C, $16, $0A, $05, $21, $27, $17, $14, $02, $09, $01, $04, $1B, $10, $1E, $0B, $0D, $1A, $29, $2E, $2D, $0F, $0C, $26, $1D, $12, $0E, $22, $23, $19, $08, $2A, $1F, $20, $25, $28 ), {2C}
    ( $2F, $20, $2A, $05, $01, $26, $13, $0E, $1E, $0A, $06, $02, $1A, $17, $2B, $04, $15, $18, $10, $0F, $21, $27, $25, $1B, $09, $0D, $2E, $28, $1F, $16, $0B, $14, $1C, $12, $24, $00, $07, $11, $08, $2D, $03, $0C, $2C, $1D, $19, $29, $22, $23 ), {2D}
    ( $07, $2F, $19, $0E, $23, $16, $10, $0C, $24, $20, $0F, $2C, $2B, $02, $03, $06, $12, $0B, $1E, $25, $1A, $08, $17, $13, $0A, $18, $04, $1F, $2D, $1C, $1D, $21, $01, $27, $28, $1B, $05, $22, $29, $0D, $11, $2E, $14, $15, $00, $09, $2A, $26 ), {2E}
    ( $1D, $03, $2A, $1A, $0E, $15, $25, $04, $29, $07, $21, $01, $13, $20, $0B, $22, $28, $06, $1B, $27, $17, $00, $0F, $1C, $2B, $02, $0A, $10, $1F, $2C, $09, $16, $2D, $05, $2E, $12, $23, $14, $2F, $26, $1E, $18, $0C, $19, $11, $24, $0D, $08 ), {2F}
    ( $15, $1F, $2E, $1D, $05, $2F, $00, $16, $2D, $2C, $2B, $23, $21, $18, $10, $12, $07, $2A, $03, $1C, $19, $14, $0E, $26, $1A, $20, $22, $0A, $27, $28, $0D, $0C, $1B, $01, $17, $04, $11, $09, $29, $13, $0B, $1E, $08, $06, $24, $0F, $25, $02 ), {30}
    ( $1C, $1D, $27, $2B, $21, $29, $07, $13, $1F, $0C, $1B, $00, $2F, $26, $14, $19, $09, $2C, $16, $0D, $24, $03, $1A, $10, $0F, $17, $2A, $23, $2D, $0E, $0A, $12, $02, $1E, $2E, $18, $15, $06, $22, $20, $05, $04, $25, $0B, $11, $28, $08, $01 ), {31}
    ( $0B, $10, $0A, $1C, $24, $27, $2A, $2D, $28, $0F, $03, $23, $2E, $05, $16, $14, $0D, $01, $04, $2F, $1A, $08, $22, $0E, $00, $1F, $29, $21, $1D, $1E, $11, $09, $2B, $25, $06, $19, $2C, $17, $07, $15, $18, $02, $0C, $20, $12, $1B, $26, $13 ), {32}
    ( $23, $05, $15, $2E, $02, $12, $0E, $21, $17, $11, $0D, $2D, $04, $2B, $22, $13, $1C, $20, $2A, $25, $16, $0A, $28, $03, $06, $0B, $26, $14, $01, $18, $10, $29, $0C, $07, $1A, $0F, $09, $1D, $00, $2C, $1B, $1F, $1E, $08, $19, $27, $2F, $24 ), {33}
    ( $0E, $14, $24, $07, $17, $04, $1C, $08, $02, $1F, $00, $12, $2C, $1D, $06, $0C, $1E, $15, $2B, $19, $22, $0D, $18, $26, $11, $0A, $2D, $28, $13, $27, $2E, $16, $23, $03, $2F, $2A, $29, $09, $01, $20, $0F, $1B, $25, $21, $1A, $10, $0B, $05 ), {34}
    ( $0B, $22, $2E, $04, $24, $13, $1A, $26, $1D, $2F, $09, $2A, $08, $15, $0F, $23, $20, $12, $1B, $06, $02, $11, $0A, $1E, $2D, $03, $0C, $1C, $2B, $19, $10, $01, $27, $2C, $0D, $28, $25, $21, $16, $07, $00, $17, $1F, $05, $18, $0E, $29, $14 ), {35}
    ( $19, $13, $28, $21, $29, $1A, $2E, $0F, $09, $02, $12, $2F, $14, $00, $03, $2B, $17, $27, $01, $10, $11, $0D, $18, $0C, $1F, $2A, $15, $26, $04, $22, $05, $0E, $0B, $23, $20, $0A, $1D, $1B, $08, $1E, $25, $2D, $07, $24, $16, $2C, $06, $1C ), {36}
    ( $13, $17, $1C, $1E, $08, $0B, $2F, $29, $0A, $00, $25, $2D, $03, $1F, $02, $22, $01, $12, $2C, $2B, $15, $09, $1B, $11, $24, $0F, $05, $27, $0C, $0D, $28, $16, $19, $1A, $26, $04, $2E, $10, $14, $21, $1D, $0E, $23, $06, $2A, $07, $18, $20 ), {37}
    ( $21, $16, $2C, $08, $2D, $1F, $28, $00, $2A, $0B, $22, $2E, $10, $2F, $1D, $0D, $1B, $19, $0C, $27, $1C, $20, $17, $15, $01, $29, $24, $1A, $12, $02, $25, $26, $0A, $23, $05, $2B, $0E, $06, $1E, $11, $14, $18, $04, $07, $03, $0F, $13, $09 ), {38}
    ( $1D, $2E, $2A, $06, $2F, $03, $0F, $1C, $26, $21, $19, $1B, $15, $02, $12, $17, $11, $0C, $20, $14, $1F, $13, $23, $05, $16, $2D, $10, $29, $08, $1A, $22, $01, $27, $24, $2C, $04, $18, $0A, $0D, $25, $07, $09, $00, $0B, $2B, $28, $0E, $1E ), {39}
    ( $10, $0E, $25, $22, $03, $17, $01, $23, $16, $0F, $2B, $09, $13, $20, $0A, $00, $1B, $27, $1F, $07, $2D, $11, $24, $2A, $0B, $18, $14, $0D, $12, $08, $1D, $1E, $2E, $06, $26, $0C, $15, $04, $1C, $29, $2C, $21, $19, $28, $05, $2F, $02, $1A ), {3A}
    ( $20, $0B, $16, $0F, $15, $29, $04, $1B, $0A, $2A, $01, $2F, $0E, $28, $1C, $2C, $0D, $03, $1D, $2E, $06, $2B, $05, $08, $10, $00, $12, $25, $2D, $0C, $02, $22, $26, $17, $19, $1A, $21, $14, $23, $24, $13, $18, $1E, $09, $1F, $27, $07, $11 ), {3B}
    ( $13, $0D, $0B, $0E, $2B, $11, $22, $02, $0F, $10, $23, $0C, $06, $1A, $01, $0A, $12, $15, $2D, $27, $07, $1D, $2A, $1B, $1E, $04, $17, $29, $2F, $16, $21, $00, $18, $09, $1F, $20, $28, $03, $24, $14, $05, $08, $25, $2E, $26, $2C, $1C, $19 ), {3C}
    ( $27, $2C, $20, $21, $2F, $25, $1F, $00, $2E, $16, $05, $26, $08, $0A, $15, $28, $24, $03, $10, $0B, $12, $1E, $0C, $14, $0F, $1D, $01, $2B, $06, $07, $1C, $2D, $02, $1A, $0D, $2A, $18, $0E, $17, $19, $09, $11, $23, $04, $29, $13, $22, $1B ), {3D}
    ( $20, $09, $1A, $29, $16, $1B, $05, $1E, $0D, $11, $00, $19, $12, $25, $13, $0C, $0B, $04, $0F, $26, $1D, $1C, $2A, $15, $01, $22, $21, $02, $27, $0E, $2D, $10, $06, $08, $28, $2C, $2F, $2B, $0A, $1F, $07, $2E, $03, $23, $14, $24, $17, $18 ), {3E}
    ( $0A, $14, $01, $15, $1F, $22, $2E, $2D, $07, $2C, $2A, $29, $13, $1E, $21, $2F, $06, $18, $24, $1A, $0D, $08, $19, $11, $12, $04, $17, $0B, $00, $20, $02, $0C, $05, $10, $09, $26, $27, $28, $0F, $25, $1C, $1B, $2B, $0E, $03, $1D, $16, $23 ), {3F}
    ( $27, $08, $05, $0D, $15, $0F, $09, $25, $2B, $01, $1D, $11, $1C, $04, $22, $03, $0A, $0E, $13, $17, $10, $28, $00, $26, $06, $20, $2C, $1E, $07, $29, $23, $21, $19, $24, $12, $2D, $2F, $0C, $2E, $14, $2A, $1A, $02, $18, $1B, $1F, $0B, $16 ), {40}
    ( $11, $1A, $29, $1C, $1B, $00, $0B, $09, $1E, $0E, $1F, $18, $0A, $16, $26, $13, $03, $12, $0F, $2E, $0D, $22, $02, $07, $23, $14, $08, $20, $2C, $27, $10, $28, $17, $2D, $2B, $24, $04, $2A, $21, $06, $0C, $19, $2F, $25, $1D, $01, $05, $15 ), {41}
    ( $02, $13, $2D, $28, $00, $1A, $01, $2F, $2B, $03, $1C, $14, $22, $26, $24, $2C, $29, $2E, $0B, $19, $1D, $17, $08, $2A, $0D, $1B, $06, $18, $15, $04, $21, $25, $05, $0E, $0F, $27, $07, $20, $09, $23, $16, $12, $11, $1F, $0C, $1E, $0A, $10 ), {42}
    ( $1C, $2F, $01, $2D, $0A, $0F, $29, $15, $2E, $04, $19, $2B, $1D, $28, $00, $05, $27, $13, $1B, $09, $0E, $23, $2A, $22, $1A, $08, $25, $1F, $24, $17, $18, $07, $1E, $26, $03, $12, $0C, $06, $0B, $2C, $10, $14, $21, $20, $0D, $02, $16, $11 ), {43}
    ( $20, $0F, $09, $1B, $28, $1F, $02, $1C, $0B, $23, $0C, $25, $04, $22, $27, $11, $2C, $0E, $0A, $24, $29, $1D, $2F, $03, $2D, $17, $06, $07, $18, $10, $05, $15, $26, $2A, $14, $01, $00, $1A, $2E, $12, $1E, $2B, $19, $16, $21, $0D, $08, $13 ), {44}
    ( $07, $27, $18, $2A, $03, $0F, $29, $17, $15, $11, $20, $14, $26, $05, $19, $23, $2B, $01, $24, $22, $09, $1C, $21, $0A, $25, $00, $1B, $04, $0D, $28, $06, $2C, $1F, $1E, $0C, $08, $2D, $10, $13, $1A, $0B, $0E, $2E, $2F, $1D, $12, $02, $16 ), {45}
    ( $2E, $2D, $1A, $14, $07, $13, $04, $00, $12, $2B, $26, $0A, $20, $0B, $2C, $0D, $0C, $23, $03, $1D, $27, $01, $0F, $2A, $22, $16, $05, $19, $24, $1F, $18, $0E, $11, $28, $29, $15, $10, $1C, $09, $08, $02, $06, $25, $21, $1B, $17, $2F, $1E ), {46}
    ( $2D, $0C, $05, $13, $14, $1A, $12, $0E, $1D, $11, $29, $27, $26, $01, $0B, $19, $21, $07, $10, $2E, $2F, $08, $18, $20, $17, $1F, $06, $15, $2B, $0D, $23, $00, $24, $03, $1C, $1E, $16, $22, $1B, $28, $2C, $25, $0A, $0F, $02, $04, $09, $2A ), {47}
    ( $26, $22, $19, $25, $1F, $01, $1C, $1A, $11, $0C, $10, $18, $09, $03, $07, $0D, $24, $20, $08, $27, $2A, $0E, $12, $29, $00, $17, $02, $0F, $2E, $2F, $2B, $1D, $2C, $13, $2D, $0A, $04, $15, $23, $14, $06, $28, $21, $16, $05, $0B, $1E, $1B ), {48}
    ( $05, $15, $0A, $1E, $2F, $00, $10, $23, $1B, $25, $16, $22, $29, $0F, $06, $0E, $13, $26, $1C, $09, $1F, $28, $11, $12, $0C, $27, $2B, $24, $17, $2E, $01, $2A, $1D, $2D, $07, $18, $20, $04, $2C, $21, $03, $0B, $14, $19, $0D, $1A, $08, $02 ), {49}
    ( $10, $0B, $2B, $26, $15, $2A, $02, $19, $13, $29, $06, $21, $24, $18, $16, $22, $14, $2E, $25, $0C, $08, $1B, $27, $05, $0F, $12, $2F, $04, $00, $2D, $07, $09, $17, $0A, $01, $0E, $0D, $1D, $1F, $1A, $23, $28, $1C, $11, $03, $1E, $20, $2C ), {4A}
    ( $0B, $0A, $14, $0C, $11, $0F, $09, $03, $06, $1C, $2B, $01, $13, $25, $26, $1B, $19, $04, $21, $1E, $12, $1D, $2D, $0E, $27, $29, $23, $2A, $10, $2F, $1F, $2C, $08, $20, $24, $00, $18, $28, $15, $05, $22, $16, $0D, $17, $07, $02, $1A, $2E ), {4B}
    ( $0C, $19, $0B, $18, $1A, $16, $17, $09, $01, $26, $20, $08, $07, $13, $29, $28, $0A, $1D, $0E, $2F, $06, $1B, $11, $04, $2B, $25, $1C, $21, $24, $10, $02, $05, $14, $2C, $23, $12, $0F, $0D, $2A, $03, $00, $15, $2D, $27, $2E, $1E, $1F, $22 ), {4C}
    ( $02, $09, $1D, $2C, $0C, $1C, $19, $26, $07, $2B, $0B, $0E, $01, $25, $27, $24, $21, $2F, $29, $23, $2E, $22, $13, $1B, $28, $1F, $11, $0A, $14, $03, $08, $1A, $12, $06, $04, $17, $2A, $00, $0F, $0D, $15, $16, $1E, $05, $20, $18, $2D, $10 ), {4D}
    ( $03, $23, $14, $25, $0E, $0A, $07, $0B, $04, $2E, $00, $22, $16, $1A, $2D, $05, $29, $2A, $18, $12, $1F, $27, $26, $01, $09, $24, $1B, $06, $11, $2F, $10, $21, $02, $1D, $0C, $1E, $2C, $08, $1C, $13, $0D, $2B, $28, $15, $0F, $17, $20, $19 ), {4E}
    ( $09, $28, $1F, $29, $0C, $26, $1B, $23, $12, $0D, $2F, $11, $20, $2A, $07, $2C, $25, $1D, $1A, $04, $19, $02, $10, $16, $2B, $15, $06, $0B, $05, $22, $2D, $18, $0E, $0F, $21, $03, $1E, $27, $13, $24, $0A, $17, $00, $2E, $08, $1C, $01, $14 ), {4F}
    ( $1D, $1F, $0D, $0B, $17, $2B, $2D, $0C, $0F, $2E, $07, $28, $01, $21, $10, $1C, $23, $0A, $24, $19, $29, $2A, $02, $09, $26, $0E, $1B, $25, $16, $06, $11, $22, $00, $12, $1A, $15, $05, $2F, $14, $03, $1E, $27, $20, $04, $13, $08, $18, $2C ), {50}
    ( $0A, $27, $14, $1E, $05, $04, $1B, $1D, $12, $07, $03, $10, $08, $16, $24, $0C, $22, $09, $15, $1C, $28, $20, $25, $21, $01, $26, $0B, $2F, $1A, $13, $2B, $23, $2E, $2D, $0E, $29, $0D, $17, $00, $18, $11, $2C, $02, $1F, $19, $06, $2A, $0F ), {51}
    ( $2A, $1F, $2B, $14, $03, $27, $2E, $01, $11, $1E, $1B, $1A, $25, $00, $15, $23, $21, $19, $18, $29, $0E, $09, $0B, $13, $20, $05, $1C, $12, $28, $2D, $24, $0D, $04, $22, $08, $0C, $17, $07, $2C, $06, $16, $1D, $2F, $0A, $0F, $26, $10, $02 ), {52}
    ( $1F, $21, $1A, $05, $25, $17, $20, $1B, $0C, $28, $13, $2A, $02, $2D, $1D, $01, $29, $0F, $07, $09, $1C, $19, $04, $14, $15, $11, $23, $0A, $03, $2E, $0E, $00, $22, $2B, $2F, $0D, $12, $26, $10, $1E, $24, $06, $2C, $18, $16, $08, $0B, $27 ), {53}
    ( $0F, $2A, $05, $00, $28, $2F, $24, $1A, $13, $1C, $2C, $04, $25, $1D, $26, $09, $2D, $15, $08, $2E, $17, $14, $11, $1F, $0E, $12, $19, $10, $18, $0A, $1B, $07, $2B, $0C, $1E, $29, $01, $23, $27, $0D, $16, $03, $06, $20, $0B, $21, $02, $22 ), {54}
    ( $0D, $1C, $2A, $09, $29, $28, $1A, $11, $14, $19, $16, $1F, $18, $20, $2D, $1D, $27, $2F, $23, $01, $25, $2B, $2C, $0A, $02, $1B, $15, $0E, $21, $10, $00, $24, $07, $0F, $05, $2E, $03, $13, $1E, $06, $0B, $12, $17, $08, $22, $04, $26, $0C ), {55}
    ( $2D, $0C, $21, $1C, $11, $06, $07, $2C, $12, $20, $01, $2E, $1F, $18, $04, $0B, $13, $2B, $2A, $2F, $24, $05, $27, $29, $14, $26, $19, $0A, $0D, $23, $09, $02, $16, $1B, $10, $08, $00, $28, $0F, $1D, $1A, $15, $22, $25, $17, $03, $1E, $0E ), {56}
    ( $06, $17, $1D, $0B, $19, $1F, $2A, $2F, $29, $15, $0C, $2C, $1A, $04, $0D, $20, $18, $1C, $28, $25, $22, $09, $07, $2B, $21, $14, $0E, $02, $23, $26, $08, $13, $1E, $16, $0A, $05, $27, $11, $1B, $24, $2E, $00, $2D, $12, $10, $01, $03, $0F ), {57}
    ( $0E, $1D, $04, $22, $0F, $1B, $2C, $2A, $2B, $06, $09, $0D, $24, $28, $2F, $13, $08, $17, $03, $1E, $05, $02, $2E, $20, $29, $0C, $27, $12, $1A, $07, $1C, $21, $16, $18, $15, $0B, $10, $26, $23, $2D, $01, $19, $0A, $1F, $25, $14, $11, $00 ), {58}
    ( $0D, $03, $25, $16, $0A, $12, $2A, $19, $10, $04, $0F, $18, $05, $15, $23, $11, $09, $28, $26, $0C, $0B, $13, $22, $1C, $0E, $07, $14, $2B, $29, $27, $06, $2D, $1B, $2E, $1E, $2C, $20, $2F, $24, $17, $02, $1A, $08, $21, $01, $1D, $00, $1F ), {59}
    ( $03, $09, $0C, $24, $10, $20, $1F, $25, $0E, $28, $08, $0F, $2C, $19, $16, $18, $22, $06, $2E, $0B, $01, $04, $14, $21, $05, $23, $0A, $00, $2D, $1C, $12, $02, $2B, $13, $1D, $27, $17, $0D, $11, $29, $1A, $1B, $1E, $2F, $2A, $26, $15, $07 ), {5A}
    ( $05, $0D, $23, $16, $1D, $28, $12, $09, $25, $14, $1A, $1B, $0F, $10, $03, $1E, $15, $2B, $18, $2D, $22, $0E, $0C, $11, $04, $21, $07, $0A, $2C, $0B, $19, $2E, $06, $2A, $08, $17, $01, $1C, $20, $24, $1F, $2F, $02, $00, $29, $27, $26, $13 ), {5B}
    ( $07, $26, $1C, $2B, $16, $20, $2E, $10, $23, $27, $04, $0A, $02, $14, $17, $19, $1D, $1E, $1B, $01, $0F, $24, $29, $13, $0E, $06, $22, $05, $03, $12, $09, $11, $18, $15, $00, $1A, $08, $2D, $25, $2C, $0B, $28, $0D, $2A, $21, $2F, $0C, $1F ), {5C}
    ( $13, $0A, $2F, $0C, $1C, $2D, $1F, $14, $15, $05, $1B, $10, $03, $11, $18, $28, $2E, $21, $0D, $2A, $02, $25, $00, $20, $1D, $08, $19, $0F, $04, $27, $22, $26, $06, $0E, $17, $01, $23, $2B, $1E, $1A, $24, $2C, $09, $29, $16, $0B, $07, $12 ), {5D}
    ( $26, $17, $2E, $18, $06, $19, $09, $0F, $1B, $28, $05, $27, $16, $23, $1A, $21, $1D, $03, $2C, $24, $20, $14, $25, $2F, $1E, $01, $02, $2D, $1F, $13, $2A, $07, $00, $04, $08, $0C, $12, $29, $0E, $22, $15, $2B, $10, $0B, $0D, $0A, $11, $1C ), {5E}
    ( $06, $28, $2B, $07, $18, $1D, $29, $12, $01, $1B, $0E, $03, $0A, $20, $02, $2C, $16, $17, $0D, $1F, $00, $2A, $1E, $05, $04, $0B, $2D, $1A, $26, $23, $2F, $22, $13, $10, $14, $19, $25, $1C, $11, $08, $09, $2E, $0C, $24, $0F, $21, $15, $27 ), {5F}
    ( $19, $15, $06, $2F, $01, $12, $20, $02, $2E, $07, $22, $14, $17, $1B, $2B, $1A, $08, $21, $1E, $16, $29, $25, $23, $18, $0E, $09, $1F, $26, $2D, $28, $2C, $0A, $05, $1D, $27, $0F, $04, $2A, $10, $00, $0D, $0B, $24, $03, $1C, $0C, $13, $11 ), {60}
    ( $2A, $21, $19, $2E, $22, $0B, $0D, $1F, $27, $0C, $17, $02, $2C, $0A, $11, $29, $07, $13, $16, $2F, $10, $24, $2B, $14, $20, $1A, $1E, $18, $1B, $06, $00, $01, $23, $04, $15, $08, $0E, $1D, $09, $12, $2D, $0F, $05, $26, $03, $1C, $28, $25 ), {61}
    ( $1E, $18, $07, $1C, $15, $0C, $2F, $10, $24, $13, $0D, $12, $00, $2E, $14, $0A, $20, $04, $05, $02, $17, $08, $11, $2A, $29, $25, $0B, $16, $19, $23, $27, $2C, $0E, $28, $1F, $21, $1B, $2D, $2B, $1A, $0F, $09, $26, $22, $06, $01, $03, $1D ), {62}
    ( $0B, $05, $2C, $1B, $20, $09, $0A, $26, $11, $10, $1F, $03, $2F, $14, $1D, $02, $15, $0D, $2D, $07, $04, $18, $1A, $22, $23, $21, $0E, $12, $2A, $28, $2B, $25, $1E, $29, $24, $19, $00, $27, $1C, $16, $17, $0F, $01, $13, $2E, $06, $0C, $08 ), {63}
    ( $19, $06, $11, $2D, $0C, $1C, $2A, $28, $05, $1F, $1B, $21, $1D, $0B, $1A, $2F, $07, $01, $0A, $17, $16, $09, $29, $24, $2B, $03, $13, $22, $1E, $08, $20, $10, $0D, $18, $0E, $0F, $02, $26, $27, $25, $00, $2C, $14, $04, $23, $12, $15, $2E ), {64}
    ( $2B, $09, $1B, $16, $1C, $0F, $21, $12, $0B, $25, $22, $2D, $27, $08, $28, $0E, $23, $19, $1E, $05, $26, $0A, $15, $14, $2A, $24, $20, $07, $1F, $0D, $18, $11, $10, $0C, $2E, $00, $2F, $01, $1D, $13, $17, $03, $1A, $02, $06, $04, $2C, $29 ), {65}
    ( $2C, $12, $0F, $0E, $07, $22, $08, $11, $01, $2E, $0C, $1F, $0D, $26, $21, $14, $2F, $03, $1B, $2A, $18, $23, $13, $1C, $05, $25, $28, $17, $2B, $15, $1D, $04, $0B, $09, $1E, $27, $2D, $29, $19, $16, $06, $00, $02, $0A, $24, $20, $10, $1A ), {66}
    ( $21, $17, $03, $26, $11, $1E, $22, $2E, $0A, $2B, $2A, $19, $18, $1C, $25, $1A, $20, $23, $07, $29, $02, $2C, $0D, $05, $09, $15, $16, $2D, $06, $04, $0F, $0B, $1F, $0E, $14, $10, $27, $12, $1D, $13, $1B, $08, $01, $2F, $00, $28, $0C, $24 ), {67}
    ( $04, $00, $28, $0F, $26, $1A, $23, $0D, $2A, $0E, $22, $2B, $06, $14, $29, $03, $25, $01, $1F, $1C, $13, $05, $24, $0A, $0C, $09, $1D, $15, $2F, $11, $1B, $17, $27, $10, $12, $16, $21, $2E, $2D, $19, $07, $1E, $18, $2C, $20, $02, $08, $0B ), {68}
    ( $18, $04, $08, $21, $23, $29, $0C, $2B, $07, $1E, $14, $09, $03, $10, $17, $00, $0D, $0F, $15, $22, $20, $0A, $06, $1A, $27, $12, $1B, $26, $19, $25, $28, $0B, $1C, $01, $05, $2E, $11, $16, $2C, $2D, $0E, $13, $02, $2A, $2F, $1F, $1D, $24 ), {69}
    ( $25, $27, $17, $13, $01, $2E, $16, $19, $0D, $29, $15, $1E, $11, $00, $06, $23, $09, $26, $0B, $07, $0C, $1A, $14, $22, $0A, $20, $1F, $18, $02, $10, $0E, $08, $0F, $12, $21, $28, $2C, $2D, $2B, $24, $1B, $05, $1D, $1C, $04, $03, $2F, $2A ), {6A}
    ( $16, $07, $23, $20, $0F, $2B, $2D, $08, $02, $00, $2E, $2C, $22, $25, $13, $19, $04, $18, $15, $06, $0D, $01, $1E, $27, $2A, $21, $29, $0C, $2F, $09, $14, $05, $24, $1B, $0E, $10, $12, $26, $28, $1A, $0B, $03, $17, $0A, $1F, $11, $1C, $1D ), {6B}
    ( $24, $06, $0B, $1F, $10, $02, $1A, $21, $25, $16, $1E, $27, $00, $22, $0C, $09, $01, $0D, $19, $0F, $07, $28, $12, $18, $26, $2E, $13, $08, $03, $20, $17, $1B, $2F, $29, $23, $15, $2C, $2D, $1D, $0A, $1C, $2B, $05, $11, $2A, $04, $14, $0E ), {6C}
    ( $05, $2F, $09, $1D, $1F, $1C, $20, $0A, $0F, $29, $01, $24, $25, $03, $08, $00, $0C, $21, $1B, $2A, $28, $02, $07, $19, $2C, $0B, $14, $2E, $15, $1E, $23, $13, $04, $17, $16, $10, $2B, $22, $1A, $2D, $27, $26, $11, $0E, $12, $06, $0D, $18 ), {6D}
    ( $27, $11, $01, $08, $13, $0D, $24, $0F, $1E, $15, $16, $12, $18, $1C, $14, $1B, $28, $20, $03, $04, $0B, $26, $1A, $1D, $17, $07, $2B, $22, $0E, $06, $21, $29, $0A, $02, $10, $2A, $25, $2C, $05, $00, $2E, $2F, $09, $0C, $2D, $1F, $19, $23 ), {6E}
    ( $24, $09, $04, $0A, $28, $25, $26, $0D, $29, $12, $01, $1C, $17, $14, $23, $20, $02, $05, $13, $2A, $11, $1F, $22, $0F, $06, $2D, $16, $1D, $0B, $2E, $0E, $15, $21, $1A, $19, $0C, $03, $07, $00, $18, $2F, $1B, $27, $10, $08, $1E, $2C, $2B ), {6F}
    ( $29, $1A, $1D, $18, $0C, $1F, $04, $11, $03, $25, $26, $02, $28, $15, $16, $01, $00, $07, $22, $12, $0E, $06, $17, $24, $14, $09, $27, $2A, $19, $0A, $21, $2B, $0B, $2C, $08, $13, $0D, $2F, $2E, $20, $10, $1C, $1B, $2D, $1E, $05, $0F, $23 ), {70}
    ( $13, $03, $2F, $25, $19, $10, $18, $2A, $2C, $2B, $22, $20, $1E, $21, $1A, $14, $04, $29, $11, $0E, $1D, $00, $0C, $06, $2D, $0A, $09, $27, $28, $2E, $23, $1C, $1F, $0D, $02, $15, $1B, $17, $05, $0F, $07, $24, $16, $26, $08, $12, $0B, $01 ), {71}
    ( $14, $13, $17, $0E, $0C, $21, $02, $1A, $20, $24, $00, $2B, $1B, $1D, $15, $10, $23, $27, $16, $26, $0A, $0B, $1F, $03, $0D, $0F, $08, $2F, $2E, $09, $07, $05, $18, $2C, $19, $25, $2A, $12, $1C, $04, $2D, $06, $22, $29, $01, $28, $11, $1E ), {72}
    ( $1C, $26, $12, $2C, $1D, $22, $03, $2B, $23, $2D, $29, $11, $0E, $04, $27, $0A, $1F, $08, $25, $00, $24, $17, $0B, $0D, $01, $2F, $05, $0F, $28, $14, $15, $09, $1A, $20, $0C, $07, $10, $1E, $06, $2E, $19, $2A, $1B, $13, $02, $16, $21, $18 ), {73}
    ( $25, $11, $07, $10, $1C, $09, $0F, $16, $0A, $17, $24, $1D, $2C, $21, $2E, $20, $05, $19, $23, $14, $13, $1A, $01, $2B, $28, $08, $1B, $06, $1E, $12, $2A, $2D, $0B, $00, $0E, $02, $22, $18, $1F, $15, $0C, $2F, $03, $26, $0D, $27, $29, $04 ), {74}
    ( $17, $2D, $24, $2E, $26, $1E, $2C, $1B, $09, $27, $0F, $20, $08, $29, $13, $22, $2B, $2A, $1C, $05, $16, $03, $25, $12, $1D, $28, $01, $15, $14, $23, $02, $06, $1F, $0A, $2F, $0B, $11, $21, $04, $0C, $1A, $0D, $19, $10, $0E, $07, $00, $18 ), {75}
    ( $22, $03, $09, $00, $23, $25, $20, $2D, $1E, $13, $19, $04, $01, $18, $1F, $2A, $26, $08, $07, $21, $2C, $2E, $10, $0A, $02, $24, $0E, $1A, $28, $2B, $06, $0F, $11, $29, $0B, $05, $1C, $12, $16, $2F, $1D, $1B, $14, $0C, $17, $0D, $27, $15 ), {76}
    ( $07, $11, $0E, $2D, $2E, $03, $15, $24, $10, $06, $05, $1A, $19, $20, $29, $13, $02, $1B, $2C, $00, $0C, $0B, $0F, $01, $09, $21, $18, $1C, $16, $1E, $14, $2F, $23, $26, $1F, $22, $0D, $0A, $17, $28, $12, $27, $04, $25, $2A, $1D, $08, $2B ), {77}
    ( $04, $2E, $05, $15, $18, $19, $1D, $2C, $0E, $1F, $1C, $0D, $00, $27, $09, $2D, $23, $2F, $17, $0B, $1A, $21, $10, $22, $03, $11, $20, $28, $0C, $25, $01, $1B, $26, $06, $14, $07, $08, $2B, $0A, $16, $29, $02, $0F, $12, $24, $1E, $13, $2A ), {78}
    ( $25, $24, $00, $11, $0E, $22, $17, $2E, $06, $12, $28, $18, $2F, $20, $05, $16, $08, $0F, $1D, $02, $0D, $2D, $27, $2B, $2A, $1E, $26, $03, $07, $0C, $19, $01, $09, $0B, $0A, $29, $14, $2C, $10, $23, $04, $1A, $15, $1F, $13, $21, $1B, $1C ), {79}
    ( $2C, $21, $27, $2A, $1A, $0D, $03, $2D, $2F, $02, $0F, $28, $0B, $11, $01, $1F, $0E, $1C, $18, $08, $2E, $19, $24, $15, $23, $17, $22, $20, $00, $0A, $04, $10, $25, $26, $13, $14, $0C, $29, $06, $1D, $1B, $16, $05, $07, $12, $09, $1E, $2B ), {7A}
    ( $1A, $1E, $00, $1C, $1D, $21, $2C, $0F, $01, $14, $2E, $0A, $17, $2D, $1B, $10, $11, $0D, $0B, $0E, $19, $05, $23, $09, $12, $2B, $27, $2A, $06, $26, $08, $2F, $0C, $02, $18, $15, $03, $24, $07, $1F, $20, $22, $29, $13, $28, $25, $16, $04 ), {7B}
    ( $10, $19, $0D, $09, $2A, $12, $01, $2E, $03, $04, $25, $1D, $06, $14, $2C, $07, $00, $2B, $2F, $16, $0F, $1C, $27, $02, $08, $0C, $11, $2D, $05, $1A, $0E, $20, $0A, $29, $23, $18, $1E, $26, $17, $21, $22, $1F, $28, $0B, $24, $1B, $15, $13 ), {7C}
    ( $22, $04, $20, $12, $10, $2F, $1C, $1B, $01, $2C, $0B, $15, $0F, $1F, $11, $0A, $03, $1D, $16, $0C, $24, $29, $19, $2E, $0D, $14, $2B, $0E, $17, $08, $09, $21, $26, $28, $13, $1E, $07, $18, $05, $2A, $06, $2D, $1A, $00, $25, $23, $02, $27 ), {7D}
    ( $2A, $2B, $1E, $02, $2E, $27, $03, $0D, $05, $23, $17, $13, $19, $2F, $2D, $21, $1C, $0A, $1D, $1F, $06, $00, $04, $29, $11, $08, $22, $1A, $0E, $26, $28, $12, $09, $15, $25, $0B, $10, $16, $24, $1B, $18, $14, $07, $01, $0C, $2C, $0F, $20 ), {7E}
    ( $21, $14, $24, $28, $2C, $00, $0F, $17, $1C, $2D, $27, $15, $11, $1B, $18, $1F, $16, $06, $0D, $05, $0E, $2F, $2E, $1A, $03, $01, $0B, $07, $10, $19, $13, $1E, $0C, $29, $04, $09, $1D, $26, $22, $23, $20, $2A, $08, $25, $2B, $0A, $12, $02 ), {7F}
    ( $02, $07, $15, $29, $25, $0C, $0D, $00, $28, $1A, $0B, $1B, $18, $2C, $03, $11, $23, $27, $1F, $1E, $06, $2E, $21, $24, $08, $05, $22, $2D, $09, $01, $12, $26, $17, $13, $2B, $1C, $14, $0A, $0E, $16, $2A, $0F, $1D, $20, $2F, $10, $19, $04 ), {80}
    ( $1D, $2F, $22, $08, $29, $18, $05, $0B, $11, $17, $2D, $13, $21, $1E, $12, $2A, $06, $1F, $28, $26, $00, $0F, $1A, $0D, $2E, $0E, $19, $27, $2C, $15, $20, $14, $2B, $03, $01, $02, $09, $24, $07, $0A, $04, $0C, $25, $1C, $1B, $23, $10, $16 ), {81}
    ( $01, $22, $2F, $19, $2E, $0E, $2A, $14, $12, $1A, $1F, $2D, $0D, $07, $04, $26, $18, $28, $0F, $16, $1B, $17, $09, $11, $29, $10, $0A, $25, $03, $06, $2C, $24, $0B, $27, $02, $13, $2B, $1D, $20, $05, $15, $08, $1C, $1E, $0C, $21, $23, $00 ), {82}
    ( $1E, $25, $1B, $13, $1D, $21, $2D, $16, $0D, $28, $12, $05, $06, $20, $19, $24, $2E, $03, $14, $10, $07, $26, $22, $1A, $1C, $0B, $0F, $1F, $0A, $17, $15, $0C, $2C, $2B, $2A, $01, $08, $02, $27, $18, $04, $11, $00, $2F, $09, $0E, $29, $23 ), {83}
    ( $2D, $22, $28, $15, $16, $0A, $09, $05, $0E, $1B, $2C, $07, $2E, $1A, $1F, $04, $01, $29, $00, $19, $23, $11, $08, $10, $25, $0F, $14, $03, $26, $06, $2B, $27, $2A, $0C, $24, $20, $21, $13, $17, $12, $18, $1C, $1E, $02, $2F, $0D, $0B, $1D ), {84}
    ( $17, $2E, $01, $16, $2D, $2B, $11, $1F, $27, $08, $23, $00, $18, $06, $0A, $0C, $22, $24, $2A, $1A, $07, $09, $13, $19, $0E, $1C, $12, $15, $1D, $02, $10, $2F, $26, $2C, $03, $1E, $1B, $0B, $25, $0D, $0F, $05, $20, $28, $04, $29, $14, $21 ), {85}
    ( $2F, $0E, $13, $07, $25, $1F, $2B, $20, $29, $19, $18, $0F, $1E, $27, $2C, $10, $12, $26, $21, $1B, $17, $08, $2A, $11, $24, $06, $2E, $09, $04, $15, $14, $05, $0D, $03, $0C, $01, $1A, $00, $22, $1D, $23, $28, $0B, $02, $1C, $16, $2D, $0A ), {86}
    ( $05, $0F, $10, $2D, $29, $07, $22, $08, $18, $0B, $1D, $02, $27, $0E, $01, $13, $2B, $28, $25, $20, $0A, $24, $21, $1E, $17, $2E, $2C, $06, $19, $04, $26, $03, $15, $2F, $11, $0D, $12, $23, $1C, $09, $14, $1B, $16, $1A, $0C, $00, $1F, $2A ), {87}
    ( $1B, $2E, $18, $0F, $05, $23, $08, $01, $10, $27, $09, $24, $1E, $12, $1F, $07, $02, $2B, $26, $2D, $0D, $29, $2F, $16, $1D, $17, $04, $03, $13, $28, $25, $1C, $11, $00, $21, $14, $0A, $0B, $2C, $22, $19, $2A, $0C, $20, $15, $06, $1A, $0E ), {88}
    ( $07, $20, $2B, $0A, $2D, $21, $1E, $13, $05, $04, $1A, $25, $0B, $01, $26, $09, $19, $1C, $15, $0E, $29, $03, $0D, $2C, $1B, $28, $02, $24, $08, $11, $0C, $22, $12, $2E, $1D, $2F, $10, $17, $0F, $16, $27, $1F, $23, $06, $18, $2A, $14, $00 ), {89}
    ( $17, $16, $0E, $00, $06, $15, $0A, $1C, $28, $02, $23, $1E, $2A, $03, $22, $11, $07, $1D, $24, $18, $20, $0C, $2B, $25, $2C, $2F, $26, $08, $0B, $1B, $0F, $10, $19, $05, $04, $0D, $27, $13, $1F, $09, $01, $2E, $12, $21, $29, $14, $2D, $1A ), {8A}
    ( $23, $10, $2A, $11, $1E, $01, $04, $0A, $25, $03, $1C, $2B, $28, $12, $2E, $1A, $08, $19, $17, $0C, $2F, $14, $06, $13, $29, $0F, $0B, $2D, $0E, $00, $16, $20, $22, $1F, $26, $1D, $02, $24, $21, $18, $07, $27, $05, $1B, $15, $0D, $09, $2C ), {8B}
    ( $28, $03, $00, $10, $12, $15, $0F, $1F, $1A, $1C, $2E, $14, $23, $0E, $17, $18, $1E, $07, $06, $0B, $08, $29, $26, $2C, $2D, $20, $25, $02, $0C, $09, $0A, $13, $27, $1D, $05, $24, $2B, $04, $22, $2F, $1B, $16, $01, $11, $21, $2A, $19, $0D ), {8C}
    ( $18, $29, $25, $09, $2B, $00, $26, $21, $0C, $12, $17, $06, $1F, $24, $1A, $1C, $05, $14, $16, $20, $1B, $19, $01, $28, $2C, $27, $0D, $22, $10, $2D, $2E, $23, $15, $2A, $03, $1D, $08, $0B, $2F, $0F, $0A, $1E, $02, $0E, $13, $04, $11, $07 ), {8D}
    ( $08, $16, $07, $2E, $1A, $0C, $27, $2F, $21, $0A, $09, $01, $02, $17, $10, $0B, $11, $22, $23, $26, $28, $2A, $1B, $0D, $04, $1C, $15, $18, $05, $0F, $03, $19, $24, $1F, $29, $1E, $12, $06, $2C, $0E, $13, $1D, $14, $00, $2B, $25, $20, $2D ), {8E}
    ( $24, $00, $2A, $10, $09, $0D, $23, $0F, $0B, $17, $02, $1F, $1E, $04, $1C, $12, $2E, $2C, $15, $03, $1A, $16, $20, $28, $25, $05, $01, $11, $18, $19, $22, $1D, $1B, $0A, $21, $29, $13, $0C, $2F, $2D, $06, $08, $26, $14, $07, $0E, $2B, $27 ), {8F}
    ( $1B, $16, $2D, $05, $15, $2C, $0E, $09, $1D, $08, $26, $19, $24, $01, $13, $1F, $2A, $0B, $06, $02, $12, $0A, $1C, $23, $1E, $07, $10, $20, $21, $00, $2F, $2E, $04, $29, $0C, $22, $2B, $11, $14, $03, $0D, $18, $17, $1A, $27, $28, $0F, $25 ), {90}
    ( $19, $20, $06, $1F, $0D, $17, $11, $03, $07, $2D, $1B, $2C, $14, $09, $15, $27, $2E, $0E, $22, $04, $18, $10, $0F, $2A, $28, $02, $25, $21, $16, $2F, $00, $1C, $1D, $1A, $0A, $13, $0B, $05, $0C, $12, $23, $1E, $29, $01, $26, $08, $2B, $24 ), {91}
    ( $1E, $0D, $26, $15, $22, $2D, $14, $25, $23, $2B, $0C, $2A, $04, $24, $12, $01, $28, $2E, $08, $2C, $19, $29, $18, $27, $20, $0E, $03, $0B, $0A, $00, $1F, $16, $2F, $1C, $13, $11, $1B, $06, $1A, $09, $17, $07, $0F, $02, $1D, $05, $21, $10 ), {92}
    ( $27, $1B, $10, $17, $29, $19, $12, $18, $14, $1C, $11, $06, $21, $28, $0A, $23, $1E, $24, $2B, $04, $20, $01, $1D, $02, $13, $0C, $2E, $0D, $25, $1F, $08, $2C, $03, $07, $0F, $1A, $05, $15, $2A, $26, $00, $09, $2F, $16, $0E, $22, $2D, $0B ), {93}
    ( $2C, $1E, $1B, $24, $1D, $2F, $26, $01, $2E, $19, $21, $0E, $23, $07, $17, $27, $14, $10, $11, $28, $22, $03, $04, $18, $16, $1A, $0D, $13, $20, $2A, $0B, $25, $29, $0C, $12, $06, $15, $0F, $1F, $05, $09, $1C, $0A, $08, $2D, $02, $00, $2B ), {94}
    ( $0B, $2A, $26, $09, $1C, $0C, $0E, $02, $11, $28, $18, $01, $1B, $20, $08, $15, $2D, $0F, $27, $2C, $1D, $1A, $29, $00, $12, $2F, $2E, $16, $0D, $2B, $17, $10, $21, $19, $07, $13, $06, $04, $23, $0A, $14, $24, $1F, $1E, $25, $22, $05, $03 ), {95}
    ( $1F, $1E, $2C, $27, $1B, $1C, $24, $0E, $2F, $2A, $07, $0F, $2E, $0B, $11, $28, $02, $17, $25, $08, $12, $23, $19, $0C, $0A, $2B, $03, $18, $16, $21, $22, $14, $10, $01, $05, $1A, $13, $15, $20, $09, $1D, $04, $2D, $06, $00, $26, $0D, $29 ), {96}
    ( $1D, $05, $1E, $0B, $2E, $13, $08, $26, $14, $2C, $20, $25, $02, $17, $07, $03, $27, $2A, $00, $11, $1C, $0A, $06, $0E, $12, $2F, $16, $19, $10, $29, $0C, $1A, $15, $04, $1F, $24, $28, $2B, $22, $18, $2D, $0D, $1B, $09, $0F, $01, $23, $21 ), {97}
    ( $29, $09, $22, $04, $13, $1D, $2F, $11, $05, $17, $06, $0A, $2B, $1C, $08, $07, $2C, $1A, $0D, $1B, $03, $24, $00, $2E, $02, $2D, $1F, $26, $20, $14, $19, $0C, $1E, $18, $2A, $10, $16, $0B, $01, $21, $12, $23, $0F, $15, $0E, $25, $27, $28 ), {98}
    ( $01, $25, $06, $20, $27, $28, $02, $1E, $09, $1C, $17, $2C, $16, $13, $2E, $29, $0F, $1D, $00, $0E, $21, $2A, $24, $1F, $2D, $0D, $11, $12, $14, $19, $15, $2B, $2F, $22, $10, $0A, $0C, $03, $07, $0B, $26, $08, $04, $1B, $23, $1A, $18, $05 ), {99}
    ( $20, $19, $1F, $11, $0D, $06, $21, $17, $2E, $2B, $1E, $10, $13, $03, $23, $05, $22, $0A, $2C, $15, $0C, $1B, $28, $1D, $1C, $0E, $2D, $02, $29, $2A, $07, $08, $12, $09, $16, $04, $14, $0B, $27, $0F, $24, $2F, $1A, $01, $18, $00, $26, $25 ), {9A}
    ( $1A, $18, $07, $01, $17, $02, $0D, $04, $0C, $2E, $03, $21, $20, $1D, $09, $2F, $25, $16, $1E, $11, $08, $1C, $19, $12, $27, $00, $0A, $2A, $2B, $24, $0F, $0E, $2C, $26, $1B, $2D, $15, $13, $28, $06, $14, $0B, $22, $23, $10, $29, $05, $1F ), {9B}
    ( $01, $2D, $0B, $04, $11, $25, $1B, $0E, $15, $14, $06, $2C, $10, $2F, $1C, $1A, $1E, $00, $21, $02, $16, $12, $07, $0D, $28, $26, $2A, $2E, $19, $22, $29, $20, $03, $24, $1D, $27, $0F, $1F, $18, $0A, $05, $13, $23, $08, $0C, $09, $17, $2B ), {9C}
    ( $08, $18, $1D, $0D, $28, $14, $01, $16, $27, $05, $0B, $06, $0A, $03, $12, $09, $13, $10, $00, $19, $07, $1E, $20, $2E, $0F, $0C, $04, $1A, $24, $2A, $25, $0E, $1B, $2F, $11, $1F, $29, $22, $02, $15, $2B, $21, $2D, $1C, $23, $26, $2C, $17 ), {9D}
    ( $09, $02, $2D, $0B, $10, $06, $25, $2C, $2E, $16, $2F, $1C, $0E, $0C, $18, $08, $14, $13, $04, $27, $03, $2A, $1D, $05, $15, $20, $23, $24, $22, $2B, $26, $0D, $01, $1F, $0A, $1A, $11, $00, $28, $17, $21, $0F, $29, $07, $1E, $1B, $19, $12 ), {9E}
    ( $08, $09, $0B, $27, $2B, $14, $0E, $03, $13, $20, $1F, $21, $10, $05, $1B, $04, $26, $24, $01, $16, $25, $22, $1E, $15, $1C, $06, $1D, $0F, $17, $0A, $18, $12, $02, $07, $2F, $2A, $2C, $11, $0D, $0C, $19, $23, $2E, $29, $00, $1A, $28, $2D ), {9F}
    ( $17, $2F, $25, $19, $22, $29, $13, $2B, $20, $1B, $18, $26, $12, $2E, $07, $27, $04, $0D, $08, $03, $15, $06, $02, $16, $01, $21, $0C, $1D, $0E, $05, $0F, $28, $00, $24, $11, $1E, $14, $10, $09, $0B, $1A, $2C, $1F, $2D, $23, $2A, $0A, $1C ), {A0}
    ( $0D, $14, $08, $06, $1C, $1A, $25, $1D, $00, $03, $0E, $11, $16, $07, $10, $23, $18, $2C, $17, $2D, $28, $0F, $15, $20, $1F, $02, $26, $2A, $0A, $12, $13, $2E, $2B, $0B, $24, $22, $1E, $29, $0C, $1B, $01, $21, $27, $2F, $04, $09, $19, $05 ), {A1}
    ( $28, $12, $05, $2A, $2F, $02, $01, $15, $07, $08, $09, $1C, $1B, $19, $25, $24, $17, $21, $16, $1F, $27, $1D, $14, $1A, $22, $04, $2E, $03, $23, $1E, $20, $06, $29, $18, $11, $0E, $0F, $2D, $00, $2B, $13, $10, $2C, $0D, $0C, $0B, $26, $0A ), {A2}
    ( $2D, $0D, $18, $23, $19, $13, $1E, $20, $16, $25, $2F, $14, $27, $1C, $10, $29, $06, $00, $2B, $1A, $1B, $0C, $2E, $07, $1D, $22, $2C, $0F, $28, $26, $01, $24, $12, $0A, $15, $04, $21, $1F, $2A, $17, $0E, $02, $0B, $03, $05, $11, $09, $08 ), {A3}
    ( $21, $1B, $04, $2E, $09, $1C, $10, $06, $19, $18, $1F, $12, $1E, $0E, $05, $0D, $01, $07, $16, $23, $2C, $26, $2D, $13, $0C, $2A, $2F, $1D, $20, $24, $03, $02, $0F, $00, $14, $0A, $29, $1A, $28, $0B, $17, $25, $08, $22, $27, $2B, $15, $11 ), {A4}
    ( $18, $10, $2C, $20, $08, $24, $1F, $15, $14, $22, $02, $25, $29, $2A, $2B, $01, $19, $2E, $0C, $03, $16, $06, $21, $2D, $2F, $00, $28, $13, $26, $1B, $11, $12, $23, $0F, $0A, $1D, $27, $07, $17, $1C, $04, $0B, $1A, $1E, $05, $09, $0D, $0E ), {A5}
    ( $16, $04, $0C, $07, $02, $23, $01, $29, $2A, $1D, $20, $12, $28, $17, $0A, $1A, $24, $1C, $11, $00, $1B, $13, $0E, $1E, $0F, $15, $21, $08, $27, $0B, $22, $2F, $2B, $0D, $19, $06, $1F, $26, $09, $2D, $18, $10, $2E, $2C, $03, $25, $05, $14 ), {A6}
    ( $13, $23, $0D, $2D, $00, $09, $0A, $24, $20, $1A, $14, $02, $26, $19, $2F, $15, $06, $0C, $17, $29, $1E, $2A, $27, $2E, $03, $0B, $08, $05, $1F, $22, $16, $12, $25, $1D, $1B, $1C, $2B, $2C, $0F, $07, $21, $0E, $18, $01, $04, $11, $28, $10 ), {A7}
    ( $10, $04, $2E, $2C, $1F, $12, $0B, $2A, $2D, $21, $24, $27, $0F, $2F, $18, $26, $29, $25, $1B, $28, $00, $1E, $15, $1D, $16, $11, $06, $0D, $13, $1C, $19, $09, $01, $05, $20, $22, $02, $17, $23, $08, $0C, $0A, $07, $1A, $14, $03, $2B, $0E ), {A8}
    ( $09, $15, $2C, $20, $02, $14, $10, $06, $1D, $27, $22, $05, $07, $0C, $24, $1C, $17, $01, $13, $08, $2A, $2D, $00, $04, $03, $1B, $18, $2B, $1A, $25, $12, $1E, $21, $0A, $16, $26, $0B, $23, $2F, $29, $1F, $2E, $0D, $0F, $28, $11, $0E, $19 ), {A9}
    ( $0B, $04, $1F, $27, $07, $10, $29, $0E, $03, $06, $15, $0C, $0A, $00, $21, $2D, $2C, $24, $14, $05, $09, $0F, $1D, $08, $11, $28, $2F, $01, $1E, $02, $2B, $13, $19, $0D, $2A, $1C, $26, $12, $18, $2E, $25, $1A, $17, $20, $23, $16, $1B, $22 ), {AA}
    ( $12, $0F, $0C, $22, $0E, $04, $27, $28, $25, $18, $05, $2C, $1D, $1C, $02, $10, $00, $17, $06, $20, $03, $2E, $0A, $23, $01, $1E, $19, $21, $15, $09, $24, $2A, $1A, $2B, $0B, $16, $1F, $0D, $08, $1B, $2F, $14, $11, $26, $29, $07, $13, $2D ), {AB}
    ( $1E, $0A, $24, $14, $13, $01, $16, $12, $2F, $06, $08, $2E, $0E, $29, $15, $21, $2A, $22, $20, $07, $00, $2D, $1A, $28, $1C, $0F, $17, $03, $25, $04, $09, $19, $0C, $05, $27, $26, $18, $0B, $1F, $02, $1D, $2C, $1B, $11, $10, $2B, $23, $0D ), {AC}
    ( $2E, $16, $20, $01, $06, $2B, $1A, $1D, $24, $04, $2C, $17, $13, $26, $28, $09, $05, $2D, $14, $12, $03, $10, $0C, $1B, $29, $11, $21, $07, $15, $1F, $08, $0A, $22, $2F, $1C, $19, $25, $1E, $0E, $2A, $0D, $02, $27, $23, $00, $0B, $18, $0F ), {AD}
    ( $15, $20, $05, $0C, $11, $04, $14, $16, $2F, $1D, $1E, $22, $01, $0A, $27, $2E, $0E, $2B, $2C, $1A, $28, $25, $06, $09, $07, $13, $18, $23, $26, $0D, $10, $2A, $1F, $08, $29, $21, $17, $1B, $02, $24, $2D, $19, $0F, $0B, $1C, $12, $00, $03 ), {AE}
    ( $0F, $1A, $23, $15, $16, $2E, $1D, $0B, $00, $24, $14, $09, $10, $17, $2A, $25, $2B, $1E, $0A, $27, $1F, $26, $05, $2C, $1B, $1C, $0E, $19, $04, $0C, $08, $20, $2D, $13, $28, $18, $02, $07, $12, $03, $11, $22, $2F, $0D, $29, $21, $01, $06 ), {AF}
    ( $10, $07, $1C, $1F, $02, $20, $2A, $2D, $2C, $03, $2B, $1B, $11, $0C, $26, $29, $00, $0A, $23, $0E, $28, $2F, $04, $16, $01, $17, $2E, $25, $0F, $21, $22, $0B, $14, $05, $0D, $08, $27, $1E, $18, $19, $12, $13, $06, $1D, $1A, $24, $09, $15 ), {B0}
    ( $09, $25, $19, $02, $2B, $1B, $13, $2E, $17, $01, $15, $2F, $05, $0B, $28, $1E, $0E, $23, $29, $21, $2D, $16, $10, $24, $03, $2C, $0D, $18, $26, $1C, $27, $1A, $00, $1D, $08, $11, $0F, $1F, $04, $14, $07, $12, $20, $2A, $06, $0C, $22, $0A ), {B1}
    ( $11, $10, $1F, $09, $05, $1D, $1E, $0B, $02, $2A, $27, $2D, $07, $26, $00, $06, $0F, $20, $0E, $24, $08, $18, $1B, $2C, $21, $13, $03, $15, $23, $04, $14, $0D, $1C, $2F, $19, $17, $28, $1A, $0C, $25, $2B, $01, $22, $2E, $12, $0A, $29, $16 ), {B2}
    ( $08, $17, $0E, $0B, $22, $02, $13, $14, $04, $1C, $26, $20, $10, $06, $15, $1B, $2A, $0F, $2C, $23, $11, $28, $18, $09, $1D, $24, $2B, $1A, $00, $03, $0D, $2D, $01, $27, $21, $19, $1E, $2E, $07, $29, $1F, $12, $0A, $2F, $05, $16, $25, $0C ), {B3}
    ( $18, $02, $05, $23, $0A, $20, $01, $1E, $0D, $06, $2A, $10, $04, $0B, $27, $03, $07, $24, $1A, $1B, $15, $0E, $1C, $25, $0C, $2B, $29, $12, $19, $28, $2F, $09, $08, $2E, $2C, $1D, $14, $1F, $16, $26, $22, $2D, $00, $11, $17, $21, $13, $0F ), {B4}
    ( $19, $1A, $27, $02, $28, $14, $17, $12, $18, $11, $03, $26, $2A, $09, $23, $13, $0C, $25, $2C, $0A, $2D, $2E, $01, $00, $04, $1D, $24, $05, $2B, $0E, $20, $1B, $07, $0F, $15, $08, $22, $1E, $2F, $16, $10, $06, $1C, $0D, $0B, $1F, $21, $29 ), {B5}
    ( $01, $13, $2A, $0C, $26, $07, $20, $22, $04, $17, $15, $03, $16, $0E, $19, $1C, $06, $2D, $24, $1B, $08, $00, $12, $2F, $0D, $1F, $05, $09, $1E, $0B, $25, $18, $28, $23, $1D, $2E, $14, $27, $29, $2C, $1A, $11, $21, $0A, $10, $2B, $0F, $02 ), {B6}
    ( $25, $11, $13, $1F, $2D, $2E, $00, $0B, $23, $2A, $28, $0D, $1C, $0F, $1B, $27, $18, $05, $1E, $22, $1D, $2C, $2B, $04, $07, $24, $10, $2F, $09, $17, $12, $0E, $0C, $14, $20, $19, $01, $26, $16, $02, $29, $0A, $06, $08, $15, $1A, $21, $03 ), {B7}
    ( $1C, $0B, $00, $25, $20, $07, $16, $27, $29, $2C, $1A, $03, $21, $2D, $06, $11, $1D, $1F, $19, $10, $09, $23, $18, $15, $04, $1B, $0C, $2E, $2F, $0A, $01, $1E, $26, $13, $02, $0D, $2A, $22, $28, $17, $08, $12, $0E, $14, $2B, $24, $05, $0F ), {B8}
    ( $1F, $13, $03, $27, $21, $26, $2D, $0D, $15, $16, $23, $29, $11, $2C, $2B, $1D, $0A, $12, $2E, $0B, $24, $25, $17, $0E, $20, $2A, $0F, $08, $02, $2F, $04, $1A, $1B, $00, $09, $05, $1C, $18, $01, $07, $14, $10, $0C, $28, $06, $22, $19, $1E ), {B9}
    ( $2D, $07, $27, $29, $12, $22, $21, $19, $04, $26, $18, $05, $28, $0C, $14, $03, $2A, $1A, $0E, $09, $1E, $16, $2B, $08, $1B, $13, $01, $00, $2C, $23, $1F, $0A, $15, $0B, $25, $0F, $0D, $1C, $06, $10, $2E, $24, $2F, $02, $20, $1D, $11, $17 ), {BA}
    ( $0E, $1C, $14, $1E, $18, $2D, $08, $13, $15, $1B, $05, $07, $01, $06, $2F, $0B, $0C, $27, $09, $0A, $21, $24, $29, $22, $19, $23, $25, $11, $17, $00, $26, $04, $03, $16, $2E, $28, $2A, $0D, $02, $1D, $1F, $1A, $10, $2C, $12, $2B, $0F, $20 ), {BB}
    ( $12, $24, $0B, $06, $20, $0A, $28, $02, $10, $18, $27, $0F, $03, $00, $0D, $1B, $13, $0C, $26, $0E, $1C, $09, $1F, $29, $2E, $1D, $08, $23, $1A, $01, $11, $2B, $07, $05, $17, $22, $15, $2A, $2D, $2F, $2C, $16, $1E, $04, $19, $21, $25, $14 ), {BC}
    ( $0F, $15, $0E, $0D, $14, $11, $2C, $04, $2B, $16, $19, $18, $2A, $23, $1D, $29, $1E, $0B, $2F, $0C, $02, $10, $25, $2E, $1A, $00, $09, $13, $0A, $1F, $28, $03, $12, $06, $27, $26, $2D, $01, $1B, $21, $08, $1C, $20, $24, $22, $05, $17, $07 ), {BD}
    ( $09, $1D, $2F, $07, $20, $27, $0B, $21, $1A, $1F, $04, $29, $18, $0A, $22, $10, $2A, $05, $15, $2B, $01, $26, $0F, $13, $14, $25, $03, $2C, $02, $08, $06, $1C, $19, $0C, $1B, $00, $0E, $12, $23, $11, $0D, $17, $2E, $16, $28, $1E, $24, $2D ), {BE}
    ( $0F, $2F, $07, $08, $19, $15, $23, $20, $12, $2D, $2C, $03, $04, $11, $01, $26, $1D, $0D, $27, $02, $0C, $2B, $0A, $2E, $1B, $05, $16, $18, $22, $13, $17, $0E, $09, $1E, $25, $24, $1C, $0B, $1F, $29, $2A, $28, $21, $06, $14, $1A, $00, $10 ), {BF}
    ( $03, $2C, $0C, $22, $05, $2B, $29, $11, $19, $12, $07, $1E, $23, $04, $1D, $1F, $26, $1A, $02, $0D, $15, $0B, $13, $10, $21, $18, $17, $08, $28, $2D, $09, $14, $06, $25, $16, $00, $0E, $24, $0F, $01, $27, $2F, $0A, $1B, $1C, $2E, $2A, $20 ), {C0}
    ( $26, $2D, $14, $01, $1B, $24, $07, $20, $05, $1D, $11, $12, $21, $29, $0D, $0A, $19, $17, $13, $22, $0C, $28, $09, $1A, $15, $27, $2A, $0B, $2C, $02, $2F, $16, $2E, $06, $1E, $0E, $03, $00, $10, $04, $08, $18, $1C, $0F, $23, $2B, $25, $1F ), {C1}
    ( $20, $2E, $2D, $11, $21, $19, $25, $0E, $24, $1E, $06, $13, $05, $23, $28, $12, $29, $2A, $07, $0A, $00, $18, $1B, $27, $2C, $22, $10, $16, $01, $08, $17, $03, $14, $0D, $04, $2B, $0C, $15, $0F, $1D, $09, $1F, $02, $2F, $0B, $1A, $1C, $26 ), {C2}
    ( $25, $26, $0E, $1D, $11, $0A, $1C, $1F, $19, $1A, $03, $09, $1B, $08, $24, $29, $0B, $22, $01, $04, $17, $02, $23, $1E, $2D, $2E, $05, $14, $18, $15, $27, $21, $2C, $00, $06, $20, $2F, $28, $0D, $16, $2B, $07, $10, $12, $2A, $13, $0F, $0C ), {C3}
    ( $07, $28, $12, $19, $1B, $0B, $23, $15, $26, $25, $0C, $2D, $16, $0A, $1E, $21, $1A, $02, $03, $0F, $2A, $04, $1F, $10, $2B, $06, $2F, $0E, $01, $14, $2E, $0D, $17, $2C, $09, $18, $22, $13, $11, $00, $29, $1D, $24, $05, $27, $20, $1C, $08 ), {C4}
    ( $0C, $14, $08, $0A, $02, $09, $05, $23, $24, $01, $0B, $10, $27, $03, $11, $1D, $1F, $2F, $1C, $22, $13, $21, $29, $16, $00, $20, $28, $19, $18, $15, $0E, $07, $06, $2A, $1A, $26, $0F, $2C, $25, $2B, $17, $0D, $04, $2D, $12, $2E, $1B, $1E ), {C5}
    ( $25, $03, $1E, $21, $1F, $06, $0E, $10, $1B, $0C, $02, $26, $23, $1C, $2E, $24, $01, $16, $28, $0D, $07, $13, $1D, $2D, $19, $0A, $22, $2A, $2F, $11, $29, $20, $08, $0B, $2C, $14, $09, $0F, $04, $00, $15, $17, $2B, $27, $1A, $05, $12, $18 ), {C6}
    ( $14, $0F, $2D, $19, $26, $22, $1E, $0D, $17, $02, $06, $24, $2B, $09, $2A, $18, $11, $03, $2F, $16, $10, $29, $1C, $20, $15, $27, $13, $01, $23, $2E, $00, $0E, $12, $2C, $1A, $07, $04, $0A, $0B, $1F, $05, $21, $08, $28, $25, $1D, $0C, $1B ), {C7}
    ( $0C, $1F, $24, $13, $07, $2D, $1A, $2C, $01, $10, $25, $1D, $02, $0B, $16, $27, $1C, $1B, $20, $22, $29, $12, $26, $1E, $23, $08, $18, $28, $2F, $0D, $06, $03, $21, $19, $2B, $17, $0A, $14, $00, $2E, $15, $04, $05, $0E, $11, $0F, $2A, $09 ), {C8}
    ( $24, $1C, $11, $01, $16, $1E, $0C, $0D, $2E, $26, $18, $17, $09, $1D, $0F, $13, $03, $1F, $29, $15, $1B, $27, $23, $0A, $2A, $04, $05, $1A, $22, $12, $14, $10, $00, $0E, $2F, $02, $07, $19, $2B, $08, $2C, $20, $06, $0B, $2D, $25, $21, $28 ), {C9}
    ( $22, $00, $12, $14, $11, $0B, $15, $24, $28, $2F, $23, $10, $03, $1A, $1F, $1E, $2B, $2E, $13, $1D, $21, $18, $0A, $0F, $08, $25, $02, $2A, $1B, $2C, $01, $05, $26, $07, $2D, $06, $17, $16, $29, $0E, $27, $09, $19, $1C, $0D, $04, $20, $0C ), {CA}
    ( $27, $1E, $1A, $22, $17, $11, $24, $20, $1F, $19, $0D, $05, $10, $2D, $09, $2B, $13, $01, $04, $1C, $00, $0B, $2C, $28, $06, $29, $2A, $1D, $23, $0F, $0C, $2E, $1B, $18, $08, $02, $0A, $03, $21, $26, $0E, $16, $12, $25, $15, $2F, $14, $07 ), {CB}
    ( $1B, $0E, $07, $02, $12, $25, $1E, $15, $2D, $10, $13, $0A, $14, $22, $20, $06, $19, $2F, $2C, $21, $1D, $11, $09, $26, $2B, $17, $08, $1C, $0B, $18, $16, $27, $29, $24, $00, $0D, $23, $2A, $01, $1A, $2E, $0F, $03, $1F, $28, $05, $04, $0C ), {CC}
    ( $1C, $17, $0C, $05, $25, $06, $2C, $22, $07, $27, $10, $09, $2B, $19, $0B, $15, $1B, $18, $14, $04, $2F, $1E, $0E, $08, $13, $1D, $03, $24, $0F, $1A, $21, $11, $28, $2A, $23, $1F, $2D, $02, $12, $01, $29, $00, $16, $2E, $20, $0D, $0A, $26 ), {CD}
    ( $1F, $29, $1A, $1D, $0F, $07, $14, $24, $1B, $18, $2F, $21, $15, $12, $06, $04, $11, $05, $0E, $27, $02, $0C, $0B, $10, $2A, $16, $28, $2C, $20, $1C, $2E, $0A, $1E, $0D, $00, $17, $25, $2D, $22, $23, $26, $2B, $19, $01, $08, $09, $13, $03 ), {CE}
    ( $0C, $15, $1C, $1A, $1E, $2C, $1D, $00, $19, $02, $2B, $0A, $04, $26, $25, $21, $1B, $28, $10, $16, $23, $22, $2E, $07, $11, $0F, $0D, $09, $08, $2D, $20, $13, $03, $0B, $1F, $01, $2F, $18, $06, $0E, $24, $2A, $12, $05, $17, $29, $14, $27 ), {CF}
    ( $26, $0B, $13, $23, $2F, $2D, $17, $1F, $16, $0F, $21, $20, $14, $15, $2A, $1E, $1C, $0A, $25, $00, $27, $0D, $12, $04, $1A, $0C, $07, $08, $1B, $03, $0E, $2C, $24, $22, $2E, $11, $02, $09, $10, $01, $05, $19, $2B, $29, $28, $06, $18, $1D ), {D0}
    ( $1A, $20, $1C, $04, $1F, $2B, $07, $06, $22, $11, $2F, $00, $01, $18, $15, $28, $19, $0C, $17, $1B, $0A, $16, $05, $0F, $23, $29, $0B, $2C, $03, $10, $27, $08, $2A, $24, $12, $02, $0D, $1E, $1D, $0E, $14, $2E, $09, $2D, $25, $21, $26, $13 ), {D1}
    ( $1F, $19, $23, $1B, $01, $08, $15, $10, $0D, $1D, $2A, $13, $17, $22, $07, $26, $1A, $09, $04, $06, $2C, $24, $27, $0A, $0B, $1E, $05, $25, $0E, $03, $20, $0C, $02, $28, $2B, $2D, $0F, $2E, $12, $11, $2F, $1C, $21, $29, $16, $00, $14, $18 ), {D2}
    ( $2B, $02, $19, $14, $25, $15, $1B, $05, $1E, $23, $00, $2E, $03, $26, $10, $2D, $22, $1F, $29, $0A, $09, $21, $07, $1C, $2C, $16, $01, $04, $2A, $0D, $0C, $1D, $1A, $08, $0F, $20, $0B, $24, $0E, $2F, $12, $11, $28, $17, $27, $18, $06, $13 ), {D3}
    ( $2C, $00, $21, $28, $24, $1D, $0C, $23, $15, $18, $2E, $22, $04, $07, $13, $20, $1F, $29, $27, $1E, $19, $1B, $2B, $0D, $14, $06, $12, $26, $08, $0A, $25, $2D, $01, $17, $03, $0F, $05, $02, $11, $1C, $2F, $0E, $16, $1A, $2A, $10, $09, $0B ), {D4}
    ( $24, $21, $06, $20, $0E, $2C, $05, $2E, $1D, $14, $19, $11, $2B, $1E, $2A, $26, $03, $02, $23, $04, $27, $09, $17, $25, $29, $28, $00, $0C, $13, $08, $0D, $1A, $0F, $2D, $15, $01, $1F, $1B, $0A, $18, $1C, $2F, $22, $07, $0B, $16, $12, $10 ), {D5}
    ( $04, $0F, $20, $10, $19, $0A, $02, $11, $25, $01, $26, $12, $2E, $2D, $1E, $08, $15, $06, $2A, $2F, $00, $0D, $29, $2C, $0C, $2B, $23, $1F, $07, $0E, $13, $18, $17, $27, $16, $22, $28, $05, $1B, $1C, $09, $0B, $21, $03, $1D, $14, $1A, $24 ), {D6}
    ( $1E, $2C, $06, $1B, $21, $0B, $13, $12, $00, $22, $0F, $14, $15, $0C, $1A, $1C, $0D, $26, $07, $11, $2B, $01, $24, $1D, $25, $23, $16, $0A, $2E, $20, $05, $02, $04, $19, $08, $03, $29, $27, $17, $0E, $2A, $2D, $28, $10, $1F, $18, $2F, $09 ), {D7}
    ( $1C, $08, $0F, $0C, $06, $13, $2D, $04, $01, $23, $14, $27, $28, $2A, $10, $17, $00, $1E, $26, $03, $05, $0A, $20, $11, $19, $02, $2F, $09, $25, $18, $07, $1B, $22, $0D, $0B, $21, $2E, $2C, $2B, $1F, $24, $12, $15, $1A, $0E, $1D, $29, $16 ), {D8}
    ( $1A, $15, $08, $1D, $2D, $06, $12, $1E, $2B, $2A, $29, $07, $2C, $2F, $25, $2E, $03, $09, $02, $20, $0D, $0B, $04, $27, $24, $28, $01, $19, $00, $1B, $18, $0F, $13, $0E, $0A, $16, $17, $11, $1C, $10, $22, $05, $1F, $0C, $21, $23, $26, $14 ), {D9}
    ( $2F, $2C, $1C, $28, $2A, $21, $0F, $13, $16, $01, $18, $04, $12, $15, $09, $24, $0D, $1E, $03, $2D, $23, $07, $00, $05, $29, $1D, $25, $17, $27, $06, $1F, $19, $10, $11, $14, $2B, $0E, $1A, $1B, $0C, $0A, $20, $22, $08, $2E, $0B, $02, $26 ), {DA}
    ( $0E, $2B, $16, $2A, $1F, $23, $2C, $15, $12, $00, $10, $17, $05, $13, $11, $02, $20, $2E, $1A, $25, $0C, $2F, $04, $06, $08, $0D, $1D, $14, $07, $24, $1B, $1E, $03, $19, $01, $2D, $0A, $1C, $27, $21, $09, $22, $0B, $28, $26, $18, $29, $0F ), {DB}
    ( $1C, $29, $2C, $0A, $26, $16, $2F, $01, $0E, $24, $1D, $1E, $13, $06, $12, $2E, $08, $1A, $10, $18, $2B, $22, $0D, $27, $04, $15, $00, $07, $02, $19, $2A, $20, $17, $28, $11, $14, $05, $1F, $03, $0F, $23, $25, $21, $0B, $2D, $0C, $1B, $09 ), {DC}
    ( $06, $0A, $24, $18, $2D, $10, $00, $1C, $26, $28, $04, $23, $09, $1D, $1F, $05, $21, $1E, $14, $17, $0D, $2A, $07, $1B, $27, $0E, $2B, $2C, $20, $11, $1A, $2F, $25, $13, $02, $22, $0F, $0C, $0B, $15, $16, $08, $01, $19, $03, $2E, $12, $29 ), {DD}
    ( $2E, $22, $19, $10, $1C, $03, $21, $2C, $0F, $2F, $1B, $28, $06, $2A, $25, $11, $16, $00, $09, $2B, $1F, $0C, $2D, $0E, $02, $07, $15, $1E, $14, $05, $26, $17, $1A, $04, $18, $27, $08, $12, $01, $29, $23, $20, $0A, $13, $1D, $24, $0B, $0D ), {DE}
    ( $22, $2B, $01, $16, $27, $0E, $0B, $19, $21, $26, $2E, $1D, $1F, $12, $1A, $23, $1B, $0A, $08, $1E, $03, $28, $17, $2D, $07, $0F, $05, $20, $04, $2F, $2C, $29, $0D, $25, $06, $15, $14, $0C, $2A, $13, $09, $00, $18, $10, $02, $1C, $24, $11 ), {DF}
    ( $06, $11, $22, $04, $20, $2A, $0A, $0C, $17, $00, $01, $29, $08, $19, $1B, $03, $26, $15, $27, $2E, $1F, $21, $28, $18, $05, $1D, $0B, $0E, $12, $25, $14, $0D, $2C, $0F, $2B, $2F, $07, $16, $24, $1E, $1C, $10, $13, $1A, $23, $02, $2D, $09 ), {E0}
    ( $08, $22, $03, $24, $21, $1A, $19, $27, $09, $01, $00, $0D, $2B, $29, $2A, $28, $2D, $0C, $11, $14, $26, $0B, $2F, $1B, $0A, $25, $23, $0F, $15, $05, $1C, $16, $13, $1E, $1F, $1D, $06, $2E, $12, $17, $04, $07, $10, $0E, $18, $20, $2C, $02 ), {E1}
    ( $1C, $1E, $24, $00, $07, $25, $04, $06, $2D, $0E, $18, $2E, $12, $20, $17, $2F, $22, $0B, $0D, $19, $02, $27, $28, $2C, $1B, $2A, $29, $09, $03, $14, $26, $2B, $05, $23, $10, $11, $01, $0C, $0A, $13, $1F, $21, $16, $1D, $15, $08, $1A, $0F ), {E2}
    ( $26, $18, $2C, $05, $19, $11, $23, $2E, $21, $15, $01, $0D, $0B, $25, $2D, $0E, $1A, $27, $22, $06, $1E, $29, $09, $1C, $16, $20, $13, $0F, $1D, $04, $2A, $0A, $2B, $14, $07, $03, $02, $10, $08, $28, $12, $1B, $1F, $0C, $24, $2F, $17, $00 ), {E3}
    ( $02, $1A, $13, $23, $00, $25, $21, $10, $1C, $0B, $08, $17, $0D, $2B, $27, $04, $0A, $28, $0C, $29, $01, $19, $18, $14, $11, $16, $1E, $0E, $05, $24, $1D, $2F, $2A, $12, $06, $22, $20, $15, $03, $1F, $1B, $26, $0F, $2C, $2D, $07, $09, $2E ), {E4}
    ( $29, $26, $2E, $13, $28, $16, $05, $14, $2B, $0D, $07, $0E, $08, $22, $1F, $0C, $04, $19, $20, $2F, $09, $0F, $10, $06, $24, $0B, $15, $23, $1B, $2D, $1A, $12, $18, $00, $17, $25, $01, $0A, $21, $11, $1C, $02, $1D, $1E, $27, $2A, $2C, $03 ), {E5}
    ( $02, $05, $16, $06, $00, $2C, $17, $19, $0C, $13, $2B, $1F, $1D, $11, $04, $0E, $0D, $09, $1E, $15, $2A, $23, $14, $0A, $2D, $28, $18, $1C, $2E, $26, $29, $1B, $01, $03, $12, $20, $1A, $24, $0F, $21, $25, $10, $2F, $27, $07, $22, $0B, $08 ), {E6}
    ( $1D, $25, $27, $29, $09, $08, $02, $2A, $10, $22, $2E, $1E, $20, $0B, $00, $26, $1C, $18, $13, $03, $28, $2B, $2C, $1A, $17, $23, $0C, $2F, $0D, $04, $0A, $06, $2D, $16, $01, $14, $1F, $0F, $11, $19, $21, $05, $0E, $15, $24, $12, $07, $1B ), {E7}
    ( $01, $09, $1D, $2D, $26, $15, $2C, $06, $17, $29, $12, $0D, $1C, $23, $21, $0A, $0B, $10, $28, $20, $27, $08, $0C, $02, $2E, $2A, $1E, $03, $11, $19, $13, $04, $22, $1B, $0F, $25, $1A, $2F, $14, $00, $16, $24, $05, $07, $18, $1F, $2B, $0E ), {E8}
    ( $1D, $18, $10, $12, $13, $1C, $1A, $2B, $06, $16, $02, $22, $0F, $11, $14, $17, $2D, $05, $07, $28, $23, $19, $27, $2C, $08, $15, $1F, $2A, $1E, $00, $0A, $25, $1B, $09, $0D, $2F, $0B, $03, $0E, $01, $20, $0C, $21, $24, $04, $2E, $26, $29 ), {E9}
    ( $27, $20, $21, $01, $1A, $11, $2B, $2A, $0D, $14, $0C, $15, $03, $1B, $2F, $10, $07, $23, $05, $16, $1F, $13, $0B, $08, $00, $17, $2D, $06, $09, $1E, $2E, $2C, $18, $0F, $25, $29, $12, $04, $19, $1C, $24, $0A, $1D, $0E, $02, $26, $28, $22 ), {EA}
    ( $05, $2D, $23, $27, $08, $29, $0A, $25, $2A, $19, $12, $1C, $2E, $01, $20, $1B, $0E, $10, $1A, $26, $2B, $1E, $28, $22, $09, $11, $18, $1F, $00, $24, $03, $06, $13, $04, $17, $2C, $14, $1D, $0D, $07, $2F, $15, $16, $02, $0B, $0C, $0F, $21 ), {EB}
    ( $1B, $00, $09, $2F, $1E, $07, $2D, $2C, $1A, $2A, $0F, $06, $0B, $10, $21, $1D, $17, $08, $26, $1C, $25, $0E, $0D, $19, $05, $15, $11, $04, $03, $13, $14, $2B, $0C, $18, $02, $28, $29, $20, $1F, $12, $2E, $23, $0A, $24, $27, $16, $22, $01 ), {EC}
    ( $23, $16, $0C, $26, $20, $24, $0B, $28, $2D, $1C, $18, $13, $0E, $1B, $2F, $1E, $01, $2C, $1D, $09, $17, $0A, $12, $00, $2E, $04, $15, $22, $07, $29, $08, $25, $1F, $10, $2B, $06, $11, $02, $0D, $0F, $19, $05, $1A, $27, $21, $14, $03, $2A ), {ED}
    ( $07, $2D, $1F, $17, $23, $22, $13, $12, $02, $25, $1A, $2F, $1B, $04, $29, $2B, $1C, $05, $24, $26, $1E, $11, $03, $06, $19, $0D, $28, $0C, $2A, $27, $0A, $10, $1D, $14, $0F, $0B, $2C, $20, $18, $15, $08, $16, $00, $09, $2E, $21, $0E, $01 ), {EE}
    ( $04, $21, $10, $0D, $2E, $08, $1B, $0B, $0F, $03, $05, $01, $20, $24, $16, $0E, $1A, $06, $02, $0C, $2D, $27, $1D, $11, $0A, $25, $12, $17, $18, $26, $2F, $19, $29, $23, $2A, $15, $00, $28, $1E, $14, $2C, $1F, $13, $22, $09, $07, $2B, $1C ), {EF}
    ( $2F, $23, $2A, $27, $0A, $17, $2E, $19, $24, $08, $10, $2D, $00, $0B, $0C, $20, $11, $1D, $14, $28, $02, $16, $13, $12, $1E, $09, $1F, $25, $0E, $07, $1C, $26, $2B, $1B, $18, $06, $21, $29, $1A, $0F, $04, $03, $22, $15, $05, $2C, $01, $0D ), {F0}
    ( $1F, $0A, $0B, $2A, $2E, $13, $25, $18, $0D, $28, $1D, $1C, $2F, $0F, $15, $22, $04, $23, $27, $14, $29, $0E, $1E, $21, $10, $01, $08, $11, $1B, $02, $2D, $17, $05, $20, $00, $19, $26, $2C, $06, $03, $2B, $0C, $16, $12, $1A, $07, $24, $09 ), {F1}
    ( $27, $12, $0C, $17, $2A, $11, $20, $2E, $18, $29, $08, $22, $13, $2F, $0B, $1F, $02, $26, $28, $1E, $2C, $06, $19, $03, $1A, $16, $0D, $25, $24, $01, $2B, $05, $1C, $15, $1B, $0A, $00, $1D, $07, $23, $09, $2D, $10, $14, $0F, $04, $0E, $21 ), {F2}
    ( $1E, $0C, $0D, $08, $2B, $28, $0B, $2A, $22, $0F, $13, $19, $27, $05, $23, $12, $24, $09, $0A, $1F, $1D, $18, $02, $2C, $06, $10, $07, $0E, $14, $17, $04, $1A, $16, $11, $03, $1C, $25, $29, $1B, $26, $2D, $21, $01, $15, $2F, $2E, $00, $20 ), {F3}
    ( $09, $02, $2C, $2D, $14, $2F, $11, $0F, $0C, $07, $23, $2A, $21, $27, $25, $04, $1D, $0E, $0D, $05, $06, $0B, $01, $1E, $1F, $26, $10, $00, $08, $16, $1C, $29, $12, $28, $1A, $1B, $2E, $2B, $15, $03, $18, $22, $0A, $19, $20, $17, $24, $13 ), {F4}
    ( $18, $28, $2F, $05, $00, $16, $15, $0B, $03, $23, $07, $01, $11, $2E, $1C, $06, $13, $2D, $1A, $14, $27, $22, $21, $29, $0F, $2C, $0A, $08, $12, $20, $0C, $17, $1E, $2B, $1D, $10, $19, $1B, $0D, $09, $02, $2A, $04, $0E, $24, $1F, $26, $25 ), {F5}
    ( $1A, $11, $2C, $07, $06, $1E, $27, $2F, $28, $14, $16, $0C, $26, $2B, $20, $05, $2A, $0F, $0A, $24, $1C, $17, $2D, $1F, $22, $08, $12, $29, $13, $01, $0B, $09, $04, $2E, $00, $25, $1B, $02, $21, $19, $23, $0D, $18, $03, $10, $15, $0E, $1D ), {F6}
    ( $06, $0F, $07, $25, $0A, $04, $13, $18, $27, $03, $2C, $05, $1E, $10, $09, $2E, $24, $02, $0C, $12, $11, $2A, $01, $0B, $17, $1F, $2F, $16, $08, $28, $19, $14, $2B, $2D, $1D, $0E, $20, $21, $0D, $1C, $15, $00, $1B, $29, $1A, $22, $23, $26 ), {F7}
    ( $0F, $26, $17, $16, $07, $1D, $10, $0C, $00, $12, $02, $08, $20, $27, $1E, $05, $28, $04, $2F, $2C, $18, $22, $1A, $29, $0B, $1B, $11, $0D, $23, $21, $15, $24, $25, $1F, $0E, $06, $0A, $2E, $01, $2A, $14, $03, $1C, $09, $19, $2B, $13, $2D ), {F8}
    ( $28, $11, $19, $1B, $06, $18, $16, $23, $1F, $0B, $27, $10, $09, $24, $1C, $25, $00, $26, $0E, $0D, $0A, $05, $2E, $21, $29, $2F, $2D, $03, $0F, $08, $2A, $01, $1E, $02, $17, $2B, $13, $14, $12, $07, $22, $0C, $20, $15, $1D, $04, $1A, $2C ), {F9}
    ( $2D, $2C, $00, $12, $29, $23, $2B, $1F, $26, $1A, $1C, $0E, $11, $2E, $0D, $19, $1B, $1D, $22, $2A, $0F, $04, $03, $28, $24, $07, $09, $10, $16, $1E, $06, $20, $02, $01, $08, $0B, $0C, $21, $0A, $05, $13, $14, $15, $2F, $18, $27, $25, $17 ), {FA}
    ( $22, $0E, $26, $2F, $02, $17, $25, $1C, $14, $12, $27, $20, $08, $05, $1E, $07, $24, $15, $28, $0C, $10, $00, $18, $11, $09, $0B, $1B, $2D, $29, $0F, $01, $0A, $03, $2A, $06, $21, $13, $2B, $19, $23, $16, $2C, $0D, $2E, $1F, $1A, $04, $1D ), {FB}
    ( $02, $28, $17, $0D, $2A, $19, $2E, $22, $0B, $04, $08, $1C, $0A, $10, $20, $03, $15, $21, $23, $06, $1B, $2D, $1D, $0E, $14, $2C, $24, $29, $11, $09, $16, $2B, $12, $26, $2F, $0F, $1F, $1A, $27, $01, $07, $05, $25, $1E, $00, $13, $0C, $18 ), {FC}
    ( $15, $0B, $18, $00, $1F, $1D, $2B, $2F, $23, $28, $14, $07, $24, $27, $05, $0E, $16, $12, $11, $02, $2A, $01, $25, $08, $29, $0D, $1C, $13, $0C, $2C, $22, $06, $21, $09, $1B, $1E, $0F, $0A, $2D, $20, $2E, $04, $1A, $03, $17, $19, $10, $26 ), {FD}
    ( $16, $1B, $2F, $17, $0C, $03, $1A, $2E, $26, $0B, $0D, $10, $12, $18, $2C, $08, $28, $0A, $2A, $00, $1D, $2B, $1F, $0E, $04, $01, $27, $20, $1E, $11, $24, $09, $05, $06, $14, $21, $2D, $07, $15, $02, $1C, $23, $29, $0F, $25, $13, $22, $19 ), {FE}
    ( $12, $19, $21, $04, $2B, $24, $0F, $1C, $10, $05, $08, $11, $22, $1E, $1D, $2D, $15, $07, $2E, $18, $28, $1A, $1B, $2A, $25, $0E, $20, $1F, $06, $0C, $2C, $16, $29, $0D, $0A, $0B, $23, $03, $00, $17, $26, $13, $2F, $14, $27, $01, $02, $09 )  {FF}
  );

const
  TABLE_RND: array[0..255] of TDigestBuf =
  (
    ( $6E, $14, $D3, $F6, $3E, $BE, $46, $8D, $3A, $A3, $72, $3D, $D3, $1A, $79, $F7, $B3, $82, $A9, $24, $F5, $1C, $AA, $7F, $A1, $4A, $56, $60, $6A, $C5, $BD, $72, $AE, $0C, $FA, $AD, $BF, $12, $47, $3A, $73, $F0, $DD, $A5, $9F, $F1, $3C, $F2 ), {00}
    ( $FC, $F4, $D6, $BC, $22, $2C, $6F, $5A, $40, $F9, $D3, $02, $32, $DF, $0A, $35, $F3, $78, $EB, $D2, $F5, $74, $E5, $BB, $02, $45, $CE, $76, $B1, $C4, $E2, $FD, $5A, $37, $DB, $52, $E8, $94, $25, $24, $E4, $23, $18, $B4, $40, $EC, $1A, $32 ), {01}
    ( $33, $18, $F8, $C3, $B5, $FB, $4D, $76, $CC, $39, $FD, $3C, $47, $99, $D6, $C2, $1D, $30, $FA, $28, $7D, $8A, $01, $01, $CC, $72, $89, $3F, $FC, $19, $6F, $6E, $51, $62, $BE, $36, $EB, $3B, $1C, $E0, $CD, $E5, $DB, $14, $72, $B6, $39, $DB ), {02}
    ( $DC, $FA, $5F, $08, $6D, $DE, $A4, $94, $34, $20, $4E, $91, $22, $DC, $4D, $E7, $46, $9E, $50, $D9, $C0, $F3, $91, $F4, $F0, $85, $52, $D7, $7A, $80, $54, $51, $2E, $51, $9A, $FD, $A0, $55, $AB, $73, $C1, $CD, $97, $CD, $9D, $F4, $76, $23 ), {03}
    ( $F4, $D3, $1E, $51, $82, $CA, $0F, $E2, $B3, $D3, $F8, $86, $73, $FB, $C2, $8D, $E1, $8F, $2A, $85, $85, $F1, $BC, $04, $9F, $1C, $D2, $2E, $08, $19, $E9, $CD, $01, $BE, $6B, $72, $7C, $25, $6C, $A5, $87, $4D, $F0, $01, $D5, $00, $27, $0F ), {04}
    ( $1C, $CC, $C8, $80, $89, $30, $C7, $44, $3F, $09, $8F, $6E, $9E, $5E, $F0, $CE, $22, $DA, $B6, $80, $97, $86, $5A, $AB, $66, $A8, $BD, $98, $84, $F5, $00, $D6, $40, $BC, $9E, $60, $08, $54, $16, $ED, $BB, $97, $DA, $60, $BD, $37, $92, $70 ), {05}
    ( $9D, $3E, $7B, $33, $0F, $AA, $82, $DB, $7E, $80, $C1, $F7, $29, $AC, $5E, $02, $33, $7F, $00, $E1, $CD, $69, $76, $F0, $92, $5F, $6C, $21, $59, $45, $D1, $C9, $7C, $12, $81, $AF, $E5, $DF, $FD, $F9, $EE, $68, $F7, $D3, $4F, $C8, $2D, $F4 ), {06}
    ( $A3, $F2, $10, $DF, $39, $B7, $3D, $5C, $2F, $3C, $D5, $EB, $DB, $58, $D7, $63, $25, $C5, $F5, $46, $D7, $59, $52, $C6, $B6, $09, $C0, $94, $D3, $EC, $CC, $4D, $66, $78, $0E, $A4, $3A, $09, $5A, $34, $1B, $A1, $0D, $A3, $DB, $31, $AB, $08 ), {07}
    ( $F2, $3A, $29, $63, $3E, $3C, $74, $FA, $D3, $84, $DC, $04, $F0, $8A, $A3, $DF, $60, $C8, $EE, $E5, $96, $09, $99, $6A, $4B, $CB, $F6, $C0, $2E, $93, $93, $3C, $41, $41, $6E, $A9, $B0, $CA, $D3, $EF, $6E, $AC, $E7, $D3, $16, $C0, $E8, $E1 ), {08}
    ( $AF, $6E, $ED, $05, $32, $59, $B0, $3E, $1D, $08, $6A, $FE, $19, $7B, $EE, $50, $20, $FC, $23, $1D, $EF, $15, $9E, $72, $82, $74, $63, $49, $34, $7B, $41, $F0, $CF, $25, $9B, $8B, $F9, $63, $7B, $06, $E4, $D8, $48, $0B, $2E, $66, $48, $0F ), {09}
    ( $6E, $DF, $03, $C9, $6B, $28, $BD, $BD, $70, $85, $18, $0E, $11, $1B, $C3, $83, $10, $BE, $1E, $67, $E4, $10, $64, $DF, $DE, $5B, $8B, $87, $87, $20, $F3, $01, $B8, $44, $92, $C4, $89, $04, $A3, $BD, $1F, $F6, $82, $3A, $79, $BF, $1A, $F1 ), {0A}
    ( $DF, $79, $4A, $CE, $99, $7D, $BE, $C2, $56, $D1, $0B, $E4, $2A, $A8, $A9, $BB, $48, $1A, $7D, $28, $FD, $6D, $C8, $A7, $7C, $DB, $BA, $58, $25, $A7, $6D, $CC, $39, $71, $7C, $04, $17, $C6, $65, $73, $44, $4E, $07, $6F, $C0, $08, $F2, $CE ), {0B}
    ( $44, $71, $DE, $71, $BE, $C1, $E9, $33, $29, $63, $7B, $04, $64, $F5, $CC, $88, $57, $93, $A8, $69, $D8, $25, $44, $F2, $07, $5F, $BC, $A2, $AF, $6D, $51, $18, $5C, $D3, $5E, $66, $40, $36, $CF, $31, $4C, $A6, $82, $B2, $3D, $7E, $CF, $27 ), {0C}
    ( $CE, $A7, $CC, $D9, $8F, $D8, $84, $C6, $83, $0D, $63, $9B, $FE, $52, $28, $51, $F7, $8D, $2C, $AD, $A5, $C9, $ED, $39, $7D, $51, $35, $33, $A7, $B5, $A3, $E4, $4A, $A6, $02, $D1, $B0, $69, $D9, $BD, $F7, $8C, $2C, $C8, $4B, $2C, $18, $6A ), {0D}
    ( $CD, $27, $BC, $04, $A6, $24, $E3, $FC, $3C, $A7, $2A, $48, $DE, $DD, $B5, $A2, $EF, $9A, $C0, $85, $EE, $EC, $5E, $E6, $56, $18, $D5, $84, $11, $2C, $2E, $26, $16, $2C, $72, $75, $18, $6A, $69, $99, $08, $FF, $BD, $36, $AE, $9A, $DD, $3E ), {0E}
    ( $91, $93, $4D, $94, $46, $AD, $59, $43, $A6, $1D, $EC, $17, $24, $5F, $49, $7C, $5F, $F4, $EB, $9A, $61, $27, $DC, $2C, $17, $2B, $97, $F6, $C9, $DA, $3F, $CC, $DA, $92, $C6, $28, $77, $DA, $CE, $16, $49, $B9, $D9, $8E, $A6, $66, $15, $16 ), {0F}
    ( $3A, $A3, $85, $E9, $72, $A4, $32, $8E, $EC, $4C, $1E, $49, $61, $BD, $31, $BA, $D7, $6C, $91, $98, $48, $34, $52, $1B, $2A, $0D, $01, $C9, $3A, $95, $B2, $A2, $06, $C8, $EF, $E3, $CD, $BF, $CC, $47, $62, $12, $B5, $51, $41, $F1, $4D, $AE ), {10}
    ( $37, $A9, $07, $75, $7B, $CA, $BE, $76, $0A, $C2, $2C, $C2, $8F, $BB, $75, $93, $73, $F2, $A1, $03, $5C, $5C, $EA, $AD, $2E, $60, $D6, $75, $F5, $F3, $13, $A1, $AD, $84, $05, $17, $7F, $EE, $C3, $FF, $7F, $9A, $5F, $8C, $DC, $33, $61, $91 ), {11}
    ( $CA, $75, $89, $A1, $86, $F5, $8C, $D9, $C0, $BE, $87, $C4, $31, $BF, $17, $9C, $02, $DF, $FB, $3C, $F9, $17, $BA, $A0, $2E, $A2, $79, $3C, $0A, $81, $2A, $2C, $43, $25, $A8, $19, $61, $92, $21, $E8, $2E, $53, $A6, $AC, $2F, $07, $BD, $7C ), {12}
    ( $E3, $D6, $0D, $37, $FF, $CD, $A4, $4A, $24, $43, $7C, $6B, $2A, $F4, $09, $96, $6F, $41, $3C, $98, $5A, $64, $C0, $B8, $EE, $87, $13, $7E, $95, $93, $8B, $A9, $9D, $C7, $E1, $28, $D8, $BB, $34, $6E, $13, $0B, $CC, $8D, $26, $71, $CD, $1A ), {13}
    ( $08, $23, $29, $FC, $E0, $09, $DC, $3C, $B0, $BB, $AD, $4D, $36, $26, $D3, $B2, $08, $1A, $5B, $D2, $F7, $A4, $09, $74, $65, $AF, $79, $0F, $75, $CB, $E3, $96, $45, $37, $DD, $6D, $6B, $E3, $05, $1A, $77, $E5, $AC, $C8, $58, $5A, $F7, $FA ), {14}
    ( $2F, $DC, $BB, $5F, $24, $39, $8B, $09, $5F, $21, $DF, $BF, $EB, $52, $C2, $A6, $30, $F9, $7D, $FD, $1F, $57, $5E, $42, $F6, $89, $06, $81, $09, $CD, $91, $33, $E7, $BE, $F4, $4D, $97, $63, $A7, $F9, $CB, $52, $45, $47, $C1, $BC, $73, $02 ), {15}
    ( $A0, $F3, $7E, $20, $95, $D7, $5C, $BC, $16, $C3, $84, $5D, $D3, $81, $71, $69, $91, $D1, $33, $5A, $89, $C5, $D9, $E5, $FE, $1E, $14, $A8, $CD, $DA, $4B, $B9, $B8, $57, $F4, $1E, $79, $C7, $A9, $9C, $36, $1C, $6F, $C9, $6B, $5B, $CE, $3C ), {16}
    ( $6C, $83, $D1, $51, $D1, $7B, $1B, $02, $FA, $3D, $4A, $31, $5E, $CB, $DE, $CB, $FF, $04, $56, $5C, $1F, $64, $1C, $A1, $45, $CA, $69, $FF, $4D, $56, $2C, $94, $ED, $7B, $A3, $96, $21, $85, $C1, $AF, $4F, $5C, $36, $E6, $AE, $47, $29, $D9 ), {17}
    ( $51, $C2, $8F, $5C, $96, $15, $D0, $8F, $77, $90, $34, $CD, $6F, $7B, $EE, $CE, $8B, $1E, $CB, $66, $F0, $D8, $9D, $7A, $28, $31, $9C, $EE, $D0, $8A, $0C, $C2, $D6, $F7, $DC, $F7, $52, $22, $7C, $30, $21, $84, $47, $D7, $79, $6A, $44, $48 ), {18}
    ( $DD, $B8, $1F, $DE, $2B, $64, $1D, $EC, $4C, $48, $16, $8D, $F0, $AB, $16, $B1, $84, $74, $14, $50, $0D, $6F, $E6, $6C, $54, $D0, $19, $24, $CF, $7E, $E1, $28, $91, $DA, $5A, $0C, $0E, $FE, $83, $84, $E3, $C0, $4E, $AE, $43, $CA, $57, $7E ), {19}
    ( $EE, $8A, $6D, $2D, $D0, $48, $A0, $E2, $C3, $0E, $3D, $ED, $C2, $AE, $B8, $F5, $75, $95, $C4, $AB, $28, $E4, $9C, $1C, $02, $35, $7D, $82, $0E, $C9, $E1, $28, $3E, $58, $35, $3D, $D7, $F7, $A1, $F1, $BB, $1D, $D6, $E2, $B5, $88, $54, $C5 ), {1A}
    ( $83, $1B, $EF, $46, $0B, $26, $D6, $FC, $64, $01, $AA, $8A, $EF, $F2, $F5, $53, $D8, $44, $9E, $44, $C5, $82, $9B, $1B, $39, $90, $FE, $E0, $44, $5F, $B8, $4C, $E7, $20, $FF, $FD, $EC, $B9, $45, $D6, $46, $02, $A7, $47, $46, $A0, $64, $BA ), {1B}
    ( $72, $35, $DC, $FF, $05, $CC, $D8, $66, $7D, $D9, $47, $60, $20, $25, $8B, $11, $9F, $CD, $C6, $9F, $5A, $CC, $E7, $A1, $A0, $3F, $3D, $BD, $CC, $06, $FE, $57, $09, $15, $83, $D2, $C4, $A5, $3A, $37, $FE, $F7, $52, $07, $B6, $D3, $65, $38 ), {1C}
    ( $37, $D3, $53, $0D, $A9, $B3, $F8, $9B, $93, $97, $11, $A9, $8D, $2C, $9C, $5C, $B0, $51, $11, $E4, $FE, $1A, $C9, $42, $AD, $B4, $AE, $88, $6F, $EF, $44, $4D, $B8, $2C, $FC, $38, $85, $8E, $39, $C5, $C1, $41, $49, $B3, $64, $92, $00, $5A ), {1D}
    ( $0B, $78, $03, $70, $0E, $1D, $1C, $4B, $5A, $96, $64, $F1, $CD, $3E, $C4, $6B, $C4, $8D, $98, $C6, $36, $15, $60, $E7, $72, $91, $BB, $BA, $62, $33, $14, $00, $FB, $AB, $BC, $88, $E4, $29, $CD, $A2, $4C, $B0, $59, $32, $B4, $B7, $36, $E2 ), {1E}
    ( $89, $A6, $B9, $76, $68, $50, $68, $D6, $21, $78, $BD, $10, $AF, $55, $0F, $DD, $59, $67, $A4, $D0, $3E, $BC, $35, $39, $25, $DF, $82, $BD, $D7, $89, $86, $CC, $56, $CE, $26, $8E, $50, $49, $26, $58, $3C, $6E, $07, $8A, $E5, $84, $D8, $B1 ), {1F}
    ( $5B, $7C, $58, $A4, $9B, $5B, $98, $85, $9E, $10, $21, $FC, $9A, $0C, $C1, $09, $F0, $2A, $5E, $A7, $53, $53, $47, $22, $19, $D9, $A3, $82, $5D, $85, $79, $08, $8B, $8E, $6C, $CF, $2C, $04, $AA, $E4, $0C, $57, $1C, $C3, $1B, $28, $D9, $78 ), {20}
    ( $17, $92, $4D, $C7, $C2, $90, $D9, $A9, $59, $16, $FB, $2B, $4F, $FA, $5F, $E2, $7B, $D3, $E1, $76, $4D, $C3, $C4, $AE, $18, $ED, $13, $31, $66, $89, $AC, $28, $EC, $17, $7B, $0E, $BC, $93, $AE, $94, $0F, $4D, $47, $B8, $BD, $DF, $11, $DB ), {21}
    ( $A3, $59, $3E, $48, $CE, $7A, $E5, $7C, $69, $EC, $5B, $6D, $E6, $2F, $58, $A6, $01, $2F, $21, $41, $46, $16, $0E, $14, $E2, $DA, $13, $C0, $9F, $B1, $AC, $60, $E2, $6E, $51, $C1, $B0, $1C, $90, $C3, $A8, $DB, $0B, $3B, $63, $70, $C2, $02 ), {22}
    ( $DA, $19, $58, $D7, $E8, $94, $43, $B0, $CC, $88, $B3, $23, $E3, $8E, $D2, $D4, $3B, $59, $C4, $68, $31, $32, $B8, $D0, $C4, $15, $64, $4A, $A6, $E2, $BE, $C4, $41, $53, $9A, $19, $24, $FD, $64, $AF, $59, $F4, $D5, $9D, $51, $A7, $E7, $6E ), {23}
    ( $DB, $01, $AE, $F7, $5F, $8C, $2F, $86, $F2, $BD, $25, $00, $42, $C8, $39, $A2, $92, $4F, $49, $3E, $82, $9C, $67, $CA, $00, $A1, $E1, $D8, $B4, $3B, $C9, $2B, $26, $C3, $BD, $F8, $BE, $66, $59, $B4, $DA, $5C, $9A, $2C, $BA, $AF, $02, $B2 ), {24}
    ( $AA, $47, $D9, $96, $D5, $32, $A4, $DD, $23, $9E, $77, $85, $30, $A7, $39, $4F, $F9, $DA, $62, $25, $D4, $5B, $89, $88, $C1, $22, $9E, $DB, $41, $AA, $97, $8D, $EE, $CA, $20, $DA, $92, $85, $65, $C4, $3C, $28, $55, $66, $88, $5E, $0B, $8A ), {25}
    ( $5F, $F6, $2F, $C9, $8E, $C7, $BD, $B9, $08, $B5, $D7, $1A, $A4, $55, $43, $7D, $97, $0F, $D0, $B4, $30, $2E, $BB, $69, $32, $E5, $05, $8C, $7B, $91, $25, $F1, $F7, $F4, $FD, $00, $CF, $5B, $98, $E5, $F4, $3B, $77, $3A, $9F, $31, $80, $85 ), {26}
    ( $64, $01, $C4, $C9, $7C, $C0, $1B, $F8, $10, $2E, $B7, $99, $07, $D1, $4C, $B4, $F1, $87, $37, $80, $26, $B5, $31, $CE, $7B, $A5, $C0, $96, $98, $7C, $1E, $81, $1E, $94, $AA, $13, $32, $C8, $3B, $CC, $47, $6C, $A7, $FA, $FA, $33, $63, $8F ), {27}
    ( $DB, $2E, $66, $45, $79, $A4, $83, $17, $C0, $B5, $47, $B9, $88, $75, $B5, $2E, $29, $C1, $B2, $0D, $3B, $D0, $07, $08, $9A, $63, $C0, $CA, $17, $D0, $CC, $92, $1A, $52, $98, $1B, $48, $F7, $EA, $B8, $18, $16, $68, $31, $0B, $A7, $C3, $8B ), {28}
    ( $AA, $31, $F2, $FB, $78, $F1, $8B, $D0, $07, $C0, $43, $56, $D2, $87, $DA, $BF, $B8, $F2, $B6, $81, $0C, $06, $2B, $CE, $44, $27, $40, $BA, $48, $C7, $3A, $8A, $67, $D0, $A7, $1F, $5E, $DA, $D7, $10, $44, $5D, $89, $44, $52, $35, $08, $DF ), {29}
    ( $B1, $1C, $CF, $29, $FF, $AF, $D1, $64, $28, $73, $12, $7F, $BB, $36, $7D, $81, $1C, $65, $E3, $94, $50, $22, $6E, $1E, $BB, $D4, $B0, $58, $15, $F8, $0B, $4E, $EB, $7E, $CC, $40, $29, $14, $C5, $64, $AB, $4C, $AC, $7C, $F3, $2B, $0E, $E2 ), {2A}
    ( $5F, $66, $69, $64, $56, $3F, $2A, $98, $CF, $77, $22, $F7, $DA, $FF, $31, $B8, $56, $34, $B9, $DB, $32, $D5, $EB, $4D, $FD, $1D, $EA, $51, $43, $85, $D7, $DF, $53, $9A, $E0, $A6, $DF, $60, $BF, $92, $82, $88, $17, $B8, $97, $29, $6E, $AD ), {2B}
    ( $3B, $9C, $A9, $90, $B5, $CA, $B9, $DC, $36, $F3, $26, $E7, $87, $EB, $DE, $20, $5A, $C2, $FA, $D0, $26, $16, $DD, $BF, $87, $E3, $4C, $98, $F9, $F8, $AC, $EF, $6C, $EF, $A1, $B7, $2C, $5D, $8E, $FB, $55, $11, $67, $A6, $F2, $B8, $59, $AC ), {2C}
    ( $BC, $10, $50, $30, $60, $73, $EF, $9E, $57, $97, $D1, $3B, $84, $AD, $A3, $2E, $ED, $3D, $C3, $00, $53, $13, $9A, $4F, $D0, $57, $97, $8F, $F3, $A3, $FA, $2B, $A0, $BF, $06, $18, $86, $EC, $42, $95, $5C, $04, $D5, $CF, $87, $4D, $F6, $34 ), {2D}
    ( $C9, $54, $7B, $8E, $3B, $47, $1C, $2A, $A7, $84, $98, $B9, $07, $1F, $8A, $77, $C7, $34, $E0, $AF, $84, $E0, $12, $A0, $DD, $72, $B3, $B4, $FE, $F6, $9A, $1B, $B0, $B2, $2B, $55, $53, $0C, $DD, $0D, $F8, $56, $45, $54, $19, $89, $9A, $17 ), {2E}
    ( $1A, $D1, $8E, $44, $E1, $94, $28, $98, $BF, $F8, $9C, $86, $65, $C7, $CA, $EA, $6C, $CC, $A8, $6A, $63, $7E, $A3, $3A, $BA, $61, $80, $5A, $E7, $54, $8B, $A9, $CD, $5B, $CC, $56, $7E, $8D, $E1, $58, $04, $84, $23, $34, $71, $20, $FD, $6B ), {2F}
    ( $00, $4C, $1F, $81, $64, $E1, $24, $BD, $BA, $8A, $3C, $0D, $30, $89, $A5, $0F, $E8, $7A, $D0, $2A, $3F, $2A, $34, $E6, $E0, $8E, $9A, $73, $02, $C4, $A6, $B0, $B7, $3F, $51, $BD, $D6, $7B, $66, $9A, $F3, $DB, $A3, $E3, $9C, $C0, $2A, $5A ), {30}
    ( $8C, $97, $3B, $34, $E2, $0A, $A7, $70, $65, $8B, $0D, $12, $A6, $84, $E6, $09, $CA, $EF, $37, $63, $8F, $66, $BC, $4B, $01, $99, $0E, $4D, $2D, $31, $CB, $9D, $A7, $4A, $5D, $FD, $67, $87, $5A, $DF, $17, $FB, $2A, $C6, $5C, $38, $72, $C2 ), {31}
    ( $A6, $8B, $AC, $C2, $ED, $63, $72, $1D, $3F, $BF, $6C, $C9, $C3, $65, $2A, $A6, $7C, $27, $39, $93, $FB, $41, $1C, $EB, $DA, $C2, $03, $AE, $77, $AD, $EE, $3E, $0A, $EF, $EC, $6C, $60, $81, $EE, $19, $77, $DC, $BF, $D3, $4A, $36, $30, $23 ), {32}
    ( $47, $24, $46, $5B, $92, $9E, $77, $6B, $06, $0A, $B6, $ED, $97, $C9, $A3, $FF, $5F, $E0, $66, $A1, $E7, $63, $86, $7D, $A8, $F0, $E4, $78, $FE, $90, $91, $FA, $0B, $B2, $BA, $B1, $74, $70, $28, $D6, $8E, $32, $E0, $AA, $C7, $CC, $D4, $F2 ), {33}
    ( $A8, $14, $F2, $A5, $C6, $48, $C4, $38, $90, $74, $17, $9A, $48, $01, $85, $D4, $A1, $54, $E9, $70, $9E, $92, $DD, $17, $E4, $CC, $32, $39, $30, $6F, $88, $C5, $38, $C0, $F5, $93, $DF, $49, $25, $BB, $77, $ED, $81, $2D, $5C, $21, $7D, $B1 ), {34}
    ( $92, $2D, $EB, $1F, $56, $B1, $67, $CE, $A0, $47, $0A, $39, $5D, $A5, $C9, $72, $8B, $88, $BA, $DB, $75, $BC, $41, $05, $BA, $AA, $8C, $5E, $E4, $4F, $6B, $70, $57, $58, $EC, $7E, $BC, $A0, $16, $E5, $AF, $CB, $78, $80, $59, $57, $FE, $92 ), {35}
    ( $D1, $05, $35, $B0, $F5, $EA, $45, $77, $91, $2C, $D9, $1E, $25, $23, $6E, $75, $12, $B3, $20, $01, $B5, $D4, $50, $95, $BE, $DF, $56, $D0, $C0, $81, $2A, $F1, $17, $94, $5F, $DB, $15, $7A, $4C, $B6, $B6, $C2, $BF, $5A, $80, $3B, $60, $27 ), {36}
    ( $F8, $1B, $82, $C6, $29, $2A, $68, $DC, $72, $0C, $48, $0B, $71, $60, $E8, $E0, $8A, $BB, $77, $97, $94, $2F, $07, $BB, $E2, $EB, $9B, $45, $76, $B7, $58, $8C, $29, $48, $43, $D2, $68, $FF, $E5, $0F, $B4, $B5, $BB, $E5, $34, $1B, $B8, $78 ), {37}
    ( $89, $F8, $13, $47, $F0, $F8, $96, $08, $46, $31, $2C, $49, $79, $57, $86, $79, $7C, $7E, $CA, $A3, $F7, $52, $3C, $B8, $74, $F1, $B0, $38, $20, $FA, $48, $D4, $CD, $A2, $96, $3E, $FF, $95, $4C, $D3, $8D, $AE, $7A, $8D, $76, $10, $04, $97 ), {38}
    ( $E2, $03, $EB, $8F, $4E, $C5, $57, $6D, $3B, $95, $D9, $B8, $C8, $5C, $4D, $09, $D0, $B8, $EE, $C2, $8A, $8A, $B8, $11, $79, $04, $F3, $F8, $35, $6C, $73, $C6, $E1, $59, $81, $E1, $F9, $7F, $6F, $38, $79, $F3, $C8, $FD, $B7, $2D, $EB, $B2 ), {39}
    ( $78, $1F, $99, $2A, $E3, $7F, $C8, $4F, $7F, $18, $6E, $7B, $8F, $2E, $B1, $F3, $D0, $70, $CE, $88, $A8, $3C, $11, $42, $FA, $83, $F1, $F0, $96, $51, $F4, $C6, $5D, $53, $46, $E2, $D6, $34, $6C, $D6, $0B, $F8, $13, $E5, $03, $47, $41, $3D ), {3A}
    ( $4E, $76, $26, $F4, $24, $E5, $E2, $9A, $1C, $3E, $D3, $17, $B6, $AE, $C8, $23, $4C, $8D, $93, $DB, $60, $DC, $91, $D1, $18, $DA, $6D, $28, $01, $F8, $E9, $46, $B5, $02, $62, $32, $89, $A6, $DB, $42, $24, $19, $C8, $E5, $B0, $50, $53, $04 ), {3B}
    ( $91, $2C, $C9, $D8, $70, $57, $28, $61, $12, $C6, $96, $B0, $EF, $5E, $21, $E1, $42, $2D, $F5, $10, $A4, $8C, $C1, $94, $C1, $60, $DF, $F8, $D5, $B5, $84, $68, $55, $E0, $0D, $CB, $27, $BA, $BF, $34, $45, $B2, $1F, $ED, $F9, $AB, $C6, $7D ), {3C}
    ( $35, $2D, $14, $C4, $65, $AC, $24, $19, $9D, $BB, $98, $5A, $D7, $75, $99, $D7, $4E, $8B, $48, $44, $FE, $36, $EC, $85, $C9, $F0, $7F, $5C, $4A, $0A, $92, $FC, $39, $16, $90, $BC, $90, $D0, $44, $D1, $0C, $EC, $EC, $B6, $86, $AA, $CC, $A1 ), {3D}
    ( $65, $43, $70, $AE, $F2, $B4, $47, $DC, $B3, $AC, $64, $E3, $0B, $21, $F4, $8B, $61, $03, $4F, $29, $CE, $E7, $92, $47, $E2, $11, $04, $50, $B0, $48, $D9, $0D, $64, $5D, $3E, $A6, $D5, $76, $B7, $2A, $A0, $EE, $AC, $1C, $80, $96, $40, $AF ), {3E}
    ( $81, $0E, $C1, $83, $19, $93, $6D, $05, $95, $A9, $83, $C4, $41, $4F, $0C, $57, $9D, $4B, $DC, $2F, $F1, $6F, $B7, $84, $CC, $5D, $FC, $57, $3F, $50, $0F, $BA, $E6, $C3, $F1, $B6, $63, $59, $B2, $A5, $F3, $76, $60, $43, $D7, $37, $D8, $BF ), {3F}
    ( $3D, $8B, $4D, $09, $A8, $6D, $79, $FD, $0E, $37, $1B, $E9, $70, $F6, $56, $4A, $7F, $F8, $F3, $08, $9C, $BD, $EE, $8E, $43, $D8, $D5, $65, $BB, $03, $EF, $44, $9F, $14, $F6, $55, $0D, $F6, $2F, $07, $BA, $F3, $D7, $4D, $25, $3C, $8B, $5F ), {40}
    ( $98, $5A, $59, $6F, $99, $D8, $8A, $5A, $45, $21, $37, $9F, $01, $9B, $AE, $B2, $CC, $AB, $8F, $2D, $C3, $19, $82, $13, $98, $0D, $B0, $CB, $DD, $6B, $FD, $31, $C0, $1E, $FC, $60, $4E, $F6, $EF, $26, $67, $4E, $38, $60, $4C, $CB, $AD, $13 ), {41}
    ( $A4, $F4, $C3, $C6, $1A, $30, $CF, $61, $C1, $A2, $80, $C7, $01, $44, $CF, $45, $BF, $DC, $A6, $1A, $B3, $8A, $6E, $71, $4A, $14, $1D, $DE, $E3, $7F, $51, $20, $1F, $4E, $EA, $47, $37, $0A, $F0, $57, $E8, $23, $CD, $CB, $67, $B5, $D5, $38 ), {42}
    ( $FE, $68, $07, $CE, $B3, $C2, $79, $45, $23, $65, $9D, $CF, $79, $57, $D9, $C6, $3A, $22, $0B, $30, $70, $46, $87, $B2, $08, $84, $DA, $D4, $EE, $53, $1D, $DA, $EA, $BA, $EF, $AE, $55, $2A, $D0, $19, $EE, $9D, $D2, $63, $EB, $57, $FE, $8E ), {43}
    ( $E8, $6F, $5A, $70, $87, $2B, $43, $3E, $63, $72, $19, $11, $EF, $4C, $F4, $03, $F4, $9F, $CA, $CB, $1B, $DC, $FA, $21, $A9, $3D, $24, $54, $CC, $E4, $8C, $20, $D0, $EB, $39, $F6, $68, $44, $77, $06, $92, $9F, $3F, $C9, $06, $7F, $85, $95 ), {44}
    ( $A1, $63, $8D, $C3, $D5, $DA, $72, $F3, $42, $0A, $AD, $9D, $21, $1F, $E2, $BF, $38, $AF, $FB, $41, $FE, $81, $CF, $7B, $AB, $B3, $24, $BA, $34, $67, $15, $0B, $50, $48, $F1, $81, $8A, $B7, $74, $FE, $F7, $0B, $79, $B9, $19, $46, $1C, $42 ), {45}
    ( $13, $FB, $28, $56, $EC, $6A, $D2, $13, $E2, $49, $3E, $5D, $85, $11, $53, $0C, $25, $D2, $6A, $A5, $E4, $0D, $AB, $BE, $B6, $45, $55, $E1, $A7, $38, $63, $42, $0A, $1B, $65, $43, $B3, $14, $FF, $AE, $F4, $7B, $3D, $5B, $EB, $F0, $B7, $B6 ), {46}
    ( $04, $0F, $2C, $DB, $C9, $D0, $3F, $02, $7E, $6B, $37, $F3, $DC, $F4, $64, $9A, $B5, $82, $3B, $FA, $97, $42, $E0, $F2, $F0, $EE, $26, $A7, $A7, $FD, $9E, $BA, $50, $4F, $95, $3F, $CE, $39, $94, $15, $D0, $59, $D5, $0A, $3C, $DB, $C1, $12 ), {47}
    ( $2F, $01, $7C, $A0, $84, $DE, $DC, $C9, $0A, $8B, $06, $3A, $36, $2E, $FF, $69, $79, $F6, $8C, $FA, $4D, $8B, $2D, $DC, $62, $D3, $E4, $48, $79, $6C, $49, $F3, $A3, $5B, $C1, $04, $09, $21, $83, $F0, $6A, $FF, $76, $E3, $38, $E4, $E8, $05 ), {48}
    ( $2C, $A8, $56, $DC, $21, $5B, $02, $AD, $83, $78, $0E, $15, $CD, $20, $89, $8E, $7A, $48, $D6, $78, $93, $B3, $F3, $89, $44, $03, $7B, $8A, $54, $97, $73, $B6, $C8, $4B, $C0, $96, $2C, $83, $46, $A3, $2F, $5C, $C5, $D7, $E9, $9A, $45, $A2 ), {49}
    ( $08, $C9, $92, $3A, $47, $1E, $76, $C4, $A2, $25, $04, $F9, $FD, $2F, $37, $FA, $31, $50, $64, $7A, $A3, $43, $E8, $72, $22, $37, $CB, $04, $74, $35, $0A, $49, $B2, $B8, $66, $3D, $15, $CF, $3A, $B7, $0E, $36, $0F, $66, $66, $C1, $5B, $5B ), {4A}
    ( $C3, $DA, $83, $8E, $33, $45, $D5, $7D, $F0, $F0, $40, $67, $56, $85, $E1, $60, $EA, $CE, $75, $9E, $C5, $B5, $E7, $A6, $B5, $CE, $59, $EB, $B0, $C0, $AE, $E5, $A9, $1D, $1A, $36, $3B, $2A, $D4, $C3, $C5, $F1, $21, $65, $48, $A7, $06, $E7 ), {4B}
    ( $D4, $70, $E0, $B0, $D5, $42, $81, $43, $0C, $C5, $CD, $F0, $A4, $C1, $DD, $AF, $4A, $B1, $08, $A2, $96, $B5, $7B, $AB, $2E, $04, $44, $CD, $62, $3A, $F1, $39, $6A, $85, $9F, $71, $10, $9D, $B7, $2A, $26, $BA, $B6, $E2, $02, $1B, $00, $F8 ), {4C}
    ( $51, $C2, $68, $D7, $B1, $3F, $CA, $69, $9D, $CD, $72, $55, $B9, $72, $25, $DC, $07, $0F, $6A, $84, $FA, $F8, $F5, $60, $19, $BB, $6D, $3E, $8F, $07, $38, $DA, $EE, $BA, $73, $44, $8C, $B1, $55, $21, $50, $2A, $0C, $85, $CD, $DA, $8B, $18 ), {4D}
    ( $57, $28, $81, $BB, $3C, $E9, $05, $18, $96, $9E, $8E, $91, $08, $41, $43, $69, $4D, $1C, $D9, $D1, $94, $08, $F5, $CC, $4A, $83, $1F, $29, $53, $FF, $B1, $12, $10, $5A, $DB, $48, $3D, $53, $F6, $B7, $F5, $7C, $3D, $93, $A5, $79, $D3, $38 ), {4E}
    ( $EA, $4B, $0D, $A7, $7E, $E7, $D4, $2A, $30, $20, $CA, $FB, $D8, $98, $29, $F4, $67, $28, $CF, $B2, $6C, $DA, $BD, $4E, $72, $5D, $B7, $D4, $D3, $E1, $78, $1E, $22, $28, $9A, $26, $64, $9D, $00, $87, $1C, $EB, $99, $72, $56, $CA, $D7, $D9 ), {4F}
    ( $23, $7B, $F7, $11, $80, $94, $EA, $7B, $10, $9C, $4A, $0E, $1B, $FB, $E0, $27, $E7, $23, $C1, $0B, $32, $CE, $B2, $E9, $33, $4B, $FF, $72, $46, $50, $E9, $49, $13, $F0, $26, $9C, $74, $34, $7C, $60, $D1, $E3, $60, $44, $95, $3C, $EC, $E8 ), {50}
    ( $2F, $3C, $70, $F7, $B5, $CE, $96, $E6, $5B, $16, $E3, $6A, $37, $48, $42, $A1, $89, $41, $86, $EA, $42, $2B, $3B, $EE, $6D, $3A, $04, $5F, $FC, $2F, $60, $8F, $AA, $5E, $05, $5E, $A0, $C3, $B1, $C7, $B4, $C6, $B1, $6A, $79, $FD, $87, $D8 ), {51}
    ( $F6, $EC, $7A, $3A, $BD, $01, $A2, $41, $58, $19, $68, $8B, $1C, $37, $C6, $A4, $75, $A0, $50, $BE, $62, $84, $65, $21, $EE, $34, $40, $A1, $29, $06, $54, $E9, $8C, $5E, $8B, $D6, $32, $E5, $37, $A1, $40, $4C, $AE, $F0, $5D, $10, $01, $C6 ), {52}
    ( $6A, $70, $DF, $AE, $C8, $A7, $01, $DE, $71, $1F, $AF, $25, $75, $FC, $7F, $22, $1A, $48, $6B, $45, $BD, $3B, $92, $9D, $A1, $C3, $26, $B2, $38, $FE, $1B, $3A, $88, $09, $72, $D4, $86, $3F, $E1, $40, $39, $77, $25, $21, $BC, $13, $EA, $47 ), {53}
    ( $66, $76, $16, $1A, $8B, $12, $3B, $9F, $5F, $91, $DB, $FB, $32, $50, $87, $24, $83, $53, $69, $9E, $67, $05, $35, $AF, $C2, $C0, $45, $07, $DD, $21, $B1, $AE, $A5, $4F, $FC, $0B, $DA, $E2, $A6, $4E, $61, $06, $D1, $E1, $29, $6B, $1A, $E0 ), {54}
    ( $AD, $AA, $A8, $B7, $2A, $85, $33, $B6, $C6, $4E, $86, $20, $65, $9E, $67, $45, $74, $AC, $18, $AC, $CF, $16, $6B, $F4, $97, $0B, $AA, $DD, $59, $4D, $62, $59, $90, $69, $A7, $1B, $6A, $81, $17, $F4, $6B, $DF, $8C, $54, $3F, $AC, $C7, $54 ), {55}
    ( $60, $FD, $EF, $62, $D3, $99, $48, $67, $95, $D0, $8F, $0E, $8C, $17, $2D, $AB, $E2, $53, $CC, $D7, $60, $69, $D0, $F1, $2A, $ED, $AC, $A4, $76, $F9, $7F, $E7, $2C, $BF, $05, $53, $A6, $73, $AA, $62, $84, $14, $C9, $EA, $1D, $65, $68, $17 ), {56}
    ( $D1, $E1, $F8, $9F, $8B, $7F, $95, $CF, $0E, $03, $8C, $2D, $4A, $D2, $F8, $D3, $79, $16, $94, $AD, $B4, $35, $CB, $0D, $35, $B6, $A9, $DC, $67, $EB, $27, $95, $32, $FA, $5D, $18, $DF, $1A, $42, $4B, $24, $B9, $79, $0B, $96, $5D, $40, $F5 ), {57}
    ( $6F, $AF, $85, $2D, $17, $30, $03, $CD, $3C, $1B, $26, $00, $D9, $59, $95, $42, $A5, $97, $2C, $93, $CA, $DD, $A0, $F4, $DD, $91, $42, $8C, $17, $28, $71, $CE, $78, $DB, $F6, $9A, $5D, $4F, $75, $2B, $94, $B9, $8B, $A7, $0F, $2C, $A8, $06 ), {58}
    ( $B5, $02, $62, $77, $68, $15, $CD, $B2, $3D, $38, $9A, $3B, $7A, $7C, $C0, $D2, $9D, $AA, $36, $83, $FB, $70, $FC, $8C, $EC, $BD, $15, $56, $2E, $2D, $1C, $A8, $B9, $CA, $A2, $39, $01, $EE, $DE, $3B, $D9, $16, $89, $A3, $B5, $BD, $82, $23 ), {59}
    ( $7C, $19, $B6, $87, $50, $4C, $55, $E9, $A6, $4C, $80, $20, $78, $E7, $F2, $91, $46, $7F, $E9, $59, $4A, $40, $BF, $EB, $52, $0A, $C4, $F3, $91, $EC, $58, $A8, $10, $19, $71, $0F, $60, $54, $D1, $E5, $87, $35, $66, $E1, $6F, $E2, $8B, $6A ), {5A}
    ( $03, $60, $1F, $9F, $FE, $87, $1E, $B2, $7F, $47, $2A, $8C, $54, $C9, $AF, $6D, $EB, $63, $DE, $3B, $6D, $2F, $50, $EC, $91, $F1, $52, $56, $0A, $D3, $E4, $F4, $B5, $C4, $BD, $65, $6C, $9A, $A2, $55, $34, $DD, $69, $2D, $07, $36, $AE, $5D ), {5B}
    ( $E3, $E8, $C2, $8E, $6A, $3B, $23, $08, $F8, $44, $56, $55, $FB, $54, $F6, $97, $33, $E6, $3E, $B0, $6A, $53, $42, $EC, $EA, $1C, $0F, $B2, $84, $BE, $F9, $07, $03, $4C, $EE, $C8, $22, $37, $5D, $C1, $51, $12, $C2, $6C, $ED, $99, $AA, $5F ), {5C}
    ( $FC, $ED, $5B, $56, $FA, $FB, $13, $18, $4D, $24, $7C, $FA, $E5, $16, $85, $1E, $5D, $7D, $4E, $30, $CB, $7C, $45, $F4, $96, $3C, $95, $B4, $89, $39, $CF, $4F, $74, $E7, $1E, $06, $10, $58, $FD, $85, $03, $64, $C1, $B7, $7F, $BE, $0A, $C6 ), {5D}
    ( $B7, $B3, $B7, $E5, $40, $8D, $DB, $17, $8F, $18, $57, $9A, $0E, $D2, $F4, $43, $11, $BD, $3C, $53, $AB, $A9, $BB, $02, $B7, $B3, $8C, $0A, $4E, $A1, $9B, $29, $73, $9C, $64, $A1, $FB, $77, $FA, $1E, $E3, $9B, $E9, $15, $F0, $DB, $EB, $AE ), {5E}
    ( $41, $D7, $38, $2B, $73, $FD, $83, $9A, $D7, $25, $FE, $86, $90, $27, $AA, $F4, $BA, $E4, $A9, $FE, $9F, $01, $3E, $96, $E1, $6C, $25, $BA, $42, $61, $0C, $EE, $35, $63, $84, $42, $27, $41, $1B, $15, $87, $C0, $B3, $C6, $04, $07, $EA, $F9 ), {5F}
    ( $FE, $C6, $FA, $BC, $DC, $F4, $4E, $9A, $44, $0E, $85, $21, $E6, $6D, $05, $E1, $6F, $A3, $5B, $6D, $CB, $68, $CC, $BF, $C5, $5D, $9D, $62, $CD, $95, $64, $CF, $39, $98, $EA, $E1, $57, $2C, $98, $DB, $DD, $D9, $4D, $28, $83, $53, $26, $E6 ), {60}
    ( $1B, $46, $37, $D5, $03, $E7, $0B, $4E, $E5, $06, $6A, $BE, $4B, $49, $94, $34, $74, $05, $26, $6E, $7A, $30, $9C, $5A, $53, $B4, $D9, $E4, $7A, $42, $28, $C0, $36, $E0, $C1, $AF, $C9, $F6, $A4, $3D, $25, $00, $BB, $A7, $CC, $E6, $33, $3B ), {61}
    ( $12, $C4, $26, $ED, $C3, $3C, $20, $FC, $18, $55, $16, $E3, $FF, $88, $C8, $1A, $BB, $0A, $EC, $4A, $8D, $E3, $92, $3C, $63, $9C, $52, $EF, $60, $30, $40, $51, $A3, $9F, $A1, $F0, $37, $76, $7C, $4E, $15, $87, $6A, $C6, $38, $F9, $86, $78 ), {62}
    ( $74, $7F, $6F, $4C, $B2, $79, $33, $B6, $43, $5E, $60, $82, $1A, $E0, $20, $78, $9D, $A2, $34, $FF, $0D, $4A, $19, $1F, $A2, $46, $66, $41, $A3, $48, $47, $05, $FC, $C7, $20, $2C, $36, $92, $96, $3A, $57, $0B, $7E, $79, $2A, $35, $26, $D0 ), {63}
    ( $B4, $A5, $6F, $6B, $88, $06, $C7, $3B, $F3, $60, $15, $C3, $D0, $62, $38, $F4, $E8, $9B, $6C, $48, $CC, $02, $33, $A5, $87, $66, $78, $16, $88, $76, $BA, $B4, $2D, $63, $3B, $EE, $A7, $D4, $22, $CC, $58, $14, $80, $2C, $6C, $CA, $78, $F0 ), {64}
    ( $9E, $7C, $76, $1F, $7B, $95, $88, $97, $CE, $73, $97, $CB, $4E, $86, $1D, $EB, $2F, $D4, $10, $74, $81, $6C, $3D, $68, $DD, $74, $65, $6C, $5C, $F7, $FB, $96, $F7, $2D, $5E, $BF, $FA, $E3, $44, $C5, $F9, $0C, $29, $7E, $87, $BF, $B7, $06 ), {65}
    ( $8F, $5C, $C8, $1D, $07, $82, $AD, $68, $E4, $F1, $BE, $AA, $2D, $B4, $61, $29, $63, $6A, $A0, $A3, $15, $A0, $EA, $D5, $6C, $E4, $7E, $75, $6A, $3E, $F7, $1B, $B4, $E1, $6A, $2D, $96, $40, $E0, $50, $7B, $2C, $19, $7E, $43, $07, $FB, $14 ), {66}
    ( $7C, $3A, $5D, $F9, $4F, $BD, $B1, $A5, $C0, $E6, $A9, $F0, $56, $18, $FA, $C1, $6D, $33, $D9, $72, $60, $09, $3E, $DC, $B8, $1C, $4F, $1C, $A5, $63, $53, $18, $F3, $C6, $F4, $F2, $4F, $C8, $44, $67, $43, $6D, $8B, $37, $65, $3E, $4F, $92 ), {67}
    ( $4F, $DB, $CC, $BF, $6F, $18, $2D, $C4, $13, $0E, $F5, $93, $52, $F9, $20, $EC, $67, $51, $54, $43, $C6, $2C, $3B, $E3, $30, $CD, $FC, $BC, $E6, $48, $D9, $79, $D8, $A8, $FC, $B2, $63, $BE, $A2, $9F, $D5, $C6, $ED, $CD, $EC, $9E, $3E, $57 ), {68}
    ( $9E, $38, $BD, $36, $2A, $33, $B0, $FC, $C9, $BA, $09, $8B, $6E, $74, $49, $1F, $6D, $81, $37, $37, $58, $BF, $47, $3C, $54, $E5, $26, $95, $D2, $F9, $C1, $26, $9B, $F6, $8A, $73, $82, $A8, $2B, $A2, $1A, $D1, $C0, $40, $E2, $BA, $AF, $1F ), {69}
    ( $8C, $76, $6B, $4A, $19, $04, $83, $02, $12, $1E, $25, $A1, $7F, $BB, $18, $63, $5F, $28, $A8, $5E, $88, $33, $D9, $48, $7E, $27, $6E, $36, $22, $CA, $CE, $4C, $04, $E9, $0F, $3D, $BC, $24, $86, $DB, $66, $A5, $F0, $CA, $51, $A1, $82, $7F ), {6A}
    ( $C7, $A5, $95, $89, $9B, $FD, $17, $43, $CC, $A6, $56, $AD, $02, $50, $D3, $60, $FD, $C8, $B0, $62, $C0, $A3, $74, $17, $E2, $9A, $C6, $54, $01, $9D, $2A, $EF, $AA, $9C, $D9, $69, $87, $DB, $1E, $C0, $EE, $BE, $D7, $14, $7D, $7E, $BD, $4A ), {6B}
    ( $E9, $1B, $51, $FF, $A0, $F2, $F2, $BF, $8B, $64, $AA, $96, $C5, $54, $43, $76, $22, $F0, $FC, $71, $95, $82, $29, $3A, $54, $EE, $4D, $21, $F6, $5A, $42, $F7, $CB, $71, $B4, $E1, $59, $9D, $AD, $34, $75, $95, $30, $46, $F3, $5F, $4B, $A3 ), {6C}
    ( $60, $88, $EB, $38, $56, $99, $12, $19, $84, $96, $86, $E9, $5B, $60, $29, $30, $BE, $98, $5A, $74, $16, $DC, $F8, $DB, $56, $6E, $2B, $00, $B6, $A4, $CC, $DF, $C0, $4A, $96, $F8, $BC, $92, $70, $CD, $BE, $0F, $D0, $DE, $76, $F0, $3C, $76 ), {6D}
    ( $14, $83, $D4, $3B, $CF, $31, $44, $6A, $8C, $56, $65, $89, $70, $7C, $99, $6B, $C8, $E7, $8F, $09, $4D, $6A, $49, $48, $F7, $B2, $F7, $FB, $EE, $3E, $D0, $2D, $78, $C0, $05, $FD, $91, $2D, $F7, $FE, $1B, $9D, $6D, $EA, $08, $09, $9A, $A4 ), {6E}
    ( $BA, $A9, $6D, $53, $77, $8A, $36, $A2, $8C, $1E, $7E, $54, $5D, $88, $29, $A5, $06, $52, $F6, $2F, $D5, $04, $90, $65, $4D, $0B, $BA, $A2, $E6, $A4, $29, $92, $73, $CA, $1A, $64, $3A, $1F, $3E, $82, $6F, $C9, $C1, $51, $5B, $44, $7F, $0D ), {6F}
    ( $08, $2F, $D0, $D9, $39, $31, $43, $0F, $F5, $AB, $49, $7D, $10, $C6, $77, $79, $3B, $17, $57, $27, $7E, $94, $8D, $6D, $41, $74, $DB, $9B, $C4, $BA, $12, $21, $EF, $4D, $14, $30, $1D, $35, $35, $FF, $B9, $A5, $DC, $2B, $A7, $9D, $63, $EF ), {70}
    ( $C7, $AC, $97, $17, $F8, $76, $B7, $D3, $F6, $E4, $78, $E5, $64, $AD, $DC, $97, $AA, $C0, $8E, $D7, $10, $11, $AF, $89, $B4, $BD, $E7, $2A, $76, $4E, $19, $68, $09, $B9, $44, $FD, $CA, $05, $39, $0B, $6D, $10, $80, $F6, $73, $60, $41, $AC ), {71}
    ( $FF, $7F, $D0, $2F, $F8, $A9, $64, $FF, $DF, $42, $4A, $06, $3E, $6B, $09, $1E, $A7, $80, $1F, $F2, $53, $72, $2C, $E7, $DC, $7E, $46, $BA, $4B, $15, $A9, $6F, $3C, $2F, $C5, $83, $69, $9C, $F2, $5D, $18, $B2, $69, $43, $17, $72, $00, $EA ), {72}
    ( $B8, $15, $D7, $D7, $9C, $81, $1F, $EE, $15, $35, $EB, $76, $07, $87, $8D, $7B, $29, $96, $94, $02, $7E, $6B, $DC, $54, $03, $5A, $AD, $7C, $B3, $98, $3E, $EB, $AA, $94, $ED, $3A, $49, $EF, $11, $96, $75, $6A, $E6, $F2, $D5, $4D, $D5, $9B ), {73}
    ( $6D, $D9, $FA, $FE, $A8, $C3, $AD, $D6, $79, $E1, $DE, $C4, $DB, $C5, $23, $7D, $DC, $FA, $31, $B5, $2D, $2C, $36, $CC, $D3, $A9, $37, $01, $1C, $36, $6A, $FD, $59, $65, $07, $1C, $FA, $5D, $20, $C8, $C1, $14, $DF, $A5, $EA, $DB, $C3, $EB ), {74}
    ( $5E, $9F, $7D, $42, $DD, $5D, $10, $DE, $13, $B2, $A3, $0A, $84, $7F, $29, $09, $FC, $D3, $8F, $5D, $35, $DD, $C8, $36, $0E, $2B, $83, $BC, $E2, $A5, $0D, $DB, $F7, $86, $AF, $58, $B6, $32, $91, $90, $9A, $5A, $D1, $B1, $4A, $4F, $A0, $6D ), {75}
    ( $FE, $9E, $C0, $9D, $46, $A7, $4B, $A5, $E5, $E5, $0D, $5A, $D4, $D7, $5E, $51, $3C, $D8, $A7, $B4, $9D, $B7, $6D, $2F, $76, $A6, $C4, $46, $06, $D8, $2E, $A1, $7C, $34, $95, $5E, $4B, $44, $12, $60, $36, $3A, $91, $A4, $16, $65, $93, $5C ), {76}
    ( $1A, $38, $7B, $AA, $52, $F7, $CD, $83, $14, $63, $BF, $65, $E1, $3B, $61, $CB, $B4, $98, $7C, $65, $8F, $FE, $11, $4B, $D3, $17, $91, $49, $0A, $BA, $B6, $9D, $89, $6A, $48, $A5, $9B, $37, $A7, $91, $83, $05, $FE, $9C, $9D, $56, $61, $FF ), {77}
    ( $16, $D8, $C3, $40, $D1, $DF, $25, $A4, $65, $55, $4E, $B9, $AA, $5B, $1C, $5B, $4D, $4F, $61, $95, $67, $78, $1E, $32, $5C, $8E, $E7, $5C, $49, $48, $9C, $7A, $0C, $01, $DF, $BE, $85, $91, $7C, $F9, $EE, $E7, $74, $CA, $E5, $1B, $90, $F2 ), {78}
    ( $57, $52, $DA, $AD, $D3, $1B, $8D, $04, $5E, $77, $72, $C6, $07, $83, $94, $31, $7C, $4A, $AF, $00, $68, $1F, $6E, $70, $44, $DE, $83, $15, $F9, $ED, $A1, $57, $5B, $F9, $5D, $E0, $32, $FA, $1C, $DE, $64, $9D, $32, $80, $32, $43, $D3, $99 ), {79}
    ( $BE, $EC, $9D, $37, $E2, $C3, $BF, $51, $F2, $1C, $F6, $65, $F1, $0C, $C7, $D1, $37, $59, $40, $7E, $93, $8C, $06, $4F, $39, $22, $F0, $5A, $DF, $F4, $BB, $2C, $06, $7B, $A9, $DC, $12, $04, $6F, $B0, $DC, $1A, $63, $78, $D8, $5E, $75, $E1 ), {7A}
    ( $99, $CE, $66, $51, $A9, $2D, $8B, $92, $33, $A3, $2E, $64, $93, $54, $DC, $96, $C8, $B1, $E4, $DB, $90, $EF, $28, $65, $98, $37, $31, $A3, $BB, $21, $01, $8E, $0B, $78, $EE, $C1, $B3, $2C, $74, $FB, $8C, $7B, $AD, $84, $46, $EC, $98, $21 ), {7B}
    ( $13, $BE, $20, $E9, $B3, $F9, $96, $A6, $9C, $EA, $6F, $A7, $02, $FF, $AC, $F6, $22, $9A, $AC, $B0, $7A, $4A, $51, $35, $0D, $FA, $D9, $EF, $1B, $C9, $DE, $6B, $48, $66, $18, $6C, $B2, $6F, $F1, $71, $B0, $B6, $54, $83, $FD, $7B, $27, $57 ), {7C}
    ( $B8, $22, $C1, $DF, $0D, $D7, $72, $50, $53, $67, $DE, $13, $07, $6D, $89, $29, $B2, $4F, $3E, $42, $7D, $C4, $51, $36, $D5, $EC, $3D, $2D, $3B, $A1, $1A, $F1, $49, $DA, $4E, $05, $54, $F3, $E6, $8F, $36, $DD, $D4, $86, $08, $EF, $4F, $DE ), {7D}
    ( $47, $41, $40, $A4, $DD, $9C, $2E, $2E, $8C, $D9, $A6, $09, $0B, $71, $6F, $BC, $02, $11, $19, $E7, $A3, $7E, $46, $79, $51, $23, $0D, $2D, $9D, $01, $6F, $91, $A7, $7E, $12, $06, $78, $A1, $35, $10, $8B, $CE, $4C, $B4, $2C, $0E, $C8, $F3 ), {7E}
    ( $8A, $61, $34, $E2, $1E, $3C, $A9, $45, $60, $58, $CE, $41, $66, $98, $C8, $DD, $51, $CF, $5C, $54, $33, $8E, $A4, $2A, $AF, $A3, $82, $29, $FD, $08, $EA, $C3, $EB, $9C, $AF, $46, $45, $CA, $35, $F2, $D5, $70, $01, $0F, $B5, $CC, $D6, $6E ), {7F}
    ( $9F, $38, $FC, $AB, $D0, $E4, $C9, $40, $00, $1F, $11, $FF, $76, $6E, $87, $80, $E0, $51, $44, $EC, $8C, $D4, $A1, $1B, $C6, $0B, $2F, $CE, $3F, $3A, $44, $9F, $D4, $E5, $2A, $63, $5C, $1F, $E3, $C0, $B0, $1C, $EC, $46, $DC, $DC, $CF, $7B ), {80}
    ( $A9, $1B, $65, $2F, $7E, $11, $9F, $0D, $9A, $18, $CD, $9B, $E4, $6B, $0B, $04, $5F, $D1, $1E, $E0, $89, $3B, $C6, $C3, $EB, $9A, $68, $ED, $FA, $25, $90, $13, $AA, $21, $B1, $76, $26, $64, $89, $F4, $D9, $1E, $39, $19, $8F, $4D, $63, $8F ), {81}
    ( $9B, $12, $71, $53, $A2, $3F, $F9, $2C, $A4, $68, $EA, $60, $51, $4A, $E8, $BF, $5C, $03, $B3, $55, $12, $C8, $58, $2C, $24, $4D, $03, $C0, $52, $46, $97, $69, $3F, $89, $61, $0C, $FE, $E9, $DA, $48, $C0, $3E, $3A, $87, $8D, $C3, $74, $56 ), {82}
    ( $5C, $D0, $61, $A1, $83, $65, $FD, $7B, $AB, $C0, $6B, $3F, $A9, $33, $C5, $D1, $EA, $F3, $E1, $9F, $E3, $A3, $4D, $90, $B7, $A3, $AC, $C7, $7B, $3B, $C1, $1E, $55, $91, $3E, $F3, $3C, $F9, $81, $05, $3E, $90, $41, $B2, $30, $8F, $7E, $65 ), {83}
    ( $2A, $5C, $4E, $65, $3E, $3A, $38, $B4, $CB, $C0, $80, $CE, $9A, $5B, $D3, $5F, $33, $58, $FA, $5D, $DC, $A2, $EC, $F2, $A6, $4D, $B2, $93, $C9, $F7, $43, $CE, $9D, $E5, $B1, $88, $47, $9F, $BD, $36, $36, $53, $63, $7E, $47, $53, $CD, $11 ), {84}
    ( $51, $38, $67, $EE, $40, $0B, $80, $39, $42, $F0, $4F, $A7, $C2, $10, $E4, $54, $7D, $70, $7D, $0C, $F0, $78, $8C, $83, $3A, $13, $74, $39, $BF, $7C, $09, $8D, $C1, $C9, $9B, $2D, $E1, $6F, $8E, $D6, $71, $25, $AA, $F5, $CC, $63, $65, $ED ), {85}
    ( $55, $45, $7A, $44, $AA, $50, $4B, $55, $EC, $A2, $91, $10, $8D, $66, $C4, $07, $4B, $BE, $18, $6D, $A7, $52, $2A, $95, $80, $B7, $3D, $0E, $B0, $B9, $7D, $B8, $B9, $09, $32, $29, $29, $25, $12, $AB, $2F, $F1, $83, $A8, $A2, $44, $2A, $40 ), {86}
    ( $D9, $7A, $23, $21, $30, $9F, $00, $5E, $5E, $AB, $89, $25, $07, $04, $4A, $AF, $12, $DC, $06, $BB, $F5, $97, $8A, $1B, $0E, $55, $A2, $DF, $E1, $A1, $95, $CB, $F1, $D6, $09, $6E, $69, $5C, $B8, $EA, $0F, $F1, $61, $F6, $F4, $71, $4D, $F9 ), {87}
    ( $94, $AE, $AC, $96, $2A, $2E, $51, $F6, $10, $3E, $5A, $2F, $19, $52, $61, $C2, $1D, $8C, $A4, $23, $5D, $46, $BE, $29, $04, $B8, $44, $46, $6C, $B5, $51, $4D, $85, $BB, $8C, $F5, $B6, $83, $C4, $60, $3F, $39, $79, $D5, $3B, $79, $09, $87 ), {88}
    ( $FA, $0B, $45, $DA, $96, $6A, $C2, $FB, $8B, $E0, $D4, $01, $43, $0E, $1A, $C1, $12, $11, $52, $36, $F9, $16, $8A, $F2, $64, $A0, $EC, $41, $3A, $54, $1F, $F5, $A6, $98, $CE, $DC, $5E, $87, $C8, $86, $38, $28, $4B, $51, $A3, $36, $89, $C0 ), {89}
    ( $EC, $3C, $F1, $8E, $70, $FB, $AF, $19, $11, $EE, $3C, $E8, $21, $B6, $64, $9C, $EA, $B2, $DE, $A7, $3D, $5B, $E2, $49, $93, $42, $04, $A8, $2E, $CF, $18, $0A, $19, $65, $7E, $A3, $36, $13, $59, $88, $78, $A7, $8C, $1A, $4B, $11, $25, $63 ), {8A}
    ( $F1, $2C, $48, $C7, $60, $47, $19, $C0, $D6, $9E, $26, $1F, $1E, $83, $CB, $3C, $16, $AC, $BB, $A6, $D4, $52, $08, $DF, $27, $8A, $9C, $B0, $E5, $D9, $63, $89, $E6, $D8, $AD, $9B, $5C, $72, $D2, $55, $57, $F5, $94, $28, $C7, $96, $2F, $23 ), {8B}
    ( $84, $84, $75, $58, $2D, $43, $F3, $05, $09, $8F, $26, $C0, $4A, $31, $23, $63, $37, $0E, $C3, $4F, $1B, $E0, $DA, $DC, $55, $7B, $7B, $07, $6E, $6C, $2B, $99, $0F, $12, $9C, $01, $81, $7F, $E1, $C3, $B6, $A9, $D8, $28, $2B, $3D, $E7, $32 ), {8C}
    ( $D6, $A1, $31, $D6, $AB, $98, $26, $1D, $89, $92, $09, $56, $4E, $71, $04, $B8, $80, $01, $41, $13, $82, $A0, $84, $15, $7E, $40, $F4, $83, $85, $AB, $99, $B6, $72, $1C, $F4, $63, $BD, $04, $09, $62, $C9, $79, $32, $E1, $0F, $E6, $FF, $88 ), {8D}
    ( $FC, $8D, $5E, $D5, $DD, $2D, $85, $89, $2F, $5B, $BC, $08, $AE, $55, $DC, $A7, $0F, $85, $E7, $6A, $27, $0B, $78, $5E, $53, $37, $6F, $DA, $46, $39, $AB, $02, $BF, $7D, $E0, $1A, $DA, $5E, $02, $DC, $3C, $E1, $6A, $48, $E8, $01, $36, $09 ), {8E}
    ( $C7, $4B, $87, $54, $3E, $71, $30, $E8, $47, $7F, $CC, $A4, $B8, $0A, $01, $77, $03, $1D, $4E, $86, $44, $D9, $D5, $02, $42, $DC, $92, $66, $68, $CA, $A4, $13, $CA, $2D, $9B, $5A, $E4, $EA, $5E, $D6, $CE, $5F, $E6, $9D, $FD, $F1, $1F, $1D ), {8F}
    ( $0D, $6F, $3D, $6B, $16, $D7, $14, $B9, $84, $54, $48, $D2, $A0, $CB, $27, $7B, $CB, $BF, $A8, $6F, $50, $7E, $06, $BD, $8E, $94, $72, $64, $8A, $2D, $CD, $88, $2F, $81, $91, $AF, $C9, $B8, $05, $F7, $E3, $CC, $63, $7E, $4B, $19, $FE, $E3 ), {90}
    ( $51, $2E, $72, $F9, $CC, $34, $9A, $FF, $99, $23, $3D, $F3, $90, $B7, $78, $FB, $BD, $F0, $F5, $31, $89, $E6, $2E, $EB, $F2, $5E, $BE, $4B, $6E, $0B, $7F, $56, $F9, $6B, $0F, $20, $3F, $EA, $15, $29, $64, $CA, $C3, $FB, $FF, $7F, $8C, $14 ), {91}
    ( $5B, $87, $8F, $8D, $73, $18, $30, $DC, $AB, $C0, $8A, $1A, $7D, $E5, $55, $5F, $7D, $16, $FB, $F3, $ED, $64, $F8, $A6, $88, $B1, $64, $7C, $5E, $21, $61, $5D, $F7, $F9, $A8, $99, $86, $6C, $AE, $AE, $D4, $E9, $2D, $6E, $63, $32, $D1, $9F ), {92}
    ( $90, $A5, $22, $49, $9D, $09, $AF, $0E, $51, $A1, $54, $78, $B7, $F9, $7F, $ED, $1C, $4B, $7E, $71, $1B, $7A, $12, $CC, $84, $3D, $32, $66, $02, $B7, $3D, $C4, $59, $E7, $BA, $FD, $A5, $8C, $00, $F0, $DC, $1D, $E6, $B9, $30, $D9, $91, $00 ), {93}
    ( $5E, $3B, $B0, $F9, $A4, $B5, $7C, $A5, $BC, $25, $5F, $E7, $D2, $46, $CE, $B7, $08, $2A, $D3, $B5, $BC, $D1, $E9, $C1, $4D, $40, $30, $4A, $40, $51, $27, $23, $84, $CC, $F7, $B0, $FE, $28, $81, $F5, $B6, $36, $14, $49, $AD, $28, $5B, $F8 ), {94}
    ( $67, $E8, $C3, $3A, $3B, $A8, $19, $26, $2E, $D5, $90, $CB, $D6, $5B, $D1, $89, $35, $78, $60, $BC, $BD, $50, $74, $3B, $4F, $F1, $0B, $29, $FA, $2C, $88, $97, $8B, $63, $2D, $E0, $63, $5E, $03, $53, $AC, $72, $89, $31, $A0, $C4, $E1, $5D ), {95}
    ( $A7, $43, $28, $F0, $AB, $7B, $24, $6B, $9E, $E3, $D7, $D3, $BD, $72, $18, $B9, $95, $BF, $28, $F7, $F4, $8D, $B8, $A7, $80, $7B, $DC, $CF, $C5, $54, $BB, $40, $01, $47, $6B, $12, $36, $36, $BF, $FA, $80, $E2, $F4, $19, $B2, $CC, $08, $B0 ), {96}
    ( $D3, $20, $C5, $6D, $1F, $11, $02, $C2, $66, $6A, $DD, $96, $EA, $63, $78, $DD, $D1, $17, $AB, $72, $B7, $BE, $D9, $CA, $50, $AD, $E4, $DB, $44, $F4, $42, $60, $8E, $41, $49, $55, $13, $76, $07, $18, $BA, $B8, $E7, $18, $B7, $10, $34, $BD ), {97}
    ( $D2, $66, $BE, $80, $11, $AD, $B6, $28, $B5, $3D, $E7, $CC, $24, $CA, $4E, $3F, $E3, $48, $0E, $BA, $86, $6F, $67, $E4, $34, $02, $5E, $9C, $E7, $7A, $23, $4E, $7A, $BF, $9A, $00, $BC, $4A, $12, $18, $EC, $C4, $6D, $CB, $0D, $6A, $FF, $53 ), {98}
    ( $B3, $34, $47, $DA, $0C, $E4, $3B, $66, $90, $72, $F2, $85, $70, $24, $16, $44, $B6, $2B, $6A, $2D, $10, $16, $B0, $67, $1F, $91, $EA, $E0, $39, $39, $8D, $01, $DE, $CF, $EF, $B9, $F0, $F0, $C9, $EA, $AB, $7B, $0E, $65, $82, $0F, $06, $71 ), {99}
    ( $95, $0B, $D7, $7A, $87, $A8, $7F, $66, $3A, $FD, $F0, $93, $BC, $4E, $6D, $C1, $05, $41, $FC, $8E, $2D, $2E, $D9, $F7, $D1, $E0, $20, $F0, $CF, $7A, $64, $C9, $04, $A8, $A4, $12, $E7, $D0, $79, $E0, $31, $2F, $24, $EB, $13, $EF, $11, $B5 ), {9A}
    ( $29, $5E, $8F, $0D, $D9, $65, $5E, $C1, $C7, $BD, $AC, $AE, $38, $77, $E7, $E7, $53, $13, $10, $1D, $E9, $7E, $F9, $5F, $B9, $76, $FA, $41, $D7, $F9, $1F, $0F, $AA, $DA, $47, $72, $28, $B4, $A4, $5E, $C7, $85, $C5, $69, $EC, $57, $28, $2A ), {9B}
    ( $07, $72, $EA, $5F, $66, $69, $9F, $26, $3A, $B1, $A2, $E0, $68, $8F, $F2, $C1, $6E, $E7, $AA, $1A, $68, $9B, $0E, $80, $62, $4F, $87, $B0, $4D, $AC, $3A, $2D, $BD, $CA, $62, $E6, $EE, $CE, $06, $D9, $AF, $4D, $70, $2B, $BF, $7E, $13, $9E ), {9C}
    ( $76, $D6, $A5, $4B, $02, $D1, $36, $33, $CE, $92, $B5, $53, $09, $88, $6C, $69, $C6, $13, $2E, $AA, $11, $F6, $0E, $F3, $B1, $73, $72, $36, $9F, $32, $D1, $2B, $40, $E3, $70, $E4, $C7, $C5, $C5, $4D, $8D, $5D, $7C, $99, $2C, $5C, $5E, $7B ), {9D}
    ( $C1, $EC, $12, $16, $61, $A2, $5B, $5C, $E2, $FF, $D2, $35, $86, $68, $5A, $62, $2F, $4A, $0F, $D1, $08, $A3, $3B, $98, $93, $40, $FA, $35, $8B, $DF, $89, $58, $E2, $85, $AD, $12, $29, $8C, $00, $D1, $A5, $22, $C9, $97, $8A, $F5, $DD, $93 ), {9E}
    ( $37, $4B, $F8, $DF, $88, $BB, $69, $3D, $3C, $A6, $78, $B1, $0C, $7D, $67, $A3, $EF, $88, $3C, $2A, $4C, $47, $1B, $0D, $E0, $B5, $71, $7B, $77, $9A, $1F, $5B, $8C, $DD, $1A, $33, $70, $A3, $8C, $77, $3E, $2D, $02, $59, $D4, $9C, $F1, $60 ), {9F}
    ( $A3, $9C, $48, $D0, $01, $B2, $7D, $E3, $2F, $04, $57, $AC, $A3, $0A, $88, $29, $C5, $41, $DE, $FF, $D1, $D7, $17, $F3, $58, $B3, $41, $7D, $42, $7A, $91, $22, $43, $3E, $8A, $E7, $1E, $5D, $F4, $A3, $F3, $40, $66, $51, $4F, $A6, $2C, $4C ), {A0}
    ( $2A, $1D, $07, $1F, $D5, $BE, $DB, $45, $FC, $14, $0E, $C0, $B5, $24, $C6, $F3, $53, $04, $CB, $9E, $39, $E5, $32, $49, $4D, $19, $F9, $DA, $26, $B8, $45, $B0, $07, $3E, $E7, $87, $5F, $BB, $FE, $B7, $9B, $40, $BF, $30, $49, $52, $EC, $D0 ), {A1}
    ( $DE, $EC, $24, $30, $97, $DD, $DB, $C1, $AE, $FE, $ED, $29, $DD, $75, $4A, $D6, $CB, $E6, $97, $25, $70, $33, $8C, $32, $71, $0B, $52, $B3, $70, $81, $8E, $36, $89, $C8, $72, $F8, $EA, $B2, $52, $EA, $ED, $B8, $E6, $50, $64, $6A, $CF, $75 ), {A2}
    ( $68, $27, $9A, $49, $65, $A7, $0F, $C6, $9B, $08, $53, $91, $B4, $4B, $C2, $94, $84, $A2, $E7, $D8, $1D, $D5, $3C, $C0, $FF, $14, $F9, $40, $C5, $5E, $03, $46, $B6, $D4, $16, $B6, $48, $59, $20, $FC, $DF, $5A, $89, $D9, $CD, $AD, $A7, $CD ), {A3}
    ( $65, $FF, $F7, $F2, $E6, $CF, $3F, $6D, $26, $C9, $38, $5E, $08, $1D, $95, $45, $A9, $61, $77, $27, $1A, $F9, $D3, $8D, $8C, $D5, $38, $45, $D7, $72, $ED, $6E, $F7, $64, $BB, $29, $9B, $5A, $F4, $07, $F5, $07, $80, $EC, $7B, $D8, $2F, $91 ), {A4}
    ( $8D, $41, $94, $9F, $C8, $4C, $A7, $92, $D7, $3D, $6E, $6C, $F4, $7F, $C6, $25, $A0, $4D, $AE, $11, $52, $03, $E6, $38, $92, $2C, $A2, $2A, $43, $6D, $9F, $35, $86, $90, $F4, $E0, $DE, $7C, $2F, $C2, $F4, $C2, $2F, $8E, $4F, $6B, $0E, $3B ), {A5}
    ( $E5, $87, $95, $ED, $BB, $13, $C9, $AC, $E4, $1E, $59, $D6, $91, $9E, $66, $44, $50, $A1, $0F, $2A, $ED, $93, $05, $45, $C6, $B2, $96, $07, $A4, $C0, $15, $7E, $55, $A2, $03, $C0, $59, $78, $C6, $7A, $E4, $68, $75, $0D, $7A, $7B, $BC, $B9 ), {A6}
    ( $C7, $53, $DF, $F5, $B0, $9D, $BD, $72, $8E, $18, $ED, $28, $C6, $58, $BC, $A3, $D7, $FE, $E8, $2C, $A5, $8B, $D6, $57, $81, $25, $25, $1B, $78, $04, $37, $36, $9B, $3A, $4A, $FD, $6A, $5D, $14, $45, $17, $38, $44, $BB, $34, $B1, $14, $11 ), {A7}
    ( $8C, $FB, $F9, $1C, $80, $D6, $11, $8D, $D6, $6A, $75, $9B, $AC, $77, $D7, $5F, $BE, $D7, $5E, $4D, $EE, $BF, $81, $5C, $45, $2B, $FC, $0B, $2D, $70, $10, $81, $15, $DA, $6A, $5E, $50, $84, $38, $9A, $3D, $5C, $CC, $F4, $C2, $B7, $E1, $17 ), {A8}
    ( $7D, $AE, $0A, $D8, $F2, $D0, $15, $8E, $75, $36, $F1, $63, $50, $BD, $84, $FA, $E4, $82, $B1, $84, $84, $7F, $DE, $58, $1C, $F0, $85, $C5, $E6, $9E, $69, $F7, $48, $BE, $15, $7B, $28, $AB, $22, $80, $08, $33, $CB, $46, $25, $96, $34, $60 ), {A9}
    ( $88, $74, $F6, $C4, $00, $FD, $A1, $77, $8A, $F3, $8E, $9D, $49, $F0, $37, $3A, $BE, $B6, $22, $F8, $44, $F7, $72, $99, $AC, $65, $5A, $E4, $E3, $D2, $DC, $9B, $40, $8F, $4D, $37, $CA, $0E, $A6, $02, $90, $6D, $EF, $F2, $F1, $B0, $58, $00 ), {AA}
    ( $F3, $B7, $F6, $B5, $1B, $38, $C9, $A4, $B1, $6D, $85, $2D, $6A, $FA, $41, $AC, $D4, $20, $07, $3A, $96, $24, $62, $53, $AA, $42, $8C, $F4, $D2, $6B, $91, $CA, $CA, $09, $50, $1A, $B9, $C1, $4B, $F6, $94, $8E, $6B, $0E, $58, $3E, $E3, $D3 ), {AB}
    ( $49, $49, $1C, $25, $8E, $A8, $47, $E3, $B4, $26, $1C, $80, $62, $9F, $D4, $BA, $1F, $F8, $C5, $CD, $87, $EA, $E3, $E3, $11, $26, $A6, $73, $66, $9C, $84, $1C, $CA, $5B, $F2, $D6, $40, $0D, $99, $53, $CA, $BE, $6E, $5B, $A9, $50, $17, $FA ), {AC}
    ( $0D, $9B, $3A, $9C, $81, $00, $E3, $4A, $0D, $76, $CB, $44, $D0, $2B, $62, $F7, $B5, $C8, $30, $9F, $08, $5A, $88, $0B, $9C, $35, $17, $DC, $FF, $B1, $FA, $17, $9F, $3B, $EF, $E0, $1D, $DB, $34, $D2, $1F, $C1, $BC, $B8, $20, $97, $4A, $C3 ), {AD}
    ( $F0, $83, $1E, $56, $FE, $37, $61, $30, $6F, $BE, $E7, $B0, $56, $18, $50, $EE, $48, $DC, $19, $16, $06, $44, $B6, $83, $02, $43, $D3, $CC, $96, $04, $C6, $DC, $BE, $81, $7F, $E5, $3E, $6C, $63, $6E, $83, $7C, $8E, $88, $78, $1F, $3E, $40 ), {AE}
    ( $5B, $47, $7C, $40, $C4, $EC, $0D, $47, $AE, $CA, $1C, $3A, $46, $55, $4E, $63, $46, $BD, $42, $EC, $0A, $24, $5E, $BD, $29, $49, $06, $A9, $EF, $A6, $0B, $E0, $0A, $C5, $1F, $39, $09, $14, $B4, $49, $C5, $40, $CB, $4C, $CF, $1B, $F4, $59 ), {AF}
    ( $70, $ED, $12, $77, $BF, $F9, $32, $3F, $F8, $6A, $37, $B3, $B1, $9B, $DA, $0C, $65, $A0, $6C, $14, $E5, $FE, $52, $CB, $A1, $2A, $00, $E8, $40, $74, $B0, $B8, $69, $C4, $E6, $EB, $D3, $40, $44, $D7, $BE, $7C, $07, $4F, $AC, $A3, $1E, $A6 ), {B0}
    ( $11, $F4, $58, $76, $E8, $EA, $4C, $F7, $C2, $C3, $58, $80, $9D, $F6, $02, $0E, $32, $0C, $92, $1C, $08, $E8, $DF, $79, $E2, $0C, $22, $9E, $0B, $94, $72, $74, $EB, $E3, $90, $E9, $E6, $77, $B7, $2E, $09, $B5, $06, $E5, $63, $A0, $1D, $68 ), {B1}
    ( $84, $FB, $6F, $54, $B4, $D9, $3B, $68, $81, $7E, $F6, $E4, $54, $BC, $C2, $0A, $80, $EE, $99, $BF, $2A, $00, $EC, $CA, $19, $06, $82, $9C, $A6, $F0, $70, $80, $D7, $98, $74, $C7, $14, $6C, $F9, $6E, $D6, $71, $CC, $A6, $EB, $85, $19, $51 ), {B2}
    ( $D0, $8C, $75, $0B, $8A, $B9, $D9, $AC, $6B, $D8, $45, $97, $60, $66, $7E, $DE, $D5, $B5, $9D, $70, $CB, $87, $91, $5C, $74, $4B, $EA, $A7, $70, $63, $B0, $9A, $FA, $2B, $A6, $53, $07, $47, $A4, $C8, $20, $0C, $F4, $7C, $BB, $28, $A4, $3D ), {B3}
    ( $5F, $2E, $5A, $66, $4E, $ED, $85, $4B, $8C, $E1, $84, $12, $A6, $9E, $08, $AE, $C1, $26, $45, $7A, $93, $19, $06, $AA, $C0, $18, $91, $73, $FD, $5C, $A4, $B2, $1D, $BC, $1D, $FB, $96, $36, $AB, $7B, $22, $08, $59, $A3, $15, $56, $D5, $88 ), {B4}
    ( $25, $35, $62, $02, $38, $83, $F8, $BF, $96, $EA, $05, $BD, $71, $0F, $2F, $E0, $FF, $84, $5A, $B4, $57, $4D, $E3, $2A, $70, $D8, $57, $F5, $0F, $A9, $E8, $20, $75, $EE, $32, $FF, $A5, $4A, $83, $78, $61, $BF, $09, $06, $53, $16, $A0, $2C ), {B5}
    ( $0B, $50, $20, $86, $54, $62, $7B, $7D, $E2, $E5, $B8, $DE, $D4, $7C, $E7, $38, $17, $1F, $47, $09, $63, $52, $E1, $6A, $23, $38, $8C, $DF, $D0, $C6, $DF, $33, $28, $75, $0F, $E7, $62, $A9, $3B, $62, $92, $DA, $26, $A9, $2C, $D7, $CB, $AA ), {B6}
    ( $13, $EC, $5F, $F2, $95, $83, $92, $97, $E6, $B4, $E4, $FD, $84, $DD, $C6, $C8, $4B, $AA, $21, $5B, $8A, $CF, $50, $EA, $E6, $B6, $00, $FB, $A9, $E8, $D5, $90, $34, $56, $B8, $45, $EA, $BD, $B7, $9D, $33, $80, $28, $68, $E4, $75, $18, $70 ), {B7}
    ( $7A, $B0, $B9, $F0, $C9, $42, $BD, $54, $41, $F0, $B9, $2F, $57, $7A, $A4, $DB, $3E, $CD, $D6, $02, $A3, $77, $9B, $90, $5C, $3C, $CC, $D2, $E0, $5A, $4E, $77, $B1, $FA, $04, $BA, $94, $E5, $E2, $0C, $C4, $82, $37, $E1, $55, $58, $EE, $B0 ), {B8}
    ( $9C, $84, $02, $03, $B0, $C8, $9D, $3C, $F1, $A4, $F7, $45, $A6, $5F, $A8, $38, $59, $E3, $A7, $91, $EC, $75, $44, $9A, $C9, $0F, $90, $F3, $1A, $16, $E9, $69, $A0, $68, $02, $5E, $4A, $6C, $B8, $D1, $9B, $10, $D2, $66, $63, $60, $57, $0C ), {B9}
    ( $35, $AA, $AD, $46, $02, $16, $55, $87, $A7, $FA, $BD, $32, $84, $19, $68, $50, $1A, $8E, $1F, $7E, $84, $3C, $1D, $7A, $8E, $FE, $F7, $0E, $14, $AC, $DA, $23, $50, $5A, $7B, $9C, $4A, $D2, $F2, $C7, $4F, $3C, $45, $6B, $FE, $11, $D8, $F3 ), {BA}
    ( $0B, $8D, $A1, $53, $8E, $52, $96, $5A, $F9, $41, $32, $5C, $42, $2C, $5E, $42, $E4, $48, $5F, $D1, $E3, $AB, $B8, $27, $0F, $49, $BB, $8E, $81, $97, $EA, $48, $15, $53, $3A, $86, $5E, $78, $AE, $AB, $DD, $A0, $2B, $AD, $D7, $30, $FA, $EF ), {BB}
    ( $48, $43, $78, $F7, $20, $50, $9D, $A9, $71, $A1, $92, $74, $24, $D8, $DC, $7C, $96, $1C, $EA, $AF, $5D, $A0, $0E, $42, $AD, $0F, $6F, $CC, $34, $9E, $C2, $B3, $F9, $15, $8B, $51, $D3, $67, $F9, $3E, $F4, $AD, $A2, $68, $25, $2F, $26, $E8 ), {BC}
    ( $D8, $16, $17, $99, $2D, $5E, $39, $93, $8B, $77, $EC, $FF, $D0, $68, $18, $2C, $77, $03, $07, $F1, $61, $15, $39, $D2, $EE, $65, $12, $20, $6C, $A0, $DA, $2C, $00, $2A, $EA, $8D, $CF, $3E, $BF, $28, $BA, $45, $A7, $21, $7C, $5C, $D5, $26 ), {BD}
    ( $8A, $95, $A8, $1C, $DB, $07, $48, $F9, $09, $E9, $55, $AE, $89, $12, $72, $FE, $88, $3F, $BE, $4E, $17, $10, $41, $8E, $02, $7B, $D2, $23, $61, $92, $D5, $6B, $C2, $DD, $AA, $F8, $36, $2B, $99, $BC, $8C, $89, $8F, $C8, $C7, $BF, $EB, $3F ), {BE}
    ( $25, $14, $0A, $B0, $06, $78, $12, $2F, $58, $2B, $A6, $60, $BE, $DC, $4B, $3E, $2E, $83, $2B, $A5, $48, $B0, $06, $6A, $72, $DF, $43, $CC, $A0, $93, $C3, $82, $34, $4B, $4F, $FB, $28, $A0, $E2, $06, $33, $A8, $79, $0F, $EC, $64, $87, $F9 ), {BF}
    ( $D7, $35, $B0, $58, $06, $FA, $FB, $B7, $C4, $C3, $C8, $8D, $71, $FC, $93, $C1, $0B, $82, $3E, $CC, $D7, $AB, $74, $24, $59, $E2, $FC, $BF, $76, $C4, $D9, $2D, $08, $2B, $EF, $38, $FD, $0A, $6A, $0C, $CD, $BC, $E0, $50, $B8, $A6, $A1, $90 ), {C0}
    ( $A5, $EC, $93, $51, $D9, $16, $69, $81, $36, $FC, $73, $85, $80, $FA, $0A, $FD, $91, $48, $A6, $9E, $46, $1E, $55, $93, $34, $26, $78, $FE, $E0, $14, $9E, $94, $68, $E4, $13, $28, $70, $50, $C8, $1D, $D3, $3E, $1D, $40, $98, $A6, $4A, $CD ), {C1}
    ( $69, $CE, $D3, $D4, $C3, $8A, $80, $A9, $0C, $36, $F1, $E1, $75, $F3, $60, $4A, $2D, $2C, $98, $BF, $B2, $68, $47, $8E, $C4, $FC, $72, $B5, $36, $A4, $79, $D6, $EC, $E3, $ED, $3A, $ED, $F6, $29, $A0, $C9, $F4, $05, $D3, $0F, $2E, $BB, $03 ), {C2}
    ( $4A, $7A, $DF, $8D, $55, $4E, $03, $9B, $53, $32, $8E, $A5, $23, $DC, $56, $BB, $52, $DE, $23, $BF, $A8, $D6, $72, $4F, $2E, $49, $93, $D7, $DC, $44, $E4, $E3, $26, $9C, $F5, $6F, $8E, $9E, $AE, $83, $4C, $C4, $32, $4E, $D7, $E3, $C2, $26 ), {C3}
    ( $FD, $C2, $9B, $E0, $B2, $F0, $22, $90, $69, $14, $D3, $14, $4A, $09, $6F, $72, $1E, $FB, $A0, $F5, $F8, $E7, $1A, $C6, $5C, $23, $90, $20, $36, $74, $55, $CF, $36, $CA, $8B, $2C, $C0, $5E, $10, $94, $63, $08, $0F, $8D, $80, $82, $3C, $DD ), {C4}
    ( $A5, $EA, $DA, $B6, $95, $17, $92, $CB, $DA, $3F, $3F, $12, $5E, $14, $87, $A5, $AF, $BD, $DC, $0B, $1F, $B3, $EC, $38, $B2, $FC, $F3, $FB, $6F, $22, $39, $24, $73, $42, $FE, $2C, $14, $69, $03, $42, $FB, $2C, $E5, $63, $C6, $1C, $ED, $F2 ), {C5}
    ( $2A, $A2, $E0, $31, $C6, $5D, $6A, $59, $B0, $DA, $89, $D1, $B5, $32, $D1, $42, $99, $05, $9D, $9C, $D1, $96, $23, $B3, $04, $17, $72, $F6, $FA, $9C, $F4, $9F, $71, $08, $F0, $54, $08, $A4, $C5, $B0, $0A, $B6, $65, $96, $0D, $30, $78, $A6 ), {C6}
    ( $E3, $3D, $BD, $BE, $30, $DF, $98, $D3, $01, $02, $D8, $40, $FA, $10, $17, $28, $A7, $94, $B5, $1E, $AA, $60, $4D, $96, $78, $75, $19, $D2, $B7, $6D, $47, $45, $4D, $E0, $8C, $C1, $91, $9C, $1D, $A0, $D2, $BC, $95, $20, $87, $66, $10, $98 ), {C7}
    ( $30, $35, $21, $55, $4C, $82, $47, $3B, $39, $F3, $8E, $8F, $F3, $74, $B9, $79, $7A, $25, $BF, $C8, $CB, $43, $FF, $4E, $68, $11, $4C, $AE, $A4, $7C, $C7, $22, $AF, $DF, $0D, $D1, $FA, $C0, $A9, $CB, $A8, $C4, $DD, $E5, $D0, $07, $7D, $69 ), {C8}
    ( $1F, $3B, $E3, $06, $4A, $F3, $E4, $F5, $06, $EF, $36, $7D, $DD, $FB, $C5, $C0, $45, $A2, $4B, $E4, $47, $1D, $E5, $DC, $01, $46, $45, $8D, $CC, $C3, $8B, $DC, $53, $35, $ED, $0B, $6C, $02, $2D, $29, $5C, $B6, $FF, $4B, $CA, $8E, $30, $2A ), {C9}
    ( $51, $DD, $AF, $63, $D3, $33, $96, $89, $DD, $84, $58, $8B, $2B, $68, $49, $CF, $22, $A0, $7F, $D1, $85, $FB, $80, $66, $9C, $8D, $BD, $CD, $17, $4B, $F0, $A3, $15, $43, $8A, $EC, $1B, $81, $2C, $75, $FF, $73, $13, $DD, $3C, $21, $15, $F4 ), {CA}
    ( $AD, $EA, $9F, $B2, $C6, $8F, $9B, $C8, $13, $4C, $9A, $00, $00, $90, $38, $E9, $28, $3F, $D7, $52, $DC, $9C, $75, $0A, $98, $C7, $43, $71, $DA, $82, $23, $08, $CA, $2B, $AD, $FD, $5E, $78, $8B, $36, $1B, $54, $E7, $11, $6B, $63, $12, $CE ), {CB}
    ( $4A, $1F, $08, $C2, $76, $C5, $A9, $B0, $62, $AD, $7B, $28, $82, $57, $BC, $FC, $67, $8C, $AA, $4F, $BF, $90, $FB, $F7, $F9, $87, $C2, $C9, $D3, $0D, $F6, $69, $F0, $26, $9F, $1D, $D2, $A2, $7A, $56, $FC, $62, $ED, $98, $78, $B8, $38, $E1 ), {CC}
    ( $41, $E8, $A5, $81, $09, $CB, $C0, $BB, $C3, $0E, $7A, $C0, $CC, $D3, $D1, $B7, $AD, $BB, $44, $BD, $60, $5D, $BC, $DD, $56, $F3, $8F, $82, $5D, $C3, $0C, $CC, $63, $57, $CD, $30, $14, $BE, $98, $A8, $FF, $71, $CE, $D2, $75, $B0, $DC, $9C ), {CD}
    ( $AE, $64, $E6, $73, $1A, $0E, $FE, $0A, $92, $F2, $3F, $D0, $DD, $83, $97, $73, $6F, $17, $45, $A6, $DE, $1B, $4D, $AE, $DE, $43, $EF, $2E, $4E, $53, $6C, $A2, $7C, $68, $9A, $47, $7D, $A3, $94, $95, $5A, $41, $44, $BF, $62, $BD, $0E, $95 ), {CE}
    ( $8A, $08, $00, $51, $31, $F6, $DE, $E4, $97, $48, $87, $1E, $1A, $7B, $EA, $F2, $68, $B4, $42, $84, $FB, $D8, $EE, $DA, $34, $BA, $6E, $AA, $8B, $07, $A9, $48, $51, $D9, $E8, $C4, $36, $02, $E0, $6D, $D4, $9D, $2C, $5D, $DB, $29, $3B, $0E ), {CF}
    ( $8B, $42, $FB, $A4, $17, $97, $02, $0C, $70, $D1, $F8, $48, $24, $11, $9F, $5E, $50, $FE, $88, $6D, $27, $C8, $DA, $41, $AD, $4D, $1A, $2C, $BA, $DA, $53, $7F, $BB, $E4, $98, $22, $58, $FA, $6F, $30, $87, $69, $05, $3D, $E3, $EE, $DF, $F9 ), {D0}
    ( $00, $2E, $FC, $51, $F1, $4A, $CA, $F6, $74, $A8, $84, $0A, $E2, $E7, $64, $23, $5B, $E4, $73, $9C, $61, $93, $A4, $C8, $A1, $DC, $2A, $75, $6C, $98, $CB, $68, $91, $D9, $37, $72, $A4, $9F, $21, $0B, $14, $1B, $A0, $D8, $BD, $4E, $54, $55 ), {D1}
    ( $19, $20, $DF, $B6, $9E, $8C, $B0, $89, $E3, $D2, $6F, $91, $08, $67, $2F, $60, $60, $B4, $EA, $C5, $99, $72, $1E, $39, $1C, $CB, $92, $9C, $50, $6C, $41, $EA, $81, $CC, $08, $FE, $B6, $04, $F6, $19, $C1, $C8, $D5, $DF, $6C, $BC, $C4, $55 ), {D2}
    ( $35, $C7, $CC, $74, $6D, $D7, $DC, $67, $72, $68, $9E, $5A, $D8, $62, $22, $AD, $02, $50, $3B, $1D, $3C, $65, $99, $66, $65, $22, $8A, $8B, $C8, $0D, $5D, $EC, $93, $3A, $5F, $4C, $43, $37, $95, $D7, $DE, $7E, $A8, $54, $DF, $B0, $D1, $D3 ), {D3}
    ( $BC, $6A, $7D, $3A, $F0, $1A, $17, $30, $11, $EC, $57, $6F, $1E, $A7, $25, $03, $33, $9F, $8E, $E6, $AA, $EF, $3C, $50, $96, $C9, $AF, $5B, $E5, $E7, $FE, $9F, $EE, $2C, $0B, $29, $1D, $F4, $54, $BC, $07, $26, $EB, $46, $CF, $F7, $3B, $B3 ), {D4}
    ( $C2, $31, $A2, $19, $FD, $DE, $51, $80, $99, $3F, $FD, $58, $D5, $BE, $C5, $1D, $1A, $C5, $82, $83, $08, $5C, $70, $2B, $39, $D7, $78, $A7, $38, $BC, $21, $0A, $0B, $99, $AC, $2D, $CB, $B2, $D1, $0F, $42, $A8, $98, $BE, $B0, $F1, $87, $27 ), {D5}
    ( $BB, $78, $93, $91, $1F, $BF, $97, $FE, $7D, $17, $E5, $62, $21, $F9, $58, $E4, $F5, $94, $2F, $4F, $9F, $3F, $B1, $99, $80, $E8, $34, $42, $FA, $DC, $3C, $D9, $8F, $F4, $E4, $23, $3B, $95, $8B, $A9, $1E, $AB, $E2, $03, $B0, $FF, $96, $63 ), {D6}
    ( $93, $4E, $EB, $3C, $D6, $B4, $BE, $2E, $86, $00, $38, $06, $F9, $0F, $6B, $04, $ED, $B9, $F6, $6C, $68, $6A, $9E, $8A, $46, $DC, $85, $A0, $48, $4B, $18, $32, $E7, $A5, $F6, $FC, $61, $22, $6C, $7C, $0B, $E0, $0E, $95, $9D, $16, $5C, $00 ), {D7}
    ( $4F, $15, $24, $7D, $A8, $9D, $05, $9A, $6A, $D0, $66, $28, $89, $38, $0A, $5E, $9A, $61, $C2, $76, $40, $D6, $5F, $A0, $B3, $14, $23, $75, $63, $25, $8E, $D6, $A2, $9B, $D5, $92, $F8, $0D, $71, $A3, $EC, $CA, $6D, $5C, $DF, $84, $A1, $CA ), {D8}
    ( $C3, $E2, $9A, $F0, $E7, $1C, $2A, $92, $B8, $1E, $F1, $38, $1A, $92, $E0, $7B, $4D, $A2, $23, $B9, $38, $3F, $7A, $21, $57, $46, $66, $F3, $B2, $40, $A5, $5A, $6C, $4F, $6A, $EF, $C4, $22, $F2, $AF, $BB, $6E, $1D, $84, $6A, $71, $03, $DF ), {D9}
    ( $F6, $A5, $71, $0B, $69, $12, $EB, $36, $AA, $CF, $4A, $BB, $8A, $2A, $A3, $A4, $4C, $9E, $53, $39, $EC, $1B, $6B, $EE, $77, $09, $82, $B9, $41, $14, $01, $C9, $40, $3F, $F5, $6A, $72, $60, $D4, $1A, $F8, $59, $4A, $AA, $D6, $25, $23, $94 ), {DA}
    ( $04, $42, $FE, $26, $98, $5C, $ED, $E8, $5B, $66, $84, $C9, $64, $CE, $D8, $24, $D4, $84, $4B, $78, $F9, $BC, $F0, $59, $8C, $DC, $00, $0D, $45, $15, $D7, $7F, $C6, $B2, $D1, $03, $72, $82, $56, $E9, $0E, $9F, $40, $7B, $DB, $94, $DF, $6E ), {DB}
    ( $6E, $1F, $85, $02, $C5, $2D, $52, $E4, $BE, $90, $0F, $0D, $CF, $53, $BE, $64, $67, $BA, $65, $25, $88, $5A, $C9, $B3, $4A, $8D, $00, $12, $22, $2A, $13, $6D, $0B, $6F, $C3, $65, $D5, $97, $01, $2F, $58, $96, $BE, $FF, $C2, $70, $F2, $13 ), {DC}
    ( $24, $6A, $B2, $0F, $F6, $7F, $89, $E8, $48, $36, $23, $72, $F2, $D2, $97, $E0, $E8, $D5, $6E, $7C, $8E, $85, $B0, $3A, $2D, $42, $55, $74, $9D, $1B, $36, $FE, $CD, $C3, $46, $15, $0D, $99, $84, $F1, $EB, $12, $62, $6A, $10, $3B, $DF, $66 ), {DD}
    ( $B9, $24, $E6, $86, $7B, $7F, $09, $0D, $28, $D8, $39, $F5, $2E, $A9, $7C, $D9, $C2, $2E, $D1, $0B, $42, $53, $24, $37, $52, $9D, $76, $B4, $51, $AF, $BD, $A1, $11, $75, $81, $44, $98, $2E, $B3, $00, $82, $54, $73, $BC, $10, $98, $1E, $28 ), {DE}
    ( $F1, $86, $B0, $5C, $E2, $9E, $64, $0C, $74, $C0, $B7, $D7, $B6, $7A, $F8, $3C, $4E, $8F, $B6, $2E, $2C, $02, $71, $BD, $1C, $96, $66, $A1, $59, $EE, $1B, $B4, $EC, $4F, $E9, $CC, $CD, $5D, $39, $74, $F5, $96, $D0, $4C, $AF, $33, $38, $59 ), {DF}
    ( $9C, $AF, $23, $7E, $57, $FE, $F0, $48, $A3, $F9, $0C, $0A, $5B, $1D, $87, $A7, $1B, $8A, $DC, $2B, $59, $FC, $C2, $6C, $CD, $31, $41, $63, $18, $82, $05, $E9, $C5, $6D, $F1, $1B, $DF, $42, $55, $30, $70, $28, $93, $F9, $88, $64, $F4, $38 ), {E0}
    ( $DD, $03, $F9, $DA, $32, $61, $37, $A5, $80, $C9, $8E, $7D, $BF, $8D, $2C, $2B, $66, $31, $13, $CA, $08, $26, $40, $EF, $51, $AA, $D8, $15, $80, $06, $40, $97, $C7, $A8, $7B, $F4, $73, $DD, $D2, $88, $59, $49, $9C, $C2, $AC, $13, $C6, $84 ), {E1}
    ( $DE, $95, $38, $54, $40, $BC, $15, $B7, $7D, $73, $1D, $DC, $51, $6F, $12, $53, $66, $F5, $45, $61, $33, $61, $2D, $EE, $BA, $4B, $E6, $A8, $A1, $CC, $28, $6F, $D1, $96, $FD, $81, $3D, $8C, $BF, $1B, $9A, $E2, $71, $DF, $91, $20, $69, $0A ), {E2}
    ( $1D, $13, $DE, $6C, $3B, $64, $29, $29, $E2, $EB, $60, $9A, $73, $EE, $B7, $03, $7F, $59, $C4, $DC, $0E, $80, $34, $DB, $7F, $3E, $F2, $2A, $01, $AA, $92, $23, $C2, $5A, $59, $02, $FC, $B2, $25, $FF, $41, $47, $BD, $81, $BC, $BD, $5E, $A0 ), {E3}
    ( $A8, $79, $59, $3C, $C2, $E1, $F2, $46, $7A, $06, $72, $68, $38, $11, $2E, $43, $6C, $69, $C1, $D8, $EF, $F1, $8B, $D8, $96, $B5, $6B, $D1, $68, $86, $77, $B3, $32, $3D, $1B, $CD, $0B, $95, $67, $53, $4A, $6A, $0D, $50, $EC, $FF, $7C, $DB ), {E4}
    ( $81, $BD, $33, $01, $80, $85, $FA, $8D, $7A, $4F, $E1, $71, $E2, $CD, $63, $C2, $E4, $90, $D8, $3D, $56, $CA, $2A, $61, $FD, $82, $8E, $B3, $C4, $04, $56, $30, $A0, $A2, $5B, $91, $CB, $A5, $42, $23, $F8, $8D, $10, $01, $D4, $C0, $D8, $5B ), {E5}
    ( $BD, $C6, $85, $73, $95, $AF, $5F, $34, $1E, $CF, $EB, $99, $D6, $8A, $43, $74, $23, $13, $DC, $16, $85, $84, $8B, $B2, $03, $A9, $D3, $82, $8D, $4E, $9D, $7A, $EB, $4A, $3D, $26, $4C, $9A, $BC, $73, $29, $40, $58, $7A, $2A, $2E, $46, $C3 ), {E6}
    ( $DB, $0F, $0E, $54, $9F, $A7, $3B, $BE, $7A, $2C, $4D, $7F, $AF, $68, $DB, $91, $54, $90, $CB, $91, $51, $EA, $CD, $09, $10, $9F, $2B, $D7, $8E, $E1, $22, $AF, $85, $26, $C3, $BA, $FE, $66, $FB, $1C, $7F, $52, $0E, $68, $E5, $43, $36, $1E ), {E7}
    ( $75, $CE, $7A, $8E, $92, $B0, $A8, $E5, $AF, $BB, $A4, $28, $5B, $62, $80, $EF, $04, $65, $25, $D2, $2E, $77, $69, $08, $A0, $48, $AB, $EE, $64, $2D, $15, $69, $49, $BF, $BC, $94, $36, $DC, $92, $B2, $63, $29, $69, $80, $50, $DA, $74, $B0 ), {E8}
    ( $F2, $9D, $6D, $3E, $F3, $86, $1A, $64, $A5, $30, $E3, $97, $7C, $72, $FE, $02, $59, $2B, $7D, $A6, $0D, $8F, $25, $5E, $A6, $CA, $C3, $CA, $B1, $7D, $72, $3D, $11, $58, $74, $19, $5F, $57, $98, $86, $D1, $88, $25, $B5, $3C, $86, $CA, $DC ), {E9}
    ( $3C, $A1, $D3, $B1, $D4, $C5, $8D, $54, $95, $00, $A7, $56, $0D, $4B, $4B, $92, $97, $D7, $5A, $2B, $24, $93, $81, $64, $B4, $AF, $EE, $79, $9B, $84, $78, $39, $7C, $51, $0C, $F7, $37, $FD, $2A, $8B, $4C, $FD, $8F, $CC, $48, $9D, $13, $84 ), {EA}
    ( $56, $59, $2B, $35, $AE, $D0, $F3, $5D, $33, $3A, $5A, $2C, $0E, $C3, $28, $50, $A8, $6A, $C3, $37, $0E, $4E, $3A, $66, $64, $C3, $1C, $C6, $1A, $B2, $A2, $30, $62, $22, $19, $27, $F9, $CC, $B1, $2D, $BE, $9D, $01, $11, $9C, $9B, $B8, $17 ), {EB}
    ( $A3, $C1, $48, $A4, $F1, $98, $20, $5B, $C7, $5A, $E8, $CC, $A9, $C1, $DC, $9C, $89, $CE, $4A, $67, $DC, $94, $F2, $44, $13, $1E, $19, $FF, $59, $02, $FF, $E0, $80, $DF, $9E, $27, $D6, $AB, $56, $DA, $08, $B5, $29, $44, $71, $D5, $E9, $0A ), {EC}
    ( $E9, $EB, $E4, $5F, $23, $CF, $7F, $2E, $40, $EC, $FB, $3F, $45, $51, $C2, $5C, $7F, $52, $FF, $9C, $8D, $6C, $28, $C7, $51, $8C, $69, $AD, $D8, $B6, $8A, $1C, $68, $F0, $04, $2B, $00, $45, $40, $AA, $4C, $29, $82, $FE, $6C, $7A, $59, $9E ), {ED}
    ( $7B, $8F, $04, $61, $FF, $40, $8E, $01, $88, $34, $B5, $B2, $A1, $4A, $97, $7D, $3A, $6D, $A3, $7B, $A5, $8D, $DB, $7D, $52, $A9, $32, $38, $E0, $92, $A9, $07, $81, $92, $CF, $83, $02, $87, $E7, $15, $9C, $79, $25, $ED, $43, $56, $FA, $69 ), {EE}
    ( $96, $E4, $F0, $94, $8C, $0E, $75, $45, $E7, $1F, $91, $F6, $4E, $6D, $3E, $82, $6A, $C1, $52, $AB, $10, $62, $6D, $5B, $61, $A2, $A8, $A8, $0A, $3D, $37, $DA, $11, $BA, $9A, $D6, $FE, $A8, $1A, $D0, $56, $82, $AB, $F6, $B1, $1E, $D7, $F6 ), {EF}
    ( $88, $BB, $59, $F0, $00, $CB, $57, $FF, $9C, $1E, $C9, $DA, $89, $08, $2F, $F6, $8B, $E3, $80, $BF, $3D, $5C, $57, $4F, $E1, $75, $F9, $AD, $13, $CE, $59, $66, $6D, $70, $4B, $69, $38, $FF, $14, $11, $0C, $7D, $17, $12, $C5, $88, $FB, $FE ), {F0}
    ( $DD, $B1, $0B, $17, $26, $41, $7D, $66, $8F, $49, $8B, $11, $59, $F0, $D2, $27, $D5, $DE, $F1, $ED, $6A, $7A, $EE, $E3, $4D, $42, $DC, $52, $84, $9F, $BD, $6F, $47, $8B, $78, $7A, $CA, $48, $F6, $F8, $DF, $3D, $B7, $CF, $9D, $31, $11, $A6 ), {F1}
    ( $9E, $A4, $F8, $10, $15, $36, $04, $98, $62, $5B, $FA, $A2, $B0, $D3, $15, $5E, $E7, $B6, $74, $5C, $76, $30, $B2, $B6, $C5, $7C, $B2, $B7, $78, $A6, $DB, $7C, $27, $2A, $32, $5B, $C4, $5E, $2A, $79, $A9, $48, $7E, $37, $D5, $9E, $30, $7C ), {F2}
    ( $78, $DA, $8E, $F6, $64, $F8, $06, $4F, $8E, $4C, $58, $17, $D5, $8E, $F2, $5C, $99, $16, $3F, $99, $13, $03, $0E, $A7, $32, $8A, $CC, $BC, $C6, $1D, $3B, $4D, $DD, $0E, $50, $CB, $33, $47, $77, $5A, $94, $B5, $DC, $3F, $70, $E0, $07, $F3 ), {F3}
    ( $8A, $99, $C2, $07, $FF, $A2, $E7, $0C, $AD, $BD, $2A, $53, $09, $73, $48, $13, $26, $AE, $D1, $9D, $2C, $A2, $EE, $51, $1D, $A3, $A4, $EE, $94, $AA, $DE, $FC, $F1, $DB, $4B, $D8, $26, $69, $28, $31, $3B, $40, $57, $6B, $82, $64, $DE, $FA ), {F4}
    ( $B1, $70, $C0, $77, $89, $20, $99, $6B, $BE, $24, $6E, $E0, $AE, $69, $51, $8C, $16, $2B, $98, $5F, $88, $F0, $3A, $6A, $06, $89, $0E, $88, $55, $94, $6E, $E7, $1C, $D8, $DF, $AE, $60, $79, $CE, $F2, $D1, $66, $DC, $E6, $00, $E4, $03, $D2 ), {F5}
    ( $2A, $C3, $13, $BB, $45, $32, $F0, $23, $8D, $1F, $9B, $94, $95, $60, $8A, $43, $23, $51, $30, $F4, $9D, $28, $46, $4A, $0C, $ED, $E0, $9C, $75, $82, $CA, $16, $85, $12, $37, $FB, $F2, $A8, $5B, $FD, $F0, $EF, $DF, $C7, $56, $4D, $2C, $49 ), {F6}
    ( $D8, $57, $DF, $DF, $47, $F0, $0A, $14, $BE, $8A, $9E, $37, $AE, $0B, $31, $1F, $A1, $0A, $6B, $E2, $64, $B7, $47, $88, $5D, $A1, $C4, $98, $50, $4A, $65, $E8, $D7, $B2, $69, $33, $DE, $04, $5C, $48, $CE, $1C, $01, $26, $08, $B8, $0E, $AE ), {F7}
    ( $AA, $27, $ED, $44, $96, $52, $02, $C2, $AB, $A5, $6D, $B9, $F4, $8B, $D4, $0C, $DF, $A0, $F8, $5C, $F2, $F5, $84, $50, $BE, $50, $03, $73, $BC, $C7, $D6, $3F, $EC, $32, $8C, $79, $76, $B1, $A8, $8D, $DD, $73, $94, $13, $12, $64, $9B, $A5 ), {F8}
    ( $28, $14, $CF, $99, $CA, $31, $D7, $9A, $4C, $CA, $E8, $18, $A6, $01, $99, $F8, $5E, $4E, $03, $9D, $B5, $4F, $F0, $1A, $56, $B8, $14, $FE, $20, $38, $A1, $FA, $70, $A1, $AF, $FD, $C2, $CF, $4B, $53, $34, $C0, $2D, $AE, $0E, $F5, $35, $B8 ), {F9}
    ( $3A, $60, $1C, $12, $A2, $8E, $19, $85, $2A, $B6, $EB, $65, $30, $80, $31, $0A, $AF, $09, $9A, $08, $9F, $28, $52, $CB, $12, $24, $2A, $85, $EE, $64, $0F, $00, $79, $E3, $66, $4D, $CE, $2E, $83, $89, $53, $BB, $1B, $DA, $00, $0A, $98, $96 ), {FA}
    ( $07, $89, $C4, $27, $21, $6B, $0A, $2A, $6B, $F6, $A8, $CC, $28, $E2, $18, $28, $84, $B2, $B3, $92, $54, $4A, $CC, $93, $02, $DB, $DF, $BB, $34, $A4, $21, $33, $A6, $FB, $EE, $A5, $09, $8F, $AE, $41, $C6, $EB, $87, $6F, $23, $B8, $8E, $52 ), {FB}
    ( $7D, $9E, $ED, $97, $78, $3A, $45, $9D, $14, $19, $C0, $96, $55, $24, $8A, $2F, $96, $E4, $BE, $AB, $7A, $C3, $09, $0A, $A5, $3D, $28, $0B, $FB, $AB, $7A, $AD, $87, $B0, $C9, $C0, $20, $25, $70, $C7, $BB, $30, $84, $D6, $85, $92, $3F, $24 ), {FC}
    ( $89, $8C, $68, $BF, $0F, $0D, $6B, $D6, $74, $72, $0B, $48, $F1, $5C, $B0, $EC, $41, $69, $0B, $98, $5E, $B0, $14, $B7, $76, $C0, $FD, $02, $74, $8C, $B9, $EB, $38, $4E, $F5, $D7, $98, $C1, $A9, $89, $B5, $91, $02, $DF, $F8, $EA, $FD, $AA ), {FD}
    ( $C0, $A7, $78, $CF, $0C, $5A, $A1, $54, $FE, $60, $FF, $7A, $8E, $6F, $D0, $F6, $3D, $32, $B2, $51, $CD, $AE, $2A, $77, $5F, $58, $82, $E8, $E6, $C0, $2E, $F7, $A3, $CE, $20, $B8, $88, $7B, $69, $8B, $18, $0D, $EE, $58, $4F, $5B, $3E, $0D ), {FE}
    ( $08, $91, $65, $D3, $32, $2B, $AD, $34, $30, $FF, $DF, $EC, $E7, $9D, $8F, $2D, $60, $FE, $58, $A8, $F7, $50, $CD, $58, $E7, $AE, $7E, $FA, $69, $BF, $12, $92, $EB, $99, $C0, $04, $D4, $4B, $41, $77, $0C, $C0, $7D, $FC, $DA, $13, $FA, $9E )  {FF}
  );

const
  TABLE_SBX: array[0..255] of TDigestBuf =
  (
    ( $98, $98, $45, $D6, $D7, $8F, $01, $B3, $62, $1D, $18, $6D, $80, $B1, $C9, $68, $1E, $83, $86, $84, $E8, $F2, $87, $10, $B3, $69, $4B, $46, $91, $03, $43, $0A, $34, $33, $2F, $ED, $7E, $D9, $99, $F2, $44, $5F, $98, $9F, $06, $77, $33, $5D ), {00}
    ( $8E, $CF, $52, $75, $99, $47, $1E, $A5, $AF, $83, $98, $C4, $6F, $34, $AB, $14, $60, $E8, $FC, $CB, $CD, $34, $A1, $20, $52, $0A, $6E, $B7, $14, $C7, $98, $BC, $9C, $9E, $15, $FC, $B6, $DC, $EB, $FC, $5C, $1C, $06, $5C, $59, $E2, $7A, $DB ), {01}
    ( $33, $2C, $AD, $BF, $50, $F1, $06, $06, $B8, $51, $1D, $B9, $78, $57, $85, $66, $C9, $D3, $4F, $BC, $EE, $F7, $6C, $34, $D2, $09, $44, $EB, $8C, $76, $74, $9C, $83, $09, $31, $D6, $49, $A2, $3A, $42, $B7, $EB, $00, $0E, $16, $A3, $18, $1D ), {02}
    ( $C3, $51, $39, $9F, $19, $36, $2A, $BE, $EF, $5B, $82, $A8, $7B, $6B, $69, $A4, $6C, $8D, $F7, $89, $16, $52, $32, $DD, $B4, $04, $3D, $CC, $68, $DC, $9B, $C0, $04, $6D, $42, $9E, $67, $A4, $B5, $E1, $0B, $ED, $30, $C7, $24, $22, $45, $96 ), {03}
    ( $05, $21, $19, $8B, $9D, $7A, $72, $CC, $5B, $2A, $4F, $1E, $4F, $F3, $E8, $E2, $F0, $10, $60, $52, $37, $35, $76, $BB, $F7, $4C, $59, $D9, $1B, $44, $CD, $B9, $08, $7A, $4B, $C4, $47, $EF, $94, $8B, $D7, $F7, $CE, $87, $83, $B5, $56, $C3 ), {04}
    ( $22, $1B, $F9, $B7, $E9, $B9, $4D, $30, $50, $64, $C6, $31, $42, $E5, $DE, $FE, $FB, $77, $78, $73, $4D, $28, $08, $F5, $FE, $13, $C6, $B0, $63, $BC, $5D, $34, $D6, $6E, $60, $6E, $A4, $0C, $9D, $46, $9D, $8F, $69, $6F, $BE, $45, $BE, $37 ), {05}
    ( $B5, $39, $41, $8A, $FE, $C0, $A9, $2C, $54, $48, $E6, $BF, $A9, $30, $57, $6D, $27, $58, $3D, $4C, $1F, $41, $F2, $EB, $89, $4E, $95, $87, $67, $69, $D3, $B2, $A5, $1C, $7A, $BF, $B0, $54, $E2, $84, $A9, $6F, $33, $0B, $86, $E4, $78, $30 ), {06}
    ( $2C, $A6, $0C, $D0, $CB, $84, $7E, $8A, $AC, $90, $39, $33, $75, $4C, $1E, $43, $04, $70, $7E, $19, $BB, $61, $EC, $1D, $90, $68, $F1, $9B, $1E, $F4, $2B, $43, $D2, $82, $F3, $B0, $D7, $68, $B0, $87, $F7, $02, $83, $A4, $EC, $2F, $42, $B9 ), {07}
    ( $C6, $EB, $71, $92, $76, $73, $B4, $D6, $97, $14, $57, $20, $03, $3F, $10, $EB, $CF, $91, $E0, $9D, $30, $EB, $1F, $0F, $F8, $38, $07, $30, $0B, $C1, $40, $12, $9D, $49, $B2, $DE, $5C, $84, $63, $38, $97, $86, $BC, $E0, $F9, $0A, $87, $0D ), {08}
    ( $01, $F5, $4E, $20, $CA, $63, $1A, $E3, $A4, $04, $F9, $7E, $F5, $F9, $5B, $3B, $B2, $28, $94, $9E, $C4, $1E, $56, $CF, $97, $5B, $2D, $88, $25, $36, $D5, $DB, $92, $D2, $3B, $1B, $33, $79, $2B, $64, $B8, $B3, $D6, $9C, $B2, $99, $1D, $E4 ), {09}
    ( $7A, $26, $76, $03, $2A, $B8, $9E, $45, $47, $27, $02, $65, $48, $F0, $CF, $59, $56, $EF, $4C, $46, $50, $18, $6F, $79, $34, $74, $BF, $31, $8F, $4A, $52, $84, $4F, $0C, $1C, $CE, $CE, $32, $80, $FF, $CD, $C9, $6F, $93, $CF, $DE, $68, $3D ), {0A}
    ( $62, $E8, $95, $12, $24, $EF, $04, $D7, $87, $75, $D8, $3B, $BB, $03, $4F, $3C, $C1, $CD, $56, $D1, $71, $78, $6B, $37, $84, $F3, $EB, $D4, $8D, $C6, $80, $35, $00, $6B, $72, $E4, $84, $75, $4B, $1F, $29, $89, $74, $2F, $8F, $7A, $95, $B5 ), {0B}
    ( $96, $66, $FF, $1E, $22, $AB, $8B, $F0, $C6, $FA, $BB, $52, $4A, $59, $BB, $74, $54, $26, $1C, $BB, $80, $5B, $45, $81, $C5, $79, $A1, $4D, $87, $D1, $18, $5A, $D5, $20, $87, $12, $A1, $6C, $58, $1A, $E4, $00, $F2, $A0, $9F, $3A, $6D, $E2 ), {0C}
    ( $16, $64, $4C, $40, $AF, $FA, $DA, $D1, $40, $87, $74, $13, $AE, $D8, $5C, $B3, $5A, $CE, $06, $E5, $BD, $47, $B2, $2C, $27, $55, $DB, $69, $1F, $A0, $57, $2E, $CE, $FC, $A4, $E7, $20, $4B, $FA, $58, $59, $FC, $CF, $3D, $F7, $14, $C7, $0F ), {0D}
    ( $4D, $30, $93, $0A, $BD, $B0, $A0, $59, $1B, $42, $0F, $A7, $77, $EB, $F5, $8C, $12, $9E, $25, $E9, $04, $81, $7F, $75, $D3, $F5, $AD, $21, $B7, $3C, $36, $F9, $2F, $37, $90, $58, $BC, $6E, $8F, $2A, $02, $6D, $EF, $05, $3C, $68, $46, $BC ), {0E}
    ( $8D, $3A, $F5, $06, $89, $82, $5F, $0B, $31, $D4, $52, $FE, $3E, $47, $8D, $95, $CB, $7D, $F6, $CC, $67, $43, $60, $19, $92, $4F, $00, $5E, $A4, $E2, $E6, $E2, $F2, $31, $95, $69, $A7, $34, $91, $E9, $80, $4B, $FC, $D5, $D9, $18, $B5, $23 ), {0F}
    ( $1D, $43, $9F, $D8, $BB, $52, $A8, $D8, $6E, $15, $A4, $C2, $4B, $92, $1C, $60, $01, $D9, $E7, $B3, $46, $A9, $DA, $2F, $8C, $9B, $EC, $B5, $43, $28, $34, $6C, $82, $4E, $CC, $6D, $15, $FD, $6C, $E4, $03, $78, $93, $6D, $C7, $12, $DE, $6A ), {10}
    ( $4F, $3E, $3C, $CD, $C9, $99, $5B, $EF, $CE, $00, $77, $32, $B0, $B3, $C3, $F3, $AC, $AE, $98, $98, $6C, $D9, $EF, $E9, $35, $B1, $01, $CE, $E5, $9B, $D6, $64, $B8, $6F, $F7, $4F, $0A, $28, $DB, $61, $8A, $9C, $01, $B6, $E7, $B6, $50, $D4 ), {11}
    ( $F7, $46, $20, $80, $F2, $98, $75, $C4, $27, $F0, $E5, $B1, $43, $37, $E2, $24, $08, $93, $85, $07, $DE, $7C, $40, $48, $68, $0D, $0C, $DE, $A5, $52, $10, $41, $AD, $F4, $D7, $3B, $89, $AF, $C7, $AF, $FB, $45, $6C, $7B, $EE, $2D, $81, $99 ), {12}
    ( $F1, $E6, $7C, $38, $9F, $45, $86, $F2, $5D, $6C, $4C, $6C, $21, $1E, $A4, $21, $5E, $55, $93, $99, $51, $DB, $BF, $33, $85, $85, $03, $06, $0A, $21, $47, $79, $95, $96, $E5, $8B, $AA, $01, $9E, $C9, $D5, $FB, $3A, $23, $3D, $DB, $1E, $BE ), {13}
    ( $F3, $C8, $2B, $68, $B7, $96, $3C, $B4, $DE, $F9, $F7, $05, $59, $88, $C7, $E0, $1F, $BC, $7C, $F2, $F3, $0C, $42, $E2, $A4, $B0, $37, $6D, $5D, $C2, $49, $E5, $01, $01, $64, $0B, $DE, $9D, $09, $77, $58, $DE, $97, $47, $0C, $D0, $84, $49 ), {14}
    ( $E7, $65, $B9, $0F, $0D, $B4, $83, $74, $7E, $AF, $5F, $96, $AA, $FA, $42, $08, $29, $9B, $37, $B6, $1A, $29, $98, $88, $F2, $7C, $6D, $FE, $71, $56, $24, $D3, $06, $05, $89, $82, $5A, $40, $76, $89, $73, $5C, $37, $26, $A1, $24, $6A, $2C ), {15}
    ( $32, $EC, $98, $48, $71, $76, $17, $68, $3A, $37, $90, $94, $EE, $E2, $34, $2A, $AF, $A2, $EE, $70, $0E, $9C, $9F, $97, $04, $41, $D5, $2B, $DA, $14, $DB, $B0, $02, $56, $C2, $50, $07, $1B, $0E, $7F, $F5, $81, $E1, $8F, $63, $96, $20, $0E ), {16}
    ( $FC, $00, $5B, $BE, $BF, $4D, $22, $20, $DC, $81, $1F, $95, $E7, $98, $3A, $2C, $E6, $74, $B7, $3A, $CC, $E9, $89, $FB, $D0, $0E, $88, $2F, $CB, $DA, $B6, $2C, $A1, $EA, $C1, $8F, $24, $74, $D6, $41, $4B, $52, $C4, $BE, $1C, $FD, $7F, $B2 ), {17}
    ( $CC, $FC, $30, $83, $27, $27, $FC, $EA, $DD, $71, $1E, $8D, $12, $78, $E3, $DE, $84, $95, $5A, $E2, $F6, $D8, $6E, $DC, $C6, $60, $8E, $D7, $50, $34, $03, $A2, $53, $2E, $03, $51, $6E, $0D, $A8, $3B, $87, $15, $ED, $3F, $80, $6D, $3F, $56 ), {18}
    ( $34, $5C, $12, $5A, $F9, $ED, $F1, $DA, $89, $AE, $F3, $C6, $24, $0B, $5D, $D6, $D9, $2D, $87, $38, $5B, $67, $38, $6D, $41, $BE, $AC, $A0, $D6, $AC, $22, $95, $F5, $F0, $EE, $A5, $38, $5D, $8B, $2B, $2B, $27, $31, $2D, $F2, $D7, $83, $68 ), {19}
    ( $78, $B6, $D7, $2E, $09, $E9, $F4, $FB, $A5, $FD, $6C, $07, $66, $17, $49, $96, $D1, $BD, $20, $9A, $45, $E7, $59, $57, $4B, $CC, $C4, $38, $88, $4F, $70, $A0, $C4, $FA, $09, $AF, $13, $39, $3C, $6C, $B2, $E3, $21, $F3, $74, $E1, $D5, $A8 ), {1A}
    ( $F9, $3D, $23, $B5, $88, $3B, $F7, $E7, $E0, $4A, $0D, $39, $0A, $97, $95, $1C, $EC, $73, $6A, $E7, $6B, $37, $AA, $46, $63, $D2, $E7, $E5, $B5, $EE, $66, $4B, $EA, $84, $0C, $C9, $29, $7B, $2D, $D6, $0A, $4E, $D1, $E5, $38, $BF, $75, $B7 ), {1B}
    ( $1E, $AD, $C4, $CF, $3B, $D0, $9F, $AF, $96, $35, $B1, $8A, $5F, $3B, $5F, $28, $C8, $21, $CF, $FE, $D6, $EA, $55, $2D, $8B, $34, $AE, $A5, $E1, $F9, $93, $63, $A2, $08, $40, $05, $18, $53, $61, $F8, $98, $3E, $72, $CD, $DE, $AF, $D3, $86 ), {1C}
    ( $D7, $4B, $CC, $84, $52, $6E, $4C, $5D, $B4, $92, $C0, $1A, $10, $D9, $79, $12, $46, $8A, $B2, $17, $AC, $E3, $37, $5D, $AE, $06, $7B, $12, $E0, $7F, $21, $29, $E6, $22, $1A, $C3, $E3, $1F, $C4, $D0, $5E, $E6, $D3, $B5, $1E, $6F, $09, $2F ), {1D}
    ( $C2, $6A, $D5, $16, $98, $A3, $87, $65, $CD, $03, $9E, $42, $20, $05, $04, $69, $78, $07, $74, $C1, $C3, $8A, $4E, $38, $10, $9F, $D8, $03, $23, $4C, $FD, $3A, $EC, $77, $94, $00, $17, $D5, $E9, $65, $62, $F3, $18, $95, $6A, $00, $DF, $DE ), {1E}
    ( $3A, $A9, $ED, $52, $E3, $D1, $4A, $91, $E4, $73, $B2, $5A, $C2, $A6, $8A, $51, $1C, $B3, $10, $47, $91, $DE, $3F, $B4, $29, $B2, $32, $43, $1D, $65, $EF, $B1, $94, $D3, $F9, $EB, $94, $58, $C8, $4A, $2C, $A8, $C0, $0A, $8A, $5E, $5A, $82 ), {1F}
    ( $D6, $CB, $CB, $D9, $29, $B2, $31, $6C, $76, $45, $0A, $F3, $8B, $27, $7C, $AD, $3E, $67, $A4, $09, $61, $1D, $24, $25, $13, $59, $4E, $56, $D5, $6B, $A1, $94, $1A, $63, $AC, $24, $46, $DD, $90, $BA, $6F, $D6, $B8, $FD, $54, $09, $AB, $9A ), {20}
    ( $E0, $12, $56, $F9, $0F, $BE, $93, $FC, $61, $65, $75, $AE, $B2, $DD, $B5, $BE, $06, $47, $04, $87, $15, $CC, $86, $E3, $B1, $9E, $81, $4B, $17, $9C, $00, $EB, $A8, $4D, $B0, $C0, $9E, $4E, $CB, $F0, $E0, $B6, $62, $C3, $25, $B7, $D0, $98 ), {21}
    ( $27, $7C, $91, $47, $20, $A1, $66, $7E, $25, $95, $13, $5D, $E9, $8E, $D4, $FF, $F3, $69, $C8, $FB, $60, $30, $4A, $E7, $19, $29, $EE, $42, $80, $A3, $6E, $A5, $77, $0E, $B3, $11, $7A, $62, $F0, $0B, $94, $3F, $3C, $02, $5B, $80, $E6, $7B ), {22}
    ( $24, $7F, $7E, $2C, $AA, $DF, $C9, $9C, $3F, $49, $7C, $37, $92, $D0, $B1, $D2, $2B, $C5, $47, $D4, $94, $92, $94, $4A, $E7, $18, $DF, $BC, $B9, $0F, $2D, $7E, $AC, $D7, $D3, $A9, $6B, $BB, $5B, $25, $9E, $72, $DB, $3C, $BF, $3C, $38, $9E ), {23}
    ( $AD, $3F, $83, $C9, $46, $23, $26, $12, $23, $44, $DA, $C1, $29, $90, $CD, $CB, $5F, $D5, $D9, $74, $63, $5E, $65, $7B, $46, $63, $73, $18, $46, $60, $33, $2A, $70, $3A, $6F, $23, $C5, $89, $95, $F1, $9C, $0E, $E3, $8B, $9D, $54, $6C, $F7 ), {24}
    ( $51, $99, $E1, $D7, $43, $D9, $84, $9D, $83, $8C, $20, $18, $79, $36, $EE, $ED, $39, $B6, $0D, $12, $E2, $53, $E0, $3E, $DD, $A5, $20, $23, $5F, $22, $4C, $E8, $D0, $75, $3D, $BC, $EE, $1E, $17, $28, $82, $24, $E7, $73, $85, $78, $13, $5F ), {25}
    ( $5E, $8D, $A1, $5C, $A0, $4E, $EB, $B7, $3C, $E7, $E3, $9B, $7A, $F4, $9E, $5D, $B4, $C1, $64, $CA, $BC, $C8, $F7, $8B, $6F, $BD, $48, $7D, $51, $93, $B0, $0F, $7B, $B7, $5D, $17, $C8, $22, $BA, $4B, $F0, $CB, $9A, $B0, $C3, $F2, $65, $06 ), {26}
    ( $18, $5B, $99, $45, $D4, $4B, $FB, $77, $08, $F3, $D5, $49, $2A, $FE, $BF, $32, $85, $4C, $5E, $15, $A1, $91, $48, $A1, $DB, $9C, $17, $C3, $4B, $B9, $B8, $3F, $1E, $44, $E8, $0E, $95, $42, $CA, $A1, $07, $1F, $36, $4E, $41, $FE, $8A, $F0 ), {27}
    ( $5D, $55, $AB, $77, $02, $81, $CA, $4A, $2E, $66, $3E, $E1, $1A, $B5, $1F, $AB, $FD, $34, $B8, $69, $F0, $F5, $3D, $F6, $57, $C3, $B3, $22, $C7, $66, $A8, $F5, $1F, $BA, $6B, $8A, $76, $3E, $13, $D4, $EA, $18, $22, $AD, $2D, $1C, $E8, $05 ), {28}
    ( $57, $0D, $BD, $DD, $DA, $62, $67, $32, $AA, $47, $6F, $EF, $54, $65, $11, $E3, $9B, $64, $12, $93, $EA, $39, $57, $A2, $A1, $65, $AF, $3A, $30, $F0, $11, $57, $74, $4A, $97, $F6, $8D, $C9, $B8, $94, $13, $66, $C2, $A5, $6C, $C5, $F5, $50 ), {29}
    ( $8F, $C1, $AC, $6C, $DE, $3D, $38, $F3, $D2, $ED, $FA, $71, $40, $25, $FB, $F8, $9A, $84, $6E, $F9, $FF, $8E, $17, $03, $39, $36, $92, $AB, $75, $D3, $0F, $ED, $F1, $EC, $78, $7E, $C4, $DE, $F6, $32, $81, $2C, $0C, $67, $B3, $4A, $71, $64 ), {2A}
    ( $0A, $1E, $48, $EC, $79, $19, $29, $1C, $B3, $70, $92, $98, $F1, $F6, $93, $6A, $8A, $3F, $C7, $E1, $4A, $90, $EE, $AF, $62, $00, $93, $AF, $09, $CC, $73, $4D, $B1, $95, $D5, $16, $1C, $0E, $29, $96, $5A, $42, $C5, $B3, $FF, $F7, $1C, $32 ), {2B}
    ( $1B, $C5, $A2, $EF, $DD, $FC, $3D, $AA, $45, $F5, $7B, $D0, $45, $C5, $25, $E8, $5C, $C8, $88, $2B, $98, $02, $29, $B9, $1D, $6B, $80, $4C, $58, $81, $F7, $02, $42, $7D, $26, $F5, $53, $E0, $DD, $C6, $9F, $91, $E5, $A2, $3A, $C6, $26, $97 ), {2C}
    ( $15, $72, $85, $3F, $86, $E4, $25, $EE, $3B, $A1, $FB, $3F, $BC, $0F, $55, $F9, $55, $7A, $02, $E8, $A8, $F8, $63, $30, $93, $C0, $6B, $59, $E7, $D7, $4E, $F7, $97, $89, $3F, $53, $2A, $7E, $A3, $C0, $84, $2B, $C9, $4D, $4D, $FF, $08, $60 ), {2D}
    ( $90, $60, $5D, $F3, $07, $50, $B2, $2B, $99, $88, $FC, $0C, $83, $FC, $58, $CA, $44, $D2, $6B, $67, $32, $DD, $1A, $1F, $ED, $93, $45, $BF, $89, $DF, $37, $31, $4D, $2A, $54, $FA, $3F, $91, $46, $85, $68, $F6, $A8, $12, $FB, $CB, $4E, $79 ), {2E}
    ( $29, $DE, $96, $1C, $B1, $43, $30, $5E, $81, $6E, $D9, $61, $3D, $20, $CC, $A5, $7C, $8F, $68, $A4, $AA, $D4, $27, $8C, $8A, $E4, $06, $7E, $93, $35, $79, $C3, $C8, $8D, $1E, $BE, $88, $50, $EA, $3C, $B0, $33, $39, $E2, $0D, $05, $1F, $69 ), {2F}
    ( $C5, $D3, $8C, $50, $A7, $86, $71, $83, $ED, $17, $47, $E3, $BE, $4A, $9F, $FC, $1D, $37, $9A, $95, $B3, $BA, $5A, $27, $58, $2F, $86, $1B, $4E, $D2, $09, $65, $99, $07, $14, $F1, $FC, $BA, $C2, $CA, $09, $29, $EA, $99, $EF, $D6, $2F, $9C ), {30}
    ( $54, $95, $DD, $63, $D6, $77, $0A, $F6, $BA, $E5, $E4, $2D, $A5, $BB, $A1, $70, $82, $F2, $6C, $77, $9D, $F4, $BD, $49, $CB, $26, $DD, $16, $FB, $A6, $0B, $8A, $FB, $6C, $0D, $7A, $59, $D3, $38, $14, $C1, $79, $08, $84, $A9, $B2, $04, $D9 ), {31}
    ( $23, $C6, $C0, $4B, $D2, $89, $88, $0C, $DB, $B3, $1A, $0B, $F0, $60, $91, $31, $35, $A0, $57, $35, $34, $7E, $CC, $06, $1B, $9D, $E4, $E6, $69, $16, $BB, $BD, $E2, $B5, $68, $48, $77, $29, $14, $9B, $7B, $96, $95, $DD, $6F, $0B, $0C, $AA ), {32}
    ( $C9, $45, $EE, $79, $CE, $68, $74, $2F, $90, $A9, $04, $7F, $5A, $9D, $98, $03, $4B, $50, $BA, $63, $C1, $27, $F0, $14, $2C, $1B, $26, $B4, $AF, $CB, $8D, $7F, $7C, $86, $4F, $BB, $3E, $4D, $E1, $7A, $AE, $01, $24, $BC, $DC, $1A, $25, $2B ), {33}
    ( $BF, $B7, $8A, $28, $28, $78, $B9, $F4, $4C, $06, $76, $7A, $4C, $E0, $7E, $CF, $B1, $61, $97, $7F, $8A, $CF, $67, $CE, $7E, $81, $55, $AC, $B2, $12, $3E, $5B, $DC, $46, $10, $85, $9C, $18, $FD, $2C, $53, $39, $40, $C9, $8B, $F3, $BA, $83 ), {34}
    ( $4E, $5A, $FA, $91, $66, $80, $15, $D4, $60, $62, $9F, $B3, $0D, $70, $4A, $9A, $79, $0D, $D0, $0C, $1E, $C5, $D3, $ED, $18, $A0, $9D, $41, $FE, $57, $90, $70, $7F, $52, $92, $91, $6C, $3A, $0F, $A6, $32, $67, $76, $E4, $E3, $6A, $F2, $FE ), {35}
    ( $AA, $EA, $31, $6E, $F1, $16, $91, $9B, $53, $50, $43, $F5, $F6, $2E, $BE, $76, $47, $65, $23, $01, $48, $3D, $10, $12, $D8, $6F, $8C, $CF, $6A, $BB, $8F, $20, $B0, $CA, $9F, $7F, $2B, $03, $9A, $F7, $E9, $AD, $19, $9B, $FE, $0D, $64, $29 ), {36}
    ( $2A, $FB, $33, $D5, $AB, $70, $AC, $8F, $FD, $57, $81, $64, $22, $06, $FF, $0F, $40, $C0, $E1, $36, $D7, $BF, $C9, $68, $45, $2A, $50, $BE, $DD, $FF, $CB, $9F, $38, $DC, $CB, $9F, $83, $C7, $E4, $5E, $C3, $C1, $F0, $09, $81, $20, $15, $8C ), {37}
    ( $03, $F8, $89, $36, $9E, $2A, $2C, $DC, $94, $8B, $56, $E7, $E4, $A5, $A7, $53, $69, $15, $3C, $6E, $E5, $B8, $43, $63, $94, $8D, $E0, $0E, $7B, $B5, $32, $61, $21, $BE, $11, $BD, $F9, $A1, $96, $FA, $52, $8B, $5B, $C2, $D5, $C1, $D8, $53 ), {38}
    ( $BC, $B0, $DC, $5B, $2B, $4F, $F5, $FA, $8F, $5D, $86, $1F, $FC, $A1, $A5, $22, $2C, $86, $C4, $60, $69, $B5, $3A, $C3, $70, $F9, $41, $C8, $DF, $95, $17, $7B, $5E, $65, $C5, $08, $44, $7D, $1A, $18, $A1, $73, $DE, $A8, $19, $85, $E5, $B6 ), {39}
    ( $E6, $DF, $21, $C7, $59, $48, $AF, $41, $70, $B6, $D2, $75, $74, $B8, $39, $0D, $A7, $81, $0B, $E0, $8D, $B4, $7C, $5F, $22, $1A, $FE, $90, $52, $27, $44, $8C, $EB, $7E, $D0, $40, $C6, $55, $5F, $1C, $2A, $62, $B2, $7F, $1B, $02, $47, $F5 ), {3A}
    ( $58, $77, $6D, $95, $C6, $CD, $FA, $50, $CC, $7D, $79, $79, $08, $19, $84, $BB, $7F, $D6, $EC, $C5, $8C, $F6, $B6, $80, $D7, $E7, $2A, $92, $6C, $C3, $CA, $21, $C7, $51, $DF, $F9, $A6, $F9, $56, $A2, $BF, $21, $7A, $C6, $E4, $34, $16, $08 ), {3B}
    ( $09, $F7, $74, $F6, $97, $24, $D9, $F7, $1A, $AB, $33, $BE, $CB, $DF, $D6, $1E, $AB, $59, $8B, $92, $F8, $4C, $26, $4E, $4C, $E6, $58, $C7, $FF, $83, $5E, $19, $2D, $AD, $93, $C6, $91, $EB, $AA, $76, $34, $2F, $99, $77, $65, $41, $23, $15 ), {3C}
    ( $A8, $70, $49, $7B, $EC, $D2, $52, $6F, $10, $BB, $DB, $AC, $D1, $BF, $BA, $06, $CD, $1C, $4D, $7E, $0D, $A5, $F3, $0D, $E1, $A1, $8B, $57, $78, $37, $85, $6F, $BB, $97, $18, $4D, $DD, $A6, $F5, $9A, $B4, $F1, $AF, $86, $C4, $A1, $F8, $6C ), {3D}
    ( $66, $25, $B6, $0C, $2F, $C3, $DD, $98, $FB, $B4, $D6, $06, $44, $6F, $71, $4F, $D6, $B2, $FB, $56, $0C, $64, $BC, $82, $EF, $82, $FF, $98, $BC, $6F, $9D, $D2, $9B, $2D, $33, $5F, $31, $AB, $B2, $05, $D9, $6A, $B4, $EA, $F1, $6C, $EB, $6B ), {3E}
    ( $C1, $1F, $22, $00, $F3, $0C, $02, $3E, $19, $2E, $A1, $DC, $9A, $FF, $48, $1D, $F9, $FB, $B0, $4A, $C6, $E0, $77, $6E, $B2, $88, $BC, $7B, $F0, $4B, $E1, $67, $91, $2C, $BB, $3A, $96, $0F, $66, $35, $B6, $F8, $57, $B9, $D7, $48, $D1, $F8 ), {3F}
    ( $2D, $F2, $F0, $CB, $30, $4C, $05, $E9, $64, $EB, $4A, $21, $A8, $E6, $50, $BC, $F8, $0F, $46, $23, $84, $A7, $73, $28, $50, $F0, $1F, $9E, $C4, $1B, $A5, $F4, $6C, $AE, $2D, $B5, $30, $46, $73, $99, $0D, $4A, $5C, $22, $53, $A2, $43, $89 ), {40}
    ( $0B, $04, $5A, $DA, $D0, $13, $61, $E1, $A2, $EE, $AE, $93, $3B, $85, $1D, $9C, $E7, $90, $11, $5B, $C9, $09, $2D, $FD, $E0, $FB, $FD, $3B, $2D, $3B, $0D, $D9, $D3, $F8, $AE, $47, $8E, $2B, $CF, $6F, $86, $BD, $42, $3E, $71, $B1, $37, $F4 ), {41}
    ( $BB, $D1, $32, $E1, $D3, $7D, $43, $9A, $E6, $09, $66, $F0, $8A, $EA, $6E, $97, $57, $6B, $24, $FD, $2D, $21, $31, $6A, $AC, $A2, $0A, $D8, $33, $2A, $02, $D7, $10, $B2, $C0, $1D, $26, $ED, $C9, $D5, $4E, $C4, $CC, $1C, $A4, $28, $D7, $55 ), {42}
    ( $DC, $5F, $3F, $E8, $57, $D7, $07, $54, $6F, $E2, $29, $7D, $2C, $22, $0A, $75, $93, $B5, $BF, $91, $FC, $74, $0E, $86, $48, $E0, $BE, $58, $A3, $8D, $91, $18, $E4, $EB, $0E, $81, $64, $14, $7E, $04, $C9, $0F, $A6, $DA, $F5, $ED, $62, $04 ), {43}
    ( $07, $8A, $81, $1B, $10, $E8, $78, $14, $26, $2B, $54, $CE, $FA, $89, $75, $F0, $14, $7F, $ED, $D0, $CB, $3E, $B4, $02, $21, $CD, $5C, $FF, $96, $6E, $71, $88, $FE, $F5, $B7, $C1, $F4, $10, $3D, $98, $00, $9E, $DF, $54, $8C, $57, $F4, $67 ), {44}
    ( $77, $AF, $A7, $FA, $58, $7F, $2B, $10, $88, $3C, $DC, $A2, $5D, $BE, $F6, $C7, $C5, $14, $EA, $22, $E0, $DA, $99, $87, $4D, $70, $5A, $A1, $1C, $C0, $C2, $81, $05, $29, $13, $1A, $3B, $35, $6A, $06, $F1, $E4, $4F, $0D, $10, $B3, $E3, $A3 ), {45}
    ( $89, $4A, $E5, $2B, $62, $53, $E8, $C3, $21, $EA, $D3, $F8, $EA, $CA, $62, $C6, $BC, $6D, $39, $04, $9B, $5F, $58, $96, $87, $61, $A7, $00, $2B, $B8, $84, $28, $54, $26, $AB, $66, $8B, $87, $35, $15, $50, $E1, $FA, $5D, $5F, $8D, $52, $18 ), {46}
    ( $FB, $88, $38, $8F, $90, $05, $E6, $1B, $C8, $0F, $16, $40, $6B, $A4, $F1, $A6, $67, $92, $1F, $AC, $C2, $07, $B5, $21, $9C, $C2, $CA, $E1, $5E, $FA, $A2, $82, $87, $A8, $A6, $9D, $7D, $EE, $AC, $43, $01, $3B, $F4, $AA, $21, $03, $A8, $5E ), {47}
    ( $DA, $44, $C5, $97, $F7, $8C, $3F, $A7, $24, $B9, $72, $45, $DE, $EC, $4C, $4A, $E8, $A3, $FE, $32, $65, $7B, $83, $9E, $32, $14, $F8, $19, $BB, $E1, $1A, $C9, $5D, $06, $B5, $65, $A3, $47, $39, $A8, $3B, $F2, $8C, $2B, $47, $11, $FE, $F9 ), {48}
    ( $DE, $D6, $E6, $B8, $0A, $A2, $33, $4E, $C9, $8F, $DF, $DE, $7D, $55, $66, $27, $DA, $33, $53, $5F, $A9, $80, $A2, $4F, $5E, $25, $94, $F6, $3B, $09, $8A, $7A, $C6, $AC, $FB, $E8, $02, $F0, $DC, $C5, $21, $EC, $10, $DC, $5D, $5A, $F7, $CA ), {49}
    ( $9D, $7E, $2D, $B3, $03, $C4, $9B, $22, $37, $A5, $07, $E0, $60, $C4, $88, $34, $95, $80, $EB, $10, $2B, $AB, $9B, $C1, $43, $B9, $3E, $15, $40, $D4, $88, $F2, $56, $19, $44, $F4, $99, $3C, $36, $67, $6A, $94, $8D, $1B, $9C, $01, $D9, $AB ), {4A}
    ( $3E, $6E, $1A, $CE, $FC, $54, $AA, $93, $E7, $C9, $F1, $CA, $CA, $E1, $63, $DF, $89, $0B, $3E, $53, $6D, $66, $D0, $15, $8E, $6E, $C7, $25, $C5, $AF, $94, $E1, $FA, $80, $2B, $03, $AF, $17, $9C, $17, $78, $44, $35, $24, $35, $9B, $7C, $A4 ), {4B}
    ( $19, $89, $03, $90, $38, $49, $09, $3C, $F7, $DE, $AB, $97, $73, $B9, $CE, $35, $9C, $40, $58, $28, $A0, $55, $DC, $C0, $D1, $D9, $E8, $1D, $42, $64, $BA, $BE, $DA, $4F, $DA, $68, $4D, $36, $3F, $7E, $66, $A3, $2E, $61, $B6, $A6, $8C, $36 ), {4C}
    ( $EF, $53, $84, $A1, $84, $A8, $18, $BC, $EE, $AC, $84, $82, $1D, $A3, $A6, $DA, $C4, $02, $31, $43, $5A, $26, $71, $40, $EB, $EB, $D2, $81, $5B, $3D, $AD, $D6, $F8, $1F, $A2, $D4, $80, $45, $18, $5B, $B1, $60, $77, $71, $72, $58, $C3, $62 ), {4D}
    ( $40, $CA, $6A, $DE, $7F, $25, $A4, $F8, $D4, $9A, $25, $2F, $7F, $95, $F4, $73, $BB, $48, $B5, $C9, $75, $95, $9A, $13, $79, $83, $13, $7C, $12, $82, $6A, $01, $B6, $DF, $47, $5C, $C0, $AD, $16, $07, $7E, $09, $48, $98, $27, $CD, $2A, $D6 ), {4E}
    ( $6A, $0F, $E7, $86, $BA, $74, $C4, $D2, $28, $EC, $C8, $55, $31, $C3, $44, $80, $7D, $C9, $32, $A8, $7E, $06, $CF, $8E, $3E, $B6, $B7, $B2, $2E, $CA, $14, $E6, $09, $A4, $37, $07, $DC, $CE, $64, $E2, $6E, $A4, $12, $70, $7B, $32, $FA, $7D ), {4F}
    ( $FA, $94, $24, $3D, $DB, $26, $9C, $57, $BD, $3E, $8F, $7C, $0F, $FB, $83, $B7, $B7, $E0, $3A, $20, $DB, $44, $92, $3D, $2A, $BF, $51, $DA, $18, $74, $5C, $52, $4E, $B6, $71, $6A, $B2, $3B, $E0, $0C, $17, $B5, $E0, $4B, $ED, $60, $C1, $1F ), {50}
    ( $E5, $3C, $4B, $EB, $CC, $F6, $9A, $85, $0C, $F4, $83, $D2, $33, $54, $E6, $7D, $F7, $E6, $50, $A3, $99, $1B, $2C, $29, $F1, $71, $70, $3E, $05, $45, $1C, $F6, $6B, $35, $45, $62, $03, $38, $0C, $21, $3A, $E7, $BB, $00, $33, $2C, $CF, $4D ), {51}
    ( $70, $27, $1D, $34, $23, $91, $D3, $4B, $D6, $41, $5E, $F2, $9C, $B6, $64, $A7, $80, $06, $65, $40, $0B, $9E, $3B, $A4, $EA, $05, $7E, $94, $D9, $E8, $C5, $97, $A7, $2B, $FF, $3C, $39, $FC, $8E, $8A, $5F, $13, $78, $97, $05, $0E, $BD, $43 ), {52}
    ( $26, $C7, $97, $43, $61, $D6, $96, $76, $04, $CC, $D0, $A9, $81, $A7, $37, $DC, $B8, $5B, $8C, $2F, $33, $D1, $1E, $6C, $FC, $80, $3F, $44, $13, $E4, $6F, $93, $67, $DA, $9D, $74, $D8, $E3, $BB, $CF, $BB, $90, $AE, $4F, $89, $3F, $28, $2D ), {53}
    ( $45, $52, $E4, $44, $18, $7E, $4F, $79, $41, $61, $BE, $EA, $9F, $BC, $94, $19, $5D, $60, $22, $6D, $08, $98, $BE, $60, $9D, $90, $5F, $97, $37, $9D, $EC, $C4, $84, $E0, $61, $27, $1B, $12, $9F, $29, $33, $C2, $9E, $F8, $D3, $C0, $34, $AC ), {54}
    ( $82, $79, $3B, $ED, $CD, $C2, $F3, $17, $EC, $3A, $CB, $D1, $30, $5F, $EF, $61, $28, $88, $DD, $39, $F2, $F1, $05, $1E, $08, $F2, $79, $34, $02, $07, $B1, $1A, $3D, $2F, $48, $E0, $9B, $8E, $42, $93, $C8, $AF, $50, $D0, $D8, $AE, $B7, $71 ), {55}
    ( $A1, $82, $D4, $C6, $EB, $28, $BB, $BF, $B7, $91, $7D, $15, $0B, $52, $7D, $09, $7A, $11, $5C, $7D, $59, $2D, $8F, $D1, $64, $AC, $A3, $78, $EC, $58, $2F, $99, $36, $7B, $7D, $06, $35, $5A, $D7, $CC, $54, $37, $1C, $6C, $7F, $84, $8E, $42 ), {56}
    ( $0E, $83, $58, $3B, $81, $6B, $AE, $24, $12, $A6, $FE, $CF, $6D, $B2, $3C, $15, $F5, $EE, $28, $F8, $19, $3A, $A7, $69, $CD, $F1, $90, $70, $07, $84, $83, $BB, $7D, $CF, $DC, $72, $A9, $F6, $5E, $50, $3F, $0B, $EB, $D1, $13, $BC, $9F, $1A ), {57}
    ( $C7, $7D, $D8, $71, $04, $88, $0C, $AD, $1D, $CA, $EE, $22, $E2, $5E, $3D, $48, $1B, $4D, $A9, $FC, $F7, $75, $06, $B6, $AD, $A3, $69, $36, $F6, $AD, $C1, $E3, $B7, $9F, $73, $8D, $AC, $61, $12, $6A, $63, $1E, $2B, $79, $57, $64, $C4, $26 ), {58}
    ( $92, $CC, $FB, $E9, $0E, $9F, $D5, $B5, $78, $24, $94, $E4, $E1, $9B, $EB, $57, $9D, $01, $35, $A1, $9C, $79, $7A, $AA, $81, $C7, $8A, $5A, $FC, $AA, $1E, $9B, $20, $99, $D4, $AA, $CA, $8F, $FE, $A7, $0C, $47, $88, $F7, $EA, $D8, $54, $07 ), {59}
    ( $88, $D5, $AA, $CA, $96, $03, $B3, $70, $B9, $FF, $4E, $D8, $DD, $DC, $A9, $CE, $CC, $05, $8F, $AA, $7F, $3B, $A0, $42, $14, $33, $EF, $5C, $9E, $B0, $B3, $37, $88, $14, $3E, $F7, $FA, $C4, $93, $E0, $56, $88, $63, $E3, $B0, $BD, $B3, $8D ), {5A}
    ( $56, $3B, $92, $EE, $8E, $E1, $16, $73, $74, $5A, $31, $B5, $1C, $A9, $5E, $BD, $45, $96, $AF, $D7, $DA, $9F, $F4, $58, $E5, $E1, $38, $FC, $E6, $68, $19, $1C, $23, $E5, $FA, $54, $43, $CB, $D8, $1B, $8F, $D9, $68, $2C, $E1, $8A, $F0, $3C ), {5B}
    ( $64, $01, $C1, $8C, $B2, $DE, $B0, $FD, $82, $B5, $58, $38, $C7, $66, $27, $86, $49, $94, $F8, $9F, $B5, $6C, $16, $52, $E4, $CA, $54, $29, $3E, $17, $AF, $D0, $FD, $90, $59, $71, $DA, $66, $7C, $60, $6B, $C3, $27, $CF, $76, $DF, $B0, $91 ), {5C}
    ( $74, $80, $15, $62, $8F, $C7, $35, $4F, $98, $79, $8E, $8E, $89, $4D, $67, $38, $03, $76, $A1, $3C, $D0, $C0, $B7, $F8, $BB, $DF, $3A, $3C, $AB, $96, $1D, $46, $C1, $F9, $C4, $31, $70, $23, $E3, $72, $20, $87, $5A, $D4, $4A, $1B, $B8, $FB ), {5D}
    ( $C0, $49, $B7, $3A, $B6, $D4, $12, $2D, $02, $F8, $B8, $AF, $6A, $C9, $DD, $8E, $16, $20, $CE, $55, $C0, $7F, $8C, $99, $40, $1E, $7C, $89, $34, $FB, $F0, $3C, $4B, $E3, $CD, $44, $EC, $31, $37, $39, $04, $64, $03, $66, $82, $9C, $FD, $9F ), {5E}
    ( $B3, $2B, $47, $19, $95, $40, $70, $C8, $66, $C0, $48, $9F, $DA, $2F, $0E, $87, $38, $72, $D7, $2E, $AE, $8D, $D8, $F9, $B7, $8E, $18, $FD, $16, $31, $E8, $23, $89, $91, $FD, $CF, $D6, $09, $60, $A3, $08, $69, $4A, $03, $20, $E6, $ED, $D8 ), {5F}
    ( $6E, $75, $27, $7F, $F6, $EA, $B1, $B2, $5E, $89, $1B, $90, $E5, $7D, $72, $81, $65, $A6, $E8, $B0, $CE, $5C, $78, $7D, $83, $19, $ED, $07, $83, $62, $F9, $AB, $CF, $21, $24, $28, $3D, $82, $5D, $BE, $A6, $23, $6A, $4A, $01, $04, $21, $8A ), {60}
    ( $39, $06, $8E, $17, $40, $41, $20, $21, $D3, $1C, $4B, $30, $38, $6A, $D1, $82, $43, $79, $4B, $EB, $95, $08, $21, $FA, $FF, $DA, $2E, $D2, $C9, $55, $27, $C6, $D7, $F1, $25, $F2, $FF, $D4, $10, $BF, $88, $0C, $4C, $63, $09, $76, $73, $80 ), {61}
    ( $C4, $29, $29, $FD, $80, $CE, $6F, $C9, $4E, $FE, $17, $0E, $99, $3A, $E9, $E9, $87, $8E, $C3, $16, $79, $71, $5F, $71, $BE, $12, $14, $64, $A6, $D9, $06, $DA, $B4, $C9, $57, $9B, $2D, $27, $4E, $49, $DC, $7E, $84, $DB, $4C, $FB, $85, $28 ), {62}
    ( $53, $BD, $68, $67, $3F, $FE, $34, $33, $C5, $20, $88, $44, $CF, $67, $26, $C9, $52, $B9, $77, $4D, $18, $72, $82, $59, $17, $C9, $30, $02, $D7, $10, $F2, $B7, $3C, $E4, $BE, $86, $0B, $4C, $AD, $C1, $8B, $59, $70, $35, $AD, $EB, $C5, $EA ), {63}
    ( $50, $61, $BE, $D4, $E7, $4A, $6C, $C5, $77, $43, $9D, $43, $BF, $81, $0D, $7B, $42, $2B, $F4, $66, $AB, $88, $E1, $DA, $0E, $3C, $AB, $73, $BF, $94, $8C, $EC, $28, $3C, $6C, $A8, $2E, $F5, $31, $19, $F4, $DF, $1B, $81, $49, $D2, $86, $CE ), {64}
    ( $A6, $42, $66, $3C, $AC, $6C, $CD, $48, $43, $12, $24, $17, $85, $D1, $20, $C1, $D8, $FC, $01, $25, $92, $B6, $E2, $05, $30, $CB, $16, $68, $A1, $6A, $9E, $26, $A9, $3B, $DD, $F3, $C9, $B5, $FB, $34, $70, $30, $9F, $FF, $8E, $F6, $36, $DD ), {65}
    ( $5A, $32, $A4, $AE, $A1, $6D, $76, $94, $7F, $11, $2F, $58, $55, $CC, $68, $8B, $A5, $53, $BE, $3F, $D3, $E1, $EB, $85, $A8, $E8, $9C, $3F, $72, $5E, $30, $53, $E9, $CE, $91, $19, $0D, $BE, $4D, $B6, $BC, $14, $8A, $85, $37, $06, $17, $72 ), {66}
    ( $D9, $E5, $80, $24, $DF, $A0, $60, $35, $D9, $D9, $E9, $FF, $11, $7E, $89, $3D, $7B, $A9, $9C, $2C, $2F, $62, $F9, $C6, $78, $D7, $DE, $17, $7E, $DE, $61, $A4, $33, $D4, $49, $AE, $66, $41, $59, $B8, $40, $0A, $C7, $D7, $88, $73, $E7, $34 ), {67}
    ( $6F, $90, $B2, $41, $9C, $31, $95, $5B, $9B, $08, $55, $3A, $ED, $87, $FA, $4B, $8B, $AD, $FA, $64, $56, $F9, $9E, $43, $F5, $27, $A0, $05, $0E, $2E, $2C, $F0, $2B, $C5, $52, $F0, $2C, $73, $54, $D7, $55, $54, $02, $11, $9A, $9E, $66, $B1 ), {68}
    ( $E1, $4D, $BA, $1A, $4F, $CB, $53, $27, $5F, $59, $40, $53, $36, $71, $30, $16, $BF, $27, $C6, $90, $D9, $FC, $6A, $66, $05, $B3, $9E, $A8, $EB, $1F, $77, $90, $A4, $78, $67, $CA, $74, $A8, $82, $90, $B3, $D4, $B3, $A1, $18, $A8, $82, $3F ), {69}
    ( $36, $4E, $B1, $89, $B3, $1B, $2D, $84, $71, $72, $68, $C5, $DC, $D2, $14, $2F, $E9, $F3, $C1, $FA, $4C, $B9, $1D, $91, $23, $2C, $68, $AD, $70, $CF, $31, $CD, $A0, $DB, $30, $5E, $37, $D6, $C6, $8E, $26, $32, $79, $F2, $2C, $3B, $9C, $C7 ), {6A}
    ( $2E, $D2, $F1, $57, $EE, $AD, $AD, $58, $91, $7B, $D1, $BC, $49, $26, $D8, $37, $EE, $54, $99, $CF, $86, $13, $02, $2E, $06, $84, $E1, $8E, $F1, $9E, $42, $69, $C5, $B9, $CA, $B8, $60, $B3, $1C, $52, $49, $FA, $45, $72, $52, $8E, $49, $0C ), {6B}
    ( $3F, $FA, $A9, $6F, $2C, $12, $C1, $81, $5C, $DA, $F2, $08, $5E, $C2, $3E, $42, $66, $AA, $07, $AB, $A6, $32, $D2, $11, $DC, $3E, $36, $11, $A7, $AB, $A7, $DF, $4A, $38, $69, $0A, $87, $43, $83, $FB, $7D, $F5, $C6, $60, $E6, $1D, $EE, $8B ), {6C}
    ( $21, $FD, $B5, $30, $0B, $BA, $46, $BA, $FA, $02, $C3, $67, $AC, $76, $6A, $D9, $1A, $6F, $F3, $42, $90, $15, $E6, $0B, $09, $A7, $C3, $8F, $45, $FE, $68, $75, $43, $62, $AD, $D1, $34, $B8, $21, $66, $FE, $0D, $94, $B7, $0B, $C2, $EA, $51 ), {6D}
    ( $E9, $EE, $C8, $6B, $A9, $79, $41, $A1, $95, $B2, $7A, $72, $B7, $80, $FE, $1F, $30, $E2, $A2, $D6, $CA, $86, $5D, $64, $42, $96, $22, $9F, $57, $80, $07, $2D, $3E, $AB, $5C, $AD, $A8, $71, $6D, $9D, $D8, $DA, $D7, $41, $C2, $CE, $EF, $93 ), {6E}
    ( $71, $92, $59, $1D, $E0, $A6, $C0, $7B, $03, $99, $22, $68, $71, $5D, $07, $A1, $E3, $CB, $B1, $C4, $AD, $D3, $F5, $1C, $AF, $66, $4C, $6C, $FA, $C4, $C6, $BF, $51, $C4, $07, $EC, $92, $2F, $E7, $36, $7A, $D2, $A4, $0C, $DD, $9D, $2E, $A6 ), {6F}
    ( $63, $47, $7D, $98, $B0, $20, $BC, $5F, $BC, $AA, $7E, $3E, $C5, $58, $96, $F7, $4A, $09, $16, $2A, $00, $6E, $2F, $D2, $98, $0B, $C5, $95, $48, $0D, $C4, $68, $2C, $41, $E0, $C2, $6A, $C5, $F3, $1D, $A8, $AC, $73, $A7, $67, $97, $3A, $4F ), {70}
    ( $28, $7A, $2C, $39, $64, $30, $73, $87, $EB, $C8, $35, $C9, $86, $AB, $DB, $78, $76, $57, $90, $1C, $7D, $7A, $51, $E0, $A9, $8B, $47, $F7, $BA, $A5, $6C, $91, $5A, $50, $46, $84, $E8, $E6, $BE, $B4, $EF, $CF, $2D, $07, $CC, $AD, $60, $70 ), {71}
    ( $AC, $40, $7A, $73, $B4, $0D, $1D, $6B, $E9, $31, $F6, $60, $06, $1B, $F8, $AC, $A1, $F8, $92, $E6, $B6, $FD, $03, $23, $C1, $FE, $7D, $51, $11, $E6, $60, $DE, $4C, $C8, $EB, $49, $E0, $25, $C5, $F3, $AB, $D7, $2C, $5F, $1D, $E7, $80, $FA ), {72}
    ( $67, $9E, $70, $DC, $A2, $9E, $99, $31, $48, $6F, $EA, $BD, $96, $E9, $06, $25, $22, $B0, $63, $8D, $03, $BB, $AF, $26, $E8, $35, $1D, $33, $77, $26, $8E, $08, $0E, $AA, $5B, $DC, $FD, $88, $15, $45, $D4, $4D, $51, $36, $97, $51, $94, $21 ), {73}
    ( $CF, $58, $FE, $F5, $C3, $8D, $D0, $B1, $86, $3B, $01, $A3, $A1, $1F, $2B, $64, $B0, $B4, $D2, $A7, $3F, $6D, $9C, $CD, $E6, $50, $B6, $9D, $5A, $F2, $20, $9E, $5F, $ED, $77, $0C, $81, $B7, $6B, $30, $E1, $B0, $AB, $2E, $4E, $40, $19, $4E ), {74}
    ( $49, $93, $73, $BB, $C2, $2C, $0E, $F5, $A8, $D7, $E1, $A0, $2D, $DE, $35, $B0, $F2, $43, $51, $29, $54, $1F, $CE, $00, $20, $7A, $F9, $76, $A9, $F7, $23, $EA, $60, $E8, $5A, $4E, $4A, $90, $87, $AE, $FA, $7D, $14, $7A, $45, $15, $69, $84 ), {75}
    ( $80, $8C, $06, $22, $75, $C6, $2F, $02, $DF, $22, $21, $4B, $DF, $F7, $A8, $88, $71, $8B, $67, $18, $5F, $49, $30, $BA, $BD, $52, $83, $9A, $FD, $5C, $0C, $F3, $D4, $D6, $A8, $6B, $8C, $A5, $72, $B3, $A0, $5B, $6D, $53, $7E, $B9, $74, $73 ), {76}
    ( $EB, $A2, $0E, $D3, $55, $C1, $14, $AE, $69, $54, $34, $2E, $A2, $F8, $BC, $EE, $3D, $52, $5D, $F6, $89, $96, $CD, $7C, $53, $AA, $B5, $8C, $84, $91, $2A, $C5, $9A, $E9, $99, $22, $F1, $6B, $6F, $CE, $06, $D1, $0A, $57, $29, $1E, $AE, $6F ), {77}
    ( $02, $36, $44, $65, $82, $5D, $8C, $99, $5A, $56, $50, $ED, $B3, $00, $D9, $A8, $8E, $EA, $DB, $1B, $81, $F0, $0F, $76, $99, $7B, $2F, $C1, $9B, $3A, $E4, $C8, $FF, $88, $02, $AC, $4E, $AA, $8D, $69, $47, $10, $29, $9D, $5A, $DA, $DA, $C0 ), {78}
    ( $11, $13, $42, $FE, $C1, $DC, $92, $7D, $58, $A7, $B4, $CD, $58, $F2, $41, $05, $FF, $46, $91, $8F, $E6, $68, $FA, $5A, $7F, $97, $75, $8A, $4A, $08, $D0, $2F, $8A, $8C, $B8, $93, $22, $05, $B6, $78, $27, $56, $F3, $C4, $84, $6E, $39, $77 ), {79}
    ( $DF, $71, $F4, $32, $41, $04, $42, $7F, $E2, $46, $A7, $D5, $70, $A0, $0C, $44, $8F, $1A, $AB, $D5, $CF, $CE, $35, $F0, $A7, $3B, $DC, $14, $19, $D8, $15, $0C, $29, $BC, $8F, $EA, $3C, $D0, $2C, $62, $9B, $3A, $E4, $6B, $62, $EC, $5C, $DA ), {7A}
    ( $46, $48, $9B, $E0, $D9, $F4, $51, $7A, $46, $29, $B5, $8F, $8C, $EE, $FD, $02, $0E, $98, $18, $68, $49, $CA, $7D, $51, $82, $11, $34, $EF, $CD, $23, $D4, $3E, $F3, $93, $85, $B2, $98, $F1, $C3, $31, $28, $A1, $28, $B8, $87, $0F, $72, $D2 ), {7B}
    ( $17, $F6, $78, $4C, $AD, $A7, $1B, $8B, $D1, $B8, $ED, $48, $BD, $75, $D2, $56, $02, $AB, $1E, $F7, $20, $C1, $C5, $18, $3A, $6D, $09, $E0, $B6, $5D, $B5, $8E, $69, $40, $8B, $80, $E5, $9B, $1F, $E6, $DD, $C8, $BF, $5E, $0E, $26, $5F, $0B ), {7C}
    ( $D4, $73, $09, $74, $15, $E0, $F6, $8E, $C1, $0A, $F0, $73, $1B, $3E, $78, $F4, $9E, $E9, $42, $5E, $13, $85, $19, $7F, $6D, $A8, $8F, $C4, $26, $01, $64, $22, $45, $3F, $E3, $E9, $42, $57, $EE, $53, $D6, $92, $16, $AB, $99, $38, $CB, $B8 ), {7D}
    ( $FF, $34, $26, $B0, $33, $1F, $8A, $5A, $06, $E1, $51, $5B, $C3, $31, $9D, $3A, $0B, $C3, $6F, $1E, $FA, $3F, $C2, $A3, $96, $95, $1B, $5F, $F7, $4D, $86, $30, $EE, $CD, $B9, $F8, $9D, $16, $43, $AD, $96, $28, $A5, $64, $FC, $46, $E9, $41 ), {7E}
    ( $D3, $2D, $75, $DF, $26, $6F, $55, $A2, $8B, $D6, $59, $9C, $A3, $ED, $E0, $10, $EF, $68, $73, $61, $F5, $6A, $E8, $92, $31, $6C, $5D, $1C, $8A, $72, $5A, $8F, $41, $8B, $82, $9C, $E9, $B0, $4C, $A5, $95, $EF, $3E, $19, $73, $8B, $06, $39 ), {7F}
    ( $73, $78, $CF, $7D, $1A, $7C, $63, $C0, $2C, $94, $08, $D7, $68, $AE, $54, $55, $D7, $39, $72, $05, $6A, $9B, $C8, $17, $A0, $1D, $4A, $71, $0C, $A7, $16, $96, $1B, $A7, $C9, $A3, $D2, $BC, $4A, $7D, $83, $2E, $43, $7D, $79, $2E, $FB, $27 ), {80}
    ( $37, $09, $7B, $CC, $39, $F0, $54, $1F, $36, $BA, $36, $99, $91, $40, $92, $45, $98, $BB, $69, $08, $21, $BC, $DD, $EF, $28, $D5, $D7, $72, $6E, $F5, $AE, $F1, $11, $DD, $16, $41, $A2, $AC, $CC, $73, $31, $4F, $6B, $46, $04, $D5, $A2, $F3 ), {81}
    ( $86, $18, $F3, $69, $54, $02, $77, $E6, $0D, $C1, $EC, $C3, $2F, $CE, $31, $D1, $8D, $DA, $05, $C7, $D1, $01, $FC, $AE, $C2, $BB, $35, $24, $65, $F1, $B7, $59, $F6, $0F, $56, $CD, $0F, $D2, $F8, $9F, $1A, $20, $CA, $D2, $7A, $9A, $30, $A9 ), {82}
    ( $8B, $24, $62, $B2, $73, $85, $62, $D9, $F2, $21, $A3, $57, $94, $14, $D3, $D4, $75, $2C, $14, $FF, $57, $4E, $EA, $FF, $4A, $24, $5B, $2C, $E9, $0E, $89, $4C, $5B, $7F, $36, $01, $C7, $B4, $A6, $2F, $1F, $68, $0F, $F4, $C5, $27, $3E, $BA ), {83}
    ( $F0, $B9, $64, $2F, $56, $EE, $48, $C2, $F6, $DC, $38, $5C, $2E, $3C, $8E, $1A, $E4, $CF, $13, $37, $0A, $73, $A3, $A5, $2F, $D6, $CB, $7A, $EA, $6D, $0E, $1D, $76, $85, $8A, $75, $C2, $A0, $85, $E3, $46, $1B, $C1, $88, $50, $36, $14, $BD ), {84}
    ( $C8, $8F, $5C, $E6, $E5, $21, $6B, $44, $01, $C3, $0E, $81, $50, $21, $36, $46, $B9, $66, $81, $F5, $8B, $B2, $4F, $9B, $54, $21, $7F, $01, $3F, $90, $AA, $0B, $40, $24, $F0, $32, $AD, $D1, $89, $16, $CB, $84, $E8, $7E, $AE, $37, $D6, $B4 ), {85}
    ( $68, $AA, $13, $AF, $68, $07, $0F, $51, $8D, $69, $26, $26, $27, $CB, $B8, $5F, $FA, $13, $30, $EC, $9A, $97, $1C, $22, $38, $E5, $1A, $2E, $C1, $A2, $35, $BA, $F4, $DE, $D6, $A6, $6F, $E9, $7F, $AA, $B5, $19, $91, $69, $77, $1F, $90, $E1 ), {86}
    ( $76, $5D, $0A, $96, $7B, $58, $56, $18, $BE, $63, $09, $41, $01, $8A, $DA, $5B, $C7, $56, $A8, $3D, $62, $D5, $12, $31, $1E, $4D, $C1, $08, $20, $A9, $96, $8B, $AA, $FD, $0F, $3E, $FB, $99, $3E, $DB, $3C, $F0, $0E, $27, $C8, $E8, $B1, $7F ), {87}
    ( $2F, $B4, $A3, $AA, $EA, $95, $32, $AC, $7C, $34, $0C, $4F, $C1, $FD, $99, $C8, $53, $CC, $1A, $F0, $31, $57, $79, $B1, $33, $B5, $C8, $85, $D2, $49, $BE, $B4, $0D, $C2, $3C, $EE, $7F, $B1, $19, $63, $6C, $D5, $81, $55, $68, $93, $3D, $AD ), {88}
    ( $D0, $03, $65, $29, $5D, $E5, $37, $40, $E8, $38, $CF, $DB, $13, $68, $65, $4C, $07, $9C, $B6, $E4, $58, $B0, $BA, $04, $51, $62, $BA, $D1, $08, $DB, $56, $55, $1C, $47, $A3, $73, $1F, $15, $2F, $0F, $6D, $80, $D0, $25, $B7, $7B, $DC, $D1 ), {89}
    ( $14, $9F, $72, $31, $B9, $32, $50, $F9, $79, $7C, $A2, $02, $A7, $46, $D7, $29, $0C, $00, $7D, $BF, $E3, $0B, $93, $B8, $07, $0F, $3C, $54, $00, $8B, $E0, $17, $B9, $18, $F4, $D9, $4F, $56, $C0, $CD, $E5, $5D, $DA, $B4, $CB, $C8, $9D, $A2 ), {8A}
    ( $CD, $1A, $2E, $04, $E1, $1D, $F2, $0E, $C2, $E6, $6E, $A4, $D0, $49, $E1, $E6, $4D, $1E, $33, $D9, $3E, $CB, $F1, $77, $49, $47, $19, $48, $94, $38, $75, $C2, $64, $59, $E2, $5B, $F2, $BF, $44, $BB, $99, $7F, $F8, $E7, $B9, $21, $B9, $F6 ), {8B}
    ( $95, $5E, $9C, $08, $D8, $59, $4B, $2E, $AD, $DD, $2B, $C0, $07, $1D, $03, $C0, $3B, $24, $BC, $57, $01, $42, $23, $4D, $C3, $67, $B0, $84, $92, $33, $B4, $27, $66, $8F, $9A, $EF, $C1, $44, $DF, $E5, $19, $48, $4D, $17, $5C, $08, $29, $90 ), {8C}
    ( $25, $C0, $BB, $A7, $C0, $8B, $E1, $53, $3E, $4F, $19, $A6, $4D, $E3, $F2, $7C, $10, $5A, $3F, $65, $29, $46, $A5, $CC, $9B, $16, $A2, $67, $1A, $02, $0A, $5C, $D8, $B4, $F2, $02, $45, $9A, $5A, $13, $38, $9D, $8F, $7C, $36, $EA, $D4, $1B ), {8D}
    ( $72, $6C, $EF, $AB, $A5, $D5, $E2, $47, $15, $BF, $15, $2C, $B4, $5A, $C2, $C2, $3C, $F0, $CA, $49, $82, $24, $1B, $D0, $00, $91, $77, $13, $82, $98, $D9, $24, $12, $4C, $EA, $A4, $E4, $21, $D1, $4D, $42, $7C, $B6, $AE, $E5, $49, $C0, $2A ), {8E}
    ( $D2, $A3, $51, $D1, $63, $7B, $4E, $CB, $0A, $74, $3A, $E5, $FD, $B0, $09, $71, $E5, $A1, $09, $48, $85, $76, $49, $62, $3B, $2E, $DA, $9C, $8E, $B2, $81, $80, $EF, $73, $35, $88, $B8, $24, $BD, $4F, $EE, $E8, $A3, $62, $02, $74, $DD, $92 ), {8F}
    ( $7B, $85, $A0, $4F, $5E, $B1, $D8, $D3, $A0, $9D, $53, $FC, $39, $35, $2E, $4D, $61, $30, $84, $13, $27, $9D, $3E, $C5, $DE, $ED, $CD, $F9, $2C, $EA, $E3, $71, $8E, $5E, $00, $46, $0E, $93, $97, $C2, $E2, $AB, $0D, $83, $B8, $56, $A9, $CB ), {90}
    ( $7C, $E1, $77, $4A, $C7, $34, $A2, $2A, $38, $67, $7F, $CB, $57, $45, $AD, $5C, $6D, $C4, $DA, $00, $1D, $93, $5E, $D3, $FA, $15, $E6, $0F, $90, $00, $97, $C7, $07, $A1, $D8, $9A, $36, $5F, $51, $22, $FF, $B4, $82, $AC, $0A, $EF, $22, $7E ), {91}
    ( $F2, $AC, $87, $A6, $AE, $55, $7B, $FF, $30, $4D, $0B, $12, $C8, $1C, $6B, $C4, $2E, $F4, $B9, $75, $2C, $9A, $5B, $0E, $6E, $7F, $4D, $47, $BE, $5B, $F1, $11, $27, $D0, $6E, $6F, $10, $DB, $11, $68, $45, $FF, $25, $8C, $7C, $67, $A5, $17 ), {92}
    ( $4A, $DB, $82, $9C, $17, $90, $CE, $6A, $17, $B1, $2C, $5F, $9D, $53, $CB, $2E, $D0, $5C, $21, $5C, $78, $05, $95, $89, $CF, $02, $27, $63, $C6, $F6, $F6, $A8, $2E, $B8, $1B, $FB, $ED, $FE, $20, $8D, $24, $26, $07, $40, $BA, $33, $AA, $74 ), {93}
    ( $F4, $D8, $4F, $23, $49, $F7, $D4, $3D, $F0, $9C, $67, $E2, $97, $99, $13, $5A, $23, $DD, $DC, $14, $25, $AC, $D4, $BD, $67, $17, $85, $A9, $44, $06, $C3, $5D, $3A, $D9, $9E, $36, $8A, $DA, $B1, $FE, $EB, $CD, $20, $C0, $39, $5B, $5E, $33 ), {94}
    ( $DD, $37, $01, $60, $E8, $5E, $47, $E2, $F5, $18, $62, $1D, $8D, $DA, $23, $A2, $68, $31, $2C, $DD, $93, $6B, $DF, $39, $B5, $8C, $87, $26, $B0, $70, $63, $B5, $37, $5B, $BC, $97, $D0, $8A, $D5, $D9, $1B, $05, $A7, $3B, $22, $65, $3C, $8E ), {95}
    ( $BA, $BA, $14, $E3, $C4, $C5, $E0, $9F, $FC, $CF, $42, $6E, $23, $7C, $45, $B6, $A3, $17, $D4, $B1, $70, $A6, $ED, $1B, $BC, $BC, $0D, $83, $35, $0A, $DD, $66, $8C, $3D, $ED, $B3, $54, $26, $48, $59, $AA, $43, $3B, $28, $90, $9F, $7D, $09 ), {96}
    ( $B2, $CE, $1C, $6D, $00, $29, $69, $6E, $84, $D3, $BF, $DF, $C9, $F5, $F9, $EF, $17, $38, $C2, $82, $DF, $A0, $47, $C7, $A3, $DE, $9B, $86, $41, $50, $7B, $AC, $B5, $1A, $FE, $E2, $9F, $5C, $08, $2E, $10, $1A, $EC, $2A, $1F, $7C, $05, $7A ), {97}
    ( $FE, $B1, $28, $70, $3C, $AF, $0D, $39, $C3, $4B, $85, $B0, $25, $8B, $28, $93, $72, $DC, $A0, $AF, $FE, $ED, $70, $84, $4E, $86, $11, $E7, $8B, $85, $E7, $86, $81, $5D, $4E, $D0, $00, $6F, $79, $4E, $79, $DB, $F5, $B2, $2F, $BE, $4C, $1E ), {98}
    ( $F8, $9A, $07, $FC, $4E, $72, $FE, $1D, $11, $AD, $9A, $D9, $15, $13, $2F, $0E, $E0, $FA, $5F, $B2, $35, $EC, $4C, $F3, $AA, $89, $42, $C0, $AD, $BE, $46, $7C, $AB, $A5, $41, $30, $05, $69, $CD, $80, $61, $17, $F7, $32, $34, $A7, $F3, $DF ), {99}
    ( $08, $08, $B0, $10, $37, $9C, $49, $69, $7A, $0C, $9C, $FD, $A6, $72, $08, $B9, $F6, $03, $45, $DF, $BE, $40, $5C, $56, $DA, $F7, $52, $91, $36, $92, $FF, $0D, $0B, $A9, $76, $3D, $CD, $C0, $FC, $3E, $57, $BA, $4E, $BB, $A8, $D3, $63, $AE ), {9A}
    ( $99, $6F, $7F, $4D, $51, $A5, $7A, $9E, $CA, $97, $3D, $EC, $53, $0C, $3B, $2D, $4E, $D8, $27, $44, $26, $E5, $0D, $24, $D5, $EE, $A4, $CB, $EE, $71, $53, $B3, $30, $10, $84, $25, $2F, $C3, $B3, $88, $E3, $B8, $F9, $4C, $31, $A0, $35, $11 ), {9B}
    ( $5F, $F1, $A5, $E2, $C5, $10, $89, $C6, $92, $36, $2E, $25, $05, $2D, $19, $BF, $33, $FF, $36, $DE, $68, $A2, $B9, $DF, $C8, $F6, $49, $B1, $3C, $7D, $D7, $06, $E0, $E7, $53, $0D, $79, $2D, $53, $70, $37, $95, $15, $82, $42, $82, $8B, $57 ), {9C}
    ( $20, $A1, $F7, $58, $4D, $0B, $E5, $92, $F3, $FB, $03, $03, $C6, $42, $7F, $4E, $D4, $82, $26, $34, $F1, $8B, $54, $0A, $65, $DD, $BD, $8D, $22, $CD, $6B, $58, $44, $B1, $E6, $98, $14, $8D, $AE, $5C, $F9, $B1, $0B, $C5, $9B, $F5, $2D, $2E ), {9D}
    ( $47, $A0, $0D, $C2, $36, $CA, $2E, $4C, $65, $58, $78, $B2, $EF, $B7, $3F, $30, $EA, $ED, $7B, $58, $B2, $C6, $28, $50, $5F, $23, $33, $55, $74, $E0, $59, $15, $19, $54, $F1, $5D, $5F, $1C, $BF, $A9, $74, $6E, $F1, $A6, $DF, $CC, $8F, $9B ), {9E}
    ( $81, $4C, $EB, $81, $83, $BB, $DE, $56, $CB, $55, $C2, $63, $5C, $EF, $2C, $B5, $5B, $35, $41, $83, $B9, $14, $88, $E1, $56, $45, $08, $37, $7F, $73, $7C, $A7, $CA, $13, $22, $95, $82, $4F, $B9, $3F, $BE, $65, $A0, $FB, $FA, $BB, $9E, $8F ), {9F}
    ( $87, $7B, $0B, $FF, $4A, $46, $39, $89, $EA, $E4, $A8, $2A, $37, $08, $F3, $6E, $37, $E3, $83, $8C, $22, $8C, $C3, $E6, $61, $94, $A5, $32, $DE, $15, $9C, $CF, $1D, $15, $D9, $AB, $52, $20, $55, $DA, $4C, $BB, $FF, $56, $08, $13, $7E, $F1 ), {A0}
    ( $2B, $DD, $46, $82, $1C, $E6, $DC, $3F, $4F, $80, $AA, $74, $FF, $B4, $0B, $8A, $92, $0E, $D5, $E3, $24, $54, $FE, $BE, $3D, $32, $D3, $B8, $AC, $E5, $13, $33, $79, $3E, $06, $1F, $48, $63, $47, $EF, $39, $71, $1E, $B1, $03, $75, $91, $CD ), {A1}
    ( $12, $56, $B3, $1F, $E4, $AA, $79, $19, $49, $6B, $FD, $DD, $FE, $8D, $C6, $6B, $BA, $D7, $71, $0F, $A7, $84, $01, $5C, $A5, $3A, $D9, $50, $CE, $B3, $7F, $51, $5C, $D8, $34, $77, $B9, $3F, $70, $81, $8D, $06, $FE, $DE, $91, $CA, $C6, $ED ), {A2}
    ( $E8, $0C, $F2, $A8, $35, $CF, $5A, $B9, $1E, $16, $64, $62, $5B, $0A, $DC, $65, $94, $C2, $9F, $B9, $C5, $94, $4B, $AB, $60, $22, $89, $0A, $B4, $04, $CE, $74, $E1, $28, $7F, $92, $DB, $BD, $81, $26, $2E, $53, $7D, $F5, $15, $FC, $67, $3B ), {A3}
    ( $35, $E0, $DF, $14, $60, $2D, $6E, $82, $39, $13, $BD, $6F, $FB, $4F, $12, $84, $DD, $D1, $C5, $AE, $6F, $AE, $E7, $AD, $9F, $D3, $25, $CD, $B3, $E7, $8B, $50, $35, $1E, $62, $2F, $5E, $85, $78, $9E, $FC, $E2, $92, $BD, $12, $55, $9A, $5C ), {A4}
    ( $3B, $4F, $02, $4E, $D5, $F3, $D2, $43, $09, $53, $06, $0A, $F7, $AF, $24, $9D, $CA, $89, $FF, $4E, $C8, $BE, $14, $C8, $80, $10, $15, $E4, $F8, $B1, $38, $4A, $15, $23, $58, $7C, $40, $30, $45, $8C, $11, $76, $AC, $06, $6D, $4B, $DB, $E6 ), {A5}
    ( $6B, $62, $0F, $F8, $67, $6A, $CB, $3A, $BF, $1E, $91, $76, $1F, $8F, $CA, $1B, $88, $2E, $E5, $6B, $11, $E6, $33, $D9, $7D, $42, $3B, $BD, $B8, $18, $95, $87, $32, $58, $8D, $A0, $EB, $67, $5C, $4C, $C0, $C0, $87, $E6, $2E, $F9, $41, $66 ), {A6}
    ( $ED, $FE, $25, $7C, $6B, $C8, $EC, $25, $05, $E3, $3F, $1C, $B1, $41, $5A, $8D, $83, $1B, $7A, $85, $66, $2A, $B0, $5E, $03, $E3, $B8, $53, $5C, $39, $1B, $73, $72, $C0, $32, $6C, $F3, $7F, $A9, $11, $71, $50, $8E, $CC, $51, $AC, $0E, $01 ), {A7}
    ( $4B, $BC, $63, $46, $21, $93, $7F, $A4, $8A, $1F, $B6, $B7, $3F, $84, $18, $72, $DB, $18, $55, $A2, $88, $33, $FB, $A7, $F3, $C6, $04, $74, $76, $89, $29, $32, $18, $A0, $38, $C8, $F0, $06, $01, $33, $93, $F9, $DD, $FA, $66, $E3, $0F, $40 ), {A8}
    ( $38, $F4, $67, $A0, $DC, $EB, $9D, $1E, $14, $01, $27, $10, $AF, $DB, $01, $89, $B6, $04, $B3, $DB, $47, $4A, $2B, $53, $15, $1F, $66, $DB, $79, $A4, $12, $42, $BE, $36, $E1, $DA, $85, $81, $4F, $10, $72, $9F, $89, $EF, $EB, $DC, $E1, $20 ), {A9}
    ( $6D, $6B, $18, $54, $5F, $BD, $19, $96, $2B, $B0, $41, $69, $47, $D7, $C0, $17, $BD, $C7, $1D, $50, $76, $E2, $80, $B7, $1C, $FC, $F3, $A3, $61, $3E, $F4, $05, $9F, $42, $EF, $2E, $6D, $AE, $68, $B9, $E6, $40, $A9, $C1, $B4, $6B, $1A, $61 ), {AA}
    ( $60, $B3, $AF, $C8, $3A, $97, $1C, $42, $20, $3F, $B3, $EB, $56, $0D, $56, $FA, $96, $85, $40, $ED, $4F, $BD, $84, $C4, $24, $08, $43, $CA, $CF, $29, $72, $48, $93, $12, $BA, $4C, $97, $94, $27, $37, $7F, $DC, $D9, $31, $98, $C7, $FC, $58 ), {AB}
    ( $42, $50, $79, $EA, $14, $B5, $5C, $88, $BB, $A4, $45, $46, $DB, $51, $90, $D0, $74, $9F, $B4, $7A, $97, $77, $39, $9A, $A6, $EA, $A8, $6B, $9D, $19, $EE, $03, $16, $66, $39, $C5, $CB, $07, $22, $86, $90, $CC, $2F, $BF, $E0, $59, $07, $35 ), {AC}
    ( $65, $35, $11, $07, $4C, $EC, $FD, $D5, $B6, $CE, $60, $C8, $17, $D3, $59, $62, $05, $22, $75, $8E, $2E, $C3, $8E, $A9, $0C, $4A, $78, $E8, $24, $F8, $48, $9D, $CB, $0D, $04, $DF, $B5, $37, $C1, $B0, $F6, $A2, $AA, $76, $0F, $2A, $40, $AF ), {AD}
    ( $9A, $F0, $A8, $87, $85, $83, $59, $36, $4B, $9E, $12, $14, $E6, $93, $52, $FD, $32, $4B, $7F, $B5, $14, $89, $A4, $4B, $36, $46, $0E, $65, $32, $FC, $45, $AF, $68, $11, $F6, $2B, $F8, $65, $0D, $3A, $41, $97, $C8, $F6, $D2, $4C, $BC, $65 ), {AE}
    ( $9C, $2F, $C3, $99, $6C, $38, $A5, $BD, $1F, $26, $73, $D3, $E3, $04, $C4, $00, $26, $78, $E6, $B4, $8E, $A8, $E5, $61, $B9, $48, $28, $DD, $E2, $86, $DF, $6B, $7E, $F7, $1F, $90, $E7, $52, $0B, $91, $DA, $16, $C3, $6E, $D4, $47, $A1, $78 ), {AF}
    ( $69, $C4, $DA, $C1, $BC, $8A, $F8, $67, $7B, $10, $C5, $16, $88, $E8, $61, $18, $64, $E5, $2F, $76, $B7, $FB, $0B, $93, $44, $5D, $74, $4A, $2A, $47, $A3, $2B, $8F, $74, $DB, $D8, $19, $8B, $D0, $24, $64, $BC, $2A, $04, $B5, $A9, $5B, $24 ), {B0}
    ( $B4, $E4, $C2, $BC, $5A, $9B, $7D, $66, $A7, $A0, $95, $51, $D7, $24, $ED, $67, $8C, $41, $08, $71, $AF, $19, $97, $E5, $2B, $99, $C2, $E2, $E4, $BF, $A6, $6D, $C9, $D1, $7B, $5A, $50, $5E, $30, $12, $D0, $C5, $5F, $45, $3B, $52, $CE, $E3 ), {B1}
    ( $B9, $02, $53, $9A, $FB, $51, $BF, $CA, $A6, $C5, $93, $6B, $9E, $61, $46, $BA, $AA, $2F, $DE, $30, $B0, $69, $0C, $83, $6A, $EF, $B9, $49, $98, $75, $FA, $09, $E5, $D5, $2A, $83, $23, $6D, $DE, $09, $16, $E9, $54, $65, $F4, $61, $6F, $63 ), {B2}
    ( $CE, $D4, $EA, $7A, $B8, $06, $F9, $A8, $B2, $07, $D7, $9E, $32, $96, $7A, $AA, $13, $DB, $C0, $F1, $E9, $E8, $BB, $95, $0A, $C5, $A9, $A7, $4D, $0B, $4A, $72, $6F, $CB, $88, $1E, $56, $CD, $F4, $95, $69, $9A, $E9, $16, $A7, $F1, $03, $E8 ), {B3}
    ( $D1, $91, $CD, $18, $7C, $5C, $90, $0F, $35, $28, $E2, $AD, $D5, $28, $E5, $0A, $D5, $AF, $F0, $54, $DD, $03, $36, $35, $25, $01, $46, $ED, $F9, $7C, $08, $D8, $3B, $8A, $E7, $57, $4C, $F7, $23, $7B, $92, $6B, $64, $34, $F8, $87, $0A, $B3 ), {B4}
    ( $F6, $97, $BF, $66, $F8, $65, $E3, $ED, $FF, $2C, $B9, $4E, $AB, $10, $33, $07, $D3, $49, $C9, $88, $3B, $7D, $D1, $16, $66, $5A, $64, $A6, $04, $78, $51, $E0, $86, $5A, $23, $8E, $3A, $0A, $28, $40, $F3, $07, $B1, $96, $AB, $53, $C9, $81 ), {B5}
    ( $B1, $B8, $55, $AC, $08, $FD, $CC, $A3, $29, $2D, $00, $A5, $35, $07, $0F, $04, $9F, $8C, $A6, $A0, $5C, $0F, $46, $5B, $CE, $77, $F0, $93, $CA, $E3, $7D, $62, $63, $03, $51, $76, $72, $7A, $8C, $F4, $8E, $11, $FB, $52, $E9, $C9, $D2, $94 ), {B6}
    ( $13, $0B, $A6, $FB, $42, $01, $BA, $5C, $E3, $8E, $CA, $77, $64, $62, $97, $B2, $73, $08, $29, $97, $A5, $AA, $41, $F1, $DF, $0C, $FC, $FA, $4F, $7A, $4F, $45, $A3, $81, $C7, $DB, $21, $B9, $E5, $F5, $C4, $3D, $FD, $D6, $A2, $44, $11, $38 ), {B7}
    ( $D8, $57, $D2, $A3, $16, $DA, $A6, $F1, $B0, $4C, $14, $1B, $28, $38, $82, $01, $81, $6E, $52, $5A, $38, $65, $DB, $D4, $5B, $7D, $63, $2A, $06, $2F, $62, $EE, $BA, $C1, $79, $45, $61, $E1, $26, $0A, $A3, $B2, $34, $ED, $6B, $7F, $0B, $BB ), {B8}
    ( $0D, $EF, $00, $59, $F4, $AE, $8E, $60, $22, $40, $A9, $27, $BA, $6C, $F7, $F6, $97, $7B, $96, $D2, $4E, $D7, $C0, $55, $91, $FA, $53, $F8, $6F, $BD, $D1, $38, $26, $02, $6D, $FD, $8F, $78, $41, $75, $CA, $8E, $B0, $DF, $BB, $B4, $C8, $A7 ), {B9}
    ( $E3, $31, $AE, $5D, $8D, $17, $57, $DF, $C7, $84, $6B, $E9, $CD, $D5, $40, $F2, $86, $0C, $17, $94, $44, $B7, $74, $67, $37, $6A, $AA, $2D, $9F, $3F, $C7, $0E, $6D, $F2, $19, $7B, $AB, $9E, $D9, $79, $65, $34, $71, $13, $A0, $F0, $4A, $4B ), {BA}
    ( $FD, $05, $E3, $D2, $69, $B6, $03, $78, $2A, $82, $49, $3D, $F3, $33, $60, $20, $91, $D4, $A5, $C3, $9E, $48, $04, $BC, $73, $39, $A6, $4E, $D4, $CE, $6D, $AD, $0F, $FE, $D1, $2C, $27, $92, $07, $DF, $2F, $A6, $3D, $51, $F3, $92, $51, $C1 ), {BB}
    ( $8A, $ED, $D0, $5F, $6E, $A4, $FF, $E8, $1C, $DF, $05, $01, $52, $7B, $29, $90, $BE, $23, $34, $27, $53, $F3, $C1, $FE, $C9, $AF, $60, $99, $21, $99, $05, $E9, $78, $25, $1D, $29, $4B, $6A, $A0, $82, $A7, $7A, $BD, $CB, $96, $35, $98, $FF ), {BC}
    ( $7D, $E2, $54, $25, $FA, $9D, $3B, $37, $59, $6D, $DE, $84, $3A, $6D, $E4, $58, $6E, $AC, $15, $C0, $F4, $6F, $96, $F4, $D4, $92, $D4, $82, $7A, $EB, $A9, $A6, $58, $A3, $B1, $39, $04, $77, $9B, $08, $ED, $75, $5D, $1E, $A6, $94, $F6, $F2 ), {BD}
    ( $A2, $11, $3E, $E7, $0C, $FF, $45, $71, $9C, $5E, $A5, $00, $D4, $CD, $EC, $63, $B3, $36, $BD, $0E, $96, $A3, $15, $8F, $D9, $BA, $2C, $6F, $DC, $25, $EB, $1E, $DF, $4B, $2E, $21, $F5, $98, $7A, $9C, $18, $EA, $38, $33, $F0, $F8, $77, $7C ), {BE}
    ( $52, $E9, $FC, $56, $9B, $3C, $BD, $6D, $34, $19, $E0, $91, $F2, $02, $1B, $7F, $A9, $42, $82, $41, $3C, $99, $18, $FC, $59, $A6, $F6, $EC, $B1, $D5, $E5, $7D, $61, $E6, $7C, $E3, $68, $A9, $1D, $F9, $CF, $8D, $47, $74, $5E, $B0, $97, $4C ), {BF}
    ( $44, $23, $4D, $93, $78, $DD, $10, $8D, $55, $BC, $3C, $70, $93, $C7, $22, $D7, $00, $CA, $9E, $F4, $23, $4B, $CB, $47, $9A, $58, $E3, $F5, $31, $8E, $F8, $25, $59, $16, $9B, $64, $EA, $95, $6E, $0D, $67, $E5, $E2, $BA, $44, $D1, $BB, $D0 ), {C0}
    ( $AB, $81, $37, $F1, $01, $5B, $D6, $13, $D5, $D8, $6D, $9A, $AD, $15, $43, $D3, $6A, $12, $19, $96, $7C, $FA, $A9, $74, $69, $FF, $6A, $10, $56, $E9, $5F, $78, $CC, $83, $17, $A2, $FE, $64, $AB, $DE, $23, $D3, $A1, $49, $CD, $F4, $27, $CF ), {C1}
    ( $30, $17, $9A, $55, $70, $F2, $97, $1A, $42, $CB, $F4, $24, $0C, $11, $EA, $83, $DC, $A7, $43, $1F, $12, $36, $62, $07, $FD, $54, $C0, $79, $0D, $53, $3B, $A9, $03, $9B, $0A, $70, $CF, $72, $ED, $DD, $C6, $3C, $55, $0F, $E8, $E9, $EC, $3A ), {C2}
    ( $00, $10, $E2, $B6, $87, $2B, $7C, $01, $D0, $F7, $D4, $AA, $26, $7F, $2D, $D8, $ED, $9A, $BB, $7B, $BA, $A1, $D6, $7A, $12, $51, $EA, $BA, $4C, $1A, $F5, $CE, $E7, $64, $A0, $37, $E2, $59, $84, $8F, $77, $A9, $17, $6A, $56, $95, $8D, $31 ), {C3}
    ( $A3, $E7, $DB, $8E, $45, $37, $80, $0A, $6B, $9B, $99, $B8, $09, $D4, $8B, $3E, $0D, $A4, $F2, $6F, $73, $DC, $AD, $65, $2E, $D0, $E9, $66, $9A, $2C, $65, $DC, $AF, $BB, $4D, $67, $D4, $F2, $CE, $5F, $25, $63, $7C, $D3, $75, $D9, $A4, $E0 ), {C4}
    ( $6C, $DA, $D1, $61, $05, $2E, $36, $90, $52, $A8, $A6, $29, $04, $C8, $B9, $9F, $B5, $7E, $E3, $5D, $EF, $1C, $E9, $B3, $5D, $C1, $1C, $B6, $2F, $32, $D2, $00, $47, $F6, $70, $87, $1A, $1A, $50, $03, $C7, $61, $49, $78, $69, $C4, $76, $75 ), {C5}
    ( $41, $8B, $34, $85, $77, $BC, $AB, $AB, $8C, $4E, $61, $47, $3C, $82, $74, $FB, $58, $62, $76, $DC, $8F, $0E, $20, $9C, $47, $A9, $62, $1F, $AA, $79, $54, $3B, $52, $A2, $0B, $E1, $1E, $70, $0A, $D8, $DB, $8C, $96, $5A, $92, $5F, $96, $16 ), {C6}
    ( $9B, $A4, $5F, $11, $1E, $3E, $21, $11, $2D, $78, $97, $36, $D2, $1A, $86, $AE, $59, $FD, $CD, $C6, $07, $D2, $E4, $A0, $7A, $9A, $65, $5D, $E3, $FD, $BC, $92, $71, $87, $A9, $61, $E6, $7C, $04, $B2, $D2, $D0, $D2, $43, $48, $07, $32, $EE ), {C7}
    ( $BE, $F9, $16, $5E, $A6, $56, $13, $EB, $7D, $5C, $5D, $BA, $E8, $50, $FC, $C5, $CE, $3C, $EF, $21, $BF, $00, $69, $D7, $72, $E2, $23, $8B, $F2, $97, $58, $B8, $22, $9C, $08, $10, $7B, $E2, $A4, $54, $0E, $8A, $BE, $20, $A5, $EE, $57, $85 ), {C8}
    ( $A4, $76, $69, $9E, $5B, $9A, $82, $86, $0E, $C2, $32, $D6, $61, $E4, $B3, $98, $11, $F5, $3B, $D3, $3D, $8F, $4D, $73, $77, $D4, $98, $5B, $59, $13, $C0, $13, $31, $EE, $01, $B7, $09, $EA, $EC, $55, $14, $82, $D8, $68, $C6, $4E, $44, $48 ), {C9}
    ( $A5, $D9, $94, $BA, $7A, $F9, $C3, $B0, $B1, $05, $F5, $DA, $95, $BD, $AE, $52, $25, $EB, $D3, $62, $F9, $EF, $64, $54, $4F, $64, $F2, $C6, $9C, $1C, $04, $A3, $DE, $55, $CF, $14, $D1, $04, $62, $7C, $51, $AA, $1F, $59, $AF, $90, $48, $EB ), {CA}
    ( $10, $8E, $F6, $15, $A4, $5A, $65, $23, $A3, $A3, $AC, $FB, $0E, $44, $D0, $A9, $6F, $2A, $0A, $AD, $D5, $AF, $A6, $2A, $A2, $E9, $F5, $D6, $86, $11, $92, $44, $8D, $71, $5E, $4A, $BB, $08, $98, $C8, $F2, $B9, $44, $58, $64, $89, $53, $C5 ), {CB}
    ( $B8, $A7, $CA, $8D, $32, $57, $A3, $26, $FE, $77, $CD, $50, $CE, $7A, $4B, $6C, $3A, $4F, $E9, $51, $1B, $70, $34, $F7, $CA, $20, $9F, $40, $60, $B7, $C9, $16, $39, $AF, $A5, $42, $5B, $19, $00, $C4, $AC, $F4, $CB, $D9, $BC, $CF, $B4, $87 ), {CC}
    ( $55, $A8, $6E, $21, $3D, $1E, $68, $DD, $A9, $F2, $4D, $7B, $D9, $12, $B4, $7E, $A0, $F7, $CC, $A9, $6E, $C9, $D7, $2B, $C4, $A4, $0F, $A2, $D1, $4E, $55, $D1, $8B, $76, $A1, $BA, $75, $B6, $B7, $D2, $DF, $DD, $86, $37, $11, $FA, $10, $C8 ), {CD}
    ( $5C, $86, $8D, $27, $3E, $44, $28, $28, $A1, $0B, $70, $E8, $EC, $2B, $16, $94, $99, $99, $D6, $80, $DC, $B3, $13, $3C, $1F, $DC, $0B, $52, $3A, $7E, $3C, $A1, $A6, $EF, $A7, $96, $32, $DF, $2E, $92, $05, $D8, $23, $30, $C1, $63, $12, $47 ), {CE}
    ( $E4, $AE, $CE, $26, $6A, $0F, $C5, $0D, $13, $DB, $5B, $AB, $98, $09, $9C, $B1, $C3, $3A, $38, $C2, $B8, $0A, $7B, $3B, $FB, $AD, $57, $D3, $15, $61, $DE, $EF, $9E, $60, $05, $18, $B7, $02, $34, $AC, $C5, $38, $B7, $F0, $26, $7D, $79, $13 ), {CF}
    ( $79, $D7, $8F, $51, $A8, $94, $00, $B6, $E1, $52, $8B, $56, $8E, $9F, $C5, $CC, $34, $B1, $79, $79, $FB, $4F, $68, $6B, $B8, $DB, $B4, $80, $D3, $C9, $BD, $39, $17, $43, $27, $D7, $55, $A3, $49, $0E, $4D, $36, $13, $75, $32, $50, $A7, $02 ), {D0}
    ( $DB, $D0, $5E, $37, $48, $E7, $B6, $BB, $4D, $23, $11, $FA, $B8, $86, $87, $6F, $21, $44, $8A, $78, $10, $5D, $72, $E8, $75, $B7, $1E, $6E, $54, $8C, $67, $8D, $C0, $39, $80, $15, $78, $A7, $BC, $A0, $EC, $C7, $58, $D8, $55, $72, $99, $03 ), {D1}
    ( $3D, $E3, $E8, $94, $93, $18, $5D, $49, $9F, $D5, $44, $85, $84, $C6, $70, $A0, $AD, $1D, $FD, $7C, $5D, $12, $B1, $EE, $8F, $D1, $71, $EE, $73, $88, $FB, $5F, $BD, $32, $3A, $DD, $69, $F8, $75, $CB, $0F, $AE, $B5, $15, $E2, $30, $61, $C2 ), {D2}
    ( $1F, $A5, $2F, $02, $25, $1A, $B5, $E0, $80, $A2, $8C, $04, $B9, $56, $21, $E7, $62, $71, $D1, $EA, $36, $C2, $CA, $09, $C0, $75, $CC, $3D, $62, $AE, $FE, $FC, $6A, $9D, $50, $1C, $16, $97, $B4, $F6, $22, $2A, $60, $5B, $AC, $D4, $24, $CC ), {D3}
    ( $EA, $6D, $6B, $B1, $D1, $B3, $C8, $46, $F9, $2F, $8D, $4A, $C0, $A2, $81, $33, $A2, $E4, $66, $8B, $87, $0D, $61, $94, $71, $F4, $9A, $6A, $99, $0C, $26, $76, $D1, $45, $86, $99, $B4, $EC, $A5, $27, $7C, $51, $4B, $1A, $61, $0C, $6B, $4A ), {D4}
    ( $84, $07, $17, $C5, $EF, $1C, $64, $72, $F1, $39, $AF, $11, $82, $6E, $DF, $5E, $09, $B7, $8E, $0A, $ED, $2F, $09, $45, $95, $72, $FA, $96, $CC, $51, $25, $54, $75, $0A, $EC, $C7, $D9, $2C, $33, $BD, $CE, $57, $26, $F1, $07, $62, $4D, $54 ), {D5}
    ( $EE, $63, $4A, $78, $FF, $3F, $08, $16, $18, $8A, $EB, $59, $EB, $2A, $4D, $8F, $D2, $32, $62, $33, $55, $58, $53, $08, $74, $C4, $84, $0D, $81, $8F, $F3, $3D, $F7, $70, $C3, $A7, $BD, $CC, $8A, $00, $A4, $35, $7B, $C8, $D1, $17, $A0, $14 ), {D6}
    ( $59, $20, $36, $7E, $2D, $C9, $6A, $05, $AE, $C7, $9B, $35, $63, $64, $A2, $77, $19, $4E, $2B, $6C, $64, $17, $75, $C2, $F9, $4B, $97, $C5, $7D, $DD, $7A, $6A, $AE, $17, $4C, $78, $5D, $13, $E6, $EC, $A5, $22, $9B, $42, $FD, $86, $BF, $10 ), {D7}
    ( $9F, $BE, $88, $64, $A3, $DB, $D1, $DB, $85, $98, $EF, $83, $90, $CF, $51, $99, $EB, $3E, $CB, $BA, $A3, $A4, $DE, $78, $0F, $2D, $10, $0C, $53, $41, $41, $49, $73, $5C, $5F, $79, $93, $00, $25, $2D, $15, $03, $52, $A3, $58, $7E, $00, $5A ), {D8}
    ( $31, $2A, $2A, $C3, $74, $FB, $F0, $04, $C4, $BE, $5A, $0F, $7E, $79, $B6, $A3, $C6, $97, $0F, $4F, $B4, $2B, $25, $B5, $7C, $8F, $CF, $1A, $49, $BA, $9F, $83, $F9, $68, $2C, $CC, $DF, $E4, $1B, $01, $3D, $41, $59, $89, $2A, $A4, $31, $3E ), {D9}
    ( $4C, $68, $50, $A9, $F5, $67, $DB, $08, $9A, $D0, $DD, $28, $C4, $9E, $77, $3F, $E1, $DE, $0E, $31, $A2, $51, $AC, $7E, $76, $2B, $72, $60, $64, $1D, $AB, $AA, $DD, $53, $75, $D3, $11, $D8, $A7, $D1, $36, $C6, $09, $39, $D6, $16, $02, $FD ), {DA}
    ( $E2, $87, $35, $C4, $91, $15, $DF, $61, $6A, $6A, $B7, $3C, $8F, $69, $B0, $0B, $77, $29, $F9, $2D, $52, $D6, $00, $CA, $01, $56, $7A, $D5, $D0, $40, $4D, $04, $57, $CC, $28, $0F, $9A, $86, $77, $B5, $5B, $B7, $E6, $8D, $2B, $66, $B6, $C6 ), {DB}
    ( $0F, $33, $EC, $33, $5C, $33, $A7, $C1, $F4, $86, $BA, $09, $1E, $4B, $9B, $DB, $2D, $19, $95, $26, $7A, $D0, $F6, $41, $9E, $FD, $76, $61, $E8, $2D, $2E, $98, $49, $61, $96, $94, $51, $3D, $88, $44, $C2, $6C, $7E, $94, $4B, $BA, $F9, $FC ), {DC}
    ( $97, $BB, $10, $F0, $1B, $A9, $EE, $34, $44, $60, $C7, $A1, $F4, $48, $B7, $47, $50, $C6, $AC, $6A, $41, $C7, $E3, $36, $2D, $F8, $31, $F3, $66, $59, $4B, $D4, $80, $C3, $E9, $52, $A5, $F3, $86, $6B, $60, $BF, $EE, $90, $60, $69, $4F, $12 ), {DD}
    ( $EC, $59, $E9, $76, $4B, $8E, $94, $80, $72, $7A, $96, $9D, $E0, $39, $76, $54, $FC, $0A, $5B, $03, $EC, $82, $11, $E4, $8D, $D8, $6F, $C2, $0F, $1E, $D8, $FF, $7A, $6A, $7E, $4B, $CC, $FF, $71, $E8, $1D, $85, $BA, $92, $4F, $8C, $B2, $22 ), {DE}
    ( $F5, $9D, $43, $B4, $8C, $69, $C6, $09, $AB, $96, $BC, $4C, $46, $5C, $05, $79, $6B, $3B, $E4, $8A, $2A, $23, $9D, $3A, $1A, $30, $96, $A4, $A0, $5A, $B2, $E7, $55, $F3, $63, $A1, $90, $C8, $FF, $DC, $75, $93, $1A, $29, $70, $5C, $6E, $A5 ), {DF}
    ( $AE, $74, $1B, $F2, $7E, $11, $D7, $C7, $0B, $EF, $71, $86, $19, $A8, $80, $2B, $2A, $51, $D8, $D8, $09, $04, $FD, $0C, $B6, $B4, $F4, $B9, $D8, $8A, $01, $60, $65, $B0, $DE, $8C, $86, $76, $24, $C3, $CC, $9B, $67, $E1, $46, $4F, $AD, $1C ), {E0}
    ( $93, $28, $8B, $6A, $06, $CC, $A1, $CF, $C0, $1B, $C4, $F9, $72, $91, $6C, $EA, $0F, $A8, $1B, $06, $05, $87, $A8, $EC, $EC, $5F, $39, $F0, $29, $9F, $DA, $FE, $0A, $67, $B6, $56, $D3, $C2, $32, $E7, $D3, $31, $04, $38, $3E, $42, $9B, $5B ), {E1}
    ( $1C, $C2, $90, $0B, $44, $66, $ED, $62, $9D, $7E, $10, $2B, $7C, $C1, $BD, $40, $A8, $BE, $49, $CD, $17, $63, $91, $72, $F4, $98, $05, $DC, $A2, $ED, $EA, $FD, $DB, $FF, $20, $B1, $F7, $48, $74, $48, $AD, $46, $B9, $AF, $94, $3E, $58, $B0 ), {E2}
    ( $06, $DC, $61, $2D, $B5, $35, $BE, $7C, $68, $F6, $2A, $F6, $02, $63, $C1, $85, $4F, $5F, $00, $72, $7B, $1A, $D9, $1A, $CC, $B8, $6C, $20, $BD, $B4, $5B, $89, $C3, $48, $9C, $FF, $58, $11, $65, $71, $89, $A5, $56, $14, $6E, $8F, $CA, $EF ), {E3}
    ( $94, $B2, $D9, $F4, $94, $0A, $CF, $EC, $16, $5F, $65, $87, $2B, $F1, $15, $23, $0A, $D0, $2D, $0B, $E7, $DF, $0A, $A8, $AB, $CE, $C9, $F4, $38, $43, $50, $6E, $62, $0B, $8C, $09, $62, $E5, $F1, $57, $43, $25, $32, $3A, $14, $31, $A6, $BF ), {E4}
    ( $61, $C3, $60, $35, $2E, $E2, $C2, $DE, $D7, $C6, $E8, $F1, $F8, $9C, $A3, $49, $36, $75, $AE, $45, $77, $4D, $6D, $44, $F6, $03, $2B, $F2, $DB, $54, $3A, $CA, $25, $04, $81, $38, $08, $4A, $AF, $B7, $FD, $49, $66, $A9, $93, $98, $E0, $45 ), {E5}
    ( $CB, $0A, $C9, $2A, $ED, $14, $E7, $63, $07, $B7, $2D, $CC, $6E, $9A, $AA, $AF, $48, $F6, $AA, $B8, $42, $50, $85, $8D, $7B, $CF, $91, $35, $C2, $B6, $ED, $10, $2A, $7C, $65, $60, $EF, $80, $D2, $47, $D1, $99, $11, $1D, $1A, $C3, $88, $E7 ), {E6}
    ( $8C, $CD, $D6, $42, $6F, $0E, $11, $B8, $67, $BD, $37, $78, $4E, $23, $AC, $E5, $F1, $25, $9D, $F3, $4B, $B1, $44, $4C, $88, $07, $BB, $09, $3D, $63, $39, $4F, $96, $57, $98, $34, $06, $5B, $F2, $FD, $76, $83, $9C, $8E, $30, $3D, $01, $C4 ), {E7}
    ( $04, $1D, $1F, $9B, $FD, $D3, $E4, $A6, $D8, $E9, $C1, $34, $B6, $77, $00, $50, $DF, $FE, $9B, $24, $E1, $25, $B8, $8A, $0D, $87, $8D, $D0, $55, $C8, $69, $FA, $E3, $34, $29, $55, $63, $9F, $52, $BC, $A2, $2D, $61, $48, $CE, $19, $E2, $C9 ), {E8}
    ( $A9, $F3, $57, $53, $F0, $F8, $44, $75, $B5, $32, $A0, $EE, $00, $8C, $E7, $9E, $E2, $F9, $54, $86, $0F, $FF, $C4, $70, $BA, $53, $B1, $4F, $10, $87, $CF, $56, $13, $27, $D2, $2D, $BE, $2E, $EF, $D3, $30, $5E, $85, $CE, $DA, $81, $70, $19 ), {E9}
    ( $BD, $2E, $6F, $72, $12, $22, $8D, $07, $F8, $F1, $1C, $23, $D8, $AC, $6D, $DD, $2F, $BA, $0C, $1A, $9F, $FE, $90, $F2, $D6, $44, $99, $AE, $01, $42, $87, $CC, $B3, $FB, $74, $13, $73, $FA, $3B, $3D, $AF, $55, $05, $CA, $D0, $71, $E4, $D5 ), {EA}
    ( $83, $BF, $B8, $A4, $1F, $08, $24, $8C, $3D, $93, $CC, $6A, $65, $0E, $A0, $26, $C2, $F1, $F1, $CE, $D2, $11, $81, $D8, $3F, $31, $E2, $EA, $85, $05, $9A, $07, $CD, $1B, $8E, $D2, $AE, $49, $F9, $23, $4F, $FD, $D5, $21, $23, $DD, $FF, $EC ), {EB}
    ( $B6, $0E, $E0, $A2, $34, $5F, $3E, $CD, $4A, $D1, $28, $B4, $6C, $5B, $4E, $41, $41, $E1, $70, $81, $3A, $EE, $D5, $9D, $02, $1C, $D6, $28, $6B, $C5, $E9, $E4, $3F, $9A, $12, $59, $0C, $1D, $E8, $20, $E8, $77, $8B, $EE, $DB, $A5, $C2, $D3 ), {EC}
    ( $43, $9C, $D3, $E4, $7D, $F5, $6D, $A0, $00, $3D, $63, $C7, $41, $2C, $AF, $92, $3F, $DF, $A7, $0D, $FD, $10, $07, $EA, $E3, $EC, $21, $DF, $03, $EF, $B9, $9A, $0C, $BF, $F5, $2A, $7C, $CA, $02, $51, $9A, $5A, $41, $08, $A3, $E0, $AF, $DC ), {ED}
    ( $B7, $54, $BC, $3E, $CF, $E3, $40, $29, $CF, $FC, $B0, $E6, $F9, $32, $53, $F5, $31, $9D, $03, $BD, $E4, $83, $52, $01, $5C, $37, $82, $62, $AE, $EC, $28, $AE, $B2, $E2, $AA, $E6, $B3, $C1, $40, $83, $BD, $58, $1D, $EB, $7D, $83, $59, $6D ), {EE}
    ( $9E, $AB, $05, $0E, $13, $BF, $98, $E5, $9E, $33, $E7, $D4, $D6, $C0, $F0, $36, $15, $7C, $6D, $59, $06, $56, $22, $B0, $B0, $3F, $F7, $75, $F5, $D6, $99, $F8, $14, $8E, $6A, $33, $65, $D7, $7B, $EB, $F8, $4C, $80, $80, $AA, $AB, $4B, $D7 ), {EF}
    ( $7F, $B5, $DE, $DB, $31, $2F, $23, $52, $75, $8D, $87, $5E, $87, $3D, $8C, $91, $90, $45, $59, $EF, $D8, $59, $C7, $B2, $E2, $AE, $5E, $1E, $F4, $24, $7E, $36, $46, $1D, $43, $04, $28, $83, $D3, $6D, $48, $A0, $9D, $E8, $00, $23, $CC, $A1 ), {F0}
    ( $5B, $C9, $3A, $B9, $47, $00, $3A, $97, $63, $D2, $C9, $F7, $76, $4E, $47, $0C, $7E, $1F, $89, $9B, $A4, $AD, $7E, $CB, $3C, $AB, $40, $27, $6D, $46, $DC, $C1, $50, $A6, $BD, $63, $BA, $C6, $69, $EE, $35, $E0, $AD, $50, $C9, $43, $AC, $6E ), {F1}
    ( $3C, $15, $B4, $A5, $9A, $42, $B8, $00, $6C, $7F, $8A, $54, $D3, $01, $2A, $E1, $FE, $EC, $8D, $11, $43, $2C, $8A, $9F, $6B, $78, $67, $39, $C0, $20, $A4, $77, $98, $B3, $CE, $B4, $41, $F4, $F7, $EA, $1E, $70, $65, $9A, $95, $91, $F1, $52 ), {F2}
    ( $A0, $84, $C7, $AD, $1D, $92, $58, $15, $6D, $0D, $80, $80, $9B, $74, $17, $13, $F4, $BF, $4A, $02, $1C, $2E, $66, $DE, $C7, $8A, $02, $B3, $97, $A8, $78, $FB, $6E, $E1, $AF, $CB, $57, $2A, $D4, $1E, $3E, $FE, $90, $9E, $28, $4D, $0D, $E5 ), {F3}
    ( $CA, $19, $F8, $F7, $E6, $64, $1F, $A9, $32, $1A, $AD, $4D, $14, $AD, $9A, $E4, $A6, $6A, $44, $1D, $72, $20, $AB, $D6, $6C, $C8, $4F, $F1, $A8, $5F, $A0, $D5, $BF, $C7, $B4, $B9, $71, $B2, $05, $5A, $E7, $04, $7F, $44, $9E, $AA, $2B, $95 ), {F4}
    ( $0C, $16, $FD, $49, $E2, $39, $E9, $CE, $93, $30, $23, $8B, $A4, $73, $8F, $39, $63, $6C, $80, $3B, $5E, $C4, $3C, $C9, $5A, $43, $B2, $E3, $7C, $48, $76, $47, $BC, $69, $21, $7D, $F6, $60, $67, $6E, $BA, $98, $46, $8A, $3F, $70, $93, $59 ), {F5}
    ( $B0, $41, $40, $01, $8A, $87, $85, $D0, $E5, $E0, $F8, $66, $B5, $29, $C8, $D5, $AE, $4A, $DF, $EE, $D4, $E4, $C6, $3F, $26, $76, $61, $04, $ED, $2B, $1F, $DD, $48, $79, $E4, $D5, $12, $51, $06, $AB, $91, $CA, $D4, $FE, $17, $2B, $1B, $A0 ), {F6}
    ( $1A, $69, $04, $C0, $BE, $B7, $0B, $03, $57, $C4, $30, $89, $A0, $BA, $D5, $C3, $4C, $A5, $F5, $3E, $40, $45, $B3, $DB, $86, $3D, $29, $45, $47, $67, $E2, $5E, $E8, $92, $BF, $35, $A0, $96, $1E, $A4, $5D, $7B, $5E, $FC, $B1, $29, $89, $25 ), {F7}
    ( $A7, $FF, $9D, $88, $11, $60, $8F, $FE, $73, $E8, $6A, $19, $16, $D6, $32, $11, $DE, $5D, $E2, $BE, $B1, $22, $8D, $6F, $E9, $57, $24, $C9, $95, $9A, $3F, $40, $85, $30, $4A, $89, $1D, $FB, $DA, $B1, $1C, $CE, $53, $F9, $40, $79, $5D, $46 ), {F8}
    ( $7E, $9B, $C6, $9D, $53, $D8, $B7, $55, $2F, $0E, $FF, $0D, $51, $AA, $6F, $7A, $24, $E7, $61, $4B, $C7, $38, $F8, $AC, $F0, $73, $D0, $0B, $28, $6C, $AC, $1B, $24, $5F, $F8, $20, $BF, $E7, $2A, $C7, $4A, $08, $75, $91, $78, $10, $CD, $9D ), {F9}
    ( $75, $38, $9E, $0D, $6D, $09, $C7, $38, $DA, $85, $89, $F4, $67, $E7, $1A, $CD, $C0, $63, $48, $B7, $EB, $31, $AE, $D5, $EE, $5E, $56, $E9, $EF, $A1, $C8, $4E, $C2, $C6, $C8, $B6, $B1, $33, $03, $02, $2D, $A7, $DC, $E9, $C0, $39, $7B, $00 ), {FA}
    ( $91, $1C, $3D, $13, $C8, $75, $EF, $64, $56, $CD, $CE, $B6, $CC, $43, $B2, $9B, $51, $87, $AD, $A6, $83, $5A, $FF, $90, $0B, $49, $CE, $BB, $27, $30, $82, $CB, $F0, $98, $FC, $3F, $C3, $E8, $A1, $97, $DE, $EE, $3F, $10, $BD, $5D, $A3, $76 ), {FB}
    ( $48, $14, $86, $05, $65, $3A, $81, $E4, $51, $25, $5C, $8C, $34, $18, $38, $F1, $18, $16, $2A, $C8, $28, $60, $50, $A6, $BF, $28, $FB, $FB, $C8, $77, $CC, $85, $D9, $00, $55, $FE, $01, $CF, $7D, $ED, $12, $BE, $6E, $EC, $CA, $25, $92, $0A ), {FC}
    ( $D5, $22, $08, $E5, $8B, $AC, $27, $95, $8E, $9F, $3B, $92, $18, $16, $73, $EC, $70, $B8, $4E, $DA, $74, $16, $2E, $BF, $55, $7E, $E5, $7F, $39, $D0, $3D, $14, $90, $BD, $83, $26, $D5, $0B, $57, $74, $85, $74, $A2, $01, $8D, $B8, $3B, $44 ), {FD}
    ( $AF, $67, $1E, $09, $92, $61, $5E, $4D, $0F, $68, $46, $BB, $62, $83, $7B, $B8, $A4, $3D, $A3, $9C, $39, $3C, $2A, $32, $16, $5C, $12, $AA, $C3, $F3, $BF, $B6, $FC, $94, $66, $43, $E1, $9C, $A2, $56, $B9, $1D, $CD, $18, $F6, $E5, $55, $88 ), {FE}
    ( $85, $96, $6C, $BD, $72, $71, $EA, $3B, $33, $76, $69, $88, $69, $94, $02, $B4, $20, $5E, $2E, $A5, $02, $CD, $8B, $98, $11, $40, $D1, $77, $F3, $7B, $FC, $1F, $ED, $72, $C6, $E5, $25, $8C, $92, $5D, $8C, $12, $F6, $1F, $43, $88, $2C, $E9 )  {FF}
  );

{ ---------------------------------------------- }
{ Public Functions                               }
{ ---------------------------------------------- }

constructor TMHash384.Create();
begin
  Reset(); {Initialize!}
end;

procedure TMHash384.Update(const input: Byte);
var
  i: Cardinal;
  ptr_xor, ptr_mix, ptr_rnd, ptr_src, ptr_dst: PDigestBuf;
  val: Byte;
begin
  ptr_xor := @TABLE_XOR[input];
  ptr_mix := @TABLE_MIX[m_rnd];
  ptr_rnd := @TABLE_RND[m_rnd];
  ptr_src := @m_digest[    m_rnd and 1];
  ptr_dst := @m_digest[not m_rnd and 1];
  for i := 0 to MHASH384_LEN-1 do
  begin
    val := (ptr_src[ptr_mix[i]] xor ptr_xor[i]) + ptr_rnd[i];
    ptr_dst[i] := ptr_dst[i] xor TABLE_SBX[val][i];
  end;
  m_rnd := (m_rnd + 1) and $FF
end;

procedure TMHash384.Update(const input: TByteArray);
var
  i: Cardinal;
begin
  for i := 0 to length(input) do
  begin
    Update(input[i]);
  end;
end;

procedure TMHash384.Update(const input: TByteArray; const offset: LongWord; const len: LongWord);
var
  i, limit: Cardinal;
begin
  limit := offset + len - 1;
  for i := offset to limit do
  begin
    Update(input[i]);
  end;
end;

procedure TMHash384.Update(const input: String);
var
  i: Cardinal;
begin
  for i := 1 to Length(input) do
  begin
    Update(Byte(input[i]));
  end;
end;

procedure TMHash384.Update(const input: TStream);
var
  i, n: Cardinal;
  buffer: array[0..4095] of Byte;
begin
  repeat
  begin
    n := input.Read(buffer, Length(buffer));
    if n > 0 then
    begin
      for i := 0 to n-1 do
      begin
        Update(buffer[i]);
      end;
    end;
  end;
  until n < Cardinal(Length(buffer));
end;

function TMHash384.Update(const input: TStream; const count: Cardinal): Cardinal;
var
  i, n, remain: Cardinal;
  buffer: array[0..4095] of Byte;
begin
  remain := count;
  repeat
  begin
    n := input.Read(buffer, Min(remain, Length(buffer)));
    if n > 0 then
    begin
      remain := remain - n;
      for i := 0 to n-1 do
      begin
        Update(buffer[i]);
      end;
    end;
  end;
  until (n < Cardinal(Length(buffer))) or (count <= 0);
  Result := count - remain;
end;

procedure TMHash384.Result(var output: TByteArray);
var
  i: Cardinal;
  ptr_xor, ptr_mix, ptr_rnd, ptr_src, ptr_dst: PDigestBuf;
  val: Byte;
begin
  SetLength(output, MHASH384_LEN);
  ptr_xor := @TABLE_XOR[256];
  ptr_mix := @TABLE_MIX[m_rnd];
  ptr_rnd := @TABLE_RND[m_rnd];
  ptr_src := @m_digest[    m_rnd and 1];
  ptr_dst := @m_digest[not m_rnd and 1];
  for i := 0 to MHASH384_LEN-1 do
  begin
    val := (ptr_src[ptr_mix[i]] xor ptr_xor[i]) + ptr_rnd[i];
    output[i] := ptr_dst[i] xor TABLE_SBX[val][i];
  end;
end;

procedure TMHash384.Reset();
begin
  m_rnd := 0;
  Move(TABLE_INI[0], m_digest[0], MHASH384_LEN);
  Move(TABLE_INI[1], m_digest[1], MHASH384_LEN);
end;

class procedure TMHash384.GetVersion(var major: Word; var minor: Word; var patch: Word);
begin
  major := VERSION_MAJOR;
  minor := VERSION_MINOR;
  patch := VERSION_PATCH;
end;

{ ---------------------------------------------- }
{ Debugging Functions                            }
{ ---------------------------------------------- }

function GetProgress(const total: Integer; const current: Integer): Integer;
var
  fTotal, fCurrent: Double;
begin
  fTotal := total; fCurrent := current;
  Result := Round((fCurrent / fTotal) * 100.0);
end;

class procedure TMHash384.SelfTest(const callback: TProgress; const abortFlag: TSimpleEvent);
type
  TTestVector = record
    iterations: Cardinal;
    input: String;
  end;
const
  TEST_VECTOR: array[0..15] of TTestVector =
  (
    (
      iterations: $0000001;
      input: '';
    ),
    (
      iterations: $0000001;
      input: 'abc';
    ),
    (
      iterations: $0000001;
      input: 'abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq';
    ),
    (
      iterations: $0000001;
      input: 'abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu';
    ),
    (
      iterations: $0000001;
      input: 'Franz jagt im komplett verwahrlosten Taxi quer durch Bayern';
    ),
    (
      iterations: $0000001;
      input: 'Frank jagt im komplett verwahrlosten Taxi quer durch Bayern';
    ),
    (
      iterations: $0000001;
      input:
            'Lorem ipsum dolor sit amet, consetetur sadipscing elitr, sed diam nonumy eirmod tempor invidunt ut labore et dolore magna aliquyam erat, sed diam voluptua. At vero eos et accusam et justo duo dolores et ea rebum. Stet clita kasd gubergren, no sea takimata' +
            ' sanctus est Lorem ipsum dolor sit amet. Lorem ipsum dolor sit amet, consetetur sadipscing elitr, sed diam nonumy eirmod tempor invidunt ut labore et dolore magna aliquyam erat, sed diam voluptua. At vero eos et accusam et justo duo dolores et ea rebum. S' +
            'tet clita kasd gubergren, no sea takimata sanctus est Lorem ipsum dolor sit amet. Lorem ipsum dolor sit amet, consetetur sadipscing elitr, sed diam nonumy eirmod tempor invidunt ut labore et dolore magna aliquyam erat, sed diam voluptua. At vero eos et ac' +
            'cusam et justo duo dolores et ea rebum. Stet clita kasd gubergren, no sea takimata sanctus est Lorem ipsum dolor sit amet.'#$A'Duis autem vel eum iriure dolor in hendrerit in vulputate velit esse molestie consequat, vel illum dolore eu feugiat nulla facil' +
            'isis at vero eros et accumsan et iusto odio dignissim qui blandit praesent luptatum zzril delenit augue duis dolore te feugait nulla facilisi. Lorem ipsum dolor sit amet, consectetuer adipiscing elit, sed diam nonummy nibh euismod tincidunt ut laoreet dol' +
            'ore magna aliquam erat volutpat.'#$A'Ut wisi enim ad minim veniam, quis nostrud exerci tation ullamcorper suscipit lobortis nisl ut aliquip ex ea commodo consequat. Duis autem vel eum iriure dolor in hendrerit in vulputate velit esse molestie consequat, v' +
            'el illum dolore eu feugiat nulla facilisis at vero eros et accumsan et iusto odio dignissim qui blandit praesent luptatum zzril delenit augue duis dolore te feugait nulla facilisi.'#$A'Nam liber tempor cum soluta nobis eleifend option congue nihil imperdi' +
            'et doming id quod mazim placerat facer possim assum. Lorem ipsum dolor sit amet, consectetuer adipiscing elit, sed diam nonummy nibh euismod tincidunt ut laoreet dolore magna aliquam erat volutpat. Ut wisi enim ad minim veniam, quis nostrud exerci tation ' +
            'ullamcorper suscipit lobortis nisl ut aliquip ex ea commodo consequat.'#$A'Duis autem vel eum iriure dolor in hendrerit in vulputate velit esse molestie consequat, vel illum dolore eu feugiat nulla facilisis.'#$A'At vero eos et accusam et justo duo dolore' +
            's et ea rebum. Stet clita kasd gubergren, no sea takimata sanctus est Lorem ipsum dolor sit amet. Lorem ipsum dolor sit amet, consetetur sadipscing elitr, sed diam nonumy eirmod tempor invidunt ut labore et dolore magna aliquyam erat, sed diam voluptua. A' +
            't vero eos et accusam et justo duo dolores et ea rebum. Stet clita kasd gubergren, no sea takimata sanctus est Lorem ipsum dolor sit amet. Lorem ipsum dolor sit amet, consetetur sadipscing elitr, At accusam aliquyam diam diam dolore dolores duo eirmod eos' +
            ' erat, et nonumy sed tempor et et invidunt justo labore Stet clita ea et gubergren, kasd magna no rebum. sanctus sea sed takimata ut vero voluptua. est Lorem ipsum dolor sit amet. Lorem ipsum dolor sit amet, consetetur sadipscing elitr, sed diam nonumy ei' +
            'rmod tempor invidunt ut labore et dolore magna aliquyam erat.'#$A'Consetetur sadipscing elitr, sed diam nonumy eirmod tempor invidunt ut labore et dolore magna aliquyam erat, sed diam voluptua. At vero eos et accusam et justo duo dolores et ea rebum. Stet' +
            ' clita kasd gubergren, no sea takimata sanctus est Lorem ipsum dolor sit amet. Lorem ipsum dolor sit amet, consetetur sadipscing elitr, sed diam nonumy eirmod tempor invidunt ut labore et dolore magna aliquyam erat, sed diam voluptua. At vero eos et accus' +
            'am et justo duo dolores et ea rebum. Stet clita kasd gubergren, no sea takimata sanctus est Lorem ipsum dolor sit amet. Lorem ipsum dolor sit amet, consetetur sadipscing elitr, sed diam nonumy eirmod tempor invidunt ut labore et dolore magna aliquyam erat' +
            ', sed diam voluptua. At vero eos et accusam et justo duo dolores et ea rebum. Stet clita kasd gubergren, no sea takimata sanctus.'#$A'Lorem ipsum dolor sit amet, consetetur sadipscing elitr, sed diam nonumy eirmod tempor invidunt ut labore et dolore magna' +
            ' aliquyam erat, sed diam voluptua. At vero eos et accusam et justo duo dolores et ea rebum. Stet clita kasd gubergren, no sea takimata sanctus est Lorem ipsum dolor sit amet. Lorem ipsum dolor sit amet, consetetur sadipscing elitr, sed diam nonumy eirmod ' +
            'tempor invidunt ut labore et dolore magna aliquyam erat, sed diam voluptua. At vero eos et accusam et justo duo dolores et ea rebum. Stet clita kasd gubergren, no sea takimata sanctus est Lorem ipsum dolor sit amet. Lorem ipsum dolor sit amet, consetetur ' +
            'sadipscing elitr, sed diam nonumy eirmod tempor invidunt ut labore et dolore magna aliquyam erat, sed diam voluptua. At vero eos et accusam et justo duo dolores et ea rebum. Stet clita kasd gubergren, no sea takimata sanctus est Lorem ipsum dolor sit amet' +
            '.'#$A'Duis autem vel eum iriure dolor in hendrerit in vulputate velit esse molestie consequat, vel illum dolore eu feugiat nulla facilisis at vero eros et accumsan et iusto odio dignissim qui blandit praesent luptatum zzril delenit augue duis dolore te fe' +
            'ugait nulla facilisi. Lorem ipsum dolor sit amet, consectetuer adipiscing elit, sed diam nonummy nibh euismod tincidunt ut laoreet dolore magna aliquam erat volutpat.'#$A'Ut wisi enim ad minim veniam, quis nostrud exerci tation ullamcorper suscipit lobort' +
            'is nisl ut aliquip ex ea commodo consequat. Duis autem vel eum iriure dolor in hendrerit in vulputate velit esse molestie consequat, vel illum dolore eu feugiat nulla facilisis at vero eros et accumsan et iusto odio dignissim qui blandit praesent luptatum' +
            ' zzril delenit augue duis dolore te feugait nulla facilisi.'#$A'Nam liber tempor cum soluta nobis eleifend option congue nihil imperdiet doming id quod mazim placerat facer possim assum. Lorem ipsum dolor sit amet, consectetuer adipiscing elit, sed diam n' +
            'onummy nibh euismod tincidunt ut laoreet dolore magna aliquam erat volutpat. Ut wisi enim ad minim veniam, quis nostrud exerci tation ullamcorper suscipit lobortis nisl ut aliquip ex ea commodo.';
    ),
    (
      iterations: $0000001;
      input:
            #$92#$AD#$AC#$49#$7F#$FA#$B3#$0B#$40#$AC#$67#$4F#$9D#$53#$A9#$A6#$24#$B2#$47#$FA#$23#$72#$94#$FA#$75#$88#$41#$A8#$5F#$4D#$DE#$FD#$DB#$FD#$23#$71#$C1#$E2#$CA#$F8#$5B#$0E#$06#$4B#$BC#$4F#$3A#$C0 +
            #$A9#$C6#$FD#$6C#$18#$5D#$C4#$B1#$2C#$8A#$AC#$C3#$E6#$25#$BD#$E8#$7A#$DF#$CB#$FF#$60#$1B#$F0#$66#$DF#$08#$7C#$0A#$63#$CE#$2C#$11#$D6#$64#$8D#$1E#$D0#$57#$CE#$1B#$5D#$93#$C6#$CB#$6F#$33#$19#$81 +
            #$B9#$FC#$B5#$BC#$F4#$FB#$6F#$77#$50#$54#$06#$D7#$1A#$87#$EC#$7E#$C1#$99#$55#$97#$FB#$B7#$96#$9D#$EC#$0D#$AD#$01#$93#$9E#$88#$89#$F5#$30#$96#$3F#$E9#$2E#$23#$A8#$F5#$DA#$E5#$47#$3E#$2C#$1E#$22 +
            #$38#$2A#$D4#$F7#$0E#$A2#$E4#$5E#$BB#$CB#$AD#$11#$C8#$7A#$74#$8F#$78#$CB#$56#$77#$8E#$F4#$1C#$0A#$EC#$A5#$63#$2E#$75#$E3#$52#$B2#$EA#$B5#$53#$3C#$06#$F7#$C6#$80#$8E#$77#$D0#$0B#$2D#$B5#$F9#$64 +
            #$F5#$9F#$89#$83#$D9#$25#$79#$22#$26#$E2#$F3#$E6#$D4#$B0#$FD#$E7#$BA#$74#$6C#$59#$BA#$9A#$20#$21#$2C#$25#$DA#$02#$D2#$3A#$EE#$5F#$88#$5A#$75#$F1#$1A#$C5#$BD#$D5#$CC#$6F#$9C#$4B#$72#$BC#$47#$D8 +
            #$40#$89#$72#$00#$F1#$B7#$EF#$97#$7F#$23#$52#$2C#$83#$38#$A3#$FC#$FD#$D7#$72#$C6#$15#$D3#$4A#$93#$46#$BC#$B1#$CE#$37#$14#$FC#$0E#$2B#$CB#$99#$C7#$34#$4D#$AC#$D8#$48#$A1#$EE#$58#$43#$6D#$B4#$78 +
            #$D0#$30#$E4#$09#$F1#$7B#$3A#$E9#$60#$68#$D9#$06#$F8#$59#$22#$6A#$66#$47#$D6#$09#$8A#$95#$18#$3B#$01#$46#$0D#$29#$36#$7B#$A5#$22#$B4#$05#$7E#$30#$AC#$8B#$C2#$D5#$4C#$BB#$63#$50#$0B#$76#$A1#$36 +
            #$EB#$36#$A0#$77#$D8#$F7#$98#$14#$57#$91#$0E#$8D#$9C#$42#$95#$E9#$39#$42#$01#$C9#$B8#$DF#$82#$87#$FF#$E5#$E6#$66#$80#$0B#$19#$B4#$13#$AC#$EC#$71#$2A#$5D#$35#$C5#$CA#$0F#$56#$E6#$A3#$51#$70#$58 +
            #$21#$B2#$20#$8D#$C9#$DE#$CC#$25#$19#$D5#$CF#$A1#$B9#$2B#$68#$BE#$D2#$64#$32#$94#$9B#$02#$7C#$F2#$00#$FE#$3D#$8B#$40#$A2#$19#$50#$0F#$EE#$41#$61#$23#$A8#$C5#$A3#$F5#$FC#$37#$35#$1E#$6A#$51#$42 +
            #$AA#$33#$8D#$34#$11#$F8#$B3#$BD#$2B#$74#$99#$16#$FF#$AD#$26#$8F#$00#$C5#$F5#$1C#$59#$22#$02#$3E#$B1#$71#$08#$43#$BF#$A7#$6B#$CD#$03#$33#$0E#$53#$9D#$BA#$E5#$28#$CA#$09#$E6#$1B#$AE#$11#$76#$A9 +
            #$7A#$89#$DF#$48#$DC#$85#$F1#$9A#$ED#$9F#$E1#$9F#$2D#$B7#$DA#$21#$BA#$D1#$0C#$CA#$67#$05#$5B#$CC#$24#$5B#$EA#$F0#$B8#$6C#$C4#$B8;
    ),
    (
      iterations: $0000001;
      input:
            #$05#$85#$7B#$AA#$C0#$C9#$E8#$62#$60#$62#$B0#$A0#$22#$03#$88#$A6#$84#$43#$AA#$8C#$77#$0C#$82#$94#$5D#$35#$82#$98#$04#$1C#$DD#$A8#$D4#$4E#$90#$6F#$6D#$B9#$81#$E2#$7B#$3F#$95#$74#$24#$3E#$9F#$53 +
            #$7B#$EF#$0A#$25#$47#$79#$89#$5C#$B6#$B2#$37#$FC#$4F#$74#$6F#$7F#$CD#$E3#$73#$8B#$F5#$D1#$A2#$3C#$BC#$70#$F1#$41#$84#$F2#$EB#$37#$9B#$CB#$C3#$D7#$46#$2E#$52#$C3#$97#$3A#$B6#$74#$F3#$D5#$BB#$5A +
            #$89#$36#$F3#$1D#$D5#$A1#$0E#$F1#$8E#$F4#$17#$26#$CE#$66#$00#$28#$DE#$17#$3D#$5B#$3B#$83#$7B#$D3#$AB#$C4#$A1#$EF#$8A#$59#$A2#$E2#$BA#$6E#$13#$EE#$EA#$83#$6B#$C1#$B1#$17#$9C#$65#$1A#$BE#$45#$A7 +
            #$B2#$40#$91#$99#$07#$9F#$4B#$5F#$BE#$F8#$C6#$7E#$4E#$0A#$FF#$5B#$B5#$5A#$7C#$2F#$9E#$77#$EE#$3B#$E0#$B5#$62#$80#$22#$9D#$2D#$13#$CB#$BC#$33#$80#$DD#$AB#$15#$AA#$AD#$04#$3A#$AD#$8C#$D5#$32#$4D +
            #$30#$32#$F0#$71#$47#$27#$5E#$4B#$12#$FB#$EF#$7B#$AB#$83#$AC#$9D#$CD#$ED#$EE#$2C#$25#$69#$19#$6D#$60#$EF#$18#$B5#$A0#$80#$9D#$A2#$83#$31#$87#$07#$53#$52#$E1#$5B#$4C#$1E#$FF#$41#$61#$34#$1C#$3E +
            #$1F#$5F#$A2#$58#$D7#$67#$56#$23#$B6#$EC#$F5#$BC#$27#$CE#$CD#$B4#$B2#$57#$B1#$3B#$EB#$A6#$58#$95#$E0#$B5#$E8#$B1#$9E#$18#$3F#$4B#$60#$96#$4E#$FB#$59#$25#$06#$F4#$95#$83#$80#$98#$5D#$9F#$59#$9B +
            #$83#$26#$97#$89#$CE#$AC#$BB#$D0#$5C#$95#$37#$DC#$E4#$DD#$2F#$5C#$22#$9C#$65#$10#$FB#$61#$A1#$73#$A1#$6A#$89#$61#$93#$7C#$C8#$6E#$80#$B8#$48#$57#$48#$4F#$88#$7E#$6A#$CE#$90#$F3#$EB#$B3#$9F#$50 +
            #$F0#$0F#$B6#$29#$60#$28#$E5#$8B#$DF#$EA#$CA#$FB#$3C#$DD#$AA#$60#$60#$73#$B3#$AE#$45#$4F#$C9#$1D#$B2#$63#$31#$30#$3B#$85#$BA#$8A#$37#$41#$D6#$5A#$52#$11#$A2#$2A#$11#$39#$D2#$1E#$2D#$BF#$1A#$53 +
            #$8A#$13#$24#$C7#$A3#$DE#$BE#$0B#$23#$72#$CC#$92#$29#$B8#$2B#$92#$8F#$08#$0A#$B4#$A6#$47#$76#$29#$39#$BF#$40#$BC#$59#$9F#$CD#$9D#$CF#$35#$51#$16#$F5#$44#$40#$A4#$C3#$1D#$A8#$8C#$73#$C0#$A1#$12 +
            #$84#$15#$4D#$B1#$7B#$4A#$44#$F4#$12#$8E#$0F#$E4#$4C#$7E#$5C#$B2#$92#$5F#$A9#$B3#$71#$DA#$40#$50#$78#$C7#$02#$1F#$20#$79#$84#$F3#$91#$1F#$47#$47#$40#$67#$2B#$52#$A1#$67#$91#$05#$CA#$B4#$AB#$A8 +
            #$69#$40#$FD#$7F#$28#$F6#$17#$B0#$DD#$1A#$82#$0F#$20#$5D#$D2#$39#$A8#$9A#$73#$F1#$AC#$10#$18#$3C#$69#$99#$49#$B8#$B2#$90#$24#$B5#$2E#$0C#$57#$93#$6A#$1B#$0B#$9C#$B9#$F5#$45#$71#$DD#$AF#$94#$6F +
            #$A5#$F5#$CF#$B1#$BC#$1C#$0E#$5E#$5A#$0E#$F6#$9D#$9E#$71#$11#$99#$9F#$A4#$57#$E6#$6D#$D5#$83#$00#$D7#$85#$0C#$54#$01#$D4#$49#$1C#$F6#$8C#$8E#$00#$8B#$F8#$48#$50#$4D#$6F#$F1#$EA#$EF#$2D#$64#$D4 +
            #$F5#$00#$A9#$03#$E6#$CB#$5F#$CF#$6C#$D6#$26#$C1#$F4#$97#$CF#$B5#$16#$A1#$62#$DE#$FF#$C2#$00#$D0#$9A#$5E#$BB#$B4#$76#$86#$FD#$4B#$F2#$29#$4F#$0F#$69#$15#$7D#$AC#$AA#$2B#$D1#$BA#$6E#$A2#$1A#$C1 +
            #$6B#$9A#$14#$90#$63#$BD#$C4#$31#$36#$BB#$26#$A7#$69#$7D#$11#$76#$E5#$3F#$05#$C5#$4F#$6D#$B0#$12#$2E#$BC#$72#$18#$8A#$0F#$07#$A1#$2A#$D0#$C0#$DE#$2B#$63#$19#$6E#$A8#$7F#$C7#$9D#$47#$2F#$A6#$62 +
            #$A3#$57#$F0#$37#$58#$9F#$80#$4D#$9D#$A9#$2F#$2A#$39#$79#$16#$6A#$CA#$DF#$22#$8F#$2F#$AA#$A2#$72#$75#$96#$63#$19#$E2#$DF#$7E#$37#$1D#$68#$F8#$D9#$5D#$8D#$40#$23#$7F#$F5#$1B#$40#$33#$CB#$38#$81 +
            #$4F#$4A#$5E#$25#$3C#$F0#$19#$D7#$01#$D1#$21#$F6#$C5#$6F#$DA#$17#$83#$60#$39#$25#$66#$F2#$25#$31#$58#$A6#$A4#$AD#$46#$78#$E7#$76#$57#$19#$CB#$7C#$9C#$6D#$E0#$6D#$F8#$FD#$28#$14#$14#$0D#$F0#$0C +
            #$8F#$D4#$68#$59#$D2#$96#$15#$BF#$EC#$8A#$34#$F7#$1A#$3B#$2C#$8C#$4A#$98#$16#$37#$7D#$95#$15#$BD#$45#$CB#$F8#$C1#$A1#$9E#$64#$3A#$17#$4C#$C4#$D2#$D1#$A0#$A4#$BD#$5A#$79#$CB#$53#$BC#$37#$84#$C8 +
            #$4F#$9B#$D7#$10#$42#$3D#$26#$40#$D6#$F2#$50#$20#$17#$8C#$2A#$81#$3A#$B5#$D2#$8B#$66#$E5#$9F#$CB#$EB#$29#$C8#$41#$D1#$B1#$9F#$BA#$77#$2A#$18#$D7#$7F#$8D#$46#$60#$FE#$DB#$E4#$C1#$A5#$73#$DD#$88 +
            #$7B#$17#$AA#$3D#$58#$30#$9B#$F6#$B2#$66#$4E#$AC#$C3#$E3#$A4#$7A#$AE#$DF#$33#$BE#$52#$89#$B7#$F9#$B0#$18#$FB#$70#$63#$9A#$C2#$B5#$FD#$EF#$3C#$8E#$91#$69#$5F#$A8#$E0#$F6#$83#$25#$EF#$16#$AA#$AF +
            #$27#$B5#$D5#$2C#$58#$57#$35#$67#$5D#$11#$31#$A3#$25#$39#$0D#$5C#$F1#$AC#$D4#$FB#$03#$4C#$8A#$C0#$FB#$B2#$E5#$0E#$AD#$7F#$C9#$B1#$33#$AE#$D9#$EE#$3D#$71#$CB#$23#$54#$F9#$7F#$BF#$02#$4D#$E1#$D2 +
            #$BC#$66#$A9#$5C#$9D#$36#$47#$5A#$62#$76#$EF#$C5#$C8#$30#$BC#$78#$79#$92#$62#$A8#$E9#$0B#$83#$E5#$91#$42#$8F#$FC#$E9#$A8#$6C#$AB#$A9#$75#$A1#$9B#$6B#$A2#$E5#$87#$96#$BE#$16#$84#$0A#$04#$29#$9F +
            #$2E#$2D#$C9#$C8#$DA#$EA#$F6#$9A#$25#$0F#$F8#$C0#$FD#$B5#$D7#$82;
    ),
    (
      iterations: $0000001;
      input:
            #$FF#$D3#$B0#$A7#$2D#$6F#$68#$3A#$0B#$80#$75#$AA#$E0#$E7#$1E#$77#$32#$81#$7D#$74#$65#$EE#$85#$B5#$7A#$D8#$D6#$53#$CE#$9F#$4F#$B7#$BD#$D4#$B0#$B6#$7B#$44#$A5#$2C#$2E#$69#$C9#$60#$E3#$AE#$C7#$AD +
            #$66#$04#$DC#$1D#$B1#$AE#$B2#$AE#$5E#$F0#$78#$CB#$08#$FF#$40#$1C#$36#$34#$0C#$10#$E3#$41#$9B#$80#$8F#$62#$BF#$D6#$25#$B7#$2D#$74#$83#$4F#$24#$7E#$BA#$B7#$0D#$87#$9A#$5C#$72#$E6#$F0#$09#$79#$51 +
            #$2B#$1F#$E6#$4A#$C0#$22#$17#$58#$66#$E9#$8B#$B2#$92#$E3#$62#$FC#$73#$6E#$BC#$72#$D5#$CF#$08#$05#$4B#$B6#$14#$3C#$91#$33#$3E#$60#$BA#$81#$C3#$FD#$32#$A9#$B7#$64#$D8#$1F#$9B#$44#$63#$2D#$1A#$70 +
            #$E1#$F5#$4E#$A0#$8D#$3C#$20#$8C#$46#$59#$F5#$F4#$E2#$E6#$A2#$A7#$E8#$18#$CB#$4D#$DB#$5A#$C6#$90#$93#$AC#$DF#$6A#$FF#$D1#$00#$96#$4F#$FC#$EF#$B7#$6A#$EB#$BB#$87#$30#$4F#$FC#$19#$CE#$12#$33#$C1 +
            #$13#$DE#$DF#$A3#$7B#$89#$C2#$A3#$DC#$A4#$A9#$BD#$04#$10#$96#$EE#$B8#$77#$FA#$98#$5C#$42#$01#$F3#$F5#$D4#$EF#$18#$0E#$B4#$0F#$2E#$2C#$FC#$9A#$D0#$CF#$4C#$99#$E1#$20#$F0#$65#$5B#$FE#$AF#$F0#$9B +
            #$B5#$DF#$D5#$2B#$C2#$3D#$B0#$FD#$E8#$31#$3C#$7F#$A6#$65#$6F#$BA#$0E#$D9#$C9#$8D#$5B#$D1#$EA#$7C#$7F#$4B#$52#$ED#$80#$4F#$26#$17#$95#$91#$6B#$B8#$08#$2C#$1E#$62#$4F#$D2#$D0#$7A#$00#$75#$A6#$13 +
            #$29#$93#$CB#$C4#$C5#$A7#$9E#$90#$B1#$7C#$8C#$1C#$7A#$59#$6C#$1D#$5D#$FC#$46#$AF#$0C#$96#$BA#$31#$95#$44#$7B#$CB#$5E#$AB#$64#$42#$F7#$35#$7A#$1F#$87#$F2#$B3#$3E#$42#$33#$28#$CF#$52#$47#$47#$13 +
            #$9A#$A3#$63#$D1#$BD#$7A#$F0#$0C#$90#$68#$1D#$0C#$03#$83#$A9#$8E#$9D#$7A#$5C#$6A#$1E#$66#$A9#$F1#$2B#$B8#$CE#$A2#$D2#$2B#$17#$21#$B1#$83#$87#$FC#$F1#$F3#$46#$78#$DC#$D3#$96#$66#$C2#$B4#$D3#$83 +
            #$02#$A8#$D7#$A5#$95#$DF#$30#$04#$F2#$D1#$52#$8C#$5A#$44#$62#$68#$BC#$CD#$A7#$69#$CB#$4E#$B4#$5B#$10#$83#$59#$AC#$28#$29#$13#$04#$C3#$B2#$7A#$C5#$35#$A7#$B0#$0A#$D4#$F0#$F0#$EC#$B4#$C0#$F2#$D2 +
            #$C2#$2B#$88#$03#$D5#$E6#$4E#$6F#$EF#$AD#$7D#$D8#$0F#$F0#$6C#$B0#$B1#$09#$D8#$9D#$99#$14#$E3#$00#$05#$23#$C7#$15#$B9#$B1#$B7#$F1#$EE#$9F#$CE#$04#$39#$B7#$E2#$4D#$0C#$5A#$13#$7A#$72#$08#$21#$11 +
            #$28#$B2#$35#$EC#$7A#$E5#$23#$D5#$66#$D6#$4C#$7A#$82#$FD#$6E#$1F#$3A#$01#$DC#$38#$3A#$21#$7B#$23#$B5#$FA#$CC#$05#$65#$8D#$DC#$DA#$5C#$A7#$0A#$1A#$A0#$02#$F4#$B6#$A0#$32#$34#$77#$C5#$E2#$BB#$10 +
            #$46#$CE#$57#$E3#$A9#$0B#$65#$73#$8F#$D5#$A5#$17#$83#$00#$BD#$40#$B1#$EE#$44#$F4#$EB#$28#$5B#$96#$A6#$C1#$94#$D3#$B6#$4B#$AE#$F4#$72#$F6#$EC#$AC#$D4#$52#$9E#$C7#$7A#$A8#$4B#$3D#$1D#$BC#$C0#$57 +
            #$7B#$91#$62#$EE#$4D#$67#$F3#$BA#$8D#$BB#$6D#$E7#$48#$5E#$BE#$16#$48#$85#$6C#$52#$48#$00#$FF#$06#$95#$E1#$7E#$D9#$13#$98#$E0#$6D#$96#$13#$B7#$4B#$28#$3B#$AC#$C5#$0A#$37#$85#$BB#$6E#$E5#$E9#$2F +
            #$2C#$0E#$9E#$B7#$44#$E7#$85#$F2#$5B#$FD#$EE#$8B#$C2#$B8#$30#$E0#$81#$29#$94#$DB#$33#$FF#$88#$32#$91#$F9#$CB#$14#$BC#$2D#$91#$26#$13#$4A#$A2#$E2#$D7#$D0#$CA#$9F#$1F#$36#$7A#$12#$7D#$0C#$9A#$0B +
            #$E6#$5C#$E2#$B7#$4D#$E5#$0B#$A4#$3E#$25#$F8#$D9#$8D#$9D#$F4#$E3#$2C#$8A#$9B#$98#$AD#$6B#$F5#$0F#$FB#$7D#$DA#$2A#$FE#$21#$18#$76#$9A#$1A#$55#$FC#$25#$CA#$3E#$F9#$FD#$E4#$B6#$2D#$10#$3E#$58#$79 +
            #$D8#$30#$ED#$DD#$4B#$B6#$0F#$D5#$D6#$F6#$B4#$B1#$37#$7C#$4E#$29#$E6#$F7#$79#$23#$74#$01#$66#$5F#$02#$EB#$F2#$9F#$51#$02#$39#$31#$AA#$8B#$BC#$DE#$6E#$5A#$6D#$AF#$3A#$84#$23#$34#$F4#$13#$9C#$04 +
            #$FD#$7E#$6C#$E7#$22#$CC#$B9#$32#$6B#$D9#$C2#$77#$C2#$7B#$DF#$E9#$B6#$17#$53#$AF#$7B#$A6#$94#$AE#$39#$39#$00#$EF#$75#$E2#$4D#$15#$F6#$45#$1A#$B2#$9F#$D6#$82#$24#$45#$B2#$07#$A7#$00#$3D#$14#$01 +
            #$62#$40#$DD#$D2#$B8#$F0#$96#$9B#$07#$0C#$49#$F8#$74#$B2#$BF#$E9#$31#$B3#$2E#$9D#$2F#$30#$C2#$F6#$ED#$4C#$56#$42#$D4#$47#$55#$6A#$F4#$39#$C2#$23#$5F#$58#$BD#$2C#$7B#$C5#$4A#$DC#$8A#$AD#$40#$E5 +
            #$84#$A0#$65#$B3#$65#$73#$74#$9F#$AB#$BD#$E6#$09#$A2#$24#$65#$8D#$F6#$5B#$5B#$14#$4B#$C0#$6A#$78#$F2#$D6#$F3#$76#$45#$A0#$77#$DD#$AA#$DB#$13#$F9#$9F#$0F#$5A#$A5#$A0#$B9#$28#$48#$95#$24#$03#$2D +
            #$94#$6E#$E1#$05#$F7#$5C#$23#$82#$29#$95#$A1#$7C#$F2#$E3#$B9#$2F#$AC#$29#$11#$EC#$60#$D1#$4C#$5F#$B8#$D2#$1E#$9D#$EC#$76#$1D#$FC#$5F#$C0#$FE#$D1#$50#$33#$0A#$42#$D7#$83#$C9#$F9#$49#$54#$BB#$50 +
            #$17#$6C#$6B#$38#$C1#$31#$19#$DB#$FC#$DE#$CE#$17#$AD#$B2#$D1#$DA#$0A#$13#$EE#$2B#$ED#$3A#$EA#$EA#$20#$8E#$17#$14#$0B#$88#$EE#$B9#$A5#$02#$1B#$E8#$6C#$F7#$FB#$E4#$18#$72#$E7#$35#$88#$DE#$8E#$B8 +
            #$56#$FB#$D9#$8C#$B8#$81#$A7#$39#$AC#$F2#$D2#$39#$0B#$AF#$7F#$B6#$D2#$0E#$A5#$C7#$A0#$B3#$EF#$6F#$4B#$F4#$08#$78#$B2#$A8#$89#$CD#$0E#$0E#$8A#$B1#$8F#$8C#$AC#$E3#$23#$23#$B6#$D1#$6C#$80#$FD#$F5 +
            #$4D#$65#$3B#$75#$17#$FC#$81#$A3#$88#$15#$2A#$EE#$19#$E0#$FD#$F5#$F1#$C2#$C3#$DD#$EE#$7A#$08#$12#$25#$72#$75#$42#$77#$8D#$51#$5B#$D6#$31#$61#$88#$21#$73#$86#$2E#$E5#$8E#$0B#$3E#$BE#$61#$B8#$E4 +
            #$FD#$FA#$67#$C6#$44#$78#$B2#$31#$D9#$8B#$BF#$A8#$07#$8F#$7E#$BE#$46#$54#$53#$24#$9A#$25#$E5#$D3#$5F#$C0#$22#$20#$EA#$C0#$86#$CD#$6C#$F8#$7A#$4A#$6C#$18#$BD#$0F#$CC#$F3#$5F#$7F#$E0#$BE#$B0#$53 +
            #$94#$5F#$E0#$89#$11#$86#$92#$F2#$3F#$1A#$BD#$E8#$68#$B8#$35#$3B#$91#$8F#$85#$73#$98#$B9#$7E#$D1#$C0#$AD#$F8#$4D#$72#$04#$4F#$1D#$AD#$82#$5F#$86#$DD#$89#$17#$73#$38#$F8#$25#$34#$BA#$57#$24#$F8 +
            #$F3#$E3#$50#$AE#$41#$D9#$91#$F8#$AC#$8E#$C4#$87#$F6#$58#$D2#$1C#$31#$9B#$C9#$0E#$28#$56#$C9#$5F#$09#$43#$73#$A0#$79#$13#$58#$F2#$5E#$25#$89#$8F#$17#$58#$9E#$B3#$5E#$30#$6C#$8A#$AB#$81#$43#$27 +
            #$6C#$53#$82#$08#$19#$B9#$C0#$FB#$DA#$F4#$DD#$BD#$55#$66#$8D#$71#$FF#$67#$40#$4E#$53#$B3#$70#$22#$7B#$80#$20#$8C#$92#$3F#$0F#$73#$AB#$F1#$E4#$5A#$A2#$57#$3D#$DF#$DE#$39#$D0#$E5#$47#$1C#$41#$DA +
            #$7A#$95#$0C#$5E#$E4#$A3#$84#$34#$6E#$3E#$A4#$9A#$FD#$22#$46#$A9#$79#$65#$69#$19#$6A#$CE#$8A#$E9#$58#$94#$4A#$E1#$D9#$71#$02#$D9#$8F#$46#$A6#$5C#$59#$48#$1F#$63#$97#$1F#$AF#$FC#$11#$F0#$08#$02 +
            #$11#$02#$FF#$70#$A4#$E7#$3A#$B9#$5C#$F1#$70#$CA#$58#$F3#$1A#$00#$14#$FA#$7C#$BF#$7E#$1B#$DE#$7E#$D6#$4F#$58#$40#$FB#$8F#$EA#$DC#$8B#$EC#$E2#$2A#$DD#$39#$97#$04#$BE#$3D#$8E#$0C#$AD#$9B#$48#$EF +
            #$92#$5F#$76#$E3#$0C#$0D#$40#$20#$08#$4A#$46#$E7#$65#$38#$F1#$B4#$43#$E6#$99#$06#$5D#$D5#$4D#$7B#$DF#$D6#$46#$7A#$92#$55#$56#$08#$E4#$5F#$08#$D8#$C5#$94#$A6#$5E#$FE#$3B#$EC#$A3#$E8#$54#$90#$2B +
            #$F7#$F3#$37#$25#$90#$26#$0A#$1C#$78#$E6#$E5#$8D#$FF#$53#$96#$4B#$BB#$E4#$DE#$46#$05#$68#$7C#$BF#$05#$19#$D4#$D2#$C6#$8E#$AE#$E9#$45#$31#$60#$F8#$4C#$D1#$9A#$F1#$CA#$6E#$6E#$61#$33#$D0#$57#$15 +
            #$0E#$15#$6A#$14#$06#$1D#$CE#$65#$CA#$45#$71#$A7#$BB#$7A#$DE#$CE#$F6#$EA#$62#$A6#$FA#$54#$C1#$AE#$B4#$77#$26#$07#$C0#$10#$30#$B1#$CC#$24#$BF#$E8#$45#$C2#$A9#$D3#$4B#$66#$A1#$E3#$F0#$57#$B0#$FE;
    ),
    (
      iterations: $0000001;
      input: 'fatherblessedthemallwithreasonandthisiswhattheychooseandthisiswhattheychoose';
    ),
    (
      iterations: $0000001;
      input: 'inerw'#$C3#$A4'gungdassdah'#$C3#$A4'userstehenw'#$C3#$A4'hrendihrunsohnebleibelassthabenwirbeschlossenjetztdorteinzuziehenweilesunsinunsernl'#$C3#$B6'chernnichtmehrpasst';
    ),
    (
      iterations: $0000001;
      input: 'hewasborninacoopraisedinacagechildrenfearhimcriticsrageheshalfaliveheshalfdeadHESGOTABUCKETONHISHEAD!';
    ),
    (
      iterations: $000002A;
      input:
            '(~2GS:=%/R&En0Z2@eY^a9WMifNUdx]?)6j8}En7R8{n7-aYXV99g|r(SU5]+wOvuq9A)]x|kTr*\mm;skW:aZ}89B`c$3>aHtli9_VeFo_%dr,Hi"kkD8O_Tra3L2/W' +
            '>sO>5P6f{dJv+w}S=HW4>>E|JNU&$[W4**IxMEbIvzD%qfj+h^_v-b2DA`z!Hvs@,YN!M>t+;Q2z_<6C4S<T6Koqb](YCm6?{]QuEJ_;}{1:/E$5+Ss1]T>,,hoF=zT`';
    ),
    (
      iterations: $0000001;
      input:
            '3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844' +
            '60955058223172535940812848111745028410270193852110555964462294895493038196442881097566593344612847564823378678316527120190914564' +
            '85669234603486104543266482133936072602491412737245870066063155881748815209209628292540917153643678925903600113305305488204665213' +
            '84146951941511609433057270365759591953092186117381932611793105118548074462379962749567351885752724891227938183011949129833673362' +
            '44065664308602139494639522473719070217986094370277053921717629317675238467481846766940513200056812714526356082778577134275778960' +
            '91736371787214684409012249534301465495853710507922796892589235420199561121290219608640344181598136297747713099605187072113499999' +
            '98372978049951059731732816096318595024459455346908302642522308253344685035261931188171010003137838752886587533208381420617177669' +
            '14730359825349042875546873115956286388235378759375195778185778053217122680661300192787661119590921642019893809525720106548586328';
    ),
    (
      iterations: $1000000;
      input: 'abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmno';
    )
  );
  TEST_RESULT: array[0..15] of TDigestBuf =
  (
    ( $4A, $07, $8B, $E6, $56, $B2, $94, $64, $38, $9C, $71, $09, $F0, $F4, $08, $B0, $AD, $9C, $92, $BD, $B4, $36, $03, $63, $73, $AC, $50, $BC, $99, $3C, $21, $F7, $34, $8B, $9E, $E2, $09, $80, $F7, $7F, $4D, $2B, $ED, $49, $0E, $BC, $9D, $2E ), {00}
    ( $F8, $41, $FD, $28, $F9, $75, $A8, $27, $ED, $57, $55, $CD, $BA, $E2, $A0, $F0, $FB, $4D, $D8, $A0, $2C, $84, $10, $04, $0F, $EC, $8B, $42, $59, $0E, $EF, $70, $FF, $0F, $50, $4A, $E4, $4F, $38, $1B, $AF, $99, $59, $96, $03, $76, $C9, $E9 ), {01}
    ( $60, $1F, $8D, $7E, $31, $94, $48, $A6, $3E, $32, $86, $00, $64, $91, $2F, $34, $4D, $D7, $15, $52, $AC, $70, $61, $31, $19, $39, $6E, $B4, $BB, $CB, $AC, $95, $95, $E6, $F1, $80, $6C, $12, $7F, $83, $B0, $7A, $7D, $44, $9A, $FA, $83, $71 ), {02}
    ( $EF, $AC, $BD, $E1, $8D, $6A, $B7, $29, $45, $74, $D2, $F7, $68, $86, $49, $D9, $6D, $74, $AE, $A6, $85, $F6, $D4, $34, $38, $76, $BB, $9C, $25, $7C, $ED, $B5, $A4, $E6, $0F, $8E, $01, $42, $96, $FB, $5E, $46, $C1, $7A, $C6, $79, $E7, $A8 ), {03}
    ( $DA, $23, $8B, $FF, $84, $2B, $DA, $78, $D3, $0F, $1F, $A2, $A1, $2D, $FA, $7D, $6A, $C5, $F1, $B0, $28, $BB, $41, $FB, $1F, $FA, $85, $D3, $63, $21, $FE, $08, $A6, $00, $64, $CD, $28, $CB, $B0, $F5, $73, $EA, $61, $6B, $FE, $08, $C2, $72 ), {04}
    ( $92, $39, $31, $81, $E1, $41, $3D, $83, $DB, $9B, $83, $52, $20, $83, $FC, $6A, $EE, $CE, $E2, $78, $AA, $CF, $53, $E1, $F2, $4F, $7A, $39, $36, $05, $75, $14, $1F, $71, $1E, $76, $2C, $D1, $BB, $43, $96, $38, $F7, $11, $EC, $D9, $1C, $15 ), {05}
    ( $0B, $99, $27, $1A, $6F, $34, $40, $52, $FF, $38, $01, $36, $F5, $28, $BC, $F3, $60, $2A, $AB, $73, $3D, $59, $86, $B2, $57, $EC, $91, $65, $AA, $61, $50, $A1, $79, $25, $54, $B1, $9E, $7E, $D9, $11, $BE, $B6, $48, $6D, $C0, $9D, $85, $A2 ), {06}
    ( $4C, $A3, $1A, $25, $69, $2C, $FC, $7C, $71, $8A, $9D, $D1, $E0, $5D, $F8, $BE, $AA, $E5, $38, $66, $9F, $E7, $11, $0F, $72, $63, $EF, $C0, $E3, $CE, $10, $8E, $CD, $A0, $C3, $D9, $85, $46, $42, $AA, $C7, $56, $DF, $C4, $C6, $D4, $22, $8E ), {07}
    ( $9B, $F9, $38, $C8, $33, $E2, $58, $44, $3E, $DE, $FD, $51, $0A, $62, $2D, $ED, $ED, $72, $1C, $01, $45, $D5, $C9, $D0, $F9, $88, $15, $74, $0A, $23, $9D, $D8, $50, $5B, $FF, $7B, $2D, $0E, $6A, $1B, $0A, $4D, $F2, $C5, $F6, $7B, $41, $28 ), {08}
    ( $87, $37, $44, $97, $9E, $FA, $03, $2F, $78, $D0, $66, $97, $30, $13, $FD, $44, $2F, $91, $97, $E7, $D4, $12, $B9, $AA, $EF, $E5, $CB, $B1, $74, $BE, $63, $C0, $CC, $02, $A6, $E0, $30, $E6, $14, $75, $EB, $89, $4A, $E1, $2D, $16, $57, $51 ), {09}
    ( $56, $7F, $32, $54, $FD, $4F, $58, $C9, $66, $F5, $11, $99, $27, $F1, $2F, $0B, $59, $3E, $5A, $CF, $90, $02, $88, $F1, $C5, $9D, $CC, $C0, $E4, $77, $C3, $4A, $00, $F4, $BF, $6B, $71, $F2, $F9, $08, $FD, $1A, $F2, $9C, $27, $88, $87, $97 ), {0A}
    ( $0E, $73, $2B, $B3, $8D, $C5, $35, $9D, $A7, $C9, $7B, $63, $D5, $2C, $5E, $E1, $20, $73, $33, $7D, $E7, $4E, $16, $95, $48, $40, $D6, $E3, $63, $0E, $C3, $5B, $D2, $FF, $EA, $D6, $5D, $FD, $11, $5C, $1B, $58, $1E, $EF, $1B, $32, $DD, $2A ), {0B}
    ( $8E, $7E, $C7, $36, $DF, $F6, $C7, $9C, $55, $AE, $12, $99, $05, $5D, $9D, $61, $D3, $26, $2E, $84, $EE, $CC, $56, $33, $47, $29, $6B, $89, $BF, $81, $2F, $DC, $C6, $30, $D2, $62, $CE, $AC, $4C, $BE, $21, $1F, $57, $9E, $B6, $EC, $1C, $B2 ), {0C}
    ( $34, $DE, $19, $51, $1B, $E9, $A2, $84, $D8, $B9, $B4, $C7, $6D, $D6, $93, $4F, $C0, $B5, $EB, $C2, $0D, $6E, $B7, $E3, $9B, $54, $13, $B3, $39, $8D, $34, $9F, $80, $6B, $13, $36, $7A, $11, $F4, $B5, $D4, $C3, $D0, $CE, $6F, $8E, $A3, $A7 ), {0D}
    ( $0F, $F2, $A2, $F0, $AC, $6F, $43, $AF, $48, $CE, $8A, $38, $0B, $1A, $B1, $34, $CE, $6A, $31, $76, $CE, $C8, $13, $B2, $AE, $6C, $3B, $7A, $B6, $DA, $EF, $D1, $BE, $99, $E7, $F3, $C9, $E6, $1A, $1B, $3E, $63, $1D, $07, $7C, $27, $35, $13 ), {0E}
    ( $09, $19, $F0, $2A, $72, $1C, $4A, $DA, $98, $79, $C3, $BE, $6C, $EC, $98, $FB, $6E, $A0, $D4, $53, $C2, $0D, $B4, $BD, $9B, $28, $7E, $D6, $25, $2B, $98, $25, $CF, $66, $A0, $65, $A2, $09, $AF, $C8, $D5, $A2, $2A, $DF, $2E, $45, $7C, $6E )  {0F}
  );
var
  Digest: TMHash384;
  i, j: Cardinal;
  Hash: TByteArray;
begin
  for i := 0 to High(TEST_VECTOR) do
  begin
    Digest := TMHash384.Create();
    for j := 0 to TEST_VECTOR[i].iterations-1 do
    begin
      if (@callback <> nil) and (j and $7FFF = 0) then
      begin
        callback(Length(TEST_VECTOR), i + 1, GetProgress(TEST_VECTOR[i].iterations, j));
      end;
      if (abortFlag <> nil) then
      begin
        if (abortFlag.WaitFor(0) = wrSignaled) then
        begin
          raise Exception.Create('Operation was aborted by the user!');
        end;
      end;
      Digest.Update(TEST_VECTOR[i].input);
    end;
    if (@callback <> nil) then
    begin
      callback(Length(TEST_VECTOR), i + 1, 100);
    end;
    Digest.Result(Hash);
    if not CompareMem(@Hash[0], @TEST_RESULT[i][0], MHASH384_LEN) then
    begin
      raise Exception.Create('Test vector #'+IntToStr(i)+' did NOT compare equal!');
    end;
  end;
end;

end.

