#include "common.h"
#include <vector>
#include <algorithm>

extern "C"
{
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <termios.h>
#include <dirent.h>
#include <libgen.h>
}

bool gCmdLineActive;
static vector<string> gCmdLine;
static int gCLCursor;
static int gCursor;

static vector<string> gPrograms;
vector<string> gCCandidate;

static int gMaster;
vector<sMasterTty*> gMasters;

char gOutput[kOutputSize];
int gOutputTail;

static int get_master_pty(RESULT char name[])
{
    int i, j;
    int master = -1;

    strcpy(name, "/dev/ptyXX");

    for(i=0; i<16 && master <=0; i++) {
        for(j=0; j<16 && master <= 0; j++) {
            name[8] = "pqrstuvwxyzPQRST"[i];
            name[9] = "0123456789abcdef"[j];

            if((master = open(name, O_RDWR)) < 0) {
                if(errno == ENOENT) {
                    return (master);
                }
            }
        }
    }
    if((master<0) && (i==16) && (j==16)){
        return master;
    }

    name[5] = 't';
    return master;
}

static int get_slave_pty(char* name)
{
    struct group* gptr;
    gid_t gid;
    int slave = -1;

    if((gptr = getgrnam("tty")) != 0) {
        gid = gptr->gr_gid;
    }
    else {
        gid = -1;
    }

    chown(name, getuid(), gid);

    chmod(name, S_IRUSR|S_IWUSR);

    slave = open(name, O_RDWR);

    return slave;
}


sighandler_t gOldWinch;

void sigwinch_handler(int signal)
{
TBEGIN();

    struct winsize ws;
    if(ioctl(STDOUT_FILENO, TIOCGWINSZ, &ws) < 0) {
        perror("ptypair: could not get window size");
    }
    if(ioctl(gMaster, TIOCSWINSZ, &ws) < 0) {
        perror("could not restore window size");
    }

    gOldWinch(signal);
    
TEND();
}

static bool sort_name(string left, string right)
{
    return strcmp(left.c_str(), right.c_str()) < 0;
}

void cmdline_init()
{
TBEGIN();

    gCmdLineActive = false;
    gCLCursor = 0;
    gCursor = 0;
    
    gOutputTail = 0;

    char* path = getenv("PATH");
    if(path == NULL) {
        fprintf(stderr, "$PATH is NULL\n");
        exit(1);
    }

    char* p = path;
    char buf[512];
    char* p2 = buf;
    
    while(*p) {
        if(*p != ':') {
            *p2++ = *p;
        }
        else {
            *p2 = 0;

            DIR* dir = opendir(buf);
            if(dir) {
                struct dirent* entry;
                while(entry = readdir(dir)) {
                    char path2[512];
                    sprintf(path2, "%s/%s", buf, entry->d_name);

                    struct stat stat_;
                    memset(&stat_, 0, sizeof(struct stat));
                    stat(path2, &stat_);
                    
                    if(strcmp(entry->d_name, ".") != 0
                        && strcmp(entry->d_name, "..") != 0
                        &&
                        (stat_.st_mode & S_IXUSR
                            ||stat_.st_mode & S_IXGRP
                            ||stat_.st_mode & S_IXOTH))
                        {
                        gPrograms.push_back(entry->d_name);
                        }
                }

                closedir(dir);
            }
            
            p2 = buf;
        }
        
        p++;
    }
    
    *p2 = 0;

    DIR* dir = opendir(buf);
    if(dir) {
        struct dirent* entry;
        while(entry = readdir(dir)) {
            if(strcmp(entry->d_name, ".") != 0
                && strcmp(entry->d_name, "..") != 0)
            {
               gPrograms.push_back(entry->d_name);
            }
        }

        closedir(dir);
    }

    sort(gPrograms.begin(), gPrograms.end(), sort_name);
    
TEND();    
}

void cmdline_final()
{
    for(int i=0; i<gMasters.size(); i++) {
        FREE(gMasters[i]);
    }
}

static void cmdline_move(int cursor)
{
TBEGIN();

    gCLCursor += cursor;

    if(gCLCursor < 0) gCLCursor = 0;
    if(gCLCursor >= gCmdLine.size()) gCLCursor = gCmdLine.size() -1;

TEND();    
}

static void cursor_move(int v)
{
TBEGIN();

    gCursor += v;

    if(gCursor < 0) gCursor = 0;
    if(gCursor > gCmdLine[gCLCursor].size()) gCursor = gCmdLine[gCLCursor].size();

TEND();    
}

// !! out must have enable allocated memory space
static void extend_macro(char* in, RESULT char* out, RESULT bool* ruby_mode, RESULT bool* output, RESULT bool* quick, RESULT bool* shell_mode,  bool flg)
{
TBEGIN();

    sFile cursor = ActiveDir()->CursorFile();
    *ruby_mode = false;
    *output = true;
    *shell_mode = false;
    *quick = false;
    
    char* p = in;
    char* cp = out;
    while(*p) {
        if(*p != '%') {
            *cp = *p;
            cp++;
            p++;
        }
        else {
            p++;

            switch(*p) {
            case 'r':
                p++;

                if(flg) {
                    *ruby_mode = true;
                }
                else {
                    *cp++ = '%';
                    *cp++ = 'r';
                }
                break;

            case 'q':
                p++;

                if(flg) {
                    *output = false;
                }
                else {
                    *cp++ = '%';
                    *cp++ = 'q';
                }
                break;

            case 'Q':
                p++;

                if(flg) {
                    *quick = true;
                }
                else {
                    *cp++ = '%';
                    *cp++ = 'Q';
                }
                break;

            case 's':
                p++;

                if(flg) {
                    *shell_mode = true;
                }
                else {
                    *cp++ = '%';
                    *cp++ = 's';
                }
                break;

            case 'f': {
                p++;
                    
                const int len = strlen(cursor.mName);
                *cp++ = '\'';
                memcpy(cp, cursor.mName, len);
                cp += len;
                *cp++ = '\'';
                }
                break;

            case 'F': {
                p++;

                *cp++ = '\'';
                const int len = strlen(ActiveDir()->Path());
                memcpy(cp, ActiveDir()->Path(), len);
                cp += len;

                const int len2 = strlen(cursor.mName);
                memcpy(cp, cursor.mName, len2);
                cp += len2;
                
                *cp++ = '\'';
                }
                break;

            case 'x': {
                p++;
                    
                char* extension = extname(cursor.mName);
                const int len = strlen(extension);

                *cp++ = '\'';
                memcpy(cp, extension, len);
                cp += len;
                *cp++ = '\'';

                FREE(extension);
                }
                break;

            case 'X': {
                p++;

                char* nwe = noextname(cursor.mName);
                const int len = strlen(nwe);

                *cp++='\'';
                memcpy(cp, nwe, len);
                cp += len;
                *cp++='\'';
                    
                FREE(nwe);
                }
                break;

            case 'd': {
                p++;

                if(*p == '1') {
                    p++;

                    char* tmp = STRDUP(ActiveDir()->Path());
                    tmp[strlen(tmp)-1] = 0;
                    char* name = basename(tmp);
                    FREE(tmp);
                    const int len = strlen(name);

                    *cp++='\'';
                    memcpy(cp, name, len);
                    cp += len;
                    *cp++='\'';
                }
                else if(*p == '2') {
                    p++;

                    char* tmp = STRDUP(SleepDir()->Path());
                    tmp[strlen(tmp)-1] = 0;
                    char* name = basename(tmp);
                    FREE(tmp);
                    const int len = strlen(name);

                    *cp++='\'';
                    memcpy(cp, name, len);
                    cp += len;
                    *cp++='\'';
                }
                else {
                    cp++;
                    err_msg("illeagal macro char");
                }
                }
                break;

            case 'D': {
                p++;

                if(*p == '1') {
                    p++;

                    const int len = strlen(ActiveDir()->Path());
                    *cp++='\'';
                    memcpy(cp, ActiveDir()->Path(), len-1);
                    cp += len-1;
                    *cp++='\'';
                }
                else if(*p == '2') {
                    p++;
                        
                    const int len = strlen(SleepDir()->Path());

                    *cp++='\'';
                    memcpy(cp, SleepDir()->Path(), len-1);
                    cp += len-1;
                    *cp++='\'';
                }
                else {
                    cp++;
                    err_msg("illeagal macro char");
                }
                    
                }
                break;

            case 'm':
                p++;

                if(*p == '1') {
                    p++;

                    if(flg) {
                        char buf[4096];
                        ActiveDir()->MarkFilesSQuote(buf);      
                        const int len = strlen(buf);

                        memcpy(cp, buf, len);
                        cp += len;
                    }
                    else {
                        *cp++ = '%';
                        *cp++ = 'm';
                        *cp++ = '1';
                    }
                }
                else if(*p == '2') {
                    p++;

                    if(flg) {
                        char buf[4096];
                        SleepDir()->MarkFilesSQuote(buf);      
                        const int len = strlen(buf);
    
                        memcpy(cp, buf, len);
                        cp += len;
                    }
                    else {
                        *cp++ = '%';
                        *cp++ = 'm';
                        *cp++ = '2';
                    }
                }
                else {
                    cp++;
                    err_msg("illeagal macro char");
                }

                break;

            default:
                cp++;
                err_msg("illeagal macro char");
                break;
            }
        }
    }
    *cp = 0;
    
TEND();
}

void cmdline_start(char* cmd, int position)
{
TBEGIN();

    gCmdLineActive = true;

    char cmd2[1024];
    bool ruby_mode;
    bool output;
    bool shell_mode;
    bool quick;
    extend_macro(cmd, cmd2, &ruby_mode, &output, &quick, &shell_mode, false);
    
    gCmdLine.push_back(cmd2);
    gCLCursor = gCmdLine.size()-1;
    if(position > 0)
        gCursor = position-1;
    else
        gCursor = strlen(cmd2) + position;
        
TEND();
}

int cmdline_select_pty()
{
    WINDOW* scr = newwin(gMasters.size() + 2, 45, 0, 2);
    box(scr, '|', '-');

    wattron(scr, A_BOLD);
    mvwprintw(scr, 0, 2, "ptty select");
    wattroff(scr, A_BOLD);
    
    for(int i=0; i<gMasters.size(); i++) {
        sMasterTty* master = gMasters[i];
        
        mvwprintw(scr, i+1, 2, "%d %s", i+1, master->mName);
    }

    wrefresh(scr);

    int key = getch();
    if(key < '1' || key > '0'+gMasters.size())  return -1;

    return key - '0';
}

inline bool ptty_loop()
{
    bool result = false;

    struct termios ot, t;

    tcgetattr(STDIN_FILENO, &ot);
    t = ot;
    t.c_lflag &= ~(ICANON | ISIG | ECHO | ECHOCTL | ECHOE| ECHOK | ECHOKE
                | ECHONL | ECHOPRT);

    t.c_iflag |= IGNBRK;
    t.c_iflag &= ~(IXOFF|IXON);
    t.c_cc[VMIN] = 1;
    t.c_cc[VTIME] = 0;
    tcsetattr(STDIN_FILENO, TCSANOW, &t);

    gOldWinch = signal(SIGWINCH, SIG_IGN);
    
    struct sigaction act;
    act.sa_handler = sigwinch_handler;
    sigemptyset(&(act.sa_mask));
    act.sa_flags = 0;
    if(sigaction(SIGWINCH, &act, NULL) < 0) {
        err_msg("ptty_loop: could not handle SIGWINCH ");
        return result;
    }
    
    int done = 0;
    do {
        fd_set ready;
        
        FD_ZERO(&ready);
        FD_SET(STDIN_FILENO, &ready);
        FD_SET(gMaster, &ready);

        select(gMaster+1, &ready, NULL, NULL, NULL);

        if(FD_ISSET(gMaster, &ready)) {
            char buf[8192];
            
            int i = read(gMaster, buf, 8191);
            if(i > 0) {
                if((gOutput + gOutputTail + i) >= (gOutput + kOutputSize)) {
                    memcpy(gOutput + kOutputSize / 2
                            , gOutput
                            , kOutputSize - kOutputSize / 2);
                    gOutputTail -= kOutputSize / 2;
/*                
                    memcpy(gOutput, buf, i);
                    gOutputTail = i;
*/
                }

                memcpy(gOutput + gOutputTail, buf, i);
                gOutputTail += i;
            
                write(STDOUT_FILENO, buf, i);
            } else {
                done = 2;
            }
        }
        else if(FD_ISSET(STDIN_FILENO, &ready)) {
            char buf[8192];
            
            int i = read(STDIN_FILENO, buf, 8191);

            if(i > 0) {
                int ctrl_z = 0;
            
                for(int j=0; j<i; j++) {
                    if(buf[j] == 26) {            // CTRL-Z
                        result = true;
                        done = 1;

                        char buf2[8192];
                        int c = 0;
                        
                        for(int k=0; k<i; k++) {
                            if(buf[k] != 26) {
                                buf2[c++] = buf[k];
                            }
                            else {
                                ctrl_z++;
                            }
                        }

                        memcpy(buf, buf2, i);
                        break;
                    }
                }
                
                write(gMaster, buf, i-ctrl_z);
            }
            else {
                done = 2;
            }
        }
        
    } while(!done);

    if(done == 2) {
        for(vector<sMasterTty*>::iterator i=gMasters.begin();
            i!=gMasters.end();
            i++)
        {
            if((*i)->mFD == gMaster) {
                FREE(*i);
                gMasters.erase(i);
                break;
            }
        }
    }

    tcsetattr(STDIN_FILENO, TCSANOW, &ot);
    act.sa_handler = gOldWinch;
    sigemptyset(&(act.sa_mask));
    act.sa_flags = 0;
    if(sigaction(SIGWINCH, &act, NULL) < 0) {
        err_msg("pty_loop: could not handle SIGWINCH ");
        return result;
    }

    clear();
    view();
    refresh();

    return result;
}

void cmdline_restore_pty(int n)
{
    if(n < 1 || n > gMasters.size()) return;

    gMaster = gMasters[n-1]->mFD;
    memcpy(gOutput, gMasters[n-1]->mOutput, kOutputSize);
    gOutputTail = gMasters[n-1]->mOutputTail;

    clear();
    move(0, 0);
    refresh();

    const int m = getmaxx(stdscr) * getmaxy(stdscr) * 3;
    if(gOutputTail < m) {
        write(STDOUT_FILENO, gOutput, gOutputTail);
    }
    else {
        write(STDOUT_FILENO, gOutput + gOutputTail -m, m);
    }
    
    if(ptty_loop()) {
        memcpy(gMasters[n-1]->mOutput, gOutput, kOutputSize);
        gMasters[n-1]->mOutputTail = gOutputTail;
    }
}

static void new_ptty(char* title, char* cmd, bool output)
{
TBEGIN();
    gOutputTail = 0;

    char* argv[4];

    char buf[1024];
            
    argv[0] = "/bin/sh";
    argv[1] = "-c";
    argv[2] = buf;
    argv[3] = NULL;

    if(output)
        sprintf(argv[2], "echo \"$ %s\";( %s ); mfiler %d reread; echo \"\nHIT ENTER KEY\"; read TMP", cmd, cmd, getpid());
    else 
        sprintf(argv[2], "( %s ); mfiler %d reread", cmd, getpid());

    char name[256];
    gMaster = get_master_pty(name);
    if(gMaster < 0) {
        err_msg("new_ptty: could not open master pty");;
        return;
    }
    
    struct winsize ws;
    if(ioctl(STDIN_FILENO, TIOCGWINSZ, &ws) < 0) {
         err_msg("new_ptty: could not get window size");
         return;
    }

    int pid = fork();
    if(pid < 0) {
        err_msg("new_ptty: fork false");
        return;
    }

    /// child process ///    
    if(pid == 0) {
        int slave;

        close(gMaster);

        if((slave = get_slave_pty(name)) < 0) {
            fprintf(stderr, "new_ptty: could not open slave pty. pleas check permission of %s", name);
            exit(1);
        }

        if(setsid() < 0) {
            perror("could not set new controlling tty");
        }

        if(ioctl(slave, TIOCSCTTY, NULL)) {
            perror("could not set new controlling tty");
        }

        dup2(slave, STDIN_FILENO);
        dup2(slave, STDOUT_FILENO);
        dup2(slave, STDERR_FILENO);

        if(slave > 2) {
            close(slave);
        }
        if(ioctl(STDOUT_FILENO, TIOCSWINSZ, &ws) < 0) {
            perror("could not restore window size");
        }

        execvp("/bin/sh", argv);

        exit(1);
    }
    
    if(ptty_loop()) {
        sMasterTty* master  = (sMasterTty*)MALLOC(sizeof(sMasterTty));

        master->mFD = gMaster;
        strcpy(master->mName, title);

        memcpy(master->mOutput, gOutput, kOutputSize);
        master->mOutputTail = gOutputTail;
                        
        gMasters.push_back(master);
    }

TEND();    
}

void cmdline_run(char* str, char* title)
{
TBEGIN();
    char cmd[1024];
    bool ruby_mode;
    bool output;
    bool shell_mode;
    bool quick;
    extend_macro(str, cmd, &ruby_mode, &output, &quick, &shell_mode, true);

    char title2[1024];
    bool tmp1;
    bool tmp2;
    bool tmp3;
    bool tmp4;
    extend_macro(title, title2, &tmp1, &tmp2, &tmp3, &tmp4, false);

    char* term = getenv("TERM");
    if(term == NULL) {
        fprintf(stderr, "TERM is null");
        exit(1);
    }

    struct winsize ws;
    ioctl(STDOUT_FILENO,TIOCGWINSZ,&ws);

    const int maxy = ws.ws_row;
    
    myclear_online(maxy-1);
    myclear_online(maxy-2);
    move(maxy-2, 0);
    refresh();
    
    if(!quick) {
        //clear();
        //refresh();
        endwin();
    }
    else {
        echo();
        keypad(stdscr, FALSE);
        noraw();
    }

    /// ruby mode ///
    if(ruby_mode) {
        char buf[256];
        sprintf(buf, "echo \"> %s\"", title2);
        if(output && !quick) system(buf);
        
        int error;
        rb_eval_string_protect(cmd, &error);

        if(output && !quick) printf("\nHIT ENTER KEY");
    }
    /// shell mode ///
    else  if(shell_mode || quick) {
        char buf[256];
        sprintf(buf, "echo \"$ %s\"", cmd);
        if(output && !quick) system(buf);
            
        system(cmd);
            
        if(output && !quick) system("echo \"\nHIT ENTER KEY\"; read TMP");
    }
    /// screen terminal ///
    else if(strstr(term, "screen") == term) {
        char* argv[7];

        char buf[4096];
            
        argv[0] = "screen";
        argv[1] = "-t";
        argv[2] = title2;
        argv[3] = "/bin/sh";
        argv[4] = "-c";
        argv[5] = buf;
        argv[6] = NULL;

        if(output && !quick)
            sprintf(argv[5], "echo \"$ %s\";( %s ); mfiler %d reread; echo \"\nHIT ENTER KEY\"; read TMP", cmd, cmd, getpid());
        else 
            sprintf(argv[5], "( %s ); mfiler %d reread", cmd, getpid());

        pid_t pid = fork();
        if(pid == -1) {
            fprintf(stderr, "fork failed");
            exit(1);
        }
        
        if(pid == 0) execvp("screen", argv);
    }

    /// normal terminal ///
    else {
        clear();
        move(0, 0);
        refresh();
    
        new_ptty(title2, cmd, output);
    }

    if(!quick) {
        initscr();
        noecho();
        keypad(stdscr, 1);
        raw();
    }
    else {
        noecho();
        keypad(stdscr, 1);
        raw();
    }

    if(ruby_mode && output && !quick) {     // ruby mode
        getch();
    }
        
    gLDir->Reread();
    gRDir->Reread();
    
TEND();
}

static void completion(vector<string>& all_candidate, char* editing)
{
    gCCandidate.clear();
    if(strcmp(editing, "") == 0) {
        for(vector<string>::iterator i = all_candidate.begin(); i != all_candidate.end(); i++) {
            gCCandidate.push_back((*i));
        }
    }
    else {
        for(vector<string>::iterator i = all_candidate.begin(); i != all_candidate.end(); i++) {
            if(strstr((*i).c_str(), editing) == (*i).c_str()) {
                gCCandidate.push_back((*i));
            }
        }
    }
            
    if(gCCandidate.size() > 0) {
        int l = strlen(editing);
    
        while(l < gCCandidate[0].length()) {
            bool same = true;
            for(vector<string>::iterator i = gCCandidate.begin(); i != gCCandidate.end(); i++) {
                if((*i)[l] != gCCandidate[0][l]) {
                    same = false;
                }
            }
    
            if(same) {
                char tmp[2];
                sprintf(tmp, "%c", gCCandidate[0][l]);
                gCmdLine[gCLCursor].insert(gCursor, tmp);
                gCursor++;
            }
            else {
                break;
            }
    
            l++;
        }
    }
}

void cmdline_input(int meta, int key)
{
TBEGIN();

    if(key == 6 || key == KEY_RIGHT) {    // CTRL-F
        if(is_kanji(gCmdLine[gCLCursor][gCursor])) {
            cursor_move(2);
        }
        else {
            cursor_move(1);
        }

        gCCandidate.clear();
    }
    else if(key == 2 || key == KEY_LEFT) {    // CTRL-B
        if(is_kanji(gCmdLine[gCLCursor][gCursor-1])) {
            cursor_move(-2);
        }
        else {
            cursor_move(-1);
        }

        gCCandidate.clear();
    }
    else if(key == 1 || key == KEY_HOME) {    // CTRL-A
        cursor_move(-999);

        gCCandidate.clear();
    }
    else if(key == 5 || key == KEY_END) {    // CTRL-E
        cursor_move(999);

        gCCandidate.clear();
    }
    else if(key == 4 || key == KEY_DC) {    // CTRL-D
        if(gCmdLine[gCLCursor].length() == 0) {
            gCmdLineActive = false;

            gCmdLine.pop_back();
        }
        else {
            if(is_kanji(gCmdLine[gCLCursor][gCursor])) {
                string_erase(gCmdLine[gCLCursor], gCursor);
                string_erase(gCmdLine[gCLCursor], gCursor);
                
//                gCmdLine[gCLCursor].erase(gCursor, 2);
            }
            else {
                string_erase(gCmdLine[gCLCursor], gCursor);
                
//                gCmdLine[gCLCursor].erase(gCursor, 1);
            }
        }

        gCCandidate.clear();
    }
    else if(key == 8 || key == KEY_BACKSPACE || key == 127) {    // CTRL-H
        if(gCursor > 0) {
            if(is_kanji(gCmdLine[gCLCursor][gCursor-1])) {
                string_erase(gCmdLine[gCLCursor], gCursor-2);
                string_erase(gCmdLine[gCLCursor], gCursor-2);
                
                //gCmdLine[gCLCursor].erase(gCursor-2, 2);
                cursor_move(-2);
            } else {
                string_erase(gCmdLine[gCLCursor], gCursor-1);
                
                //gCmdLine[gCLCursor].erase(gCursor-1, 1);
                cursor_move(-1);
            }
        }

        gCCandidate.clear();
    }
    else if(key == 11) {    // CTRL-K
//        gCmdLine[gCLCursor].erase(gCursor, gCmdLine[gCLCursor].size()-gCursor);

        string::iterator i = gCmdLine[gCLCursor].end();
        i--;
        for(int j=gCmdLine[gCLCursor].size()-1;
            j>=0;
            j--)
        {
            if(j >= gCursor) {
                gCmdLine[gCLCursor].erase(i);
            }
            i--;
        }

        gCCandidate.clear();
    }
    else if(key == 14 || key == KEY_DOWN) {    // CTRL-N
        cmdline_move(1);
        gCursor = gCmdLine[gCLCursor].size();

        gCCandidate.clear();
    }
    else if(key == 16 || key == KEY_UP) {    //CTRL-P
        cmdline_move(-1);
        gCursor = gCmdLine[gCLCursor].size();

        gCCandidate.clear();
    }
    else if(key == 12) {// CTRL-L
        clear();
        gLDir->Reread();
        gRDir->Reread();
    }
    else if(key == 3) { // CTRL-C
        //gMainLoop = false;

        gCmdLineActive = false;

        if(gCmdLine[gCmdLine.size()-1].length() == 0) {
            gCmdLine.pop_back();
        }

        gCCandidate.clear();
    }
    else if(key == 7 || key == 27) {    // CTRL-G Escape
        gCmdLineActive = false;

        if(gCmdLine[gCmdLine.size()-1].length() == 0) {
            gCmdLine.pop_back();
        }

        gCCandidate.clear();
    }
    else if(key == 10 || key == 13) {		// CTRL-J CTRL-M
/*
        /// !cmd ///
        if(gCmdLine[gCLCursor].length()>0 && gCmdLine[gCLCursor][0]=='!')
        {
            if(gCmdLine[gCLCursor].length() == 2
                && gCmdLine[gCLCursor][0]=='!'
                && gCmdLine[gCLCursor][1]=='!'
                && (gCLCursor-1) >= 0)
            {
                cmdline_run((char*)gCmdLine[gCLCursor-1].c_str()
                              , (char*)gCmdLine[gCLCursor-1].c_str());
            }
            else if(gCmdLine[gCLCursor].length() > 0
                    && gCmdLine[gCLCursor][0] == '!'
                    && (gCLCursor-1) >= 0)
            {
                /// get cmd ///
                char cmd[1024];
                int c = 0;
                for(int i=1;
                    gCmdLine[gCLCursor][i]!=' '
                            && i<gCmdLine[gCLCursor].length();
                    i++)
                    {
                    cmd[c++] = gCmdLine[gCLCursor][i];
                    }
                cmd[c] = 0;
            
                /// search command in history ///
                for(int i=gCLCursor-1; i>=0; i--) {
                    char cmd2[1024];

                    string cmdline = gCmdLine[i];

                    int c2 = 0;
                    int j;
                    for(j=0; cmdline[j]!=' ' && j<cmdline.length(); j++) {
                        cmd2[c2++] = cmdline[j];
                    }
                    cmd2[c2] = 0;
    
                    if(strstr(cmd2, cmd) == cmd2) {
                        cmdline_run((char*)gCmdLine[i].c_str()
                                        , (char*)gCmdLine[i].c_str());
                                        
                        break;
                    }                
                }
            }

            /// remove command ///
            int j=0;
            for(vector<string>::iterator i = gCmdLine.begin();
                i!=gCmdLine.end();
                i++)
            {
                if(j == gCLCursor) {
                    gCmdLine.erase(i);
                    break;
                }
               
                j++;
            }

            const int size = gCmdLine.size();
            if(size-1 >= 0 && gCmdLine[size-1].length() == 0)
                gCmdLine.pop_back();
        }
        else {
*/        
            cmdline_run((char*)gCmdLine[gCLCursor].c_str()
                                , (char*)gCmdLine[gCLCursor].c_str());

            /// run history ///
            const int size = gCmdLine.size();
            if(gCLCursor != size-1) {
                if(gCmdLine[size-1].length() == 0) gCmdLine.pop_back();

                if(gCmdLine[gCLCursor].length() != 0)
                    gCmdLine.push_back(gCmdLine[gCLCursor]);

                int j=0;
                for(vector<string>::iterator i = gCmdLine.begin();
                    i!=gCmdLine.end();
                    i++)
                {
                    if(j == gCLCursor) {
                        gCmdLine.erase(i);
                        break;
                    }
            
                    j++;
                }
            }
//        }
        
        gCmdLineActive = false;

        gCCandidate.clear();
    }
    else if(key == 9) {        // CTRL-I
        string cmdline = gCmdLine[gCLCursor];

        /// get editing text and position ///
        int i = 0;
        int editing_position = 0;

        char cmd_name[512];
        cmd_name[0] = 0;
        
        char editing[512];
        editing[0] = 0;
        char* e = editing;
        
        while(i < gCursor) {
            if(cmdline[i] == ' ' || cmdline[i] == '\t') {
                i++;

                editing[0] = 0;
                e = editing;
            }
            else if(cmdline[i] == '|' || cmdline[i] == ';' || cmdline[i] == '&') {
                i++;

                editing[0] = 0;
                e = editing;
                
                editing_position = 0;
            }
            else {
                editing[0] = 0;
                e = editing;
                
                while(i < gCursor) {
                    if(cmdline[i]==' ' || cmdline[i]=='\t' || cmdline[i]=='|' || cmdline[i]==';' || cmdline[i]=='&') {
                        editing_position++;
                        break;
                    }
                    *e++ = cmdline[i++];
                }
                *e = 0;

                if(editing_position == 1) {
                    strcpy(cmd_name, editing);
                }
            }
        }

M(("cmdname %s editing %s editing_position %d", cmd_name, editing, editing_position));
        
        /// divide editing into dir part and file part ///
        char editing_dir[512];
        char editing_file[512];

        strcpy(editing_dir, "");
        strcpy(editing_file, "");
        
        e = editing + strlen(editing);
        while(e >= editing) {
            if(*e == '/') {
                memcpy(editing_dir, editing, e - editing + 1);
                editing_dir[e-editing + 1] = 0;
    
                memcpy(editing_file, e + 1, editing + strlen(editing) - e);
                editing_file[editing + strlen(editing) -e] = 0;
                break;
            }
                
            e--;
        }
    
M(("editing %s dir %s editing_file %s", editing, editing_dir, editing_file));        
        if(strcmp(editing_dir, "") == 0) {
            strcpy(editing_file, editing);
        }
        
        if(strcmp(editing_dir, "") != 0) {
            DIR* dir = opendir(editing_dir);
            vector<string> files;

            if(dir) {
                struct dirent* entry;
                while(entry = readdir(dir)) {
                    if(strcmp(entry->d_name, ".") != 0
                        && strcmp(entry->d_name, "..") != 0)
                    {
                    char buf[PATH_MAX];
                    
                    struct stat _stat;
                    strcpy(buf, editing_dir);
                    strcat(buf, entry->d_name);
                    stat(buf, &_stat);

                    strcpy(buf, entry->d_name);
                    if(S_ISDIR(_stat.st_mode)) {
                        strcat(buf, "/");
                    }
                    
                    files.push_back(buf);
                    }
                }

                closedir(dir);

                sort(files.begin(), files.end(), sort_name);
                completion(files, editing_file);
            }
        }
        else {
            /// editing program name ///
            if(editing_position == 0) {
                completion(gPrograms, editing);
            }
            
            /// editing file name ///
            else {
                DIR* dir = opendir(".");
                vector<string> files;

                if(dir) {
                    struct dirent* entry;
                    while(entry = readdir(dir)) {
                        if(strcmp(entry->d_name, ".") != 0
                            && strcmp(entry->d_name, "..") != 0)
                        {
                        char buf[PATH_MAX];
                    
                        struct stat _stat;
                        strcpy(buf, "./");
                        strcat(buf, entry->d_name);
                        stat(buf, &_stat);

                        strcpy(buf, entry->d_name);
                        if(S_ISDIR(_stat.st_mode)) {
                            strcat(buf, "/");
                        }
                    
                        files.push_back(buf);
                        }
                    }

                    closedir(dir);

                    sort(files.begin(), files.end(), sort_name);
                    completion(files, editing_file);
                }
            }
        }
    }
    else if(is_kanji(key)) {
        int key2 = getch();

        char tmp[3];
        sprintf(tmp, "%c%c", key, key2);
        gCmdLine[gCLCursor].insert(gCursor, tmp);
        gCursor+=2;
    }
    else if(meta == 0 && ' ' <= key && key <= '}') {
        if(key == ' ') {
            gCCandidate.clear();
        }
    
        char tmp[2];
        sprintf(tmp, "%c", key);
        gCmdLine[gCLCursor].insert(gCursor, tmp);
        gCursor++;
    }
    
TEND();
}
  
void cmdline_view()
{
TBEGIN();

    const int maxx = getmaxx(stdscr);
    const int maxy = getmaxy(stdscr);
    
    mvprintw(maxy - 2, 0, "$ ");
    bool kanji = false;
    bool kanji_zure = false;
    for(int i=0; i<gCmdLine[gCLCursor].size(); i++) {
        if(kanji) {
            kanji = false;
        }
        else if(is_kanji(gCmdLine[gCLCursor][i])) {
            kanji = true;
        }

        if(i+2 < maxx-1) {
            mvprintw(maxy -2, 2 + i, "%c", gCmdLine[gCLCursor][i]);
        }
        else if(i+2 == maxx-1) {
            if(kanji) {
                mvprintw(maxy -1, 2 + i - maxx+1, "%c", gCmdLine[gCLCursor][i]);
                kanji_zure = true;
            }
            else {
                mvprintw(maxy -2, 2 + i, "%c", gCmdLine[gCLCursor][i]);
            }
        }
        else {
            if(kanji_zure)
                mvprintw(maxy -1, 2 + i - maxx+1, "%c", gCmdLine[gCLCursor][i]);
            else
                mvprintw(maxy -1, 2 + i - maxx, "%c", gCmdLine[gCLCursor][i]);
        }
    }

    if(kanji_zure) {    
        if(gCursor+2 < maxx-1)
            move(maxy -2, 2 + gCursor);
        else
            move(maxy-1, 2+ gCursor -maxx+1);
    }
    else {
        if(gCursor+2 < maxx)
            move(maxy -2, 2 + gCursor);
        else
            move(maxy -1, 2 + gCursor - maxx);
    }
    
TEND();
}

void cmdline_completion_view()
{
TBEGIN();

    const int maxx = getmaxx(stdscr);
    const int maxy = getmaxy(stdscr);

    int row = 0;
    int line = 0;
    for(vector<string>::iterator i = gCCandidate.begin(); i != gCCandidate.end(); i++) {
        if(row == 4) {
            row = 0;
            line++;
        }

        if(line >= maxy-2) break;
        
        mvprintw(line, row * (maxx/4), "%s", (*i).c_str());
        row++;
    }

TEND();
}
