/*
 * Copyright (c) 2003 Shinji Kashihara. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies
 * this distribution, and is available at cpl-v10.html.
 */
package mergedoc.ui;

import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.nio.charset.Charset;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;

import javax.swing.BoxLayout;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.filechooser.FileFilter;

import mergedoc.MergeDocException;

/**
 * ファイル選択フィールドです。
 * @author Shinji Kashihara
 */
public class FileChooserField extends JPanel {

    /** ラベル部 */
    private JLabel label = new JLabel();

    /** テキストフィールド部 */
    private JTextField textField = new JTextField();

    /** ボタン部 */
    private JButton button = new JButton();

    /** コンボボックス部 */
    private JComboBox combo = new JComboBox();

    /** コンボボックスに表示する文字セット名の Set */
    private static Set charsets;

    /** コンボボックス横幅 */
    private static int comboWidth;

    /** デフォルトエンコーディング */
    private static final String DEFAULT_ENCODING = System.getProperty("file.encoding");

    /** ファイル選択ダイアログ */
    private JFileChooser chooser = new JFileChooser();

    /** ファイル選択時のリスナー */
    private ActionListener chooseListener;

    /**
     * コンストラクタです。
     * デフォルトではコンボボックスの文字セットはデフォルトエンコーディングが
     * 選択状態となります。
     */
    public FileChooserField() throws MergeDocException {

        setLayout(new BoxLayout(this, BoxLayout.X_AXIS));

        // ラベルの設定
        add(label);
        add(ComponentFactory.createSpacer(10, 0));

        // テキストフィールドの設定
        int maxWidth = (int) ComponentFactory.createMaxDimension().getWidth();
        textField.setMaximumSize(new Dimension(maxWidth, 20));
        add(textField);

        // ボタンの設定
        ComponentFactory.ensureSize(button, 20, 18);
        button.setText("...");
        add(button);
        add(ComponentFactory.createSpacer(5, 0));

        // コンボの設定
        initCharsets();
        ComponentFactory.ensureSize(combo, comboWidth, 20);
        combo.setModel(new DefaultComboBoxModel(charsets.toArray()));
        setCharset( Charset.forName(DEFAULT_ENCODING) );
        add(combo);

        // ファイル選択ダイアログの設定
        button.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                int result = chooser.showOpenDialog(new JFrame());
                if (result == JFileChooser.APPROVE_OPTION) {
                    textField.setText(chooser.getSelectedFile().getPath());
                    if (chooseListener != null) {
                        chooseListener.actionPerformed(e);
                    }
                }
            }
        });
        setDirectoryOnly(false);
    }

    /**
     * コンボボックスに表示する文字セット名の Set を初期化します。
     * この Set はインスタンス間で共有されるため最初の 1 回だけ初期化されます。
     * ただし同期化は考慮されていません。
     */
    private void initCharsets() {

        if (charsets != null) return;
        
        // デフォルトを含む文字セットの Set を作成
        charsets = new TreeSet();
        charsets.add( Charset.forName("EUC-JP") );
        charsets.add( Charset.forName("ISO-2022-JP") );
        charsets.add( Charset.forName("MS932") );
        charsets.add( Charset.forName("Shift_JIS") );
        charsets.add( Charset.forName("UTF-16") );
        charsets.add( Charset.forName("UTF-16BE") );
        charsets.add( Charset.forName("UTF-16LE") );
        charsets.add( Charset.forName("UTF-8") );
        charsets.add( Charset.forName(DEFAULT_ENCODING) );

        // 表示上の最大幅を求める
        FontMetrics metrics = getFontMetrics(combo.getFont());
        for (Iterator it = charsets.iterator(); it.hasNext();) {
            int width = metrics.stringWidth(it.next().toString());
            if (comboWidth < width) {
                comboWidth = width;
            }
        }
        comboWidth += 30; //プルダウン部を加算
    }

    /**
     * ファイル選択ダイアログで、アーカイブファイルのみ、ディレクトリのみを
     * 選択できるようにします。デフォルトは false（アーカイブファイルのみ）です。 
     * @param directoryOnly ディレクトリのみにする場合は true
     */
    public void setDirectoryOnly(boolean directoryOnly) {
        if (directoryOnly) {
            chooser = new JFileChooser();
            chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
            chooser.setFileFilter(new FileFilter() {
                public boolean accept(File f) {
                    return f.isDirectory();
                }
                public String getDescription() {
                    return "ディレクトリのみ";
                }
            });
        } else {
            chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            chooser.setFileFilter(new FileFilter() {
                public boolean accept(File f) {
                    return f.isDirectory() ||
                           f.getName().endsWith(".zip") ||
                           f.getName().endsWith(".jar");
                }
                public String getDescription() {
                    return "*.zip, *.jar";
                }
            });
        }
    }
    
    /**
     * ファイル選択時のリスナーをセットします．
     * @param chooseListener ファイル選択時のリスナー
     */    
    public void setChooseListener(ActionListener chooseListener) {
        this.chooseListener = chooseListener;
    }

    /**
     * ラベルを取得します。
     * @return ラベル
     */
    public JLabel getLabel() {
        return label;
    }

    /**
     * ファイルを設定します。
     * @param file ファイル
     */
    public void setFile(File file) {
        textField.setText(file.getPath());
        chooser.setCurrentDirectory(file.getParentFile());
        chooser.setSelectedFile(file);
    }

    /**
     * ファイルを取得します。
     * @return ファイル
     */
    public File getFile() {
        return new File(textField.getText());
    }
    
    /**
     * 文字セットを設定します。
     * @param charset 文字セット
     */
    public void setCharset(Charset charset) {
        combo.setSelectedItem(charset);
    }
    
    /**
     * 文字セットを取得します。
     * @return 文字セット
     */
    public Charset getCharset() {
        return (Charset) combo.getSelectedItem();
    }
}
