/*
 * Copyright (c) 2003 Shinji Kashihara. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies
 * this distribution, and is available at cpl-v10.html.
 */
package mergedoc.core;



/**
 * シグネチャです。
 * @author Shinji Kashihara
 */
public class Signature {

    /** 短い形式のクラス名（パッケージ部分除く） */
    private String shortClassName;

    /** メソッド名 */
    private String methodName;

    /** 引数 */
    private String arguments;

    /** インナークラス宣言判定 */
    private boolean declareInnerClass;

    /**
     * コンストラクタです。
     * @param className このシグネチャが属するクラス名（短い形式でも良い）
     * @param sig シグネチャ文字列
     */
    public Signature(String className, String sig) {

        // 改行をスペースに置換
        sig = sig.replaceAll("(\\r|\\n)", " ");

        // クラスの拡張部を除去
        sig = sig.replaceFirst(" (extends|implements) .*", " ");

        // 先頭、末尾にスペース付加
        sig = sig.replaceAll("(^|$)", " ");
            
        // "(" ")" "," の前後にスペース付加
        sig = sig.replaceAll("(\\(|\\)|,)", " $1 ");
        
        // 複数スペースをスペース1個に置換（\\t含む）
        sig = sig.replaceAll("\\s+", " ");

        // 型がフルクラス名の場合はパッケージ名部分を除去
        // ex) " java.io.Serializable" -> "Serializable"
        sig = sig.replaceAll(" (\\w+?\\.)+", " ");

        // 引数がある場合
        if (sig.matches(".*?\\((.+?)\\).*?")) {

            // final を除去
            sig = sig.replaceAll( " final ", " ");

            // "[]" の前のスペース除去
            sig = sig.replaceAll( " \\[\\]", "[]");

            // 引数名を取り除き、型のみにする。
            // 配列の場合、型ではなく変数名に [] がついている場合があるので
            // 型の後ろに [] をつけるように統一する。
            //   ex) public void get(String str[], int num1, int[] num2)
            //    -> public void get(String[],int,int[])
            sig = sig.replaceAll(" \\w+?( ,| \\)|\\[\\] (,|\\)))", "$1");
        }


        this.shortClassName = className.replaceFirst(".+\\.", "");

        // クラスシグネチャの場合の宣言クラス名を取得
        String classRegex = ".*? (class|interface) (\\w+) .*";
        if (sig.matches(classRegex)) {
            String declaClassName = sig.replaceFirst(classRegex, "$2");

            // インナークラス宣言の場合はこのシグネチャの属する
            // クラス名としてメンバに上書きする
            if (!declaClassName.equals(shortClassName)) {
                this.declareInnerClass = true;
                this.shortClassName = declaClassName;
            }
        }

        // name(type,,,) の形式に変換
        sig = sig.replaceFirst( ".* (\\w+?(| \\(.*?\\)))(|\\[\\]) $", "$1");
        sig = sig.replaceAll(" +", "");
        //System.out.println(shortClassName + "#" + sig);

        this.methodName = sig.replaceFirst("(\\w+).*", "$1");
        this.arguments = sig.replaceFirst("\\w+(.*)", "$1");
    }
    
    /**
     * インナークラス宣言シグネチャか判定します．
     * @return インナークラス宣言シグネチャの場合は true
     */
    public boolean isDeclareInnerClass() {
        return declareInnerClass;
    }
    
    /**
     * このシグネチャが属するクラス名を取得します．
     * @return このシグネチャが属するクラス名
     */
    public String getClassName() {
        return shortClassName;
    }
    
    /**
     * 他のシグネチャと等しいか比較します．
     * @param obj 他のシグネチャ
     * @return 等しい場合は true 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object obj) {
        if (obj == null) return false; 
        return toString().equals(obj.toString());
    }

    /**
     * このオブジェクトのハッシュコードを取得します．
     * @return ハッシュコード
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return toString().hashCode();
    }

    /**
     * このオブジェクトの文字列表現を取得します．
     * 形式は "属するクラス名#メソッド名(引数型,,,)" となります．
     * @return 文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return shortClassName + "#" + methodName + arguments;
    }
}
