/*
 * Copyright (C) 2000-2003 ASANO Masahiro
 */

#include "def.h"

#define __KERNEL__
#include <linux/fs.h>
#include <linux/mm.h>

#include "printdev.h"
#include "flags_fs.h"

void
mprint_kdev_t(kdev)
	kdev_t kdev;
{
	mprintf("%3d,%-3d", MAJOR(kdev), MINOR(kdev));
}

void
mprint_dev_t(dev)
	dev_t dev;
{
	mprintf("%3d,%-3d", MAJOR(dev), MINOR(dev));
}

void
prhead_buffer_head()
{
	mprintf(SPTR"    BLOCK SIZE CNT   DEV    RDEV   "SPTR" "SPTR" FLAG\n",
		"ADDR", "DATA", "PAGE");
}

addr_t
print_buffer_head(addr, full, lru)
	addr_t addr;
	int full;
	int lru;
{
	struct buffer_head buf;

	memread(addr, sizeof(struct buffer_head), &buf, "buffer_head");
	mprintf(FPTR " ", addr);
	mprintf("%8lx %4x %3x ", buf.b_blocknr, buf.b_size, ATOMIC_READ(buf.b_count));
	mprint_kdev_t(buf.b_dev);
	mprintf(" ");
	mprint_kdev_t(buf.b_rdev);
	mprintf(" " FPTR " " FPTR, buf.b_data, buf.b_page);
	mprintbit(bh_state_bits_bitname, buf.b_state);
	mprintf("\n");
	if (full) {
		mprintf("\t"SPTR" "SPTR" "SPTR" "SPTR"  RSECTOR "SPTR"\n", "FLUSH", "PAGE", "END_IO", "PRIVATE", "REQNEXT");
		mprintf("\t" FPTR " " FPTR " " FPTR " " FPTR " %8lx " FPTR "\n\n",
			buf.b_flushtime, buf.b_page, buf.b_end_io,
			buf.b_private, buf.b_rsector, buf.b_reqnext);
	}
	return (lru == -1)? (addr_t)buf.b_next: (addr_t)buf.b_next_free;
}

void
prhead_dentry()
{
	mprintf(SPTR" CNT "SPTR" "SPTR" HLCSA "SPTR" "SPTR" F NAME\n",
		"ADDR", "INODE", "PARENT", "D_OP", "SB");
}

addr_t
print_dentry(addr, sw, subdir, parent)
	addr_t addr;
	int sw, subdir, parent;
{
	struct dentry de;
	addr_t child, next;

	switch (sw) {
	case 1: /* hash */
		addr -= OFFSET(struct dentry, d_hash);
		break;
	case 2: /* lru */
		addr -= OFFSET(struct dentry, d_lru);
		break;
	}
	memread(addr, sizeof(de), &de, "dentry");

	if (subdir) {
		for (next = (addr_t)de.d_subdirs.next; next != (addr_t)&((struct dentry *)addr)->d_subdirs; next = (addr_t)de.d_child.next) {
			child = next - OFFSET(struct dentry, d_child);
			memread(child, sizeof(de), &de, "dentry");
			(void)print_dentry(child, 0, 0, 0);
		}
		return 0;
	}

	mprintf(FPTR, addr);
	mprintf(" %3x " FPTR " " FPTR, ATOMIC_READ(de.d_count), de.d_inode, de.d_parent);
	mprintf(" %c%c%c%c%c ",
		(de.d_hash.next   !=&((struct dentry *)addr)->d_hash   )? 'H': '-',
		(de.d_lru.next    !=&((struct dentry *)addr)->d_lru    )? 'L': '-',
		(de.d_child.next  !=&((struct dentry *)addr)->d_child  )? 'C': '-',
		(de.d_subdirs.next!=&((struct dentry *)addr)->d_subdirs)? 'S': '-',
		(de.d_alias.next  !=&((struct dentry *)addr)->d_alias  )? 'A': '-');
	mprintf(FPTR " " FPTR " ", de.d_op, de.d_sb);
#if SUBLEVEL>=4
	mprintf("%lx ", de.d_vfs_flags);
#endif
	if (de.d_name.name == ((struct dentry *)addr)->d_iname)
		mprintstr(de.d_iname, sizeof(de.d_iname));
	else if (de.d_name.name == NULL)
		mprintf("[null]");
	else {
		char qbuf[256];
		int len = de.d_name.len < sizeof(qbuf)? de.d_name.len: sizeof(qbuf);
		memread((addr_t)de.d_name.name, len, qbuf, "dentry.d_name");
		mprintstr(qbuf, len);
	}
	mprintf("\n");

	if (parent) {
		if (de.d_parent && (addr_t)de.d_parent != addr) {
			return print_dentry(de.d_parent, 0, 0, 1);
		}
		return 0;
	}

	if (sw == 2)
		return (addr_t)de.d_lru.next;
	else
		return (addr_t)de.d_hash.next;
}

void
prhead_file()
{
	mprintf(SPTR" "SPTR" "SPTR" MOD       POS CNT   UID   GID ERR "SPTR" FLAGS\n",
		"ADDR", "DENTRY", "F_OP", "PRIVATE");
}

addr_t
print_file(addr, full)
	addr_t addr;
	int full;
{
	struct file f;
	static const struct bitname fflags[] = {
		{ O_APPEND,	"append" },
		{ O_NONBLOCK,	"nonblock" },
		{ O_SYNC,	"sync" },
		{ FASYNC,	"fasync" },
		{ O_DIRECT,	"direct" },
		{ O_LARGEFILE,	"largefile" },
		{ 0,		NULL }
	};

	memread(addr, sizeof(f), &f, "file");
	mprintf(FPTR " ", addr);

	mprintf(FPTR " " FPTR " ", f.f_dentry, f.f_op);
	switch (f.f_mode) {
	case 0:	mprintf(" -");	break;
	case 1: mprintf("r ");	break;
	case 2: mprintf(" w");	break;
	case 3: mprintf("rw");	break;
	default:mprintf(" ?");	break;
	}
	mprintf(" %10Lx %3x %5d %5d %3d", f.f_pos, ATOMIC_READ(f.f_count), f.f_uid, f.f_gid, f.f_error);
	mprintf(" " FPTR, f.private_data);
	mprintbit(fflags, f.f_flags & ~3);
	mprintf("\n");

	if (full) {
		mprintf("\tvfsmnt:     " FPTR "\n", f.f_vfsmnt);
		mprintf("\treada:      %lx  %lx  %lx  %lx  %lx\n",
			f.f_reada, f.f_ramax, f.f_raend,
			f.f_ralen, f.f_rawin);
		mprintf("\towener.pid: %x\n", f.f_owner.pid);
		mprintf("\tversion:    %lx\n", f.f_version);
	}
	return (addr_t)f.f_list.next;
}

void
prhead_filesystem()
{
	mprintf(SPTR"        NAME "SPTR" "SPTR" FLAG\n",
		"ADDR", "READ_SP", "OWNER");
}

addr_t
print_file_system_type(addr)
	addr_t addr;
{
	struct file_system_type fs;
	char buf[16];

	mprintf(FPTR, addr);
	memread(addr, sizeof(fs), &fs, "file_system_type");
	memread((addr_t)fs.name, sizeof(buf), buf, "file system type name");

#ifdef KERN_MNT
	mprintf(" %11s " FPTR " " FPTR " " FPTR, buf, fs.read_super, fs.owner, fs.kern_mnt);
#else
	mprintf(" %11s " FPTR " " FPTR, buf, fs.read_super, fs.owner);
#endif
	mprintbit(fsflags, fs.fs_flags);
	mprintf("\n");
	return (addr_t)fs.next;
}

void
prhead_inode()
{
	mprintf(SPTR"        INO CNT   DEV  LINK MODE   UID       SIZE "SPTR" HLDD\n",
		"ADDR", "MAPPING");
}

addr_t
print_inode(addr, full)
	addr_t addr;
	int full;
{
	struct inode inode;
	const char *p;

	memread(addr, sizeof(struct inode), &inode, "inode");
	mprintf(FPTR " ", addr);
	mprintf("%10ld %3x ", inode.i_ino, ATOMIC_READ(inode.i_count));
	mprint_kdev_t(inode.i_dev);
	mprintf(" %3x ", inode.i_nlink);
	mprint_mode(inode.i_mode);
	mprintf(" %5d %10Lx", inode.i_uid, inode.i_size);
	mprintf(" " FPTR, inode.i_mapping);
	mprintf(" %c%c%c%c\n",
		(inode.i_hash.next         !=&((struct inode *)addr)->i_hash         )? 'H': '-',
		(inode.i_list.next         !=&((struct inode *)addr)->i_list         )? 'L': '-',
		(inode.i_dentry.next       !=&((struct inode *)addr)->i_dentry       )? 'D': '-',
		(inode.i_dirty_buffers.next!=&((struct inode *)addr)->i_dirty_buffers)? 'D': '-');

	if (full) {
		mprintf("\trdev:      ");
		mprint_kdev_t(inode.i_rdev);
		mprintf("\n");
		mprintf("\t[amc]time: %08lx  %08lx  %08lx  (date)\n",
			inode.i_atime,
			inode.i_mtime,
			inode.i_ctime);
		mprintf("\tblocks:    %ld  (* blksize %ld)\n", inode.i_blocks, inode.i_blksize);
		mprintf("\tversion:   %ld\n", inode.i_version);
		mprintf("\tsem:       ");
		mprint_semaphore(&inode.i_sem);
		mprintf("\n");
		mprintf("\top:        " FPTR "  (operation -i)  ", inode.i_op);
		p = getsymstr((addr_t)inode.i_op);
		if (p)
			mprintf("(%s)\n", p);
		else
			mprintf("\n");
		mprintf("\tfop:       " FPTR "  (operation -f)  ", inode.i_fop);
		p = getsymstr((addr_t)inode.i_fop);
		if (p)
			mprintf("(%s)\n", p);
		else
			mprintf("\n");
		mprintf("\tsb:        " FPTR "  (super_block)\n", inode.i_sb);
		mprintf("\tflock:     " FPTR "  (file_lock)\n", inode.i_flock);
		if (inode.i_pipe)
		mprintf("\tpipe:      " FPTR "\n", inode.i_pipe);
		if (inode.i_bdev)
		mprintf("\tbdev:      " FPTR "  (block_device)\n", inode.i_bdev);
#ifdef _inode_has_i_cdev
		if (inode.i_cdev)
		mprintf("\tcdev:      " FPTR "  (char_device)\n", inode.i_cdev);
#endif
		mprintf("\tstate:     %lx ", inode.i_state);
		mprintbit(istate, inode.i_state);
		mprintf("\n\tflags:     %x\n", inode.i_flags);
		mprintf("\tgen:       %u\n", inode.i_generation);
		mprintf("\tu:        *" FPTR "  = " FPTR "\n",
			addr + OFFSET(struct inode, u), inode.u.generic_ip);
		mprintf("\n");
	}
	return (addr_t)inode.i_hash.next;
}

void
prhead_superblock()
{
	mprintf(SPTR"   DEV   BSIZE D        TYPE "SPTR" "SPTR" "SPTR" LDIF FLAGS\n",
		"ADDR", "S_OP", "ROOT", "BDEV");
}

addr_t
print_superblock(addr, full)
	addr_t addr;
	int full;
{
	struct super_block sb;
	struct file_system_type fstype;
	char buf[16];

	memread(addr, sizeof(struct super_block), &sb, "super_block");
	mprintf(FPTR " ", addr);

	mprint_kdev_t(sb.s_dev);
	mprintf(" %5lx %x ", sb.s_blocksize, sb.s_dirt);

	if (sb.s_type) {
		memread((addr_t)sb.s_type, sizeof(fstype), &fstype, "file system type");
		memread((addr_t)fstype.name, sizeof(buf), buf, "file system type name");
		mprintf("%11s", buf);
	} else {
		mprintf("%11s", "-");
	}

	mprintf(" " FPTR " " FPTR " " FPTR, sb.s_op, sb.s_root, sb.s_bdev);

#define ISEMPTY(x)	((addr_t)sb.x.next == addr+OFFSET(struct super_block,x))
	mprintf(" %c%c%c%c",
		ISEMPTY(s_list)         ? '-': 'l',
		ISEMPTY(s_dirty)        ? '-': 'd',
		ISEMPTY(s_locked_inodes)? '-': 'i',
		ISEMPTY(s_files)        ? '-': 'f');

	mprintbit(sflags, sb.s_flags);
	mprintf("\n");

	if (full) {
		mprintf("\tmaxbytes:         %llx\n", sb.s_maxbytes);
		mprintf("\tdquot_operations: " FPTR "\n", sb.dq_op);
		mprintf("\tmagic:            %lx\n", sb.s_magic);
		mprintf("\tlock:             ");
		mprint_semaphore(&sb.s_lock);
		mprintf("\n");
#ifdef _super_block_has_s_count
		mprintf("\tcount:            %x\n", sb.s_count);
#endif
#ifdef _super_block_has_s_active
		mprintf("\tactive:           %x\n", ATOMIC_READ(sb.s_active));
#endif
#define SEMPTY(list) \
	(addr + OFFSET(struct super_block, list) == (addr_t)sb.list.next)
		mprintf("\tdirty:            ");
		if (SEMPTY(s_dirty))
			mprintf(SPTR " " SPTR "\n", "-", "-");
		else
			mprintf(FPTR " " FPTR "\n",
				sb.s_dirty.next, sb.s_dirty.prev);
		mprintf("\tlocked_inodes:    ");
		if (SEMPTY(s_locked_inodes))
			mprintf(SPTR " " SPTR "\n", "-", "-");
		else
			mprintf(FPTR " " FPTR "\n",
				sb.s_locked_inodes.next,
				sb.s_locked_inodes.prev);
		mprintf("\tfiles:            ");
		if (SEMPTY(s_files))
			mprintf(SPTR " " SPTR "\n", "-", "-");
		else
			mprintf(FPTR " " FPTR "\n",
				sb.s_files.next, sb.s_files.prev);
		mprintf("\tu:                " FPTR "\n",
			addr + OFFSET(struct super_block, u));
	}

	if (full >= 2) {
		addr_t faddr = (addr_t)sb.s_files.next;
		prhead_file();
		while (faddr != addr + OFFSET(struct super_block, s_files)) {
			faddr = print_file(faddr - OFFSET(struct file, f_list), 0);
		}
		mprintf("\n");
	}

	return (addr_t)sb.s_list.next;
}

addr_t
print_superblock_union(addr, fstype, func, arg)
	addr_t addr;
	addr_t fstype;
	addr_t (*func)();
	int arg;
{
	struct super_block sb;

	memread(addr, sizeof(struct super_block), &sb, "super_block");

	if ((addr_t)sb.s_type == fstype)
		func(sb.u.generic_sbp, addr + OFFSET(struct super_block, u), arg);
	return (addr_t)sb.s_list.next;
}

void
prhead_vfsmount()
{
	mprintf(SPTR" "SPTR" "SPTR" "SPTR" "SPTR"  CNT  F  DEVICE\n",
		"ADDR", "MOUNT-D", "ROOT-D", "PARENT", "SB");
}

addr_t
print_vfsmount(addr, chain)
	addr_t addr;
	int chain;
{
	struct vfsmount m;
	char path[PATH_MAX];

	if (chain) {
#if SUBLEVEL>=18
		addr -= OFFSET(struct vfsmount, mnt_hash);
#else
		addr -= OFFSET(struct vfsmount, mnt_list);
#endif
	}

	memread(addr, sizeof(struct vfsmount), &m, "vfsmount");
	mprintf(FPTR " ", addr);
	mprintf(FPTR " " FPTR " " FPTR " " FPTR " ",
		m.mnt_mountpoint, m.mnt_root, m.mnt_parent, m.mnt_sb);
	mprintf("%4x %2x  ", ATOMIC_READ(m.mnt_count), m.mnt_flags);

	if (m.mnt_devname) {
		memread((addr_t)m.mnt_devname, sizeof(path), path, "mnt_devname");
		mprintf("%s", path);
	} else {
		mprintf("-");
	}
	mprintf("\n");

	if (chain) {
#if SUBLEVEL>=18
		return (addr_t)m.mnt_hash.next;
#else
		return (addr_t)m.mnt_list.next;
#endif
	}
	return (addr_t)m.mnt_list.next - OFFSET(struct vfsmount, mnt_list);
}

void
print_files_stat(addr)
	addr_t addr;
{
	struct files_stat_struct filst;

	memread(addr, sizeof(filst), &filst, "files_stat");
	mprintf("nr_files:      %5x  (%d)\n", filst.nr_files, filst.nr_files);
	mprintf("nr_free_files: %5x  (%d)\n", filst.nr_free_files, filst.nr_free_files);
	mprintf("max_files:     %5x  (%d)\n", filst.max_files, filst.max_files);
}

void
print_address_space(addr, plist)
	addr_t addr;
	int plist;
{
	struct address_space as;
	extern void print_gfp();
	addr_t paddr, top;
	addr_t print_page();
	void prhead_page();

	mprintf("ADDR:         " FPTR "\n", addr);
	memread(addr, sizeof(as), &as, "address_space");

#undef ISEMPTY
#define ISEMPTY(list) \
	((addr_t)as.list.next == addr + OFFSET(struct address_space, list))
	mprintf("clean_pages:  ");
	if (ISEMPTY(clean_pages))
		mprintf(SPTR "  " SPTR "\n", "-", "-");
	else
		mprintf(FPTR "  " FPTR "\n",
			as.clean_pages.next, as.clean_pages.prev);
	mprintf("dirty_pages:  ");
	if (ISEMPTY(dirty_pages))
		mprintf(SPTR "  " SPTR "\n", "-", "-");
	else
		mprintf(FPTR "  " FPTR "\n",
			as.dirty_pages.next, as.dirty_pages.prev);
	mprintf("locked_pages: ");
	if (ISEMPTY(locked_pages))
		mprintf(SPTR "  " SPTR "\n", "-", "-");
	else
		mprintf(FPTR "  " FPTR "\n",
			as.locked_pages.next, as.locked_pages.prev);

	mprintf("nrpages:      %8lx\n", as.nrpages);
	mprintf("ops:          " FPTR "\n", as.a_ops);
	mprintf("host:         " FPTR "\n", as.host);
	mprintf("mmap:         " FPTR "\n", as.i_mmap);
	mprintf("mmap_shared:  " FPTR "\n", as.i_mmap_shared);
	mprintf("gfp_mask:    ");
	print_gfp(as.gfp_mask);

	if (plist & 1) {
		prhead_page();
		top = addr + OFFSET(struct address_space, clean_pages) - OFFSET(struct page, list);
		paddr = (addr_t)as.clean_pages.next - OFFSET(struct page, list);
		while (paddr && paddr != top) {
			paddr = print_page(paddr, 1);
		}
	}
	if (plist & 2) {
		prhead_page();
		top = addr + OFFSET(struct address_space, dirty_pages) - OFFSET(struct page, list);
		paddr = (addr_t)as.dirty_pages.next - OFFSET(struct page, list);
		while (paddr && paddr != top) {
			paddr = print_page(paddr, 1);
		}
	}
	if (plist & 4) {
		prhead_page();
		top = addr + OFFSET(struct address_space, locked_pages) - OFFSET(struct page, list);
		paddr = (addr_t)as.locked_pages.next - OFFSET(struct page, list);
		while (paddr && paddr != top) {
			paddr = print_page(paddr, 1);
		}
	}
}

#if SUBLEVEL>=5
void
prhead_char_device()
{
	mprintf(SPTR" COUNT   DEV   OPEN SEM C SLP\n", "ADDR");
}

addr_t
print_char_device(addr)
	addr_t addr;
{
	struct char_device cd;

	memread(addr, sizeof(cd), &cd, "char_device");
	mprintf(FPTR " ", addr);

	mprintf("%5x ", ATOMIC_READ(cd.count));
	mprint_dev_t(cd.dev);
	mprintf(" %4x ", ATOMIC_READ(cd.openers));
	mprintf("%5d %3d\n", ATOMIC_READ(cd.sem.count), cd.sem.sleepers);
	return (addr_t)cd.hash.next;
}
#endif

void
prhead_block_device()
{
#if SUBLEVEL>=15
	mprintf(SPTR" COUNT   DEV   OPEN "SPTR" SEM C SLP "SPTR"  INODES..\n", "ADDR", "BD_OP", "INODE");
#else
	mprintf(SPTR" COUNT   DEV   OPEN "SPTR" SEM C SLP\n", "ADDR", "BD_OP");
#endif
}

addr_t
print_block_device(addr, eflag)
	addr_t addr;
	int eflag;
{
	struct block_device bd;
#if SUBLEVEL>=15
	addr_t iaddr;
	struct inode inode;
#endif

	memread(addr, sizeof(bd), &bd, "block_device");
#if SUBLEVEL>=10
	if (eflag == 0 && bd.bd_openers == 0)
		return (addr_t)bd.bd_hash.next;
#else
	if (eflag == 0 && ATOMIC_READ(bd.bd_openers) == 0)
		return (addr_t)bd.bd_hash.next;
#endif

	mprintf(FPTR " ", addr);
	mprintf("%5x ", ATOMIC_READ(bd.bd_count));
	mprint_dev_t(bd.bd_dev);
	mprintf(" %4x " FPTR, bd.bd_openers, bd.bd_op);
	mprintf(" %5d %3d", ATOMIC_READ(bd.bd_sem.count), bd.bd_sem.sleepers);
#if SUBLEVEL>=15
	mprintf(" " FPTR " ", bd.bd_inode);
	iaddr = (addr_t)bd.bd_inodes.next;
	while (iaddr && iaddr != addr + OFFSET(struct block_device, bd_inodes)) {
		iaddr -= OFFSET(struct inode, i_devices);
		mprintf(" " FPTR, iaddr);
		memread(iaddr, sizeof(inode), &inode, "inode");
		iaddr = (addr_t)inode.i_devices.next;
	}
#endif
	mprintf("\n");
	return (addr_t)bd.bd_hash.next;
}

void
prhead_file_lock()
{
	mprintf(SPTR " " SPTR "   PID " SPTR " T      START        END FLAGS\n",
		"ADDR", "OWNER", "FILE");
}

addr_t
print_file_lock(addr, full, chain)
	addr_t addr;
	int full;
	int chain;
{
	struct file_lock fl;

	switch (chain) {
	case 1:
		addr -= OFFSET(struct file_lock, fl_link);
		break;
	}
	memread(addr, sizeof(fl), &fl, "file_lock");
	mprintf(FPTR " " FPTR " %5d " FPTR " ",
		addr, fl.fl_owner, fl.fl_pid, fl.fl_file);
	switch (fl.fl_type) {
	case F_UNLCK:			mprintf("-");	break;
	case F_WRLCK:			mprintf("W");	break;
	case F_RDLCK:			mprintf("R");	break;
	case F_WRLCK | F_INPROGRESS:	mprintf("w");	break;
	case F_RDLCK | F_INPROGRESS:	mprintf("r");	break;
	default:			mprintf("?");	break;
	}
	mprintf(" %10llx", fl.fl_start);
	if (fl.fl_end == OFFSET_MAX) {
		mprintf(" offset_max");
	} else {
		mprintf(" %10llx", fl.fl_end);
	}
	mprintbit(flflags, fl.fl_flags);
	mprintf("\n");

	if (full) {
		mprintf("  next:       " FPTR "\n", fl.fl_next);
		mprintf("  block:      ");
		if ((addr_t)fl.fl_block.next == addr + OFFSET(struct file_lock, fl_block)) {
			mprintf(SPTR " " SPTR "\n", "-", "-");
		} else {
			mprintf(FPTR " " FPTR "\n",
				fl.fl_block.next, fl.fl_block.prev);
		}
		mprintf("  wait:       ");
		if ((addr_t)fl.fl_wait.task_list.next == addr + OFFSET(struct file_lock, fl_wait.task_list)) {
			mprintf(SPTR " " SPTR "\n", "-", "-");
		} else {
			mprintf(FPTR " " FPTR "\n",
				fl.fl_wait.task_list.next,
				fl.fl_wait.task_list.prev);
		}
		mprintf("  not/ins/rem:" FPTR " " FPTR " " FPTR "\n",
			fl.fl_notify, fl.fl_insert, fl.fl_remove);
		mprintf("  fasync:     " FPTR "\n", fl.fl_fasync);
#ifdef _file_lock_has_fl_break_time
		mprintf("  break_time: %8lx\n", fl.fl_break_time);
#endif
		mprintf("  nfs_fl:     %x  %x  %lx\n",
			fl.fl_u.nfs_fl.state,
			fl.fl_u.nfs_fl.flags,
			fl.fl_u.nfs_fl.host);
	}

	switch (chain) {
	default:
	case 0:	return (addr_t) fl.fl_next;
	case 1:	return (addr_t) fl.fl_link.next;
	case 2:	return (addr_t) fl.fl_block.next - OFFSET(struct file_lock, fl_block);
	}
}
