/*
 * Copyright (c) 2006-2008 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package org.maskat.ui.wizards.nature;

import java.net.URI;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.internal.ide.IDEWorkbenchMessages;
import org.eclipse.ui.internal.ide.IDEWorkbenchPlugin;
import org.eclipse.ui.internal.ide.filesystem.FileSystemSupportRegistry;
import org.maskat.ui.MaskatUIPlugin;
import org.maskat.ui.Messages;

/**
 * Standard main page for a wizard that is creates a project resource.
 * <p>
 * This page may be used by clients as-is; it may be also be subclassed to suit.
 * </p>
 * <p>
 * Example usage:
 * <pre>
 * mainPage = new WizardNewProjectCreationPage("basicNewProjectPage");
 * mainPage.setTitle("Project");
 * mainPage.setDescription("Create a new project resource.");
 * </pre>
 * </p>
 */
public class AddMaskatNatureCreationPage extends WizardPage {
	
	/** The wizard page ID */
	public static final String PAGE_ID = MaskatUIPlugin.PLUGIN_ID
			+ ".addMaskatNatureCreationPage";

    // widgets
    Text projectNameField;

    // constants
    private static final int SIZING_TEXT_FIELD_WIDTH = 250;
    
    private Text maskatFWPath;
    
    private Button shouldCopy;

	private IPath frameworkPath;

	private boolean shouldCopyFlag;
	
	private IProject project;
	
	public AddMaskatNatureCreationPage(IProject project, IPath path, boolean copy) {
		this(project, PAGE_ID, path, copy);
		setTitle(Messages.getString("wizard.nature.title")); //$NON-NLS-1$
		setDescription(Messages.getString("wizard.nature.description")); //$NON-NLS-1$
	}
	
    /**
     * Creates a new project creation wizard page.
     *
     * @param pageName the name of this page
     */
    public AddMaskatNatureCreationPage(IProject project, String pageName,
    		IPath path, boolean copy) {
        super(pageName);
        
        this.project = project;
        this.frameworkPath = path;
        this.shouldCopyFlag = copy;
        setPageComplete(false);
    }

    /** (non-Javadoc)
     * Method declared on IDialogPage.
     */
    public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NULL);
        composite.setFont(parent.getFont());

        initializeDialogUnits(parent);

        composite.setLayout(new GridLayout());
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        createProjectNameGroup(composite, project.getName());

        Group fwGroup = new Group(composite, SWT.NONE);
        GridLayout layout = new GridLayout(3, false);
        layout.marginLeft = 5;
        fwGroup.setLayout(layout);
        fwGroup.setLayoutData(new GridData(GridData.FILL_BOTH));
        
        fwGroup.setText(Messages.getString("wizard.nature.group.framework")); //$NON-NLS-1$
		Label dirLabel = new Label(fwGroup, SWT.NULL);
		dirLabel.setText(Messages.getString("wizard.nature.label.path")); //$NON-NLS-1$
		
		maskatFWPath = new Text(fwGroup, SWT.BORDER);
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 2;
		maskatFWPath.setLayoutData(gridData);
		maskatFWPath.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				setPageComplete(validatePage());
			}
		});
		shouldCopy = new Button(fwGroup, SWT.CHECK | SWT.RIGHT);
		GridData copyData = new GridData();
		copyData.horizontalSpan = 5;
		shouldCopy.setLayoutData(copyData);
		shouldCopy.setText(Messages.getString("wizard.nature.label.shuldCopy")); //$NON-NLS-1$
		
		// Scale the button based on the rest of the dialog
		initPage();
        setPageComplete(validatePage());
        // Show description on opening
        setErrorMessage(null);
        setMessage(null);
        setControl(composite);
    }
    
    private void initPage() {
    	maskatFWPath.setText(frameworkPath.toString());
    	shouldCopy.setSelection(shouldCopyFlag);
    }
    
    /**
     * Creates the project name specification controls.
     *
     * @param parent the parent composite
     */
    private final void createProjectNameGroup(Composite parent, String projectName) {
        // project specification group
        Composite projectGroup = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout();
        layout.numColumns = 2;
        projectGroup.setLayout(layout);
        projectGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

        // new project label
        Label projectLabel = new Label(projectGroup, SWT.NONE);
        projectLabel.setText(IDEWorkbenchMessages.WizardNewProjectCreationPage_nameLabel);
        projectLabel.setFont(parent.getFont());

        // new project name entry field
        projectNameField = new Text(projectGroup, SWT.BORDER);
        GridData data = new GridData(GridData.FILL_HORIZONTAL);
        data.widthHint = SIZING_TEXT_FIELD_WIDTH;
        projectNameField.setLayoutData(data);
        projectNameField.setFont(parent.getFont());
        projectNameField.setEnabled(false);
		projectNameField.setText(projectName);
    }
  
    /**
     * Returns whether this page's controls currently all contain valid 
     * values.
     *
     * @return <code>true</code> if all controls are valid, and
     *   <code>false</code> if at least one is invalid
     */
    protected boolean validatePage() {
        IWorkspace workspace = IDEWorkbenchPlugin.getPluginWorkspace();
        String path = maskatFWPath.getText();
		if (path == null || path.length() == 0) {
			setErrorMessage(Messages.getString(
					"wizard.nature.msg.error.notfwpath")); //$NON-NLS-1$);
			maskatFWPath.setFocus();
			return false;
		}
		URI fwUrl = FileSystemSupportRegistry.getInstance()
			.getDefaultConfiguration().getContributor().getURI(path);
		
        IStatus fwLocationStatus = workspace.validateProjectLocationURI(this.project, fwUrl);
        if (!fwLocationStatus.isOK()) {
            setErrorMessage(fwLocationStatus.getMessage());
			maskatFWPath.setFocus();
            return false;
        }
        setErrorMessage(null);
        setMessage(null);
        return true;
    }

	public boolean shouldCopyMaskatFramework() {
		return shouldCopy.getSelection();
	}

	public IPath getMaskatFrameworkPath() {
		return new Path(maskatFWPath.getText());
	}
}

