/*
 * Copyright (c) 2006-2008 Maskat Project.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Maskat Project - initial API and implementation
 */
package org.maskat.ui;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.preferences.IScopeContext;
import org.eclipse.jface.preference.IPersistentPreferenceStore;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.maskat.core.event.Event;

public class MaskatNature implements IProjectNature {

	/** The project nature ID */
	public static final String NATURE_ID = MaskatUIPlugin.PLUGIN_ID + ".maskatNature"; //$NON-NLS-1$

	public static final String FRAMEWORK_PATH = "frameworkPath"; //$NON-NLS-1$
	
	public static final String FRAMEWORK_PATH_DEFAULT = "/maskatFw/"; //$NON-NLS-1$

	public static final String GRID_SELECTION = "gridSelection"; //$NON-NLS-1$
	
	public static final String GRID_SELECTION_DEFAULT = "false"; //$NON-NLS-1$

	public static final String SNAPTOGRID_SELECTION = "snapToGridSelection"; //$NON-NLS-1$
	
	public static final String SNAPTOGRID_SELECTION_DEFAULT = "false"; //$NON-NLS-1$
	
	public static final String SNAPTOGEOMETRY_SELECTION = "snapToGeometrySelection"; //$NON-NLS-1$
	
	public static final String SNAPTOGEOMETRY_SELECTION_DEFAULT = "false"; //$NON-NLS-1$

	public static final String GRID_SIZE = "gridSize"; //$NON-NLS-1$
	
	public static final String GRID_SIZE_DEFAULT = "15"; //$NON-NLS-1$
	
	public static void addNature(IProject project, IProgressMonitor monitor)
			throws CoreException {
		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}
		if (monitor.isCanceled() || project == null) {
			return;
		}

		if (!project.hasNature(NATURE_ID)) {
			IProjectDescription description = project.getDescription();
			String[] oldIds = description.getNatureIds();
			String[] newIds = new String[oldIds.length + 1];
			System.arraycopy(oldIds, 0, newIds, 0, oldIds.length);
			newIds[oldIds.length] = NATURE_ID;
			description.setNatureIds(newIds);
			project.setDescription(description, monitor);
			project.refreshLocal(IResource.DEPTH_INFINITE, monitor);
		}
	}

	public static void removeNature(IProject project, IProgressMonitor monitor)
			throws CoreException {
		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}
		if (monitor.isCanceled() || project == null) {
			return;
		}

		if (project.hasNature(NATURE_ID)) {
			IProjectDescription description = project.getDescription();
			String[] oldIds = description.getNatureIds();
			String[] newIds = new String[oldIds.length - 1];
			for (int i = 0; i < oldIds.length; i++) {
				if (oldIds[i].equals(NATURE_ID)) {
					System.arraycopy(oldIds, 0, newIds, 0, i);
					System.arraycopy(oldIds, i + 1, newIds, i, newIds.length - i);
					break;
				}
			}
			description.setNatureIds(newIds);
			project.setDescription(description, monitor);
			project.refreshLocal(IResource.DEPTH_INFINITE, monitor);
		}
	}

	public static MaskatNature getNature(IProject project) {
		try {
			if (project != null && project.hasNature(NATURE_ID)) {
				return (MaskatNature) project.getNature(NATURE_ID);
			}
		} catch (CoreException e) {
			MaskatUIPlugin.log(e.getStatus());
		}
		return null;
	}

	private IProject project;

	private IPersistentPreferenceStore store;

	private EventEditorContributor contributor = null;
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.core.resources.IProjectNature#getProject()
	 */
	public IProject getProject() {
		return project;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.core.resources.IProjectNature#setProject(org.eclipse.core.resources.IProject)
	 */
	public void setProject(IProject project) {
		if (project == null) {
			return;
		}
		this.project = project;

		IScopeContext scope = new ProjectScope(project);
		store = new ScopedPreferenceStore(scope, MaskatUIPlugin.PLUGIN_ID);
		store.setDefault(FRAMEWORK_PATH, FRAMEWORK_PATH_DEFAULT);
		store.setDefault(GRID_SELECTION, GRID_SELECTION_DEFAULT);
		store.setDefault(SNAPTOGRID_SELECTION, SNAPTOGRID_SELECTION_DEFAULT);
		store.setDefault(SNAPTOGEOMETRY_SELECTION, SNAPTOGEOMETRY_SELECTION_DEFAULT);
		store.setDefault(GRID_SIZE, GRID_SIZE_DEFAULT);
		
		// TODO: 実際のContributorをセットしてください。
		contributor = new NullEventEditorContributor();
	}

	public IPersistentPreferenceStore getPreferenceStore() {
		return store;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.core.resources.IProjectNature#configure()
	 */
	public void configure() throws CoreException {
		// NOP
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.core.resources.IProjectNature#deconfigure()
	 */
	public void deconfigure() throws CoreException {
		// NOP
	}

	public IPath getMaskatFrameworkPath() {
		return new Path(store.getString(FRAMEWORK_PATH));
	}

	public void setMaskatFrameworkPath(IPath path) {
		store.setValue(FRAMEWORK_PATH, path.toString());
	}
	
	public boolean getGridSelection() {
		return new Boolean(store.getString(GRID_SELECTION)).booleanValue();
	}
	
	public void setGridSelection(boolean selection) {
		store.setValue(GRID_SELECTION, String.valueOf(selection));
	}
	
	public boolean getSnapToGridSelection() {
		return new Boolean(store.getString(SNAPTOGRID_SELECTION)).booleanValue();
	}
	
	public void setSnapToGridSelection(boolean selection) {
		store.setValue(SNAPTOGRID_SELECTION, String.valueOf(selection));
	}
	
	public boolean getSnapToGeometrySelection() {
		return new Boolean(store.getString(SNAPTOGEOMETRY_SELECTION)).booleanValue();
	}
	
	public void setSnapToGeometrySelection(boolean selection) {
		store.setValue(SNAPTOGEOMETRY_SELECTION, String.valueOf(selection));
	}
	
	public int getGridSize() {
		String v = store.getString(GRID_SIZE);
		if (v == null || v.length() < 1) {
			v = GRID_SIZE_DEFAULT;
		}
		return new Integer(v).intValue();
	}
	
	public void setGridSize(int size) {
		store.setValue(GRID_SIZE, String.valueOf(size));
	}

	public void setEventEditorContributor(EventEditorContributor contributor) {
		this.contributor = contributor;
	}

	public void openEventHandlerImpl(Event event) throws CoreException {
		if (this.contributor != null) {
			this.contributor.openEventHandlerImpl(event);
		}
	}
}
