/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.maru.common.Key;
import org.maru.common.MapFactory;
import org.maru.dog.core.Definition;

final class BindingDefinitionCache {

    private BindingDefinitionCache(){}

    /** The definitions of the target object */
    private static final ConcurrentMap<Key<?>, Definition<?>> CACHED_TARGET_DEFINITIONS = MapFactory.createConcurrentSoftMap();

    /** The definitions of the input object */
    private static final ConcurrentMap<Key<?>, Definition<?>> CACHED_INPUT_DEFINITIONS  = MapFactory.createConcurrentSoftMap();

    /** The definitions of the target object by field name */
    private static final ConcurrentMap<Key<?>, Definition<?>> CACHED_FIELD_NAME_TARGET_DEFINITIONS = MapFactory.createConcurrentSoftMap();

    /** The definitions of the input object by field name */
    private static final ConcurrentMap<Key<?>, Definition<?>> CACHED_FIELD_NAME_INPUT_DEFINITIONS = MapFactory.createConcurrentSoftMap();

    /**
     *  Gets the input {@link Definition} for annotated field bindings.
     * @param key key
     * @return {@link Definition}
     */
    public static Definition<?> getInputDefinition(Key<?> key) {
        return CACHED_INPUT_DEFINITIONS.get(key);
    }

    /**
     * Gets the target {@link Definition} for annotated field bindings.
     * @param key key
     * @return {@link Definition}
     */
    public static Definition<?> getTargetDefinition(Key<?> key) {
        return CACHED_TARGET_DEFINITIONS.get(key);
    }

    /**
     * Gets the input {@link Definition} for the field name bindings.
     * @param key key
     * @return {@link Definition}
     */
    public static Definition<?> getFieldNameInputDefinition(Key<?> key) {
        return CACHED_FIELD_NAME_INPUT_DEFINITIONS.get(key);
    }

    /**
     * Gets the target{@link Definition} for the field name bindings.
     * @param key key
     * @return {@link Definition}
     */
    public static Definition<?> getFieldNameTargetDefinition(Key<?> key) {
        return CACHED_FIELD_NAME_TARGET_DEFINITIONS.get(key);
    }

    /**
     * Checks if the key contains in CACHED_TARGET_DEFINITIONS.
     * @param key key
     * @return true if key contains, otherwise false.
     */
    public static boolean containsTargetDefinition(Key<?> key) {
        return CACHED_TARGET_DEFINITIONS.containsKey(key);
    }

    /**
     * Checks if the key contains in CACHED_INPUT_DEFINITIONS.
     * @param key key
     * @return true if key contains, otherwise false.
     */
    public static boolean containsInputDefinition(Key<?> key) {
        return CACHED_INPUT_DEFINITIONS.containsKey(key);
    }

    /**
     * Checks if the key contains in CACHED_FIELD_NAME_TARGET_DEFINITIONS.
     * @param key key
     * @return true if key contains, otherwise false.
     */
    public static boolean containsFieldNameTargetDefinition(Key<?> key) {
        return CACHED_FIELD_NAME_TARGET_DEFINITIONS.containsKey(key);
    }

    /**
     * Checks if the key contains in CACHED_FIELD_NAME_INPUT_DEFINITIONS.
     * @param key key
     * @return true if key contains, otherwise false.
     */
    public static boolean containsFieldNameInputDefiniton(Key<?> key) {
        return CACHED_FIELD_NAME_INPUT_DEFINITIONS.containsKey(key);
    }

    /**
     * Applies {@link Key} and target {@link Definition} into CACHED_TARGET_DEFINITIONS
     * if it is absent.
     * @param key key
     * @param definition definition
     * @return applied definition.
     */
    public static Definition<?> applyTargetDefinition(Key<?> key, Definition<?> definition) {
        return CACHED_TARGET_DEFINITIONS.putIfAbsent(key, definition);
    }

    /**
     * Applies {@link Key} and input {@link Definition} into CACHED_INPUT_DEFINITIONS
     * if it is absent.
     * @param key key
     * @param definition definition
     * @return applied definition.
     */
    public static Definition<?> applyInputDefinition(Key<?> key, Definition<?> definition) {
        return CACHED_INPUT_DEFINITIONS.putIfAbsent(key, definition);
    }

    /**
     * Applies {@link Key} and target field name {@link Definition} into CACHED_FIELD_NAME_TARGET_DEFINITIONS
     * if it is absent.
     * @param key key
     * @param definition definition
     * @return applied definition.
     */
    public static Definition<?> applyFieldNameTargetDefinition(Key<?> key, Definition<?> definition) {
        return CACHED_FIELD_NAME_TARGET_DEFINITIONS.putIfAbsent(key, definition);
    }

    /**
     * Applies {@link Key} and input field name {@link Definition} into CACHED_FIELD_NAME_INPUT_DEFINITIONS
     * if it is absent.
     * @param key key
     * @param definition definition
     * @return applied definition.
     */
    public static Definition<?> applyFieldNameInputDefinition(Key<?> key, Definition<?> definition) {
        return CACHED_FIELD_NAME_INPUT_DEFINITIONS.putIfAbsent(key, definition);
    }

    /**
     * Clear all cached data.
     */
    public static void clearAll() {
        Lock lock = new ReentrantLock();
        lock.lock();
        try {
            if (!CACHED_TARGET_DEFINITIONS.isEmpty()) {
                CACHED_TARGET_DEFINITIONS.clear();
            }
            if (!CACHED_INPUT_DEFINITIONS.isEmpty()) {
                CACHED_INPUT_DEFINITIONS.clear();
            }
            if (!CACHED_FIELD_NAME_TARGET_DEFINITIONS.isEmpty()) {
                CACHED_FIELD_NAME_TARGET_DEFINITIONS.clear();
            }
            if (!CACHED_FIELD_NAME_INPUT_DEFINITIONS.isEmpty()) {
                CACHED_FIELD_NAME_INPUT_DEFINITIONS.clear();
            }
        } finally {
            lock.unlock();
        }
    }

    /**
     * Removes target definition from CACHED_TARGET_DEFINITIONS by key.
     * @param key key
     * @return removed definition.
     */
    public static Definition<?> removeTargetDefinition(Key<?> key) {
        return CACHED_TARGET_DEFINITIONS.remove(key);
    }

    /**
     * Removes input definition from CACHED_INPUT_DEFINITIONS by key.
     * @param key key
     * @return removed definition.
     */
    public static Definition<?> removeInputDefinition(Key<?> key) {
        return CACHED_INPUT_DEFINITIONS.remove(key);
    }

    /**
     * Removes target field name definition from CACHED_FIELD_NAME_TARGET_DEFINITIONS by key.
     * @param key key
     * @return removed definition.
     */
    public static Definition<?> removeFieldNameTargetDefinition(Key<?> key) {
        return CACHED_FIELD_NAME_TARGET_DEFINITIONS.remove(key);
    }

    /**
     * Removes input field name definition from CACHED_FIELD_NAME_INPUT_DEFINITIONS by key.
     * @param key key
     * @return removed definition.
     */
    public static Definition<?> removeFieldNameInputDefinition(Key<?> key) {
        return CACHED_FIELD_NAME_INPUT_DEFINITIONS.remove(key);
    }
}
