<?php
/**
 * HttpSession.
 * @package magic.core
 * @subpackage helper.http
 */
/**
 * 永続的なセッションとクッキーを管理するクラスです.
 * <p>
 * <var>$_SESSION</var>や<var>$_COOKIE</var>を操作するための機能を提供します。<br/>
 * 一時的なセッションを利用するには、{@link HttpFlash}クラスを使用してください。
 * </p>
 * @package magic.core
 * @subpackage helper.http
 * @author T.Okumura
 * @version 1.0.0
 * @see HttpFlash
 * @final
 */
final class HttpSession {
    /**
     * セッションが開始されているかどうかを保持します.
     * @var bool
     */
    private $_isStarted = FALSE;
    /**
     * コンストラクタ.
     * <p>
     * クッキーが利用可能かどうかを判定するために、デフォルトのクッキーを発行します。
     * </p>
     * @param string $sessionName セッション名
     */
    public function __construct($sessionName) {
        $this->setName($sessionName);
        $this->setCookieValue('MAGIC', 'MAGIC', time() + 60 * 60);
    }
    /**
     * セッションが開始されているかどうかを取得します.
     *
     * @return bool 開始されている場合はTRUE
     */
    public function isStarted() {
        return $this->_isStarted;
    }
    /**
     * セッションを開始します.
     * <p>
     * 既に開始されている場合は何もしません。
     * </p>
     * @return bool 既に開始されているか、正常に開始した場合はTRUE
     */
    public function start() {
        if (!$this->_isStarted) {
            $this->_isStarted = @session_start();
        }
        return $this->isStarted();
    }
    /**
     * セッションから値を取得します.
     *
     * @param string $key 取得する値のキー
     * @return mixed 取得できた場合はその値、できなければFALSE
     */
    public function get($key) {
        return array_key_exists($key, $_SESSION) ? $_SESSION[$key] : FALSE;
    }
    /**
     * すべてのセッションを取得します.
     * <p>
     * 返されるのはセッションのコピーです。<br/>
     * これによって返された配列に変更を加えても、元の<var>$_SESSION</var>の内容は変更されない事に注意してください。
     * </p>
     * @return array コピーされたセッションの配列
     */
    public function getAll() {
        $copy = $_SESSION;
        return $copy;
    }
    /**
     * クッキーから値を取得します.
     * <p>
     * クッキーの値がカンマで区切られていれば、配列に分割されます。
     * </p>
     * @param string $key 取得する値のキー
     * @param int $index [optional] 取得する値が複数の場合、0から始まるインデックス番号(オプション)
     * @return mixed 値がなければFALSE、インデックスが指定されていればその値、それ以外は値の配列
     */
    public function getCookieValue($key, $index = NULL) {
        if (!isset($_COOKIE[$key])) {
            return FALSE;
        }
        $cookieValues = explode(',', $_COOKIE[$key]);
        return is_null($index) ? $cookieValues : $cookieValues[$index];
    }
    /**
     * カレントのセッション名を取得します.
     * @return string セッション名
     */
    public function getName() {
        return session_name();
    }
    /**
     * カレントのセッションIDを取得します.
     * @return string セッションID
     */
    public function getId() {
        return session_id();
    }
    /**
     * セッションに値を設定します.
     * <p>
     * 同じキーがあれば、値が上書きされます。
     * </p>
     * @param string $key 設定する値のキー
     * @param mixed $value 設定する値
     */
    public function set($key, $value) {
        $_SESSION[$key] = $value;
    }
    /**
     * 渡された配列をセッションにマージして設定します.
     * <p>
     * 同じキーがあれば、値が上書きされます。
     * </p>
     * @param array $list セッションに設定する値の配列
     */
    public function setAll(array $list) {
        $_SESSION = array_merge($_SESSION, $list);
    }
    /**
     * クッキーに値を設定します.
     * <p>
     * 同じクッキーがあれば、置き換えられます。
     * </p>
     * @param string $key 設定する値のキー
     * @param mixed $value 設定する値、または設定する値の配列
     * @param int $expire クッキーの存続時間(秒)
     */
    public function setCookieValue($key, $value, $expire) {
        $value = is_array($value) ? implode(',', $value) : $value;
        setcookie($key, $value, $expire, '/');
    }
    /**
     * セッションIDを設定します.
     * @param string セッションID
     */
    public function setId($sessionId) {
        session_id($sessionId);
    }
    /**
     * セッション名を設定します.
     * @param string セッション名
     */
    public function setName($sessionName) {
        session_name($sessionName);
    }
    /**
     * セッションから値を削除します.
     * @param string $key 削除する値のキー
     */
    public function remove($key) {
        if (array_key_exists($key, $_SESSION)) {
            unset($_SESSION[$key]);
        }
    }
    /**
     * セッションを置換します.
     * <p>
     * セッションの内容がすべて置き換わる事に注意してください。
     * </p>
     * @param array $list 置換する配列
     */
    public function replace(array $list) {
        $_SESSION = $list;
    }
    /**
     * セッションを終了します.
     * <p>
     * このファンクションが呼び出された後は、再度セッションをスタートさせるまで、
     * セッションは利用できません。
     * </p>
     */
    public function clear() {
        if (isset($_COOKIE[session_name()])) {
            $this->setCookieValue(session_name(), '', time() - 3600);
        }
        $this->setCookieValue('MAGIC', '', time() - 3600);
        session_destroy();
        $_SESSION = array();
    }
}
// EOF.