<?php
/**
 * HttpFlash.
 * @package magic.core
 * @subpackage helper.http
 */
/**
 * 一時的なセッションを管理するクラスです.
 * <p>
 * このクラスに保持されている情報は、次のActionクラスの呼び出しまで、セッションに保持されます。<br/>
 * (次のActionクラスがコールされた時点で、セッションからは破棄されますが、
 * コールされたActionクラスの中では使用する事ができます)<br/>
 * これはリダイレクトを使用したActionのチェインで、情報を引き継ぐために利用できます。<br/>
 * またGETリクエストのクエリパラメータに情報を設定せずに済むため、情報を隠蔽する事ができます。<br/>
 * このクラスに保持されている情報は、一度取得すると、このクラスからも削除されます。<br/>
 * 永続的なセッションを利用するには、{@link HttpSession}クラスを使用してください。<br/>
 * セッションを利用しない環境では、このクラスも利用できません。
 * </p>
 * @package magic.core
 * @subpackage helper.http
 * @author T.Okumura
 * @version 1.0.0
 * @see HttpSession
 * @final
 */
final class HttpFlash {
    /**
     * 一時的な情報を保持します.
     * @var array
     */
    private $_flash = array();
    /**
     * コンストラクタ.
     */
    public function __construct() {
    }
    /**
     * 保持されている値を取得します.
     * <p>
     * 一度取得した値は、このクラスからも削除される事に注意してください。
     * </p>
     * @param string $key 取得する値のキー
     * @return mixed 存在すればその値、存在しない場合はFALSE
     */
    public function get($key) {
        if (array_key_exists($key, $this->_flash)) {
            $result = $this->_flash[$key];
            $this->remove($key);
            return $result;
        }
        return FALSE;
    }
    /**
     * 保持されている情報の配列を取得します.
     * <p>
     * 一度取得すると、このクラスの情報はすべて削除される事に注意してください。
     * </p>
     * @return array このクラスが保持している情報の配列
     */
    public function getAll() {
        $result = $this->_flash;
        $this->_flash = array();
        return $result;
    }
    /**
     * 値を設定します.
     * <p>
     * 同じキーがあれば、値が上書きされます。
     * </p>
     * @param string $key 設定する値のキー
     * @param mixed $value 設定する値
     */
    public function set($key, $value) {
        $this->_flash[$key] = $value;
    }
    /**
     * 渡された配列をマージして設定します.
     * <p>
     * 同じキーがあれば、値が上書きされます。
     * </p>
     * @param array $list 設定する値の配列
     */
    public function setAll(array $list) {
        $this->_flash = array_merge($this->_flash, $list);
    }
    /**
     * 値を削除します.
     * @param string $key 削除する値のキー
     */
    public function remove($key) {
        if (array_key_exists($key, $this->_flash)) {
            unset($this->_flash[$key]);
        }
    }
    /**
     * このクラスで保持している情報を置換します.
     * <p>
     * 保持している情報がすべて置き換わる事に注意してください。
     * </p>
     * @param array $list 置換する配列
     */
    public function replace(array $list) {
        $this->_flash = $list;
    }
    /**
     * このクラスで保持している情報をすべて削除します.
     */
    public function clear() {
        $this->_flash = array();
    }
}
// EOF.