package map.model;

import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.io.Serializable;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import labeling.Labels;

/**
 * 市区町村クラス
 * 数値地図、国土数値情報などのデータを管理する
 * @author ma38su
 */
public class City implements Serializable {

	/**
	 * 市町村番号フォーマット
	 */
	private final static NumberFormat CODE_FORMAT = new DecimalFormat("00000");
	
	/**
	 * 都道府県番号フォーマット
	 */
	private final static NumberFormat PREFECTURE_FORMAT = new DecimalFormat("00");

	/**
	 * 市区町村番号(int)を(String)に変換
	 * @param code 市区町村番号
	 * @return 市区町村番号
	 */
	public static String cityFormat(int code) {
		return City.CODE_FORMAT.format(code);
	}

	/**
	 * 都道府県番号(int)を(String)に変換
	 * @param code 都道府県番号
	 * @return 都道府県番号
	 */
	public static String prefectureFormat(int code) {
		return City.PREFECTURE_FORMAT.format(code);
	}

	/**
	 * 外接長方形
	 */
	private transient Rectangle area;
	
	/**
	 * 行政界
	 */
	private transient Curve[] border;


	/**
	 * 数値地図25000の水域界
	 * {湖岸, 海岸線}
	 * 海岸線は描画する必要がある。
	 */
	private transient Curve[][] coast;
	/**
	 * 地域番号
	 */
	private final int code;

	/**
	 * 施設配列
	 */
	private transient Map<String, Facility[]> facility;
	/**
	 * 店舗情報フラグ
	 */
	private transient boolean hasShop;
	
	/**
	 * 詳細データを保持しているかどうか
	 */
	private transient boolean isLoaded;

	/**
	 * 国土数値情報の市町村界
	 */
	private Polygon[] ksj;

	/**
	 * 標高メッシュ配列
	 */
	private transient Mesh[][] mesh;
	
	/**
	 * 数値地図2500の水域
	 */
	private transient Polygon[] mizu;
	
	/**
	 * 水域アーク（水域の輪郭）
	 */
	private transient Curve[] mizuArc;
	
	/**
	 * 市区町村名
	 */
	private String name;
	
	/**
	 * 頂点配列
	 */
	private transient Map<Integer, Node> node;

	/**
	 * 鉄道
	 */
	private transient Curve[][] rail;

	/**
	 * 河川
	 */
	private transient Curve[] river;
	
	/**
	 * 道路
	 */
	private transient Road[][] road;
	
	/**
	 * 店舗情報
	 */
	private transient Map<String, Labels> shop;

	/**
	 * 駅
	 */
	private transient Station[] station;

	private int x;
	
	private int y;
	
	/**
	 * 市区町村データのコンストラクタ
	 * @param code 市区町村番号
	 * @param polygon 行政界
	 */
	public City(int code, Polygon[] polygon) {
		this.code = code;
		this.ksj = polygon;
		this.isLoaded = false;
		this.shop = new HashMap<String, Labels>();

		long x = 0;
		long y = 0;
		int size = 0;
		for (Polygon p : polygon) {
			size += p.npoints;
			for (int i = 0; i < p.npoints; i++) {
				x += p.xpoints[i];
				y += p.ypoints[i];
			}
		}
		this.x = (int) (x / size);
		this.y = (int) (y / size);
	}

	/**
	 * 市区町村データのコンストラクタ
	 * @param code 市町村番号
	 * @param area バウンディングボックス
	 * @param point 行政代表点
	 */
	public City(int code, Rectangle area) {
		this.code = code;
		this.area = area;
		this.isLoaded = false;
		this.shop = new HashMap<String, Labels>();
		this.x = area.x + area.width / 2;
		this.y = area.y + area.height / 2;
	}
	

	/**
	 * チェーン店情報を追加します。
	 * @param station チェーン店名
	 * @param location 店舗の所在地（経緯度）
	 */
	public void addLabels(String station, Collection<Point> location) {
		Labels labels = this.shop.get(station);
		if (labels == null) {
			labels = new Labels(station);
			this.shop.put(station, labels);
		}
		labels.add(location);
	}

	/**
	 *　データの破棄
	 * @param rect 表示領域
	 */
	public void dump() {
		this.mizu = null;
		this.mizuArc = null;
		this.rail = null;
		this.station = null;
		this.node = null;
		this.road = null;
		this.river = null;
		this.border = null;
		this.coast = null;
		this.facility = null;
		this.mesh = null;
		this.isLoaded = false;
		this.shop.clear();
		this.hasShop = false;
	}
	
	/**
	 * 広域のデータの破棄
	 */
	public void dumpKsj() {
		this.ksj = null;
		this.name = null;
	}

	/**
	 * エリアを返す
	 * @return エリア
	 */
	public Rectangle getArea() {
		return this.area;
	}

	
	/**
	 * 行政界を返します。
	 * @return 行政界
	 */
	public Curve[] getBorder() {
		return this.border;
	}
	
	/**
	 * 数値地図2500の水域界を取得します。
	 * @return 数値地図2500の水域界
	 */
	public Curve[] getCoast() {
		return this.coast[0];
	}

	/**
	 * 市町村番号の取得
	 * @return 市町村番号
	 */
	public int getCode() {
		return this.code;
	}

	/**
	 * JRの鉄道区間の取得
	 * @return 鉄道区間
	 */
	public Curve[] getJRRailway() {
		return this.rail[0];
	}

	public Facility[] getLabel(String key) {
		return this.facility.get(key);
	}

	/**
	 * @return 複数ラベル
	 */
	public Collection<Labels> getLabels() {
		return this.shop.values();
	}

	public Mesh[][] getMesh() {
		return this.mesh;
	}

	public Polygon[] getMizu() {
		return this.mizu;
	}
	
	public Curve[] getMizuArc() {
		return this.mizuArc;
	}
	/**
	 * 市区町村名
	 * @return 市区町村名
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * 道路頂点の取得
	 * @return 道路頂点
	 */
	public Collection<Node> getNode() {
		return this.node.values();
	}
	
	/**
	 * 道路頂点の取得
	 * @param id
	 * @return 道路頂点
	 */
	public Node getNode(Integer id) {
		return this.node.get(id);
	}

	/**
	 * 原点のX座標を取得
	 * @return 原点のX座標
	 */
	public int getOriginX() {
		return this.area.x;
	}

	/**
	 * 原点のY座標を取得
	 * @return 原点のY座標
	 */
	public int getOriginY() {
		return this.area.y;
	}

	/**
	 * JR以外の鉄道区間の取得
	 * @return 鉄道区間
	 */
	public Curve[] getOtherRailway() {
		return this.rail[1];
	}

	/**
	 * 行政界ポリゴンの取得
	 * @return 行政界ポリゴン(国土数値情報)
	 */
	public Polygon[] getPolygon() {
		return this.ksj;
	}
	
	/**
	 * 都道府県番号の取得
	 * @return 都道府県番号
	 */
	public String getPrefectureCode() {
		return City.PREFECTURE_FORMAT.format(this.code / 1000);
	}
	
	/**
	 * 河川の取得
	 * @return 河川
	 */
	public Curve[] getRiver() {
		return this.river;
	}
	
	/**
	 * 道路の取得
	 * @return 道路
	 */
	public Road[][] getRoad() {
		return this.road;
	}
	
	/**
	 * 数値地図25000の海岸線を取得します。
	 * @return 数値地図25000の海岸線
	 */
	public Curve[] getSeaBorder() {
		return this.coast[1];
	}
	
	/**
	 * 駅区間の取得
	 * @return 駅区間
	 */
	public Station[] getStation() {
		return this.station;
	}
	
	public int getX() {
		return this.x;
	}
	
	public int getY() {
		return this.y;
	}
	
	/**
	 * データを読み込んでいるかどうか
	 * @return データを読み込んでいればtrue
	 */
	public boolean hasData() {
		return this.isLoaded;
	}
	
	public void init() {
		this.isLoaded = false;
		this.shop = new HashMap<String, Labels>();
	}
	
	public boolean hasNode() {
		return this.node != null;
	}

	public boolean hasSdk25Bounds() {
		return this.area != null;
	}

	/**
	 * 店舗情報を読み込んでいるか確認します。
	 * @return 店舗情報を読み込んでいればtrue
	 */
	public boolean hasStore() {
		return this.hasShop;
	}
	
	public boolean isLoaded() {
		return this.area != null;
	}
		
	/**
	 * 数値地図2500の読み込みを確認します。
	 * @return 読み込んでいればtrue
	 */
	public boolean isSdk2500() {
		return this.mizu != null;
	}

	/**
	 * データの読み込みが行われていなければ指定した番号の頂点を削除する
	 * @param id 頂点番号
	 */
	public void removeNode (Integer id) {
		this.node.remove(id);
	}
	
	public void setArea(Rectangle area) {
		this.area = area;
	}
	
	public void setData(Curve[] border, Curve[][] coast, Curve[] river, Curve[][] rail, Station[] station, Map<Integer, Node> node, Road[][] road, Map<String, Facility[]> facility, Mesh[][] mesh) {
		this.rail = rail;
		this.station = station;
		this.node = node;
		this.road = road;
		this.river = river;
		this.border = border;
		this.coast = coast;
		this.facility = facility;
		this.mesh = mesh;
		this.isLoaded = true;
	}
	
	
	/**
	 * 頂点データを追加
	 * @param node 頂点データを追加する。
	 */
	public void setData(Map<Integer, Node> node) {
		this.node = node;
	}
	
	/**
	 * 名前を設定します。
	 * @param name
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * 数値地図2500を追加します。
	 * @param polygon
	 * @param arc
	 */
	public void setSdk2500(Polygon[] polygon, Curve[] arc) {
		this.mizuArc = arc;
		this.mizu = polygon;
	}

	public void setStore() {
		this.hasShop = true;
	}
	@Override
	public String toString() {
		return this.getName() + "(" + City.cityFormat(this.getCode()) + ")";
	}
}
