package map;

import index.CellMethod;

import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.sourceforge.ma38su.util.Log;

import map.model.City;
import map.model.Node;
import map.store.Store;
import view.DialogFactory;
import view.MapPanel;
import view.StatusBar;
import database.FileDatabase;
import database.CodeDatabase;

/**
 * 市区町村管理クラス
 * @author ma38su
 */
public class CityMap implements Map<Integer, City>, Runnable {

	/**
	 * 地図パネルクラス
	 */
	private final MapPanel panel;

	/**
	 * セル型の地域検索クラス
	 */
	private final CellMethod cell;
	
	/**
	 * 数値地図25000を読み込むためのクラス
	 */
	private final Sdf25kReader sdf25k;

	/**
	 * 数値地図2500を読み込むためのクラス
	 */
	private final Sdf2500Reader sdf2500;

	/**
	 * ステータスバー
	 */
	private final StatusBar statusbar;

	/**
	 * 都道府県番号の読み込み状態
	 * true  -> 既読
	 * false -> 未読
	 */
	private boolean[] prefecture;

	private Rectangle screen;

	/**
	 * 市区町村データのMap
	 */
	private final Map<Integer, City> cityMap;

	/**
	 * 街区レベル位置参照情報ファクトリー
	 */
	private final IsjReader isjFactory;

	/**
	 * 店舗情報ファクトリー
	 */
	private final Collection<Store> shops;

	public CityMap(MapPanel panel, CellMethod cell, FileDatabase storage, CodeDatabase db, IsjReader isjFactory, Collection<Store> shop, StatusBar statusbar) {
		this.db = db;
		this.cityMap = new HashMap<Integer, City>();
		this.panel = panel;
		this.cell = cell;
		this.shops = shop;
		this.sdf25k = new Sdf25kReader(storage);
		this.sdf2500 = new Sdf2500Reader(storage);
		this.isjFactory = isjFactory;
		this.statusbar = statusbar;
		this.prefecture = new boolean[47];
	}
	
	/**
	 * スレッドを開始します。
	 */
	public void start() {
		this.screen = this.panel.getScreen();
		Log.out(this, "start Thread.");
		Thread thread = new Thread(this);
		thread.setPriority(2);
		thread.start();
	}

	/**
	 * 地図情報を加える
	 * @param key 市区町村番号
	 * @param value 市区町村データ
	 * 
	 * @return 市区町村データ
	 */
	public synchronized City put(Integer key, City value) {
		return this.cityMap.put(key, value);
	}

	/**
	 * 指定した頂点番号の頂点を取得します。
	 * @param id 頂点番号
	 * @param isDetailRoad 詳細な道路区間データを取得する場合true
	 * @return 頂点
	 * @throws IOException 頂点を取得できなかった場合
	 */
	public Node getNode(long id, boolean isDetailRoad) throws IOException {
		City map = this.cityMap.get((int) (id / 1000000));
		if (map == null) {
			map = this.sdf25k.productNode((int) (id / 1000000), isDetailRoad);
		} else if (!map.hasNode()) {
			this.sdf25k.productNode(map, isDetailRoad);
		}
		Node node = map.getNode((int) (id % 1000000));
		if (node == null) {
			Log.out(this, " : node is not found, and reproduct node.");
			this.sdf25k.productNode(map, isDetailRoad);
			node = map.getNode((int) (id % 1000000));
			if (node == null) {
				throw new IOException("node is not found.");
			}
		}
		return node;
	}

	/**
	 * 市区町村データの取得
	 * @param key 市区町村番号
	 * @return 市区町村データ
	 */
	public synchronized City get(Object key) {
		City city = this.cityMap.get(key);
		return city;
	}

	/**
	 * とりあえず行政代表点を求めます。
	 * @param code 市区町村番号
	 * @return 市区町村番号に対応した座標
	 */
	public Point getPoint(int code) {
		Point point = null;
		try {
			point = this.sdf25k.getPoint(code);
		} catch (Exception e) {
			point = null;
			Log.err(this, e);
			Log.err(this, "code: "+ Integer.toString(code));
		}
		return point;
	}
	/**
	 * 頂点のCollectionを取得
	 * @param code 市区町村番号
	 * @return 頂点のCollection
	 */
	public synchronized Collection<Node> getNodes(int code) {
		return this.cityMap.get(code).getNode();
	}

	public void run() {
		while (true) {
			this.screen = new Rectangle(this.panel.getScreen());
			Log.out(this, "running...");
			try {
				boolean isReadNewData = false;
				try {
					if (this.panel.mode() > 1) {
						Map<Integer, Set<Integer>> codes = this.cell.search(this.screen);
						Log.out(this, "codes = " + codes);
						if (!this.panel.isOperation()) {
							this.dumpPrefecture(codes.keySet());
						}
						for (Map.Entry<Integer, Set<Integer>> entry : codes.entrySet()) {
							if (this.screen.equals(this.panel.getScreen())) {
								Log.out(this, "read KSJ border:"+ City.cityFormat(entry.getKey()));
								this.productPrefecture(entry.getKey());
								if (this.panel.mode() == 3) {
									for (Integer code : entry.getValue()) {
										if (this.screen.equals(this.panel.getScreen())) {
											isReadNewData |= this.readCity(code);
										}
									}
								}
							}
						}
					}
				} catch (IOException e) {
					DialogFactory.errorDialog(this.panel, e);
					isReadNewData = false;
					Log.err(this, e);
				} catch (OutOfMemoryError e) {
					DialogFactory.memoryDialog(this.panel, e);
					isReadNewData = false;
					Log.err(this, e);
				} finally {
					this.dump();
				}
				if (!isReadNewData) {
					Log.out(this, "sleep");
					Thread.sleep(2500L);
				}
			} catch (InterruptedException e) {
				Log.err(this, e);
			}
		}
	}
	
	/**
	 * 国土数値情報の行政界の読み込み市区町村データを生成
	 * @param code 市区町村番号
	 * @throws IOException 入出力エラー
	 */
	private void productPrefecture(int code) throws IOException {
		if (!this.prefecture[code - 1]) {
			String format = City.prefectureFormat(code);
			Map<Integer, List<Polygon>> polygonMap = KsjReader.readSerializeKsj(format);
			for (Map.Entry<Integer, List<Polygon>> entry : polygonMap.entrySet()) {
				synchronized (this) {
					Integer key = entry.getKey();
					this.put(key, new City(key, entry.getValue().toArray(new Polygon[] {})));
				}
			}
			this.prefecture[code - 1] = true;
			this.panel.repaint();
		}
	}
	/**
	 * 市町村データの読み込み
	 * @param code 読み込む市区町村番号
	 * @return データを読み込めばtrue
	 * @throws IOException 入出力エラー
	 */
	private boolean readCity(int code) throws IOException {
		Log.out(this, "read City started.");
		// セルメソッドから地図の読み込み
		Log.out(this, "read City: " + code);
		City city = this.get(code);
		Log.out(this, "city: " + city);
		boolean ret = false;
		String content = this.db.get(code);
		if (content == null) {
			content = City.cityFormat(code);
		}
		if (city == null) {
			/* 数値地図25000を読み込む */
			this.statusbar.startReading(content, StatusBar.SDF25000);
			city = this.sdf25k.product(code);
			this.put(code, city);
			Log.out(this, "put " + code + ", " + city);
			this.panel.repaint();
			ret = true;
		} else if (!city.hasData()) {
			/* 数値地図25000を読み込む */
			Log.out(this, "read SDF25000: "+ code);
			this.statusbar.startReading(content, StatusBar.SDF25000);
			this.sdf25k.product(city);
			this.panel.repaint();
			ret = true;
		}
		if (!city.hasStore()) {
			/* 店舗情報を読み込む */
			Log.out(this, "read ISJ: "+ code);
			this.statusbar.startReading(content, StatusBar.ISJ);
			Map<String, Point> locationMap = this.isjFactory.productStreaming(code);
			// TODO 富士山頂付近でnullが返ることがある。
			if (locationMap != null) {
				for (Store shop : this.shops) {
					try {
						List<Point> location = shop.getLocation(city, locationMap);
						city.addLabels(shop.getName(), location);
					} catch (Throwable e) {
						DialogFactory.errorDialog(this.panel, e);
					}
				}
			}
			city.setStore();
			this.panel.repaint();
			ret = true;
		}
		if (!city.isSdk2500() && this.db.hasSdf2500(code)) {
			/* 数値地図2500を読み込む */
			Log.out(this, "read SDF2500: "+ code);
			this.statusbar.startReading(content, StatusBar.SDF2500);
			this.sdf2500.productMizu(city);
			this.panel.repaint();
			ret = true;
		}
		this.statusbar.finishReading();
		return ret;
	}
	
	private final CodeDatabase db;

	/**
	 * 経路探索のために読み込んだ頂点を破棄します。
	 */
	public synchronized void clearNode() {
		for (City map : this.cityMap.values()) {
			if ((!map.hasData() && map.hasNode()) || (map.hasData() && !this.screen.intersects(map.getArea()))) {
				map.dump();
			}
		}
	}

	public void searchedRecover() {
		this.panel.repaint();
		this.clearNode();
	}

	public void setSearchThread(Thread thread) {
		this.statusbar.setSearchThread(thread);
	}

	/**
	 * 不要なデータの解放
	 */
	public synchronized void dump() {
		for (City map : this.cityMap.values()) {
			if (map.hasData() && (this.panel.mode() < 3 || !this.screen.intersects(map.getArea()))) {
				map.dump();
			}
		}
	}

	/**
	 * 引数の都道府県番号に含まれていないデータの解放
	 * @param codes 都道府県番号
	 */
	public synchronized void dumpPrefecture(Set<Integer> codes) {
		for (City map : this.cityMap.values()) {
			int code = map.getCode() / 1000;
			if (!codes.contains(code)) {
				map.dumpKsj();
				this.prefecture[code - 1] = false;
			}
		}
	}

	public int size() {
		return this.cityMap.size();
	}

	public boolean isEmpty() {
		return this.cityMap.isEmpty();
	}

	public boolean containsKey(Object key) {
		return this.cityMap.containsKey(key);
	}

	public boolean containsValue(Object value) {
		return this.cityMap.containsValue(value);
	}

	public City remove(Object key) {
		return this.cityMap.remove(key);
	}

	public void putAll(Map<? extends Integer, ? extends City> t) {
		throw new IllegalAccessError("未実装");
	}

	public void clear() {
		this.cityMap.clear();
	}

	public Set<Integer> keySet() {
		return this.cityMap.keySet();
	}

	public Collection<City> values() {
		return this.cityMap.values();
	}

	public Set<Entry<Integer, City>> entrySet() {
		return this.cityMap.entrySet();
	}
}
