/*
 * Digital Map
 * 地図データの閲覧のためのプログラムです。
 * Copyright(C) 2005-2006 ma38su
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
import index.CellMethod;

import java.awt.BorderLayout;
import java.awt.Polygon;
import java.io.File;
import java.io.IOException;
import java.util.Collection;

import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;

import jp.sourceforge.ma38su.util.Log;

import map.CityMap;
import map.IsjReader;
import map.store.Store;
import util.Loader;
import util.Setting;
import util.Version;
import view.DialogFactory;
import view.MapMenu;
import view.MapPanel;
import view.SearchPanel;
import view.StatusBar;
import controller.MainController;
import database.CodeDatabase;
import database.FileDatabase;

/**
 * 地図描画アプリケーション起動のためのクラス
 * @author ma38su
 */
public class StartUp {

	public static final String SETTING = ".setting";
	
	/**
	 * フレーム高さ
	 */
	private static final int F_WIDTH = 800;

	/**
	 * フレーム幅
	 */
	private static final int F_HEIGHT = 600;

	/**
	 * args[0]には地図データのルートディレクトリの位置を指定します。
	 * args[1]にはプラグインのディレクトリの位置を指定します。
	 * @param libDir 
	 * @param pluginDir 
	 * @param mapDir 
	 */
	public static void startup(String libDir, String pluginDir, String mapDir) {
		Log.isDebug = false;

		String version = Version.get("/history.txt");
		String title = version == null ? "Digital Map" : "Digital Map ver." + version;
		JFrame frame = new JFrame(title);
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.setLayout(new BorderLayout());
		
		CodeDatabase codeDB;
		try {
			codeDB = new CodeDatabase("/.data/city.csv", mapDir + "city.idx", mapDir + "city.dat");
		} catch (IOException e) {
			codeDB = null;
			DialogFactory.errorDialog(null, e);
			return;
		}
		StatusBar statusbar = new StatusBar(" ");
		MapPanel panel = new MapPanel(codeDB);

		Setting setting = Setting.readSetting(mapDir);

		MainController controller = new MainController(panel, setting);
		
		JMenuBar menu = new MapMenu(controller);

		if (setting.isCheckTerms()) {
			DialogFactory.termsDialog(setting);
		}
		if (setting.isCheckUpdate()) {
			try {
				String latest = Version.getLatest("ma38su", "Digital Map");
				if (latest != null && version != null && !version.equals(latest)) {
					DialogFactory.versionDialog(latest, frame, controller);
				}
			} catch (IOException ex) {
				DialogFactory.errorDialog(panel, ex);
			}
		}

		JTabbedPane tab = new JTabbedPane();
		tab.addTab("市区町村検索", new SearchPanel(codeDB, controller));
		
		JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		splitPane.setRightComponent(panel);
		splitPane.setLeftComponent(tab);
		splitPane.setOneTouchExpandable(true);

		frame.add(statusbar, BorderLayout.SOUTH);
		frame.add(splitPane, BorderLayout.CENTER);
		frame.add(menu, BorderLayout.NORTH);

		
		frame.setSize(StartUp.F_WIDTH, StartUp.F_HEIGHT);
		frame.setLocationRelativeTo(null);
		frame.setVisible(true);

		try {
			statusbar.startReading("初期設定");

			Collection<Store> store = Loader.loadStorePlugin(pluginDir, mapDir + "store" + File.separatorChar);
			
			FileDatabase fileDB = new FileDatabase(mapDir);
			fileDB.addObserver(statusbar);
			final CityMap maps = new CityMap(panel, new CellMethod(mapDir + "index" + File.separatorChar), fileDB, codeDB, new IsjReader(mapDir + "isj" + File.separatorChar), store, statusbar);
			statusbar.finishReading();

			statusbar.startReading("OpenGIS Worlddata");
			Polygon[][] world = fileDB.getWorldPolygon();

			statusbar.startReading("国土数値情報 都道府県界");
			Polygon[][] prefectures = fileDB.getPrefecturePolygon();


			statusbar.finishReading();

			panel.init(maps, world, prefectures);

			maps.start();
		} catch (IOException e) {
			statusbar.startReading(e.getMessage());
		}

		statusbar.finishReading();
		panel.addMouseListener(controller);
		panel.addMouseMotionListener(controller);
		panel.addMouseWheelListener(controller);

		statusbar.setThreadPriority(Thread.MIN_PRIORITY);
	}
	
	/**
	 * インスタンス生成不要
	 */
	private StartUp() {
	}
}
