package database;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.RandomAccessFile;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.sourceforge.ma38su.util.Log;

/**
 * 都道府県番号、市区町村名番号データベース
 * 番号から名前、利用できるデータの種類などを検索するためのデータベースのクラス
 * @author ma38su
 */
public class CodeDatabase {

	public static final int MATCH_CONTAINS = 1;

	private final String file;
	
	private HashMap<SerializableInteger, SerializableInteger> map;

	private final SerializableInteger p;

	private RandomAccessFile random;
	
	/**
	 * データベース
	 * @param from
	 * @param db
	 * @param index
	 * @throws IOException
	 */
	public CodeDatabase(String from, String db, String index) throws IOException {
		this.file = from;
		this.p = new SerializableInteger();
		File disc = new File(db);
		BufferedReader bi = null;
		ObjectOutputStream out = null;
		if (disc.exists()) {
			this.map = this.readIndexMap(new File(index));
		} else {
			if (!disc.getParentFile().isDirectory()) {
				disc.getParentFile().mkdirs();
			}
		}
		if (this.map != null) {
			this.random = new RandomAccessFile(disc, "r");
		} else {
			this.map = new HashMap<SerializableInteger, SerializableInteger>();
			try {
				bi = new BufferedReader(new InputStreamReader(CodeDatabase.class.getResourceAsStream(from), "SJIS"));
				this.random = new RandomAccessFile(disc, "rw");
				this.random.seek(0);
				this.random.setLength(0);
				String line;
				Pattern csv = Pattern.compile(",");
				while ((line = bi.readLine()) != null) {
					String[] param = csv.split(line);
					SerializableInteger entry = new SerializableInteger(this.random.getFilePointer());
					this.map.put(new SerializableInteger(param[1]), entry);
					int flag = (Integer.parseInt(param[2]) << 1) | Integer.parseInt(param[3]);
					this.random.writeByte(flag);
					this.random.writeUTF(param[0]);
				}
				File file = new File(index);
				if (file.getParentFile().isDirectory()) {
					file.getParentFile().mkdirs();
				}
				out = new ObjectOutputStream(new FileOutputStream(file));
				out.writeObject(this.map);
				out.flush();
			} finally {
				if (bi != null) {
					bi.close();
				}
				if (out != null) {
					out.close();
				}
			}
		}
	}
	
	public boolean hasSdf25k(int code) {
		this.p.setValue(code);
		SerializableInteger entry = this.map.get(this.p);
		if (entry == null) {
			return false;
		}
		int name = 0;
		try {
			this.random.seek(entry.getValue() + 1);
			name = this.random.readByte();
		} catch (IOException e) {
			Log.err(this, e);
			name = 0;
		}
		return name != 0;
	}

	public boolean hasSdf2500(int code) {
		this.p.setValue(code);
		SerializableInteger entry = this.map.get(this.p);
		if (entry == null) {
			return false;
		}
		int flag = 0;
		try {
			this.random.seek(entry.getValue());
			flag = this.random.readByte();
		} catch (IOException e) {
			Log.err(this, e);
			flag = 0;
		}
		return (flag & 1) != 0;
	}
	
	/**
	 * @param code 市区町村番号
	 * @return 文字列
	 */
	public String get(int code) {
		this.p.setValue(code);
		SerializableInteger entry = this.map.get(this.p);
		if (entry == null) {
			return null;
		}
		String name = null;
		try {
			this.random.seek(entry.getValue() + 1);
			name = this.random.readUTF();
		} catch (IOException e) {
			e.printStackTrace();
			name = null;
		}
		return name;
	}
	
	/**
	 * 市区町村名から市区町村名を検索します。
	 * @param name 市区町村名
	 * @param type 検索するタイプ
	 * @return 市区町村番号
	 * @throws IOException 
	 */
	public Map<Integer, String> get(String name, int type) throws IOException {
		StringBuilder sb = new StringBuilder();
		switch (type) {
			case MATCH_CONTAINS :
				sb.append(".*");
				sb.append(name);
				sb.append(".*");
		}
		BufferedReader bi = null;
		Pattern pattern = Pattern.compile(sb.toString());
		Pattern csv = Pattern.compile(",");
		Map<Integer, String> map = new TreeMap<Integer, String>();
		Map<Integer, String> pref = new HashMap<Integer, String>();
		Set<Integer> set = new HashSet<Integer>();
		try {
			bi = new BufferedReader(new InputStreamReader(CodeDatabase.class.getResourceAsStream(this.file), "SJIS"));
			String line;
			while ((line = bi.readLine()) != null) {
				String[] param = csv.split(line);
				int code = Integer.parseInt(param[1]);
				int prefCode = code / 1000;
				Matcher matcher = pattern.matcher(param[0]);
				if (set.contains(prefCode) || matcher.matches()) {
					if (prefCode == 0) {
						set.add(code);
					} else {
						StringBuilder s = new StringBuilder();
						s.append(pref.get(prefCode));
						s.append(param[0]);
						map.put(code, s.toString());
					}
				}
				if (prefCode == 0) {
					pref.put(code, param[0]);
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return map;
	}

	@SuppressWarnings("unchecked")
	private HashMap<SerializableInteger, SerializableInteger> readIndexMap(File index) {
		ObjectInputStream in = null;
		HashMap<SerializableInteger, SerializableInteger> map = null;
		if (index.isFile()) {
			try {
				try {
					in = new ObjectInputStream(new FileInputStream(index));
					map = (HashMap<SerializableInteger, SerializableInteger>) in.readObject();
				} catch (Exception e) {
					this.map = null;
					index.delete();
				} finally {
					if (in != null) {
						in.close();
					}
				}
			} catch (Exception e) {
				map = null;
			}
		}
		return map;
	}

	@Override
	protected void finalize() throws Throwable {
		if (this.random != null) {
			this.random.close();
			this.random = null;
		}
		super.finalize();
	}
}
