package database;

import java.awt.Point;
import java.awt.Polygon;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import jp.sourceforge.ma38su.util.Log;

/**
 * 国土数値情報を読み込むクラス
 * @author ma38su
 */
public class KsjReader {

	/**
	 * 直列化して保存した行政界を読み込みます。
	 * @param name 入力ストリーム
	 * @return Polygon配列
	 * @throws IOException 入出力エラー
	 */
	public static Polygon[] readPolygon(String name) throws IOException {
		Polygon[] polygon = null;
		ObjectInputStream in = null;
		try {
			in = new ObjectInputStream(System.class.getResourceAsStream(name));
			polygon = (Polygon[]) in.readObject();
		} catch (ClassNotFoundException e) {
			polygon = null;
			Log.err(KsjReader.class, e);
		} finally {
			if (in != null) {
				in.close();
			}
		}
		return polygon;
	}

	/**
	 * 都道府県内の各市区町村に対応した国土数値情報を読み込みます。
	 * 読み込めるのは、直列化して保存したデータに限ります。
	 * 
	 * @param code 都道府県番号
	 * @return 市区町村に対応した国土数値情報の行政界
	 * @throws IOException 入出力エラー
	 * @throws ClassNotFoundException 
	 */
	@SuppressWarnings("unchecked")
	public static Map<Integer, List<Polygon>> readSerializeKsj(String code) throws IOException {
		Map<Integer, List<Polygon>> map = null;
		ObjectInputStream in = null;
		try {
			in = new ObjectInputStream(KsjReader.class.getResourceAsStream("/.data/" + code + "/ksj.dat"));
			map = (LinkedHashMap<Integer, List<Polygon>>) in.readObject();
		} catch (ClassNotFoundException e) {
			throw new IllegalArgumentException(e);
		} finally {
			if (in != null) {
				in.close();
			}
		}
		return map;
	}

	
	/**
	 * 地図データファイルを管理するクラス
	 */
	private FileDatabase db;
	
	public KsjReader(FileDatabase db) {
		this.db = db;
	}
	
	/**
	 * 文字列を切り出してintに変換します。
	 * @param str 変換する文字列
	 * @param s 切り出す始点
	 * @param t 切り出す終点
	 * @return 変換後のint
	 */
	private int parseInt(String str, int s, int t) {
		return Integer.parseInt(str.substring(s, t).trim());
	}

	/**
	 * 国土数値情報の行政界を読み込みます。
	 * 行政界は都道府県単位で読み込みます。
	 * ファイルが存在しない場合には、国土数値情報から取得します。
	 * 
	 * @param code 都道府県番号
	 * @return 市区町村番号対応した行政界を持つMap
	 */
	public Map<Integer, List<Polygon>> readKsjBorder(int code) {
		Map<Integer, List<Polygon>> ksjMap = null;
		try {
			File file = this.db.getKsjBoder(code);
			ksjMap = this.readKsjFace(file);
		} catch (IOException e) {
			ksjMap = null;
		}
		return ksjMap;
	}
	
	/**
	 * 国土数値情報の面ポリゴンを読み込みます
	 * @param file 国土数値情報のファイル
	 * @return 市区町村番号に対応した面ポリゴン
	 * @throws IOException 入出力エラー
	 */
	public Map<Integer, List<Polygon>> readKsjFace(File file) throws IOException {
		Map<Integer, Map<Integer, Point[]>> points = new HashMap<Integer, Map<Integer, Point[]>>();
		Map<Integer, List<Polygon>> polygons = new HashMap<Integer, List<Polygon>>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(file), "SJIS"));
			String line;
			while((line = bi.readLine()) != null) {
				switch (line.charAt(0)) {
					case 'L': {
							final int mesh = this.parseInt(line, 3, 9);
							final int linkID = this.parseInt(line, 27, 33);
							int length = this.parseInt(line, 45, 51);
							Point[] np = new Point[length];
							Point[] rp = new Point[length];
							// 中間点データの読み込み
							int index = 0;
							int end = (length - 1) / 5 + 1;
							for(int j = 0; j < end; j++) {
								line = bi.readLine();
								StringTokenizer st = new StringTokenizer(line);
								while(st.hasMoreTokens()) {
									Point p = new Point(Integer.parseInt(st.nextToken()) * 100, Integer.parseInt(st.nextToken()) * 100);
									np[index] = p;
									rp[length - 1 - index] = p;
									index++;
								}
							}
							if(length != index) {
								throw new UnknownError("length:"+ length +", index:"+ index +"\n");
							}
							if(!points.containsKey(mesh)) {
								points.put(mesh, new HashMap<Integer, Point[]>());
							}
							points.get(mesh).put( linkID, np);
							points.get(mesh).put(-linkID, rp);
						break;
					}
					case 'A' : {
						final int code = this.parseInt(line, 40, 45);
						final int length = this.parseInt(line, 45, 51);
						int end = (length - 1) / 5 + 1;
						List<Integer> aryX = new ArrayList<Integer>();
						List<Integer> aryY = new ArrayList<Integer>();
						for (int j = 0; j < end; j++) {
							line = bi.readLine();
							for (int k = 0; k < 5; k++) {
								int index = k * 14;
								if(12 + index >= line.length()) {
									break;
								}
								int meshL = this.parseInt(line, 0 + index, 6 + index);
								int idL = this.parseInt(line, 6 + index, 12 + index);
								for(Point p : points.get(meshL).get(idL)) {
									aryX.add(p.x);
									aryY.add(p.y);
								}
							}
						}
						int[] x = new int[aryX.size()];
						int[] y = new int[aryY.size()];
						for (int j = 0; j < aryX.size(); j++) {
							x[j] = aryX.get(j);
							y[j] = aryY.get(j);
						}
						if(!polygons.containsKey(code)) {
							polygons.put(code, new ArrayList<Polygon>());
						}
						polygons.get(code).add(new Polygon(x, y, x.length));
						break;
					}
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return polygons;
	}
}
