import java.awt.BorderLayout;
import java.awt.Polygon;
import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Properties;

import javax.swing.JFrame;
import javax.swing.UIManager;

import map.CityMap;
import map.IsjFactory;
import map.KsjFactory;
import map.Sdf25kFactory;
import map.SdfStorage;
import map.sdk25k.cell.CellSearch;
import map.store.Store;
import util.Log;
import util.Loader;
import view.MapPanel;
import view.StatusBar;
import controller.Controller;

/**
 * 地図描画アプリケーション起動のためのクラス
 * 
 * @author Masayasu Fujiwara
 */
public class Main {

	/**
	 * フレーム高さ
	 */
	private static final int F_WIDTH = 800;

	/**
	 * フレーム幅
	 */
	private static final int F_HEIGHT = 600;

	/**
	 * args[0]には地図データのルートディレクトリの位置を指定します。
	 * args[1]にはプラグインのディレクトリの位置を指定します。
	 * 
	 * @param args 引数
	 */
	public static void main(String[] args) {
		String mapDir = ".digital_map" + File.separatorChar;
		String pluginDir = "plugin" + File.separatorChar;
		String libDir = "lib" + File.separatorChar;
		switch (args.length) {
			case 3: libDir = args[2];
			case 2: pluginDir = args[1];
			case 1: mapDir = args[0];
		}

		// 外部JARをロードします。
		Loader.addExternalJar(libDir);
		
		// 都道府県データの相対位置リスト
		final String prefData = "/data/prefecture.csv";

		// 都道府県界のデータ
		final String prefBorderData = "/data/prefecture.dat";

		// 都道府県セルのデータ
		final String cellData = "/data/prefecture.cell";
		
		try {
			Properties properties = System.getProperties();
			if (((String) properties.get("java.version")).toLowerCase().contains("1.6")) {
				if (((String) properties.get("sun.desktop")).toLowerCase().contains("windows")){
					properties.setProperty("sun.java2d.d3d", "true");
					Log.out(Main.class, "setProperty sun.java2d.d3d=true");
				}
			}
			// properties.list(System.out);
		} catch (Exception e) {
			Log.err(Main.class, e);
		}

		final JFrame frame = new JFrame("Digital Map");
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.setLayout(new BorderLayout());
		
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (Exception e) {
			Log.err(Main.class, e);
		}

		MapPanel panel = new MapPanel();
		Log.out(Main.class, "created panel.");

		StatusBar statusbar = new StatusBar(" ");
		Log.out(Main.class, "created statusbar");

		
		// Listener
		Controller listener = new Controller(panel);
		Log.out(Main.class, "created listener.");

		frame.add(statusbar, BorderLayout.SOUTH);
		frame.add(panel, BorderLayout.CENTER);
		frame.add(listener.getMenuBar(), BorderLayout.NORTH);

		frame.setSize(Main.F_WIDTH, Main.F_HEIGHT);
		
		frame.setVisible(true);
		Log.out(Main.class, "frame set visible.");

		
		try {
			statusbar.setMessage("初期設定");

			Collection<Store> store = Loader.loadStorePlugin(pluginDir, mapDir + "store" + File.separatorChar);
			Log.out(Main.class, "loaded plugins.");

			
			final CityMap maps = new CityMap(panel, new CellSearch(cellData),
					new Sdf25kFactory(new SdfStorage(mapDir, prefData, statusbar)),
					new IsjFactory(mapDir + "isj" + File.separatorChar), store, statusbar);
			statusbar.clearReading();
			Log.out(Main.class, "created maps.");

			statusbar.setMessage("国土数値情報 都道府県");
			final Polygon[] prefecture = KsjFactory.readPolygon(prefBorderData);
			// int[] coast = new int[]{82, 83, 84, 143, 288, 457, 505, 582, 725, 878, 941, 967, 1007, 1008, 1032, 1592, 1593, 1595, 1694, 1828, 1909, 1910, 1911, 1912, 1913, 1914, 1915, 2073, 2253, 2345, 2641, 3335, 3466, 3503, 3595, 3705, 4191, 4418, 4531, 4600, 5193};
			Log.out(Main.class, "created prefecture.");

			// 湖などを繰りぬく
			final int[] coast = new int[] { 2345 };
			final Polygon[] prefectureCoast = new Polygon[coast.length];
			int count = 0;
			for (final int i : coast) {
				prefectureCoast[count++] = prefecture[i];
			}
			statusbar.clearReading();

			panel.init(maps, prefecture, prefectureCoast);
			maps.start();
			Log.out(Main.class, "started maps.");

		} catch (IOException e) {
			statusbar.setMessage(e.getMessage());
			Log.err(Main.class, e);
		}
		statusbar.clearReading();

		frame.addKeyListener(listener);
		panel.addKeyListener(listener);
		panel.addMouseListener(listener);
		panel.addMouseMotionListener(listener);
		panel.addMouseWheelListener(listener);

		statusbar.setThreadPriority(Thread.MIN_PRIORITY);
		
		Log.out(Main.class, "finished main().");
	}
	
	/**
	 * インスタンス生成不要
	 */
	private Main() {
	}
}
