package map.route;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;


import map.data.MapMap;
import map.data.Node;
import map.data.Road;
/**
 * Unidirectional Search algorithm
 * ヒューリスティック関数を設定すればA*アルゴリズム
 * 設定しなければDijkstraのアルゴリズムにより最短経路を求める。
 * @author Masayasu FUJIWARA
 */
public class SearchThread extends Thread {
	/**
	 * 地図データ
	 */
	private final MapMap maps;
	private final Node start;
	private final Node terminal;
	private final Set<Road> route;
	private Map<Long, Long> path;
	private boolean isSearched;
	private HeapMap heap;
	private boolean isAlive;
	private Node p;
	private float cost;
	/**
	 * A* algorithm
	 * @param heuristic ヒューリステック関数
	 * @param maps 地図データ
	 */
	public SearchThread(Node start, Node terminal, MapMap maps, HeapMap heap) {
		this.start = start;
		this.terminal = terminal;
		this.maps = maps;
		this.heap = heap;
		this.route = new HashSet<Road>();
		this.path = new HashMap<Long, Long>();
		this.isSearched = false;
		this.isAlive = true;
		/**
		 * これ以上にすると満足に操作できなくなる。
		 */
		this.maps.setSearchThread(this);
		this.setPriority(Thread.MIN_PRIORITY);
		this.start();
	}
	@Override
	public void run() {
		try {
			long t = this.terminal.getID();
			// 初期化
			Map<Long, Float> vp = new HashMap<Long, Float>();
			this.heap.put(this.start, 0);
			do {
				HeapMap.Entry entry = this.heap.poll();
				this.p = entry.getKey();
				float d = entry.getValue();
				if (this.p.getID() == t) {
					this.cost = d;
					this.maps.clearNode();
					this.vp = vp.size();
					this.traceback(this.terminal.getID());
					break;
				}
				// p を訪問済み頂点とする
				vp.put(this.p.getID(), d);
				// 隣接頂点までの距離を設定
				for (Map.Entry<Long, Road> edge : this.p.getEdge().entrySet()) {
					Long key = edge.getKey();
					if (vp.containsKey(key)) {
						continue;
					}
					final Node v = this.maps.getNode(key, false);
					final Road value = edge.getValue();
					final float cost = (value == null) ? 0 : this.getCost(value);

					if(cost == Float.POSITIVE_INFINITY) {
						continue;
					}

					if(this.heap.put(v, d + cost)) {
						this.path.put(v.getID(), this.p.getID());
					}
				}
			} while (this.isAlive && !this.heap.isEmpty());
			this.isSearched = true;
		} catch (IOException e) {
			e.printStackTrace();
			this.route.clear();
		} finally {
			this.path = null;
			this.heap = null;
			this.maps.searchedRecover();
		}
	}
	/**
	 * 求めた最短経路をトレースバックします。
	 * @param r
	 * @throws IOException
	 */
	private void traceback(Long r) throws IOException {
		while (this.isAlive && this.path.containsKey(r)) {
			Long next = this.path.get(r);
			if (next == null) {
				break;
			}
			Node n = this.maps.getNode(r, true);
			if (n == null) {
				System.out.println(r + " > "+ next);
				break;
			}
			Road road = n.getEdge().get(next);
			if(road != null) {
				if (road.getArrayX() == null) {
					throw new IOException();
				}
				this.route.add(road);
			}
			r = next;
		}
	}
	/**
	 * 探索済みなら最短経路を返す
	 * @return 最短経路
	 */
	public Set<Road> getRoute() {
		if (this.isSearched && this.isAlive) {
			return this.route;
		} else {
			return null;
		}
	}
	private int vp;
	public int getVP () {
		if (this.isSearched && this.isAlive) {
			return this.vp;
		} else {
			return 0;
		}
	}
	/**
	 * このスレッドを殺す
	 *
	 */
	public void kill() {
		this.isAlive = false;
	}
	@Override
	public String toString() {
		if (this.isAlive) {
			StringBuilder sb = new StringBuilder(" / SEARCH : ");
			sb.append(this.start.getID());
			sb.append(" => ");
			sb.append(this.terminal.getID());
			if (!this.isSearched) {
				sb.append(" : ");
				if(this.p != null) {
					double b = this.start.distance(this.p);
					double a = this.terminal.distance(this.p);
					sb.append((int)(b * 100 / (a + b)));
				} else {
					sb.append(0);
				}
				sb.append('%');
			} else {
				sb.append(" : COST = ");
				sb.append(this.cost);
			}
			return sb.toString();
		} else {
			return "";
		}
	}
	/**
	 * 道路のコストを求める
	 * @param road 道路
	 * @return 道路のコスト
	 */
	private float getCost(Road road) {
		float cost = road.getCost();
		switch (road.getType()) {
			case 3 :
				if(Road.useHighway()) {
					return cost;
				}
				return Float.POSITIVE_INFINITY;
			default :
				float rate = 1 + (5 - road.getWidth()) * 0.5f;
				if (Road.useHighway()) {
					rate *= Road.GENERAL_RATE;
				}
				return cost * rate;
		}
	}
}