package map.map25000;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.regex.Pattern;

import map.data.MapData;

import util.StatusBar;


import jp.gr.java_conf.dangan.util.lha.LhaHeader;
import jp.gr.java_conf.dangan.util.lha.LhaInputStream;

public class Warehouse {
	/**
	 * 文字列の分割
	 */
	private static final Pattern splitCSV = Pattern.compile(",");
	private final String baseDir;
	private final String gsi;
	private final String[] prefecture;
	private final StatusBar status;

	/**
	 * コンストラクタ
	 * @param baseDir - データ格納ディレクトリ
	 * @param gsi - 国土地理院のURL
	 * @param list - リストのアドレス
	 * @throws IOException 
	 */
	public Warehouse(String baseDir, String gsi, InputStream list, StatusBar status) throws IOException {
		this.baseDir = baseDir;
		File dir = new File(baseDir);
		
		if(!dir.isDirectory()) {
			dir.mkdirs();
		}

		this.gsi = gsi;
		// 都道府県番号（都道府県数47）
		this.prefecture = new String[48];
		this.status = status;

		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(list, "SJIS"));
			int i = 1;
			while(bi.ready()) {
				this.prefecture[i++] = Warehouse.splitCSV.split(bi.readLine())[1];
			}
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
	}
	/**
	 * エリアコードに対応するファイル配列を返す
	 * @param code - エリアコード
	 * @return エリアコードに対応したファイル配列
	 * @throws IOException 
	 */
	public synchronized File[] get(String code) throws IOException {

		File[] file = null;
		// 圧縮ファイル
		final File lzh = new File(this.baseDir + code + ".lzh");
		// 保存先ディレクトリ
		final File dir = new File(this.baseDir + code);

		if (lzh.exists() || !dir.isDirectory() || dir.list().length == 0) {
			dir.mkdir();
			this.download(code, lzh);
		}
		if(lzh.exists()) {
			// ファイルの展開
			file = this.extract(lzh);
			if(!lzh.delete()){
				System.out.print(" / lzh delete failed.");
			}
		} else if(dir.isDirectory()) {
			if(dir.toString().endsWith("13421")) {
				Collection<File> subFileList = new ArrayList<File>();
				for(File subdir : dir.listFiles()) {
					for(File subfile : subdir.listFiles()) {
						subFileList.add(subfile);
					}
				}
				file = subFileList.toArray(new File[]{});
			}else{
				file = dir.listFiles();
			}
		}
		
		if(file == null || file.length == 0) {
			throw new IOException();
		}

		return file;
	}
	/**
	 * ファイルのダウンロード
	 * @param code
	 * @param file
	 * @throws IOException 
	 */
	private void download(String code, File file) throws IOException {
		int index = Integer.parseInt(code.substring(0, 2));

		URL url = null;
		try {
			url = new URL(this.gsi + this.prefecture[index] + code+ ".lzh");
		}catch(Exception e) {
			e.printStackTrace();
			System.out.printf("code : %s, %d", code, index);
		}
		InputStream in = null;
		OutputStream out = null;
		try {
			URLConnection connect = url.openConnection();
			// ファイルのチェック（ファイルサイズの確認）
			int contentLength = connect.getContentLength();
			if (contentLength != file.length()) {
				System.out.print(" / download");
				if(this.status != null) {
					this.status.set(code, contentLength, file);
				}
				final long start = System.currentTimeMillis();
				// ダウンロード
				in = connect.getInputStream();
				out = new FileOutputStream(file);
				this.copy(in, out);
				System.out.printf(": %dms", (System.currentTimeMillis() - start));
			}
		} finally {
			if(in != null) {
				in.close();
			}
			if(out != null) {
				out.close();
			}
		}
	}
	/**
	 * ファイルのコピーを行う。
	 * 入出力のストリームは閉じないので注意すること。
	 * @param in 入力ストリーム
	 * @param out 出力ストリーム
	 * @throws IOException
	 */
	private void copy(InputStream in, OutputStream out) throws IOException {
		final byte buf[] = new byte[1024];
		int size;
		while ((size = in.read(buf)) != -1) {
			out.write(buf, 0, size);
			out.flush();
		}
	}
	/**
	 * 圧縮ファイルを展開する
	 * @param file 展開するファイル
	 * @return 展開したファイル配列
	 * @throws IOException 
	 * @throws FileNotFoundException
	 * @throws IOException
	 */
	private File[] extract(File file) throws IOException {
		
		System.out.println(file);
		final long start = System.currentTimeMillis();

		boolean isExtracted = false;

		LhaInputStream in = null;
		FileOutputStream out = null;
		Collection<File> extracted = new ArrayList<File>();
		try {
			in = new LhaInputStream(new FileInputStream(file));
			LhaHeader entry;
			
			while ((entry = in.getNextEntry()) != null) {
				// 出力先ファイル
				File outFile;
				// 13421のときは例外
				if(entry.getPath().startsWith("1342")) {
					outFile = new File(this.baseDir + "13421/" + entry.getPath());
					final File dirParent = new File(this.baseDir + "13421/" + entry.getPath()).getParentFile();
					if(!dirParent.isDirectory()) {
						dirParent.mkdir();
						System.out.println(dirParent);
					}
				} else {
					outFile = new File(this.baseDir + entry.getPath());
					if(entry.getPath().endsWith("/")) {
						if(outFile.isDirectory()) {
							continue;
						}else{
							System.err.println("Error!!");
							System.exit(1);
						}
					}
				}
				// 出力先ファイル
				if (!outFile.exists() || entry.getOriginalSize() != outFile.length()) {
					out = new FileOutputStream(outFile);
					this.copy(in, out);
					isExtracted = true;
				}
				extracted.add(outFile);
			}
		} finally {
			if (in != null) {
				in.close();
			}
			if (out != null) {
				out.close();
			}
		}
		final long end = System.currentTimeMillis();
		if (isExtracted) {
			System.out.printf(" / extract: %dms", end - start);
		}
		return extracted.toArray(new File[]{});
	}
	/**
	 * 頂点の外部への接続情報の取得
	 * @param code 市町村番号
	 * @return 市町村番号に対応する頂点の外部への接続情報
	 */
	public InputStream getBoundaryNode(int code) {
		String codeStr = MapData.codeFormat.format(code);
		return this.getClass().getResourceAsStream("/data/" + codeStr.substring(0, 2) + "/" + codeStr +".nod");
	}
}
