package map.data;

import java.awt.Shape;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.HashMap;

import util.StatusBar;

import map.MapPanel;
import map.cell.CellSearch;
import map.map25000.MapFactory;
import map.route.SearchThread;


public class MapMap extends HashMap<Integer, MapData> implements Runnable {

	/**
	 * 地図パネルクラス
	 */
	private final MapPanel panel;
	
	/**
	 * セル型の地域検索クラス
	 */
	private final CellSearch cell;

	/**
	 * 地図データ作成クラス
	 */
	private final MapFactory factory;
	/**
	 * ステータスバー
	 */
	private final StatusBar statusbar;
	
	private Shape screen;
	
	public MapMap(MapPanel panel, CellSearch cell, MapFactory factory, StatusBar statusbar) {
		this.panel = panel;
		this.cell = cell;
		this.factory = factory;
		this.statusbar = statusbar;
	}
	public void start() {
		Thread thread = new Thread(this);
		thread.setPriority(3);
		thread.start();
	}
	/**
	 * 地図情報を加える
	 * 
	 * @param map 加える地図情報
	 */
	public synchronized void put(MapData map) {
		super.put(map.getCode(), map);
	}
	/**
	 * 地図情報の削除
	 * @param code 削除する市町村番号
	 */
	public synchronized void remove(String code) {
		super.remove(code);
	}

	/**
	 * 指定した頂点番号の頂点を取得
	 * @param id 頂点番号
	 * @return 頂点
	 * @throws IOException 
	 * @throws URISyntaxException 
	 */
	public Node getNode(long id, boolean isDetailRoad) throws IOException {
		MapData map = super.get((int)(id / 1000000));
		if (!map.hasNode()) {
			this.factory.productNode(map, isDetailRoad);
		}
		Node node = map.getNode((int)(id % 1000000));
		if(node == null) {
			System.out.println("Search node is not found, and reproduct node. "+ this);
			this.factory.productNode(map, isDetailRoad);
			node = map.getNode((int)(id % 1000000));
			if (node == null) {
				throw new IOException();
			}
		}
		return node;
	}
	public void removeNode(long id) {
		MapData map = super.get((int)(id / 1000000));
		if(!map.isLoaded()) {
			map.removeNode((int)(id % 1000000));
		}
	}
	public synchronized Collection<Node> getNodes(int code) {
		return super.get(code).getNode();
	}
	public void run() {
		while(true) {
			this.screen = this.panel.getScreen();
			this.dump();
			if(this.panel.mode() == 2 && this.screen != null) {
				try {
					// セルメソッドから地図の読み込み
					for(int code : this.cell.search(this.screen)) {
						MapData map = this.get(code);
						try {
							if(map.getArea() == null) {
								this.statusbar.setCheckCode(MapData.codeFormat.format(map.getCode()));
								this.factory.preproduct(map);
								this.statusbar.setCheckCode();
							}
						} catch (NullPointerException e) {
							System.out.println("Error code : " + code);
							map = this.factory.preproduct(code);
							this.put(code, map);
						}
						if (this.screen.intersects(map.getArea())) {
							if(!map.hasData()) {
								this.statusbar.setReading(MapData.codeFormat.format(map.getCode()));
								this.factory.product(map);
								this.statusbar.setReading();
								this.panel.repaint();
							}
						}
					}
				} catch (IOException e) {
					e.printStackTrace();
				} catch (OutOfMemoryError e) {
					e.printStackTrace();
					this.dump();
				}
			}
			try {
				Thread.sleep(2000L);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
	}
	public synchronized void clearNode() {
		if (this.screen != null) {
			for (MapData map : super.values()) {
				if (!map.hasData() && map.hasNode()) {
					map.dump();
				}
				if (map.isLoaded() && !this.screen.intersects(map.getArea())) {
					map.dump();
				}
			}
		}
	}
	public void clearStatus() {
		this.statusbar.clearLocation();
	}
	public void setStatus(int x, int y) {
		int height = Integer.MIN_VALUE;
		double meshDist = Double.POSITIVE_INFINITY;
		for (MapData map : this.values()) {
			if (!map.hasData() || !map.getArea().contains(x, y)) {
				continue;
			}
			for (final Mesh mesh : map.getMesh()) {
				final double d = mesh.distance(x, y);
				if (d + 0.5 <= Mesh.SIZE && meshDist > d) {
					height = mesh.getHeight();
					meshDist = d;
				}
			}
		}
		this.statusbar.setLocation((float)(x / 36000) / 100, (float)(y / 36000) / 100, height);
	}
	public synchronized void searchedRecover() {
		this.panel.repaint();
		this.clearNode();
	}
	public void setSearchThread(SearchThread thread) {
		this.statusbar.set(thread);
	}
	public synchronized void dump() {
		if (this.screen != null) {
			for (MapData map : super.values()) {
				if(map.hasData() && !this.screen.intersects(map.getArea())) {
					System.out.println("dump "+ map.getCode());
					map.dump();
				}
			}
		}
	}
}
