package map.data;

import java.awt.Polygon;
import java.awt.Rectangle;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Collection;
import java.util.Map;

public class MapData {
	/**
	 * 市町村番号フォーマット
	 */
	public final static NumberFormat codeFormat = new DecimalFormat("00000");
	/**
	 * 都道府県番号フォーマット
	 */
	public final static NumberFormat prefFormat = new DecimalFormat("00");
	/**
	 * 外接長方形
	 */
	private Rectangle area;

	/**
	 * 行政界
	 */
	private Curve[] border;
	/**
	 * 市町村名
	 */
	private Facility[] city;
	/**
	 * 海岸線
	 */
	private Map<Curve, Boolean> coast;
	/**
	 * 地域番号
	 */
	private final int code;
	
	/**
	 * 施設配列
	 */
	private Facility[] facility;
	
	/**
	 * 詳細データを保持しているかどうか
	 */
	private boolean isLoaded;
	
	/**
	 * 標高メッシュ配列
	 */
	private Mesh[] mesh;
	
	/**
	 * 頂点配列
	 */
	private Map<Integer, Node> node;

	/**
	 * 国土数値情報の市町村界
	 */
	private final Polygon[] polygon;

	/**
	 * 鉄道
	 */
	private Curve[] rail;
	
	/**
	 * 河川
	 */
	private Curve[] river;
	
	/**
	 * 道路
	 */
	private Road[][] road;

	/**
	 * 駅
	 */
	private Curve[] station;
	
	public MapData (int code, Polygon[] polygon) {
		this.code = code;
		this.polygon = polygon;
		this.isLoaded = false;
	}
	/**
	 * コンストラクタ
	 * @param code 市町村番号
	 * @param area バウンディングボックス
	 */
	public MapData(int code, Rectangle area) {
		this.code = code;
		this.area = area;
		this.isLoaded = false;
		this.polygon = new Polygon[0];
	}
	public MapData(String code, Rectangle area, Curve[] border, Map<Curve, Boolean> coast, Curve[] river, Curve[] rail, Curve[] station, Map<Integer, Node> node, Road[][] road, Facility[] city, Facility[] facility, Mesh[] mesh) {
		this.code = Integer.parseInt(code);
		this.area = area;
		this.rail = rail;
		this.station = station;
		this.node = node;
		this.road = road;
		this.river = river;
		this.border = border;
		this.coast = coast;
		this.city = city;
		this.facility = facility;
		this.mesh = mesh;
		this.isLoaded = true;
		this.polygon = new Polygon[0];
	}
	/**
	 *　データの破棄
	 */
	public void dump() {
		this.rail = null;
		this.station = null;
		this.node = null;
		this.road = null;
		this.river = null;
		this.border = null;
		this.coast = null;
		this.facility = null;
		this.mesh = null;
		this.isLoaded = false;
	}
	/**
	 * エリアを返す
	 * @return
	 */
	public Rectangle getArea() {
		return this.area;
	}
	/**
	 * 地域高さを取得
	 * @return
	 */
	public int getAreaHeight() {
		return this.area.height;
	}
	/**
	 * 地域幅を取得
	 * @return
	 */
	public int getAreaWidth() {
		return this.area.width;
	}
	/**
	 * 行政界の取得
	 * @return 行政界
	 */
	public Curve[] getBorder() {
		return this.border;
	}
	/**
	 * 市町村名の取得
	 * @return 市町村ラベル
	 */
	public Facility[] getCity() {
		return this.city;
	}
	public Map<Curve, Boolean> getCoast() {
		return this.coast;
	}
	/**
	 * 市町村番号の取得
	 * @return 市町村番号
	 */
	public int getCode() {
		return this.code;
	}
	public Facility[] getFacility() {
		return this.facility;
	}
	public Mesh[] getMesh() {
		return this.mesh;
	}
	/**
	 * 道路頂点の取得
	 * @return 道路頂点
	 */
	public Collection<Node> getNode() {
		return this.node.values();
	}
	/**
	 * 道路頂点の取得
	 * @return 道路頂点
	 */
	public Node getNode(Integer id) {
		return this.node.get(id);
	}

	/** 原点座標Xを取得
	 * 
	 * @return
	 */
	public int getOriginX() {
		return this.area.x;
	}
	/**
	 * 原点座標Yを取得
	 * @return
	 */
	public int getOriginY() {
		return this.area.y;
	}
	/**
	 * 行政界ポリゴンの取得
	 * @return 行政界ポリゴン(国土数値情報)
	 */
	public Polygon[] getPolygon() {
		return this.polygon;
	}
	/**
	 * 都道府県番号の取得
	 * @return 都道府県番号
	 */
	public String getPrefectureCode() {
		return MapData.codeFormat.format(this.code).substring(0, 2);
	}
	/**
	 * 鉄道区間の取得
	 * @return 鉄道区間
	 */
	public Curve[] getRailway() {
		return this.rail;
	}
	/**
	 * 河川の取得
	 * @return 河川
	 */
	public Curve[] getRiver() {
		return this.river;
	}
	/**
	 * 道路の取得
	 * @return 道路
	 */
	public Road[][] getRoad() {
		return this.road;
	}
	/**
	 * 駅区間の取得
	 * @return 駅区間
	 */
	public Curve[] getStation() {
		return this.station;
	}
	/**
	 * データを読み込んでいるかどうか
	 * @return データを読み込んでいればtrue
	 */
	public boolean hasData() {
		return this.isLoaded;
	}
	public boolean hasNode() {
		return this.node != null;
	}
	public boolean isLoaded() {
		return this.area != null;
	}
	/**
	 * データの読み込みが行われていなければ指定した番号の頂点を削除する
	 * @param id 頂点番号
	 */
	public void removeNode (Integer id) {
		this.node.remove(id);
	}
	public void setData(Curve[] border, Map<Curve, Boolean> coast, Curve[] river, Curve[] rail, Curve[] station, Map<Integer, Node> node, Road[][] road, Facility[] city, Facility[] facility, Mesh[] mesh) {
		this.rail = rail;
		this.station = station;
		this.node = node;
		this.road = road;
		this.river = river;
		this.border = border;
		this.coast = coast;
		this.city = city;
		this.facility = facility;
		this.mesh = mesh;
		this.isLoaded = true;
	}
	/**
	 * 頂点データを追加
	 * @param node 頂点データを追加する。
	 */
	public void setData(Map<Integer, Node> node) {
		this.node = node;
		this.isLoaded = false;
	}
	public void setScreen(Rectangle area) {
		this.area = area;
	}
}
