package util;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;

/**
 * ディスク上にデータ構造を作るヒープ
 * @author fujiwara
 *
 */
public class DiscHeapMap {
	public static Collection<File> list = new HashSet<File>();
	public static void main(String[] args) throws IOException {
		DiscHeapMap disc = new DiscHeapMap();
		List<Integer> list = new ArrayList<Integer>();
		for (int j = 0; j < 5; j++) {
			for (int i = 0; i < 50; i++) {
				int value = (int)(10000 * Math.random());
				System.out.print("add("+value+")");
				list.add(value);
				disc.put(value, value);
				if(disc.size() != list.size()){
					System.out.println(" size error");
					System.exit(1);
				}else{
					System.out.println();
				}
			}
			for (int i = 0; i < 50; i++) {
				int value = (int)(10000 * Math.random());
				System.out.print("contains("+value+")");
				if (disc.containsKey(value) != list.contains(value)) {
					System.out.println(" oper error");
					System.exit(1);
				} else {
					System.out.println();
				}
			}
		}
		Collections.sort(list);
		for(int value : list){
			if(value != disc.poll().getKey()) {
				System.out.println("error");
			}
		}
	}
	/**
	 * データ構造を持つファイル
	 */
	private File tmp;
	/**
	 * ヒープ長
	 */
	private int size;
	/**
	 * データの単位
	 * [long:8][double:8]
	 */
	private int datasize = 8 + 8;
	/**
	 * コンストラクタ
	 * @throws IOException
	 */
	public DiscHeapMap() throws IOException {
		do{
			this.tmp = new File((int)(1000 * Math.random())+"dh.mem");
		} while(DiscHeapMap.list.contains(this.tmp));
		this.tmp.deleteOnExit();
		DiscHeapMap.list.add(this.tmp);
		RandomAccessFile disc = new RandomAccessFile(this.tmp, "rw");
		disc.setLength(0);
		disc.close();
		this.size = 0;
	}
	/**
	 * 指定された要素がセットの要素として存在しない場合に、その要素をセットに追加します。
	 * @param data
	 * @throws IOException
	 */
	public boolean put(long k, double v) throws IOException {
		RandomAccessFile disc = new RandomAccessFile(this.tmp, "rw");
		if (this.size > 0) {
			for(int i = 0; i < disc.length() / this.datasize; i++) {
				disc.seek(i * this.datasize);
				long key = disc.readLong();
				if(k == key) {
					double value = disc.readDouble();
					if(v < value) {
						this.fixUp(disc, i, k, v);
						return true;
					}
					break;
				}
			}
		}
		this.fixUp(disc, ++this.size, k, v);
		disc.close();
		return false;
	}
	/**
	 * 親との状態を確認して，マズければ入れ替える
	 * @param disc
	 * @param index
	 * @throws IOException 
	 */
	private void fixUp(RandomAccessFile disc, int index, long k, double v) throws IOException {
		int parent;
		while((parent = index >> 1) > 0) {
			disc.seek((parent - 1) * this.datasize);
			long pk = disc.readLong();
			double pv = disc.readDouble();
			disc.seek((index - 1) * this.datasize);
			if(pv <= v) {
				disc.writeLong(k);
				disc.writeDouble(v);
				break;
			}
			disc.writeLong(pk);
			disc.writeDouble(pv);
			index = parent;
		}
		if(parent == 0) {
			disc.seek(0);
			disc.writeLong(k);
			disc.writeDouble(v);
		}
	}
	/**
	 * 子との状態を確認して，マズければ入れ替える
	 * @param disc
	 * @param index
	 * @throws IOException 
	 */
	private void fixDown(RandomAccessFile disc, int index, long k, double v) throws IOException {
		int son;
		while((son = index << 1) <= this.size) {
			disc.seek((son - 1) * this.datasize);
			long sk = disc.readLong();
			double sv = disc.readDouble();
			if (son < this.size) {
				long sk2 = disc.readLong();
				double sv2 = disc.readDouble();
				if(sv > sv2) {
					sk = sk2;
					sv = sv2;
					son++;
				}
			}
			disc.seek((index - 1) * this.datasize);
			if(v <= sv) {
				disc.writeLong(k);
				disc.writeDouble(v);
				break;
			}
			disc.writeLong(sk);
			disc.writeDouble(sv);
			index = son;
		}
		if(son > this.size) {
			disc.seek((index - 1) * this.datasize);
			disc.writeLong(k);
			disc.writeDouble(v);
		}
	}
	/**
	 * ヒープの先頭を取得および削除します。
	 * キューが空の場合は null を返します。
	 * @return キューの先頭。キューが空の場合は null
	 * @throws IOException
	 */
	public Entry poll() throws IOException {
		if (this.size == 0) {
			return null;
		}
		RandomAccessFile disc = new RandomAccessFile(this.tmp, "rw");
		long k = disc.readLong();
		double v = disc.readDouble();

		disc.seek(--this.size * this.datasize);
		long key = disc.readLong();
		double value = disc.readDouble();
		this.fixDown(disc, 1, key, value);
		disc.setLength(this.size * this.datasize);
		disc.close();
		return new Entry(k, v);
	}
//	/**
//	 * 入れ替える
//	 * @param disc - データ構造を構築しているRandomAccessFile
//	 * @param index1 - 入れ替える位置
//	 * @param index2 - 入れ替える位置
//	 * @throws IOException 
//	 */
//	private void swap(RandomAccessFile disc, int index1, int index2) throws IOException {
//		disc.seek(index1 * this.datasize);
//		long k1 = disc.readLong();
//		double v1 = disc.readDouble();
//		disc.seek(index2 * this.datasize);
//		long k2 = disc.readLong();
//		double v2 = disc.readDouble();
//		disc.seek(index2 * this.datasize);
//		disc.writeLong(k1);
//		disc.writeDouble(v1);
//		disc.seek(index1 * this.datasize);
//		disc.writeLong(k2);
//		disc.writeDouble(v2);
//	}
	/**
	 * セットが、指定された要素を保持している場合に true を返します。
	 * @param data - このセットにあるかどうかを判定するオブジェクト
	 * @return セットが、指定された要素を保持している場合は true
	 * @throws IOException
	 */
	public boolean containsKey(long k) throws IOException {
		if (this.size == 0) {
			return false;
		}
		RandomAccessFile disc = new RandomAccessFile(this.tmp, "r");
		for(int i = 0; i < disc.length() / this.datasize; i++) {
			disc.seek(i * this.datasize);
			long key = disc.readLong();
			if(k == key) {
				disc.close();
				return true;
			}
		}
		disc.close();
		return false;
	}
	/**
	 * リスト内の要素数を返します。
	 * @return リスト内の要素数
	 */
	public int size() {
		return this.size;
	}
	/**
	 * マップが空のときtrueを返します．
	 * @return
	 */
	public boolean isEmpty() {
		return this.size == 0;
	}
	/**
	 * マップからすべての要素を削除します。
	 * @throws IOException
	 */
	public void clear() throws IOException {
		RandomAccessFile disc = new RandomAccessFile(this.tmp, "rw");
		disc.setLength(0);
		disc.close();
		this.size = 0;
	}
	/**
	 * インスタンス破棄時の処理
	 */
	@Override
	protected void finalize() throws Throwable {
		DiscHeapMap.list.remove(this.tmp);
		super.finalize();
	}
	/**
	 * 値を返すためのクラス
	 * @author fujiwara
	 *
	 */
	public static class Entry {
		private long key;
		private double value;
		/**
		 * コンストラクタ
		 * @param key - キー
		 * @param value - 値
		 */
		public Entry(long key, double value) {
			this.key = key;
			this.value = value;
		}
		/**
		 * キーを取得
		 * @return　キー
		 */
		public long getKey() {
			return this.key;
		}
		/**
		 * 値を取得
		 * @return 値
		 */
		public double getValue() {
			return this.value;
		}
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		try {
			RandomAccessFile disc = new RandomAccessFile(this.tmp, "rw");
			for(int i = 0; i < this.size; i++) {
				long key = disc.readLong();
				sb.append("[");
				sb.append(key);
				sb.append("]");
				disc.readDouble();
			}
			disc.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return sb.toString();
	}
}
