package map.route;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;

import map.data.Node;

public class HeapMap {
	/**
	 * ソートされるオブジェクト
	 */
	private Entry[] entries;
	/**
	 * ヒープ長
	 */
	private int size;
	private Map<Long, Integer> table;
	private final Comparator<Entry> comparator;
	/**
	 * コンストラクタ
	 *
	 */
	public HeapMap () {
		this(10, null);
	}
	/**
	 * コンストラクタ
	 * @param initialCapacity 初期容量
	 */
	public HeapMap (int initialCapacity) {
		this(initialCapacity, null);
	}
	/**
	 * コンストラクタ
	 * @param comparator
	 */
	public HeapMap (Comparator<Entry> comparator) {
		this(10, comparator);
	}
	/**
	 * コンストラクタ
	 * @param initialCapacity 初期容量
	 * @param comparator
	 */
	public HeapMap (int initialCapacity, Comparator<Entry> comparator) {
		if (initialCapacity < 1)
			throw new IllegalArgumentException();
		this.size = 0;
		this.entries = new Entry[initialCapacity + 1];
		this.table = new HashMap<Long, Integer>();
		this.comparator = comparator;
	}
	/**
	 * 挿入する
	 * @param key
	 * @param value
	 */
	public boolean put(Node key, float value) {
		Entry entry = new Entry(key, value);
		if (this.table.containsKey(key.getID())) {
			int index = this.table.get(key.getID());
			if (this.comparator == null) {
				if(((Comparable<Entry>)this.entries[index]).compareTo(entry) > 0) {
					this.entries[index] = entry;
					this.fixUp(index);
				} else {
					return false;
				}
			} else {
				if (this.comparator.compare(this.entries[index], entry) > 0) {
					this.entries[index] = entry;
					this.fixUp(index);
				} else {
					return false;
				}
			}
		} else {
			this.grow(++this.size);
			this.table.put(key.getID(), this.size);
			this.entries[this.size] = entry;
			this.fixUp(this.size);
		}
		return true;
	}
	/**
	 * 入れ替える
	 * @param index1
	 * @param index2
	 */
	private void swap(int index1, int index2) {
		Entry tmp = this.entries[index1];
		this.entries[index1] = this.entries[index2];
		this.entries[index2] = tmp;
		this.table.put(this.entries[index1].getKey().getID(), index1);
		this.table.put(this.entries[index2].getKey().getID(), index2);
	}
	/**
	 * 根を取り出す
	 * @return
	 */
	public Entry poll() {
		if (this.size == 0)
			return null;

		Entry entry = this.entries[1];
		this.table.remove(entry.getKey());
		if(this.size > 1) {
			this.entries[1] = this.entries[this.size];
			this.table.put(this.entries[1].getKey().getID(), 1);
		}
		this.entries[this.size] = null; // Drop extra ref to prevent memory leak
		if(--this.size > 1) {
			this.fixDown(1);
		}
		return entry;
	}
	/**
	 * 根を削除せずに取り出す
	 * @return
	 */
	public Entry peek() {
		return this.entries[1];
	}
	/**
	 * key から value を取得
	 * @param key
	 * @return
	 */
	public float get(Object key){
		return this.entries[this.table.get(key)].getValue();
	}
	/**
	 * @param key
	 * @return
	 */
	public boolean containsKey(Object key) {
		return this.table.containsKey(key);
	}
	/**
	 * 子との状態の比較
	 * @param index
	 */
	private void fixDown(int index) {
		int son;
		if(this.comparator == null) {
			while((son = index << 1) <= this.size) {
				if (son < this.size && this.entries[son].compareTo(this.entries[son+1]) > 0) {
					son++;
				}
				if(this.entries[index].compareTo(this.entries[son]) <= 0) {
					break;
				}
				this.swap(index, son);
				index = son;
			}
		}else{
			while((son = index << 1) <= this.size) {
				if (son < this.size && this.comparator.compare(this.entries[son], this.entries[son+1]) > 0) {
					son++;
				}
				if (this.comparator.compare(this.entries[index], this.entries[son]) <= 0) {
					break;
				}
				this.swap(index, son);
				index = son;
			}
		}
	}
	/**
	 * 親との状態を確認
	 * @param index
	 */
	private void fixUp(int index) {
		int parent;
		if(this.comparator == null) {
			while((parent = index >> 1) > 0) {
				if(this.entries[index].compareTo(this.entries[parent]) >= 0) {
					break;
				}
				this.swap(index, parent);
				index = parent;
			}
		}else{
			while((parent = index >> 1) > 0) {
				if(this.comparator.compare(this.entries[index], this.entries[parent]) >= 0) {
					break;
				}
				this.swap(index, parent);
				index = parent;
			}
		}
	}
	/**
	 * 
	 * @return 空なら真
	 */
	public boolean isEmpty() {
		return this.size == 0;
	}
	/**
	 * Resize Arrays
	 * @param index
	 */
	private void grow(int index) {
		int newLength = this.entries.length;
		if (index < newLength) {
			// don't need to grow
			return;
		}
		if (index == Integer.MAX_VALUE) {
			throw new OutOfMemoryError();
		}
		while (newLength <= index) {
			if (newLength >= Integer.MAX_VALUE / 2) // avoid overflow
				newLength = Integer.MAX_VALUE;
			else
				newLength <<= 2;
		}
		Entry[] newEntrys = new Entry[newLength];
		System.arraycopy(this.entries, 0, newEntrys, 0, this.entries.length);

		this.entries = newEntrys;
	}
	/**
	 * 値を返すためのクラス
	 * @author fujiwara
	 *
	 * @param <K>
	 * @param <V>
	 */
	public class Entry implements Comparable<Entry> {
		private Node key;
		private float value;
		public Entry(Node key, float value) {
			this.key = key;
			this.value = value;
		}
		public Node getKey() {
			return this.key;
		}
		public float getValue() {
			return this.value;
		}
		@Override
		public String toString() {
			return this.key +"->"+ Float.toString(this.value);
		}
		public int compareTo(Entry o) {
			if (this.value > o.value) {
				return 1;
			} else if (this.value < o.value) {
				return -1;
			} else {
				return 0;
			}
		}
	}
	/**
	 * 文字列で返す
	 */
	@Override
	public String toString() {
		if(this.size == 0) return "";
		StringBuilder sb = new StringBuilder(this.entries[1].toString());
		for(int i = 2; i <= this.size; i++) {
			sb.append("," + this.entries[i].toString());
		}
		return sb.toString();
	}
	/**
	 * サイズ
	 * @return
	 */
	public int size() {
		return this.size;
	}
}