package map.map25000;

import java.awt.Point;
import java.awt.Rectangle;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import map.data.Curve;
import map.data.CoastNode;
import map.data.Facility;
import map.data.MapData;
import map.data.Mesh;
import map.data.Node;
import map.data.Road;

public class MapFactory {
	/**
	 * ファイル読み込みのための正規表現
	 */
	private final Pattern pattern = Pattern.compile(
			"([A-Z]{2})"+
			"(?:"+
				"(?:"+
					"\\("+
					"([^)]*)"+ "\\)"+
				")"+
				"|"+
				"\\{"+ "([^\\}\\{]*)"+ "\\}"+
			")"+
			"(?:"+
				"\\{"+
				"(?:"+
					"("+
						"(?:"+ "\\d{6},)*" + "\\d{6}" +
					")" +
				"|" +
					"(" +
						"(?:" +
							"[A-Z]{2}" +
							"\\(" + "[^\\)]*" + "\\)" +
						")*" +
					")" +
				")" +
			"\\}" +
			")?");
	/**
	 * 必要なファイルを取得するためのクラス
	 */
	private final Warehouse warehouse;

	/**
	 * コンストラクタ
	 * @param warehouse - 必要なファイルを取得するためのクラス
	 */
	public MapFactory(Warehouse warehouse) {
		this.warehouse = warehouse;
	}
	/**
	 * エリアコードからファイルを取得
	 * @param code - エリアコード
	 * @return - ファイルマップ
	 * @throws IOException 
	 */
	private Map<String, File> getFile(int code) throws IOException {
		Map<String, File> mapfiles = new HashMap<String, File>();
		for (File file : this.warehouse.get(MapData.codeFormat.format(code))) {
			String str = file.toString();
			if (str.endsWith(".sal")) {
				if (str.endsWith("DS.sal")) {
					// 道路接点
					mapfiles.put("DS.sal", file);
				} else if (str.endsWith("DK.sal")) {
					// 道路区間
					mapfiles.put("DK.sal", file);
				} else if (str.endsWith("SK.sal")) {
					// 水域界
					mapfiles.put("SK.sal", file);
				} else if (str.endsWith("GK.sal")) {
					// 行政界
					mapfiles.put("GK.sal", file);
				} else if (str.endsWith("CM.sal")) {
					// 市町村名
					mapfiles.put("CM.sal", file);
				} else if (str.endsWith("MH.sal")) {
					// メッシュ標高
					mapfiles.put("MH.sal", file);
				} else if (str.endsWith("KK.sal")) {
					// 河川区間
					mapfiles.put("KK.sal", file);
				} else if (str.endsWith("KO.sal")) {
					// 施設
					mapfiles.put("KO.sal", file);
				} else if (str.endsWith("TK.sal")) {
					// 鉄道区間
					mapfiles.put("TK.sal", file);
				} else if (str.endsWith("EK.sal")) {
					// 駅
					mapfiles.put("EK.sal", file);
				}
			} else if (str.endsWith(".slp")) {
				if (str.endsWith("MH.slp")) {
					// メッシュ標高
					mapfiles.put("MH.slp", file);
				} else {
					mapfiles.put(".slp", file);
				}
			} else if (str.endsWith(".slm")) {
				mapfiles.put(".slm", file);
			}
		}
		return mapfiles;
	}
	/**
	 * エリア範囲を取得
	 * @param code
	 * @return
	 * @throws IOException 
	 */
	public MapData preproduct(int code) throws IOException {
		Map<String, File> mapfile = this.getFile(code);
		Rectangle area = this.readSLM(mapfile.get(".slm"));
		return new MapData(code, area);
	}
	/**
	 * エリア範囲を取得
	 * @param code
	 * @return
	 * @throws IOException 
	 */
	public void preproduct(MapData map) throws IOException {
		Map<String, File> mapfile = this.getFile(map.getCode());
		Rectangle area = this.readSLM(mapfile.get(".slm"));
		map.setScreen(area);
	}
	public void productNode (MapData map, boolean isDetailRoad) throws IOException {
		Map<String, File> mapfile = this.getFile(map.getCode());
		if(!map.hasData()) {
			Rectangle area = this.readSLM(mapfile.get(".slm"));
			map.setScreen(area);
		}
		Point[] slp = this.readSLP(mapfile.get(".slp"), map.getArea());
		// 道路接点の読み込み
		Map<Integer, Node> nodes = this.readNode(MapData.codeFormat.format(map.getCode()), mapfile.get("DS.sal"), slp);
		// 道路区間の読み込み
		this.readRoad(map.getCode(), mapfile.get("DK.sal"), slp, nodes, isDetailRoad);
		map.setData(nodes);
	}
	/**
	 * エリア情報を生成
	 * @param map エリアデータ
	 * @throws IOException 
	 * @throws URISyntaxException 
	 */
	public void product(MapData map) throws IOException {
		
		Map<String, File> mapfile = this.getFile(map.getCode());

		Mesh[] mesh = this.readMesh(mapfile.get("MH.slp"), mapfile.get("MH.sal"), map.getArea()).toArray(new Mesh[]{});

		Point[] slp = this.readSLP(mapfile.get(".slp"), map.getArea());
		
		Facility[] city = this.readFacility(mapfile.get("CM.sal"), slp);
		
		Facility[] facility = this.readFacility(mapfile.get("KO.sal"), slp);
		
		// 道路接点の読み込み
		Map<Integer, Node> nodes = this.readNode(MapData.codeFormat.format(map.getCode()), mapfile.get("DS.sal"), slp);

		// 道路区間の読み込み
		Road[][] road = this.readRoad(map.getCode(), mapfile.get("DK.sal"), slp, nodes, true);
		
		// 鉄道区間の読み込み
		Map<Integer, Curve> railMap = this.readRailway(mapfile.get("TK.sal"), slp);
		
		// 駅区間の読み込み
		Curve[] station = this.readStation(mapfile.get("EK.sal"), railMap);
		
		// 河川区間の読み込み
		Curve[] river = this.readBorder(mapfile.get("KK.sal"), slp);
		
		// 水域界の読み込み
		Map<Curve, Boolean> coast = this.readCoast(mapfile.get("SK.sal"), slp);
		
		// 行政界の読み込み
		Curve[] border = this.readBorder(mapfile.get("GK.sal"), slp);

		Curve[] rail = railMap.values().toArray(new Curve[]{});

		map.setData(border, coast, river, rail, station, nodes, road, city, facility, mesh);
	}

	/**
	 * エリア情報を生成
	 * @param code - エリアコード
	 * @return エリア情報
	 * @throws IOException 
	 * @throws URISyntaxException 
	 */
	public MapData product(String code) throws IOException {
		Map<String, File> mapfile = this.getFile(Integer.parseInt(code));
		Rectangle area = this.readSLM(mapfile.get(".slm"));
		
		// ファイルの読み込み
		Mesh[] mesh = this.readMesh(mapfile.get("MH.slp"), mapfile.get("MH.sal"), area).toArray(new Mesh[]{});

		Point[] slp = this.readSLP(mapfile.get(".slp"), area);
		
		Facility[] city = this.readFacility(mapfile.get("CM.sal"), slp);
		
		Facility[] facility = this.readFacility(mapfile.get("KO.sal"), slp);
		
		// 道路接点の読み込み
		Map<Integer, Node> nodes = this.readNode(code, mapfile.get("DS.sal"), slp);
		
		// 道路区間の読み込み
		Road[][] road = this.readRoad(Integer.parseInt(code), mapfile.get("DK.sal"), slp, nodes, true);
		
		// 鉄道区間の読み込み
		Map<Integer, Curve> railMap = this.readRailway(mapfile.get("TK.sal"), slp);
		
		// 駅区間の読み込み
		Curve[] station = this.readStation(mapfile.get("EK.sal"), railMap);
		
		// 河川区間の読み込み
		Curve[] river = this.readBorder(mapfile.get("KK.sal"), slp);
		
		// 行政界の読み込み
		Curve[] border = this.readBorder(mapfile.get("GK.sal"), slp);

		// 水域界の読み込み
		Map<Curve, Boolean> coast = this.readCoast(mapfile.get("SK.sal"), slp);
		

		// 配列に変換
		Curve[] rail = railMap.values().toArray(new Curve[]{});
		
		// マップデータを返す
		return new MapData(code, area, border, coast, river, rail, station, nodes, road, city, facility, mesh);
	}
	/**
	 * SLPファイルの読み込み
	 * @param file
	 * @param area
	 * @return
	 * @throws IOException 
	 * @throws NumberFormatException 
	 */
	private Point[] readSLP(File file, Rectangle area) throws NumberFormatException, IOException {
		Map<Point, Point> map = new HashMap<Point, Point>();
		List<Point> slp = new ArrayList<Point>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new FileReader(file));
			while(bi.ready()) {
				String line = bi.readLine();
				int x = Integer.parseInt(line.substring(0, 7)) + area.x;
				int y = Integer.parseInt(line.substring(8, 15)) + area.y;
				Point p = new Point(x, y);
				if(map.containsKey(p)) {
					p = map.get(p);
				}
				slp.add(p);
			}
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
		return slp.toArray(new Point[]{});
	}
	/**
	 * 領域などの読み込み
	 * @param file
	 * @return
	 * @throws IOException 
	 */
	private Rectangle readSLM(File file) throws IOException {
		Rectangle rect = null;
		BufferedReader bi = new BufferedReader(new FileReader(file));
		// 原点読み込み
		StringTokenizer st = new StringTokenizer(bi.readLine(), ",");
		int x = (int)(Long.parseLong(st.nextToken()) / 10);
		int y = (int)(Long.parseLong(st.nextToken()) / 10);

		// 領域読み込み
		st = new StringTokenizer(bi.readLine(), ",");
		int width  = Integer.parseInt(st.nextToken()) / 10;
		int height = Integer.parseInt(st.nextToken()) / 10;
		rect = new Rectangle(x, y, width, height);
		bi.close();
		return rect;
	}
	/**
	 * 施設情報の読み込み
	 * @throws IOException 
	 */
	private Facility[] readFacility (File file, Point[] slp) throws IOException {
		List<Facility> facility = new ArrayList<Facility>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(file), "SJIS"));
			while(bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.pattern.matcher(line.substring(17));
				Point point = null;
				boolean isNM = false;
				String name = null;
				while(matcher.find()) {
					String param = matcher.group(1);
					if(param.equals("NM")) {
						name = matcher.group(3);
						isNM = true;
					} else if(param.equals("PT")) {
						point = slp[Integer.parseInt(matcher.group(4)) - 1];
					}
					if(isNM && point != null) {
						facility.add(new Facility(name, point.x, point.y));
					}
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return facility.toArray(new Facility[]{});
	}
	/**
	 * メッシュデータの読み込み
	 * @throws IOException 
	 */
	private List<Mesh> readMesh (File slp, File sal, Rectangle area) throws IOException {
		List<Mesh> mesh = new ArrayList<Mesh>();
		List<Point> points = new ArrayList<Point>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new FileReader(slp));
			while(bi.ready()) {
				String line = bi.readLine();
				int x = Integer.parseInt(line.substring(0, 7)) + area.x;
				int y = Integer.parseInt(line.substring(8, 15)) + area.y;
				points.add(new Point(x, y));
			}
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(sal), "SJIS"));
			while(bi.ready()) {
				String line = bi.readLine();
				//String category = line.substring(0, 2);
				Matcher matcher = this.pattern.matcher(line.substring(17));
				Point point = null;
				int height = 0;
				while(matcher.find()) {
					String param = matcher.group(1);
					if(param.equals("PT")) {
						point = points.get(Integer.parseInt(matcher.group(4)) - 1);
					} else if(param.equals("HK")) {
						height = Integer.parseInt(matcher.group(3));
					}
				}
				mesh.add(new Mesh(point.x, point.y, height));
			}
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
		return mesh;
	}
	/**
	 * 接点の読み込み
	 * @param file
	 * @param slp
	 * @return
	 * @throws IOException 
	 * @throws URISyntaxException 
	 */
	private Map<Integer, Node> readNode (String code, File file, Point[] slp) throws IOException {
		Map<Integer, Node> node = new HashMap<Integer, Node>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(file), "SJIS"));
			while(bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.pattern.matcher(line.substring(17));
				String id = null;
				Point point = null;
				int slpindex = -1;
				while(matcher.find()) {
					String param = matcher.group(1);
					if(param.equals("PT")) {
						slpindex = Integer.parseInt(matcher.group(4)) - 1;
						point = slp[slpindex];
					} else if(param.equals("ND")) {
						id = matcher.group(2).substring(5, 11);
					}
				}
				if(point != null && id != null) {
					if(code.equals("40101")) {
						if(id.equals("000985")) {
							point.setLocation(471453115, 122261236);
						}else if(id.equals("001020")) {
							point.setLocation(471457816, 122265788);
						}
					}else if (code.equals("38202") && id.equals("001867")) {
						point.setLocation(478795248, 122834185);
					}
					node.put(Integer.parseInt(id), new Node(Long.parseLong(code + id), point));
				}
			}
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
		return node;
	}
	/**
	 * 多角形面の向き
	 * @param x 多角形面を構成するx座標の配列
	 * @param y 多角形面を構成するy座標の配列
	 * @return
	 */
	public int polygonDirection (int[] x, int[] y) {
		int vector = 0;
		for (int i = 1; i < x.length - 1; i++) {
			int dx0 = x[i] - x[0];
			int dy0 = y[i] - y[0];

			int dx1 = x[i + 1] - x[0];
			int dy1 = y[i + 1] - y[0];
			
			vector += dx1 * dy0 - dy1 * dx0;
		}
		return (vector >= 0) ? 1 : -1;
	}
	/**
	 * 境界の読み込み
	 * @param file
	 * @param slp
	 * @return
	 * @throws IOException 
	 */
	private Curve[] readBorder (File file, Point[] slp) throws IOException {
		List<Curve> border = new ArrayList<Curve>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(file), "SJIS"));
			while(bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.pattern.matcher(line.substring(17));
				int[] curveX = null;
				int[] curveY = null;
				while(matcher.find()) {
					String id = matcher.group(1);
					if(id.equals("CV")) {
						// カーブメモリ
						String[] ref = matcher.group(4).split(",");
						curveX = new int[ref.length];
						curveY = new int[ref.length];
						for(int i = 0; i < ref.length; i++) {
							Point point = slp[Integer.parseInt(ref[i]) - 1];
							curveX[i] = point.x; curveY[i] = point.y;
						}
					}
				}
				Curve b = new Curve(curveX, curveY, 0);
				border.add(b);
			}
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
		return border.toArray(new Curve[]{});
	}
	/**
	 * 水域界の読み込み
	 * @param file 水域界のファイル
	 * @param slp
	 * @return 水域界データ
	 * @throws IOException 
	 */
	private Map<Curve, Boolean> readCoast (File file, Point[] slp) throws IOException {
		Map<Curve, Boolean> map = new HashMap<Curve, Boolean>();
		Map<Point, CoastNode> points = new HashMap<Point, CoastNode>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(file), "SJIS"));
			while(bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.pattern.matcher(line.substring(17));
				int type = 0;
				int[] curveX = null;
				int[] curveY = null;
				while(matcher.find()) {
					String id = matcher.group(1);
					if(id.equals("CV")) {
						// カーブメモリ
						String[] ref = matcher.group(4).split(",");
						curveX = new int[ref.length];
						curveY = new int[ref.length];
						for(int i = 0; i < ref.length; i++) {
							Point point = slp[Integer.parseInt(ref[i]) - 1];
							curveX[i] = point.x; curveY[i] = point.y;
						}
					} else if (id.equals("SR")) {
						/**
						 * 51 水涯線または湖岸線
						 * 52 海岸線
						 * 53 河口
						 * 54 湖沼と河川の境界
						 */
						type = Integer.parseInt(matcher.group(3)) - 51;
						if(type != 1) {
							type = 0;
						}
						if(type == 3) {
							System.out.println("SR 54 ? "+ file);
						}
					}
				}
				Curve border = new Curve(curveX, curveY, type);
				CoastNode p1 = new CoastNode(curveX[0], curveY[0]);
				CoastNode p2 = new CoastNode(curveX[curveX.length - 1], curveY[curveX.length - 1]);
				if(p1.equals(p2)) {
					map.put(border, true);
				} else if (border.getType() == 1) {
					map.put(border, false);
				} else {
					if(points.containsKey(p1)) {
						p1 = points.get(p1);
					} else {
						points.put(p1, p1);
					}
					if(points.containsKey(p2)) {
						p2 = points.get(p2);
					} else {
						points.put(p2, p2);
					}
					Curve b = p1.put(p2, border);
					if(b != null) {
						map.put(b, true);
					}
				}
			}
			this.extractCoast(points.values(), map);
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
		return map;
	}
	/**
	 * 海岸線の展開
	 * @param nodes
	 * @param map
	 */
	private void extractCoast(Collection<CoastNode> nodes, Map<Curve, Boolean> map) {
		Set<Curve> borderSet = new HashSet<Curve>();
		for (CoastNode node : nodes) {
			if(node.getBorder().size() == 2) {
				for (int i = 0; i < 4; i++) {
					Curve border = node.connect(i);
					if(border != null) {
						map.put(border, true);
					}
				}
			}
		}
		for (CoastNode node : nodes) {
			for (Curve border : node.getBorder()) {
				borderSet.add(border);
			}
		}
		for (Curve border : borderSet) {
			map.put(border, false);
		}
	}
	/**
	 * 駅区間の読み込み
	 * @return 駅区間の配列
	 * @throws IOException 
	 */
	public Curve[] readStation(File file, Map<Integer, Curve> railway) throws IOException {
		List<Curve> station = new ArrayList<Curve>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(file), "SJIS"));
			while(bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.pattern.matcher(line.substring(17));
				while(matcher.find()) {
					String id = matcher.group(1);
					if(id.equals("NM")) {
						// 駅名
	//					String name = matcher.group(3);
					} else if(id.equals("KN")) {
						station.add(railway.get(Integer.valueOf(matcher.group(2).substring(10, 16))));
					}
				}
			}
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
		return station.toArray(new Curve[]{});
	}
	
	/**
	 * 鉄道区間の読み込み
	 * @param file 鉄道区間ファイル
	 * @param slp
	 * @return Map<ID, 鉄道区間>
	 * @throws IOException
	 */
	private Map<Integer, Curve> readRailway (File file, Point[] slp) throws IOException {
		Map<Integer, Curve> border = new HashMap<Integer, Curve>();
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(file), "SJIS"));
			while(bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.pattern.matcher(line.substring(17));
				int[] curveX = null;
				int[] curveY = null;
				int type = 0;
				while(matcher.find()) {
					String id = matcher.group(1);
					if(id.equals("CV")) {
						// カーブメモリ
						String[] ref = matcher.group(4).split(",");
						curveX = new int[ref.length];
						curveY = new int[ref.length];
						for(int i = 0; i < ref.length; i++) {
							Point point = slp[Integer.parseInt(ref[i]) - 1];
							curveX[i] = point.x; curveY[i] = point.y;
						}
					} else if(id.equals("SB")) {
						type = Integer.parseInt(matcher.group(3)) - 43;
					}
				}
				Curve b = new Curve(curveX, curveY, type);
				border.put(Integer.parseInt(line.substring(8, 14)), b);
			}
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
		return border;
	}
	/**
	 * 道路の読み込み
	 * @param file 道路情報のファイル
	 * @param slp slpファイルの読み込んだもの
	 * @return 道路ファイル
	 * @throws IOException 
	 */
	@SuppressWarnings("unchecked")
	private Road[][] readRoad (int code, File file, Point[] slp, Map<Integer, Node> node, boolean isDetail) throws IOException {
		List<Curve>[] border = new List[6];
		List<Curve>[] highway = new List[6];
		if (isDetail) {
			for(int i = 0; i < 6; i++) {
				border[i] = new ArrayList<Curve>();
				highway[i] = new ArrayList<Curve>();
			}
		}
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new InputStreamReader(new FileInputStream(file), "SJIS"));
			while(bi.ready()) {
				String line = bi.readLine();
				Matcher matcher = this.pattern.matcher(line.substring(17));
				int[] curveX = null;
				int[] curveY = null;
				int type = 0;
				int width = 0;
				int n0 = -1;
				int n1 = -1;
				String[] ref = null;
				double cost = 0;
				while (matcher.find()) {
					String id = matcher.group(1);
					if (id.equals("CV")) {
						// カーブメモリ
						ref = matcher.group(4).split(",");
						curveX = new int[ref.length];
						curveY = new int[ref.length];
						Point p0 = null;
						for (int i = 0; i < ref.length; i++) {
							Point point = slp[Integer.parseInt(ref[i]) - 1];
							curveX[i] = point.x;
							curveY[i] = point.y;
							if (i == 0) {
								p0 = point;
							} else {
								long dx = p0.x - point.x;
								long dy = p0.y - point.y;
								cost += Math.sqrt(dx * dx + dy * dy);
								p0 = point;
							}
						}
					} else if (id.equals("EG")) {
						String edge = matcher.group(5);
						n0 = Integer.parseInt(edge.substring(13, 19));
						n1 = Integer.parseInt(edge.substring(34, 40));
					} else if (id.equals("SB")) {
						type = Integer.parseInt(matcher.group(3)) - 13;
						if (type < 0) {
							type = 0;
						}
					} else if (id.equals("FI")) {
						// 17から16進数で与えられる
						width = Integer.parseInt(matcher.group(3), 16) - 23;
						if (width == 7) {
							width = 5;
						}
					}
				}
				Road b;
				if (isDetail) {
					b = new Road(curveX, curveY, type, width, (float)cost);
				} else {
					b = new Road(type, width, (float)cost);
				}
				Node node0 = node.get(n0);
				Node node1 = node.get(n1);
				node0.connect(code * 1000000L + n1, b);
				node1.connect(code * 1000000L + n0, b);
				if (isDetail) {
					if(type == 3) {
						highway[width].add(b);
					}else {
						border[width].add(b);
					}
				}
			}
		} finally {
			if(bi != null) {
				bi.close();
			}
		}
		InputStream stream = this.warehouse.getBoundaryNode(code);
		if(stream != null) {
			DataInputStream disc = null;
			try {
				disc = new DataInputStream(stream);
				while(disc.available() >= 12) {
					node.get(disc.readInt()).connect(disc.readLong(), null);
				}
			} finally {
				if (disc != null) {
					disc.close();
				}
			}
		}
		if (isDetail) {
			return new Road[][]{border[0].toArray(new Road[]{}), border[1].toArray(new Road[]{}), border[2].toArray(new Road[]{}), border[3].toArray(new Road[]{}), border[4].toArray(new Road[]{}), border[5].toArray(new Road[]{}), highway[0].toArray(new Road[]{}), highway[1].toArray(new Road[]{}), highway[2].toArray(new Road[]{}), highway[3].toArray(new Road[]{}), highway[4].toArray(new Road[]{}), highway[5].toArray(new Road[]{})};
		} else {
			return null;
		}
	}
}
