package map;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.geom.GeneralPath;
import java.util.Collection;
import java.util.Map;

import javax.swing.JPanel;

import map.data.Curve;
import map.data.MapData;
import map.data.MapMap;
import map.data.Mesh;
import map.data.Node;
import map.data.Road;
import map.labeling.Labeling;
import map.labeling.SimpleLabeling;
import map.route.ShortestPathAlgorithm;
import map.route.DirectDistance;
import map.route.ShortestPathSearch;
import map.route.SearchThread;

/**
 * Graph Class 2005/09/23
 * 
 * @author Masayasu FUJIWARA
 */
public class MapPanel extends JPanel {

	/**
	 * 高速道路のルート色
	 */
	private final Color ROUTE_COLOR_HIGHWAY = Color.RED;

	/**
	 * 一般道のルート色
	 */
	private final Color ROUTE_COLOR_GENERAL = Color.YELLOW;
	
	/**
	 * 最短経路探索アルゴリズム
	 */
	private ShortestPathAlgorithm algorithm;

	/**
	 * 市町村まで表示する倍率
	 */
	private final float CITY_SCALE = 0.0002f;

	/**
	 * 国土数値情報と数値地図25000を切り替える倍率
	 */
	private final float DATA25K_SCALE = 0.0015f;

	/**
	 * アンチエイリアスのフラグ
	 */
	private boolean isAntialias;

	/**
	 * 水域区間の塗りつぶしフラグ
	 */
	private boolean isFill;

	/**
	 * ラベル表示のフラグ
	 */
	private boolean isLabel;

	/**
	 * マウス操作のフラグ
	 */
	private boolean isOperation;

	/**
	 * 頂点表示のフラグ
	 */
	private boolean isNodeView;

	/**
	 * 操作マニュアルの表示フラグ
	 */
	private boolean isHelp;

	/**
	 * 地図情報管理マップ
	 */
	private MapMap maps;
	
	/**
	 * 頂点のサイズ
	 */
	private int nodeSize = 3;

	/**
	 * オフスクリーンイメージ
	 */
	private Image offs;

	/**
	 * 都道府県ポリゴン
	 */
	private Polygon[] prefecture;

	/**
	 * 地図の表示倍率
	 */
	private float scale = 0.005f;

	/**
	 * 表示倍率の上限
	 */
	private float SCALE_MAX = 0.1f;

	/**
	 * 表示倍率の下限
	 */
	private float SCALE_MIN = 0.000005f;

	/**
	 * 表示倍率の変更精度
	 */
	private float SCALE_SENSE = 0.08f;

	/**
	 * スクリーンサイズ
	 */
	private Rectangle screen;

	/**
	 * 探索の始点
	 */
	private Node start;
	
	/**
	 * 道路区間，鉄道区間の描画の基本の幅
	 */
	private final int STROKE_WIDTH = 130;

	/**
	 * 探索の終点
	 */
	private Node terminal;

	/**
	 * 探索端点の変更
	 */
	public synchronized void changeBoundary(int ex, int ey, boolean flag) {
		int x = (int)(ex / this.scale + this.screen.x);
		int y = (int)((this.getHeight() - ey) / this.scale + this.screen.y);
		
		Node point = null;
		double nodeDist = Double.POSITIVE_INFINITY;
		for (MapData map : this.maps.values()) {

			if (!map.hasData() || !map.getArea().contains(x, y)) {
				continue;
			}
			for (final Node node : map.getNode()) {
				final double d = node.distance(x, y, this.drawlevel);
				if (nodeDist > d) {
					point = node;
					nodeDist = d;
				}
			}
		}
		if (point == null) {
			return;
		}
		if (flag) {
			this.terminal = point;
			this.start = null;
			if (this.thread != null) {
				this.thread.kill();
			}
		} else {
			this.start = point;
			if (this.terminal != null && this.start != this.terminal) {
				if (this.thread != null) {
					this.thread.kill();
				}
				this.thread = this.algorithm.search(this.start, this.terminal);
			}
		}
	}
	private SearchThread thread;
	/**
	 * リルート計算
	 *
	 */
	public void reroute() {
		if(this.start != null && this.terminal != null && this.start != this.terminal) {
			if (this.thread != null) {
				this.thread.kill();
			}
			this.thread = this.algorithm.search(this.start, this.terminal);
			System.out.printf("s : %d, t : %d\n", this.start.getID(), this.terminal.getID());
		}
	}
	/**
	 * 行政界の描画
	 * 
	 * @param g
	 *            描画するGraphics
	 * @param curves
	 *            描画する行政界
	 * @param color
	 *            描画する色
	 */
	private void drawBorderCurve(Graphics g, Curve[] curves, Color color) {
		if (curves != null) {
			g.setColor(color);
			for (Curve curve : curves) {
				int[] curveX = curve.getArrayX();
				int[] curveY = curve.getArrayY();

				int x0 = (int) ((curveX[0] - this.screen.x) * this.scale);
				int y0 = this.getHeight() - (int) ((curveY[0] - this.screen.y) * this.scale);
				boolean isContains = this.screen.contains(curveX[0], curveY[0]);

				for (int k = 1; k < curveX.length; k++) {

					int x = (int) ((curveX[k] - this.screen.x) * this.scale);
					int y = this.getHeight() - (int) ((curveY[k] - this.screen.y) * this.scale);
					// 表示領域外であれば次へ
					if (isContains | (isContains = this.screen.contains(curveX[k], curveY[k]))) {
						g.drawLine(x0, y0, x, y);
					}
					x0 = x;
					y0 = y;
				}
			}
		}
	}

	/**
	 * 水域界の描画
	 * @param g 描画するGraphics2D
	 * @param coast 描画する水域界
	 * @param color
	 */
	private void drawCoast(Graphics2D g, Map<Curve, Boolean> coast, Color color1) {
		if (coast != null) {
			g.setColor(color1);
			for (Map.Entry<Curve, Boolean> entry : coast.entrySet()) {
				Curve curve = entry.getKey();
				int[] curveX = curve.getArrayX();
				int[] curveY = curve.getArrayY();


				if (this.isFill && entry.getValue() && curve.getType() == 0) {
					int[] x = new int[curveX.length];
					int[] y = new int[curveY.length];
					for (int i = 0; i < curveX.length; i++) {	
						x[i] = (int) ((curveX[i] - this.screen.x) * this.scale);
						y[i] = this.getHeight() - (int) ((curveY[i] - this.screen.y) * this.scale);
					}
					g.setColor(color1);
					g.fill(new Polygon(x, y, curveX.length));
				} else {
					g.setColor(color1);
					
					int x0 = (int) ((curveX[0] - this.screen.x) * this.scale);
					int y0 = this.getHeight() - (int) ((curveY[0] - this.screen.y) * this.scale);
					boolean isContains = this.screen.contains(curveX[0], curveY[0]);
					
					for (int i = 1; i < curveX.length; i++) {
						int x = (int) ((curveX[i] - this.screen.x) * this.scale);
						int y = this.getHeight() - (int) ((curveY[i] - this.screen.y) * this.scale);
						// 表示領域外であれば次へ
						if (isContains | (isContains = this.screen.contains(curveX[i], curveY[i]))) {
							g.drawLine(x0, y0, x, y);
						}
						x0 = x;
						y0 = y;
					}
				}
			}
		}
	}
	/**
	 * 数値地図25000を描画する
	 * 
	 * @param g
	 */
	public synchronized void drawMap(Graphics2D g) {

		int meshSize = (int) (Mesh.SIZE * this.scale) + 1;

		// メッシュの描画
		for (Map.Entry<Integer, MapData> entry : this.maps.entrySet()) {
			MapData map = entry.getValue();
			if (map == null) {
				System.out.println("map null !! code : " + entry.getKey());
				continue;
			}
			if(!map.hasData()) {
				this.drawPolygon(g, map.getPolygon(), true);
			} else {
				int halfMesh = Mesh.SIZE / 2;
				for (Mesh mesh : map.getMesh()) {
					// 表示領域外であれば次へ
					if (!this.screen.intersects(mesh.getX() - halfMesh, mesh.getY() - halfMesh, Mesh.SIZE, Mesh.SIZE)) {
						continue;
					}
					int x = (int) ((mesh.getX() - halfMesh - this.screen.x) * this.scale);
					int y = this.getHeight() - (int) ((mesh.getY() + halfMesh - this.screen.y) * this.scale);
	
					g.setColor(mesh.getColor());
					g.fillRect(x, y, meshSize, meshSize);
				}
			}
		}
		for (MapData map : this.maps.values()) {

			if (!map.hasData()) {
				continue;
			}

			// 行政界を描画
			this.drawBorderCurve(g, map.getBorder(), Color.GREEN);

			// 沿岸を描画
			this.drawCoast(g, map.getCoast(), Color.BLUE);

			// 河川を描画
			this.drawBorderCurve(g, map.getRiver(), Color.BLUE);
		}
		for (MapData map : this.maps.values()) {

			if (!map.hasData()) {
				continue;
			}
			
			// 道路を描画する
			Road[][] roads = map.getRoad();
			this.drawRoadway(g, roads);

			// 鉄道区間と駅区間を描画する
			this.drawRailway(g, map.getRailway(), map.getStation());

			if (this.isNodeView) {
				g.setColor(Color.GRAY);
				for (Node node : map.getNode()) {
					// メッシュサイズを計算
					int nodeSize = (int) (this.nodeSize / this.scale);
					// 表示領域外であれば次へ
					if (!this.screen.intersects(node.getX() - nodeSize, node.getY() - nodeSize, nodeSize * 2, nodeSize * 2)) {
						continue;
					}

					int x = (int) ((node.getX() - this.screen.x) * this.scale);
					int y = this.getHeight() - (int) ((node.getY() - this.screen.y) * this.scale);

					// draw node
					g.fillRect(x - this.nodeSize, y - this.nodeSize,
							this.nodeSize * 2, this.nodeSize * 2);
				}
			}
		}
		
		this.drawRoute(g);

		g.setStroke(new BasicStroke(1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
		if (this.isLabel) {

//			Labeling labeling = new DefaultLabeling(g, area, this.screen.x, this.screen.y, this.scale);
			Labeling labeling = new SimpleLabeling(g, this.screen, this.getWidth(), this.getHeight(), this.scale);
//			Labeling labeling = new Wagner(g, this.screen, this.getWidth(), this.getHeight(), this.scale);
			for (MapData map : this.maps.values()) {

				if (!map.hasData()) {
					continue;
				}

				labeling.add(map.getCity());
				labeling.add(map.getFacility());
			}
			labeling.draw();
		}
	}
	private void drawPolygon(Graphics g, Polygon[] polygons, boolean fill) {
		for (int i = 0; i < polygons.length; i++) {
			if(polygons[i].intersects(this.screen)) {
				int[] curveX = polygons[i].xpoints;
				int[] curveY = polygons[i].ypoints;
				
				int[] polygonX = new int[polygons[i].npoints];
				int[] polygonY = new int[polygons[i].npoints];
	
				for (int j = 0; j < polygons[i].npoints; j++) {
					polygonX[j] = (int) ((curveX[j] - this.screen.x) * this.scale);
					polygonY[j] = this.getHeight() - (int) ((curveY[j] - this.screen.y) * this.scale);
				}
				if (fill) {
					g.setColor(Color.GREEN.brighter());
					g.fillPolygon(polygonX, polygonY, polygons[i].npoints);
				}
				g.setColor(Color.BLACK);
				g.drawPolygon(polygonX, polygonY, polygons[i].npoints);
			}
		}
	}
	/**
	 * 鉄道区間，駅区間を描画
	 * 
	 * @param g
	 *            描画するGraphics2D
	 * @param rails
	 *            描画する鉄道区間
	 * @param color
	 *            描画する色
	 */
	private void drawRailway(Graphics2D g, Curve[] rails, Curve[] station) {
		int w = (int)(this.STROKE_WIDTH * this.scale * 2f) + 1;
		if (rails != null) {
			GeneralPath jr = new GeneralPath();
			GeneralPath other = new GeneralPath();
			for (Curve curve : rails) {
				GeneralPath rail = (curve.getType() == 0) ? jr : other;
				int[] curveX = curve.getArrayX();
				int[] curveY = curve.getArrayY();

				boolean isContains = this.screen.contains(curveX[0], curveY[0]);
				int x0 = (int) ((curveX[0] - this.screen.x) * this.scale);
				int y0 = this.getHeight()
						- (int) ((curveY[0] - this.screen.y) * this.scale);

				rail.moveTo(x0, y0);

				for (int k = 1; k < curveX.length; k++) {

					// 表示領域外であれば次へ
					if (!isContains & !(isContains = this.screen.contains(curveX[k],
									curveY[k])))
						continue;

					int x = (int) ((curveX[k] - this.screen.x) * this.scale);
					int y = this.getHeight()
							- (int) ((curveY[k] - this.screen.y) * this.scale);
					
					rail.lineTo(x, y);
				}
			}
			
			g.setStroke(new BasicStroke(w + 2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.setColor(Color.DARK_GRAY);
			g.draw(other);
			g.setColor(Color.BLACK);
			g.draw(jr);

			g.setColor(Color.BLACK);
			g.setStroke(new BasicStroke(w, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.draw(other);
			g.setColor(Color.WHITE);
			g.setStroke(new BasicStroke(w, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND, 10f, new float[]{w * 3, w * 6}, 0));
			g.draw(jr);
		}
		if (station != null) {
			GeneralPath border = extractRoadway(station);
	
			g.setColor(Color.BLACK);
			g.setStroke(new BasicStroke(w * 3 + 2, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_ROUND));
			g.draw(border);
	
			g.setColor(Color.WHITE);
			g.setStroke(new BasicStroke(w * 3, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_ROUND));
			g.draw(border);
		}
	}

	private int drawlevel;

	/**
	 * 道路の描画
	 * 
	 * @param g Graphics2D
	 * @param road 描画する道路(一般道[6], 高速道路[6])
	 */
	private void drawRoadway(Graphics2D g, Road[][] road) {
		GeneralPath[] path = new GeneralPath[road.length];
		int[] width = new int[6];
		for (int i = 5; i >= 0; i--) {
			width[i] = (int)((i + 1) * this.STROKE_WIDTH * this.scale);
			if(width[i] > 0) {
				path[i] = this.extractRoadway(road[i]);
				this.drawlevel = i;
			}
			path[i + 6] = this.extractRoadway(road[i + 6]);
		}
		g.setColor(Color.LIGHT_GRAY);
		for (int i = 0; i < 6; i++) {
			if(path[i] != null) {
				g.setStroke(new BasicStroke(width[i] + 2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				g.draw(path[i]);
			}
		}
		g.setColor(Color.WHITE);
		for (int i = 0; i < 6; i++) {
			if(path[i] != null) {
				g.setStroke(new BasicStroke(width[i], BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
				g.draw(path[i]);
			}
		}
		g.setColor(Color.DARK_GRAY);
		for (int i = 0; i < 6; i++) {
			g.setStroke(new BasicStroke(width[i] + 3, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.draw(path[i + 6]);
		}
		g.setColor(Color.GRAY);
		for (int i = 0; i < 6; i++) {
			g.setStroke(new BasicStroke(width[i] + 1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
			g.draw(path[i + 6]);
		}
	}

	/**
	 * 探索済み頂点を描画する
	 * 
	 * @param g
	 */
	private void drawRoute(Graphics2D g) {
		if (this.thread != null) {
			Collection<Road> route = this.thread.getRoute();
			if (route != null) {
				int[] width = new int[12];
				for (int i = 0; i < 6; i++) {
					width[i] = (int)((i + 1) * this.STROKE_WIDTH * this.scale) + 1;
				}
				for (Road road : route) {
					int[] aryX = road.getArrayX();
					int[] aryY = road.getArrayY();
		
					boolean isContain = this.screen.contains(aryX[0], aryY[0]);
					int x0 = (int) ((aryX[0] - this.screen.x) * this.scale);
					int y0 = this.getHeight()
							- (int) ((aryY[0] - this.screen.y) * this.scale);
					
					if (road.getType() == 3) {
						g.setColor(this.ROUTE_COLOR_HIGHWAY);
					} else {
						g.setColor(this.ROUTE_COLOR_GENERAL);
					}
					g.setStroke(new BasicStroke(width[road.getWidth()], BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
		
					for (int k = 1; k < aryX.length; k++) {
		
						int x = (int) ((aryX[k] - this.screen.x) * this.scale);
						int y = this.getHeight()
						- (int) ((aryY[k] - this.screen.y) * this.scale);
		
						// 表示領域内であれば描画
						if (isContain | (isContain = this.screen.contains(aryX[k], aryY[k]))) {
							g.drawLine(x0, y0, x, y);
						}
						x0 = x;
						y0 = y;
					}
				}
			}
		}
		int r = (int)(this.scale * 500) + 2;
		g.setStroke(new BasicStroke(r / 2f));
		if (this.start != null) {
			int x = (int) ((this.start.getX() - this.screen.x) * this.scale) - r;
			int y = this.getHeight() - (int) ((this.start.getY() - this.screen.y) * this.scale) - r;
			g.setColor(Color.YELLOW);
			g.fillOval(x, y, r * 2, r * 2);
			g.setColor(Color.BLACK);
			g.drawOval(x, y, r * 2, r * 2);
		}
		if (this.terminal != null) {
			int x = (int) ((this.terminal.getX() - this.screen.x) * this.scale) - r;
			int y = this.getHeight() - (int) ((this.terminal.getY() - this.screen.y) * this.scale) - r;
			g.setColor(Color.RED);
			g.fillOval(x, y, r * 2, r * 2);
			g.setColor(Color.BLACK);
			g.drawOval(x, y, r * 2, r * 2);
		}
	}
	/**
	 * 道路データをGeneralPathに展開する
	 * @param curves 道路データ
	 * @return 展開した道路データ
	 */
	private GeneralPath extractRoadway(Curve[] curves) {
		GeneralPath border = new GeneralPath();
		if (curves != null) {
			for (Curve curve : curves) {
				int[] curveX = curve.getArrayX();
				int[] curveY = curve.getArrayY();
	
				boolean isContains = this.screen.contains(curveX[0], curveY[0]);
				int x0 = (int) ((curveX[0] - this.screen.x) * this.scale);
				int y0 = this.getHeight()
						- (int) ((curveY[0] - this.screen.y) * this.scale);
	
				border.moveTo(x0, y0);
				
				for (int k = 1; k < curveX.length; k++) {
	
					// 表示領域外であれば次へ
					if (!isContains & !(isContains = this.screen.contains(curveX[k],
									curveY[k])))
						continue;
	
					int x = (int) ((curveX[k] - this.screen.x) * this.scale);
					int y = this.getHeight()
							- (int) ((curveY[k] - this.screen.y) * this.scale);
					border.lineTo(x, y);
				}
			}
		}
		return border;
	}

	/**
	 * スクリーン情報の取得
	 * 
	 * @return スクリーンのRectangle
	 */
	public Rectangle getScreen() {
		return this.screen;
	}

	/**
	 * 初期設定
	 */
	public void init(MapMap map, Polygon[] prefecture) {
		this.algorithm = new ShortestPathSearch(new DirectDistance(), map);
		this.maps = map;
		
		this.prefecture = prefecture;
		
		this.screen = new Rectangle();
		this.screen.x = 122 * 3600000;
		this.screen.y = 20 * 3600000;

		this.isLabel = true;
		this.isNodeView = false;
		this.isAntialias = true;

		this.isFill = false;
		
		this.isHelp = true;

		this.isOperation = false;
		
		super.setBackground(Color.CYAN);
		
		double widthScale = (double) this.getWidth() / ((154 - 122) * 3600000);
		double heightScale = (double) this.getHeight() / ((46 - 20) * 3600000);
		this.scale = (float) ((widthScale < heightScale) ? widthScale : heightScale);
		
		this.repaint();
	}
	/**
	 * どの地図を表示する
	 * @return 
	 */
	public int mode() {
		if(this.scale > this.DATA25K_SCALE) {
			return 2;
		} else if(this.scale > this.CITY_SCALE) {
			return 1;
		} else {
			return 0;
		}
	}
	/**
	 * 平行移動を行う
	 * 
	 * @param dx
	 * @param dy
	 */
	public void moveLocation(int dx, int dy) {
		this.screen.x -= dx / this.scale;
		this.screen.y += dy / this.scale;
		repaint();
	}
	@Override
	public void paintComponent(Graphics g) {
		// オフスクリーンバッファ
		if(this.maps == null) {
			this.offs = createImage(this.getWidth(), this.getHeight());
			Graphics2D offg = (Graphics2D)this.offs.getGraphics();
			super.paintComponent(offg);
			Font defaultFont = g.getFont();
			offg.setFont(new Font("gothic", Font.PLAIN, 20));
			FontMetrics metrics = offg.getFontMetrics();
			String msg = "Now Loading...";
			offg.drawString(msg, (this.getWidth() - metrics.stringWidth(msg)) / 2, (this.getHeight() - metrics.getHeight()) / 2);
			offg.setFont(defaultFont);
		} else {
			if(this.getWidth() != this.offs.getWidth(null) || this.getHeight() != this.offs.getHeight(null)) {
				this.offs.flush();
				this.offs = createImage(this.getWidth(), this.getHeight());
			}
			Graphics2D offg = (Graphics2D)this.offs.getGraphics();
			super.paintComponent(offg);

			// スクリーン情報を指定する
			this.screen.width = (int) (this.getWidth() / this.scale);
			this.screen.height = (int) (this.getHeight() / this.scale);
			
			if(this.scale > this.CITY_SCALE) {
				if (!this.isOperation && this.isAntialias) {
					offg.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
					offg.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
				}
				if (this.scale >= this.DATA25K_SCALE) {
					this.drawMap(offg);
				} else {
					for (MapData map : this.maps.values()) {
						this.drawPolygon(offg, map.getPolygon(), true);
					}
					offg.setStroke(new BasicStroke(2, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND));
					this.drawPolygon(offg, this.prefecture, false);
					this.drawRoute(offg);
				}
			} else {
				this.drawPolygon(offg, this.prefecture, true);
			}

			offg.setColor(Color.BLACK);
			offg.drawString("SCALE : " + Float.toString(this.scale * 1000) + "m", 5, 15);
			// アルゴリズム名の表示
			int vp = this.thread != null ? this.thread.getVP() : 0;
			if (vp == 0) {
				offg.drawString(this.algorithm.toString(), 5, 35);
			} else {
				offg.drawString(this.algorithm.toString() + " : " + vp, 5, 35);
			}
			if (this.isHelp) {
				this.drawHelp(offg, this.getWidth() - 325, this.getHeight() - 240);
			}
		}
		g.drawImage(this.offs, 0, 0, null);
	}
	public void getLocation(int ex, int ey) {
		int x = (int)(ex / this.scale + this.screen.x);
		int y = (int)((this.getHeight() - ey) / this.scale + this.screen.y);
		this.maps.setStatus(x, y);
	}
	public void clearLocation() {
		this.maps.clearStatus();
	}
	/**
	 * マウス操作の状態を設定する
	 * @param flag マウス操作していればtrue
	 */
	public void setOperation(boolean flag) {
		this.isOperation = flag;
		if(!flag) {
			this.repaint();
		}
	}
	public boolean isOperation() {
		return this.isOperation;
	}
	/**
	 * 操作マニュアルの表示を切り替える
	 */
	public void switchHelp() {
		this.isHelp = !this.isHelp;
		this.repaint();
	}
	/**
	 * 水域界の塗りつぶしを切り替える
	 */
	public void switchFill() {
		this.isFill = !this.isFill;
		this.repaint();
	}
	/**
	 * ラベル表示を切り替える
	 */
	public void switchLabel() {
		this.isLabel = !this.isLabel;
		this.repaint();
	}

	/**
	 * 頂点表示を切り替える
	 */
	public void switchNodeView() {
		this.isNodeView = !this.isNodeView;
		this.repaint();
	}
	/**
	 * アンチエイリアスを切り替える
	 */
	public void switchRendering() {
		this.isAntialias = !this.isAntialias;
		this.repaint();
	}
	/**
	 * 最短経路を求めるアルゴリズムを切り替える
	 */
	public void switchShortestPathAlgorithm(String algorithm) {
		if(algorithm.endsWith("_a*")) {
			this.algorithm = new ShortestPathSearch(new DirectDistance(), this.maps);
		} else if (algorithm.endsWith("_dijkstra")) {
			this.algorithm = new ShortestPathSearch(null, this.maps);
		}
		this.reroute();
		this.repaint();
	}


	/**
	 * 拡大縮小を行う
	 * 
	 * @param x
	 * @param y
	 * @param d
	 */
	public void zoom(int x, int y, int d) {
		float newScale = this.scale * (1 + d * this.SCALE_SENSE);
		if (newScale > this.SCALE_MAX)
			newScale = this.SCALE_MAX;
		else if (newScale < this.SCALE_MIN)
			newScale = this.SCALE_MIN;

		y = this.getHeight() - y;
		this.screen.x = (int) (this.screen.x + x / this.scale - x / newScale);
		this.screen.y = (int) (this.screen.y + y / this.scale - y / newScale);

		this.scale = newScale;
		repaint();
	}
	public void drawHelp(Graphics2D g, int x, int y) {
		g.setColor(new Color(255, 255, 255, 200));

		int width = 305;
		int height = 220;
		g.fillRect(x, y, width, height);

		g.setColor(Color.BLACK);
		g.setFont(new Font("SansSerif", Font.BOLD, 14));
		FontMetrics metrics = g.getFontMetrics();
		String msg = "操作マニュアル";
		g.drawString(msg,  x + (width - metrics.stringWidth(msg)) / 2, y + 25);
		g.setFont(new Font("SansSerif", Font.BOLD, 12));
		
		g.drawString("↑ / ↓ / ← / →", x + 180, y + 50);
		g.drawString("＋ / ； / PgUp", x + 180, y + 95);
		g.drawString("－ / ＝ / PgDn", x + 180, y + 140);
		g.drawString("？", x + 180, y + 185);

		g.drawString("マウス ドラッグ", x + 20, y + 50);
		g.drawString("マウス ホイール", x + 20, y + 95);
		g.drawString("右クリック", x + 20, y + 140);
		g.drawString("Shift + 右クリック", x + 20, y + 185);


		g.setFont(new Font("SansSerif", Font.PLAIN, 12));
		g.drawString("地図の平行移動",  x + 180, y + 70);
		g.drawString("地図の拡大",     x + 180, y + 115);
 		g.drawString("地図の縮小",     x + 180, y + 160);
		g.drawString("マニュアル表示切替", x + 180, y + 205);

 		g.drawString("地図の平行移動", x + 20, y + 70);
 		g.drawString("地図の拡大縮小", x + 20, y + 115);
 		g.drawString("探索始点の選択 / 探索開始", x + 20, y + 160);
 		g.drawString("探索終点の選択 / 探索中止", x + 20, y + 205);
	}
}